#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <config.h>
#include <support.h>
#include <xcio.h>

#include "option.h"
#include "log.h"
#include "phase.h"
#include "env.h"
#include "console.h"
#include "timer.h"
#include "frame.h"
#include "fsm.h"
#include "cps.h"
#include "ccp.h"
#include "dev/device.h"

extern void PredInit();
extern int Pred1Compress();
extern u_char *Pred1Uncompress();

#ifdef	HAVE_DEFLATE
extern void DeflateInit();
extern int DeflateCompress();
extern u_char *DeflateUncompress();
#endif

void (*compInitC)(), (*compInitUC)();
int (*compC)();
u_char *(*compUC)();

#define	CCPCODE_MAX	CPCODE_RESET_ACK

#define	CCPCONF_OUI	0
#define	CCPCONF_PRED1	1	/* license free :-) rfc1978 */
#define	CCPCONF_PRED2	2	/* license free :-) rfc1978 */
#define	CCPCONF_PUDDLE	3
#define	CCPCONF_HPPPC	16	/* licensed ;-p draft-ietf-pppext-hpppc */
#define	CCPCONF_SELZS	17	/* licensed ;-p rfc1974 */
#define	CCPCONF_MSPPC	18	/* licensed ;-p draft-ietf-pppext-mppc */
#define	CCPCONF_GFZA	19	/* licensed ;-p rfc1993 */
#define	CCPCONF_V42BIS	20
#define	CCPCONF_BSDLZW	21	/* license free :-) rfc1977 */
#define	CCPCONF_LZSDCP	23	/* licensed ;-p rfc1967 */
#define	CCPCONF_MVRCA	24	/* licensed ;-p rfc1975 */
#define	CCPCONF_DCE	25	/* rfc1976 */
#define	CCPCONF_DEFLATE	26	/* license free :-) rfc1979 */
#define	CCPCONF_MAX	27

static struct ccpreg_s {
    b256_t types;
    u_int8_t deflate_win;
} remoteReg, localReg, xReg;

static struct fsmreg_s ccpFsm;
static struct cpcodeop_s ccpCop[CCPCODE_MAX + 1];
static struct cptypeop_s ccpTop[]={
    {
	"OUI", NULL,
	NULL, NULL, 0,
	CCPCONF_OUI
    }, {
	"Predictor type 1", NULL,
	NULL, NULL, 0,
	CCPCONF_PRED1
    }, {
	"Predictor type 2", NULL,
	NULL, NULL, 0,
	CCPCONF_PRED2
    }, {
	"Puddle Jumper", NULL,
	NULL, NULL, 0,
	CCPCONF_PUDDLE
    }, {
	"Hewlett-Packard PPC", NULL,
	NULL, NULL, 0,
	CCPCONF_HPPPC
    }, {
	"Stac Electronics LZS", NULL,
	NULL, NULL, 0,
	CCPCONF_SELZS
    }, {
	"Microsoft PPC", NULL,
	NULL, NULL, 0,
	CCPCONF_MSPPC
    }, {
	"Gandalf FZA", NULL,
	NULL, NULL, 0,
	CCPCONF_GFZA
    }, {
	"V.42bis compression", NULL,
	NULL, NULL, 0,
	CCPCONF_V42BIS
    }, {
	"BSD LZW Compress", NULL,
	NULL, NULL, 0,
	CCPCONF_BSDLZW
    }, {
	"LZS-DCP", NULL,
	NULL, NULL, 0,
	CCPCONF_LZSDCP
    }, {
	"Magnalink VRCA", NULL,
	NULL, NULL, 0,
	CCPCONF_MVRCA
    }, {
	"Deflate", NULL,
	NULL, NULL, 0,
	CCPCONF_DEFLATE
    }
};

static struct ccpOp_s opList[]={
#ifdef	HAVE_DEFLATE
    {"deflate", DeflateInit, DeflateCompress, DeflateUncompress,
	 CCPCONF_DEFLATE},
#endif
/*    {"bsdlzw", NULL, NULL, NULL, CCPCONF_BSDLZW},*/
    {"pred1", PredInit, Pred1Compress, Pred1Uncompress, CCPCONF_PRED1},
    {NULL}
};

#define	MAX_COMP_PROTO	((int)(sizeof(opList)/sizeof(struct ccpOp_s)) - 1)

struct ccpopt_s {
    bool_t enable;
    u_int32_t option;
    union {
	bool_t b;
	u_int32_t i;
    } options;
#define	opt_b	options.b
#define	opt_i	options.i
};

static struct ccpopt_s ccpOpt[MAX_COMP_PROTO];

static cptype_t remoteOrder[CCPCONF_MAX], localOrder[CCPCONF_MAX];

static struct cpstate_s ccpState={
    ccpCop,		/* code oprators */
    ccpTop,		/* type operators */
    (void *)&remoteReg,	/* remote registry */
    (void *)&localReg,	/* local registry */
    NULL,		/* received buffer pointers */
    NULL,		/* encode order */
    remoteOrder,	/* remote(received) order */
    localOrder,		/* request(local) order */
    {{0}},		/* encode list */
    {{0}},		/* remote(received) list */
    {{0}},		/* request(local) list */
    /* number of configuration types */
    sizeof(ccpTop)/sizeof(struct cptypeop_s),
    NBO_PROTO_CCP,
    0, 0		/* remote/local ID */
};

static int
CcpEncodeReg(u_char *buf, struct ccpreg_s *ccr)
{
    if (ccr) memcpy(&xReg, ccr, sizeof(struct ccpreg_s));
    return(CpEncodeReg(&ccpState, buf, FALSE));
}

/* FSM action functions */

static void
TLD(struct fsmreg_s *fsm)
{
    compInitC = NULL;
    compInitUC = NULL;
    compC = NULL;
    compUC = NULL;
/*
    const char *oldpath;

    oldpath = SetLoadFilePath("ip");
    LoadScriptFile(ipcpOpt.down);
    SetLoadFilePath(oldpath);
*/
}

static void
TLF(struct fsmreg_s *fsm)
{
    /*    pppInfo.p[PINFO_CCP].st = PIST_DOWN;*/
    FsmInit(fsm);
}

static void
CcpDown(bool_t process)
{
    FsmClose(&ccpFsm);
    if (!process) TLF(&ccpFsm);
    /*    pppInfo.p[PINFO_CCP].st = PIST_DOWN;*/
}

static void
TLU(struct fsmreg_s *fsm)
{
    int i;
    struct ccpOp_s *cop;
    bool_t found_c, found_uc;

    found_c = found_uc = FALSE;
    for (i = 0; i < MAX_COMP_PROTO; i ++) if (ccpOpt[i].enable) {
	cop = &opList[i];
	if (!found_c && B256_ISSET(&remoteReg.types, cop->type)) {
	    compInitC = cop->init;
	    compInitC(COMP_INITC);
	    compC = cop->compress;
	    found_c = TRUE;
	}
	if (!found_uc && B256_ISSET(&localReg.types, cop->type)) {
	    compInitUC = cop->init;
	    compInitUC(COMP_INITUC);
	    compUC = cop->uncompress;
	    found_uc = TRUE;
	}
	if (found_c && found_uc) break;
    }
    if (found_c || found_uc) RegisterDownPhase("CCP", CcpDown);
    /*    pppInfo.p[PINFO_CCP].st = PIST_UP;*/
}

static int
RCR(u_char *buf, int n)
{
    int i=0, k;
    bool_t accept=FALSE;
    cptype_t type;
    b256_t rej, nak;

    memset(&xReg, 0, sizeof(struct ccpreg_s));
    if (CpDecodeReg(&ccpState, buf, n) < 0) {
	FsmRUC(&ccpFsm);
	return(0);
    }
    B256_ZERO(&nak);
    B256_ZERO(&rej);
    ccpFsm.nak_rcr = ccpFsm.rej_rcr = 0;
    while ((type = remoteOrder[i++]) < CCPCONF_MAX) {
	for (k = 0; k < MAX_COMP_PROTO; k ++)
	    if (ccpOpt[k].enable && opList[k].type == type) {
		accept = TRUE;
		B256_SET(&xReg.types, type);
		break;
	    }
	if (!accept) {
	    /* not supported */
	    ccpFsm.rej_rcr = 1;
	    B256_SET(&rej, type);
	    continue;
	} else continue;
/*
	ccpFsm.nak_rcr = 1;
	B256_SET(&nak, type);
*/
    }
    memcpy(&remoteReg, &xReg, sizeof(struct ccpreg_s));
    if (ccpFsm.rej_rcr) B256_CPY(&ccpState.r_list, &rej);
    else if (ccpFsm.nak_rcr) B256_CPY(&ccpState.r_list, &nak);
    FsmRCR(&ccpFsm);
    return(0);
}

static int
RCA(u_char *buf, int n)
{
    int i=0;
    cptype_t type;
    b256_t rej, nak;

    memset(&xReg, 0, sizeof(struct ccpreg_s));
    CpDecodeReg(&ccpState, buf, n);
    while ((type = remoteOrder[i++]) < CCPCONF_MAX) {
	switch (type) {
	case CCPCONF_DEFLATE:
	case CCPCONF_PRED1:
	    B256_SET(&xReg.types, type);
	    continue;
	default:
	    /* not supported */
	    B256_SET(&rej, type);
	    continue;
	}
	B256_SET(&nak, type);
    }
    memcpy(&localReg, &xReg, sizeof(struct ccpreg_s));
    FsmRCA(&ccpFsm);
}

#if 0
static int
RXJ(u_char *buf, int n)
{
    memset(&xReg, 0, sizeof(struct ccpreg_s));
    CpDecodeReg(&ccpState, buf, n);
/*
    CpDecodeReg(&ccpState, buf, n);
    ccpState.l_list &= ~ccpState.r_list;
    ccpFsm.fatal_rxj = (ccpState.r_list & FATAL_RXJ) ? 0: 1;
*/
    ccpFsm.fatal_rxj = 1;
    FsmRXJ(&ccpFsm);
}
#endif

static int
RCNJ(u_char *buf, int n, bool_t rcj)
{
    int i=0;
    cptype_t type;

    memset(&xReg, 0, sizeof(struct ccpreg_s));
    CpDecodeReg(&ccpState, buf, n);
    while ((type = remoteOrder[i++]) < ENDOF_LIST) {
	if (rcj) B256_CLR(&ccpState.l_list, type);
	switch (type) {
	case CCPCONF_PRED1:
	    break;
	}
    }
    return(FsmRCN(&ccpFsm));
}

static int
RCN(u_char *buf, int n)
{
    return(RCNJ(buf, n, FALSE));
}

static int
RCJ(u_char *buf, int n)
{
    return(RCNJ(buf, n, TRUE));
}

static int
RTR(u_char *buf, int n)
{
    return(CpRtr(&ccpFsm, buf, n));
}

static int
RTA(u_char *buf, int n)
{
    return(CpRta(&ccpFsm, buf, n));
}

void
CcpSRR()
{
    CpEnqueueFrame(&ccpFsm, CPCODE_RESET_REQ, NULL, ccpState.l_id);
}

static int
RRR(u_char *buf, int n)
{
    if (compInitC) compInitC(COMP_INITC);
    CpEnqueueFrame(&ccpFsm, CPCODE_RESET_ACK, NULL, ccpState.r_id);
}

static int
RRA(u_char *buf, int n)
{
    if (compInitUC) compInitUC(COMP_INITUC);
}

static bool_t
EnvCompWithOpt(int argc, char *argv[], char *outs)
{
    int i, proto;
    char *p;

    if ((p = index(argv[0], '.')) != NULL) p ++;
    else p = argv[0];

    for (proto = 0; proto < MAX_COMP_PROTO; proto ++)
	if (!strcasecmp(opList[proto].name, p)) break;
    if (!argc) {
	if (!ccpOpt[proto].enable) strcpy(outs, "no");
	else {
	    if (!ccpOpt[proto].option) strcpy(outs, "yes");
	    else sprintf(outs, "%d", ccpOpt[proto].option);
	}
	return(FALSE);
    }
    ccpOpt[proto].enable = FALSE;
    if (*argv[1] != '0'
	&& strcasecmp("no", argv[1]) && strcasecmp("off", argv[1])) {
	ccpOpt[proto].enable = TRUE;
	ccpOpt[proto].option = atoi(argv[1]);
    }
    for (i = proto = 0; proto < MAX_COMP_PROTO; proto ++)
	if (ccpOpt[proto].enable) i ++;
    FrameReject(NBO_PROTO_CCP, (i > 0) ? 0: 1);
    return(TRUE);
}

static void
CcpEnable(bool_t sw)
{
    FrameEnable(NBO_PROTO_CCP, sw);
}

void
CcpSetup()
{
    static struct env_s envlist[]={
	/*	{"PROTO", {EnvCompProto}, ENV_SET},*/
	{"PRED1", {EnvCompWithOpt}, ENV_SET, 0, 0, 0666},
	{"DEFLATE", {EnvCompWithOpt}, ENV_SET, 0, 0, 0666},
	{NULL}
    };

    ccpFsm.name = "CCP";
    RegisterEnvs(envlist, ccpFsm.name, CcpEnable);
    RegisterCp(&ccpState, &ccpFsm);
    memset(ccpOpt, 0, sizeof(ccpOpt));
    compInitC = NULL;
    compInitUC = NULL;
    compC = NULL;
    compUC = NULL;
    ccpFsm.timer.name = "CCPFSM";
    ccpFsm.timer.arg = &ccpFsm;
    ccpCop[CPCODE_CONF_REQ].fsmevent = RCR;
    ccpCop[CPCODE_CONF_REQ].encode = CcpEncodeReg;
    ccpCop[CPCODE_CONF_ACK].fsmevent = RCA;
    ccpCop[CPCODE_CONF_ACK].encode = CcpEncodeReg;
    ccpCop[CPCODE_CONF_NAK].fsmevent = RCN;
    ccpCop[CPCODE_CONF_NAK].encode = CcpEncodeReg;
    ccpCop[CPCODE_CONF_REJ].fsmevent = RCJ;
    ccpCop[CPCODE_CONF_REJ].encode = CcpEncodeReg;
    ccpCop[CPCODE_TERM_REQ].fsmevent = RTR;
    ccpCop[CPCODE_TERM_ACK].fsmevent = RTA;

    ccpCop[CPCODE_RESET_REQ].fsmevent = RRR;
    ccpCop[CPCODE_RESET_ACK].fsmevent = RRA;
    ccpFsm.tlu = TLU;
    ccpFsm.tlf = TLF;
    ccpFsm.tld = TLD;
    /*    strcpy(pppInfo.p[PINFO_CCP].name, ccpFsm.name);*/
}

void
CcpInit()
{
    int i, k;

    memset(&remoteReg, 0, sizeof(struct ccpreg_s));
    ccpState.r_id = ccpState.l_id = 0;
    B256_ZERO(&ccpState.l_list);
    for (k = i = 0; i < MAX_COMP_PROTO; i ++) if (ccpOpt[i].enable) {
	B256_SET(&ccpState.l_list, opList[i].type);
	k ++;
    }
    if (k == 0) {
	FrameReject(NBO_PROTO_CCP, 1);
	return;
    }
/*    if (pppOpt.mode != RUN_PASSIVE && i == 0) return;*/
/*    localReg.reg_pred1 = TRUE;*/
    for (k = 0, i = 0; i < CCPCONF_MAX; i ++)
	if (B256_ISSET(&ccpState.l_list, i)) localOrder[k++] = i;
    localOrder[k] = ENDOF_LIST;

    ccpFsm.init_r_count = pppOpt.r_count;
    ccpFsm.init_r_to = pppOpt.r_to;
    FsmInit(&ccpFsm);
    FsmUp(&ccpFsm);
/*    ccpFsm.opt_p = (pppOpt.mode == RUN_PASSIVE) ? 1: 0;*/
    ccpFsm.opt_p = 0;
    FsmOpen(&ccpFsm);
    /*    pppInfo.p[PINFO_CCP].st = PIST_RUN;*/
}
