##############################################################################
#
# Print billing management system - support modules, version 4.1.2
#
# Copyright (C) 2000, 2001, 2002, 2003 Daniel Franklin
#
# This module is free software; you can redistribute it and/or modify it
# under the same terms as Perl itself.
#
# This module contains various functions (well, one actually) used by
# printbill. 'printbill' calculates number of pages and the percentage CMYK
# coverage for a given postscript file. If mono printing is requested, only
# param 1 and 5 have any meaningful value - the others are 0.
#
##############################################################################

package Printbill::printbill;

use POSIX;
use strict;
use File::Path;
use File::Temp qw/ tempdir /;

BEGIN {
	use Exporter ();
	
	@Printbill::printbill::ISA = qw (Exporter);
	@Printbill::printbill::EXPORT = qw (&printbill);
	$Printbill::printbill::VERSION = 4.1.2;
}

# Given the filename of a file to be billed, the operative colourspace, a
# path to find the binaries and the desired priority for the process (and an
# optional blah -> postscript conversion filter if desired - set to none if
# not used), compute the total page count and the CMYK coverage for this
# file.

sub printbill
{
	my ($ps_filename, $priority, $tmpdir_root, %params) = @_;
	my ($printbill_tmpdir, $rand);
	my @total_channels = (0, 0, 0, 0);
	my @channels = (0, 0, 0, 0);
	my ($total_pages, $N, $p, $i, $gsdev);
	my ($pages, $answer, @results);
	
	$rand = join '', (0..9, 'A'..'Z', 'a'..'z', '_', '-')[rand 64, rand 64, rand 64, rand 64, rand 64, rand 64, rand 64, rand 64];

	$printbill_tmpdir = "$tmpdir_root/$rand";
	
	mkdir $printbill_tmpdir, 0700 or do {
		return ("$0 (printbill.pm): cannot create directory $printbill_tmpdir: $!");
	};
	
# Generate a set of PNG images for the file in that directory

# In this case we just use a dumb postscript pagecount algorithm. This is
# very easy to fool if you know what you're doing (which, fortunately, means
# windows client environments should be fine).


	if ($params{'colourspace'} eq "pagecount") {
		open POSTSCRIPT, "<$ps_filename";
		
		my $copies = 1;
		
		while (<POSTSCRIPT>) {
			if ($_ =~ "%%Page: ") {
				$total_pages++;
			}
			
			if ($_ =~ /^ (\d+) \@copies$/) {
				$copies = $1;
			}
		}
		
		$total_pages *= $copies;
		@total_channels = (0, 0, 0, 0);
		
		close POSTSCRIPT;
	} else {
		if ($params{'colourspace'} eq "mono") {
			$gsdev = "pnggray";
		} else {
			$gsdev = "png16m";
		}

		if ($params{'filter'} eq 'none') {
			$answer = `$params{'nice'} -$priority $params{'gs'} -sDEVICE=$gsdev -r$params{'dpi'} -dBATCH -dNOPAUSE -dSAFER -sOutputFile=$printbill_tmpdir/page%d.png -q \'$ps_filename\' 2>&1`;
		} else {
			$answer = `/bin/cat \'$ps_filename\' | $params{'filter'} | $params{'nice'} -$priority $params{'gs'} -sDEVICE=$gsdev -r$params{'dpi'} -dBATCH -dNOPAUSE -dSAFER -sOutputFile=$printbill_tmpdir/page%d.png -q - 2>&1`;
		}

		chomp $answer;

# Ghostscript reported some kind of error. Return undefined.
	
		if ($answer =~ /error/i) {
			`/bin/rm -rf $printbill_tmpdir`;
			return ("$0 (printbill.pm): gs cannot parse $ps_filename (errors follow):\n$answer");
		}

# Work out the total number of pages and the percentage of each which is
# black and/or coloured. 4-5% is typical. Charge accordingly.

		$N = 1;
		$total_pages = 0;
			
		if ($params{'colourspace'} eq "mono") {
			while (-e "$printbill_tmpdir/page$N.png") {
				$p = `$params{'prog_path'}/percentblack $printbill_tmpdir/page$N.png`;
				chomp $p;
				
				@channels = split (':', $p);
			
				$total_channels [3] += $channels [0] * $channels [1];
				$total_pages += $channels [1];
	
				$N++;
			}
		} elsif ($params{'colourspace'} eq "colour" || $params{'colourspace'} eq "cmyk") {
			while (-e "$printbill_tmpdir/page$N.png") {
				$p = `$params{'prog_path'}/percentcolour $printbill_tmpdir/page$N.png`;
				chomp $p;
					
				@channels = split (':', $p);
			
# @channels is [C, M, Y, K]
				for ($i = 0; $i < 4; $i++) {
					$total_channels [$i] += $channels [$i] * $channels [4];
				}
	
				$total_pages += $channels [4];
					
				$N++;
			}
		} elsif ($params{'colourspace'} eq "cmy") {
			while (-e "$printbill_tmpdir/page$N.png") {
				$p = `$params{'prog_path'}/percent_cmy $printbill_tmpdir/page$N.png`;
				chomp $p;
					
				@channels = split (':', $p);
# @channels is [C, M, Y, (K = 0)]
				for ($i = 0; $i < 3; $i++) {
					$total_channels [$i] += $channels [$i] * $channels [3];
				}
			
				$total_pages += $channels [3];
	
				$N++;
			}

			$total_channels[3] = 0;
		}
	}

# Clean up after ourselves

	@results = ("", $total_pages, $total_channels [0], $total_channels [1], $total_channels [2], $total_channels [3]);
	
	`/bin/rm -rf $printbill_tmpdir`;

	return @results;
}

END {}

1;
