/* 
 * Prospect: a developer's system profiler.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Authors: Alex Tsariounov, HP
 *          Keith Fish, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: f_sym_name.c,v 1.10 2004/01/09 20:29:28 type2 Exp $ */

/*
********************************************************************************
**
**                            PROSPECT PROJECT
**                   Elf64 and Elf32 symbol table code.
**        Find a symbol value given the file_name and symbol_name.
**
********************************************************************************
*/

#ifndef __LINT__
static const char gRCSid[] = "@(#) $Id: f_sym_name.c,v 1.10 2004/01/09 20:29:28 type2 Exp $";
#endif


#include <sys/types.h> 
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <strings.h>
#include <sys/mount.h>
#include <a.out.h>

#include "prospect.h"
#include "malloc_debug.h"
#include "f_sym_name.h"

/*
 * Reuse some unused fields in SYMENT for:
 * The offset to the Next Global defining symbol
 * Flag to mark symbol as being local
 * Flag to mark symbol as being of duplicate value
*/
#define OffsetToNextGlobalSymb  arg_reloc
#define LocalSymbolFlag         secondary_def
#define DupSymbolFlag           dup_common

#define ElfLocalSymbolFlag         st_other
#define ElfOffsetToNextGlobalSymb  st_size
#define ElfDupSymbolFlag           st_shndx

/*
 * File Imported Variables - live in prospect.c
 * However, EXAMPLE build is stand alone.
 */
#ifdef EXAMPLE
#define ferr printf
#define ferr_nt printf
#endif

/* long long types for 64 bit files in 32 bit world */
#ifdef __ia64__
#define PRIlx "lx"
#else
#define PRIlx "llx"
#endif

/*
 * Static scope vars
 */
static syms_key_t *Pre_pSk = NULL, *gCur_pSk = NULL;
static syms_key_t gZeroHt = {-1, NULL, 0, 0};

/*
 * Static file prototypes
 */
static int f_Read(syms_key_t *pSk, char *Buf, int Len, off_t Off);
static void elf_fsymattr32(syms_key_t *pSk, elf_hit_tbl_32_t *phtp,
        syms_attrb_t *Psa);
static void elf_fsymattr64(syms_key_t *pSk, elf_hit_tbl_64_t *phtp,
        syms_attrb_t *Psa);
static int elf_process32file(Elf *elfptr, syms_key_t *pSk, char *AoutName);
static int elf_process64file(Elf *elfptr, syms_key_t *pSk, char *AoutName);
static int ElfHitComp32(const void *value, const void *vhtp);
static int ElfHitCompGlobal32(const void *value, const void *vhtp);
static int ElfHitComp64(const void *value, const void *vhtp);
static int ElfHitCompGlobal64(const void *value, const void *vhtp);
static int elfnumcomp32(const void *PhtpA, const void *PhtpB);
static int elfnumcomp64(const void *PhtpA, const void *PhtpB);
static void elf_psyment32(Pelf_hit_tbl_32_t ht, syms_key_t *pSk);
static void elf_psyment64(Pelf_hit_tbl_64_t ht, syms_key_t *pSk);


/*
 * f_sym_name read file operation
 */
static int
f_Read(syms_key_t *pSk, char *Buf, int Len, off_t Off)
{
    if (pSk == NULL || *pSk->sk_Path == 0) 
    {
        return(-1);
    }
    /* If different file than last time... */
    if (pSk != Pre_pSk)
    {
        if (Pre_pSk && Pre_pSk->sk_fd > 2)
        {
            close(Pre_pSk->sk_fd);
            Pre_pSk->sk_fd = -1;
        }
    }

    /* Check if file needs opening */
    if (pSk->sk_fd <= 2)
    {
        if  ((pSk->sk_fd = open(pSk->sk_Path, O_RDONLY)) < 0) 
        { 
            return(-1);
        }
    }

    if (lseek(pSk->sk_fd, Off, SEEK_SET) == -1)
    { 
        return(-1);
    }
    if (read(pSk->sk_fd, Buf, Len) != Len)
    {
        return(-1);
    }
    Pre_pSk = pSk;  /* for next time */
    gCur_pSk = pSk;

    return(0);

} /* f_Read() */


/*
 * ELF print symbol functions... 32 and 64 bit
 */
void
elf_print_sym_entry(void *ht, syms_key_t *pSk)
{
    if (pSk->sk_isElf32)
        elf_psyment32((elf_hit_tbl_32_t *)ht, pSk);
    else
        elf_psyment64((elf_hit_tbl_64_t *)ht, pSk);
    ferr_nt("\n");
}

static void
elf_psyment32(elf_hit_tbl_32_t *ht, syms_key_t *pSk)
{
    char *st = pSk->sk_strings;

    ferr_nt("0x%16lx", ht->elf_ht_ls->st_value + pSk->sk_exec_tmem);
    switch(ELF32_ST_TYPE(ht->elf_ht_ls->st_info))
    {
    case  STT_PARISC_MILLICODE: ferr_nt(" STT_PA_MILLI "); break;
    case  STT_FUNC:      ferr_nt(" STT_FUNC     ");     break;
    case  STT_SECTION:   ferr_nt(" STT_SECTION  ");     break;
    case  STT_FILE:      ferr_nt(" STT_FILE     ");     break;
    case  STT_OBJECT:    ferr_nt(" STT_OBJECT   ");     break;
    case  STT_NOTYPE:    ferr_nt(" STT_NOTYPE   ");     break;
    default: 
        ferr_nt(" STT_%4x", ELF32_ST_TYPE(ht->elf_ht_ls->st_info));
        break;
    }
    switch(ELF32_ST_BIND(ht->elf_ht_ls->st_info))
    {
    case    STB_WEAK:   ferr_nt(" STB_WEAK    ");  break;
    case    STB_GLOBAL: ferr_nt(" STB_GLOBAL  ");  break;
    case    STB_LOCAL:  ferr_nt(" STB_LOCAL   ");  break;
    default: 
        ferr_nt(" STB_%4x", ELF32_ST_BIND(ht->elf_ht_ls->st_info));
        break;
    }
    if (ht->elf_ht_ls->ElfLocalSymbolFlag) ferr_nt(" L");
    else                                   ferr_nt(" G");
    if (ht->elf_ht_ls->ElfDupSymbolFlag)   ferr_nt(" D");
    else                                   ferr_nt("  ");
    ferr_nt("%2d", ht->elf_ht_ls->ElfOffsetToNextGlobalSymb);
    ferr_nt(" %-20s ", (st+ht->elf_ht_ls->st_name));

    return;
}

static void
elf_psyment64(elf_hit_tbl_64_t *ht, syms_key_t *pSk)
{
    char *st = pSk->sk_strings;

    ferr_nt("0x%16"PRIlx, ht->elf_ht_ls->st_value + pSk->sk_exec_tmem);
    switch(ELF64_ST_TYPE(ht->elf_ht_ls->st_info))
    {
    case  STT_PARISC_MILLICODE: ferr_nt(" STT_PA_MILLI "); break;
    case  STT_FUNC:      ferr_nt(" STT_FUNC     ");     break;
    case  STT_SECTION:   ferr_nt(" STT_SECTION  ");     break;
    case  STT_FILE:      ferr_nt(" STT_FILE     ");     break;
    case  STT_OBJECT:    ferr_nt(" STT_OBJECT   ");     break;
    case  STT_NOTYPE:    ferr_nt(" STT_NOTYPE   ");     break;
    default: 
        ferr_nt(" STT_%4x", ELF64_ST_TYPE(ht->elf_ht_ls->st_info));
        break;
    }
    switch(ELF64_ST_BIND(ht->elf_ht_ls->st_info))
    {
    case    STB_WEAK:   ferr_nt(" STB_WEAK    ");  break;
    case    STB_GLOBAL: ferr_nt(" STB_GLOBAL  ");  break;
    case    STB_LOCAL:  ferr_nt(" STB_LOCAL   ");  break;
    default: 
        ferr_nt(" STB_%4x", ELF64_ST_BIND(ht->elf_ht_ls->st_info));
        break;
    }
    if (ht->elf_ht_ls->ElfLocalSymbolFlag)  ferr_nt(" L");
    else                                    ferr_nt(" G"); 
    if (ht->elf_ht_ls->ElfDupSymbolFlag)    ferr_nt(" D");
    else                                    ferr_nt("  ");
    ferr_nt("%2ld", (unsigned long)ht->elf_ht_ls->ElfOffsetToNextGlobalSymb);
    ferr_nt(" %-20s ", (st+ht->elf_ht_ls->st_name));

    return;
}

/*
 * Search for closest local or global symbol and return pointer
 * to symbol attributes.
 */
void*
f_sym_name(syms_key_t *pSk, char *Value)
{
    if (pSk == NULL) 
    {
        return((void*)NULL);
    }

    #ifdef DEBUG5
    printf("f_sym_name(): searching for 0x%lx in: ",Value); 
    #endif

    if (pSk->sk_sym_count != 0)
    {
        if (pSk->sk_isElf32)
        {
            Pelf_hit_tbl_32_t phtp;
            #ifdef DEBUG5
            printf ("32bit elf file...\n");
            #endif
            phtp = (Pelf_hit_tbl_32_t)
                bsearch(Value, pSk->sk_ht_tbl, pSk->sk_sym_count, 
                        cElf32HT_SZ, ElfHitComp32);
            return((void*)phtp);
        }
        else
        {
            Pelf_hit_tbl_64_t phtp;
            #ifdef DEBUG5
            printf ("64bit elf file...\n");
            #endif
            phtp = (Pelf_hit_tbl_64_t)
                bsearch(Value, pSk->sk_ht_tbl, pSk->sk_sym_count, 
                        cElf64HT_SZ, ElfHitComp64);
            return((void*)phtp);
        }
    }
    else
    {
        return(NULL);
    }
} /* f_sym_name() */

/*
 * Find closest global symbol to the passed in values,
 * and return the pointer to the symbol attributes. 
 */
void*
f_sym_name_global(syms_key_t *pSk, char *Value)
{
    if (pSk == NULL) 
    {
        return((void*)NULL);
    }

    #ifdef DEBUG5
    printf("f_sym_name_global(): searching for 0x%lx in: ",Value); 
    #endif

    if (pSk->sk_sym_count != 0)
    {
        if (pSk->sk_isElf32)
        {
            Pelf_hit_tbl_32_t phtp;
            #ifdef DEBUG5
            printf ("32bit elf file...\n");
            #endif
            phtp = (Pelf_hit_tbl_32_t)
                bsearch(Value, pSk->sk_ht_tbl, pSk->sk_sym_count, 
                        cElf32HT_SZ, ElfHitCompGlobal32);
            return((void*)phtp);
        }
        else
        {
            Pelf_hit_tbl_64_t phtp;
            #ifdef DEBUG5
            printf ("64bit elf file...\n");
            #endif
            phtp = (Pelf_hit_tbl_64_t)
                bsearch(Value, pSk->sk_ht_tbl, pSk->sk_sym_count, 
                        cElf64HT_SZ, ElfHitCompGlobal64);
            return((void*)phtp);
        }
    }
    else
    {
        return(NULL);
    }
} /* f_sym_name_global() */

/* 
 * Decode the _next_ symbol into common structure - elf version,
 * sensitive to 32- and 64-bit elf. 
 * Returns this symbol attributes if this is the last symbol.
 */
void
f_sym_attrb_next(syms_key_t *pSk, void *phtp, syms_attrb_t *Psa)
{ 
    if (!pSk->sk_ElfFlag || phtp==NULL) {
        /* catch this first - Symbol not in table */
        Psa->sa_value =  0x0;
        Psa->sa_name  = "Unknown-name";
        Psa->sa_scope = "Unknown-scope";
        Psa->sa_type  = "Unknown-type";
        return;
    }

    /* do it for 32 or 64 bits */
    if (pSk->sk_isElf32) {
        elf_hit_tbl_32_t *ht=(elf_hit_tbl_32_t*)phtp;

        /* if this is not the last symbol, increment ptr */
        if (ht->elf_ht_ls->ElfOffsetToNextGlobalSymb != 0) {
            /* advance to next non-dup symbol */
            while (ht->elf_ht_ls->ElfOffsetToNextGlobalSymb) {
                ht++;
                if (ht->elf_ht_ls->ElfDupSymbolFlag==FALSE)
                    break;
            }
        }
        elf_fsymattr32(pSk, ht, Psa);
    }
    else {
        elf_hit_tbl_64_t *ht=(elf_hit_tbl_64_t*)phtp;

        if (ht->elf_ht_ls->ElfOffsetToNextGlobalSymb != 0) {
            while (ht->elf_ht_ls->ElfOffsetToNextGlobalSymb) {
                ht++;
                if (ht->elf_ht_ls->ElfDupSymbolFlag==FALSE)
                    break;
            }
        }
        elf_fsymattr64(pSk, ht, Psa);
    }

    return;
} /* f_sym_attrb_next() */

/* 
 * Decode symbol into common structure,
 * sensitive to 32- and 64-bit elf.
 */
void
f_sym_attrb(syms_key_t *pSk, void *phtp, syms_attrb_t *Psa)
{
    if (phtp == NULL)
    {
        /* catch this first - Symbol not in table */
        Psa->sa_value =  0x0;
        Psa->sa_name  = "Unknown-name";
        Psa->sa_scope = "Unknown-scope";
        Psa->sa_type  = "Unknown-type";
        return;
    }

    /* only elf supported */
    if (!pSk->sk_ElfFlag) 
        return;

    /* 32 or 64 bit */
    if (pSk->sk_isElf32)
        elf_fsymattr32(pSk, phtp, Psa);
    else
        elf_fsymattr64(pSk, phtp, Psa);

    return;
}

static void
elf_fsymattr32(syms_key_t *pSk, elf_hit_tbl_32_t *phtp, syms_attrb_t *Psa)
{
    /* 
     * Value of last symbols searched
     * (already NULL terminated string)
     */
    Psa->sa_value = phtp->elf_ht_ls->st_value;
    Psa->sa_name  = (char*)((unsigned long)pSk->sk_strings + 
                            (unsigned long)phtp->elf_ht_ls->st_name);
    switch(ELF32_ST_TYPE(phtp->elf_ht_ls->st_info))
    {
    case  STT_PARISC_MILLICODE: Psa->sa_type = "MILLI";     break;
    case  STT_NOTYPE:       Psa->sa_type = "NOTYPE";    break;
    case  STT_OBJECT:       Psa->sa_type = "OBJECT";    break;
    case  STT_FUNC:         Psa->sa_type = "FUNC";      break;
    case  STT_SECTION:      Psa->sa_type = "SECTION";   break;
    case  STT_FILE:         Psa->sa_type = "FILE";      break;
    default:                Psa->sa_type = "UNKNOWN";   break; 
    }
    switch(ELF32_ST_BIND(phtp->elf_ht_ls->st_info))
    {
    case  STB_WEAK:         Psa->sa_scope = "WEAK";     break;
    case  STB_GLOBAL:       Psa->sa_scope = "GLOBAL";   break;
    case  STB_LOCAL:
        if (phtp->elf_ht_ls->ElfLocalSymbolFlag == FALSE)
        {
            Psa->sa_scope = "STATIC";
        } else
        {
            Psa->sa_scope = "LOCAL";
        }
        break;
    default:                Psa->sa_scope = "UNKNOWN"; break;
    }
    return;
} /* elf_fsymattr32() */

static void
elf_fsymattr64(syms_key_t *pSk, elf_hit_tbl_64_t *phtp, syms_attrb_t *Psa)
{
    /* 
     * Value of last symbols searched
     * (already NULL terminated string)
     */
    Psa->sa_value = phtp->elf_ht_ls->st_value;
    Psa->sa_name  = (char*)((unsigned long)pSk->sk_strings + 
                            (unsigned long)phtp->elf_ht_ls->st_name);
    switch(ELF64_ST_TYPE(phtp->elf_ht_ls->st_info))
    {
    case  STT_PARISC_MILLICODE: Psa->sa_type = "MILLI";     break;
    case  STT_NOTYPE:       Psa->sa_type = "NOTYPE";    break;
    case  STT_OBJECT:       Psa->sa_type = "OBJECT";    break;
    case  STT_FUNC:         Psa->sa_type = "FUNC";      break;
    case  STT_SECTION:      Psa->sa_type = "SECTION";   break;
    case  STT_FILE:         Psa->sa_type = "FILE";      break;
    default:                Psa->sa_type = "UNKNOWN";   break; 
    }
    switch(ELF64_ST_BIND(phtp->elf_ht_ls->st_info))
    {
    case  STB_WEAK:         Psa->sa_scope = "WEAK";     break;
    case  STB_GLOBAL:       Psa->sa_scope = "GLOBAL";   break;
    case  STB_LOCAL:
        if (phtp->elf_ht_ls->ElfLocalSymbolFlag == FALSE)
        {
            Psa->sa_scope = "STATIC";
        } else
        {
            Psa->sa_scope = "LOCAL";
        }
        break;
    default:                Psa->sa_scope = "UNKNOWN";  break;
    }
    return;
} /* elf_fsymatt64() */


void
f_sym_name_free(syms_key_t *pSk)
{
    /* printf("f_sym_name_free  fd = %d, pSk=0x%x, Path='%s'\n", 
           pSk->sk_fd, pSk, pSk->sk_Path); */

    if (pSk)
    {
        /* close open symbols file if open */
        if (pSk->sk_fd > 2) close(pSk->sk_fd);

        /* Free old symbol table memory */
        if (pSk->sk_lesyms)   FREE(pSk->sk_lesyms);
        if (pSk->sk_ht_tbl)   FREE(pSk->sk_ht_tbl);
        if (pSk->sk_strings)  FREE(pSk->sk_strings);
        if (pSk->sk_Pdinode)  FREE(pSk->sk_Pdinode);
        if (pSk->sk_Pfs)      FREE(pSk->sk_Pfs);

        /* Initialize the control block */
        *pSk = gZeroHt;
    }
    return;

} /* f_sym_name_free() */


/* 
 * Open and allocate memory for symbol table search.
 */ 
int
f_sym_name_list(syms_key_t *pSk, char *AoutName)
{

    int      rval;
    Elf        *elfptr;
    Elf64_Ehdr *ehdr64;
    Elf32_Ehdr *ehdr32;
    struct stat StatBuf;

    static int been_here_done_that = 0;

    /* Check if libelf.a is in sync */
    if (been_here_done_that == 0) {
        /* 
         * Note: If this function 'elf_version()' does not get called 
         *       on hpux, the 64-bit files fail to read. 
         *       32-bit files don't need this.
         */
        if (elf_version(EV_CURRENT) == EV_NONE)
        {
#if defined(DEBUG1) || defined(DEBUG2) || defined(DEBUG3) || defined(DEBUG4) /* [ */
           /* library out of date */
           printf("Elf[%d]: libelf.a is out of date, may be an issue!?!\n",
               __LINE__);
#endif /* ] */
        }
        been_here_done_that = 1;
    }

#ifdef DEBUG1
    printf("Entering elf_f_sym_name_list(0x%lx, %s)\n",(unsigned long)pSk, 
        AoutName);
#endif
    /* Check if symbol table already obtained? */
    if (pSk->sk_Inum && pSk->sk_Dev) { return(0); }

    /* Initialize the control block */
    *pSk = gZeroHt;

    /* Open the new file */
    if  ((pSk->sk_fd = open(AoutName, O_RDONLY)) < 0) { return (-errno); }

    /* Get Inum and Dev */
    if (fstat(pSk->sk_fd, &StatBuf) < 0) { return (-errno); }

    /* Save the open for next time */
    pSk->sk_Path = AoutName;
    pSk->sk_Dev = StatBuf.st_dev;
    pSk->sk_Inum = StatBuf.st_ino;

    /*
     * Open the elf file 
     */
    elfptr = elf_begin(pSk->sk_fd, ELF_C_READ, (Elf *)0);
    if (elf_kind(elfptr) != ELF_K_ELF)
    {
        /* not elf binary nor shlib */
        #ifdef EXAMPLE
        printf("File:%s is not an executable nor shared library (archive?).\n", 
            AoutName);
        #endif
        elf_end(elfptr);
        close(pSk->sk_fd);
        pSk->sk_fd = -1; 
        return 0;
    }

    /*
     * Figure out if this is a 32-bit or
     * 64-bit elf file.
     */
    ehdr64 = elf64_getehdr(elfptr);
    if (!ehdr64) 
    {
        /* try for 32-bit ... */
        ehdr32 = elf32_getehdr(elfptr);
        if (!ehdr32) 
        {
            /* what kind of file is this ?? */
            elf_end(elfptr);
            close(pSk->sk_fd);
            pSk->sk_fd = -1; 
            rval = -67;
            f_sym_name_free(pSk);
        }
        else
        {
            /* 32-bit elf file - process */
            pSk->sk_isElf32 = TRUE;
            rval = elf_process32file(elfptr, pSk, AoutName); 
        }
    }
    else 
    {
        /* 64-bit elf file - process */
        pSk->sk_isElf32 = FALSE;
        rval = elf_process64file(elfptr, pSk, AoutName);
    }

    return rval;
} /* elf_f_sym_name_list() */


static int
elf_process32file(Elf *elfptr, syms_key_t *pSk, char *AoutName)
{
    int               ii;
    Pelf_hit_tbl_32_t  PHtp_dup;
    Pelf_hit_tbl_32_t  PHtp;
    Pelf_hit_tbl_32_t  PHtp_glb;

    Elf_Scn *escntmp, *escnsym, *escnstr;

    Elf32_Ehdr *ehdr;
    Elf32_Shdr *escnhdr=NULL;
    Elf32_Phdr *phdr;

    /*
     * Open header and start processing 
     */
    pSk->sk_lesyms = NULL;
    pSk->sk_strings = NULL;
    if ((ehdr = elf32_getehdr(elfptr)) != NULL)  
    {
        char *sptr = (char*)(&(ehdr->e_ident[EI_MAG0]));
        if (strncmp(sptr, ELFMAG, SELFMAG) != 0)
        {
            fprintf(stderr,"Elf[%d] '%s' -- bad ELF magic number='%s'\n", 
                __LINE__, AoutName, sptr);
            goto elf_error_rtn;
        }

#ifdef EXAMPLE /* [ */
        printf("Reading file '%s', ELF identification:\n", AoutName);
        printf("  e_ident[EI_CLASS]      = %d (", ehdr->e_ident[EI_CLASS]);
        switch(ehdr->e_ident[EI_CLASS]) 
        {
            case ELFCLASSNONE: printf("ELFCLASSNONE)\n"); break;
            case ELFCLASS32:   printf("ELFCLASS32)\n");   break;
            case ELFCLASS64:   printf("ELFCLASS64)\n");   break;
            default:           printf("Unknown!)\n");
        }
        printf("  e_ident[EI_OSABI]      = %d (", ehdr->e_ident[EI_OSABI]);
        switch(ehdr->e_ident[EI_OSABI]) 
        {
            case ELFOSABI_HPUX:       printf("ELFOSABI_HPUX)\n");       break;
            case ELFOSABI_STANDALONE: printf("ELFOSABI_STANDALONE)\n"); break;
            default:                  
                printf("Unknown 0x%x)\n", ehdr->e_ident[EI_OSABI]);
        }
        printf("  e_ident[EI_ABIVERSION] = %d (", ehdr->e_ident[EI_ABIVERSION]);
        switch(ehdr->e_ident[EI_ABIVERSION]) 
        {
            default:
                printf("Unknown 0x%x)\n", ehdr->e_ident[EI_ABIVERSION]);
        }
        if ((ehdr->e_ident[EI_CLASS] != ELFCLASS32) ||
            (
            (ehdr->e_ident[EI_OSABI] != ELFOSABI_SYSV) 
            &&
            (ehdr->e_ident[EI_OSABI] != ELFOSABI_HPUX) 
            )
           )
        {
            printf("  OOPS: Line_%d -- bad ident stuff\n", __LINE__);
            printf("  Continuing anyway.\n");
        }
        printf("\n'%s' 32-bit elf header is:\n",AoutName);
        printf("     Object File ID, e_ident[EI_CLASS] = %d\n",
               ehdr->e_ident[EI_CLASS]);
        printf("     Object File type, e_type = %d\n",ehdr->e_type);
        printf("     Machine type, e_machine = %d\n",ehdr->e_machine);
        printf("     Object File Version, e_version = %d\n",ehdr->e_version);
        printf("     Entry Point Address, e_entry = %lx\n",
            (unsigned long)ehdr->e_entry);
        printf("     Program Header Offset, e_phoff = %lx\n",
            (unsigned long)ehdr->e_phoff);
        printf("     Section Header Offset, e_shoff = %lx\n",
            (unsigned long)ehdr->e_shoff);
        printf("     Processor Specific Flags, e_flags = %x\n",ehdr->e_flags);
        printf("     Sizeof Elf Header, e_ehsize = %d\n",ehdr->e_ehsize);
        printf("     Sizeof Program Entry Header, e_phentsize = %d\n",
               ehdr->e_phentsize);
        printf("     Number of Program Entry Headers, e_phnum = %d\n",
               ehdr->e_phnum);
        printf("     Sizeof Section Entry Header, e_shentsize = %d\n",
               ehdr->e_shentsize);
        printf("     Number of Section Entry Headers, e_shnum = %d\n",
               ehdr->e_shnum);
        printf("     Section Name String Table Ndx, e_shstrndx = %d\n",
               ehdr->e_shstrndx);
#endif /* ] */

        /* Store flags.  */
        pSk->sk_exec_magic = (unsigned short)ehdr->e_flags;

        /*
         * Now, find the location of symbols in the elf file
         * according to the vaddr of the first loadable 
         * and executable program header.  But only do this for
         * exectuable or shared object files - others do not have
         * one of these.
         */
        pSk->sk_exec_tmem = 0;
        pSk->sk_exec_tfile = 0;
        if (ehdr->e_type == ET_DYN  || ehdr->e_type == ET_EXEC) {
            if (ehdr->e_type == ET_DYN) {
                pSk->sk_isReloc = TRUE;
                pSk->sk_isShlib = TRUE;
            }
            else {
                pSk->sk_isReloc = FALSE;
                pSk->sk_isShlib = FALSE;
            }
            if ((phdr = elf32_getphdr(elfptr)) != 0) {
                for (ii=0; ii<ehdr->e_phnum; ii++) {
                    if (phdr->p_type == PT_LOAD && (phdr->p_flags & PF_X)) {
                        /* found first loadable and executable header */
                        pSk->sk_exec_tmem = phdr->p_vaddr;
                        pSk->sk_exec_tfile = phdr->p_offset;
                        break;
                    }
                    phdr++;
                }
                if (ii==ehdr->e_phnum)
                    printf("\nElf[%d]: Loadable and executable header not "
                           "found for %s\n",
                            __LINE__, AoutName );
            }
            else {
                printf("\nElf[%d]: elf32_getphdr(elfptr) returned 0 for %s\n", 
                       __LINE__, AoutName);
            }
        }
        else {
            /* assume relocatable: usually .o files (kernel modules) */
            pSk->sk_isReloc = TRUE;
            pSk->sk_isShlib = FALSE;
        }

#ifdef EXAMPLE /* [ */
        if (ehdr->e_type == ET_DYN  || ehdr->e_type == ET_EXEC) {
            Elf32_Phdr *phdr;
            int i;
            printf("\nOffsets read in from header:\n");
            printf("sk_exec_tmem: 0x%lx   sk_exec_tfile: 0x%lx   "
                "sk_exec_dmem: 0x%lx\n",
                pSk->sk_exec_tmem, pSk->sk_exec_tfile, pSk->sk_exec_dmem);
            if ((phdr = elf32_getphdr(elfptr)) != 0) {
                printf("\nProgram header table:\n");
                for (i=0; i<ehdr->e_phnum; i++) {
                    printf("ph[%d]: p_vaddr=0x%lx   p_offset=0x%lx",
                            i+1, (unsigned long)phdr->p_vaddr, 
                            (unsigned long)phdr->p_offset);
                    printf("   p_type=");
                    switch (phdr->p_type) {
                        case PT_NULL:    printf("PT_NULL"); break;
                        case PT_LOAD:    printf("PT_LOAD"); break;
                        case PT_DYNAMIC: printf("PT_DYNAMIC"); break;
                        case PT_INTERP:  printf("PT_INTERP"); break;
                        case PT_NOTE:    printf("PT_NOTE"); break;
                        case PT_SHLIB:   printf("PT_SHLIB"); break;
                        case PT_PHDR:    printf("PT_PHDR"); break;
                        default: printf("HPUX");
                    }
                    printf("   p_flags=[");
                    if (phdr->p_flags & PF_X) printf(" X");
                    if (phdr->p_flags & PF_W) printf(" W");
                    if (phdr->p_flags & PF_R) printf(" R");
                    printf(" ]\n");
                    phdr++;
                }
                printf("\n");
            }
            else {
                printf("Elf[%d]: elf32_getphdr(elfptr) returned 0\n", __LINE__);
            }
        } 
        else {
            printf("\nNot an ET_EXEC or ET_DYN section, so no program header.");
            printf("\nOffsets set to zero.\n");
        }
#endif /* ] */

        /*
         * See if we have section SHT_SYMTAB   of    ELF_T_SYM type.
         * Look for SHT_SYMTAB first, remember if we see SHT_DYNSYM so we
         * can use it if SYMTAB is not there (e.g. stripped).  Big assumption:
         * these seem to occur at opposite ends of the list!
         */

        /* No known sections */
        escnsym = escnstr = (Elf_Scn *)NULL;

        /* Look near end for symbol table */
        pSk->sk_isDynamic = FALSE;
        for (ii = ehdr->e_shnum-1; ii > ehdr->e_shnum-10; ii--) {
            escntmp = elf_getscn(elfptr, ii);
            if (escntmp == NULL)
                printf("Elf[%d] Section invalid, err='%s'\n",
                    __LINE__, elf_errmsg(-1));
            escnhdr = elf32_getshdr(escntmp);
            if (escnhdr->sh_type == SHT_SYMTAB) {
                escnsym = escntmp; 
                #ifdef EXAMPLE
                printf("Elf[%d] Found syms in section(%d)\n", __LINE__, ii);
                #endif
                break; 
            }
        }
        /* Look near beginning for dynamic symbols, but keep
         * crusing through just in case
         */
        if (escnsym == NULL) {
            #ifdef EXAMPLE
            printf("'%s' -- no symtab; looking for dynsym\n", AoutName);
            #endif
            for (ii = 0; ii < ehdr->e_shnum; ii++) {
                escntmp = elf_getscn(elfptr, ii);
                escnhdr = elf32_getshdr(escntmp);
                if (escnhdr->sh_type == SHT_DYNSYM) {
                    /* Only use the dynamic section symbols if this is a 
                     * shared library - don't for executables.
                     */
                    if (ehdr->e_type == ET_DYN) {
                        escnsym = escntmp;
                        pSk->sk_isDynamic = TRUE;
                        #ifdef EXAMPLE
                        printf("Elf[%d] Found dynamic syms in section(%d)\n",
                            __LINE__, ii);
                        #endif
                        break;
                    }
                }
                if (escnhdr->sh_type == SHT_SYMTAB) {
                    escnsym = escntmp; 
                    #ifdef EXAMPLE
                    printf("Elf[%d] Found syms in section(%d)\n", __LINE__, ii);
                    #endif
                    break; 
                }
            }
        }
        if (escnsym == NULL) {
            #ifdef EXAMPLE
            printf("Elf[%d]: '%s' -- no symbols ?!?\n", 
                   __LINE__, AoutName);
            #endif
            goto elf_no_symbols_rtn;
        }
        if (escnhdr->sh_link <= 0) {
            fprintf(stderr,
                "Elf[%d]: '%s' -- no link to string table (USELESS?!)\n", 
                __LINE__, AoutName);
            goto elf_no_symbols_rtn;
        }
        escnstr = elf_getscn(elfptr, escnhdr->sh_link);
        escnhdr = elf32_getshdr(escnstr);
        if (escnhdr->sh_type != SHT_STRTAB) {
            fprintf(stderr,
                   "Elf[%d]: '%s' -- no link to string table\n", 
                   __LINE__, AoutName);
            goto elf_no_symbols_rtn;
        }

#ifdef EXAMPLE /* [ */
        /* print the string section header we got ... */
        printf("String section header is:\n");
        printf("     Section Name, sh_name = %d\n",escnhdr->sh_name);
        printf("     Section Type, sh_type = %d\n",escnhdr->sh_type);
        printf("     Section Flags, sh_flags = %x\n",escnhdr->sh_flags);
        printf("     Virtual Addr, sh_addr = %lx\n",
            (unsigned long)escnhdr->sh_addr);
        printf("     Offset in File, sh_offset = %lx\n",
            (unsigned long)escnhdr->sh_offset);
        printf("     Section Size, sh_size = %d\n",escnhdr->sh_size);
        printf("     Link to other Section, sh_link = %d\n",escnhdr->sh_link);
        printf("     Misc Info, sh_info = %d\n",escnhdr->sh_info);
        printf("     Address Align Bndry, sh_addralign = %x\n", 
               escnhdr->sh_addralign);
        printf("     Entry Size, sh_entsize = %d\n",escnhdr->sh_entsize);
        if (escnhdr->sh_entsize > 0)
            printf("     Number of Entries, sh_size/entsize = %d\n",
                   escnhdr->sh_size/escnhdr->sh_entsize);
        printf("\n");
#endif /* ] */

        /* Now read string table into mem */
        pSk->sk_strings = (char *)MALLOC(escnhdr->sh_size);

        if (f_Read(
                   pSk,
                   (char *)pSk->sk_strings, 
                   escnhdr->sh_size,
                   (off_t)escnhdr->sh_offset 
                  )
           )
        { goto elf_perror_rtn; }

        escnhdr = elf32_getshdr(escnsym);
        ii = 0;
        if (escnhdr->sh_entsize > 0)
            ii = escnhdr->sh_size/escnhdr->sh_entsize;

#ifdef EXAMPLE /* [ */
        /* print the symbol section header we got ... */
        printf("Symbol section header is:\n");
        printf("     Section Name, sh_name = %d\n",escnhdr->sh_name);
        printf("     Section Type, sh_type = %d\n",escnhdr->sh_type);
        printf("     Section Flags, sh_flags = %x\n",escnhdr->sh_flags);
        printf("     Virtual Addr, sh_addr = %lx\n",
            (unsigned long)escnhdr->sh_addr);
        printf("     Offset in File, sh_offset = %lx\n",
            (unsigned long)escnhdr->sh_offset);
        printf("     Section Size, sh_size = %d\n",escnhdr->sh_size);
        printf("     Link to other Section, sh_link = %d\n",escnhdr->sh_link);
        printf("     Misc Info, sh_info = %d\n",escnhdr->sh_info);
        printf("     Address Align Bndry, sh_addralign = %x\n",
            escnhdr->sh_addralign);
        printf("     Entry Size, sh_entsize = %d\n",escnhdr->sh_entsize);
        if (escnhdr->sh_entsize > 0)
            printf("     Number of Entries, sh_size/entsize = %d\n", ii);
#endif /* ] */

        if (ii > 0) {
            Elf32_Sym *e32sym, *e32sym_end;
            pSk->sk_lesyms = (void*)MALLOC(escnhdr->sh_size);
            if (f_Read(pSk, (char *)pSk->sk_lesyms, 
                      escnhdr->sh_size, (off_t)escnhdr->sh_offset) )
            { goto elf_perror_rtn; }

            /* Set the ELF flag ... so we know how to handle this pSk later */
            pSk->sk_ElfFlag = 1;
            pSk->sk_isElf32 = TRUE;

            /* Get memory for the hit table */
            pSk->sk_ht_tbl = MALLOC((ii+1) * cElf32HT_SZ);
            PHtp = (Pelf_hit_tbl_32_t)pSk->sk_ht_tbl;

            /* Build the hit table with pointers to symbol table */
            e32sym = (Elf32_Sym *)pSk->sk_lesyms;
            e32sym_end = e32sym + ii;
            ii = 0;
            while (++e32sym < e32sym_end) {  /* Skip first STN_UNDEF entry */
                /* Assume this is UNDEF */
                if (e32sym->st_value == 0) continue;
                e32sym->ElfLocalSymbolFlag =
                (ELF32_ST_BIND(e32sym->st_info) == STB_LOCAL)?TRUE:FALSE;
                if (ELF32_ST_TYPE(e32sym->st_info) == STT_FUNC) {
#ifdef DBGELF
                    printf("%20s  %6s %6s %16llx  %ld\n",
                           (pSk->sk_strings+e32sym->st_name), "FUNC",
                           e32sym->ElfLocalSymbolFlag?"LOCAL":"GLOBAL",
                           e32sym->st_value, e32sym->st_size);
#endif
                    /*
                     * Since this is a function entry point, override
                     * to be a global symbol so it shows up in the profiles.
                     */
                    e32sym->ElfLocalSymbolFlag = FALSE;
                    goto elf_good_symbol;
                } else if ((ELF32_ST_TYPE(e32sym->st_info) == STT_OBJECT)) {
#ifdef DBGELF
                    printf("%20s  %6s",
                           (pSk->sk_strings+e32sym->st_name), "OBJT");
                    (e32sym->ElfLocalSymbolFlag==TRUE)
                        ? printf(" %6s","LOCAL") 
                        : printf(" %6s","GLOBAL");
                    printf(" %16llx  %ld\n",
                           e32sym->st_value, e32sym->st_size);
#endif
                    /*
                     * Global objects are not necessarily data... they can
                     * be (especially when in code space) functions labels.
                     * We see sizes for data objects, while labels have a 
                     * st_size of zero.  This is not in a spec and thus not
                     * guaranteed.  Oh well.  The other way to do this is to
                     * filter based on the section.  The .bss, .data, .rodata,
                     * etc., will have data objects, while other sections should
                     * be func labels.  But section names can be created....
                     * and what happens when there's a true data object in .text
                     * We'll go with the st_size for now and see what happens.
                     */
                    if (e32sym->st_size) continue;
                    /*
                     * Objects in code space that are local are static
                     * labels within functions; these are the local labels
                     * that will show up in the disassembled profiles.
                     * So we leave the ElfLocalSymbolFlag as is.
                     */
                    goto elf_good_symbol;
                }
                continue;
elf_good_symbol:
                ii++;
                PHtp->elf_ht_ls = e32sym;
                PHtp++;

                /*
                 * Adjust symbol Value to be "text in pregion" relative
                 */
                e32sym->st_value -= pSk->sk_exec_tmem;
            }
            PHtp->elf_ht_ls = pSk->sk_lesyms; /* Mark end of table with UNDEF */
            pSk->sk_sym_count = ii; /* Remember count of symbols in our table */

            /*
             * Sort the hit table by symbol Value (actually only the pointers)
             * but do not touch the last UNDEF entry.
             */
            qsort(pSk->sk_ht_tbl, ii, cElf32HT_SZ, elfnumcomp32);

#ifdef DEBUG4
            printf("\n\nNow Mark Global symbols with preceeding symbols "
                   "of duplicate values\n");
#endif /* DEBUG4 */

            /* 
             * Now MARK symbol names of the same Value.
             * Point to last symbol in the table (UNDEF) and call this 
             * the last global.
             */
            PHtp_glb = PHtp; /* So this is the last (fake) global symbol */
            PHtp_glb->elf_ht_ls->ElfOffsetToNextGlobalSymb = 0;
            PHtp_dup = NULL;

            /*
             * Mark symbols with preceeding symbols of duplicate values (for
             * bsearch). Skip last UNDEF.  Mark local function entry points as 
             * global so they show up in the profiles.  Don't bother for data 
             * objects and stubs.
             */
            while (--PHtp >= (Pelf_hit_tbl_32_t)pSk->sk_ht_tbl)
            {
                /*
                 * Mark offset to next Global symbol
                 */
                PHtp->elf_ht_ls->ElfOffsetToNextGlobalSymb = PHtp_glb - PHtp;

                /*
                 * Check if we have a new Global symbol 
                 */
                if (PHtp->elf_ht_ls->ElfLocalSymbolFlag == FALSE)
                {
                    PHtp_glb = PHtp;
                }

                /*
                 * Mark symbols of duplicate Value 
                 */
                if (PHtp_dup)
                {
                    if (PHtp_dup->elf_ht_ls->st_value == 
                        PHtp->elf_ht_ls->st_value)
                    {
                        PHtp_dup->elf_ht_ls->ElfDupSymbolFlag = TRUE;
                    } else
                    {
                        PHtp_dup->elf_ht_ls->ElfDupSymbolFlag = FALSE;
                    }
#ifdef DEBUG4
                    elf_psyment32(PHtp_dup, pSk); ferr_nt("\n");
#endif /* DEBUG4 */
                }

                PHtp_dup = PHtp;

            } /* while(--Phtp...) */

            /* The last (first since backwards)  symbol can never be a dup */
            PHtp_dup->elf_ht_ls->ElfDupSymbolFlag = FALSE;

#if defined(DEBUG1) || defined(DEBUG2)
            printf("Final after marking dups, pSk->sk_sym_count = %d\n\n", 
                   pSk->sk_sym_count);
#endif

#ifdef DEBUG2
            {
            unsigned int last_addr = 0;
            for (PHtp = (Pelf_hit_tbl_32_t)pSk->sk_ht_tbl, ii = 0; 
                 ii < pSk->sk_sym_count; 
                 PHtp++, ii++) 
            {
                ferr_nt("%6d\n", PHtp->elf_ht_ls->st_value - last_addr); 
                last_addr = PHtp->elf_ht_ls->st_value;
                elf_print_sym_entry(pSk);
            }
            }
#endif /* DEBUG2 */

        } /* if (ii>0) */
    } /* if valid elf file */

elf_no_symbols_rtn:
    elf_end(elfptr);
    close(pSk->sk_fd);
    pSk->sk_fd = -1; 
    return(0);

elf_error_rtn:
    errno = 251;

elf_perror_rtn:
    elf_end(elfptr);
    f_sym_name_free(pSk);
    return(-errno);

} /* elf_process32file() */

static int
elf_process64file(Elf *elfptr, syms_key_t *pSk, char *AoutName)
{
    int               ii;
    Pelf_hit_tbl_64_t  PHtp_dup;
    Pelf_hit_tbl_64_t  PHtp;
    Pelf_hit_tbl_64_t  PHtp_glb;

    Elf_Scn *escntmp, *escnsym, *escnstr;

    Elf64_Ehdr *ehdr;
    Elf64_Shdr *escnhdr=NULL;
    Elf64_Phdr *phdr;

    /* Note, this had a loop using elf_next(elf) to look at all elf objects in
     * an archive ...  I don't think we need that since we're looking at one
     * file at a time, and none of these are static archives.
     */
    pSk->sk_lesyms = NULL;
    pSk->sk_strings = NULL;
    if ( (elf_kind(elfptr) == ELF_K_ELF) && /* ELF binary or shlib */
         ((ehdr = elf64_getehdr(elfptr)) != 0) ) 
    {
        char *sptr = (char*)(&(ehdr->e_ident[EI_MAG0]));
        if (strncmp(sptr, ELFMAG, SELFMAG) != 0)
        {
            fprintf(stderr,"Line_%d '%s' -- bad ELF magic number='%s'\n", 
                __LINE__, AoutName, sptr);
            goto elf_error_rtn;
        }

#ifdef EXAMPLE /* [ */
        printf("Reading file '%s', ELF identification:\n", AoutName);
        printf("  e_ident[EI_CLASS]      = %d (", ehdr->e_ident[EI_CLASS]);
        switch(ehdr->e_ident[EI_CLASS]) 
        {
            case ELFCLASSNONE: printf("ELFCLASSNONE)\n"); break;
            case ELFCLASS32:   printf("ELFCLASS32)\n");   break;
            case ELFCLASS64:   printf("ELFCLASS64)\n");   break;
            default:           printf("Unknown!)\n");
        }
        printf("  e_ident[EI_OSABI]      = %d (", ehdr->e_ident[EI_OSABI]);
        switch(ehdr->e_ident[EI_OSABI]) 
        {
            case ELFOSABI_HPUX:
                  printf("ELFOSABI_HPUX)\n");       break;
            case ELFOSABI_STANDALONE: 
                  printf("ELFOSABI_STANDALONE)\n"); break;
            default: 
                  printf("Unknown 0x%x)\n", ehdr->e_ident[EI_OSABI]);
        }
        printf("  e_ident[EI_ABIVERSION] = %d (", 
                ehdr->e_ident[EI_ABIVERSION]);
        switch(ehdr->e_ident[EI_ABIVERSION]) 
        {
            default:
                printf("Unknown 0x%x)\n", ehdr->e_ident[EI_ABIVERSION]);
        }

        if ( (ehdr->e_ident[EI_CLASS] != ELFCLASS64) 
             ||
             (
             (ehdr->e_ident[EI_OSABI] != ELFOSABI_SYSV) 
             &&
             (ehdr->e_ident[EI_OSABI] != ELFOSABI_HPUX) 
             )
           )
        {
            printf("  OOPS: Line_%d -- bad ident stuff\n", __LINE__);
            printf("  Continuing anyway.\n");
        }

        printf("\n'%s' 64-bit elf header is:\n",AoutName);
        printf("     Object File ID, e_ident[EI_CLASS] = %d\n",
                ehdr->e_ident[EI_CLASS]);
        printf("     Object File type, e_type = %d\n",ehdr->e_type);
        printf("     Machine type, e_machine = %d\n",ehdr->e_machine);
        printf("     Object File Version, e_version = %d\n",
                ehdr->e_version);
        printf("     Entry Point Address, e_entry = %"PRIlx"\n",
                ehdr->e_entry);
        printf("     Program Header Offset, e_phoff = %"PRIlx"\n",
                ehdr->e_phoff);
        printf("     Section Header Offset, e_shoff = %"PRIlx"\n",
                ehdr->e_shoff);
        printf("     Processor Specific Flags, e_flags = %x\n",
                ehdr->e_flags);
        printf("     Sizeof Elf Header, e_ehsize = %d\n",ehdr->e_ehsize);
        printf("     Sizeof Program Entry Header, e_phentsize = %d\n",
                ehdr->e_phentsize);
        printf("     Number of Program Entry Headers, e_phnum = %d\n",
                ehdr->e_phnum);
        printf("     Sizeof Section Entry Header, e_shentsize = %d\n",
                ehdr->e_shentsize);
        printf("     Number of Section Entry Headers, e_shnum = %d\n",
                ehdr->e_shnum);
        printf("     Section Name String Table Ndx, e_shstrndx = %d\n",
                ehdr->e_shstrndx);
#endif /* ] */

        /* Store flags. */
        pSk->sk_exec_magic = (unsigned short)ehdr->e_flags;

        /*
         * Now, find the location of symbols in the elf file
         * according to the vaddr of the first loadable 
         * and executable program header. Note that this
         * also has the offset of the section from the beginning
         * of the file.  We'll use this value, however, it
         * seems that this value is 0 for all files we're
         * interested in.
         */
        pSk->sk_exec_tmem = 0;
        pSk->sk_exec_tfile = 0;
        if (ehdr->e_type == ET_DYN  || ehdr->e_type == ET_EXEC) {
            if (ehdr->e_type == ET_DYN) {
                pSk->sk_isReloc = TRUE;
                pSk->sk_isShlib = TRUE;
            }
            else {
                pSk->sk_isReloc = FALSE;
                pSk->sk_isShlib = FALSE;
            }
            if ((phdr = elf64_getphdr(elfptr)) != 0) {
                for (ii=0; ii<ehdr->e_phnum; ii++) {
                    if (phdr->p_type == PT_LOAD && (phdr->p_flags & PF_X)) {
                        /* found first loadable and executable header */
                        pSk->sk_exec_tmem = phdr->p_vaddr;
                        pSk->sk_exec_tfile = phdr->p_offset;
                        break;
                    }
                    phdr++;
                }
                if (ii==ehdr->e_phnum)
                    printf("\nElf[%d]: Loadable and executable header "
                           "not found for %s\n",
                           __LINE__, AoutName );

            }
            else {
                printf("\nElf[%d]: elf64_getphdr(elfptr) returned 0 for %s\n",
                        __LINE__, AoutName);
            }
        }
        else {
            /* assume relocatable: usually .o files (kernel modules) */
            pSk->sk_isReloc = TRUE;
            pSk->sk_isShlib = FALSE;
        }

#ifdef EXAMPLE /* [ */
        if (ehdr->e_type == ET_DYN  || ehdr->e_type == ET_EXEC) {
            Elf64_Phdr *phdr;
            int i;
            printf("\nOffsets read in from header:\n");
            printf("sk_exec_tmem: 0x%lx   sk_exec_tfile: 0x%lx   "
                   "sk_exec_dmem: 0x%lx\n",
                   pSk->sk_exec_tmem, pSk->sk_exec_tfile, pSk->sk_exec_dmem);
            if ((phdr = elf64_getphdr(elfptr)) != 0) {
                printf("\nProgram header table:\n");
                for (i=0; i<ehdr->e_phnum; i++) {
                    printf("ph[%d]: p_vaddr=0x%lx   p_offset=0x%lx",
                            i+1, (unsigned long)phdr->p_vaddr,
                            (unsigned long)phdr->p_offset);
                    printf("   p_type=");
                    switch (phdr->p_type) {
                            case PT_NULL:    printf("PT_NULL"); break;
                            case PT_LOAD:    printf("PT_LOAD"); break;
                            case PT_DYNAMIC: printf("PT_DYNAMIC"); break;
                            case PT_INTERP:  printf("PT_INTERP"); break;
                            case PT_NOTE:    printf("PT_NOTE"); break;
                            case PT_SHLIB:   printf("PT_SHLIB"); break;
                            case PT_PHDR:    printf("PT_PHDR"); break;
                            default: printf("HPUX");
                    }
                    printf("   p_flags=[");
                    if (phdr->p_flags & PF_X) printf(" X");
                    if (phdr->p_flags & PF_W) printf(" W");
                    if (phdr->p_flags & PF_R) printf(" R");
                    printf(" ]\n");
                    phdr++;
                }
                printf("\n");
            }
            else {
                printf("\nelf64_getphdr(elfptr) returned 0\n");
            }
        }
        else {
            printf("\nNot an ET_EXEC or ET_DYN section, so no program header.");
            printf("\nOffsets set to zero.\n");
        }
#endif /* ] */

        /* process the file . . . */
        /* see if we have section SHT_SYMTAB   of    ELF_T_SYM type */

        /*
         * Look for SHT_SYMTAB first, remember if we see SHT_DYNSYM so we
         * can use it if SYMTAB is not there (e.g. stripped).
         */

        /* No known sections */
        escnsym = escnstr = (Elf_Scn *)NULL;

        /* Look near end for symbol table */
        pSk->sk_isDynamic = FALSE;
        for (ii = ehdr->e_shnum; ii > ehdr->e_shnum-10; ii--) {
            escntmp = elf_getscn(elfptr, ii);
            if (escntmp == NULL) {
#ifndef EXAMPLE
                if (gConf.bug_level)
#endif
                    ferr("Elf[%d] null section descriptor %d\n", 
                        __LINE__, ii);
                continue;
            }
            escnhdr = elf64_getshdr(escntmp);
            if (escnhdr == NULL) {
#ifndef EXAMPLE
                if (gConf.bug_level)
#endif
                    ferr("Elf[%d] null section header %d\n", 
                        __LINE__, ii);
                continue;
            }
            if (escnhdr->sh_type == SHT_SYMTAB) 
            {
                #ifdef EXAMPLE
                printf("Elf[%d] Found syms in section(%d)\n", __LINE__, ii);
                #endif
                escnsym = escntmp; 
                break; 
            }
        }
        if (escnsym == NULL) {
            #ifdef EXAMPLE
            printf("'%s' -- no symtab; looking for dynsym\n", AoutName);
            #endif
            /* Look near beginning for dynamic symbols */
            for (ii = 1; ii < ehdr->e_shnum; ii++) {
                escntmp = elf_getscn(elfptr, ii);
                escnhdr = elf64_getshdr(escntmp);
                if (escnhdr->sh_type == SHT_DYNSYM) 
                {
                    /* Only use the dynamic section symbols if this is a 
                     * shared library - don't for executables.
                     */
                    if (ehdr->e_type == ET_DYN) {
                        #ifdef EXAMPLE
                        printf("Elf[%d] Found dynsyms in section(%d)\n",
                            __LINE__, ii);
                        #endif
                        escnsym = escntmp; 
                        pSk->sk_isDynamic = TRUE;
                        break; 
                    }
                }
                if (escnhdr->sh_type == SHT_SYMTAB) {
                    escnsym = escntmp; 
                    #ifdef EXAMPLE
                    printf("Elf[%d] Found syms in section(%d)\n", 
                        __LINE__, ii);
                    #endif
                    break; 
                }
            }
        }
        if (escnsym == NULL) {
            #ifdef EXAMPLE
            printf("Elf[%d]: '%s' - no symbol section found.\n", __LINE__, 
                AoutName);
            #endif
            goto elf_no_symbols_rtn;
        }
        if (escnhdr->sh_link <= 0) {
            printf("Elf[%d]: '%s' -- no link to string table (USELESS?!)\n", 
                    __LINE__, AoutName);
            goto elf_no_symbols_rtn;
        }
        escnstr = elf_getscn(elfptr, escnhdr->sh_link);
        escnhdr = elf64_getshdr(escnstr);
        if (escnhdr->sh_type != SHT_STRTAB) {
            printf("'%s' -- expected a link to string table -- NOT!?!\n", 
                    AoutName);
            goto elf_no_symbols_rtn;
        }

#ifdef EXAMPLE /* [ */
        /* print the section header we got ... */
        printf("String section header is:\n");
        printf("     Section Name, sh_name = %d\n",escnhdr->sh_name);
        printf("     Section Type, sh_type = %d\n",escnhdr->sh_type);
        printf("     Section Flags, sh_flags = %x\n",
            (unsigned int)escnhdr->sh_flags);
        printf("     Virtual Addr, sh_addr = %lx\n",
            (unsigned long)escnhdr->sh_addr);
        printf("     Offset in File, sh_offset = %lx\n",
            (unsigned long)escnhdr->sh_offset);
        printf("     Section Size, sh_size = %d\n",
            (unsigned int)escnhdr->sh_size);
        printf("     Link to other Section, sh_link = %d\n",
             escnhdr->sh_link);
        printf("     Misc Info, sh_info = %d\n",escnhdr->sh_info);
        printf("     Address Align Bndry, sh_addralign = %x\n",
            (unsigned int)escnhdr->sh_addralign);
        printf("     Entry Size, sh_entsize = %d\n",
            (unsigned int)escnhdr->sh_entsize);
        if (escnhdr->sh_entsize > 0)
            printf("     Number of Entries, sh_size/entsize = %d\n",
                   (unsigned int)(escnhdr->sh_size/escnhdr->sh_entsize));
        printf("\n");
#endif /* ] */

        /* Now read string table into mem */
        pSk->sk_strings = (char *)MALLOC(escnhdr->sh_size);

        if (f_Read (
            pSk,
            (char *)pSk->sk_strings, 
            escnhdr->sh_size,
            (off_t)escnhdr->sh_offset ))
        { goto elf_perror_rtn; }

        escnhdr = elf64_getshdr(escnsym);
        ii = 0;
        if (escnhdr->sh_entsize > 0)
            ii = escnhdr->sh_size/escnhdr->sh_entsize;

#ifdef EXAMPLE /* [ */
        /* print the section header we got ... */
        printf("Symbol section header is:\n");
        printf("     Section Name, sh_name = %d\n",escnhdr->sh_name);
        printf("     Section Type, sh_type = %d\n",escnhdr->sh_type);
        printf("     Section Flags, sh_flags = %x\n",
            (unsigned int)escnhdr->sh_flags);
        printf("     Virtual Addr, sh_addr = %"PRIlx"\n",escnhdr->sh_addr);
        printf("     Offset in File, sh_offset = %"PRIlx"\n",
            escnhdr->sh_offset);
        printf("     Section Size, sh_size = %d\n",
            (unsigned int)escnhdr->sh_size);
        printf("     Link to other Section, sh_link = %d\n",
            escnhdr->sh_link);
        printf("     Misc Info, sh_info = %d\n",escnhdr->sh_info);
        printf("     Address Align Bndry, sh_addralign = %x\n",
            (unsigned int)escnhdr->sh_addralign);
        printf("     Entry Size, sh_entsize = %d\n",
            (unsigned int)escnhdr->sh_entsize);
        if (escnhdr->sh_entsize > 0)
            printf("     Number of Entries, sh_size/entsize = %d\n", ii);
#endif /* ] */

        /*
         * Process all symbols in file and file em in table.
         */
        if (ii > 0) {
            Elf64_Sym *e64sym, *e64sym_end;
            pSk->sk_lesyms = (void*)MALLOC(escnhdr->sh_size);
            if (f_Read (pSk, (char *)pSk->sk_lesyms, 
                    escnhdr->sh_size, (off_t)escnhdr->sh_offset) )
            { goto elf_perror_rtn; }

            /* Set the ELF flag ... so we know how to handle this pSk */
            pSk->sk_ElfFlag = 1;
            pSk->sk_isElf32 = FALSE;

            /* Get memory for the hit table */
            pSk->sk_ht_tbl = MALLOC((ii+1) * cElf64HT_SZ);
            PHtp = (Pelf_hit_tbl_64_t)pSk->sk_ht_tbl;

            /* Build the hit table with pointers to symbol table */
            e64sym = (Elf64_Sym *)pSk->sk_lesyms;
            e64sym_end = e64sym + ii;
            ii = 0;
            while (++e64sym < e64sym_end) { /* Skip first STN_UNDEF entry */
                /* Assume this is UNDEF */
                if (e64sym->st_value == 0) continue;
                e64sym->ElfLocalSymbolFlag =
                    (ELF64_ST_BIND(e64sym->st_info) == 
                        STB_LOCAL)?TRUE:FALSE;
                if (ELF64_ST_TYPE(e64sym->st_info) == STT_FUNC) {
#ifdef DBGELF
                  /*if (
                      (! strcmp("msghdr_to_msghdr_32",
                        (pSk->sk_strings+e64sym->st_name))) ||
                      (! strcmp("r1",
                        (pSk->sk_strings+e64sym->st_name))) ||
                      (! strcmp("sqrt",
                        (pSk->sk_strings+e64sym->st_name))) )*/
                  printf("%20s  %6s %6s %16llx  %ld\n",
                    (pSk->sk_strings+e64sym->st_name), "FUNC",
                    e64sym->ElfLocalSymbolFlag?"LOCAL":"GLOBAL",
                    e64sym->st_value, e64sym->st_size);
#endif
                    /*
                     * Since this is a function entry point, override
                     * to be a global symbol so it shows up in the profiles.
                     */
                    e64sym->ElfLocalSymbolFlag = FALSE;
                    goto elf_good_symbol;
                } else if ((ELF64_ST_TYPE(e64sym->st_info) == STT_OBJECT)) {
#ifdef DBGELF
                  printf("%20s  %6s",
                    (pSk->sk_strings+e64sym->st_name), "OBJT");
                  (e64sym->ElfLocalSymbolFlag==TRUE)
                       ? printf(" %6s","LOCAL") 
                       : printf(" %6s","GLOBAL");
                  printf(" %16llx  %ld\n",
                    e64sym->st_value, e64sym->st_size);
#endif
                    /*
                     * Global objects are not necessarily data... they can
                     * be (especially when in code space) functions labels.
                     * We see sizes for data objects, while labels have a 
                     * st_size of zero.  This is not in a spec and thus not
                     * guaranteed.  Oh well.  The other way to do this is to
                     * filter based on the section.  The .bss, .data, 
                     * .rodata,  etc., will have data objects, while other
                     * sections should be func labels.  But section names 
                     * can be created....  and what happens when there's a 
                     * true data object in .text?  We'll go with the 
                     * st_size for now and see what happens.
                     */
                    if (e64sym->st_size) continue;
                    /*
                     * Objects in code space that are local are static
                     * labels within functions; these are the local labels
                     * that will show up in the disassembled profiles.
                     * So we leave the ElfLocalSymbolFlag as is.
                     */
                    goto elf_good_symbol;
                }
                continue;
elf_good_symbol:
                ii++;
                PHtp->elf_ht_ls = e64sym;
                PHtp++;

                /*
                 * Adjust symbol Value to be "text in pregion" relative
                 */
                e64sym->st_value -= pSk->sk_exec_tmem;
            }

            /* Mark end of table with UNDEF */
            PHtp->elf_ht_ls = pSk->sk_lesyms;
            /* Remember count of symbols in our table */
            pSk->sk_sym_count = ii;          

            /*
             * Sort the hit table by symbol Value (actually only the 
             * pointers) but do not touch the last UNDEF entry.
             */
            qsort(pSk->sk_ht_tbl, ii, cElf64HT_SZ, elfnumcomp64);

            /*
             * Now MARK symbol names of the same Value
             */
#ifdef DEBUG4
            printf("\nNow mark global symbols with preceeding symbols "
                   "of duplicate values\n");
#endif /* DEBUG4 */

            /* Point to last symbol in the table (UNDEF) and call this 
             * the last global 
             */
            PHtp_glb = PHtp; /* So this is the last (fake) global symbol */
            PHtp_glb->elf_ht_ls->ElfOffsetToNextGlobalSymb = 0;
            PHtp_dup = NULL;

            /*
             * Mark symbols with preceeding symbols of duplicate values
             * (for bsearch). Skip last UNDEF.  Mark local function entry 
             * points as global so they show up in the profiles.  Don't 
             * bother for data objects and stubs.
             */
            while (--PHtp >= (Pelf_hit_tbl_64_t)pSk->sk_ht_tbl)
            {
                /*
                 * Mark offset to next Global symbol
                 */
                PHtp->elf_ht_ls->ElfOffsetToNextGlobalSymb = 
                    PHtp_glb - PHtp;

                /*
                 * Check if we have a new Global symbol 
                 */
                if (PHtp->elf_ht_ls->ElfLocalSymbolFlag == FALSE)
                {
                    PHtp_glb = PHtp;
                }

                /*
                 * Mark symbols of duplicate Value 
                 */
                if (PHtp_dup)
                {
                    if (PHtp_dup->elf_ht_ls->st_value == 
                            PHtp->elf_ht_ls->st_value)
                    {
                        PHtp_dup->elf_ht_ls->ElfDupSymbolFlag = TRUE;
                    } else
                    {
                        PHtp_dup->elf_ht_ls->ElfDupSymbolFlag = FALSE;
                    }
#ifdef DEBUG4
                    elf_psyment64(PHtp_dup, pSk); ferr_nt("\n");
#endif /* DEBUG4 */
                }

                PHtp_dup = PHtp;

            } /* while(--Phtp...) */

            /* The last (first since backwards) symbol can never be a dup */
            PHtp_dup->elf_ht_ls->ElfDupSymbolFlag = FALSE;

#if defined(DEBUG1) || defined(DEBUG2) || defined(DEBUG4)
            printf("Final after removal of dups, pSk->sk_sym_count = %d\n\n",
                   pSk->sk_sym_count);
#endif

#ifdef DEBUG2
            {
                unsigned int last_addr = 0;
                for (PHtp = (Pelf_hit_tbl_64_t)pSk->sk_ht_tbl, ii = 0; 
                     ii < pSk->sk_sym_count; 
                     PHtp++, ii++
                    )
                {
                        ferr_nt("%6d\n", PHtp->elf_ht_ls->st_value - last_addr); 
                        last_addr = PHtp->elf_ht_ls->st_value;
                        elf_print_sym_entry(pSk);
                }
            }
#endif/* DEBUG2 */

        } /* if (ii>0) */
    } /* if elf binary or shlib */

elf_no_symbols_rtn:
    elf_end(elfptr);
    close(pSk->sk_fd);
    pSk->sk_fd = -1; 
    return(0);

elf_error_rtn:
    errno = 251;

elf_perror_rtn:
    elf_end(elfptr);
    f_sym_name_free(pSk);
    return(-errno);

} /* elf_process64file() */


/*
 * Elf compare routine for bsearch  32-bit version
 * We need different 32 and 64 bit versions because we 
 * have to pluck out the value out of the structure.
 * See elf.h....
 */
static int
ElfHitComp32(const void *value, const void *vhtp)
{
    Pelf_hit_tbl_32_t  PHtp;
    unsigned long      Value, CurVal;
    int              DupSymbolFlag;

    /* Get hit table pointer and test value */
    PHtp = (Pelf_hit_tbl_32_t)vhtp;
    Value = (unsigned long)value; 
    /* Get the current value of the symbol */
    CurVal = (unsigned long)PHtp->elf_ht_ls->st_value;
    /* Note this is duplicate symbol and how far to next Global symbol */
    DupSymbolFlag = PHtp->elf_ht_ls->ElfDupSymbolFlag;

    /* Check if Value is less than test symbol value, we're done */
    if (Value < CurVal) { return(-1); }

    /* Check if we're dead on */
    if (Value == CurVal) { goto returnmatch; }

    /* Advance to NEXT non-dup valued symbol */
    do
    {
        /* Check if this is the last symbol in table */
        if (PHtp->elf_ht_ls->ElfOffsetToNextGlobalSymb == 0)
        {
            goto returnmatch;
        }
        /* Advance to next symbol and test not a duplicate value */
    } while ((++PHtp)->elf_ht_ls->ElfDupSymbolFlag == TRUE);

    /* Check if value is between PHtp and PHtp+1 */
    if (Value < PHtp->elf_ht_ls->st_value) { goto returnmatch; }

    /* Must be equal or greater than PHtp+1 */
    return(1);

returnmatch:
    if (DupSymbolFlag) { return(-1); }
    return(0);

} /* ElfHitComp() */

/*
 * Compare Global symbols only routine for bsearch  32-bit version
 */
static int
ElfHitCompGlobal32(const void *value, const void *vhtp)
{
    Pelf_hit_tbl_32_t PHtp;
    unsigned long     Value, CurVal;
    int               DupSymbolFlag;
    int               LocalSymbolFlag;

    /* Get hit table pointer and test value (Well C is not quite done) */
    PHtp = (Pelf_hit_tbl_32_t)vhtp;
    Value = (unsigned long)value;
    /* Get the current value of the symbol */
    CurVal = (unsigned long)PHtp->elf_ht_ls->st_value;
    /* Note this is duplicate symbol and how far to next Global symbol */
    DupSymbolFlag = PHtp->elf_ht_ls->ElfDupSymbolFlag;
    LocalSymbolFlag = PHtp->elf_ht_ls->ElfLocalSymbolFlag;

    /* Check if Value is less than test symbol value, were done.  */
    if (Value < CurVal)  { return(-1); }

    /* Check if we're dead on */
    if (Value == CurVal) { goto returnmatch; }

    /* Advance to NEXT Global symbol that is not a duplicate */
    for(;;)
    {
        /* Check if this is the last symbol */
        if (PHtp->elf_ht_ls->ElfOffsetToNextGlobalSymb == 0)
        {
            goto returnmatch;
        }
        /* Advance to NEXT Global symbol.  */
        PHtp += PHtp->elf_ht_ls->ElfOffsetToNextGlobalSymb;

        if (PHtp->elf_ht_ls->ElfDupSymbolFlag == FALSE) break;
    }

    /* Check if value is between PHtp and PHtp+1 */
    if (Value < PHtp->elf_ht_ls->st_value) { goto returnmatch; }

    /* Must be equal or greater than PHtp+1 */
    return(1);

returnmatch:
    if (DupSymbolFlag || LocalSymbolFlag) { return(-1); }
    return(0);
} /* ElfHitCompGlobal32() */

static int
ElfHitComp64(const void *value, const void *vhtp)
{
    Pelf_hit_tbl_64_t  PHtp;
    unsigned long      Value, CurVal;
    int                DupSymbolFlag;

    /* Get hit table pointer and test value */
    PHtp = (Pelf_hit_tbl_64_t)vhtp;
    Value = (unsigned long)value; 
    /* Get the current value of the symbol */
    CurVal = PHtp->elf_ht_ls->st_value;
    /* Note this is duplicate symbol and how far to next Global symbol */
    DupSymbolFlag = PHtp->elf_ht_ls->ElfDupSymbolFlag;

    /* Check if Value is less than test symbol value, we're done */
    if (Value < CurVal) { return(-1); }

    /* Check if we're dead on */
    if (Value == CurVal) { goto returnmatch; }

    /* Advance to NEXT non-dup valued symbol */
    do
    {
        /* Check if this is the last symbol in table */
        if (PHtp->elf_ht_ls->ElfOffsetToNextGlobalSymb == 0)
        {
            goto returnmatch;
        }
        /* Advance to next symbol and test not a duplicate value */
    } while ((++PHtp)->elf_ht_ls->ElfDupSymbolFlag == TRUE);

    /* Check if value is between PHtp and PHtp+1 */
    if (Value < PHtp->elf_ht_ls->st_value) { goto returnmatch; }

    /* Must be equal or greater than PHtp+1 */
    return(1);

returnmatch:
    if (DupSymbolFlag) { return(-1); }
    return(0);

} /* ElfHitComp64() */

static int
ElfHitCompGlobal64(const void *value, const void *vhtp)
{
    Pelf_hit_tbl_64_t  PHtp;
    unsigned long      Value, CurVal;
    int                DupSymbolFlag;
    int                LocalSymbolFlag;

    /* Get hit table pointer and test value (Well C is not quite done) */
    PHtp = (Pelf_hit_tbl_64_t)vhtp;
    Value = (unsigned long)value;
    /* Get the current value of the symbol */
    CurVal = PHtp->elf_ht_ls->st_value;
    /* Note this is duplicate symbol and how far to next Global symbol */
    DupSymbolFlag = PHtp->elf_ht_ls->ElfDupSymbolFlag;
    LocalSymbolFlag = PHtp->elf_ht_ls->ElfLocalSymbolFlag;

    /* Check if Value is less than test symbol value, were done.  */
    if (Value < CurVal)  { return(-1); }

    /* Check if we're dead on */
    if (Value == CurVal) { goto returnmatch; }

    /* Advance to NEXT Global symbol that is not a duplicate */
    for(;;)
    {
        /* Check if this is the last symbol */
        if (PHtp->elf_ht_ls->ElfOffsetToNextGlobalSymb == 0)
        {
            goto returnmatch;
        }
        /* Advance to NEXT Global symbol.  */
        PHtp += PHtp->elf_ht_ls->ElfOffsetToNextGlobalSymb;

        if (PHtp->elf_ht_ls->ElfDupSymbolFlag == FALSE) break;
    }

    /* Check if value is between PHtp and PHtp+1 */
    if (Value < PHtp->elf_ht_ls->st_value) { goto returnmatch; }

    /* Must be equal or greater than PHtp+1 */
    return(1);

returnmatch:
    if (DupSymbolFlag || LocalSymbolFlag) { return(-1); }
    return(0);
} /* ElfHitCompGlobal64() */

/*
 * Compare routine for qsort to sort the symbol table.
 * 32-bit version.
 */
static int
elfnumcomp32(const void *PhtpA, const void *PhtpB)
{
    unsigned int  A, B;
    Pelf_hit_tbl_32_t PHtpA, PHtpB;

    PHtpA = (Pelf_hit_tbl_32_t) PhtpA;
    PHtpB = (Pelf_hit_tbl_32_t) PhtpB;

    A = PHtpA->elf_ht_ls->st_value;
    B = PHtpB->elf_ht_ls->st_value;

    if (A > B) return( 1);
    else if (A < B) return(-1);
    else /* (A == B) */
        if (ELF32_ST_BIND(PHtpA->elf_ht_ls->st_info) ==
            ELF32_ST_BIND(PHtpB->elf_ht_ls->st_info) )
        {
            /* Return in alphabetical order */
            return
            (strcmp(
                    &gCur_pSk->sk_strings[PHtpB->elf_ht_ls->st_name] ,
                    &gCur_pSk->sk_strings[PHtpA->elf_ht_ls->st_name]
                   )
            );
        }

    /* One Local and one Global symbol, return the Global the lesser value */
    if (ELF32_ST_BIND(PHtpA->elf_ht_ls->st_info) == STB_LOCAL) 
        return( 1);
    else
        return(-1);
} /* elfnumcomp32() */

/*
 * Compare routine for qsort to sort the symbol table.
 * 64-bit version.
 */
static int
elfnumcomp64(const void *PhtpA, const void *PhtpB)
{
    unsigned long            A, B;
    Pelf_hit_tbl_64_t PHtpA, PHtpB;

    PHtpA = (Pelf_hit_tbl_64_t) PhtpA;
    PHtpB = (Pelf_hit_tbl_64_t) PhtpB;

    A = PHtpA->elf_ht_ls->st_value;
    B = PHtpB->elf_ht_ls->st_value;

    if (A > B) return( 1);
    else if (A < B) return(-1);
    else /* (A == B) */
        if (ELF64_ST_BIND(PHtpA->elf_ht_ls->st_info) ==
            ELF64_ST_BIND(PHtpB->elf_ht_ls->st_info) )
        {
            /* Return in alphabetical order */
            return
            (strcmp(
                    &gCur_pSk->sk_strings[PHtpB->elf_ht_ls->st_name] ,
                    &gCur_pSk->sk_strings[PHtpA->elf_ht_ls->st_name]
                   )
            );
        }
    /* One Local and one Global symbol, return the Global the lesser value */
    if (ELF64_ST_BIND(PHtpA->elf_ht_ls->st_info) == STB_LOCAL) 
        return( 1);
    else
        return(-1);
} /* elfnumcomp64() */


#ifdef EXAMPLE /* Example test program [ */

/* print Value of asc symbols passed as parameters */
int
main(int argc, char *argv[])
{
    int ii, ret;
    char *Value;
    char *Psym_attrb;
    static syms_key_t f_sym_key;
    static syms_attrb_t syms_attrb;

    if (argc < 2) 
    {
        printf("Usage:  %s filename [symval1 syval2 symval...]\n", argv[0]);
        exit(1);
    }

    /* initialize the search table find symbol for that number */
    if ((ret = f_sym_name_list(&f_sym_key, argv[1])))
    {
        printf("'%s' is unable to open file ='%s'\n", argv[0], argv[1]);
        printf("Returned val: %d ", ret);
        printf("\n");
        exit(-1);
    }
    printf("\n%d unique symbols found\n\n", f_sym_key.sk_sym_count);

    if (argc > 2)
    for (ii = 2; ii < argc; ii++)
    {
        /* get the number supplied in argument */
        Value = (char*)strtoul(argv[ii], NULL, 0);

        /* find global symbol for that number */
        Psym_attrb = f_sym_name_global(&f_sym_key, Value);
        f_sym_attrb(&f_sym_key, Psym_attrb, &syms_attrb);
        printf("Global 0x%lx at %20s+0x%04x %-9s %-9s\n",
               (unsigned long)Value,
               syms_attrb.sa_name,
               (unsigned int)(unsigned long)(Value - syms_attrb.sa_value),
               syms_attrb.sa_scope,
               syms_attrb.sa_type
              );

        /* find closest symbol for that number */
        Psym_attrb = f_sym_name(&f_sym_key, Value);
        f_sym_attrb(&f_sym_key, Psym_attrb, &syms_attrb);
        printf("       0x%lx at %20s+0x%04x %-9s %-9s\n",
               (unsigned long)Value,
               syms_attrb.sa_name,
               (unsigned int)(unsigned long)(Value - syms_attrb.sa_value),
               syms_attrb.sa_scope,
               syms_attrb.sa_type
              );
        printf  ("\n");
    }
    f_sym_name_free(&f_sym_key);

   exit(0);

} /* f_sym_name:main() */

#endif /* EXAMPLE ] */


