// for finding memory leaks in debug mode with Visual Studio 
#if defined _DEBUG && defined _MSC_VER
#include <crtdbg.h>
#endif

#include <stdint.h>
#ifndef _WIN32
#include <unistd.h> /* usleep() */
#endif
#include "pt_header.h"
#include "pt_helpers.h"
#include "pt_visuals.h"
#include "pt_scopes.h"
#include "pt_sampler.h"
#include "pt_palette.h"
#include "pt_tables.h"

// This uses code that is not entirely thread safe, but I have never had any issues so far...

static volatile uint8_t scopesReading;
static uint64_t timeNext64, timeNext64Frac;
static SDL_Thread *scopeThread;

// globals
scopeChannel_t scope[4];
// -------

extern int8_t forceMixerOff;  // pt_audio.c
extern uint32_t *pixelBuffer; // pt_main.c

void updateScopes(void)
{
    uint8_t i;
    int32_t pos, len;
    scopeChannel_t *sc;
    moduleSample_t *s;

    if (editor.isWAVRendering)
        return;

    s = &modEntry->samples[editor.currSample];
    hideSprite(SPRITE_SAMPLING_POS_LINE);

    for (i = 0; i < AMIGA_VOICES; i++)
    {
        sc = &scope[i];
        if (sc->retriggered)
        {
            sc->retriggered = false;

            sc->pos     = 0;
            sc->posFrac = 0;

            // data/length is already set from replayer thread (important)
            sc->loopFlag  = sc->newLoopFlag;
            sc->loopStart = sc->newLoopStart;

            sc->didSwapData = false;
            sc->active      = true;
        }
        else if (sc->active)
        {
            sc->posFrac += sc->posDelta;
            if (sc->posFrac >= 65536)
            {
                pos = sc->pos + (sc->posFrac >> 16);
                sc->posFrac &= 0xFFFF;

                if (pos >= sc->length)
                {
                    // sample reached end, simulate Paula register update (sample swapping)

                    // wrap phase around one time with current length, then set new length
                    // and wrap around it (handles one-shot loops and sample swapping)

                    len  = sc->length;
                    pos -= len;
                    len  = sc->newLength;

                    if (len > 0)
                        pos %= len;

                    sc->length      = len;
                    sc->pos         = pos;
                    sc->data        = sc->newData;
                    sc->loopFlag    = sc->newLoopFlag;
                    sc->loopStart   = sc->newLoopStart;
                    sc->didSwapData = true;
                }
                else
                {
                    // no sample end, update position only
                    sc->pos = pos;
                }
            }
        }

        // move sample read position sprite (line)
        if ((modEntry->channels[i].n_samplenum == editor.currSample) && editor.ui.samplerScreenShown && !editor.muted[i])
        {
            pos = sc->pos;
            if (sc->active && (pos >= 2) && !editor.ui.samplerVolBoxShown && !editor.ui.samplerFiltersBoxShown)
            {
                // get real sampling position regardless of where the scope data points to
                // sc->data changes during loop, offset and so on, so this has to be done (sadly, because it's really hackish)

                pos = (int32_t)(&sc->data[pos] - &modEntry->sampleData[s->offset]);
                if ((pos >= 0) && (pos < s->length))
                {
                    pos = 3 + smpPos2Scr(pos);
                    if ((pos >= 3) && (pos <= 316))
                        setSpritePos(SPRITE_SAMPLING_POS_LINE, pos, 138);
                }
            }
        }
    }
}

void drawScopes(void)
{
    uint8_t y;
    int16_t scopeData;
    int32_t i, x, readPos;
    uint32_t *dstPtr, *scopePtr, scopePixel;
    scopeChannel_t cachedScope, *sc;

    scopesReading = true;
    if (editor.ui.visualizerMode == VISUAL_QUADRASCOPE)
    {
        // --- QUADRASCOPE ---

        scopePtr = &pixelBuffer[(71 * SCREEN_W) + 128];
        for (i = 0; i < AMIGA_VOICES; ++i)
        {
            // cache scope channel to lower thread race condition issues
            memcpy(&cachedScope, &scope[i], sizeof (scopeChannel_t));
            sc = &cachedScope;

            // render scope
            if (sc->active && ((sc->length > 2) || sc->loopFlag) && (sc->volume != 0) && !editor.muted[i] && (sc->data != NULL))
            {
                // scope is active

                scope[i].emptyScopeDrawn = false;

                // draw scope background

                dstPtr = &pixelBuffer[(55 * SCREEN_W) + (128 + (i * (SCOPE_WIDTH + 8)))];
                scopePixel = palette[PAL_BACKGRD]; // this palette can change

                for (y = 0; y < SCOPE_HEIGHT; ++y)
                {
                    for (x = 0; x < SCOPE_WIDTH; ++x)
                        dstPtr[x] = scopePixel;

                    dstPtr += SCREEN_W;
                }

                // render scope data

                scopePixel = palette[PAL_QADSCP];

                readPos = sc->pos;
                if (sc->loopFlag)
                {
                    // loop enabled

                    for (x = 0; x < SCOPE_WIDTH; ++x)
                    {
                        if (sc->didSwapData)
                        {
                            if (sc->length > 0)
                                readPos %= sc->length; // sc->data = loopStartPtr, wrap readPos to 0
                        }
                        else if (readPos >= sc->length)
                        {
                            readPos = sc->loopStart; // sc->data = sampleStartPtr, wrap readPos to loop start
                        }

                        scopeData = sc->data[readPos++] * sc->volume;
                        scopeData = SAR16(scopeData, 8);

                        scopePtr[(scopeData * SCREEN_W) + x] = scopePixel;
                    }
                }
                else
                {
                    // no loop

                    for (x = 0; x < SCOPE_WIDTH; ++x)
                    {
                        if (readPos >= sc->length)
                        {
                            scopePtr[x] = scopePixel; // end of data, draw center pixel
                        }
                        else
                        {
                            scopeData = sc->data[readPos++] * sc->volume;
                            scopeData = SAR16(scopeData, 8);
                            scopePtr[(scopeData * SCREEN_W) + x] = scopePixel;
                        }
                    }
                }
            }
            else
            {
                // scope is inactive, draw empty scope once until it gets active again

                sc = &scope[i];
                if (!sc->emptyScopeDrawn)
                {
                    // draw scope background

                    dstPtr     = &pixelBuffer[(55 * SCREEN_W) + (128 + (i * (SCOPE_WIDTH + 8)))];
                    scopePixel = palette[PAL_BACKGRD];

                    for (y = 0; y < SCOPE_HEIGHT; ++y)
                    {
                        for (x = 0; x < SCOPE_WIDTH; ++x)
                            dstPtr[x] = scopePixel;

                        dstPtr += SCREEN_W;
                    }

                    // draw line

                    scopePixel = palette[PAL_QADSCP];
                    for (x = 0; x < SCOPE_WIDTH; ++x)
                        scopePtr[x] = scopePixel;

                    sc->emptyScopeDrawn = true;
                }
            }

            scopePtr += (SCOPE_WIDTH + 8);
        }
    }
    scopesReading = false;
}

static int32_t SDLCALL scopeThreadFunc(void *ptr)
{
    int32_t time32;
    uint32_t diff32;
    uint64_t time64;

    // this is needed for the scopes to stutter slightly less (confirmed)
    SDL_SetThreadPriority(SDL_THREAD_PRIORITY_HIGH);

    // set next frame time
    timeNext64     = SDL_GetPerformanceCounter() + editor.vblankTimeLen;
    timeNext64Frac = editor.vblankTimeLenFrac;

    while (editor.programRunning)
    {
        updateScopes();

        // sync scopes to 60Hz (ish)

        time64 = SDL_GetPerformanceCounter();
        if (time64 < timeNext64)
        {
            MY_ASSERT((timeNext64 - time64) <= 0xFFFFFFFFULL);
            diff32 = (uint32_t)(timeNext64 - time64);

            // convert and round to microseconds
            time32 = (int32_t)((diff32 * editor.perfFreqMulMicro_f) + 0.5);

            // delay until we have reached next tick
            if (time32 > 0)
                usleep(time32);
        }

        // update next tick time
        timeNext64 += editor.vblankTimeLen;

        timeNext64Frac += editor.vblankTimeLenFrac;
        if (timeNext64Frac >= (1ULL << 32))
        {
            timeNext64++;
            timeNext64Frac &= 0xFFFFFFFF;
        }
    }

    (void)(ptr); // make compiler happy

    return (true);
}

uint8_t initScopes(void)
{
    scopeThread = SDL_CreateThread(scopeThreadFunc, "PT Clone Scope Thread", NULL);
    if (scopeThread == NULL)
    {
        showErrorMsgBox("Couldn't create scope thread!");
        return (false);
    }

    // don't let thread wait for this thread, let it clean up on its own when done
    SDL_DetachThread(scopeThread);

    return (true);
}

void waitOnScopes(void)
{
    while (scopesReading);
}

void clearScopes(void)
{
    uint8_t i;

    waitOnScopes();
    memset(scope, 0, sizeof (scope));

    for (i = 0; i < AMIGA_VOICES; ++i)
        scope[i].length = scope[i].newLength = 2;
}
