// for finding memory leaks in debug mode with Visual Studio 
#if defined _DEBUG && defined _MSC_VER
#include <crtdbg.h>
#endif

#include <stdint.h>
#include <stdbool.h>
#include <math.h> // modf()
#ifndef _WIN32
#include <unistd.h> // usleep()
#endif
#include "pt_header.h"
#include "pt_helpers.h"
#include "pt_visuals.h"
#include "pt_scopes.h"
#include "pt_sampler.h"
#include "pt_palette.h"
#include "pt_tables.h"

// this uses code that is not entirely thread safe, but I have never had any issues so far...

static volatile bool scopesReading;
static uint32_t scopeTimeLen, scopeTimeLenFrac;
static uint64_t timeNext64, timeNext64Frac;
static SDL_Thread *scopeThread;

scopeChannel_t scope[4]; // global
scopeChannelExt_t scopeExt[4]; // global

extern bool forceMixerOff; // pt_audio.c
extern uint32_t *pixelBuffer; // pt_main.c

int32_t getSampleReadPos(uint8_t ch, uint8_t smpNum)
{
	const int8_t *data;
	int32_t pos;
	scopeChannel_t *sc;
	moduleSample_t *s;
	
	sc = &scope[ch];

	// cache some stuff
	data = sc->data;
	pos = sc->pos;

	if (scopeExt[ch].active && pos >= 2)
	{
		s = &modEntry->samples[smpNum];

		/* get real sampling position regardless of where the scope data points to
		** sc->data changes during loop, offset and so on, so this has to be done
		** (sadly, because it's really hackish) */
		pos = (int32_t)(&data[pos] - &modEntry->sampleData[s->offset]);
		if (pos >= s->length)
			return -1;

		return pos;
	}

	return -1;
}

void updateScopes(void)
{
	uint32_t period;
	scopeChannel_t *sc, tmp;
	scopeChannelExt_t *se, tmpExt;

	if (editor.isWAVRendering)
		return;

	for (uint32_t i = 0; i < AMIGA_VOICES; i++)
	{
		sc = &scope[i];
		se = &scopeExt[i];

		// cache these
		tmp = *sc;
		tmpExt = *se;

		if (!tmpExt.active)
			continue; // scope is not active

		period = (uint32_t)modEntry->channels[i].n_period;
		if (period > 0)
		{
			if (period < 113)
				period = 113; // confirmed Paula behavior

#if SCOPE_HZ != 64
#error Scope Hz is not 64 (2^n), change rate calc. to use doubles+round in pt_scope.c
#endif
			tmp.posFrac += ((PAULA_PAL_CLK * (65536UL / SCOPE_HZ)) / period);
		}

		if (tmp.posFrac >= 65536)
		{
			tmp.pos += tmp.posFrac >> 16;
			tmp.posFrac &= 0xFFFF;

			if (tmp.pos >= tmp.length)
			{
				// sample reached end, simulate Paula register update (sample swapping)

				/* wrap pos around one time with current length, then set new length
				** and wrap around it (handles one-shot loops and sample swapping) */
				tmp.pos -= tmp.length;
				tmp.length = tmpExt.newLength;

				if (tmp.length > 0)
					tmp.pos %= tmp.length;

				tmp.data = tmpExt.newData;
				tmp.loopFlag = tmpExt.newLoopFlag;
				tmp.loopStart = tmpExt.newLoopStart;

				se->didSwapData = true;
			}
		}

		*sc = tmp; // update it
	}
}

void drawScopes(void)
{
	bool didSwapData;
	int8_t volume;
	int16_t scopeData;
	int32_t i, x, y, readPos;
	uint32_t *dstPtr, *scopePtr, scopePixel;
	scopeChannel_t tmpScope, *sc;
	scopeChannelExt_t *se;

	scopesReading = true;
	if (editor.ui.visualizerMode == VISUAL_QUADRASCOPE)
	{
		// --- QUADRASCOPE ---

		scopePtr = &pixelBuffer[(71 * SCREEN_W) + 128];
		for (i = 0; i < AMIGA_VOICES; i++)
		{
			sc = &scope[i];
			se = &scopeExt[i];

			// cache these
			tmpScope = *sc;
			didSwapData = se->didSwapData;

			volume = 0 - (modEntry->channels[i].n_volume >> 1);

			// render scope
			if (se->active && (tmpScope.length > 2 || tmpScope.loopFlag) &&
				volume != 0 && !editor.muted[i] && tmpScope.data != NULL)
			{
				// scope is active

				se->emptyScopeDrawn = false;

				// draw scope background

				dstPtr = &pixelBuffer[(55 * SCREEN_W) + (128 + (i * (SCOPE_WIDTH + 8)))];
				scopePixel = palette[PAL_BACKGRD]; // this palette can change

				for (y = 0; y < SCOPE_HEIGHT; y++)
				{
					for (x = 0; x < SCOPE_WIDTH; x++)
						dstPtr[x] = scopePixel;

					dstPtr += SCREEN_W;
				}

				// render scope data

				scopePixel = palette[PAL_QADSCP];

				readPos = tmpScope.pos;
				if (tmpScope.loopFlag)
				{
					// loop enabled

					for (x = 0; x < SCOPE_WIDTH; x++)
					{
						if (didSwapData)
						{
							if (tmpScope.length > 0)
								readPos %= tmpScope.length; // s.data = loopStartPtr, wrap readPos to 0
						}
						else if (readPos >= tmpScope.length)
						{
							readPos = tmpScope.loopStart; // s.data = sampleStartPtr, wrap readPos to loop start
						}

						scopeData = (tmpScope.data[readPos++] * volume) >> 8;
						scopePtr[(scopeData * SCREEN_W) + x] = scopePixel;
					}
				}
				else
				{
					// no loop

					for (x = 0; x < SCOPE_WIDTH; x++)
					{
						if (readPos >= tmpScope.length)
						{
							scopePtr[x] = scopePixel; // end of data, draw center pixel
						}
						else
						{
							scopeData = (tmpScope.data[readPos++] * volume) >> 8;
							scopePtr[(scopeData * SCREEN_W) + x] = scopePixel;
						}
					}
				}
			}
			else
			{
				// scope is inactive, draw empty scope once until it gets active again

				if (!se->emptyScopeDrawn)
				{
					// draw scope background

					dstPtr = &pixelBuffer[(55 * SCREEN_W) + (128 + (i * (SCOPE_WIDTH + 8)))];
					scopePixel = palette[PAL_BACKGRD];

					for (y = 0; y < SCOPE_HEIGHT; y++)
					{
						for (x = 0; x < SCOPE_WIDTH; x++)
							dstPtr[x] = scopePixel;

						dstPtr += SCREEN_W;
					}

					// draw line

					scopePixel = palette[PAL_QADSCP];
					for (x = 0; x < SCOPE_WIDTH; x++)
						scopePtr[x] = scopePixel;

					se->emptyScopeDrawn = true;
				}
			}

			scopePtr += SCOPE_WIDTH + 8;
		}
	}
	scopesReading = false;
}

static int32_t SDLCALL scopeThreadFunc(void *ptr)
{
	int32_t time32;
	uint32_t diff32;
	uint64_t time64;

	(void)ptr;

	// this is needed for scope stability (confirmed)
	SDL_SetThreadPriority(SDL_THREAD_PRIORITY_HIGH);

	// set next frame time
	timeNext64 = SDL_GetPerformanceCounter() + scopeTimeLen;
	timeNext64Frac = scopeTimeLenFrac;

	while (editor.programRunning)
	{
		updateScopes();

		time64 = SDL_GetPerformanceCounter();
		if (time64 < timeNext64)
		{
			assert(timeNext64-time64 <= 0xFFFFFFFFULL);
			diff32 = (uint32_t)(timeNext64 - time64);

			// convert to microseconds and round to integer
			time32 = (uint32_t)((diff32 * editor.dPerfFreqMulMicro) + 0.5);

			// delay until we have reached next tick
			if (time32 > 0)
				usleep(time32);
		}

		// update next tick time
		timeNext64 += scopeTimeLen;

		timeNext64Frac += scopeTimeLenFrac;
		if (timeNext64Frac >= (1ULL << 32))
		{
			timeNext64Frac &= 0xFFFFFFFF;
			timeNext64++;
		}
	}

	return true;
}

bool initScopes(void)
{
	double dInt, dFrac;

	// calculate scope time for performance counters and split into int/frac
	dFrac = modf(editor.dPerfFreq / SCOPE_HZ, &dInt);

	// integer part
	scopeTimeLen = (uint32_t)dInt;

	// fractional part scaled to 0..2^32-1
	dFrac *= UINT32_MAX + 1.0;
	if (dFrac > (double)UINT32_MAX)
		dFrac = (double)UINT32_MAX;
	scopeTimeLenFrac = (uint32_t)round(dFrac);

	scopeThread = SDL_CreateThread(scopeThreadFunc, NULL, NULL);
	if (scopeThread == NULL)
	{
		showErrorMsgBox("Couldn't create scope thread!");
		return false;
	}

	SDL_DetachThread(scopeThread);
	return true;
}

void waitOnScopes(void)
{
	while (scopesReading);
}

void clearScopes(void)
{
	waitOnScopes();

	memset(scope, 0, sizeof (scope));
	memset(scopeExt, 0, sizeof (scopeExt));

	for (uint8_t i = 0; i < AMIGA_VOICES; i++)
		scope[i].length = scopeExt[i].newLength = 2;
}
