package AtomicData::USZip;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: USZip.pm,v 1.1 2000/11/23 23:36:15 muaddib Exp $

use strict;

=head1 NAME

AtmoicData::USZip - for storing US zip codes

=head1 SYNOPSIS

see AtomicData.pm

=head1 DESCRIPTION

Implements data encapsulation for US zip codes

=cut

use AtomicData::Text;
@AtomicData::USZip::ISA = qw (AtomicData::Text);

=head1 METHODS

see AtomicData.pm for further methods.

=head2 parameters

see parent class for additional documentation. This class depricates the
min and max_length parameters and handles length itself since the
length must be either 5 or 9 digits. This class supports 'long'/bool
which, if true, requires that a nine digit zip be entered.

=head2 set_format

see parent class for additional documentation. Will accept
'short'/bool, and if true ,then will format the zip code to a 5 digit
zip. Also accepts 'dash'/bool, and if true, will insert a dash between
the 5'th and 6'th digit.

=head2 canonicalize

see parent class for additional documentation.

Will return data in the form of /^((\d{5})|(\d{9}))?$/.

=head2 _canonicalize

 instance/private
 (int $value) _canonicalize (int $raw_value)

DESCRIPTION:

Will accept a value and return the cannonical form or undef and set
C<$this->{_can_failed}>.

=cut

sub _canonicalize {
  my ($this, $val) = @_;

  $val = $this->SUPER::_canonicalize($val);
  defined $val or return;
  $val eq "" and return $val;

  $val =~ s/\s*-+\s*/-/;
  $val =~ s/\s+/-/;

  $val =~ /^(\d{5})$/ and return $val;
  $val =~ /^(\d{5})(\d{4})$/ and return $1."-".$2;
  $val =~ /^(\d{5})-(\d{4})$/ and return $1."-".$2;

  $this->{_can_failed} = ['Invalid US zip code.'];
  return;
}

=head2 _verify

 instance/private
 (1) _verify ()

DESCRIPTION:

Verifies that the data is 9 digits if long is set.

=cut

sub _verify {
  my ($this) = @_;

  my ($bool,$problems) = $this->SUPER::_verify();
  $bool or return (0, $problems);

  my $value = $this->canonical_value();

  my $parameters = $this->parameters();

  if ($parameters->{long} && (length($value) != 10)) {
    return (0, ['A 5+4 digit zip code is required.']);
  }
  
  return (1, []);
}

=head2 _format

  instance
  (string $zip_code) _format (int $can_value)

DESCRIPTION:

Returns a formatted zipcode.

=cut

sub _format {
  my ($this, $value) = @_;

  my $parameters = $this->{_parameters};

  if ($parameters->{short}) {
    $value =~ s/(\d{5}-?\d{4})/$1/;
    return $value; #5 digit zips don't need dashes
  }

  if ($parameters->{dash}) {
    $value =~ /(\d{5})-?(\d{4})/;
    if ($2) {
      $value = $1.'-'.$2;
    }    
  }

  return $value;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
