/* PureAdmin
 * Copyright (C) 2003 Isak Savo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Wrapper functions for FAM (File Alteration Monitor). Used to monitor changes in files and directories.
 *
 * Copyright (C) 2003-2004 Isak Savo
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

/* FIXME: Something is wrong. CHANGED events doesn't occurr.. in fact, no events occurr at all :-/ */
#include <stdio.h>
#include <gtk/gtk.h>
#include <errno.h>
#include "globals.h"

#ifdef HAVE_LIBFAM
#include <fam.h>
#include "helper.h"

static GPtrArray *monitors = NULL;
static FAMConnection *famconn = NULL;
static gint input_id = 0;
static GIOChannel *iochannel;

char *fam_event_names[] =
{
  "",
  "FAMChanged", "FAMDeleted", "FAMStartExecuting", 
  "FAMStopExecuting", "FAMCreated", "FAMMoved", 
  "FAMAcknowledge", "FAMExists", "FAMEndExist"
};

struct fam_watch {
  gchar *fname;
  void (*func)(void);
  FAMRequest *request;
};

struct fam_watch *_fam_find_node_fname (const gchar *fname);
static gboolean fam_event_handler (GIOChannel *src, GIOCondition cond, gpointer data);


/* Only for internal use, connects to local FAM-service */
gboolean _connect_to_fam ()
{
#ifdef HAVE_LIBFAM
   famconn = g_malloc0 (sizeof (FAMConnection));
   pur_log_dbg ("Opening FAM connection...");
   if (FAMOpen (famconn) != 0)
   {
      g_free (famconn);
      famconn = NULL;
      pur_log_wrn ("Unable to connect to FAM!");
      return FALSE;
   }
   
   return TRUE;
#else /* HAVE_LIBFAM */
   pur_log_dbg ("Can't connect to FAM: not supported!");
   return FALSE;
#endif /* HAVE_LIBFAM */
}

/*** Public Functions ***/

/* fam_is_running: Returns the status of the FAM-connection. Can be used to verify that FAM 
 *		   is running and that we are able to use it.
 * 
 * Returns: TRUE if we're connected, else FALSE
 */
gboolean fam_is_running (void)
{
   if (!famconn)
   {
      /* Not running, try to connect! */
      _connect_to_fam ();
      if (!famconn)
	return FALSE;
   }
   return TRUE;
}
/* fam_add_watch: Adds a watch to a directory or file and calls a function when a change is noticed. 
 *  
 *  Args:
 *	fname - The name of the file or directory that should be watched.
 *	func  - A pointer to the function that should be called. Function should be of type "void function (void)"
 *  Returns:
 *	TRUE if watch was added, else FALSE.
 */
gboolean fam_add_watch (const gchar *fname, void (*func))
{
   FAMRequest *req;
   struct fam_watch *arr_entry = NULL;

   pur_log_dbg ("Adding watch: %s", fname);
   if (!fname || g_file_test (fname, G_FILE_TEST_EXISTS) == FALSE)
     return FALSE;
   /* Connect to FAM if not done before */
   if (famconn == NULL)
   {
      if (!_connect_to_fam ())
	return FALSE;
      iochannel = g_io_channel_unix_new (FAMCONNECTION_GETFD(famconn));
      if (!iochannel)
      {
	 pur_log_err ("Unable to create iochannel");
	 return FALSE;
      }
      input_id = g_io_add_watch (iochannel, G_IO_IN | G_IO_HUP, (GIOFunc) fam_event_handler, NULL);
      pur_log_dbg ("GIOChannel Input watch added, id=%d, fd=%d", input_id, FAMCONNECTION_GETFD(famconn));
      g_io_channel_unref (iochannel); /* Remove our reference (add_watch() has one already) */
   }
   /* First time allocation */
   if (monitors == NULL)
     monitors = g_ptr_array_sized_new (10);
   
   req = g_malloc0 (sizeof (FAMRequest));
   arr_entry = g_malloc0 (sizeof (struct fam_watch));
   if (g_file_test (fname, G_FILE_TEST_IS_DIR))
   {
      if (FAMMonitorDirectory (famconn, fname, req, NULL) == -1)
      {
	 g_free (req);
	 g_free (arr_entry);
	 pur_log_err ("FAMMonitorDirectory: %s", g_strerror (errno));
	 return FALSE;
      }
      /* We've established a new monitor, add it to the array and return the request-pointer back to the calling function */
      arr_entry->fname = g_strdup (fname);
      arr_entry->request = req;
      arr_entry->func = func;
      g_ptr_array_add (monitors, (gpointer) arr_entry);
      pur_log_dbg ("FAM Directory watch added successfully");
   }
   else 
   {
      if (FAMMonitorFile (famconn, fname, req, NULL) == -1)
      {
	 g_free (req);
	 g_free (arr_entry);
	 pur_log_err ("FAMMonitorFile: %s", g_strerror (errno));
	 return FALSE;
      }
      arr_entry->fname = g_strdup (fname);
      arr_entry->request = req;
      arr_entry->func = func;
      /* We've established a new monitor, add it to the array and return the request-pointer back to the calling function */
      g_ptr_array_add (monitors, (gpointer) arr_entry);
      pur_log_dbg ("FAM File watch added successfully");
   }
   return TRUE;
}

/* fam_suspend_watch: Suspends (temporary disables) a watch
 *
 *  Args:
 *	fname - The filename of an existing watch.
 *  Returns:
 *	TRUE if suspension was successful, otherwise FALSE.
 */
gboolean fam_suspend_watch (const gchar *fname)
{
   int res;
   struct fam_watch *tmp;
   
   if (!fname || !famconn)
     return FALSE;
   pur_log_dbg ("Suspending watch: %s", fname);
   tmp = _fam_find_node_fname (fname);
   if (!tmp)
   {
      pur_log_wrn ("Unable to find node %s in list of current fam-watches\n", fname);
      return FALSE;
   }
/*    g_print ("Suspending watch: %s\n", fname); */
   res = FAMSuspendMonitor(famconn, tmp->request);
   if (res != 0)
     return FALSE; /* Error while suspending */
   return TRUE;

}

/* fam_resume_watch: Resumes a previously suspended watch
 *
 *  Args:
 *	fname - The filename of an existing (suspended) watch.
 *  Returns:
 *	TRUE if the resume was successful, otherwise FALSE.
 */
gboolean fam_resume_watch (const gchar *fname)
{
   int res;
   struct fam_watch *tmp;

   if (!fname || !famconn)
     return FALSE;
   tmp = _fam_find_node_fname (fname);
   if (!tmp)
   {
      pur_log_wrn ("Unable to find node %s in list of current fam-watches\n", fname);
      return FALSE;
   }
/*    g_print ("Resuming watch: %s\n", fname); */
   res = FAMResumeMonitor(famconn, tmp->request);
   if (res != 0)
     return FALSE; /* Error while resuming */
   return TRUE;
   
}

/* fam_delete_watch: Cancels a watch. No more notifixations will be recieved for that file!
 *
 *  Args:
 *	fname - The filename of an existing watch.
 *  Returns:
 *	TRUE if the deletion was successful, otherwise FALSE.
 */
gboolean fam_delete_watch (const gchar *fname)
{
   int res;
   struct fam_watch *tmp;

   if (!fname || !famconn)
     return FALSE;
   pur_log_dbg ("Removing watch: %s", fname);
   tmp = _fam_find_node_fname (fname);
   if (!tmp)
   {
      pur_log_wrn ("Unable to find node %s in list of current fam-watches\n", fname);
      return FALSE;
   }
/*    g_print ("Removing watch: %s\n", fname); */
   res = FAMCancelMonitor(famconn, tmp->request);
   if (res != 0)
     return FALSE; /* Error while canceling */
   g_ptr_array_remove (monitors, tmp);
   g_free (tmp->fname);
   g_free (tmp->request);
   g_free (tmp);
   return TRUE; /* FIXME: (?) Should we return FALSE if we couldn't remove it from the array? */
}

/* fam_terminate: Removes all watches and terminates connection to local FAM service.
 *
 *  Returns:
 *	TRUE if all went well, otherwise FALSE.
 */
gboolean fam_terminate (void)
{
   guint i;
   gboolean ret = TRUE;
   struct fam_watch *tmp;

   if (!famconn)
     return TRUE; /* We consider termination a success if a connection was never established... */
   /*    g_print ("FAMWrapper: Shutting down!\n"); */

   /* Remove all monitors and close FAM connection */
   if (monitors)
   {
      for (i = 0; i < monitors->len; i++)
      {
	 tmp = (struct fam_watch *) g_ptr_array_index (monitors, i);
	 g_free (tmp->fname);
	 g_free (tmp->request);
      }
      g_ptr_array_free (monitors, TRUE);
      monitors = NULL;
   }
   g_source_remove (input_id);
   input_id = 0;
   FAMClose (famconn);
   g_free (famconn);
   famconn = NULL;
   return ret;
}

/* Debuggingfunction, not used by the program! Prints all current watches! */
void dbg_fam_print_conns (void)
{
   guint i;
   FAMRequest *tmp;
   
   if (!monitors)
     return;
   g_print ("%i connections:\n", monitors->len);
   for (i = 0; i < monitors->len; i++)
   {
      tmp = (FAMRequest *) g_ptr_array_index (monitors, i);
      g_print (" Request no: %d\n", tmp->reqnum);
   }
}

/* fam_event_handler: Internal function, handles events and calls the appropriate function depending on the filename */
static gboolean fam_event_handler (GIOChannel *src, GIOCondition cond, gpointer data)
{
   FAMEvent ev;
   struct fam_watch *tmp;

   pur_log_dbg ("FAM Event occurred, GIOCondition: %d", cond);
   while (FAMPending(famconn))
   {
      if (FAMNextEvent(famconn, &ev) != 1)
      {
	 /*  Nuts!  An error!  Shut down & clean up.  */
	 pur_log_err ("Error while receiving FAM-event! This shouldn't happen");
	 /* FIXME: Should we really terminate? Perhaps its just temporary... */
	 fam_terminate ();
	 return FALSE;
      }
      pur_log_dbg ("FAM Eventcode: %d = %s", ev.code, fam_event_names[ev.code]);
      switch (ev.code)
      {
	 case FAMExists:
	 case FAMChanged:
	   /* Only call functions when file has changed or first time watch is enabled */
	   tmp = _fam_find_node_fname (ev.filename);
	   if (tmp)
	   {
	      pur_log_dbg ("File %s changed, calling handler", ev.filename);
	      tmp->func();
	   }
	   break;
	 case FAMDeleted:
	   fam_delete_watch (ev.filename);
 	   pur_log_dbg (" Removing watch [%s] (file deleted)\n", ev.filename);
	   break;
	 default:
	   break;
      }
   }
   return TRUE;
}

struct fam_watch *_fam_find_node_fname (const gchar *fname)
{
   gint i;
   struct fam_watch *tmp;

   if (!monitors)
     return NULL;
   for (i = 0; i < monitors->len; i++)
   {
      tmp = (struct fam_watch *) g_ptr_array_index (monitors, i);
      if (misc_str_is_equal (fname, tmp->fname))
	return tmp;
   }
   return NULL;
}

#else /* HAVE_LIBFAM */

gboolean fam_add_watch (const gchar *fname, void (*func))
{
   return FALSE;
}

gboolean fam_suspend_watch (const gchar *fname)
{
   return FALSE;
}

gboolean fam_resume_watch (const gchar *fname)
{
   return FALSE;
}

gboolean fam_delete_watch (const gchar *fname)
{
   return TRUE;
}

gboolean fam_terminate (void)
{
   return TRUE;
}

#endif /* HAVE_LIBFAM */
