/* PureAdmin
 * Copyright (C) 2003 Isak Savo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Misc helperfunctions used throughout the program 
 *
 * -- NOTE:
 * crypt() was taken and slightly modified from pureftpd: Copyright to the author(s)
 * --
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#define _XOPEN_SOURCE
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <signal.h>
#include <errno.h>
#include <gtk/gtk.h>
#include "helper.h"
#include "usr_manager.h"
#include "globals.h"
#include "cfg.h"

gint arr_count (gchar **arr)
{
   gint i = 0;

   if (!arr)
     return 0;
   while (arr[i] != NULL)
     i++;

   return i;
}

gboolean misc_str_is_equal (const gchar *s1, const gchar *s2)
{
   if (!s1 || !s2)
     return FALSE;
   if (strlen (s1) == strlen (s2) &&
       g_strcasecmp (s1, s2) == 0)
     return TRUE;
 
   return FALSE;
}

gboolean misc_str_is_only_whitespace (const gchar *s1)
{
   gint i;

   for (i = 0; i < strlen (s1); i++)
     if (!isspace (s1[i]))
       return FALSE;
   
   return TRUE;
}

/* FIXME: Perhaps we should use glibs g_spawn_* functions instead of system() */
gint misc_stop_server (void)
{
   gchar *cmd;
   gint ret;

   cmd = g_strdup_printf ("%s stop", srv_vars.startupscript);
   pur_log_nfo ("Stopping server: %s", cmd);
   ret = system (cmd);
   
   g_free (cmd);
   if (ret != 0)
     return FALSE;
   
   return TRUE;
}

gint misc_start_server (void)
{
   gchar *cmd;
   gint ret;

   cmd = g_strdup_printf ("%s start", srv_vars.startupscript);
   pur_log_nfo ("Starting server: %s", cmd);
   ret = system (cmd);

   g_free (cmd);
   
   if (ret != 0)
     return FALSE;
   return TRUE;
}

gint misc_close_connection (guint pid)
{
   gint ret;
   
   /* FIXME: A higher lower-bound should be by default if pureftpd_pid isn't set */
   if (pid <= (srv_vars.pureftpd_pid > 0 ? srv_vars.pureftpd_pid : 1))
     return 0;
#ifndef SIGTERM
   pur_log_wrn ("SIGTERM is not defined");
#define SIGTERM 15
#endif /* SIGTERM */
   ret = kill (pid, SIGTERM);
   if (ret == -1)
   {
      perror ("kill");
      pur_log_wrn ("Could not close kill process %d\n", pid);
      return 0;
   }
   return 1;
}

gint misc_get_pureftpd_pid (void)
{
   gchar buf[20];
   gchar pidfile[FILEPATH_MAX];
   FILE *fl;
   gint wait_count = 500; /* Wait max ~0.5 second for pidfile to be created */
   
   
   if (*srv_vars.fl_pidfile == '\0')
     strncpy (pidfile, DEF_PUREFTPD_PIDFILE, FILEPATH_MAX);
   else
     strncpy (pidfile, srv_vars.fl_pidfile, FILEPATH_MAX);

   while (g_file_test (pidfile, G_FILE_TEST_EXISTS) == FALSE && wait_count > 0)
   {
      while (g_main_context_pending (NULL))
      {
	 g_main_context_iteration (NULL, FALSE);
	 wait_count--;
      }
      g_usleep (1000);
      wait_count--;
   }
   if (g_file_test (pidfile, G_FILE_TEST_EXISTS) == FALSE)
     return -1;
   if ((fl = fopen (pidfile, "r")) == NULL)
   {
      g_print ("fopen ");
      perror (pidfile);
      return -1;
   }
   fgets (buf, 20, fl);
   fclose (fl);
   return atoi (buf);
}

gboolean misc_validate_username (const gchar *user)
{
   const gchar *ptr;
   gboolean valid = TRUE;
   ptr = user;
   if (!ptr)
     return FALSE;
   while (*ptr && valid)
   {
      if (!((*ptr >= 'A' && *ptr <= 'Z') ||
	    (*ptr >= 'a' && *ptr <= 'z') ||
	    (*ptr >= '0' && *ptr <= '9') ||
	    *ptr == '-' || *ptr == ','  ||
	    *ptr == '_' || *ptr == '.'  ||
	    *ptr == ':' || *ptr == '\'' ||
	    *ptr == '@' || *ptr == ' '))
      {
	 return FALSE;
      }
      ptr++;
   }
   return TRUE;
}

gboolean misc_command_exists (const gchar *cmd)
{
   gchar *fullpath;
   gboolean ret;
   fullpath = g_find_program_in_path (cmd);
   if (!fullpath)
     return FALSE;
   ret = g_file_test (fullpath, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_EXECUTABLE);
   g_free (fullpath);
   return ret;
}  

gint id_str_compare (const gchar *s1, const gchar *s2)
{
   /* Numeric sorting */
   gint a, b;
   if (!s1 || !s2)
     return -1;
   a = atoi (s1);
   b = atoi (s2);
   if (a < b)
     return -1;
   else if (a > b)
     return 1;
   else
     return 0;
}

GList *misc_get_user_ids (void)
{
   FILE *fl;
   GList *retval = NULL;
   gchar buf[LINE_MAX], **arr;

   if ((fl = fopen ("/etc/passwd", "r")) == NULL)
   {
      pur_log_wrn ("Unable to open password file [/etc/passwd]");
      return NULL;
   }
   while (fgets (buf, LINE_MAX, fl))
   {
      arr = g_strsplit (buf, ":", -1);
      if (arr_count (arr) < 2)
      {
	 g_strfreev (arr);
	 continue;
      }
      if (arr[0][0] && arr[2][0] &&
	  atoi (arr[2]) > 0)
	retval = g_list_insert_sorted (retval, g_strdup_printf ("%s (%s)", arr[2], arr[0]), (GCompareFunc) id_str_compare);
      g_strfreev (arr);
   }
   fclose (fl);
   return g_list_first (retval);
}

GList *misc_get_group_ids (void)
{
   FILE *fl;
   GList *retval = NULL;
   gchar buf[LINE_MAX], **arr;

   if ((fl = fopen ("/etc/group", "r")) == NULL)
   {
      pur_log_wrn ("Unable to open group file [/etc/group]");
      return NULL;
   }
   while (fgets (buf, LINE_MAX, fl))
   {
      arr = g_strsplit (buf, ":", -1);
      if (arr_count (arr) < 2)
      {
	 g_strfreev (arr);
	 continue;
      }
      if (arr[0][0] && arr[2][0] &&
	  atoi (arr[2]) > 0)
	retval = g_list_insert_sorted (retval, g_strdup_printf ("%s (%s)", arr[2], arr[0]), (GCompareFunc) id_str_compare);
      g_strfreev (arr);
   }
   fclose (fl);
   return g_list_first (retval);
}

gboolean misc_create_passwd_file (void)
{
   FILE *f;
   
   if ((f = fopen (cfg.pwfile, "w")) == NULL)
   {
      pur_log_wrn ("Could not create password file: %s", strerror (errno));
      return FALSE;
   }
   fclose (f);
   return TRUE;
}

gboolean misc_create_system_account (gboolean create_user, gboolean create_group)
{
   gboolean ret;
   gchar *cmd;
   gint child_retval;

   if (create_group && !*(srv_vars.cmd_groupadd))
     return FALSE;
   if (create_user &&  !*(srv_vars.cmd_useradd))
     return FALSE;
       
   if (create_group)
   {
      cmd = g_strdup_printf ("%s ftpgroup", srv_vars.cmd_groupadd);
      ret = g_spawn_command_line_sync (cmd, NULL, NULL, &child_retval, NULL);
      g_free (cmd);
      if (child_retval || !ret)
      {
	 pur_log_dbg ("Child 'groupadd' returned abnormally: %d or execution failed\n", child_retval);
	 return FALSE;
      }
   }
   if (create_user)
   {
      if (create_group || cfg.default_gid < 1)
	cmd = g_strdup_printf ("%s -g ftpgroup -d /dev/null -s /etc ftpuser", srv_vars.cmd_useradd);
      else
	cmd = g_strdup_printf ("%s -g %d -d /dev/null -s /etc ftpuser", srv_vars.cmd_useradd, cfg.default_gid);
      pur_log_dbg ("Running command [%s]", cmd);
      ret = g_spawn_command_line_sync (cmd, NULL, NULL, &child_retval, NULL);
      g_free (cmd);    
      if (child_retval || !ret)
      {
	 pur_log_wrn ("Child 'useradd' returned abnormally: %d or execution failed\n", child_retval);
	 return FALSE;
      }
   }
   
   return TRUE;
}

/* Crypt functions taken pure-pw distributed with pureftpd source package:
 * Copyright (c) 2001, 2002, 2003 Frank Denis <j@pureftpd.org> with the
 * help of all Pure-FTPd contributors.
 *
 * Modified slightly to use GLibs random numbers
 */
gchar *misc_crypt_passwd (const gchar * const pwd)
{
   
    static const char crcars[64] =
      "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789./";
    register const char *crypted;
    guint rdm;

    rdm = g_random_int (); 
    
    if ((crypted = (const char *)      /* Blowfish */
         crypt("test", "$2a$07$1234567890123456789012")) != NULL &&  
        strcmp(crypted, "$2a$07$123456789012345678901uKO4"
               "/IReKqBzRzT6YaajGvw20UBdHW7m") == 0) {
        char salt[] = "$2a$07$0000000000000000000000";        
        int c = 28;
        
        do {            
            c--;
            salt[c] = crcars[rdm & 63];
        } while (c > 7);
        
        return (char *) crypt(pwd, salt);        
    } else if ((crypted = (const char *)    /* MD5 */
                crypt("test", "$1$12345678$")) != NULL &&
               strcmp(crypted, "$1$12345678$oEitTZYQtRHfNGmsFvTBA/") == 0) {
        char salt[] = "$1$00000000";
        int c = 10;
        
        do {            
            c--;
            salt[c] = crcars[rdm & 63];
        } while (c > 3);
        
        return (char *) crypt(pwd, salt);
    } else if ((crypted = (const char *)    /* Extended DES */
                crypt("test", "_.../1234")) != NULL &&
               strcmp(crypted, "_.../1234PAPUVmqGzpU") == 0) {
        char salt[] = "_.../0000";
        int c = 8;
        
        do {
            c--;
            salt[c] = crcars[rdm & 63];
        } while (c > 5);
        
        return (char *) crypt(pwd, salt);
    }
    /* Simple DES */
    {
        char salt[] = "00";
        
        salt[0] = crcars[rdm & 63];
        salt[1] = crcars[rdm & 63];
        
        return (char *) crypt(pwd, salt);        
    }    
}
