/* PureAdmin
 * Copyright (C) 2003 Isak Savo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Usermanager functions
 *
 * Copyright (C) 2003 Isak Savo
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "interface.h"
#include "support.h"
#include "cfg.h"
#include "gui_helper.h"
#include "globals.h"
#include "helper.h"
#include "usr_manager.h"

struct usrman_info_widgets *usr_get_edit_widgets (void);

void _free_pwstruct (PWInfo *i)
{
   if (!i)
     return;
   g_free (i->login);
   g_free (i->pwd);
   g_free (i->home);
   g_free (i->gecos);
   g_free (i->allow_local_ip);
   g_free (i->deny_local_ip);    
   g_free (i->allow_client_ip);
   g_free (i->deny_client_ip);
   g_free (i);
}

gint _userlist_compare_available_users (gconstpointer p1, gconstpointer p2)
{
   PAvailableUser *u1, *u2;
   
   u1 = (PAvailableUser *) p1;
   u2 = (PAvailableUser *) p2;
   
   return (gint) g_strcasecmp (u1->login, u2->login);
}

pur_error_t rebuild_puredb (void)
{
   FILE *p;
   gchar buf[LINE_MAX];
   
   if (!misc_command_exists (cfg.cmd_purepw))
     return E_CMD_NOT_FOUND;
   snprintf (buf, LINE_MAX, "%s mkdb %s %s", cfg.cmd_purepw, cfg.pdbfile, cfg.pwfile);
   if ((p = popen (buf, "r")) == NULL)
   {
      pur_log_wrn ("Error executing '%s': %s", buf, strerror (errno));
      return E_GENERAL_ERROR;
   }
   while (fgets (buf, LINE_MAX, p) != NULL)
   {
      g_print ("pure-pw output: %s", buf);
   }
   pclose (p);
   return E_NO_ERROR;
}

/* add_pw_line: Adds or modifies a password entry in the plain-text password-file used by pure-pw.
 *		If 'user' already exists, do an update; otherwise add a new line.
 * Args:
 *	line: The already formatted line to add.
 *	user: The username
 * Returns: 
 *	E_PERMISSION_DENIED - Couldn't open passwd file, create temporary file or rename/remove old passwd file.
 *	E_NO_ERROR - Everything updated successfully.
 */
pur_error_t add_pw_line (const gchar *line, const gchar *user)
{
   FILE *fl, *fl_tmp;
   gint fd_tmp;
   gchar tmp_fname[30], buf[LINE_MAX], *dir;
   gboolean added = FALSE, pwfile_exists = TRUE;
   
   dir = g_path_get_dirname (cfg.pwfile);
   snprintf (tmp_fname, 30, "%s/pureadmin_XXXXXX", dir);
   if ((fl = fopen (cfg.pwfile, "r")) == NULL)
   {
      perror ("Unable to open password file");
      if (g_file_test (cfg.pwfile, G_FILE_TEST_EXISTS))
	return E_PERMISSION_DENIED;
      else
	/* The file doesn't exist, but a new line could be added anyway */
	pwfile_exists = FALSE;
   }
   fd_tmp =  g_mkstemp (tmp_fname);
   if (((fl_tmp = fdopen (fd_tmp, "w")) == NULL))
   {
      pur_log_wrn ("Unable to open temporary file: %s", strerror (errno));
      close (fd_tmp);
      return E_PERMISSION_DENIED;
   }
   
   while (pwfile_exists && fgets (buf, LINE_MAX, fl))
   {
      if (strncmp (buf, user, strlen (user)) == 0)
      {
	 fputs (line, fl_tmp);
	 fputc ('\n', fl_tmp);
	 added = TRUE;
      }
      else
	fputs (buf, fl_tmp);
   }
   if (!added)
   {
      fputs (line, fl_tmp);
      fputc ('\n', fl_tmp);
   }
   fclose (fl_tmp);
   if (pwfile_exists)
   {
      fclose (fl);
      if (remove (cfg.pwfile) != 0)
      {
	 pur_log_wrn ("Unable to remove password-file");
	 return E_PERMISSION_DENIED;
      }
   }
   if (rename (tmp_fname, cfg.pwfile) != 0)
   {
      pur_log_wrn ("Unable to rename [%s] to [%s]: %s", tmp_fname, cfg.pwfile, strerror (errno));
      return E_PERMISSION_DENIED;
   }
   return E_NO_ERROR;
}

pur_error_t remove_pw_line (const gchar *user, gchar **line)
{
   FILE *fl, *fl_tmp;
   gint fd_tmp, len;
   gchar tmp_fname[30], buf[LINE_MAX];
   gboolean removed = FALSE;
   gchar *user_locale, *dir, *sep_pos, *buf_start;

   if ((fl = fopen (cfg.pwfile, "r")) == NULL)
   {
      pur_log_wrn ("Unable to open password file");
      return E_FILE_NOT_FOUND;
   }
   dir = g_path_get_dirname (cfg.pwfile);
   snprintf (tmp_fname, 30, "%s/pureadmin_XXXXXX", dir);
   g_free (dir);
   fd_tmp = g_mkstemp (tmp_fname);
   if ((fl_tmp = fdopen (fd_tmp, "w")) == NULL)
   {
      pur_log_wrn ("Unable to open temporary password-file");
      return E_PERMISSION_DENIED;
   }
   user_locale = g_locale_from_utf8 (user, -1, NULL, NULL, NULL);
   buf_start = buf;
   while (fgets (buf, LINE_MAX, fl))
   {
      if ((sep_pos = strchr (buf, PW_LINE_SEP_CH)) == NULL)
	/* No ':' found in string */
	continue;
      len = GPOINTER_TO_INT(sep_pos) - GPOINTER_TO_INT(buf_start);
      if (strlen (user_locale) == len && strncmp (user_locale, buf, len) == 0)
      {
	 removed = TRUE;
	 *line = strdup (buf);
      }
      else
	fputs (buf, fl_tmp);
   }
   g_free (user_locale);
   fclose (fl_tmp);
   fclose (fl);
   if (removed == FALSE)
   {
      remove (tmp_fname);
      return E_USER_NOT_FOUND;
   }
   if (remove (cfg.pwfile) != 0)
   {
      pur_log_wrn ("Unable to remove password-file [%s]: %s ", cfg.pwfile, strerror (errno));
      return E_PERMISSION_DENIED;
   }
   if (rename (tmp_fname, cfg.pwfile) != 0)
   {
      pur_log_wrn ("Unable to rename [%s] to [%s]: %s", tmp_fname, cfg.pwfile, strerror (errno));
      return E_PERMISSION_DENIED;
   }
   return E_NO_ERROR;
}

/* parse_pw_line: Parses a line read from a pure-ftpd password file.
 * Args: line	 the line to parse.
 *       pwinfo: preallocated struct where the info will be stored.
 *       user:   if not NULL, then return failure if users doesn't match.
 *
 * Returns: TRUE upon success, otherwise FALSE
*/
gboolean parse_pw_line(gchar *line, PWInfo * const pwinfo, const gchar *user)
{
   gchar **arr;
   gint to_free[13] = { 0, 2, 3, 4, 5,6, 7, 8, 9, 10, 11, 12, 17 };
   gint i;
   
   memset (pwinfo, 0, sizeof (PWInfo));
   arr = g_strsplit (line, PW_LINE_SEP, 0);
   if (!arr)
     return FALSE;
   pwinfo->login = g_locale_to_utf8 (arr[0], -1, NULL, NULL, NULL);
   if (user && !misc_str_is_equal (user, pwinfo->login))
   {
      g_strfreev (arr);
      return FALSE;
   }
   pwinfo->pwd = arr[1];
   pwinfo->uid = (uid_t) strtoul(arr[2], NULL, 10);
   pwinfo->gid = (gid_t) strtoul(arr[3], NULL, 10);
   if (pwinfo->uid <= (uid_t) 0 || pwinfo->gid <= (gid_t) 0) {
      g_strfreev (arr);
      return FALSE;
   }
   pwinfo->gecos = g_locale_to_utf8(arr[4], -1, NULL, NULL, NULL);
   if (*(arr[5]) != '/')
   {
      g_strfreev (arr);
      return FALSE;
   }
   pwinfo->home = g_locale_to_utf8 (arr[5], -1, NULL, NULL, NULL);    
   if (*(arr[6])) {
      pwinfo->bw_ul = strtoul(arr[6], NULL, 10) / 1024;
   }
   if (*(arr[7])) {
      pwinfo->bw_dl = strtoul(arr[7], NULL, 10) / 1024;
   }
   if (*(arr[8])) {
        pwinfo->ul_ratio = (unsigned int) strtoul(arr[8], NULL, 10);
    }
    if (*(arr[9])) {
        pwinfo->dl_ratio = (unsigned int) strtoul(arr[9], NULL, 10);
    }
    if (*(arr[10])) {
	pwinfo->per_user_max = (unsigned int) strtoul(arr[10], NULL, 10);
    }
    if (*(arr[11])) {
        pwinfo->quota_files = strtoull(arr[11], NULL, 10);
    }
    if (*(arr[12])) {
        pwinfo->quota_size = strtoull(arr[12], NULL, 10) / 1024 / 1024;
    }
    pwinfo->allow_local_ip = arr[13];
    pwinfo->deny_local_ip = arr[14];
    pwinfo->allow_client_ip = arr[15];
    pwinfo->deny_client_ip = arr[16];
    if (sscanf(arr[17], "%u-%u", &pwinfo->time_begin, &pwinfo->time_end) == 2 &&
        pwinfo->time_begin < 2360 && (pwinfo->time_begin % 100) < 60 &&
        pwinfo->time_end < 2360 && (pwinfo->time_end % 100) < 60) 
    {
    }
    else
    {
       pwinfo->time_begin = 0;
       pwinfo->time_end = 0;
    }
    /* Free those strings that were converted to numbers/utf8-strings */
    for (i = 0; i < 13; i++)
      g_free (arr[to_free[i]]);
    g_free (arr);

    return TRUE;
}

gboolean usr_user_exists (const gchar *user)
{
   FILE *pw;
   gboolean found = FALSE;
   gchar *sep_pos, buf[200], *buf_start;
   int len;
   if (!user || strlen (user) < 1)
     return FALSE;

   if ((pw = fopen (cfg.pwfile, "r")) == NULL)
   {
      pur_log_err ("Couldn't open %s: %s", cfg.pwfile, g_strerror (errno));
      return FALSE;
   }
   buf_start = buf; /* Pointer magic :-) */
   while (fgets (buf, 200, pw))
   {
      if ((sep_pos = strchr (buf, PW_LINE_SEP_CH)) == NULL)
	continue;
      len = GPOINTER_TO_INT(sep_pos) - GPOINTER_TO_INT(buf_start);
      if (strlen (user) == len && strncmp (user, buf, len) == 0)
      {
	 found = TRUE;
	 break;
      }
   }
   fclose (pw);
   
   return found;
}

gboolean usr_select_first_user (void)
{
   static GtkWidget *tree_userman = NULL;
   static GtkTreeSelection *sel = NULL;
   GtkTreePath *path;

   if (!tree_userman)
   {
      tree_userman = lookup_widget (dlg_usrman, "treeview_dlg_users");
      sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_userman));
   }
   path = gtk_tree_path_new_from_string ("0"); /* First item */
   gtk_tree_selection_select_path (sel, path);
   gtk_tree_path_free (path);
   g_signal_emit_by_name ((gpointer) tree_userman, 
			  "cursor_changed", tree_userman, NULL);

   return TRUE;
}

GList *usr_get_available_users (void)
{
   GList *retval = NULL;
   PAvailableUser *user;
   FILE *pw;
   gchar buf[200], **arr;

   if ((pw = fopen (cfg.pwfile, "r")) == NULL)
   {
      pur_log_err ("Couldn't open %s: %s", cfg.pwfile, g_strerror (errno));
      return NULL;
   }
   while (fgets (buf, 200, pw))
   {
      g_strchomp (buf);
      arr = g_strsplit (buf, PW_LINE_SEP, 0);
      if (!MISC_VALID_STRING (arr)) /* May happen on empty lines */
	continue;
      user = (PAvailableUser *) g_malloc0 (sizeof (PAvailableUser));
      user->login = g_locale_to_utf8 (arr[0], -1, NULL, NULL, NULL);
      user->realname = g_locale_to_utf8 (arr[4], -1, NULL, NULL, NULL);
      retval = g_list_insert_sorted (retval, user, (GCompareFunc) _userlist_compare_available_users);
      g_strfreev (arr);
   }
   fclose (pw);
   return g_list_first (retval);
}

gboolean usr_manager_check_prerequisites ()
{
   static gboolean first_time = TRUE;
   gboolean need_group, need_user, ret;
   gchar *str;
   gint val;

   if (first_time == FALSE)
     return TRUE;
   pur_log_dbg ("usr_manager: First time run!");
   if (!g_file_test (cfg.pwfile, G_FILE_TEST_EXISTS))
   {
      /* No password-files found */
      ret = gui_display_confirmdialog (_("Create Password File?"),
				       _("The file containing the login-names for virtual users could not be found. You need this file in order to create and edit ftp-users.\n"
					 "Do you want to create this file?"),
				       _("_Yes, Create File"), NULL);
      if (ret == GTK_RESPONSE_YES)
      {
	 if (misc_create_passwd_file () == FALSE)
	 {
	    /* Couldn't create password file. Report error to user */
	    str = g_strdup_printf (_("There was an error when trying to create password file [%s].\n"
				     "The error reported was: '%s'"), cfg.pwfile, g_strerror (errno));
	    gui_display_msgdialog (_("File Creation Failed"),
				   str, MSGDLG_TYPE_ERROR, main_window);
	    return FALSE;
	 }
	 g_print ("pwfile set to: %s\n", cfg.pwfile);
      }
      else
	return FALSE;
   }
   else if ((val = access (srv_vars.fl_pwfile, R_OK | W_OK)) != 0)
   {
      if (val == -1)
	pur_log_err ("access(%s, R_OK | W_OK) failed: %s", srv_vars.fl_pwfile, g_strerror (errno));
      /* We didn't have permission to read/write to the password file */
      gui_display_msgdialog (_("Permission Denied"),
			     _("You don't have permission to read and write to the password file."
			       "This probably means that you have to run PureAdmin as root before beeing able to manage "
			       "your users."), MSGDLG_TYPE_ERROR, main_window);
      return FALSE;
   }
   need_group = need_user = FALSE;
   /* Check if there exist a dedicated ftpgroup and ftpuser */
   if (cfg.default_gid < 1)
     need_group = TRUE;
   if (cfg.default_uid < 1)
     need_user = TRUE;
   if (cfg.seen_usrdlg_welcome == FALSE && (need_user || need_group))
   {
      ret = gui_display_confirmdialog (_("Create User and Group"), 
				       _("It seems that you do not have a dedicated ftp-user and -group to use with the FTP users.\n"
					 "Do you want to create ftpgroup and ftpuser and assign them as default for new users?"),
				       _("_Yes, Create Account"), NULL);
      if (ret == GTK_RESPONSE_YES)
      {
	 ret = misc_create_system_account (need_user, need_group);
	 if (!ret)
	 {
	    gui_display_msgdialog (_("Account Creation Failed"), 
				   _("PureAdmin was unable to create the system-accounts. You will have to create them manually. "
				     "Use the appropriate tools for your distribution and read README.VirtualUsers distributed "
				     "with the PureFTPd package."), MSGDLG_TYPE_ERROR, main_window);
	 }
      }
      get_srv_vars ();
      if (need_group)
	cfg.default_gid = srv_vars.id_ftpgid;
      if (need_user)
	cfg.default_uid = srv_vars.id_ftpuid;
      cfg.seen_usrdlg_welcome = TRUE;
   }
   first_time = FALSE;
   return TRUE;
}
/* Fetches information for the specified user.
 * Returns: A struct w/ all info or NULL upon failure.
 */
PWInfo *usr_get_user_information (gchar *user)
{
   PWInfo *info;
   FILE *pw_fl;
   gchar *buf;
   
   if ((pw_fl = fopen (cfg.pwfile, "r")) == NULL)
   {
      pur_log_wrn ("Unable to open password file %s: %s", cfg.pwfile, g_strerror (errno));
      return NULL;
   }
   buf = (gchar *) g_malloc0 (LINE_MAX);
   info = (PWInfo *) g_malloc0 (sizeof (PWInfo));
   while (fgets (buf, LINE_MAX, pw_fl))
   {
      g_strstrip (buf);
      if (*buf == 0)
	continue;
      if (parse_pw_line (buf, info, user))
	return info;
   }
   
   g_free (info);
   return NULL;
}

pur_error_t usr_remove_user (const gchar *user)
{
   gint ret = E_GENERAL_ERROR;
   gchar *line;
   if (!user || *user == 0 || (ret = remove_pw_line (user, &line)) != E_NO_ERROR)
     return ret;
   else
   {
      ret = rebuild_puredb ();
      if (ret != E_NO_ERROR)
	add_pw_line (line, user);
      g_free (line);
      return ret;
   }
   return E_NO_ERROR;
}


pur_error_t usr_add_user (const gchar *user, const gchar *passwd)
{
   GString *line;
   gint ret;
   gchar *conv;

   conv = g_locale_from_utf8 (user, -1, NULL, NULL, NULL);
   line = g_string_new (conv);
   g_free(conv);
   g_string_append_printf (line, ":%s:", passwd);
   
   g_string_append_printf (line, "%d:%d::", cfg.default_uid, cfg.default_gid);
   g_string_append_printf (line, "%s/./", cfg.default_home); /* Make chroot by default */
   g_string_append (line, "::::::::::::");
   
   ret = add_pw_line (line->str, user);
   g_string_free (line, TRUE);
   if (ret != E_NO_ERROR)
     return ret;
   
   ret = rebuild_puredb ();
   return ret;
}

pur_error_t usr_save_user_info (gboolean existing)
{
   PWInfo *i;
   GString *cmd;
   gint ret;
   
   i = usr_widgets_get_info ();
   if (!i || !MISC_VALID_STRING (i->login))
   {
      pur_log_err ("Invalid information structure");
      return E_GENERAL_ERROR;
   }
   cmd = g_string_new (i->login);
   g_string_append_printf (cmd, ":%s:", i->pwd);
   /* UID and GID */
   g_string_append_printf (cmd, "%d:%d:", i->uid, i->gid);

   /* Real name */
   g_string_append_printf (cmd, "%s:", i->gecos ? i->gecos : "");
   /* Home */
   if (MISC_VALID_STRING (i->home))
     g_string_append_printf (cmd, "%s:", i->home);
   else
   {
      pur_log_wrn ("Invalid home directory: %s", i->home);
      g_string_free (cmd, TRUE);
      return E_GENERAL_ERROR;
   }
   /* Bandwidth- and size-restrictions */
   if (i->bw_ul)
     g_string_append_printf (cmd, "%li:", i->bw_ul * 1024);
   else
     g_string_append (cmd, ":");
   if (i->bw_dl)
     g_string_append_printf (cmd, "%li:", i->bw_dl * 1024);
   else
     g_string_append (cmd, ":");
   if (i->ul_ratio)
     g_string_append_printf (cmd, "%d:", i->ul_ratio);
   else
     g_string_append (cmd, ":");
   if (i->dl_ratio)
     g_string_append_printf (cmd, "%d:", i->dl_ratio);
   else
     g_string_append (cmd, ":");
   
   if (i->per_user_max)
     g_string_append_printf (cmd, "%d:", i->per_user_max);
   else
     g_string_append (cmd, ":");
   
   if (i->quota_files)
     g_string_append_printf (cmd, "%li:", i->quota_files);
   else
     g_string_append (cmd, ":");
   if (i->quota_size)
     g_string_append_printf (cmd, "%li:", i->quota_size * 1024 * 1024);
   else
     g_string_append (cmd, ":");
   
   /* IP-restrictions */
   if (MISC_VALID_STRING (i->allow_local_ip))
     g_string_append_printf (cmd, "%s:", i->allow_local_ip);
   else
     g_string_append (cmd, ":");
   
   if (MISC_VALID_STRING (i->deny_local_ip))
     g_string_append_printf (cmd, "%s:", i->deny_local_ip);
   else
     g_string_append (cmd, ":");
   
   if (MISC_VALID_STRING (i->allow_client_ip))
     g_string_append_printf (cmd, "%s:", i->allow_client_ip);
   else
     g_string_append (cmd, ":");
   
   if (MISC_VALID_STRING (i->deny_client_ip))
     g_string_append_printf (cmd, "%s:", i->deny_client_ip);
   else
     g_string_append (cmd, ":");
   
   /* Time restrictions */
   if (i->time_begin < 2360 && (i->time_begin % 100) < 60 &&
       i->time_end < 2360 && (i->time_end % 100))
     g_string_append_printf (cmd, "%04d-%04d", i->time_begin, i->time_end);
       
   /* TODO: emit cursor-change signal to treeview */
   if ((ret = add_pw_line (cmd->str, i->login)) != E_NO_ERROR)
   {
      pur_log_err ("Couldn't add pwline, returning %d", ret);
      g_string_free (cmd, TRUE);
      _free_pwstruct (i);
      return ret;
   }
   g_string_free (cmd, TRUE);
   _free_pwstruct (i);
   
   ret = rebuild_puredb ();
   return ret;
}
   

struct usrman_info_widgets *usr_get_edit_widgets (void)
{
   static struct usrman_info_widgets *ret = NULL;
   
   if (!ret)
   {
      ret = (struct usrman_info_widgets *) g_malloc0 (sizeof (struct usrman_info_widgets));

      ret->login	= lookup_widget (dlg_usrman, "usr_dlg_entry_uname");
      ret->realname	= lookup_widget (dlg_usrman, "usr_dlg_entry_realname");
      ret->e_uid	= lookup_widget (dlg_usrman, "usr_dlg_optmenu_uid");
      ret->e_gid	= lookup_widget (dlg_usrman, "usr_dlg_optmenu_gid");
      ret->chk_chroot	= lookup_widget (dlg_usrman, "usr_dlg_chk_chroot");
      ret->home		= lookup_widget (dlg_usrman, "usr_dlg_entry_home");
      ret->root		= lookup_widget (dlg_usrman, "usr_dlg_entry_fakeroot");
      ret->quota_files	= lookup_widget (dlg_usrman, "usr_dlg_spin_maxfiles");
      ret->quota_size	= lookup_widget (dlg_usrman, "usr_dlg_spin_maxsize");
      ret->timelimit	= lookup_widget (dlg_usrman, "usr_dlg_entry_timerestrictions");
      ret->simsess	= lookup_widget (dlg_usrman, "usr_dlg_spin_simsessions");
      ret->loc_allow	= lookup_widget (dlg_usrman, "usr_dlg_entry_allowedlocalip");
      ret->loc_deny	= lookup_widget (dlg_usrman, "usr_dlg_entry_deniedlocalip");
      ret->cli_allow	= lookup_widget (dlg_usrman, "usr_dlg_entry_allowedclientip");
      ret->cli_deny	= lookup_widget (dlg_usrman, "usr_dlg_entry_deniedclientip");
      ret->bw_ul	= lookup_widget (dlg_usrman, "usr_dlg_spin_ulbwidth");
      ret->bw_dl	= lookup_widget (dlg_usrman, "usr_dlg_spin_dlbwidth");
      ret->ul_ratio	= lookup_widget (dlg_usrman, "usr_dlg_spin_ratioup");
      ret->dl_ratio	= lookup_widget (dlg_usrman, "usr_dlg_spin_ratiodown");
      ret->btn_pwchange = lookup_widget (dlg_usrman, "usr_dlg_btn_change_pw");
      ret->btn_savechanges = lookup_widget (dlg_usrman, "usr_dlg_btn_save_changes");
      ret->btn_cancelchanges = lookup_widget (dlg_usrman, "usr_dlg_btn_cancel_changes");
      ret->lbl_invalid_uid = lookup_widget (dlg_usrman, "lbl_invalid_uid");
      ret->lbl_invalid_gid = lookup_widget (dlg_usrman, "lbl_invalid_gid");
      ret->btn_browse_homedir = lookup_widget (dlg_usrman, "btn_browse_homedir");
      ret->btn_browse_fakeroot = lookup_widget (dlg_usrman, "btn_browse_fakeroot");
   }
   return ret;
}

PWInfo *usr_widgets_get_info (void)
{
   PWInfo *i;
   struct usrman_info_widgets *w;
   gchar *tmp, *t_home, *t_root;
   GList *items;
   gint sel;

   i = (PWInfo *) g_malloc0 (sizeof (PWInfo));
   w = usr_get_edit_widgets ();
   if (!i || !w)
     return NULL;
   i->login = g_locale_from_utf8 (gtk_entry_get_text (GTK_ENTRY (w->login)), -1, NULL, NULL, NULL);
   i->gecos = g_locale_from_utf8 (gtk_entry_get_text (GTK_ENTRY (w->realname)), -1, NULL, NULL, NULL);
   i->pwd = g_strdup ((gchar *) g_object_get_data (G_OBJECT (w->btn_pwchange), "usr_password"));

   if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w->chk_chroot)))
   {
      /* Chroot user */
      i->chroot = TRUE;
      t_home = g_filename_from_utf8 (gtk_entry_get_text (GTK_ENTRY (w->home)), -1, NULL, NULL, NULL);
      t_root = g_filename_from_utf8 (gtk_entry_get_text (GTK_ENTRY (w->root)), -1, NULL, NULL, NULL);
      tmp = t_home + strlen (t_home) - 1;
      while (*tmp == '/')
      {
	 *tmp = 0;
	 tmp--;
      }
      tmp = t_root + strlen (t_root) - 1;
      while (*tmp == '/')
      {
	 *tmp = 0;
	 tmp--;
      }
      if (strlen (t_root) > strlen (t_home) || strncmp (t_home, t_root, strlen (t_root)) != 0)
      {
	 pur_log_nfo ("Fake root error: %s not part of %s", t_root, t_home);
	 i->home = g_strdup_printf ("%s/./", t_home);
	 gtk_entry_set_text (GTK_ENTRY (w->root), t_home);
	 /* The fake root that the user specified wasn't part of the home directory and thus invalid! */
	 gui_display_msgdialog (_("Invalid fake root"), 
				_("The fake root you've selected is not a part of the homedirectory. The value has been defaulted to the users homedirectory."), 
				MSGDLG_TYPE_WARNING, dlg_usrman);
      }
      else if (strcmp (t_home, t_root) == 0)
	i->home = g_strdup_printf ("%s/./", t_home);
      else
      {
	 tmp = t_home + strlen (t_root);
	 *tmp = 0;
	 tmp++; /* points to the remaining part of the home-dir */
	 i->home = g_strdup_printf ("%s/./%s", t_root, tmp);
      }
      g_free (t_home);
      g_free (t_root);
   }
   else
   {
      i->chroot = FALSE;
      i->home = g_filename_from_utf8 (gtk_entry_get_text (GTK_ENTRY (w->home)), -1, NULL, NULL, NULL);
      tmp = i->home + strlen (i->home) - 1;
      while (*tmp == '/')
      {
	 *tmp = 0;
	 tmp--;
      }
   }

   if (!g_path_is_absolute (i->home))
   {
      /* Home directory *MUST* be an absolute path, starting with a '/' */
      pur_log_wrn ("Invalid homedir: %s is not absolute!", i->home);
      gui_display_msgdialog (_("Invalid home directory"), 
			     _("The home directory you have specified is not an absolute path (i.e. not starting with a '/'). Information could not be saved"),
			     MSGDLG_TYPE_ERROR, dlg_usrman);
      _free_pwstruct (i);
      return NULL;
   }

   /* UID / GID -- Needs conversion from menuitems to int */
   sel = gtk_option_menu_get_history (GTK_OPTION_MENU (w->e_uid));
   items = g_object_get_data (G_OBJECT (w->e_uid), "items");
   tmp = (gchar *) (g_list_nth (items, sel)->data);
   if (tmp)
     i->uid = atoi (tmp);
   
   sel = gtk_option_menu_get_history (GTK_OPTION_MENU (w->e_gid));
   items = g_object_get_data (G_OBJECT (w->e_gid), "items");
   tmp = (gchar *) (g_list_nth (items, sel)->data);
   if (tmp)
     i->gid = atoi (tmp);
   
   /* Time -- Needs conversion from char to ints */
   tmp = (gchar *) gtk_entry_get_text (GTK_ENTRY (w->timelimit));
   sscanf (tmp, "%u-%u", &(i->time_begin), &(i->time_end));

   i->bw_dl = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w->bw_dl));
   i->bw_ul = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w->bw_ul));
   i->quota_files = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w->quota_files));
   i->quota_size = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w->quota_size));
   i->ul_ratio = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w->ul_ratio));
   i->dl_ratio = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w->dl_ratio));
   i->per_user_max = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w->simsess));
   
   i->allow_local_ip = g_strdup (gtk_entry_get_text (GTK_ENTRY (w->loc_allow)));
   i->deny_local_ip = g_strdup (gtk_entry_get_text (GTK_ENTRY (w->loc_deny)));
   i->allow_client_ip = g_strdup (gtk_entry_get_text (GTK_ENTRY (w->cli_allow)));
   i->deny_client_ip = g_strdup (gtk_entry_get_text (GTK_ENTRY (w->cli_deny)));

   return i;
}

pur_error_t usr_widgets_set_info (gchar *user)
{
   PWInfo *info;
   struct usrman_info_widgets *w;
   gchar *tmp, **t_arr;
   GtkWidget *menu;
   GList *items = NULL;
   gint i;
   gboolean found = FALSE;

   info = usr_get_user_information (user);
   if (!info)
     return E_GENERAL_ERROR;
   w = usr_get_edit_widgets ();
   
   gtk_entry_set_text (GTK_ENTRY (w->login), info->login);
   //   tmp = g_locale_to_utf8 (info->gecos, -1, NULL, NULL, NULL);
   gtk_entry_set_text (GTK_ENTRY (w->realname), info->gecos);
   //   g_free (tmp);
   
   /* Hmm, this should be safe if g_object_get_data returns NULL upon failure */
   g_free (g_object_get_data (G_OBJECT (w->btn_pwchange), "usr_password"));
   if (MISC_VALID_STRING (info->pwd))
     g_object_set_data (G_OBJECT (w->btn_pwchange), "usr_password", g_strdup (info->pwd));
   else
     g_object_set_data (G_OBJECT (w->btn_pwchange), "usr_password", NULL);

   t_arr = g_strsplit (info->home, "/./", 2);
   if (t_arr[1] != NULL)
   {
      /* User is chroot'ed */
      if (t_arr[1])
	tmp = g_strdup_printf ("%s/%s", t_arr[0], t_arr[1]);
      else
	tmp = g_strdup (t_arr[0]);
      gtk_entry_set_text (GTK_ENTRY (w->home), tmp);
      g_free (tmp);
      
      tmp = g_strdup (t_arr[0]);
      gtk_entry_set_text (GTK_ENTRY (w->root), tmp);
      g_free (tmp);
      
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w->chk_chroot), TRUE);
   }
   else
   {
      /* No chroot */
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w->chk_chroot), FALSE);
      gtk_entry_set_text (GTK_ENTRY (w->home), info->home);
      gtk_entry_set_text (GTK_ENTRY (w->root), "");
   }
   g_strfreev (t_arr);
   tmp = g_strdup_printf ("%u-%u", info->time_begin, info->time_end);
   gtk_entry_set_text (GTK_ENTRY (w->timelimit), tmp);
   g_free (tmp);
   
   gtk_entry_set_text (GTK_ENTRY (w->loc_allow), info->allow_local_ip);
   gtk_entry_set_text (GTK_ENTRY (w->loc_deny), info->deny_local_ip);
   gtk_entry_set_text (GTK_ENTRY (w->cli_allow), info->allow_client_ip);
   gtk_entry_set_text (GTK_ENTRY (w->cli_deny), info->deny_client_ip);
   
   /* UID / GID */
   menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (w->e_uid));
   items = (GList *) g_object_get_data (G_OBJECT (w->e_uid), "items");
   i = 0;
   found = FALSE;
   while (items)
   {
      if (info->uid == atoi ((gchar *) items->data))
      {
	 found = TRUE;
	 break;
      }
      i++;
      items = g_list_next (items);
   }
   if (found)
   {
      gtk_option_menu_set_history (GTK_OPTION_MENU (w->e_uid), i);
      gtk_widget_hide (w->lbl_invalid_uid);
   }
   else
   {
      gtk_option_menu_set_history (GTK_OPTION_MENU (w->e_uid), 0);
      gtk_widget_show (w->lbl_invalid_uid);
   }

   menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (w->e_gid));
   items = (GList *) g_object_get_data (G_OBJECT (w->e_gid), "items");
   i = 0;
   found = FALSE;
   while (items)
   {
      if (info->gid == atoi ((gchar *) items->data))
      {
	 found = TRUE;
	 break;
      }
      i++;
      items = g_list_next (items);
   }
   if (found)
   {
      gtk_option_menu_set_history (GTK_OPTION_MENU (w->e_gid), i);
      gtk_widget_hide (w->lbl_invalid_gid);
   }
   else
   {
      gtk_option_menu_set_history (GTK_OPTION_MENU (w->e_gid), 0);
      gtk_widget_show (w->lbl_invalid_gid);
   }


   gtk_spin_button_set_value (GTK_SPIN_BUTTON (w->bw_ul), (gdouble) info->bw_ul);
   gtk_spin_button_set_value (GTK_SPIN_BUTTON (w->bw_dl), (gdouble) info->bw_dl);
   gtk_spin_button_set_value (GTK_SPIN_BUTTON (w->quota_files), (gdouble) info->quota_files);
   gtk_spin_button_set_value (GTK_SPIN_BUTTON (w->quota_size), (gdouble) info->quota_size);
   gtk_spin_button_set_value (GTK_SPIN_BUTTON (w->ul_ratio), (gdouble) info->ul_ratio);
   gtk_spin_button_set_value (GTK_SPIN_BUTTON (w->dl_ratio), (gdouble) info->dl_ratio);
   gtk_spin_button_set_value (GTK_SPIN_BUTTON (w->simsess), (gdouble) info->per_user_max);
   _free_pwstruct (info);
   
   return E_NO_ERROR;
}
   
void usr_widgets_set_active_state (gboolean active)
{
   struct usrman_info_widgets *w;
   
   w = usr_get_edit_widgets ();
   //   gtk_widget_set_sensitive (w->login, active);
   gtk_widget_set_sensitive (w->realname, active);
   gtk_widget_set_sensitive (w->e_uid, active);
   gtk_widget_set_sensitive (w->e_gid, active);
   gtk_widget_set_sensitive (w->chk_chroot, active);
   gtk_widget_set_sensitive (w->home, active);

   if (active && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w->chk_chroot)))
     gtk_widget_set_sensitive (w->root, active);
   else if (!active)
     gtk_widget_set_sensitive (w->root, FALSE);

   gtk_widget_set_sensitive (w->quota_files, active);
   gtk_widget_set_sensitive (w->quota_size, active);
   gtk_widget_set_sensitive (w->timelimit, active);
   gtk_widget_set_sensitive (w->simsess, active);
   gtk_widget_set_sensitive (w->loc_allow, active);
   gtk_widget_set_sensitive (w->loc_deny, active);
   gtk_widget_set_sensitive (w->cli_allow, active);
   gtk_widget_set_sensitive (w->cli_deny, active);
   gtk_widget_set_sensitive (w->bw_ul, active);
   gtk_widget_set_sensitive (w->bw_dl, active);
   gtk_widget_set_sensitive (w->ul_ratio, active);
   gtk_widget_set_sensitive (w->dl_ratio, active);
   gtk_widget_set_sensitive (w->btn_pwchange, active);
   gtk_widget_set_sensitive (w->btn_savechanges, active);
   gtk_widget_set_sensitive (w->btn_cancelchanges, active);
   gtk_widget_set_sensitive (w->btn_browse_homedir, active);
   gtk_widget_set_sensitive (w->btn_browse_fakeroot, active);
}
