#----------------------------------------------------------------------
#      ___       ___ ___   P X E S   Universal  Linux  Thin  Client
#     /__/\\_// /__ /__    Copyright(C) 2003 by Diego Torres Milano
#    /    // \\/__  __/    All rights reserved.  http://pxes.sf.net
#
# Author: Diego Torres Milano <diego@pxes.com.ar>
# $Id: PxesconfigModel.pm,v 1.24 2003/11/13 15:02:38 diego Exp $
#----------------------------------------------------------------------
package PxesconfigModel;
require 5.000; use strict 'vars', 'refs', 'subs';

BEGIN {
	use constant TRUE => 1;
	use constant FALSE => 0;
	use constant RAMDISKSIZE => 16384;
	# WARNING:
	# if you change the ICAVERSION remember to change the size too
	# used in calculate_fssize
	use constant ICAVERSION6 => 'Version 6.30.1052';
	use constant ICA6FSSIZE => 1560;
	use constant ICAVERSION7 => 'Version 7.00.77757';
	use constant ICA7FSSIZE => 1860;

	use constant BLACK => "\e[30m";
	use constant RED => "\e[31m";
	use constant GREEN => "\e[32m";
	use constant ORANGE => "\e[33m";
	use constant BLUE => "\e[34m";
	use constant MAGENTA => "\e[35m";
	use constant CYAN => "\e[36m";
	use constant GRAY => "\e[37m";
	use constant SGR0 => "\e[m";

	use constant MKISO_INIT => 0;
	use constant MKISO_FS => 1;
	use constant MKISO_IMG => 2;

	use constant XVT => 0;

	use constant TA => 1;
	use constant XA => 2;
	use constant TI => -1;
	use constant XI => -2;

	use constant PRPM_INSTALL => 1;
	
	use English;
	use File::Copy;
	use File::Basename;
	use Exporter   ();
   our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS, $AUTOLOAD);

   # set the version for version checking
   $VERSION = do { my @r = (q$Revision: 1.24 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r }; # must be all one line, for MakeMaker

	@ISA         = qw(Exporter);
	@EXPORT      = qw();
	%EXPORT_TAGS = ( );     # eg: TAG => [ qw!name1 name2! ],

	# your exported package globals go here,
	# as well as any optionally exported functions
	@EXPORT_OK   = qw();
}

our @EXPORT_OK;
our $VERSION;
our $AUTOLOAD;

# non-exported package globals go here
# attributes and default values
our $DEBUG;
our @DEBUG;
our $verbose;
our %attributes;
our %local_attributes;
our $tftpboot;
our $prefix;
our $local_confdir;
our $monitors_db;

#
# initialize package globals
#

# debug enabled ?
$DEBUG = 0;
#@DEBUG = ( 'local_httpd_enabled', 'extra_templates_op' );
@DEBUG = ( );

$verbose = FALSE;

# attributes and default values
# attributes keys will become permitted_values
%attributes = (
	# pxes
	pxes_version => undef,
	pxes_release => undef,
	pxes_debug => FALSE,
	session_default => 'xdm',

	# kernel
	kernel_arch => 'i586',

	# required devices
	keyboard_layout => 'us',
	keyboard_model => 'pc101', 
	mouse_device => 'Autodetect',
	mouse_protocol => 'PS/2',
	mouse_protocol_prefix => '',
	mouse_wheel_enabled => FALSE,
	mouse_emulate_3_buttons_enabled => TRUE,
	mouse_emulate_3_timeout => 50,
	mouse_accel_mult => 5,
	mouse_accel_div => 1,
	mouse_accel_thr => 2,
	mouse_handed => undef,
	network_card => 'Autodetect',
	network_card_options => undef,

	# optional local services and devices
	local_printer_enabled => FALSE,
	local_printer_device => undef,
	local_printer_port => 9100,
	local_sound_enabled => FALSE,
	local_esd_enabled => FALSE,
	local_sound_card => 'Autodetect',
	local_sound_card_options => undef,
	local_sound_device => '/dev/sound/dsp',
	local_usb_enabled => TRUE,
	local_floppy_enabled => FALSE,
	local_dvdcdrom_enabled => FALSE,
	local_harddisk_enabled => FALSE,
	supermount_enabled => FALSE,
	nbd_server_enabled => FALSE,
	nbd_floppy_port => 9001,
	nbd_cdrom_port => 9002,
	local_samba_enabled => FALSE,
	local_httpd_enabled => FALSE,
	local_telnetd_enabled => FALSE,
	local_inetd_enabled => FALSE,

	# configuration
	local_configuration_enabled => FALSE,
	remote_configuration_enabled => FALSE,
	configuration_server_name => undef,
	configuration_directory => '/pxes/config',
	local_configuration_directory => '/etc/pxes/config',

	# sessions
	session_telnet => FALSE,
	session_ssh => FALSE,
	session_ltsp => FALSE,
	session_custom => FALSE,
	x_session_local => FALSE,
	x_session_xdm => TRUE,
	x_session_rdp => FALSE,
	x_session_vnc => FALSE,
	x_session_ica => FALSE,
	x_session_nx => FALSE,
	x_session_tarantella => FALSE,
	x_session_linrdp => FALSE,
	x_session_winconnect => FALSE,
	x_session_custom => FALSE,

	# x
	x_version => '3.3.6',
	x_driver => 'Autodetect',
	x_video_modes => '"800x600"',

	x_background_color => 'steelblue4',
	x_font_server_enabled => FALSE,
	x_font_server_name => undef,
	x_font_server_port => 7100,
	x_monitor_autodetect_enabled => FALSE,
	x_horizontal_sync => '30-70',
	x_horizontal_sync_default => '31.5-35.5', # Generic SVGA
	x_vertical_refresh => '50-100',
	x_vertical_refresh_default => '50-61', # Generic SVGA
	x_color_depth => 24,
	x_bpp => 32,
	x_option_noaccel => FALSE,
	x_busid => undef,
	x_option_no_hw_cursor => FALSE,
	x_option_dont_zap => TRUE,
	x_numlock => TRUE,
	x_video_mode_custom => FALSE,
	x_video_mode_custom_x => undef,
	x_video_mode_custom_y => undef,
	x_video_mode_custom_modeline => undef,
	x_remote_connections_enabled => FALSE,
	x_remote_connections_authorized_from => undef,
	x_screensaver_disabled => FALSE,
	x_extra_options => '',

	# rdp
	rdesktop_version => '1.3.0',
	rdesktop => '/bin/rdesktop',
	rdp_ask_enabled => FALSE,
	rdp_version => 5,
	rdp_server_name => undef,
	rdp_port => 3389,
	rdp_domain => undef,
	rdp_user => undef,
	rdp_password => undef,
	rdp_client_hostname => undef,
	rdp_server_version => 'W2K',
	rdp_is_full_screen => TRUE,
	rdp_title => undef,
	rdp_geometry => undef,
	rdp_color_depth => 8,
	rdp_force_bitmap_updates => FALSE,
	rdp_dont_send_motion_events => FALSE,
	rdp_disable_encryption => undef,
	rdp_dont_request_license => undef,
	rdp_hide_wm_decorations => undef,
	rdp_keep_wm_key_bindings => undef,
	rdp_use_private_colormap => undef,
	rdp_sound_enabled => FALSE,
	rdp_start_program_enabled => FALSE,
	rdp_start_program => undef,
	rdp_start_program_folder => undef,

	# ica
	ica_ask_enabled => FALSE,
	ica_installation_directory => '/usr/lib/ICAClient',
	ica_application_servers_ini => '/root/.ICAClient.ro/appsrv.ini',
	ica_client_configuration_ini => '/root/.ICAClient.ro/wfclient.ini',
	ica_server_name => undef,
	ica_server_address => undef,
	ica_server_list => undef,

	# vnc
	vnc_ask_enabled => FALSE,
	vnc_server_name => undef,
	vnc_display => 1,
	vnc_password_file => '/.vnc/passwd',
	vnc_is_full_screen => TRUE,
	vnc_geometry => undef,
	vnc_compress_level => 9,
	vnc_quality => 5,
	vnc_is_shared => FALSE,
	vnc_is_viewonly => FALSE,

	# xdm
	xdm_ask_enabled => FALSE,
	xdm_method => 'broadcast',
	xdm_server_name => undef,
	xdm_server_port => 177,

	# local session
	local_session => '/opt/bin/localsession',

	# ltsp
	ltsp_server_and_root => 'dhcp',	# ask, dhcp, pxes
	ltsp_server_name => undef,
	ltsp_root_directory => undef,

	# telnet
	telnet_ask_enabled => FALSE,
	telnet_server_name => undef,
	telnet_port => 23,

	# linrdp
	linrdp_ask_enabled => FALSE,
	linrdp_server_name => undef,
	linrdp_port => 3389,
	linrdp_user => undef,
	linrdp_password => undef,
	linrdp_domain => undef,
	linrdp_is_full_screen => TRUE,
	linrdp_depth => 24,
	linrdp_geometry => undef,
	linrdp_client_hostname => undef,
	linrdp_sound_enabled => FALSE,
	linrdp_sound_quality => 3,
	linrdp_start_program_enabled => FALSE,
	linrdp_start_program => undef,
	linrdp_start_program_folder => undef,

	# winconnect
	winconnect_ask_enabled => FALSE,
	winconnect_server_name => undef,
	winconnect_port => 3389,
	winconnect_user => undef,
	winconnect_password => undef,
	winconnect_domain => undef,
	winconnect_is_full_screen => TRUE,
	winconnect_depth => 24,
	winconnect_geometry => undef,
	winconnect_sound_enabled => FALSE,
	winconnect_start_program_enabled => FALSE,
	winconnect_start_program => undef,
	winconnect_start_program_folder => undef,
	winconnect_protocol_level => 'RDP51',

	# custom
	custom => undef,

	# inittab
	inittab_startx_action => undef,
	inittab_shell_action => 'off',
	inittab_login_action => 'askfirst',
	inittab_sysinfo_action => 'askfirst',

	# syslog
	syslog_enabled => FALSE,
	syslog_server_name => undef,
	syslog_port => undef,

	# extra
	boot_messages => 1,
	wait_on_error => FALSE,
	prompt_before_x => FALSE,
	prompt_before_client => FALSE,
	dhcp_error_non_fatal => FALSE,
	dhcp_options => undef,
	extra_templates => undef,
	generate_short_hostnames => TRUE,
	ask => undef,
	zeroconf_enabled => FALSE,
	rfb_enabled => FALSE,
	window_manager => '/usr/bin/oroborus',
	include => undef,
	install => undef,
	isoinclude => undef,
	isoinstall => undef,

	# pxes management start
	pxes_management_enabled => FALSE,
	pxes_management_server_name => undef,
	# pxes management end
);

# local parameters
$tftpboot = '/tftpboot/pxes';
$prefix = '/opt'; # pxes base prefix
$local_confdir = "/usr/share/pxesconfig";
$monitors_db = '/usr/share/hwdata/MonitorsDB';

# local_attributes keys will become local_permitted_values
%local_attributes = (
	tftpboot => $tftpboot,
	splash_screen => "$tftpboot/pxes.lss",
	prefix => $prefix,
	kernel => undef,
	initrd => undef,
	initrd_fstype => 'ext2',
	initrd_mountdir => '/tmp/pxes',
	initrd_uncompressed => '/tmp/pxes.initrd',
	initrd_fssize => undef,
	initrd_fssize_p100inc => 0,
	initrd_fssize_inc => 200,
	initrd_inodes => undef,
	initrd_inodes_inc => 200,
	mknbi => FALSE,
	mknbi_cmd => '/usr/bin/mknbi-linux',
	mknbi_args => '--output=%s --append="ramdisk_size=%d initrd=%s ro root=/dev/ram"',
	nbi	=> undef,
	mkiso => FALSE,
	mkiso_cmd => '/usr/bin/mkisofs',
	mkiso_args => '-o %s -V ISOPXES -b boot/isolinux/isolinux.bin -c boot/isolinux/boot.cat -no-emul-boot -boot-load-size 4 -boot-info-table -hide-rr-moved -J -R %s',
	iso => undef,
	isopxes_dir => '/tmp/isopxes',
	isopxes => undef,
	non_root_initialize => FALSE,
	stock_dir => undef,
	arch_dir => undef,
	initialize => FALSE,
	read_configuration => TRUE,
	no_warnings => FALSE,
	local_confdir => $local_confdir,
	local_vnc_password_file => '/root/.vnc/passwd',
	local_ica_installation_directory => '/usr/lib/ICAClient',
	local_ica_configuration_manager => 'wfcmgr',
	local_ica_application_servers_ini =>
		'/root/.ICAClient/appsrv.ini',
	local_ica_client_configuration_ini =>
		'/root/.ICAClient/wfclient.ini',
	local_ica_required_libs_directory => undef,
	local_set_root_password => FALSE,
	root_password => 'pxes',
);

sub new {
	my $proto = shift;
	my $class = ref($proto) || $proto;

	debug("PxesconfigModel::new");

	# default values
	# depends on other values
	$local_attributes{initrd} = 
		"$tftpboot/pxes-$::PXES_VERSION.initrd";
	$local_attributes{nbi} = 
		"$tftpboot/pxes-$::PXES_VERSION.nbi";
	$local_attributes{iso} =
		"/tmp/pxes-$::PXES_VERSION.iso";
	$local_attributes{stock_dir} = 
		"$prefix/pxes-$::PXES_VERSION/stock";
	$local_attributes{arch_dir} = 
		"$prefix/pxes-$::PXES_VERSION/stock/arch";
	$local_attributes{monitors_db} = $monitors_db;
	# FIXME: hardcoded
	$local_attributes{kernel} = $local_attributes{stock_dir} .
		"/arch/i586/kernel/vmlinuz-2.4.20-6pxes.i586";

	my $self = {
		permitted_fields => \%attributes,
		local_permitted_fields => \%local_attributes,
		fields => {},
		local_fields => {},
		changed => FALSE,
		initrd_already_mounted => FALSE,
		initrd_already_uncompressed => FALSE,
		sessions => { 'xdm'=>XA, 'rdp'=>XA, 'telnet'=>TA, 'ica'=>XA,
			'vnc'=>XA, 'local'=>XA, 'ssh'=>TI, 'ltsp'=>TA, 
			'nx'=>XI, 'tarantella'=>XI,
			'linrdp'=>XI, 'winconnect'=>XI,
			'custom'=>TI, },
		error_message => '',
		warn_message => '',
		observers => [],
		};

	# use default values
	foreach my $a (keys %attributes) {
		$self->{fields}->{$a} = $attributes{$a};
	}

	foreach my $a (keys %local_attributes) {
		$self->{local_fields}->{$a} = $local_attributes{$a};
	}

	# start in a known state
	#$self->{changed} = FALSE;

	debug("PxesconfigModel::new: blessing");
	bless $self, $class;

	$self->read_configuration_file('/etc/pxes/pxesconfig.conf', FALSE);
	$self->read_configuration_file("$ENV{HOME}/.pxesconfig.conf", FALSE);

	if ( $#_ == -1 || $_[0] eq '' ) {
		# no arguments passed
	}
	elsif ( $#_ == 0 && $_[0] ne '' ) {
		$self->read_configuration_file($_[0], TRUE);
	}
	else {
		die "PxesconfigModel::new: Invalid number of arguments";
	}

	return $self;
}

sub DESTROY {
}

sub AUTOLOAD {
    my $self = shift;
    my $type = ref($self)
        or die "$self is not an object so we cannot '$AUTOLOAD'\n",
            "We were called from ".join(", ", caller)."\n\n";
    my $name = $AUTOLOAD;
    $name =~ s/.*://;       # strip fully-qualified portion
	 my ($value) = @_;

    if ( exists($self->{permitted_fields}->{$name}) ) {
        # This allows dynamic data methods - see hash fields above
        # eg $class->UI('new_value');
        # or $current_value = $class->UI;
        if (@_) {
				print STDERR "PxesconfigModel::AUTOLOAD setting ", 
					$name, " = ",$value, " oldvalue: ",
					( defined($self->{fields}->{$name}) ) ? 
						$self->{fields}->{$name} : 'UNDEF',
					"\n" if $DEBUG > 4;

				if ( !defined($self->{fields}->{$name}) ||
						$self->{fields}->{$name} ne $value ) {
					warn "changed drived by $name = $value" if $DEBUG > 1;
					$self->{changed} = TRUE;
				}
            return $self->{fields}->{$name} = $value;
        }
		  else {
				print STDERR "getting $name\n"
					if $DEBUG > 5;
            return $self->{fields}->{$name};
        }
	 }
	 elsif ( exists($self->{local_permitted_fields}->{$name}) ) {
        # This allows dynamic data methods - see hash fields above
        # eg $class->UI('new_value');
        # or $current_value = $class->UI;
        if (@_) {
				print STDERR "PxesconfigModel::AUTOLOAD local setting ", 
					$name, " = ",$value, " oldvalue: ",
					$self->{local_fields}->{$name},
					"\n" if $DEBUG > 0;

				if ( !defined($self->{local_fields}->{$name}) ||
						$self->{local_fields}->{$name} ne $value ) {
					warn "changed drived by $name = $value" if $DEBUG > 1;
					$self->{changed} = TRUE;
				}
            return $self->{local_fields}->{$name} = $value;
			}
			else {
				print STDERR "getting $name\n"
					if $DEBUG > 5;
            return $self->{local_fields}->{$name};
			}
	 }
	 elsif ( exists $self->{$name} ) {
	 	if ( @_ ) {
			return $self->{$name} = $value;
		}
		else {
			return $self->{$name};
		}
    }
	 else {
        die "Can't access method `$name' in class $type\n".
            "We were called from ".join(", ", caller)."\n\n";

    }
} # End of sub AUTOLOAD


sub debug($$$) {
	my ($msg, $level, $force) = @_;
	my $color = MAGENTA;

	if ( ! defined($level) ) {
		$level = 1;
	}

	if ( $level == 1 ) {
		$color = RED;
	}
	elsif ( $level == 2 ) {
		$color = MAGENTA;
	}
	elsif ( $level == 3 ) {
		$color = CYAN;
	}
	elsif ( $level == 4 ) {
		$color = ORANGE;
	}

	warn $color . $msg . SGR0 . "\n" if $force || $DEBUG > $level;
}

sub fatal($$) {
	my ($self, $msg) = @_;

	warn "self=$self" if defined($self) && $DEBUG;
	$self->umount_initrd() if defined($self);

	die "Fatal error:\n" .
		$msg, "\n";
}

sub error($$) {
	my ($self, $msg) = @_;

	print STDERR "Error:\n" .
		$msg, "\n";
}

sub initial_checks($) {
	my $self = shift;

	debug("PxesconfigModel::initial_checks()", 3, FALSE);

	if ( $self->is_mounted() ) {
		$self->fatal("The filesystem " . $self->initrd_mountdir() . 
			" is already mounted.\n" .
			"Manually unmount it and try again.");
	}

	if ( ! $self->check_fstab() ) {
		$self->fatal("Needed entry missing from /etc/fstab.\n" .
			"Add an entry like:\n" .
			$self->fstab_entry('str') . "\n"
			);
	}

	if ( ! -d $self->stock_dir() ) {
		$self->fatal("Cannot find stock dir: " . $self->stock_dir());
	}

}

=pod

=head2
PxesconfigModel::do_read_configuration:
=cut
sub do_read_configuration($) {
	my $self = shift;

	if ( $self->read_configuration() ) {
		if ( $self->read_configuration_file($self->initrd_mountdir() . 
			"/etc/pxes.conf", FALSE) < 0 ) {
				return -1;
		}
		$self->notify();
	}

	return 0;
}

sub read_configuration_file($$) {
	my $self = shift;
	my($filename, $mandatory) = @_;

	debug("PxesconfigModel::read_configuration_file($filename," .
		"$mandatory)", 3, FALSE);

	my $h = hashFromFile($self, $filename, $mandatory);
	foreach my $k (keys %$h) {
		# change the key to lower case
		my $a = lc($k);
		if ( exists($self->{permitted_fields}->{$a}) ) {
			#$self->{fields}->{$a} = $h->{$k};
			$self->$a($h->{$k});
			print "PxesconfigModel::read_confoguration_file: ",
				"setting self->{fields}->{$a} = ",$h->{$k},"\n"
				if $DEBUG > 5;
		}
		elsif ( exists($self->{local_permitted_fields}->{$a}) ) {
			#$self->{local_fields}->{$a} = $h->{$k};
			$self->$a($h->{$k});
			print "PxesconfigModel::read_confoguration_file: ",
				"setting self->{local_fields}->{$a} = ",$h->{$k},"\n"
				if $DEBUG > 5;
		}
		else {
			#print "permitted_fields:\n";
			#foreach my $k (sort keys %{$self->{permitted_fields}} ) {
			#	print "'$k' ", (exists($self->{permitted_fields}->{$k})), "\n";
			#}
			warn "not permitted key '$a'";
		}
	}
}

sub initial($) {
	my $self = shift;

	print STDERR "PxesconfigModel::initial()\n"
		if $DEBUG > 0;

	if ( $self->read_configuration() ) {
		if ( $self->uncompress_initrd() < 0 ) {
			return -1;
		}

		if ( $self->mount_initrd() < 0 ) {
			return -1;
		}

		if ( $self->do_read_configuration() < 0 ) {
			return -1;
		}
	}
	
	return 0;

}

sub steps_to_finish($) {
	my $self = shift;
	my $steps = 9;

	$steps++ if $self->initialize();
	$steps++ if $self->mknbi();
	$steps+= 3 if $self->mkiso();

	return $steps;
}

sub finish($) {
	my $self = shift;
	my ($message_type) = @_;

	if ( defined($message_type) && $message_type eq 'text' ) {
		$verbose = TRUE;
	}

	debug("PxesconfigModel::finish()", 3, FALSE);

	if ( $self->initialize() ) {
		print STDERR "PXesconfigModel::finish: initializing initrd\n"
			if $DEBUG > 1;
		if ( ! $self->check_privileges() ) {
			$self->notify('error', 
				"Insuficient privileges to initialize ram disk.");
			return -1;
		}

		if ( $self->initialize_initrd() < 0 ) {
			return -1;
		}

		$self->notify('info', "Initial ram disk has been initialized.");
	}

	print STDERR "PXesconfigModel::finish: uncompressing initrd\n"
		if $DEBUG > 1;
	if ( $self->uncompress_initrd() < 0 ) {
		return -1;
	}
	$self->notify('info', "Initial ram disk has been uncompressed.");

	print STDERR "PXesconfigModel::finish: mounting initrd\n"
		if $DEBUG > 1;
	if ( $self->mount_initrd() < 0 ) {
		return -1;
	}
	$self->notify('info', "Initial ram disk has been mounted.");

	print STDERR "PXesconfigModel::finish: copying extra files\n"
		if $DEBUG > 1;
	if ( $self->copy_extra_files() < 0 ) {
		$self->error_message("Copying extra files: " .
			$self->error_message());
		return -1;
	}
	$self->notify('info', "Extra files has been copied.");

	print STDERR "PXesconfigModel::finish: writing configuration\n"
		if $DEBUG > 1;
	if ( $self->write_configuration() < 0 ) {
		return -1;
	}
	if ( $self->write_release() < 0 ) {
		return -1;
	}
	$self->notify('info', "Configuration has been written.");

	print STDERR "PXesconfigModel::finish: parsing inittab\n"
		if $DEBUG > 1;
	if ( $self->parse_inittab() < 0 ) {
		return -1;
	}
	$self->notify('info', "Inittab has been parsed.");
	
	$self->delete_lost_found();

	if ( $self->local_set_root_password() ) {
		if ( $self->set_root_password() < 0 ) {
			$self->fatal("Cannot set root pasword");
		}
	}

	if ( $self->mkiso() ) {
		if ( $self->do_mkiso(MKISO_INIT) != 0 ) {
			$self->fatal("Cannot make an ISO bootable image");
		}
		$self->notify('info', "ISO bootable image has been initialized.");

		if ( $self->do_mkiso(MKISO_FS) != 0 ) {
			$self->fatal("Cannot make an ISO bootable image");
		}
		$self->notify('info', "ISO bootable image has been created.");
	}

	if ( $self->umount_initrd() < 0 ) {
		$self->fatal("There were problems unmounting");
	}
	$self->notify('info', "Initial ramdisk has been unmounted.");

	if ( $self->compress_initrd() < 0 ) {
		$self->fatal("There were problems compressing initrd");
	}
	$self->notify('info', "Initial ramdisk has been compressed.");

	if ( $self->copy_kernel() < 0 ) {
		$self->fatal("There were problems copying kernel");
	}
	$self->notify('info', "Selected kernel has been copied.");

	if ( $self->mknbi() ) {
		if ( $self->do_mknbi() != 0 ) {
			$self->fatal("Cannot make a network bootable image");
		}
		$self->notify('info', "Network bootable image has been created.");
	}

	if ( $self->mkiso() ) {
		if ( $self->do_mkiso(MKISO_IMG) != 0 ) {
			$self->fatal("Cannot make an ISO bootable image");
		}
		$self->notify('info', "ISO bootable image has been created.");
	}

	print STDERR "PXesconfigModel::finish: SUCCESS", SGR0, "\n"
		if $DEBUG > 1;
	$self->notify('info', "Configuration complete.");
	return 0;
}

=pod
=head2
check_and_copy

$var		variable to extract file name from or path (if it has '/')
$dest		destination file in copy (automatically add mountdir) unless
			$dontaddmountdir is true
$dontaddmountdir	don't add the mountdir

check that the referenced file is readable, creates the needed path to
dest, and copy the file.
=cut
sub check_and_copy($$$$) {
	my $self = shift;
	my ($var, $dest, $dontaddmountdir) = @_;
	my $val;

	$dontaddmountdir = FALSE unless defined($dontaddmountdir);

	debug("PxesconfigModel::check_and_copy($var, $dest, $dontaddmountdir)", 3, FALSE);

	if ( $var =~ '/' ) {
		# if var seems to be a pathname
		$val = $var;
	}
	else {
		# otherwise, should be a file name containing the pathname
		$val = eval "\$self->$var()";
	}

	my @g = glob($val);
	if ( &File::Glob::GLOB_ERROR != 0 ) {
		$self->error_message($!);
		return -1;
	}

	if ( $#g == 0 ) {
		my $f = $g[0];
		if ( defined($f) && $f ne $val ) {
			# glob found
			if ( $dest =~ /^(.*)\/([^\/]+)$/ ) {
				# path + glob
				$dest = $1 . $f;
			}
		}
	}
	elsif ( $#g > 0 ) {
		# glob expande to many files
	}
	else {
		$self->error_message($!);
		return -1;
	}
		
	# safer, add mountdir
	warn "**** dontaddmountdir=$dontaddmountdir ****" if $DEBUG;
	$dest = $self->initrd_mountdir() . $dest unless $dontaddmountdir;

	foreach my $f (@g) {

		print STDERR "check_and_copy: val='$f' dest='$dest'\n"
			if $DEBUG > 0;

		if ( ! $dest ) {
			$self->error_message("Invalid destination.");
			return -1;
		}

		if ( $val ) {
			if ( ! -r $f ) {
				$self->error_message("Cannot access " . $f);
				return -1;
			}
		
			$self->create_path_to($dest);

			my $cmd = "cp -a " .  $f . " " . $dest;

			my $m = `$cmd 2>&1`;
			if ( $? ) {
				$self->error_message($m);
				return -1;
			}
		
		}
	}

	return 0;
}

sub prpm($$$$$) {
	my $self = shift;
	my ($op, $prpm, $dest, $dontaddmountdir) = @_;
	my $val;

	$dontaddmountdir = FALSE unless defined($dontaddmountdir);
	$dest = $self->initrd_mountdir() . $dest unless $dontaddmountdir;

	$prpm = $self->stock_dir() . "/extra/" . $prpm;
	debug("PxesconfigModel::prpm($op, $prpm, $dest, $dontaddmountdir)", 3, TRUE);

	if ( $op == PRPM_INSTALL ) {
		$self->create_path_to($dest);
		my $cmd = 'cat ' . $prpm . '| (cd ' . $dest . '; tar xzf - )';
		my $m = `$cmd 2>&1`;
		#print STDERR $m;
		if ( $? ) {
			$self->error_message($m);
			return -1;
		}
	}

	return 0;
}

sub check_ica_version($) {
	my ($self, $local_icaroot) = @_;
	my $l = (length(ICAVERSION6) > length(ICAVERSION7)) ?
		length(ICAVERSION6) : length(ICAVERSION7);

	if ( ! defined($local_icaroot) || ! $local_icaroot ) {
		$self->error_message("Empty or invalid ICAROOT");
		return -1;
	}

	if ( ! -d $local_icaroot ) {
		$self->error_message("$local_icaroot directory doesn't exist");
		return -1;
	}

	my $wfica = $local_icaroot . "/wfica";
	if ( ! -r $wfica ) {
		$self->error_message("Cannot access $wfica");
		return -1;
	}

	if ( ! open(PIPE, "strings -n $l $wfica|") ) {
		$self->error_message("Cannot open pipe");
		return -1;
	}

	my $found = 0;
	my $icaversion6 = ICAVERSION6;
	my $icaversion7 = ICAVERSION7;
	while ( <PIPE> ) {
		chomp;
		$found = 6 if /$icaversion6/;
		$found = 7 if /$icaversion7/;
		last if $found;
	}
	close(PIPE);

	if ( ! $found ) {
		$self->error_message("Required ICA Client versions '" .
			ICAVERSION6 . "or" . ICAVERSION7 . "' not found");
		return -1;
	}

	return $found;
}

sub copy_extra_files($) {
	my $self = shift;

	# XFree86
	if ( $self->initialize() && $self->x_sessions('selected') ) {
		my $from = $self->stock_dir() . "/extra/XFree86-" .
			$self->x_version() . "/.";
		my $to = "/";
		if ( $self->check_and_copy($from, $to) < 0 ) {
			warn $self->error_message();
			return -1;
		}

		#my $prpm = "XFree86-" .  $self->x_version() . '.prpm';
		#my $to = "/";
		#if ( $self->prpm(PRPM_INSTALL, $prpm, $to) < 0 ) {
		#	warn $self->error_message();
		#	return -1;
		#}
	}

	if ( $self->x_session_ica() ) {
		# ica session
		if ( $self->initialize() ) {
			# ica client files
			my $f = $self->local_confdir() . "/ica/icaclient.files";
			if ( -r $f ) {
				if ( ! open(ICA, $f) ) {
					$self->error_message("Cannot open $f");
					return -1;
				}

				my $local_icaroot =
					$self->local_ica_installation_directory();
				my $icaroot =
					$self->ica_installation_directory();

				if ( ! -d $local_icaroot ) {
					$self->error_message(
						"$local_icaroot is not a directory");
					return -1;
				}
					
				if ( $self->check_ica_version($local_icaroot) < 0 ) {
					return -1;
				}

				while ( <ICA> ) {
					next if /^\s*#/;
					chomp;
					my $from = $_;
					my $to = $_;

					$from =~ s/^\$\{ICAROOT\}/$local_icaroot/;
					$to 	=~ s/^\$\{ICAROOT\}/$icaroot/;

					if ( $self->check_and_copy($from, $to) < 0 ) {
						return -1;
					}
				}
				close(ICA);
			}
			else {
				$self->error_message("Cannot read $f");
				return -1;
			}


			# ica client libs
			my $eid = $self->stock_dir() . "/extra/ica";
			$f = $self->local_confdir() . "/ica/icaclient.requires";
			if ( -r $f ) {
				if ( ! open(LIB, $f) ) {
					$self->error_message("Cannot open $f");
					return -1;
				}

				while ( <LIB> ) {
					next if /^\s*#/;
					chomp;
					my $from = $_;
					my $to = $_;

					$from =~ s/^\$\{EXTRAICA\}/$eid/;
					$to 	=~ s/^\$\{EXTRAICA\}//;

					if ( $self->check_and_copy($from, $to) < 0 ) {
						return -1;
					}
				}
				close(LIB);
			}
			else {
				$self->error_message("Cannot read $f");
				return -1;
			}

			# ica optional files
			if ( $self->x_session_local() ) {
				my $f = $self->local_confdir() . "/ica/localsession.files";
				if ( -r $f ) {
					if ( ! open(ICA, $f) ) {
						$self->error_message("Cannot open $f");
						return -1;
					}

					my $local_icaroot =
						$self->local_ica_installation_directory();
					my $icaroot =
						$self->ica_installation_directory();

					while ( <ICA> ) {
						next if /^\s*#/;
						chomp;
						my $from = $_;
						my $to = $_;

						$from =~ s/^\$\{ICAROOT\}/$local_icaroot/;
						$to 	=~ s/^\$\{ICAROOT\}/$icaroot/;

						if ( $self->check_and_copy($from, $to) < 0 ) {
							return -1;
						}
					}
					close(ICA);
				}
			}


			# ica other files needed (maybe empty)
			if ( $self->check_and_copy('/dev/null',
					'/.netscape/preferences.js') < 0 ) {
				return -1;
			}
		}

		if ( $self->check_and_copy('local_ica_application_servers_ini',
				$self->ica_application_servers_ini()) < 0 ) {
			return -1;
		}
		if ( $self->check_and_copy('local_ica_client_configuration_ini',
				$self->ica_client_configuration_ini()) < 0 ) {
			return -1;
		}
	}

	if ( $self->x_session_vnc() ) {
		# vnc session
		if ( $self->local_vnc_password_file() ) {
			if ( $self->check_and_copy('local_vnc_password_file',
					$self->vnc_password_file()) < 0 ) {
				return -1;
			}
		}
		else {
			$self->vnc_password_file('');
		}
	}

	# local session
	if ( $self->x_session_local() ) {
		if ( $self->initialize() ) {
			my $from = $self->stock_dir() . "/extra/local/.";
			my $to = "/";
			if ( $self->check_and_copy($from, $to) < 0 ) {
				warn $self->error_message();
				return -1;
			}
		}
	}

	# rfb
	if ( $self->rfb_enabled() ) {
		if ( $self->initialize() ) {
			my $from = $self->stock_dir() . "/extra/rfb/.";
			my $to = "/";
			if ( $self->check_and_copy($from, $to) < 0 ) {
				warn $self->error_message();
				return -1;
			}
		}
	}

	# nomachine nx client
	if ( $self->x_session_nx() ) {
		if ( $self->initialize() ) {
			my $from = $self->stock_dir() . "/extra/nxclient/.";
			my $to = "/";
			if ( $self->check_and_copy($from, $to) < 0 ) {
				warn $self->error_message();
				return -1;
			}
		}
	}

	# linrdp
	if ( $self->x_session_linrdp() ) {
		if ( $self->initialize() ) {
			my $from = $self->stock_dir() . "/extra/linrdp/.";
			my $to = "/";
			if ( $self->check_and_copy($from, $to) < 0 ) {
				warn $self->error_message();
				return -1;
			}
		}
	}

	# winconnect
	if ( $self->x_session_winconnect() ) {
		if ( $self->initialize() ) {
			my $from = $self->stock_dir() . "/extra/winconnect/.";
			my $to = "/";
			if ( $self->check_and_copy($from, $to) < 0 ) {
				warn $self->error_message();
				return -1;
			}
		}
	}

	# samba
	if ( $self->local_samba_enabled() ) {
		if ( $self->initialize() ) {
			my $from = $self->stock_dir() . "/extra/samba/.";
			my $to = "/";
			if ( $self->check_and_copy($from, $to) < 0 ) {
				warn $self->error_message();
				return -1;
			}
		}

		if ( $self->local_set_root_password() ) {
			my $conf = $self->local_confdir() . "/samba/smb.conf";
			my $cmd = "smbpasswd -L -c $conf -a root " .
				$self->root_password() .
				" >/dev/null 2>&1";
			warn "cmd=$cmd" if $DEBUG;
			my $m = `$cmd`;
			if ( $? ) {
				$self->error_message($m);
				return -1;
			}

			my $from = "/tmp/smbpasswd";
			my $to = "/etc/samba/smbpasswd.ro";
			if ( $self->check_and_copy($from, $to) < 0 ) {
				warn $self->error_message();
				return -1;
			}
		}
	}

	# optional files
	if ( $self->initialize() ) {
		my $from = $self->stock_dir() . "/extra/opt/.";
		my $to = "/";
		if ( $self->check_and_copy($from, $to) < 0 ) {
			warn $self->error_message();
			return -1;
		}
	}

	# debug tools
	if ( $self->pxes_debug() ) {
		my $from = $self->stock_dir() . "/extra/debug/.";
		my $to = "/";
		if ( $self->check_and_copy($from, $to) < 0 ) {
			warn $self->error_message();
			return -1;
		}
	}

	# included directories
	if ( $self->include() ) {
		foreach my $d (@{$self->include()}) {
			warn "#### including $d";
			my $from = $d . "/.";
			my $to = "/";
			if ( $self->check_and_copy($from, $to) < 0 ) {
				warn $self->error_message();
				return -1;
			}
		}
	}

	return 0;
}

sub write_configuration($) {
	my $self = shift;

	$self->write_configuration_file($self->initrd_mountdir() .
		"/etc/pxes.conf");

	$self->write_modules_configuration($self->initrd_mountdir() .
		"/etc/modules.conf.ro");
}

sub write_release($) {
	my $self = shift;

	$self->write_release_file($self->initrd_mountdir() .
		"/etc/pxes-release");
}

sub create_path_to($$) {
	my ($self, $filename) = @_;

	debug("PxesconfigModel::create_path_to($filename)", 3, FALSE);

	$filename =~ m@(.*)/[^/]+$@;
	my $dir = $1;
	if ( $dir && ! -d $dir ) {
		mkdir($dir) or
			$self->fatal("Cannot create directory '$dir': $!");
	}
}

sub pxescomment($) {
	my ($self) = @_;

	my $date = `date`;
	my $s = <<"EOF";
#----------------------------------------------------------------------
#    ___       ___ ___
#   /__/\\\\_// /__//__
#  /    // \\\\/__  __/
#
#  PXES Universal Linux Thin Client - http://pxes.sf.net
#  Copyright(C) 2001-2003 by Diego Torres Milano
#
# Author: Diego Torres Milano <diego\@in3.com.ar>
#----------------------------------------------------------------------
# PXES Configuration
# Version: $self->{fields}->{pxes_version}
# Date: $date

EOF
	return $s;
}

sub write_configuration_file($$) {
	my $self = shift;
	my($filename) = @_;

	$self->create_path_to($filename);

	my $umask = umask(02);
	open(F, ">$filename") or
		die "Cannot open '$filename' for writing: $!";
	umask($umask);
	chmod(0664, $filename);

	$self->pxes_version($::PXES_VERSION);
	$self->pxes_release($::PXES_RELEASE);

	if ( $self->extra_templates() ) {
		warn "###### extra_templates=" . $self->extra_templates()
			if $DEBUG > 2 || grep(/extra_templates_op/, @DEBUG);
		warn "###### extra_templates=" . "@{$self->extra_templates()}"
			if $DEBUG > 2 || grep(/extra_templates_op/, @DEBUG);
	}
	else {
		warn "###### extra_templates=UNDEFINED" 
			if $DEBUG > 2 || grep(/extra_templates_op/, @DEBUG);
	}

	my $s;
	foreach my $k ( sort keys %{$self->{fields}} ) {
		my $v = $self->{fields}->{$k};
		if ( ref($v) eq 'ARRAY' ) {
			my @a = @{$v};
			warn "a=@a" if $DEBUG > 4;
			$v = "";
			for (my $i=0; $i < $#a; $i++) {
				# WARNING / FIXME
				# array elements cannot contain quotes
				my $q = '';
				$q = '"' if ( $a[$i] !~ /"/ );
				$v .= "${q}${a[$i]}${q} ";
			}
			my $q = '';
			$q = '"' if ( $a[$#a] !~ /"/ );
			$v .= "${q}${a[$#a]}${q}";
		}
				
		warn "v=$v" if $DEBUG > 4;
		$v = '' if ! defined($v);

		$s .= uc($k) . "='" . $v . "'\n";
		$s .= "export " . uc($k) . "\n\n";
	}

	print F ":\n",$self->pxescomment(),"\n";
	print F $s;
	close(F);

	print STDERR "File $filename has been written.\n"
		if $DEBUG > 0;
}

sub write_release_file($$) {
	my $self = shift;
	my($filename) = @_;

	$self->create_path_to($filename);

	my $umask = umask(02);
	open(F, ">$filename") or
		die "Cannot open '$filename' for writing: $!";
	umask($umask);
	chmod(0664, $filename);

	print F "PXES ", $self->pxes_version(), "-", $self->pxes_release(),
		"\n";
	close(F);

	print STDERR "File $filename has been written.\n"
		if $DEBUG > 0;
}

sub write_local_configuration_file($$) {
	my $self = shift;
	my($filename) = @_;

	$self->create_path_to($filename);

	my $umask = umask(02);
	open(F, ">>$filename") or
		die "Cannot open '$filename' for appending: $!";
	umask($umask);
	chmod(0664, $filename);

	my $s;
	foreach my $k ( sort keys %{$self->{local_fields}} ) {
		my $v = $self->{local_fields}->{$k};
		if ( ref($v) eq 'ARRAY' ) {
			my @a = @{$v};
			$v = "";
			for (my $i=0; $i < $#a; $i++) {
				$v .= "$a[$i] ";
			}
			$v .= $a[$#a];
		}
				
		$v = '' if ! defined($v);

		$s .= uc($k) . "='" . $v . "'\n";
	}

	print F "#\n# LOCAL CONFIGURATION\n#\n";
	print F $s;
	close(F);

	print STDERR "File $filename has been written.\n"
		if $DEBUG > 0;
}

sub dump_configuration($$) {
	my $self = shift;
	my($filename) = shift || '-';
	
	my $retval = $self->write_configuration_file($filename);
	$retval |= $self->write_local_configuration_file($filename);

	return $retval;
}

sub write_modules_configuration($$) {
	my $self = shift;
	my($filename) = @_;
	my $permanent = '';

	if ( -f $filename ) {
		open(F, "<$filename") or
			$self->fatal("Cannot open '$filename' for reading: $!");
		while ( <F> ) {
			$permanent .= $_ if /# .*permanent$/;
		}
		close(F);
	}

	my $umask = umask(02);
	open(F, ">$filename") or
		$self->fatal("Cannot open '$filename' for writing: $!");
	umask($umask);
	chmod(0664, $filename);

	print F $self->pxescomment(), "\n";

	if ( $permanent eq '' ) {
		$permanent = "alias net-pf-16 af_netlink # permanent\n";
	}

	print STDERR "\tpermanent=$permanent\n"
		if $DEBUG > 2;

	print F $permanent,"\n";

	my $nc = $self->network_card();
	if ( $nc ne 'Autodetect' ) {
		# Autodetection is handled by autodetect script in distro
		print STDERR "alias eth0 ", $nc, "\n"
			if $DEBUG > 2;
		print F "alias eth0 ", $nc, "\n";
	}

	if ( $self->local_sound_enabled() ) {
		my $sc = $self->local_sound_card();

		if ( $sc ne 'Autodetect' ) {
			# Autodetection is handled by autodetect script in distro
			print F "alias sound-slot-0 ", $sc, "\n";
			# FIXME
			# very bad place
			if ( $sc eq 'sb' ) {
				print F "alias synth0 awe_wave\n";
			}
			elsif ( $sc eq 'i810_audio' ) {
				;
			}
		}

		print F <<EOF;
post-install sound-slot-0 /bin/aumix-minimal -f /etc/.aumixrc -L >/dev/null 2>&1 || :
pre-remove sound-slot-0 /bin/aumix-minimal -f /etc/.aumixrc -S >/dev/null 2>&1 || :
EOF
	}

	if ( $self->local_printer_enabled() ) {
		print F "alias parport_lowlevel parport_pc\n";
	}

	if ( $self->local_usb_enabled() ) {
		print F "alias usb-controller usb-uhci\n";
	}

	close(F);
	print STDERR "File $filename has been written.\n"
		if $DEBUG > 0;
}

sub do_mknbi($) {
	my ($self) = @_;

	debug("PxesconfigModel::do_mknbi()", 3, FALSE);

	if ( ! -x $self->mknbi_cmd() ) {
		$self->error("Cannot find " . $self->mknbi_cmd() . "\n" .
			"Maybe you need to install mknbi tools from Etherboot");
		return -1;
	}

	$self->initrd() =~ /([^\/]+)$/;
	my $ibn = $1;
	if ( ! $ibn ) {
		$self->error("Cannot determine initrd base name from " .
			$self->initrd());
		$ibn='unknown';
	}
	my $ramdisksize = ( RAMDISKSIZE > $self->initrd_fssize() ) ?
		RAMDISKSIZE : $self->initrd_fssize();
	my $args = '';
	if ( ref($self->mknbi_args()) eq 'ARRAY' ) {
		foreach my $a (@{$self->mknbi_args()}) {
			$args .= $a;
		}
	}
	else {
		$args = $self->mknbi_args();
	}
	warn "********* args='$args'" if $DEBUG;
	$args = sprintf($args, $self->nbi(), $ramdisksize, $ibn);
	my $cmd = $self->mknbi_cmd() . " " . $args . " " .
		$self->kernel() . " " . $self->initrd();

	print STDERR "running '$cmd'\n" if $DEBUG;
	return system($cmd);
}

sub do_mkiso($) {
	my ($self, $type) = @_;
	my $cmd;
	my $m;

	debug("PxesconfigModel::do_mkiso($type)", 1, FALSE);

	if ( ! $self->check_privileges() && ! $DEBUG ) {
		$self->error("Making ISO filesystem is a privileged operation.");
		return -1;
	}

	if ( ! -x $self->mkiso_cmd() ) {
		$self->error("Cannot find " . $self->mkiso_cmd() . "\n" .
			"Maybe you need to install mkisofs");
		return -1;
	}

	if ( $type == MKISO_INIT ) {
		debug("Pxesconfig::do_mkiso: INITIALIZING", 1, FALSE);
		$cmd = "rm -rf " . $self->isopxes_dir();
		print STDERR MAGENTA,"running '$cmd'\n", SGR0 if $DEBUG;
		$m = `$cmd 2>&1`;
		if ( $? ) {
			$self->error_message($m);
			return -1;
		}

		if ( mkdir($self->isopxes_dir()) < 0 ) {
			$self->error_message($!);
			return -1;
		}
	}
	elsif ( $type == MKISO_FS ) {
		debug("Pxesconfig::do_mkiso: FILESYSTEM", 1, FALSE);
		$cmd = "cp -a " .  $self->initrd_mountdir() . "/. " . 
			$self->isopxes_dir();
		print STDERR MAGENTA,"running '$cmd'\n", SGR0 if $DEBUG;
		$m = `$cmd 2>&1`;
		if ( $? ) {
			warn "Error running cmd=$? $m";
			$self->error_message($m . " Error($?)");
			return -1 unless $DEBUG;
			warn "ERROR copying files but not returning due to DEBUG";
		}

		if ( $self->isoinclude() ) {
			foreach my $d (@{$self->isoinclude()}) {
				$cmd = "cp -a " .  $d . "/. " . $self->isopxes_dir();
				print STDERR MAGENTA,"running '$cmd'\n", SGR0 if $DEBUG;
				$m = `$cmd 2>&1`;
				if ( $? ) {
					warn "Error running cmd=$? $m";
					$self->error_message($m . " Error($?)");
					return -1 unless $DEBUG;
					warn "ERROR copying files but not returning due to DEBUG";
				}
			}
		}
	}
	elsif ( $type == MKISO_IMG ) {
		debug("Pxesconfig::do_mkiso: IMAGE", 1, FALSE);
		my $dir = "/boot/isolinux/";
		my $from = $self->stock_dir() . "/extra/isopxes";
		$from = $self->isopxes() if ( $self->isopxes() );
		$from .= "/.";
		my $to = $self->isopxes_dir();

		if ( $self->check_and_copy($from, $to, TRUE) < 0 ) {
			warn $self->error_message();
			return -1;
		}

		if ( $self->check_and_copy($self->kernel(), $to . $dir . "/vmlinuz", TRUE)
				< 0 ) {
			warn $self->error_message();
			return -1;
		}

		if ( $self->check_and_copy($self->initrd(), $to . $dir . "/pxes.img", TRUE)
				< 0 ) {
			warn $self->error_message();
			return -1;
		}

		if ( $self->parse_isolinux_cfg() < 0 ) {
			warn $self->error_message();
			return -1;
		}

		my $args = sprintf($self->mkiso_args(), $self->iso(),
			$self->isopxes_dir());
		$cmd = $self->mkiso_cmd() . " " . $args . " 2> /tmp/isopxes.err";

		print STDERR MAGENTA,"running '$cmd'\n", SGR0 if $DEBUG;
		return system($cmd);
	}

	return 0;
}

sub write_tftp_configuration($) {
	my ($self) = @_;

	print STDERR "\nPxesconfigModel::write_tftp_configuration " .
		"NOT IMPLEMENTED YET\n";
}

sub write_pxelinux_configuration($) {
	my ($self) = @_;

	print STDERR "\nPxesconfigModel::write_pxelinux_configuration " .
		"NOT IMPLEMENTED YET\n";
}

sub check_privileges($) {
	my ($self) = @_;

	return ($self->non_root_initialize() || $self->login() eq 'root');
}

sub fstab_entry($$) {
	my ($self, $type) = @_;

	if ( $type eq 're' ) {
		my $re = '^' . $self->initrd_uncompressed() . '\s+' .
			$self->initrd_mountdir() .  '\s+' .
			$self->initrd_fstype() . '\s+' . 'loop,noauto,user,owner';

		return $re;
	}
	elsif ( $type eq 'str' ) {
		my $str = $self->initrd_uncompressed() . "\t" .
			$self->initrd_mountdir() .  "\t" .
			$self->initrd_fstype() . "\t" . 'loop,noauto,user,owner' .
			"\t0 0"
			;

		return $str;
	}
	else {
		$self->fatal("Invalid argument to fstab_entry: $type");
	}
}

sub check_fstab($) {
	my ($self) = @_;

	debug("PxesconfigModel::check_fstab()", 3, FALSE);

	my $re = $self->fstab_entry('re');

	print STDERR "PxesconfigModel::check_fstab:\n"
		if $DEBUG > 2;
	print STDERR "\tsearching for '$re'\n"
		if $DEBUG > 2;

	open(F, "/etc/fstab") or 
		$self->fatal("Cannot open /etc/fstab");
	my $found = FALSE;
	while ( <F> ) {
		if ( /$re/ ) {
			$found = TRUE;
			last;
		}
	}

	close(F);

	print STDERR "\treturning $found\n"
		if $DEBUG > 2;
	return $found;
}

sub is_mounted($) {
	my ($self) = @_;
	my $u = $self->initrd_uncompressed();

	debug("PxesconfigModel::is_mounted()", 3, FALSE);

	open(P, "mount|") or die;
	while ( <P> ) {
		if ( /^$u/ ) {
			close(P);
			return TRUE;
		}
	}
	close(P);
	return FALSE;
}

sub uncompress_initrd($) {
	my ($self) = @_;

	print STDERR "PxesconfigModel::uncompress_initrd:\n" if $DEBUG > 1;
	print STDERR "Uncompressing ", $self->initrd(), " to ", 
		$self->initrd_uncompressed(), "\n"
		if $DEBUG > 0;

	if ( ! $self->initrd_already_uncompressed() ) {
		my $i = $self->initrd();
		my $u = $self->initrd_uncompressed();

		if ( ! -r $i ) {
			$self->fatal("Cannot read initrd '" . $i . "'");
		}

		if ( -f $u && ! -w $u ) {
			$self->fatal("Cannot write uncompressed initrd '" . $u . "'");
		}

		my $r = system("gunzip -c < $i  >| $u");

		if ( $r/256 ) {
			$self->fatal("Uncompressing $i to $u ($r:$!)");
		}

		$self->initrd_already_uncompressed(TRUE);
	}

	return 0;
}

sub mount_initrd($) {
	my ($self) = @_;

	debug("PxesconfigModel::mount_initrd()", 3, FALSE);

	print STDERR "PxesconfigModel::mount_initrd()\n"
		if $DEBUG > 1;
	print STDERR "Mounting initial ramdisk.\n"
		if $DEBUG;

	# Redundant test
	# It was made in initial_checks
	if ( ! $self->check_fstab() ) {
		$self->fatal("Needed entry missing from /etc/fstab.\n" .
			"Add an entry like:\n" .
			$self->fstab_entry('str') . "\n"
			);
	}

	# double check
	if ( $self->is_mounted() ) {
		if ( ! $self->initrd_already_mounted() ) {
			die "\nModel inconsistency:\n" .
				"The model and the reality are incompatibles, " .
				"the filesystem ", $self->initrd_uncompressed(), " is " .
				"already mounted.\nTry to manually unmount it and try " .
				"again,";
		}

		print STDERR "\treturning 0\n" if $DEBUG > 1;
		return 0;
	}

	if ( ! -d $self->initrd_mountdir() ) {
		$self->create_path_to($self->initrd_mountdir());
		if ( ! mkdir($self->initrd_mountdir()) ) {
			$self->fatal("Cannot make mount directory '" .
				$self->initrd_mountdir());
		}
	}

	if ( ! -r $self->initrd_uncompressed() ) {
		$self->fatal("Cannot read initrd filesystem '" .
			$self->initrd_uncompressed() . "\n" .
			"Maybe you forgot to uncompress it.\n");
	}

	my $r = system("mount " . $self->initrd_mountdir());
	if ( $r/256 ) {
		$self->fatal("Cannot mount filesystem ($r:$!)");
	}

	my $s = (stat($self->initrd_uncompressed()))[7];
	$self->initrd_fssize(int($s/1024)+1);
	$self->initrd_already_mounted(TRUE);

	print STDERR "\treturning 0 (last)\n" if $DEBUG > 2;
	return 0;
}

sub umount_initrd($) {
	my ($self) = @_;

	print STDERR "Unmounting initial ramdisk.\n"
		if $DEBUG;

	if ( ! $self->initrd_already_mounted() ) {
		if ( $self->is_mounted() ) {
			die "\nModel inconsistency:\n" .
				"The model and the reality are incompatibles, " .
				"the filesystem ", $self->initrd_uncompressed(), " is " .
				"mounted,";
		}
		else {
			return 0;
		}
	}

	if ( ! $self->is_mounted() ) {
		if ( $self->initrd_already_mounted() ) {
			die "\nModel inconsistency:\n" .
				"The model and the reality are incompatibles, " .
				"the filesystem ", $self->initrd_uncompressed(), " is " .
				"not mounted,";
		}
	}

	my $r = system("umount " . $self->initrd_mountdir());
	if ( $r/256 ) {
		print STDERR "r=$r\n" if $DEBUG;
		print STDERR "msg=$!\n" if $DEBUG;
		return -1;
	}

	$self->initrd_already_mounted(FALSE);
	$self->initrd_already_uncompressed(FALSE);

	return 0;
}

sub compress_initrd($) {
	my ($self) = @_;

	print STDERR "Filesystem ", $self->initrd_uncompressed(),
		" being compressed to ", $self->initrd(), "\n"
		if $DEBUG > 0;
	my $r = system("gzip -9c < " . $self->initrd_uncompressed() . 
		" >| " .  $self->initrd());
	if ( $r/256 ) {
		print STDERR "r=$r\n";
		print STDERR "msg=$!\n";
		return -2;
	}

	chmod(0664, $self->initrd());
	return 0;
}

sub copy_kernel($) {
	my ($self) = @_;

	print STDERR "Kernel ", $self->kernel(),
		" being copied to ", $self->tftpboot(), "\n"
		if $DEBUG > 0;

	my $k = (stat($self->kernel()))[9];
	my $n = basename($self->kernel());
	my $t = (stat($self->tftpboot() . "/" . $n))[9] || 0;
	
	warn "k=$k t=$t" if $DEBUG > 1;

	if ( $k > $t ) {
		if ( copy($self->kernel(), $self->tftpboot()) < 0 ) {
			return -1;
		}
	}

	return 0;
}

sub du($) {
	my ($self, $dir) = @_;
	
	if ( ! $dir || ! -d $dir ) {
		$self->fatal("Cannot calculate a size of an inexistent " .
			"directory: '$dir'.");
	}

	my ($k, $dk) = split(/\s+/, `du -s $dir`);
	chomp($k);

	$self->fatal("Cannot determine directory size of '$dir'")
		if ( $k !~ /\d+/ );

	return $k;
}

sub inodes($) {
	my ($self, $dir) = @_;
	
	if ( ! $dir || ! -d $dir ) {
		$self->fatal("Cannot calculate inodes of an inexistent " .
			"directory: '$dir'.");
	}

	my $i = `find $dir -print | wc -l`;
	chomp($i);
	$i =~ s/(\d+)/$1/;

	$self->fatal("Cannot determine directory size of '$dir'")
		if ( $i !~ /\d+/ );

	return $i;
}

sub calculate_fssize($) {
	my ($self) = @_;
	my $dir = $self->stock_dir() . "/dist";
	my $size;

	if ( $self->stock_dir() eq '/' ) {
		$self->fatal("Are you trying to duplicate your installation ?");
	}

	$size += $self->du($dir);

	# modules
	$dir = $self->modules_dir();

	if ( ! -d $dir ) {
		$self->fatal("Cannot calculate modules size for " .
			"directory: '$dir'.");
	}

	$size += $self->du($dir);

	# per session size variation
	if ( $self->x_session_ica() ) {
		my $local_icaroot = $self->local_ica_installation_directory();
		my $iv;
		if ( ($iv = $self->check_ica_version($local_icaroot)) < 0 ) {
			$self->fatal("Cannot determine ICA Version");
		}
		eval "\$size += ICA" . $iv . "FSSIZE;";
		if ( $self->x_session_local() ) {
			# FIXME: hardcoded
			$size += 2000;
		}
	}

	if ( $self->x_session_local() ) {
		$dir = $self->stock_dir() . "/extra/local";
		$size += $self->du($dir);
	}

	# rfb
	if ( $self->rfb_enabled() ) {
		$dir = $self->stock_dir() . "/extra/rfb";
		$size += $self->du($dir);
	}


	# nx
	if ( $self->x_session_nx() ) {
		$dir = $self->stock_dir() . "/extra/nxclient";
		$size += $self->du($dir);
	}

	# linrdp
	if ( $self->x_session_linrdp() ) {
		$dir = $self->stock_dir() . "/extra/linrdp";
		$size += $self->du($dir);
	}

	# winconnect
	if ( $self->x_session_winconnect() ) {
		$dir = $self->stock_dir() . "/extra/winconnect";
		$size += $self->du($dir);
	}

	# independent variation
	if ( $self->local_samba_enabled() ) {
		$dir = $self->stock_dir() . "/extra/samba";
		$size += $self->du($dir);
	}

	# XFree86
	if ( $self->x_sessions('selected') ) {
		$dir = $self->stock_dir() . "/extra/XFree86-" .
			$self->x_version();
		$size += $self->du($dir);
	}

	# optional files
	$dir = $self->stock_dir() . "/extra/opt";
	$size += $self->du($dir);

	# debug tools
	if ( $self->pxes_debug() ) {
		$dir = $self->stock_dir() . "/extra/debug";
		$size += $self->du($dir);
	}

	# included dirs
	if ( $self->include() ) {
		foreach my $d (@{$self->include()}) {
			$size += $self->du($d);
		}
	}

	debug("PxesconfigModel::calculate_fssize returning $size", 4, FALSE);

	return $size;
}

sub calculate_inodes($) {
	my ($self) = @_;
	my $dir = $self->stock_dir() . "/dist";

	if ( ! $dir || ! -d $dir ) {
		$self->fatal("Cannot calculate a size of an inexistent " .
			"directory.");
	}

	my $i = $self->inodes($dir);

	# per session inode variation
	if ( $self->x_session_ica() ) {
		# FIXME
		# really hardcoded
		$i += 20;
	}

	# local
	if ( $self->x_session_local() ) {
		$dir = $self->stock_dir() . "/extra/local";
		$i += $self->inodes($dir);
	}

	# rfb
	if ( $self->rfb_enabled() ) {
		$dir = $self->stock_dir() . "/extra/rfb";
		$i += $self->inodes($dir);
	}

	# nx
	if ( $self->x_session_nx() ) {
		$dir = $self->stock_dir() . "/extra/nxclient";
		$i += $self->inodes($dir);
	}

	# linrdp
	if ( $self->x_session_linrdp() ) {
		$dir = $self->stock_dir() . "/extra/linrdp";
		$i += $self->inodes($dir);
	}

	# winconnect
	if ( $self->x_session_winconnect() ) {
		$dir = $self->stock_dir() . "/extra/winconnect";
		$i += $self->inodes($dir);
	}

	# indepented variation
	if ( $self->local_samba_enabled() ) {
		$dir = $self->stock_dir() . "/extra/samba";
		$i += $self->inodes($dir);
	}

	# XFree86
	if ( $self->x_sessions('selected') ) {
		$dir = $self->stock_dir() . "/extra/XFree86-" .
			$self->x_version();
		$i += $self->inodes($dir);
	}

	# optional files
	$dir = $self->stock_dir() . "/extra/opt";
	$i += $self->inodes($dir);

	# debug tools
	if ( $self->pxes_debug() ) {
		$dir = $self->stock_dir() . "/extra/debug";
		$i += $self->inodes($dir);
	}

	# included dirs
	if ( $self->include() ) {
		foreach my $d (@{$self->include()}) {
			$i += $self->inodes($d);
		}
	}

	return $i;
}

sub initialize_initrd($) {
	my ($self) = @_;

	print STDERR "PxesconfigModel::initialize_initrd\n"
		if $DEBUG > 2;

	$self->umount_initrd();

	$self->print_message('info', "Initializing ram disk.")
		if $verbose;

	# add x% + yk to save configuration files
	my $fssize = int($self->calculate_fssize() *
		($self->initrd_fssize_p100inc()+1) + $self->initrd_fssize_inc());
	# add x inodes to save configuration files
	my $inodes = $self->calculate_inodes() + $self->initrd_inodes_inc();

	# FIXME
	# unify variables
	$self->initrd_fssize($fssize);
	$self->initrd_inodes($inodes);

	my $r;
	my $cmd;

	$cmd = "dd if=/dev/zero of=" . 
		$self->initrd_uncompressed() . " bs=1k count=$fssize " .
		"2>/dev/null";

	print STDERR $cmd,"\n"
		if $DEBUG > 1;

	$r = system($cmd);

	if ( $r != 0 ) {
		$self->fatal("Cannot create filesystem file ($r),");
	}

	$cmd  = "/sbin/mke2fs -F -s0 -q -N $inodes " .
		$self->initrd_uncompressed() . " 2>/dev/null";

	print STDERR $cmd,"\n"
		if $DEBUG > 1;

	$r = system($cmd);

	if ( $r != 0 ) {
		$self->fatal("Cannot create filesystem.\nRunning command: " .
			"'$cmd'.");
	}

	$self->initrd_already_uncompressed(TRUE);

	$r = system("/sbin/tune2fs -c 0 -i 0 -L PXES -m 0 " .
		$self->initrd_uncompressed() . " >/dev/null 2>&1");

	if ( $r != 0 ) {
		$self->error("Cannot tune filesystem,");
	}

	if ( $self->mount_initrd() < 0 ) {
		return -1;
	}

	if ( $self->populate_initrd() < 0 ) {
		if ( $DEBUG ) {
			warn "populate_initrd FAILED but in DEBUG mode";
			return 0;
		}
		return -1;
	}

	print STDERR "\treturning 0 (last)\n" if $DEBUG > 1;
	return 0;
}

sub populate_from_to($$) {
	my ($self, $from, $to) = @_;
	my $r;
	my $tmpbase = "/tmp/pxes$$";

	print STDERR "PxesconfigModel::populate_from_to($from,$to)\n"
		if $DEBUG > 0;

	if ( ! -d $from ) {
		$self->fatal("Directory $from doesn't exist");
	}

	if ( ! -d $to ) {
		system("mkdir -p $to");
	}

	$r = system("/bin/sh -c \"(cd $from; find -depth -print | cpio -pdum $to > $tmpbase.log 2> $tmpbase.err)\"");

	if ( -s "$tmpbase.log" ) {
		open(F, "<$tmpbase.log") or die "$!";
		my $l = '';
		while ( <F> ) {
			$l .= $_;
		}
		close(F);
		$self->fatal($l);
	}
	else {
		unlink("$tmpbase.log");
	}

	open(F, "<$tmpbase.err") or die "$!";
	my $l = <F>;
	close(F);
	if ( $l !~ /\d+ +block/ ) {
		open(F, "<$tmpbase.err") or die "$!";
		my $l = '';
		while ( <F> ) {
			$l .= $_;
		}
		close(F);
		$self->fatal($l);
	}
	else {
		unlink("$tmpbase.err");
	}
		
	if ( $r != 0 ) {
		################
		# FIXME
		warn "returning 0 in DEBUG MODE !
Probably the error was caused by a --nonrootinitialize and /dev copying errors
		" if $DEBUG;

		return 0 if $DEBUG;
		$self->fatal("Cannot populate $to: $!");
	}

	return 0;
}

sub populate_initrd($) {
	my ($self) = @_;
	my $to = $self->initrd_mountdir();
	my $from = $self->stock_dir() . "/dist";

	print STDERR "PxesconfigModel::populate_initrd\n" 
		if $DEBUG > 0;

	return -1 if ( ! defined($from) || ! $from || ! -d $from );

	if ( $self->populate_from_to($from, $to) < 0 ) {
		$self->fatal("Cannot populate $to");
	}

	$from = $self->modules_dir();
	return -1 if ( ! defined($from) || ! $from );
	$to .= "/lib/modules/" . $self->kernel_version();

	if ( $self->populate_from_to($from, $to) < 0 ) {
		$self->fatal("Cannot populate $to");
	}

	return 0;
}

sub copy_module($$) {
	my ($self, $module) = @_;

	print STDERR "\nPxesconfigModel::copy_module " .
		"NOT IMPLEMENTED YET\n";
}

sub check_arch_dependencies($) {
	my ($self, $arch) = @_;
	#my $d = $self->stock_dir() . "/arch/" . $arch;
	my $d = $self->arch_dir() . '/' . $arch;

	if ( ! defined($d) || ! $d || ! -d $d ) {
		$self->error_message($d);
		return -1;
	}

	return 0;
}

sub kernel_dir($) {
	my ($self, $arch) = @_;

	#return $self->stock_dir() . "/arch/" . $arch . "/kernel";
	return $self->arch_dir() . '/' . $arch . '/kernel';
}

sub find_kernel($$) {
	my ($self, $arch, $kd) = @_;

	warn "### finding kernel kernel=" . $self->kernel() if $DEBUG > 4;
	# if the default kernel exists, return it
	return $self->kernel() if -f "$self->kernel()";

	# or try to find another one
	warn "### trying to find another one" if $DEBUG > 4;
	$kd = $self->kernel_dir($arch) unless $kd;
	warn "### kd=" . $kd if $DEBUG > 4;
	my $g = $kd . "/vmlinuz-*";
	my @g = glob($g);
	if ( &File::Glob::GLOB_ERROR != 0 ) {
		$self->error_message($!);
		warn "### returning undef " . $self->error_message()
			if $DEBUG > 4;
		return undef;
	}

	if ( $#g < 0 || ! defined($g[0])) {
		$self->error_message("Cannot find kernels for $arch in $kd");
		warn "### returning undef " . $self->error_message()
			if $DEBUG > 4;
		return undef;
	}

	my $newer = undef;
	my $modified = -1;
	my $higher = " ";
	my $higher_version = " ";

	foreach my $k (@g) {
		my $v = $k;
		$v =~ s/.*vmlinuz-//;
		$v =~ s/\.$arch$//;
		$v =~ s/-/./g;
		$v =~ s/pxes//;
		my @v = split(/\./,$v);
		warn "### splitting $v -> @v" if $DEBUG > 4;
		foreach my $c (0..$#v) {
			$v[$c] = sprintf('%03d', $v[$c]);
		}
		$v = join('.', @v);
		warn "### joining $v" if $DEBUG > 4;

		warn "### comapring $v and $higher" if $DEBUG > 4;
		my $m = (stat($k))[9];
		if ( $m > $modified) {
			$newer = $k;
			$modified = $m;
		}

		if ( $v gt $higher_version ) {
			$higher_version = $v;
			$higher = $k;
		}
	}

	warn "### finding kernel newer=" . $newer if $DEBUG > 4;
	warn "### finding kernel higher=" . $higher if $DEBUG > 4;
	return $higher;
}

# hashFromFile
# returns a hash with the pairs (attr,val) from filename
# no case is changed
sub hashFromFile($$$) {
	my ($self, $filename, $mandatory) = @_;

	debug("PxesconfigModel::hashFromFile($filename, $mandatory)", 3,
		FALSE);

	my %h = ();

	if ( ! open(F, $filename) ) {
		fatal(undef, "Cannot open '$filename' for reading: $!")
			if ( $mandatory );
		return \%h;
	}

	while ( <F> ) {
		next if /^\s*#/;
		next if /^\s*$/;
		next if /^\s*export/;

		/\s*(\w+)=(.*)\s*$/ or next;

		my $name = $1;
		my $val = $2;


		my $q = substr($val, 0, 1);
		my $r = substr($val, -1, 1);
		my $s = substr($val, 1, length($val)-2);

		# check for surrounding quotes
		if ( $q eq '"' && $r eq '"' ) {
			$val = $s;
		}
		elsif ( $q eq "'" && $r eq "'" ) {
			$val = $s;
		}
		elsif ( $q =~ /["']/ && $r !~ /["']/ ) {
			die "Apparently syntax error: $_";
		}
			
		warn ">> name=$name val=[$val]" if $DEBUG > 1;
		if ( $val =~ /^(["'][\S ]+["']\s*)+/ ) {
			warn ">> val='$val' is an ARRAY" if $DEBUG;
			# check for inner quotes
			if ( $val =~ /(["'])/ ) {
				$q = $1;
				#my @val = split(/[$1 	]+/, $val);
				#my @val = split(/$1[ \t]*$1*/, $val);
				warn "splitting val='$val' on q=$1" if $DEBUG > 1;
				#my @val = split(/$1[ \t]*$1?/, $val);
				my @val = split(/$1\s*$1?/, $val);
				warn "q='$q' n=$#val \@val='@val'" if $DEBUG > 1;

				# the first element can be null because of the initial
				# delimiter
				shift @val unless $val[0];

				# add quotes
				for (my $i=0; $i <= $#val; $i++) {
					if ( $val[$i] !~ /$q/ ) {
						$val[$i] = "${q}$val[$i]${q}";
					}
				}

				print STDERR "val=@val 0..$#val\n"
					if $DEBUG > 1;
				$val = \@val;
			}
		}

		print STDERR "PxesconfigModel::hashFromFile: setting '$name' = '$val'\n"
			if $DEBUG > 1;

		$h{$name} = $val;
	}

	close(F);

	return \%h;
}

sub clean_up($) {
	my ($self) = @_;

	print STDERR "Cleaning up...\n" 
		if $DEBUG > 0;
	$self->umount_initrd();
}

sub attach($$) {
	my ($self, $view) = @_;

	push(@{$self->{observers}}, $view);
}

sub detach($$) {
	my ($self, $view) = @_;

	print STDERR "PxesconfigModel::detach: NOT IMPLEMENTED YET\n";
}

sub notify($) {
	my ($self) = shift;

	if ( $verbose ) {
		$self->print_message(@_);
	}

	foreach my $o ( @{$self->{observers}} ) {
		$o->update(@_);
	}

}

sub print_message {
	my ($self) = shift;
	my ($level) = shift;

	if ( $level eq 'info' ) {
		print @_, "\n";
	}
	elsif ( $level eq 'error' ) {
		$self->error(@_);
	}
	elsif ( $level eq 'fatal' ) {
		$self->fatal(@_);
	}
	else {
		print STDERR @_, "\n";
	}

}

sub set_state($$) {
	my ($self, %namevals) = @_;

	foreach my $a ( keys %namevals ) {
		$self->$a($namevals{$a});
	}

	$self->notify();
}

sub toString() {
	my $self = shift;
	my $s;

	foreach my $k ( sort keys(%{$self->{fields}}) ) {
		$s .= uc($k) . "=" . $self->{fields}->{$k}. "\n";
	}

	$s .= "observers=";
	foreach my $o ( @{$self->{observers}} ) {
		$s .= $o . " ";
	}

	return $s;
}

sub stock_dir {
	my $self = shift;
	my ($dir) = @_;

	if (@_) {
		if ( ! $dir || ! -d $dir ) {
			$self->fatal("Invalid or non-existent directory '$dir'");
		}
		return $self->{local_fields}->{stock_dir} = $dir;
	}
	else {
		return $self->{local_fields}->{stock_dir};
	}
}

sub inittab_startx_action {
	my $self = shift;
	my ($action) = @_;
	
	if ( @_ ) {
		return $self->{fields}->{inittab_startx_action} = $action
			if ( $action eq 'askfirst' || $action eq 'respawn' );

		return $self->{fields}->{inittab_startx_action} = 'askfirst'
			if ( $action );

		return $self->{fields}->{inittab_startx_action} = 'off';
	}
	else {
			return $self->{fields}->{inittab_startx_action};
	}
}

sub inittab_shell_action {
	my $self = shift;
	my ($action) = @_;
	
	if ( @_ ) {
		return $self->{fields}->{inittab_shell_action} = $action
			if ( $action eq 'askfirst' || $action eq 'off' );

		return $self->{fields}->{inittab_shell_action} = 'askfirst'
			if ( $action );

		return $self->{fields}->{inittab_shell_action} = 'off';
	}
	else {
			return $self->{fields}->{inittab_shell_action};
	}
}

sub inittab_login_action {
	my $self = shift;
	my ($action) = @_;
	
	if ( @_ ) {
		return $self->{fields}->{inittab_login_action} = $action
			if ( $action eq 'askfirst' || $action eq 'off' );

		return $self->{fields}->{inittab_login_action} = 'askfirst'
			if ( $action );

		return $self->{fields}->{inittab_login_action} = 'off';
	}
	else {
			return $self->{fields}->{inittab_login_action};
	}
}

sub kernel_version($) {
	my $self = shift;
	my ($d, $v) = $self->kernel() =~ /^(.*)vmlinuz-(.+)/;

	return undef if ( ! $d || ! $v );

	# remove optional arch extension
	# WARNING: Only Intel and VIA C3
	$v =~ s/\.((i[3456]86)|(viac3))$//;
	return $v;
}

sub mouse_wheel_enabled($) {
	my $self = shift;
	my ($value) = @_;

	if ( @_ ) {
		if ( $value ) {
			$self->mouse_emulate_3_buttons_enabled(FALSE);

			if ( $self->mouse_protocol() eq 'PS/2' ) {
				$self->mouse_protocol_prefix('im');
			}
			else {
				$self->mouse_protocol_prefix('');
			}
		}
		else {
			$self->mouse_emulate_3_buttons_enabled(TRUE);
		}

		return $self->{fields}->{mouse_wheel_enabled} = $value;
	}
	else {
		return $self->{fields}->{mouse_wheel_enabled};
	}
}

sub modules_dir() {
	my $self = shift;

	my $v = $self->kernel_version();
	return undef if ( ! defined($v) || ! $v );
	my $a = $self->kernel_arch();
	return undef if ( ! defined($a) || ! $a );

	return $self->arch_dir() .  "/$a/modules/lib/modules/$v";
}

sub drivers_dir() {
	my $self = shift;
	my $d = $self->modules_dir();

	return undef if ( ! defined($d) || ! $d || ! -d $d );
	$d .= "/kernel/drivers";
	if ( ! -d $d) {
		$self->fatal("Directory $d doesn't exist");
	}

	return $d;
}

sub video_drivers_dir() {
	my $self = shift;
	my $d = $self->stock_dir() . "/extra";

	return undef if ( ! defined($d) || ! $d || ! -d $d );
	$d .= "/XFree86-4.3.0/lib/modules/drivers";
	if ( ! -d $d) {
		$self->fatal("Directory $d doesn't exist");
	}

	return $d;
}

sub network_cards($) {
	my $self = shift;
	my $d = $self->drivers_dir();
	
	return undef if ( ! defined($d) || ! $d || ! -d $d );
	$d .= "/net";
	if ( ! -d $d ) {
		$self->fatal("Directory $d doesn't exist");
	}

	my @m = `(cd $d && find . -type f -print)`;
	my @r;
	foreach my $m (sort @m) {
		$m =~ m@.*/([\w-]+)\.o@;
		push @r, $1 if $1;
	}

	if ( ! @r ) {
		$self->fatal("I couldn't find any network card in '$d'");
	}

	unshift @r, "Autodetect";
	return @r;
}

sub sound_cards($) {
	my $self = shift;
	my $d = $self->drivers_dir();
	
	return undef if ( ! defined($d) || ! $d || ! -d $d );
	$d .= "/sound";
	my @r;

	if ( ! -d $d ) {
		$self->error("Directory $d doesn't exist");
		push @r, "";
		return @r;
	}

	my @m = `(cd $d && find . -type f -print)`;
	foreach my $m (sort @m) {
		$m =~ m@.*/([\w-]+)\.o@;
		push @r, $1 if $1;
	}

	if ( ! @r ) {
		$self->error("I couldn't find any sound card in '$d'");
	}

	unshift @r, "Autodetect";
	return @r;
}

sub keyboard_layouts($) {
	my $self = shift;
	my $d = $self->stock_dir();

	return undef if ( ! defined($d) || ! $d || ! -d $d );
	$d .= "/dist/lib/kmaps";

	if ( ! -d $d ) {
		$self->fatal("Directory $d doesn't exist");
	}

	my @m = `(cd $d && find . -type f -print)`;
	my @r;
	foreach my $m (sort @m) {
		$m =~ m@.*/([\w-]+)@;
		push @r, $1 if $1;
	}

	if ( ! @r ) {
		$self->fatal("I couldn't find any keyboard map in '$d'");
	}

	return @r;
}

sub video_drivers($) {
	my $self = shift;
	my $d = $self->video_drivers_dir();
	
	return undef if ( ! defined($d) || ! $d || ! -d $d );
	my @r;

	if ( ! -d $d ) {
		$self->error("Directory $d doesn't exist");
		push @r, "";
		return @r;
	}

	my @m = `(cd $d && find . -type f -name "*drv.o" -print)`;
	foreach my $m (sort @m) {
		$m =~ m@.*/([\w-]+)_drv\.o@;
		push @r, $1 if $1;
	}

	if ( ! @r ) {
		$self->error("I couldn't find any X video driver in '$d'");
	}

	unshift @r, "Autodetect";
	return @r;
}

sub monitors() {
	my $self = shift;
	my $db = $self->monitors_db();

	return undef if ( ! defined($db) || ! $db || ! -r $db );
	my @r;

	open(MDB, $db);
	while ( <MDB> ) {
		next if /^\s*#/;
		next unless $_;
		chomp;
		my ($m, $n, $e, $v, $h) = split('\s*;\s*');
		next unless $n;
		$n =~ s/^\s*//;
		my @a = ($n, $v, $h);
		push @r, \@a if $n;
	}
	close(MDB);

	return @r;
}

sub parse_ica_application_servers_ini($) {
	my $self = shift;
	my @r;

	my $ini = $self->local_ica_application_servers_ini();
	if ( ! -r $ini ) {
		# FIXME: ica_server_list should be reset
		return undef;
	}

	if ( ! open(INI, $ini) ) {
		# FIXME: ica_server_list should be reset
		return undef;
	}

	while ( <INI> ) {
		next unless /\[ApplicationServers\]/;
		$_ = <INI>;
		chomp;
		while ( $_ && $_ ne '' ) {
			/(.+)=/;
			push @r, $1 if $1;
			$_ = <INI>;
			chomp;
		}
		last;
	}
	close(INI);
		
	$self->ica_server_list(\@r);
	return @r;
}

sub parse_inittab($) {
	my $self = shift;
	my $dir = $self->initrd_mountdir();
	my $date = `date`;
	my $ia = $self->inittab_sysinfo_action();
	my $la =	$self->inittab_login_action();
	my $sa =	$self->inittab_shell_action();

	if ( $self->x_sessions('selected') ) {
		$self->inittab_startx_action(
			$self->prompt_before_x() ? 'askfirst' : 'respawn' );
	}
	else {
		$self->inittab_startx_action('off');
	}
		
	my $xa = $self->inittab_startx_action();
	my $tt = ( $xa eq 'off' ) ? 'respawn' : 'off';

	open(TPL, "$dir/etc/inittab.tpl") or
		$self->fatal("Couldn't find the inittab.tpl in $dir/etc");

	my $filename = "$dir/etc/inittab";
	my $umask = umask(02);
	open(INITTAB, ">$filename") or
		$self->fatal("Couldn't create the inittab in $dir/etc");
	umask($umask);
	chmod(0664, $filename);

	my $xt = ( $xa eq 'respawn' && $self->x_version() gt '4' ) ? XVT : 1;

	# should I call parsetemplate ?
	while ( <TPL> ) {
		s/\${DATE}/$date/g;
		s/\${INITTAB_STARTX_TTY}/$xt/g;
		s/\${INITTAB_STARTT_TTY}/$xt/g;
		s/\${INITTAB_STARTX_ACTION}/$xa/g;
		s/\${INITTAB_STARTT_ACTION}/$tt/g;
		s/\${INITTAB_SYSINFO_ACTION}/$ia/g;
		s/\${INITTAB_LOGIN_ACTION}/$la/g;
		s/\${INITTAB_SHELL_ACTION}/$sa/g;
		# busybox doesn't support off actions
		$_ = "# $_" if m@^[\w/]*::off:@;
		print INITTAB $_;
	}

	close(INITTAB) or
		$self->fatal("Closing iniitab: $!");

	close(TPL);

	return 0;
}

sub parse_isolinux_cfg($) {
	my $self = shift;
	my $dir = $self->isopxes_dir();
	my $date = `date`;
	my $k = 'vmlinuz';
	# FIXME
	#my $aimg = 'ramdisk_size=%d ro vga=0x303 console=null initrd=pxes.img root=/dev/ram';
	#my $afs = 'ramdisk_size=%d ro vga=0x303 console=null initrd=pxes.img root=/dev/cdrom';
	my $aimg = 'ramdisk_size=%d ro initrd=pxes.img root=/dev/ram';
	my $afs = 'ramdisk_size=%d ro initrd=pxes.img root=/dev/cdroms/cdrom0';
	my $lb = '-1'; # BIOS boot failure
	my $filename = "$dir/boot/isolinux/isolinux.cfg";

	$aimg = sprintf($aimg, $self->initrd_fssize());
	# by now, I'm using the same ramdisk (should change when cdrom is
	# mounted as root filesystem)
	$afs = sprintf($afs, $self->initrd_fssize());

	open(TPL, "$filename.tpl") or
		$self->fatal("Couldn't find $filename.tpl");

	my $umask = umask(02);
	open(ISOLINUXCFG, ">$filename") or
		$self->fatal("Couldn't create $filename");
	umask($umask);
	chmod(0664, $filename);

	# should I call parsetemplate ?
	while ( <TPL> ) {
		s/\${DATE}/$date/g;
		s/\${ISO_KERNEL}/$k/g;
		s/\${ISO_IMG_APPEND}/$aimg/g;
		s/\${ISO_FS_APPEND}/$afs/g;
		s/\${ISO_LOCALBOOT}/$lb/g;
		print ISOLINUXCFG $_;
	}

	close(ISOLINUXCFG) or
		$self->fatal("Closing $filename: $!");

	close(TPL);

	return 0;
}

sub rdp_server_version {
	my $self = shift;
	my ($version) = @_;

	if (@_) {
		if ( $version ne 'W2K' && $version ne 'NT4' ) {
			$self->fatal("Invalid RDP server version '$version'");
		}
		return $self->{fields}->{rdp_server_version} = $version;
	}
	else {
		return $self->{fields}->{rdp_server_version};
	}
}

sub local_ica_application_servers_ini {
	my $self = shift;
	my ($appsrv) = @_;

	if (@_) {
		$self->{local_fields}->{local_ica_application_servers_ini} =
			$appsrv;
		$self->parse_ica_application_servers_ini();
		return $self->{local_fields}->{local_ica_application_servers_ini};
	}
	else {
		return $self->{local_fields}->{local_ica_application_servers_ini};
	}
}

sub session_default {
	my $self = shift;
	my ($session) = @_;

	if (@_) {
		my $s = lc($session);
		$self->{fields}->{session_default} = $session;
		# force selection of default session
		my $t = $self->sessions()->{$s};
		my $x = ($t == XA || $t == XI) ? 'x_' : '';
		$self->{fields}->{$x . "session_" . $s} = TRUE;
	}

	return $self->{fields}->{session_default};
}

sub extra_templates_op($$) {
	my $self = shift;
	my ($op, $path) = @_;
	my $debug = $DEBUG > 2 || grep(/extra_templates_op/, @DEBUG);

	warn "extra_templates_op::$op $path" if $debug;

	if ( $op eq 'find' ) {
		if ( ! defined($self->extra_templates()) || !
				$self->extra_templates() ) {
			return undef;
		}

		#return (grep(/$path/, $self->extra_templates()))[0];
		my @r = grep(m@$path@, @{$self->extra_templates()});
		warn "extra_templates_op::finding $path -> @r" if $debug;
		#return \$r[0];
		return $r[0];
	}
	elsif ( $op eq 'add' ) {
		if ( ! defined($self->extra_templates()) || !
				$self->extra_templates() ) {
			warn "#### creating extra_templates array adding $path"
				if $debug;
			my @a = ($path);
			warn "#### inserting a=@a" if $debug;
			$self->extra_templates(\@a);
		}
		else {
			if ( ! $self->extra_templates_op('find', $path) ) {
				warn "### adding $path" if $debug;
				push(@{$self->extra_templates()}, $path);
			}
		}
	}
	elsif ( $op eq 'del' ) {
		if ( ! defined($self->extra_templates()) || ! 
				$self->extra_templates() ) {
			return;
		}
		else {
			warn "### deleting $path" if $debug;
			my @a  = @{$self->extra_templates()};
			foreach my $i (0..$#a) {
				delete $a[$i] if ( $a[$i] =~ m@$path@ );
			}
		}
	}
	else {
		die "Internal error";
	}
}

sub local_samba_enabled {
	my $self = shift;
	my ($enabled) = @_;

	warn "local_samba_enabled($enabled)" if $DEBUG > 1;

	if (@_) {
		warn "before: @{$self->extra_templates()}" if
			$self->extra_templates() && $DEBUG > 1;
		$self->{fields}->{local_samba_enabled}  = $enabled;
		$self->extra_templates_op(($enabled)?'add':'del',
			'/etc/samba/smb.conf.tpl');
		warn "after: @{$self->extra_templates()}" if 
			$self->extra_templates() && $DEBUG > 1;
	}

	return $self->{fields}->{local_samba_enabled};
}

sub local_httpd_enabled {
	my $self = shift;
	my ($enabled) = @_;
	my $debug = $DEBUG > 1 || grep(/local_httpd_enabled/, @DEBUG);

	warn "local_httpd_enabled($enabled)" if $debug;

	if (@_) {
		$self->{fields}->{local_httpd_enabled}  = $enabled;
		warn "before: @{$self->extra_templates()}" if
			$self->extra_templates() && $debug;
		$self->extra_templates_op(($enabled)?'add':'del',
			'/usr/www/html/index.html.tpl');
		warn "###### after: @{$self->extra_templates()}" if 
			$self->extra_templates() && $debug;
	}

	return $self->{fields}->{local_httpd_enabled};
}

sub login($) {
	my $self = shift;

	return (getpwuid($EUID))[0]
}

sub delete_lost_found() {
	my $self = shift;
	my $dir = $self->initrd_mountdir();

	return if ! $dir;
	rmdir($dir . "/lost+found");
}

sub check_kernel($) {
	my $self = shift;
	my $kernel = shift;

	return TRUE if $kernel =~ /vmlinuz-/;
	return FALSE;
}

sub session_by_name($) {
	my $self = shift;
	my $name = shift;

	return FALSE unless $name;

	my $x = '';
	foreach my $s (keys %{$self->sessions()}) {
		if ( $s eq $name ) {
			my $t = $self->sessions()->{$s};
			if ( $t == XA || $t == XI ) {
				$x = 'x_';
			}
			last;
		}
	}

	eval 'return $self->' . $x . 'session_' . $name . '()';
}

sub x_sessions($$) {
	my $self = shift;
	my $state = shift;
	my @r = ();

	if ( $state eq 'selected' ) {
		for my $s (keys %{$self->sessions()}) {
			next if $self->sessions()->{$s} != XA;
			if ( eval('$self->x_session_' . $s . '()') ) {
				push(@r, $s);
			}
		}

		return @r;
	}
	else {
		warn "Not implemented";
	}
}

sub set_root_password($) {
	my $self = shift;
	my $pxescrypt = $self->stock_dir() . "/dist/bin/pxescrypt";
	my $salt = "px";
	my $dir = $self->initrd_mountdir();
	my $passwd = $dir . "/etc/passwd";
	my $temp = "/tmp/pxesconfig.$$";

	if ( ! -x $pxescrypt ) {
		return -1;
	}

	my $PIPE;
	open($PIPE, "|$pxescrypt -s $salt>$temp") || $self->fatal($!);

	if ( ! $PIPE ) {
		return -1;
	}

	print $PIPE $self->root_password(), "\n";
	close($PIPE);

	my $ENCRYPTED;
	open($ENCRYPTED, $temp);

	my $e = <$ENCRYPTED>;
	close($ENCRYPTED);
	unlink($temp);
	chomp($e);


	my $PASSWD;
	open($PASSWD, "<$passwd") || $self->fatal($!);
	
	if ( ! $PASSWD ) {
		return -1;
	}

	my @p = <$PASSWD>;
	close($PASSWD);
	$p[0] =~ s/^root:[^:]*:/root:$e:/;

	open($PASSWD, ">$passwd") || $self->fatal($!);
	print $PASSWD @p;
	close($PASSWD);

	return 0;
}


1;
__END__

#=======================================================================
#==== Documentation
#=======================================================================
=pod

=head1 NAME

PxesconfigModel - version 0.01 Fri May 24 01:45:10 ART 2002

This class represent the Pxesconfig Model (MVC parlance).

=head1 SYNOPSIS

 use PxesconfigModel;
 my $model = PxesconfigModel();

 To construct an empty model, or

 use PxesconfigModel;
 my $model = PxesconfigModel($filename)

 To construct a model based on $filename settings.
 
=head1 DESCRIPTION

Unfortunately, the author has not yet written any documentation :-(

=head1 AUTHOR

Diego Torres Milano <diego@in3.com.ar>

=cut
