"""
Map projection file utility functions
"""

def name_to_descript(mapprj_name):
    """
    Returns the map projection file contents of the given common name
    (case insensitive).  Raises a ValueError is the common name is unknown.
    """
    # get the dictionary key from the given name
    prjname = mapprj_name.upper()
    # get the description (None returned if not present)
    descript = _DESCRIPT_DICT.get(prjname)
    if not descript:
        raise ValueError('Unknown map projection name "%s"' % mapprj_name)
    return descript


_DESCRIPT_DICT = {
    'ADINDAN':
        'GEOGCS["Adindan",DATUM["Adindan",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'ADINDAN / UTM ZONE 37N':
        'PROJCS["Adindan / UTM zone 37N",GEOGCS["Adindan",DATUM["Adindan",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ADINDAN / UTM ZONE 38N':
        'PROJCS["Adindan / UTM zone 38N",GEOGCS["Adindan",DATUM["Adindan",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'AFGOOYE':
        'GEOGCS["Afgooye",DATUM["Afgooye",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'AFGOOYE / UTM ZONE 38N':
        'PROJCS["Afgooye / UTM zone 38N",GEOGCS["Afgooye",DATUM["Afgooye",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'AFGOOYE / UTM ZONE 39N':
        'PROJCS["Afgooye / UTM zone 39N",GEOGCS["Afgooye",DATUM["Afgooye",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'AGADEZ':
        'GEOGCS["Agadez",DATUM["Agadez",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'AGD66':
        'GEOGCS["AGD66",DATUM["Australian_Geodetic_Datum_1966",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'AGD66 / AMG ZONE 48':
        'PROJCS["AGD66 / AMG zone 48",GEOGCS["AGD66",DATUM["Australian_Geodetic_Datum_1966",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD66 / AMG ZONE 49':
        'PROJCS["AGD66 / AMG zone 49",GEOGCS["AGD66",DATUM["Australian_Geodetic_Datum_1966",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD66 / AMG ZONE 50':
        'PROJCS["AGD66 / AMG zone 50",GEOGCS["AGD66",DATUM["Australian_Geodetic_Datum_1966",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD66 / AMG ZONE 51':
        'PROJCS["AGD66 / AMG zone 51",GEOGCS["AGD66",DATUM["Australian_Geodetic_Datum_1966",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD66 / AMG ZONE 52':
        'PROJCS["AGD66 / AMG zone 52",GEOGCS["AGD66",DATUM["Australian_Geodetic_Datum_1966",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD66 / AMG ZONE 53':
        'PROJCS["AGD66 / AMG zone 53",GEOGCS["AGD66",DATUM["Australian_Geodetic_Datum_1966",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD66 / AMG ZONE 54':
        'PROJCS["AGD66 / AMG zone 54",GEOGCS["AGD66",DATUM["Australian_Geodetic_Datum_1966",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD66 / AMG ZONE 55':
        'PROJCS["AGD66 / AMG zone 55",GEOGCS["AGD66",DATUM["Australian_Geodetic_Datum_1966",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD66 / AMG ZONE 56':
        'PROJCS["AGD66 / AMG zone 56",GEOGCS["AGD66",DATUM["Australian_Geodetic_Datum_1966",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD66 / AMG ZONE 57':
        'PROJCS["AGD66 / AMG zone 57",GEOGCS["AGD66",DATUM["Australian_Geodetic_Datum_1966",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD66 / AMG ZONE 58':
        'PROJCS["AGD66 / AMG zone 58",GEOGCS["AGD66",DATUM["Australian_Geodetic_Datum_1966",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD84':
        'GEOGCS["AGD84",DATUM["Australian_Geodetic_Datum_1984",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'AGD84 / AMG ZONE 48':
        'PROJCS["AGD84 / AMG zone 48",GEOGCS["AGD84",DATUM["Australian_Geodetic_Datum_1984",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD84 / AMG ZONE 49':
        'PROJCS["AGD84 / AMG zone 49",GEOGCS["AGD84",DATUM["Australian_Geodetic_Datum_1984",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD84 / AMG ZONE 50':
        'PROJCS["AGD84 / AMG zone 50",GEOGCS["AGD84",DATUM["Australian_Geodetic_Datum_1984",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD84 / AMG ZONE 51':
        'PROJCS["AGD84 / AMG zone 51",GEOGCS["AGD84",DATUM["Australian_Geodetic_Datum_1984",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD84 / AMG ZONE 52':
        'PROJCS["AGD84 / AMG zone 52",GEOGCS["AGD84",DATUM["Australian_Geodetic_Datum_1984",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD84 / AMG ZONE 53':
        'PROJCS["AGD84 / AMG zone 53",GEOGCS["AGD84",DATUM["Australian_Geodetic_Datum_1984",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD84 / AMG ZONE 54':
        'PROJCS["AGD84 / AMG zone 54",GEOGCS["AGD84",DATUM["Australian_Geodetic_Datum_1984",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD84 / AMG ZONE 55':
        'PROJCS["AGD84 / AMG zone 55",GEOGCS["AGD84",DATUM["Australian_Geodetic_Datum_1984",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD84 / AMG ZONE 56':
        'PROJCS["AGD84 / AMG zone 56",GEOGCS["AGD84",DATUM["Australian_Geodetic_Datum_1984",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD84 / AMG ZONE 57':
        'PROJCS["AGD84 / AMG zone 57",GEOGCS["AGD84",DATUM["Australian_Geodetic_Datum_1984",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AGD84 / AMG ZONE 58':
        'PROJCS["AGD84 / AMG zone 58",GEOGCS["AGD84",DATUM["Australian_Geodetic_Datum_1984",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'AIN EL ABD':
        'GEOGCS["Ain el Abd",DATUM["Ain_el_Abd_1970",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'AIN EL ABD / BAHRAIN GRID':
        'PROJCS["Ain el Abd / Bahrain Grid",GEOGCS["Ain el Abd",DATUM["Ain_el_Abd_1970",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'AIN EL ABD / UTM ZONE 37N':
        'PROJCS["Ain el Abd / UTM zone 37N",GEOGCS["Ain el Abd",DATUM["Ain_el_Abd_1970",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'AIN EL ABD / UTM ZONE 38N':
        'PROJCS["Ain el Abd / UTM zone 38N",GEOGCS["Ain el Abd",DATUM["Ain_el_Abd_1970",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'AIN EL ABD / UTM ZONE 39N':
        'PROJCS["Ain el Abd / UTM zone 39N",GEOGCS["Ain el Abd",DATUM["Ain_el_Abd_1970",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'AMERSFOORT':
        'GEOGCS["Amersfoort",DATUM["Amersfoort",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'ARATU':
        'GEOGCS["Aratu",DATUM["Aratu",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'ARATU / UTM ZONE 22S':
        'PROJCS["Aratu / UTM zone 22S",GEOGCS["Aratu",DATUM["Aratu",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ARATU / UTM ZONE 23S':
        'PROJCS["Aratu / UTM zone 23S",GEOGCS["Aratu",DATUM["Aratu",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ARATU / UTM ZONE 24S':
        'PROJCS["Aratu / UTM zone 24S",GEOGCS["Aratu",DATUM["Aratu",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ARC 1950':
        'GEOGCS["Arc 1950",DATUM["Arc_1950",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'ARC 1950 / UTM ZONE 34S':
        'PROJCS["Arc 1950 / UTM zone 34S",GEOGCS["Arc 1950",DATUM["Arc_1950",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ARC 1950 / UTM ZONE 35S':
        'PROJCS["Arc 1950 / UTM zone 35S",GEOGCS["Arc 1950",DATUM["Arc_1950",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ARC 1950 / UTM ZONE 36S':
        'PROJCS["Arc 1950 / UTM zone 36S",GEOGCS["Arc 1950",DATUM["Arc_1950",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ARC 1960':
        'GEOGCS["Arc 1960",DATUM["Arc_1960",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'ARC 1960 / UTM ZONE 35N':
        'PROJCS["Arc 1960 / UTM zone 35N",GEOGCS["Arc 1960",DATUM["Arc_1960",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ARC 1960 / UTM ZONE 35S':
        'PROJCS["Arc 1960 / UTM zone 35S",GEOGCS["Arc 1960",DATUM["Arc_1960",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ARC 1960 / UTM ZONE 36N':
        'PROJCS["Arc 1960 / UTM zone 36N",GEOGCS["Arc 1960",DATUM["Arc_1960",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ARC 1960 / UTM ZONE 36S':
        'PROJCS["Arc 1960 / UTM zone 36S",GEOGCS["Arc 1960",DATUM["Arc_1960",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ARC 1960 / UTM ZONE 37N':
        'PROJCS["Arc 1960 / UTM zone 37N",GEOGCS["Arc 1960",DATUM["Arc_1960",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ARC 1960 / UTM ZONE 37S':
        'PROJCS["Arc 1960 / UTM zone 37S",GEOGCS["Arc 1960",DATUM["Arc_1960",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ATF (PARIS)':
        'GEOGCS["ATF (Paris)",DATUM["Ancienne_Triangulation_Francaise",SPHEROID["Plessis 1817",6376523,308.64]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]]',
    'ATF (PARIS) / NORD DE GUERRE':
        'PROJCS["ATF (Paris) / Nord de Guerre",GEOGCS["ATF (Paris)",DATUM["Ancienne_Triangulation_Francaise",SPHEROID["Plessis 1817",6376523,308.64]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",55],PARAMETER["central_meridian",6],PARAMETER["scale_factor",0.99950908],PARAMETER["false_easting",500000],PARAMETER["false_northing",300000],UNIT["metre",1]]',
    'ATS77':
        'GEOGCS["ATS77",DATUM["Average_Terrestrial_System_1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'ATS77 / MTM NOVA SCOTIA ZONE 4':
        'PROJCS["ATS77 / MTM Nova Scotia zone 4",GEOGCS["ATS77",DATUM["Average_Terrestrial_System_1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",61.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",4500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ATS77 / MTM NOVA SCOTIA ZONE 5':
        'PROJCS["ATS77 / MTM Nova Scotia zone 5",GEOGCS["ATS77",DATUM["Average_Terrestrial_System_1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",64.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",5500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ATS77 / NEW BRUNSWICK STEREOGRAPHIC':
        'PROJCS["ATS77 / New Brunswick Stereographic",GEOGCS["ATS77",DATUM["Average_Terrestrial_System_1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Oblique_Stereographic"],PARAMETER["latitude_of_origin",46.5],PARAMETER["central_meridian",-66.5],PARAMETER["scale_factor",0.999912],PARAMETER["false_easting",300000],PARAMETER["false_northing",800000],UNIT["metre",1]]',
    'ATS77 / PRINCE EDWARD ISL. STEREOGRAPHIC':
        'PROJCS["ATS77 / Prince Edward Isl. Stereographic",GEOGCS["ATS77",DATUM["Average_Terrestrial_System_1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Oblique_Stereographic"],PARAMETER["latitude_of_origin",47.25],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.999912],PARAMETER["false_easting",700000],PARAMETER["false_northing",400000],UNIT["metre",1]]',
    'ATS77 / UTM ZONE 19N':
        'PROJCS["ATS77 / UTM zone 19N",GEOGCS["ATS77",DATUM["Average_Terrestrial_System_1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ATS77 / UTM ZONE 20N':
        'PROJCS["ATS77 / UTM zone 20N",GEOGCS["ATS77",DATUM["Average_Terrestrial_System_1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BARBADOS':
        'GEOGCS["Barbados",DATUM["Barbados",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'BATAVIA':
        'GEOGCS["Batavia",DATUM["Batavia",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'BATAVIA (JAKARTA)':
        'GEOGCS["Batavia (Jakarta)",DATUM["Batavia",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Jakarta",106.807719444444],UNIT["degree",0.0174532925199433]]',
    'BATAVIA (JAKARTA) / NEIEZ':
        'PROJCS["Batavia (Jakarta) / NEIEZ",GEOGCS["Batavia (Jakarta)",DATUM["Batavia",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Jakarta",106.807719444444],UNIT["degree",0.0174532925199433]],PROJECTION["Mercator_1SP"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",110],PARAMETER["scale_factor",0.997],PARAMETER["false_easting",3900000],PARAMETER["false_northing",900000],UNIT["metre",1]]',
    'BATAVIA / UTM ZONE 48S':
        'PROJCS["Batavia / UTM zone 48S",GEOGCS["Batavia",DATUM["Batavia",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'BATAVIA / UTM ZONE 49S':
        'PROJCS["Batavia / UTM zone 49S",GEOGCS["Batavia",DATUM["Batavia",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'BATAVIA / UTM ZONE 50S':
        'PROJCS["Batavia / UTM zone 50S",GEOGCS["Batavia",DATUM["Batavia",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'BEDUARAM':
        'GEOGCS["Beduaram",DATUM["Beduaram",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'BEIJING 1954':
        'GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'BEIJING 1954 / GAUSS-KRUGER 13N':
        'PROJCS["Beijing 1954 / Gauss-Kruger 13N",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",75],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER 14N':
        'PROJCS["Beijing 1954 / Gauss-Kruger 14N",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",81],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER 15N':
        'PROJCS["Beijing 1954 / Gauss-Kruger 15N",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",87],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER 16N':
        'PROJCS["Beijing 1954 / Gauss-Kruger 16N",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER 17N':
        'PROJCS["Beijing 1954 / Gauss-Kruger 17N",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER 18N':
        'PROJCS["Beijing 1954 / Gauss-Kruger 18N",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER 19N':
        'PROJCS["Beijing 1954 / Gauss-Kruger 19N",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER 20N':
        'PROJCS["Beijing 1954 / Gauss-Kruger 20N",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER 21N':
        'PROJCS["Beijing 1954 / Gauss-Kruger 21N",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER 22N':
        'PROJCS["Beijing 1954 / Gauss-Kruger 22N",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER 23N':
        'PROJCS["Beijing 1954 / Gauss-Kruger 23N",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER ZONE 13':
        'PROJCS["Beijing 1954 / Gauss-Kruger zone 13",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",75],PARAMETER["scale_factor",1],PARAMETER["false_easting",13500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER ZONE 14':
        'PROJCS["Beijing 1954 / Gauss-Kruger zone 14",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",81],PARAMETER["scale_factor",1],PARAMETER["false_easting",14500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER ZONE 15':
        'PROJCS["Beijing 1954 / Gauss-Kruger zone 15",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",87],PARAMETER["scale_factor",1],PARAMETER["false_easting",15500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER ZONE 16':
        'PROJCS["Beijing 1954 / Gauss-Kruger zone 16",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",1],PARAMETER["false_easting",16500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER ZONE 17':
        'PROJCS["Beijing 1954 / Gauss-Kruger zone 17",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",1],PARAMETER["false_easting",17500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER ZONE 18':
        'PROJCS["Beijing 1954 / Gauss-Kruger zone 18",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",1],PARAMETER["false_easting",18500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER ZONE 19':
        'PROJCS["Beijing 1954 / Gauss-Kruger zone 19",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",1],PARAMETER["false_easting",19500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER ZONE 20':
        'PROJCS["Beijing 1954 / Gauss-Kruger zone 20",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",1],PARAMETER["false_easting",20500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER ZONE 21':
        'PROJCS["Beijing 1954 / Gauss-Kruger zone 21",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",1],PARAMETER["false_easting",21500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER ZONE 22':
        'PROJCS["Beijing 1954 / Gauss-Kruger zone 22",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",1],PARAMETER["false_easting",22500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BEIJING 1954 / GAUSS-KRUGER ZONE 23':
        'PROJCS["Beijing 1954 / Gauss-Kruger zone 23",GEOGCS["Beijing 1954",DATUM["Beijing_1954",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",1],PARAMETER["false_easting",23500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BELGE 1950':
        'GEOGCS["Belge 1950",DATUM["Reseau_National_Belge_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'BELGE 1950 (BRUSSELS)':
        'GEOGCS["Belge 1950 (Brussels)",DATUM["Reseau_National_Belge_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Brussels",4.367975],UNIT["degree",0.0174532925199433]]',
    'BELGE 1950 (BRUSSELS) / BELGE LAMBERT 50':
        'PROJCS["Belge 1950 (Brussels) / Belge Lambert 50",GEOGCS["Belge 1950 (Brussels)",DATUM["Reseau_National_Belge_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Brussels",4.367975],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",49.8333333333333],PARAMETER["standard_parallel_2",51.1666666666667],PARAMETER["latitude_of_origin",90],PARAMETER["central_meridian",0],PARAMETER["false_easting",150000],PARAMETER["false_northing",5400000],UNIT["metre",1]]',
    'BELGE 1972':
        'GEOGCS["Belge 1972",DATUM["Reseau_National_Belge_1972",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'BELGE 1972 / BELGE LAMBERT 72':
        'PROJCS["Belge 1972 / Belge Lambert 72",GEOGCS["Belge 1972",DATUM["Reseau_National_Belge_1972",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP_Belgium"],PARAMETER["standard_parallel_1",49.8333333333333],PARAMETER["standard_parallel_2",51.1666666666667],PARAMETER["latitude_of_origin",90],PARAMETER["central_meridian",4.35693972222222],PARAMETER["false_easting",150000.01256],PARAMETER["false_northing",5400088.4378],UNIT["metre",1]]',
    'BERMUDA 1957':
        'GEOGCS["Bermuda 1957",DATUM["Bermuda_1957",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'BERN 1898':
        'GEOGCS["Bern 1898",DATUM["Bern_1898",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'BERN 1898 (BERN)':
        'GEOGCS["Bern 1898 (Bern)",DATUM["Bern_1898",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Bern",7.43958333333333],UNIT["degree",0.0174532925199433]]',
    'BERN 1898 (BERN) / SWISS OLD GRID':
        'PROJCS["Bern 1898 (Bern) / Swiss Old Grid",GEOGCS["Bern 1898 (Bern)",DATUM["Bern_1898",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Bern",7.43958333333333],UNIT["degree",0.0174532925199433]],PROJECTION["Oblique_Mercator"],PARAMETER["latitude_of_center",46.9524055555556],PARAMETER["longitude_of_center",0],PARAMETER["azimuth",90],PARAMETER["rectified_grid_angle",90],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BERN 1898 / SWISS NEW GRID':
        'PROJCS["Bern 1898 / Swiss New Grid",GEOGCS["Bern 1898",DATUM["Bern_1898",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Oblique_Mercator"],PARAMETER["latitude_of_center",46.9524055555556],PARAMETER["longitude_of_center",7.43958333333333],PARAMETER["azimuth",90],PARAMETER["rectified_grid_angle",90],PARAMETER["scale_factor",1],PARAMETER["false_easting",600000],PARAMETER["false_northing",200000],UNIT["metre",1]]',
    'BERN 1938':
        'GEOGCS["Bern 1938",DATUM["Bern_1938",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'BOGOTA':
        'GEOGCS["Bogota",DATUM["Bogota",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'BOGOTA (BOGOTA)':
        'GEOGCS["Bogota (Bogota)",DATUM["Bogota",SPHEROID["International 1924",6378388,297]],PRIMEM["Bogota",-74.0809166666667],UNIT["degree",0.0174532925199433]]',
    'BOGOTA / COLOMBIA BOGOTA ZONE':
        'PROJCS["Bogota / Colombia Bogota zone",GEOGCS["Bogota",DATUM["Bogota",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",4.59904722222222],PARAMETER["central_meridian",-74.0809166666667],PARAMETER["scale_factor",1],PARAMETER["false_easting",1000000],PARAMETER["false_northing",1000000],UNIT["metre",1]]',
    'BOGOTA / COLOMBIA EAST':
        'PROJCS["Bogota / Colombia East",GEOGCS["Bogota",DATUM["Bogota",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",4.59904722222222],PARAMETER["central_meridian",-68.0809166666667],PARAMETER["scale_factor",1],PARAMETER["false_easting",1000000],PARAMETER["false_northing",1000000],UNIT["metre",1]]',
    'BOGOTA / COLOMBIA EAST CENTRAL ZONE':
        'PROJCS["Bogota / Colombia East Central zone",GEOGCS["Bogota",DATUM["Bogota",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",4.59904722222222],PARAMETER["central_meridian",-71.0809166666667],PARAMETER["scale_factor",1],PARAMETER["false_easting",1000000],PARAMETER["false_northing",1000000],UNIT["metre",1]]',
    'BOGOTA / COLOMBIA WEST ZONE':
        'PROJCS["Bogota / Colombia West zone",GEOGCS["Bogota",DATUM["Bogota",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",4.59904722222222],PARAMETER["central_meridian",-77.0809166666667],PARAMETER["scale_factor",1],PARAMETER["false_easting",1000000],PARAMETER["false_northing",1000000],UNIT["metre",1]]',
    'BOGOTA / UTM ZONE 17N':
        'PROJCS["Bogota / UTM zone 17N",GEOGCS["Bogota",DATUM["Bogota",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BOGOTA / UTM ZONE 18N':
        'PROJCS["Bogota / UTM zone 18N",GEOGCS["Bogota",DATUM["Bogota",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'BUKIT RIMPAH':
        'GEOGCS["Bukit Rimpah",DATUM["Bukit_Rimpah",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'CAMACUPA':
        'GEOGCS["Camacupa",DATUM["Camacupa",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'CAMACUPA / TM 11.30 SE':
        'PROJCS["Camacupa / TM 11.30 SE",GEOGCS["Camacupa",DATUM["Camacupa",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",11.5],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'CAMACUPA / TM 12 SE':
        'PROJCS["Camacupa / TM 12 SE",GEOGCS["Camacupa",DATUM["Camacupa",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",12],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'CAMACUPA / UTM ZONE 32S':
        'PROJCS["Camacupa / UTM zone 32S",GEOGCS["Camacupa",DATUM["Camacupa",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'CAMACUPA / UTM ZONE 33S':
        'PROJCS["Camacupa / UTM zone 33S",GEOGCS["Camacupa",DATUM["Camacupa",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'CAMPO INCHAUSPE':
        'GEOGCS["Campo Inchauspe",DATUM["Campo_Inchauspe",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'CAMPO INCHAUSPE / ARGENTINA 1':
        'PROJCS["Campo Inchauspe / Argentina 1",GEOGCS["Campo Inchauspe",DATUM["Campo_Inchauspe",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",-90],PARAMETER["central_meridian",-72],PARAMETER["scale_factor",1],PARAMETER["false_easting",1500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'CAMPO INCHAUSPE / ARGENTINA 2':
        'PROJCS["Campo Inchauspe / Argentina 2",GEOGCS["Campo Inchauspe",DATUM["Campo_Inchauspe",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",-90],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",1],PARAMETER["false_easting",2500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'CAMPO INCHAUSPE / ARGENTINA 3':
        'PROJCS["Campo Inchauspe / Argentina 3",GEOGCS["Campo Inchauspe",DATUM["Campo_Inchauspe",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",-90],PARAMETER["central_meridian",-66],PARAMETER["scale_factor",1],PARAMETER["false_easting",3500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'CAMPO INCHAUSPE / ARGENTINA 4':
        'PROJCS["Campo Inchauspe / Argentina 4",GEOGCS["Campo Inchauspe",DATUM["Campo_Inchauspe",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",-90],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",1],PARAMETER["false_easting",4500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'CAMPO INCHAUSPE / ARGENTINA 5':
        'PROJCS["Campo Inchauspe / Argentina 5",GEOGCS["Campo Inchauspe",DATUM["Campo_Inchauspe",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",-90],PARAMETER["central_meridian",-60],PARAMETER["scale_factor",1],PARAMETER["false_easting",5500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'CAMPO INCHAUSPE / ARGENTINA 6':
        'PROJCS["Campo Inchauspe / Argentina 6",GEOGCS["Campo Inchauspe",DATUM["Campo_Inchauspe",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",-90],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",1],PARAMETER["false_easting",6500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'CAMPO INCHAUSPE / ARGENTINA 7':
        'PROJCS["Campo Inchauspe / Argentina 7",GEOGCS["Campo Inchauspe",DATUM["Campo_Inchauspe",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",-90],PARAMETER["central_meridian",-54],PARAMETER["scale_factor",1],PARAMETER["false_easting",7500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'CAPE':
        'GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'CAPE / UTM ZONE 34S':
        'PROJCS["Cape / UTM zone 34S",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'CAPE / UTM ZONE 35S':
        'PROJCS["Cape / UTM zone 35S",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'CAPE / UTM ZONE 36S':
        'PROJCS["Cape / UTM zone 36S",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'CARTHAGE':
        'GEOGCS["Carthage",DATUM["Carthage",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'CARTHAGE (PARIS)':
        'GEOGCS["Carthage (Paris)",DATUM["Carthage",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]]',
    'CARTHAGE (PARIS) / TUNISIA MINING GRID':
        'PROJCS["Carthage (Paris) / Tunisia Mining Grid",GEOGCS["Carthage (Paris)",DATUM["Carthage",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]],PROJECTION["Tunisia_Mining_Grid"],PARAMETER["latitude_of_origin",38.81924],PARAMETER["central_meridian",7.84257],PARAMETER["false_easting",270.66603],PARAMETER["false_northing",581.95123],UNIT["kilometre",1000]]',
    'CARTHAGE / NORD TUNISIE':
        'PROJCS["Carthage / Nord Tunisie",GEOGCS["Carthage",DATUM["Carthage",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",36],PARAMETER["central_meridian",9.9],PARAMETER["scale_factor",0.999625544],PARAMETER["false_easting",500000],PARAMETER["false_northing",300000],UNIT["metre",1]]',
    'CARTHAGE / SUD TUNISIE':
        'PROJCS["Carthage / Sud Tunisie",GEOGCS["Carthage",DATUM["Carthage",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",33.3],PARAMETER["central_meridian",9.9],PARAMETER["scale_factor",0.999625769],PARAMETER["false_easting",500000],PARAMETER["false_northing",300000],UNIT["metre",1]]',
    'CARTHAGE / UTM ZONE 32N':
        'PROJCS["Carthage / UTM zone 32N",GEOGCS["Carthage",DATUM["Carthage",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'CHUA':
        'GEOGCS["Chua",DATUM["Chua",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'CONAKRY 1905':
        'GEOGCS["Conakry 1905",DATUM["Conakry_1905",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'CORREGO ALEGRE':
        'GEOGCS["Corrego Alegre",DATUM["Corrego_Alegre",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'CORREGO ALEGRE / UTM ZONE 23S':
        'PROJCS["Corrego Alegre / UTM zone 23S",GEOGCS["Corrego Alegre",DATUM["Corrego_Alegre",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'CORREGO ALEGRE / UTM ZONE 24S':
        'PROJCS["Corrego Alegre / UTM zone 24S",GEOGCS["Corrego Alegre",DATUM["Corrego_Alegre",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    "COTE D'IVOIRE":
        'GEOGCS["Cote d\'Ivoire",DATUM["Cote_d_Ivoire",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'DATUM 73':
        'GEOGCS["Datum 73",DATUM["Datum_73",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'DATUM 73 / UTM ZONE 29N':
        'PROJCS["Datum 73 / UTM zone 29N",GEOGCS["Datum 73",DATUM["Datum_73",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'DEALUL PISCULUI 1933':
        'GEOGCS["Dealul Piscului 1933",DATUM["Dealul_Piscului_1933",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'DEALUL PISCULUI 1933/ STEREO 33':
        'PROJCS["Dealul Piscului 1933/ Stereo 33",GEOGCS["Dealul Piscului 1933",DATUM["Dealul_Piscului_1933",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Oblique_Stereographic"],PARAMETER["latitude_of_origin",45.9],PARAMETER["central_meridian",25.3924658888889],PARAMETER["scale_factor",0.9996667],PARAMETER["false_easting",500000],PARAMETER["false_northing",500000],UNIT["metre",1]]',
    'DEALUL PISCULUI 1970':
        'GEOGCS["Dealul Piscului 1970",DATUM["Dealul_Piscului_1970",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'DEALUL PISCULUI 1970/ STEREO 70':
        'PROJCS["Dealul Piscului 1970/ Stereo 70",GEOGCS["Dealul Piscului 1970",DATUM["Dealul_Piscului_1970",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Oblique_Stereographic"],PARAMETER["latitude_of_origin",46],PARAMETER["central_meridian",25],PARAMETER["scale_factor",0.99975],PARAMETER["false_easting",500000],PARAMETER["false_northing",500000],UNIT["metre",1]]',
    'DEIR EZ ZOR':
        'GEOGCS["Deir ez Zor",DATUM["Deir_ez_Zor",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'DHDN':
        'GEOGCS["DHDN",DATUM["Deutsche_Hauptdreiecksnetz",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'DHDN / GERMANY ZONE 1':
        'PROJCS["DHDN / Germany zone 1",GEOGCS["DHDN",DATUM["Deutsche_Hauptdreiecksnetz",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",3],PARAMETER["scale_factor",1],PARAMETER["false_easting",1500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'DHDN / GERMANY ZONE 2':
        'PROJCS["DHDN / Germany zone 2",GEOGCS["DHDN",DATUM["Deutsche_Hauptdreiecksnetz",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",6],PARAMETER["scale_factor",1],PARAMETER["false_easting",2500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'DHDN / GERMANY ZONE 3':
        'PROJCS["DHDN / Germany zone 3",GEOGCS["DHDN",DATUM["Deutsche_Hauptdreiecksnetz",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",1],PARAMETER["false_easting",3500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'DHDN / GERMANY ZONE 4':
        'PROJCS["DHDN / Germany zone 4",GEOGCS["DHDN",DATUM["Deutsche_Hauptdreiecksnetz",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",12],PARAMETER["scale_factor",1],PARAMETER["false_easting",4500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'DHDN / GERMANY ZONE 5':
        'PROJCS["DHDN / Germany zone 5",GEOGCS["DHDN",DATUM["Deutsche_Hauptdreiecksnetz",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",1],PARAMETER["false_easting",5500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'DOUALA':
        'GEOGCS["Douala",DATUM["Douala",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'DOUALA / UTM ZONE 32N':
        'PROJCS["Douala / UTM zone 32N",GEOGCS["Douala",DATUM["Douala",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50':
        'GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'ED50 / TM 0 N':
        'PROJCS["ED50 / TM 0 N",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",0],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50 / TM 5 NE':
        'PROJCS["ED50 / TM 5 NE",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",5],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50 / UTM ZONE 28N':
        'PROJCS["ED50 / UTM zone 28N",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50 / UTM ZONE 29N':
        'PROJCS["ED50 / UTM zone 29N",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50 / UTM ZONE 30N':
        'PROJCS["ED50 / UTM zone 30N",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50 / UTM ZONE 31N':
        'PROJCS["ED50 / UTM zone 31N",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50 / UTM ZONE 32N':
        'PROJCS["ED50 / UTM zone 32N",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50 / UTM ZONE 33N':
        'PROJCS["ED50 / UTM zone 33N",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50 / UTM ZONE 34N':
        'PROJCS["ED50 / UTM zone 34N",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50 / UTM ZONE 35N':
        'PROJCS["ED50 / UTM zone 35N",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50 / UTM ZONE 36N':
        'PROJCS["ED50 / UTM zone 36N",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50 / UTM ZONE 37N':
        'PROJCS["ED50 / UTM zone 37N",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED50 / UTM ZONE 38N':
        'PROJCS["ED50 / UTM zone 38N",GEOGCS["ED50",DATUM["European_Datum_1950",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ED87':
        'GEOGCS["ED87",DATUM["European_Datum_1987",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'EGYPT 1907':
        'GEOGCS["Egypt 1907",DATUM["Egypt_1907",SPHEROID["Helmert 1906",6378200,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'EGYPT 1907 / EXTENDED PURPLE BELT':
        'PROJCS["Egypt 1907 / Extended Purple Belt",GEOGCS["Egypt 1907",DATUM["Egypt_1907",SPHEROID["Helmert 1906",6378200,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",30],PARAMETER["central_meridian",27],PARAMETER["scale_factor",1],PARAMETER["false_easting",700000],PARAMETER["false_northing",1200000],UNIT["metre",1]]',
    'EGYPT 1907 / PURPLE BELT':
        'PROJCS["Egypt 1907 / Purple Belt",GEOGCS["Egypt 1907",DATUM["Egypt_1907",SPHEROID["Helmert 1906",6378200,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",30],PARAMETER["central_meridian",27],PARAMETER["scale_factor",1],PARAMETER["false_easting",700000],PARAMETER["false_northing",200000],UNIT["metre",1]]',
    'EGYPT 1907 / RED BELT':
        'PROJCS["Egypt 1907 / Red Belt",GEOGCS["Egypt 1907",DATUM["Egypt_1907",SPHEROID["Helmert 1906",6378200,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",30],PARAMETER["central_meridian",31],PARAMETER["scale_factor",1],PARAMETER["false_easting",615000],PARAMETER["false_northing",810000],UNIT["metre",1]]',
    'ETRS89':
        'GEOGCS["ETRS89",DATUM["European_Reference_System_1989",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'ETRS89 / UTM ZONE 28N':
        'PROJCS["ETRS89 / UTM zone 28N",GEOGCS["ETRS89",DATUM["European_Reference_System_1989",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ETRS89 / UTM ZONE 29N':
        'PROJCS["ETRS89 / UTM zone 29N",GEOGCS["ETRS89",DATUM["European_Reference_System_1989",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ETRS89 / UTM ZONE 30N':
        'PROJCS["ETRS89 / UTM zone 30N",GEOGCS["ETRS89",DATUM["European_Reference_System_1989",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ETRS89 / UTM ZONE 31N':
        'PROJCS["ETRS89 / UTM zone 31N",GEOGCS["ETRS89",DATUM["European_Reference_System_1989",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ETRS89 / UTM ZONE 32N':
        'PROJCS["ETRS89 / UTM zone 32N",GEOGCS["ETRS89",DATUM["European_Reference_System_1989",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ETRS89 / UTM ZONE 33N':
        'PROJCS["ETRS89 / UTM zone 33N",GEOGCS["ETRS89",DATUM["European_Reference_System_1989",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ETRS89 / UTM ZONE 34N':
        'PROJCS["ETRS89 / UTM zone 34N",GEOGCS["ETRS89",DATUM["European_Reference_System_1989",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ETRS89 / UTM ZONE 35N':
        'PROJCS["ETRS89 / UTM zone 35N",GEOGCS["ETRS89",DATUM["European_Reference_System_1989",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ETRS89 / UTM ZONE 36N':
        'PROJCS["ETRS89 / UTM zone 36N",GEOGCS["ETRS89",DATUM["European_Reference_System_1989",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ETRS89 / UTM ZONE 37N':
        'PROJCS["ETRS89 / UTM zone 37N",GEOGCS["ETRS89",DATUM["European_Reference_System_1989",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ETRS89 / UTM ZONE 38N':
        'PROJCS["ETRS89 / UTM zone 38N",GEOGCS["ETRS89",DATUM["European_Reference_System_1989",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'FAHUD':
        'GEOGCS["Fahud",DATUM["Fahud",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'FAHUD / UTM ZONE 39N':
        'PROJCS["Fahud / UTM zone 39N",GEOGCS["Fahud",DATUM["Fahud",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'FAHUD / UTM ZONE 40N':
        'PROJCS["Fahud / UTM zone 40N",GEOGCS["Fahud",DATUM["Fahud",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'FD58':
        'GEOGCS["FD58",DATUM["Final_Datum_1958",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'FD58 / IRAQ ZONE':
        'PROJCS["FD58 / Iraq zone",GEOGCS["FD58",DATUM["Final_Datum_1958",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",32.5],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.99878640776699],PARAMETER["false_easting",1500000],PARAMETER["false_northing",1166200],UNIT["metre",1]]',
    'GANDAJIKA 1970':
        'GEOGCS["Gandajika 1970",DATUM["Gandajika_1970",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'GAROUA':
        'GEOGCS["Garoua",DATUM["Garoua",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'GAROUA / UTM ZONE 33N':
        'PROJCS["Garoua / UTM zone 33N",GEOGCS["Garoua",DATUM["Garoua",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'GD49':
        'GEOGCS["GD49",DATUM["New_Zealand_Geodetic_Datum_1949",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'GD49 / NEW ZEALAND MAP GRID':
        'PROJCS["GD49 / New Zealand Map Grid",GEOGCS["GD49",DATUM["New_Zealand_Geodetic_Datum_1949",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["New_Zealand_Map_Grid"],PARAMETER["latitude_of_origin",-41],PARAMETER["central_meridian",173],PARAMETER["false_easting",2510000],PARAMETER["false_northing",6023150],UNIT["metre",1]]',
    'GD49 / NORTH ISLAND GRID':
        'PROJCS["GD49 / North Island Grid",GEOGCS["GD49",DATUM["New_Zealand_Geodetic_Datum_1949",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",-39],PARAMETER["central_meridian",175.5],PARAMETER["scale_factor",1],PARAMETER["false_easting",300000],PARAMETER["false_northing",400000],UNIT["yard (Sears)",0.914398414616029]]',
    'GD49 / SOUTH ISLAND GRID':
        'PROJCS["GD49 / South Island Grid",GEOGCS["GD49",DATUM["New_Zealand_Geodetic_Datum_1949",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",-44],PARAMETER["central_meridian",171.5],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",500000],UNIT["yard (Sears)",0.914398414616029]]',
    'GDA94':
        'GEOGCS["GDA94",DATUM["Geocentric_Datum_of_Australia_1994",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'GDA94 / MGA ZONE 48':
        'PROJCS["GDA94 / MGA zone 48",GEOGCS["GDA94",DATUM["Geocentric_Datum_of_Australia_1994",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'GDA94 / MGA ZONE 49':
        'PROJCS["GDA94 / MGA zone 49",GEOGCS["GDA94",DATUM["Geocentric_Datum_of_Australia_1994",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'GDA94 / MGA ZONE 50':
        'PROJCS["GDA94 / MGA zone 50",GEOGCS["GDA94",DATUM["Geocentric_Datum_of_Australia_1994",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'GDA94 / MGA ZONE 51':
        'PROJCS["GDA94 / MGA zone 51",GEOGCS["GDA94",DATUM["Geocentric_Datum_of_Australia_1994",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'GDA94 / MGA ZONE 52':
        'PROJCS["GDA94 / MGA zone 52",GEOGCS["GDA94",DATUM["Geocentric_Datum_of_Australia_1994",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'GDA94 / MGA ZONE 53':
        'PROJCS["GDA94 / MGA zone 53",GEOGCS["GDA94",DATUM["Geocentric_Datum_of_Australia_1994",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'GDA94 / MGA ZONE 54':
        'PROJCS["GDA94 / MGA zone 54",GEOGCS["GDA94",DATUM["Geocentric_Datum_of_Australia_1994",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'GDA94 / MGA ZONE 55':
        'PROJCS["GDA94 / MGA zone 55",GEOGCS["GDA94",DATUM["Geocentric_Datum_of_Australia_1994",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'GDA94 / MGA ZONE 56':
        'PROJCS["GDA94 / MGA zone 56",GEOGCS["GDA94",DATUM["Geocentric_Datum_of_Australia_1994",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'GDA94 / MGA ZONE 57':
        'PROJCS["GDA94 / MGA zone 57",GEOGCS["GDA94",DATUM["Geocentric_Datum_of_Australia_1994",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'GDA94 / MGA ZONE 58':
        'PROJCS["GDA94 / MGA zone 58",GEOGCS["GDA94",DATUM["Geocentric_Datum_of_Australia_1994",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'GGRS87':
        'GEOGCS["GGRS87",DATUM["Greek_Geodetic_Reference_System_1987",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'GGRS87 / GREEK GRID':
        'PROJCS["GGRS87 / Greek Grid",GEOGCS["GGRS87",DATUM["Greek_Geodetic_Reference_System_1987",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",24],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'GREEK':
        'GEOGCS["Greek",DATUM["Greek",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'GREEK (ATHENS)':
        'GEOGCS["Greek (Athens)",DATUM["Greek",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Athens",23.7163375],UNIT["degree",0.0174532925199433]]',
    'GUYANE FRANCAISE':
        'GEOGCS["Guyane Francaise",DATUM["Guyane_Francaise",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'HD72':
        'GEOGCS["HD72",DATUM["Hungarian_Datum_1972",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'HD72 / EOV':
        'PROJCS["HD72 / EOV",GEOGCS["HD72",DATUM["Hungarian_Datum_1972",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Oblique_Mercator"],PARAMETER["latitude_of_center",47.1443937222222],PARAMETER["longitude_of_center",19.0485717777778],PARAMETER["azimuth",90],PARAMETER["rectified_grid_angle",90],PARAMETER["scale_factor",0.99993],PARAMETER["false_easting",650000],PARAMETER["false_northing",200000],UNIT["metre",1]]',
    'HERAT NORTH':
        'GEOGCS["Herat North",DATUM["Herat_North",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'HITO XVIII 1963':
        'GEOGCS["Hito XVIII 1963",DATUM["Hito_XVIII_1963",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'HU TZU SHAN':
        'GEOGCS["Hu Tzu Shan",DATUM["Hu_Tzu_Shan",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'ID74':
        'GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'ID74 / UTM ZONE 46N':
        'PROJCS["ID74 / UTM zone 46N",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ID74 / UTM ZONE 46S':
        'PROJCS["ID74 / UTM zone 46S",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ID74 / UTM ZONE 47N':
        'PROJCS["ID74 / UTM zone 47N",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ID74 / UTM ZONE 47S':
        'PROJCS["ID74 / UTM zone 47S",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ID74 / UTM ZONE 48N':
        'PROJCS["ID74 / UTM zone 48N",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ID74 / UTM ZONE 48S':
        'PROJCS["ID74 / UTM zone 48S",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ID74 / UTM ZONE 49N':
        'PROJCS["ID74 / UTM zone 49N",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ID74 / UTM ZONE 49S':
        'PROJCS["ID74 / UTM zone 49S",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ID74 / UTM ZONE 50N':
        'PROJCS["ID74 / UTM zone 50N",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ID74 / UTM ZONE 50S':
        'PROJCS["ID74 / UTM zone 50S",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ID74 / UTM ZONE 51N':
        'PROJCS["ID74 / UTM zone 51N",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ID74 / UTM ZONE 51S':
        'PROJCS["ID74 / UTM zone 51S",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ID74 / UTM ZONE 52N':
        'PROJCS["ID74 / UTM zone 52N",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ID74 / UTM ZONE 52S':
        'PROJCS["ID74 / UTM zone 52S",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ID74 / UTM ZONE 53N':
        'PROJCS["ID74 / UTM zone 53N",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ID74 / UTM ZONE 53S':
        'PROJCS["ID74 / UTM zone 53S",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'ID74 / UTM ZONE 54S':
        'PROJCS["ID74 / UTM zone 54S",GEOGCS["ID74",DATUM["Indonesian_Datum_1974",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'INDIAN 1954':
        'GEOGCS["Indian 1954",DATUM["Indian_1954",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'INDIAN 1954 / UTM ZONE 46N':
        'PROJCS["Indian 1954 / UTM zone 46N",GEOGCS["Indian 1954",DATUM["Indian_1954",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'INDIAN 1954 / UTM ZONE 47N':
        'PROJCS["Indian 1954 / UTM zone 47N",GEOGCS["Indian 1954",DATUM["Indian_1954",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'INDIAN 1954 / UTM ZONE 48N':
        'PROJCS["Indian 1954 / UTM zone 48N",GEOGCS["Indian 1954",DATUM["Indian_1954",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'INDIAN 1960':
        'GEOGCS["Indian 1960",DATUM["Indian_1960",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'INDIAN 1960 / TM 106 NE':
        'PROJCS["Indian 1960 / TM 106 NE",GEOGCS["Indian 1960",DATUM["Indian_1960",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",106],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'INDIAN 1960 / UTM ZONE 48N':
        'PROJCS["Indian 1960 / UTM zone 48N",GEOGCS["Indian 1960",DATUM["Indian_1960",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'INDIAN 1960 / UTM ZONE 49N':
        'PROJCS["Indian 1960 / UTM zone 49N",GEOGCS["Indian 1960",DATUM["Indian_1960",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'INDIAN 1975':
        'GEOGCS["Indian 1975",DATUM["Indian_1975",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'INDIAN 1975 / UTM ZONE 47N':
        'PROJCS["Indian 1975 / UTM zone 47N",GEOGCS["Indian 1975",DATUM["Indian_1975",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'INDIAN 1975 / UTM ZONE 48N':
        'PROJCS["Indian 1975 / UTM zone 48N",GEOGCS["Indian 1975",DATUM["Indian_1975",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'JAD69':
        'GEOGCS["JAD69",DATUM["Jamaica_1969",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'JAD69 / JAMAICA NATIONAL GRID':
        'PROJCS["JAD69 / Jamaica National Grid",GEOGCS["JAD69",DATUM["Jamaica_1969",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",18],PARAMETER["central_meridian",-77],PARAMETER["scale_factor",1],PARAMETER["false_easting",250000],PARAMETER["false_northing",150000],UNIT["metre",1]]',
    'JAMAICA 1875':
        'GEOGCS["Jamaica 1875",DATUM["Jamaica_1875",SPHEROID["Clarke 1880",6378249.13884613,293.466307655625]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'JAMAICA 1875 / JAMAICA (OLD GRID)':
        'PROJCS["Jamaica 1875 / Jamaica (Old Grid)",GEOGCS["Jamaica 1875",DATUM["Jamaica_1875",SPHEROID["Clarke 1880",6378249.13884613,293.466307655625]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",18],PARAMETER["central_meridian",-77],PARAMETER["scale_factor",1],PARAMETER["false_easting",550000],PARAMETER["false_northing",400000],UNIT["Clarke\'s foot",0.3047972651151]]',
    'KALIANPUR':
        'GEOGCS["Kalianpur",DATUM["Kalianpur",SPHEROID["Everest 1830 (1975 Definition)",6377301.243,300.801737341596]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'KALIANPUR / INDIA ZONE 0':
        'PROJCS["Kalianpur / India zone 0",GEOGCS["Kalianpur",DATUM["Kalianpur",SPHEROID["Everest 1830 (1975 Definition)",6377301.243,300.801737341596]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",39.5],PARAMETER["central_meridian",68],PARAMETER["scale_factor",0.998461538461539],PARAMETER["false_easting",2355500],PARAMETER["false_northing",2590000],UNIT["Indian yard",0.914398553970127]]',
    'KALIANPUR / INDIA ZONE I':
        'PROJCS["Kalianpur / India zone I",GEOGCS["Kalianpur",DATUM["Kalianpur",SPHEROID["Everest 1830 (1975 Definition)",6377301.243,300.801737341596]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",32.5],PARAMETER["central_meridian",68],PARAMETER["scale_factor",0.99878640776699],PARAMETER["false_easting",3000000],PARAMETER["false_northing",1000000],UNIT["Indian yard",0.914398553970127]]',
    'KALIANPUR / INDIA ZONE IIA':
        'PROJCS["Kalianpur / India zone IIa",GEOGCS["Kalianpur",DATUM["Kalianpur",SPHEROID["Everest 1830 (1975 Definition)",6377301.243,300.801737341596]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",26],PARAMETER["central_meridian",74],PARAMETER["scale_factor",0.99878640776699],PARAMETER["false_easting",3000000],PARAMETER["false_northing",1000000],UNIT["Indian yard",0.914398553970127]]',
    'KALIANPUR / INDIA ZONE IIB':
        'PROJCS["Kalianpur / India zone IIb",GEOGCS["Kalianpur",DATUM["Kalianpur",SPHEROID["Everest 1830 (1975 Definition)",6377301.243,300.801737341596]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",26],PARAMETER["central_meridian",90],PARAMETER["scale_factor",0.99878640776699],PARAMETER["false_easting",3000000],PARAMETER["false_northing",1000000],UNIT["Indian yard",0.914398553970127]]',
    'KALIANPUR / INDIA ZONE IIIA':
        'PROJCS["Kalianpur / India zone IIIa",GEOGCS["Kalianpur",DATUM["Kalianpur",SPHEROID["Everest 1830 (1975 Definition)",6377301.243,300.801737341596]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",19],PARAMETER["central_meridian",80],PARAMETER["scale_factor",0.99878640776699],PARAMETER["false_easting",3000000],PARAMETER["false_northing",1000000],UNIT["Indian yard",0.914398553970127]]',
    'KALIANPUR / INDIA ZONE IVA':
        'PROJCS["Kalianpur / India zone IVa",GEOGCS["Kalianpur",DATUM["Kalianpur",SPHEROID["Everest 1830 (1975 Definition)",6377301.243,300.801737341596]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",12],PARAMETER["central_meridian",80],PARAMETER["scale_factor",0.99878640776699],PARAMETER["false_easting",3000000],PARAMETER["false_northing",1000000],UNIT["Indian yard",0.914398553970127]]',
    'KANDAWALA':
        'GEOGCS["Kandawala",DATUM["Kandawala",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'KERTAU':
        'GEOGCS["Kertau",DATUM["Kertau",SPHEROID["Everest 1830 Modified",6377304.063,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'KERTAU / R.S.O. MALAYA (CH)':
        'PROJCS["Kertau / R.S.O. Malaya (ch)",GEOGCS["Kertau",DATUM["Kertau",SPHEROID["Everest 1830 Modified",6377304.063,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Hotine_Oblique_Mercator"],PARAMETER["latitude_of_center",4],PARAMETER["longitude_of_center",102.25],PARAMETER["azimuth",323.0257905],PARAMETER["rectified_grid_angle",323.130102361111],PARAMETER["scale_factor",0.99984],PARAMETER["false_easting",40000],PARAMETER["false_northing",0],UNIT["chain (Benoit)",20.1167824943759]]',
    'KERTAU / SINGAPORE GRID':
        'PROJCS["Kertau / Singapore Grid",GEOGCS["Kertau",DATUM["Kertau",SPHEROID["Everest 1830 Modified",6377304.063,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Cassini_Soldner"],PARAMETER["latitude_of_origin",1.28764666666667],PARAMETER["central_meridian",103.853002222222],PARAMETER["false_easting",30000],PARAMETER["false_northing",30000],UNIT["metre",1]]',
    'KERTAU / UTM ZONE 47N':
        'PROJCS["Kertau / UTM zone 47N",GEOGCS["Kertau",DATUM["Kertau",SPHEROID["Everest 1830 Modified",6377304.063,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'KERTAU / UTM ZONE 48N':
        'PROJCS["Kertau / UTM zone 48N",GEOGCS["Kertau",DATUM["Kertau",SPHEROID["Everest 1830 Modified",6377304.063,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'KKJ':
        'GEOGCS["KKJ",DATUM["Kartastokoordinaattijarjestelma",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'KKJ / FINLAND UNIFORM COORDINATE SYSTEM':
        'PROJCS["KKJ / Finland Uniform Coordinate System",GEOGCS["KKJ",DATUM["Kartastokoordinaattijarjestelma",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",1],PARAMETER["false_easting",3500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'KKJ / FINLAND ZONE 1':
        'PROJCS["KKJ / Finland zone 1",GEOGCS["KKJ",DATUM["Kartastokoordinaattijarjestelma",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",1],PARAMETER["false_easting",1500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'KKJ / FINLAND ZONE 2':
        'PROJCS["KKJ / Finland zone 2",GEOGCS["KKJ",DATUM["Kartastokoordinaattijarjestelma",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",24],PARAMETER["scale_factor",1],PARAMETER["false_easting",2500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'KKJ / FINLAND ZONE 4':
        'PROJCS["KKJ / Finland zone 4",GEOGCS["KKJ",DATUM["Kartastokoordinaattijarjestelma",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",30],PARAMETER["scale_factor",1],PARAMETER["false_easting",4500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'KOC':
        'GEOGCS["KOC",DATUM["Kuwait_Oil_Company",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'KOC LAMBERT':
        'PROJCS["KOC Lambert",GEOGCS["KOC",DATUM["Kuwait_Oil_Company",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",32.5],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.99878640776699],PARAMETER["false_easting",1500000],PARAMETER["false_northing",1166200],UNIT["metre",1]]',
    'KUDAMS':
        'GEOGCS["KUDAMS",DATUM["Kuwait_Utility",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'KUDAMS / KTM':
        'PROJCS["KUDAMS / KTM",GEOGCS["KUDAMS",DATUM["Kuwait_Utility",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",48],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'LA CANOA':
        'GEOGCS["La Canoa",DATUM["La_Canoa",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'LA CANOA / UTM ZONE 20N':
        'PROJCS["La Canoa / UTM zone 20N",GEOGCS["La Canoa",DATUM["La_Canoa",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'LA CANOA / UTM ZONE 21N':
        'PROJCS["La Canoa / UTM zone 21N",GEOGCS["La Canoa",DATUM["La_Canoa",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'LAKE':
        'GEOGCS["Lake",DATUM["Lake",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'LEIGON':
        'GEOGCS["Leigon",DATUM["Leigon",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'LEIGON / GHANA METRE GRID':
        'PROJCS["Leigon / Ghana Metre Grid",GEOGCS["Leigon",DATUM["Leigon",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",4.66666666666667],PARAMETER["central_meridian",-1],PARAMETER["scale_factor",0.99975],PARAMETER["false_easting",274319.51],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'LIBERIA 1964':
        'GEOGCS["Liberia 1964",DATUM["Liberia_1964",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'LIETUVOS KOORDINOEI SISTEMA 1994':
        'PROJCS["Lietuvos Koordinoei Sistema 1994",GEOGCS["LKS94 (ETRS89)",DATUM["Lithuania_1994_ETRS89",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",24],PARAMETER["scale_factor",0.9998],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'LISBON':
        'GEOGCS["Lisbon",DATUM["Lisbon",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'LISBON (LISBON)':
        'GEOGCS["Lisbon (Lisbon)",DATUM["Lisbon",SPHEROID["International 1924",6378388,297]],PRIMEM["Lisbon",-9.13190611111111],UNIT["degree",0.0174532925199433]]',
    'LISBON (LISBON)/PORTUGUESE NATIONAL GRID':
        'PROJCS["Lisbon (Lisbon)/Portuguese National Grid",GEOGCS["Lisbon (Lisbon)",DATUM["Lisbon",SPHEROID["International 1924",6378388,297]],PRIMEM["Lisbon",-9.13190611111111],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",39.6666666666667],PARAMETER["central_meridian",1],PARAMETER["scale_factor",1],PARAMETER["false_easting",200000],PARAMETER["false_northing",300000],UNIT["metre",1]]',
    'LKS94 (ETRS89)':
        'GEOGCS["LKS94 (ETRS89)",DATUM["Lithuania_1994_ETRS89",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'LOMA QUINTANA':
        'GEOGCS["Loma Quintana",DATUM["Loma_Quintana",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'LOME':
        'GEOGCS["Lome",DATUM["Lome",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'LOME / UTM ZONE 31N':
        'PROJCS["Lome / UTM zone 31N",GEOGCS["Lome",DATUM["Lome",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'LUZON 1911':
        'GEOGCS["Luzon 1911",DATUM["Luzon_1911",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'LUZON 1911 / PHILIPPINES ZONE I':
        'PROJCS["Luzon 1911 / Philippines zone I",GEOGCS["Luzon 1911",DATUM["Luzon_1911",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.99995],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'LUZON 1911 / PHILIPPINES ZONE II':
        'PROJCS["Luzon 1911 / Philippines zone II",GEOGCS["Luzon 1911",DATUM["Luzon_1911",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",119],PARAMETER["scale_factor",0.99995],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'LUZON 1911 / PHILIPPINES ZONE III':
        'PROJCS["Luzon 1911 / Philippines zone III",GEOGCS["Luzon 1911",DATUM["Luzon_1911",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",121],PARAMETER["scale_factor",0.99995],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'LUZON 1911 / PHILIPPINES ZONE IV':
        'PROJCS["Luzon 1911 / Philippines zone IV",GEOGCS["Luzon 1911",DATUM["Luzon_1911",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",0.99995],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'LUZON 1911 / PHILIPPINES ZONE V':
        'PROJCS["Luzon 1911 / Philippines zone V",GEOGCS["Luzon 1911",DATUM["Luzon_1911",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",125],PARAMETER["scale_factor",0.99995],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    "M'PORALOKO":
        'GEOGCS["M\'poraloko",DATUM["M_poraloko",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    "M'PORALOKO / UTM ZONE 32N":
        'PROJCS["M\'poraloko / UTM zone 32N",GEOGCS["M\'poraloko",DATUM["M_poraloko",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    "M'PORALOKO / UTM ZONE 32S":
        'PROJCS["M\'poraloko / UTM zone 32S",GEOGCS["M\'poraloko",DATUM["M_poraloko",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'MADRID 1870 (MADRID)':
        'GEOGCS["Madrid 1870 (Madrid)",DATUM["Madrid_1870",SPHEROID["Struve 1860",6378298.3,294.73]],PRIMEM["Madrid",-3.68793888888889],UNIT["degree",0.0174532925199433]]',
    'MADRID 1870 (MADRID) / SPAIN':
        'PROJCS["Madrid 1870 (Madrid) / Spain",GEOGCS["Madrid 1870 (Madrid)",DATUM["Madrid_1870",SPHEROID["Struve 1860",6378298.3,294.73]],PRIMEM["Madrid",-3.68793888888889],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",40],PARAMETER["central_meridian",0],PARAMETER["scale_factor",0.9988085293],PARAMETER["false_easting",600000],PARAMETER["false_northing",600000],UNIT["metre",1]]',
    'MADZANSUA':
        'GEOGCS["Madzansua",DATUM["Madzansua",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'MAHE 1971':
        'GEOGCS["Mahe 1971",DATUM["Mahe_1971",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'MAKASSAR':
        'GEOGCS["Makassar",DATUM["Makassar",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'MAKASSAR (JAKARTA)':
        'GEOGCS["Makassar (Jakarta)",DATUM["Makassar",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Jakarta",106.807719444444],UNIT["degree",0.0174532925199433]]',
    'MAKASSAR (JAKARTA) / NEIEZ':
        'PROJCS["Makassar (Jakarta) / NEIEZ",GEOGCS["Makassar (Jakarta)",DATUM["Makassar",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Jakarta",106.807719444444],UNIT["degree",0.0174532925199433]],PROJECTION["Mercator_1SP"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",110],PARAMETER["scale_factor",0.997],PARAMETER["false_easting",3900000],PARAMETER["false_northing",900000],UNIT["metre",1]]',
    'MALONGO 1987':
        'GEOGCS["Malongo 1987",DATUM["Malongo_1987",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'MALONGO 1987 / UTM ZONE 32S':
        'PROJCS["Malongo 1987 / UTM zone 32S",GEOGCS["Malongo 1987",DATUM["Malongo_1987",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'MANOCA':
        'GEOGCS["Manoca",DATUM["Manoca",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'MASSAWA':
        'GEOGCS["Massawa",DATUM["Massawa",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'MASSAWA / UTM ZONE 37N':
        'PROJCS["Massawa / UTM zone 37N",GEOGCS["Massawa",DATUM["Massawa",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'MERCHICH':
        'GEOGCS["Merchich",DATUM["Merchich",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'MERCHICH / NORD MAROC':
        'PROJCS["Merchich / Nord Maroc",GEOGCS["Merchich",DATUM["Merchich",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",33.3],PARAMETER["central_meridian",-5.4],PARAMETER["scale_factor",0.999625769],PARAMETER["false_easting",500000],PARAMETER["false_northing",300000],UNIT["metre",1]]',
    'MERCHICH / SAHARA':
        'PROJCS["Merchich / Sahara",GEOGCS["Merchich",DATUM["Merchich",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",26.1],PARAMETER["central_meridian",-5.4],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",1200000],PARAMETER["false_northing",400000],UNIT["metre",1]]',
    'MERCHICH / SUD MAROC':
        'PROJCS["Merchich / Sud Maroc",GEOGCS["Merchich",DATUM["Merchich",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",29.7],PARAMETER["central_meridian",-5.4],PARAMETER["scale_factor",0.999615596],PARAMETER["false_easting",500000],PARAMETER["false_northing",300000],UNIT["metre",1]]',
    'MGI':
        'GEOGCS["MGI",DATUM["Militar_Geographische_Institute",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'MGI (FERRO)':
        'GEOGCS["MGI (Ferro)",DATUM["Militar_Geographische_Institute",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Ferro",-17.6666666666667],UNIT["degree",0.0174532925199433]]',
    'MGI (FERRO) / AUSTRIA CENTRAL ZONE':
        'PROJCS["MGI (Ferro) / Austria Central Zone",GEOGCS["MGI (Ferro)",DATUM["Militar_Geographische_Institute",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Ferro",-17.6666666666667],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",31],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'MGI (FERRO) / AUSTRIA EAST ZONE':
        'PROJCS["MGI (Ferro) / Austria East Zone",GEOGCS["MGI (Ferro)",DATUM["Militar_Geographische_Institute",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Ferro",-17.6666666666667],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",34],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'MGI (FERRO) / AUSTRIA WEST ZONE':
        'PROJCS["MGI (Ferro) / Austria West Zone",GEOGCS["MGI (Ferro)",DATUM["Militar_Geographische_Institute",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Ferro",-17.6666666666667],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",28],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'MHAST':
        'GEOGCS["Mhast",DATUM["Mhast",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'MHAST / UTM ZONE 32S':
        'PROJCS["Mhast / UTM zone 32S",GEOGCS["Mhast",DATUM["Mhast",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'MINNA':
        'GEOGCS["Minna",DATUM["Minna",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'MINNA / NIGERIA EAST BELT':
        'PROJCS["Minna / Nigeria East Belt",GEOGCS["Minna",DATUM["Minna",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",4],PARAMETER["central_meridian",12.5],PARAMETER["scale_factor",0.99975],PARAMETER["false_easting",1110369.7],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'MINNA / NIGERIA MID BELT':
        'PROJCS["Minna / Nigeria Mid Belt",GEOGCS["Minna",DATUM["Minna",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",4],PARAMETER["central_meridian",8.5],PARAMETER["scale_factor",0.99975],PARAMETER["false_easting",670553.98],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'MINNA / NIGERIA WEST BELT':
        'PROJCS["Minna / Nigeria West Belt",GEOGCS["Minna",DATUM["Minna",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",4],PARAMETER["central_meridian",4.5],PARAMETER["scale_factor",0.99975],PARAMETER["false_easting",230738.26],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'MINNA / UTM ZONE 31N':
        'PROJCS["Minna / UTM zone 31N",GEOGCS["Minna",DATUM["Minna",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'MINNA / UTM ZONE 32N':
        'PROJCS["Minna / UTM zone 32N",GEOGCS["Minna",DATUM["Minna",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'MONTE MARIO':
        'GEOGCS["Monte Mario",DATUM["Monte_Mario",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'MONTE MARIO (ROME)':
        'GEOGCS["Monte Mario (Rome)",DATUM["Monte_Mario",SPHEROID["International 1924",6378388,297]],PRIMEM["Rome",12.4523333333333],UNIT["degree",0.0174532925199433]]',
    'MONTE MARIO (ROME) / ITALY ZONE 1':
        'PROJCS["Monte Mario (Rome) / Italy zone 1",GEOGCS["Monte Mario (Rome)",DATUM["Monte_Mario",SPHEROID["International 1924",6378388,297]],PRIMEM["Rome",12.4523333333333],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",1500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'MONTE MARIO (ROME) / ITALY ZONE 2':
        'PROJCS["Monte Mario (Rome) / Italy zone 2",GEOGCS["Monte Mario (Rome)",DATUM["Monte_Mario",SPHEROID["International 1924",6378388,297]],PRIMEM["Rome",12.4523333333333],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",2520000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'MOZNET':
        'GEOGCS["Moznet",DATUM["Moznet_ITRF94",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'MOZNET / UTM ZONE 36S':
        'PROJCS["Moznet / UTM zone 36S",GEOGCS["Moznet",DATUM["Moznet_ITRF94",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'MOZNET / UTM ZONE 37S':
        'PROJCS["Moznet / UTM zone 37S",GEOGCS["Moznet",DATUM["Moznet_ITRF94",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'NAD MICHIGAN / MICHIGAN CENTRAL':
        'PROJCS["NAD Michigan / Michigan Central",GEOGCS["NAD27 Michigan",DATUM["NAD_Michigan",SPHEROID["Clarke 1866 Michigan",6378450.0472441,294.978684676652]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",44.1833333333333],PARAMETER["standard_parallel_2",45.7],PARAMETER["latitude_of_origin",43.3166666666667],PARAMETER["central_meridian",-84.3333333333333],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD MICHIGAN / MICHIGAN EAST':
        'PROJCS["NAD Michigan / Michigan East",GEOGCS["NAD27 Michigan",DATUM["NAD_Michigan",SPHEROID["Clarke 1866 Michigan",6378450.0472441,294.978684676652]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",41.5],PARAMETER["central_meridian",-83.6666666666667],PARAMETER["scale_factor",0.999942857],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD MICHIGAN / MICHIGAN NORTH':
        'PROJCS["NAD Michigan / Michigan North",GEOGCS["NAD27 Michigan",DATUM["NAD_Michigan",SPHEROID["Clarke 1866 Michigan",6378450.0472441,294.978684676652]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",45.4833333333333],PARAMETER["standard_parallel_2",47.0833333333333],PARAMETER["latitude_of_origin",44.7833333333333],PARAMETER["central_meridian",-87],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD MICHIGAN / MICHIGAN OLD CENTRAL':
        'PROJCS["NAD Michigan / Michigan Old Central",GEOGCS["NAD27 Michigan",DATUM["NAD_Michigan",SPHEROID["Clarke 1866 Michigan",6378450.0472441,294.978684676652]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",41.5],PARAMETER["central_meridian",-85.75],PARAMETER["scale_factor",0.999909091],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD MICHIGAN / MICHIGAN SOUTH':
        'PROJCS["NAD Michigan / Michigan South",GEOGCS["NAD27 Michigan",DATUM["NAD_Michigan",SPHEROID["Clarke 1866 Michigan",6378450.0472441,294.978684676652]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",42.1],PARAMETER["standard_parallel_2",43.6666666666667],PARAMETER["latitude_of_origin",41.5],PARAMETER["central_meridian",-84.3333333333333],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD MICHIGAN / MICHIGAN WEST':
        'PROJCS["NAD Michigan / Michigan West",GEOGCS["NAD27 Michigan",DATUM["NAD_Michigan",SPHEROID["Clarke 1866 Michigan",6378450.0472441,294.978684676652]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",41.5],PARAMETER["central_meridian",-88.75],PARAMETER["scale_factor",0.999909091],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27':
        'GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'NAD27 / ALABAMA EAST':
        'PROJCS["NAD27 / Alabama East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",30.5],PARAMETER["central_meridian",-85.8333333333333],PARAMETER["scale_factor",0.99996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ALABAMA WEST':
        'PROJCS["NAD27 / Alabama West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",30],PARAMETER["central_meridian",-87.5],PARAMETER["scale_factor",0.999933333],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ALASKA ZONE 1':
        'PROJCS["NAD27 / Alaska zone 1",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Hotine_Oblique_Mercator"],PARAMETER["latitude_of_center",57],PARAMETER["longitude_of_center",-133.666666666667],PARAMETER["azimuth",323.130102361111],PARAMETER["rectified_grid_angle",323.130102361111],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",16404166.67],PARAMETER["false_northing",-16404166.67],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ALASKA ZONE 10':
        'PROJCS["NAD27 / Alaska zone 10",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",53.8333333333333],PARAMETER["standard_parallel_2",51.8333333333333],PARAMETER["latitude_of_origin",51],PARAMETER["central_meridian",-176],PARAMETER["false_easting",3000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ALASKA ZONE 2':
        'PROJCS["NAD27 / Alaska zone 2",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-142],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ALASKA ZONE 3':
        'PROJCS["NAD27 / Alaska zone 3",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-146],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ALASKA ZONE 4':
        'PROJCS["NAD27 / Alaska zone 4",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-150],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ALASKA ZONE 5':
        'PROJCS["NAD27 / Alaska zone 5",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-154],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ALASKA ZONE 6':
        'PROJCS["NAD27 / Alaska zone 6",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-158],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ALASKA ZONE 7':
        'PROJCS["NAD27 / Alaska zone 7",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-162],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",700000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ALASKA ZONE 8':
        'PROJCS["NAD27 / Alaska zone 8",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-166],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ALASKA ZONE 9':
        'PROJCS["NAD27 / Alaska zone 9",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-170],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ARIZONA CENTRAL':
        'PROJCS["NAD27 / Arizona Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",31],PARAMETER["central_meridian",-111.916666666667],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ARIZONA EAST':
        'PROJCS["NAD27 / Arizona East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",31],PARAMETER["central_meridian",-110.166666666667],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ARIZONA WEST':
        'PROJCS["NAD27 / Arizona West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",31],PARAMETER["central_meridian",-113.75],PARAMETER["scale_factor",0.999933333],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ARKANSAS NORTH':
        'PROJCS["NAD27 / Arkansas North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",36.2333333333333],PARAMETER["standard_parallel_2",34.9333333333333],PARAMETER["latitude_of_origin",34.3333333333333],PARAMETER["central_meridian",-92],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ARKANSAS SOUTH':
        'PROJCS["NAD27 / Arkansas South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",34.7666666666667],PARAMETER["standard_parallel_2",33.3],PARAMETER["latitude_of_origin",32.6666666666667],PARAMETER["central_meridian",-92],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / BLM 14N (FEET)':
        'PROJCS["NAD27 / BLM 14N (feet)",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",1640416.67],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / BLM 15N (FEET)':
        'PROJCS["NAD27 / BLM 15N (feet)",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",1640416.67],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / BLM 16N (FEET)':
        'PROJCS["NAD27 / BLM 16N (feet)",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",1640416.67],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / BLM 17N (FEET)':
        'PROJCS["NAD27 / BLM 17N (feet)",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",1640416.67],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / CALIFORNIA ZONE I':
        'PROJCS["NAD27 / California zone I",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.6666666666667],PARAMETER["standard_parallel_2",40],PARAMETER["latitude_of_origin",39.3333333333333],PARAMETER["central_meridian",-122],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / CALIFORNIA ZONE II':
        'PROJCS["NAD27 / California zone II",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",39.8333333333333],PARAMETER["standard_parallel_2",38.3333333333333],PARAMETER["latitude_of_origin",37.6666666666667],PARAMETER["central_meridian",-122],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / CALIFORNIA ZONE III':
        'PROJCS["NAD27 / California zone III",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",38.4333333333333],PARAMETER["standard_parallel_2",37.0666666666667],PARAMETER["latitude_of_origin",36.5],PARAMETER["central_meridian",-120.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / CALIFORNIA ZONE IV':
        'PROJCS["NAD27 / California zone IV",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",37.25],PARAMETER["standard_parallel_2",36],PARAMETER["latitude_of_origin",35.3333333333333],PARAMETER["central_meridian",-119],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / CALIFORNIA ZONE V':
        'PROJCS["NAD27 / California zone V",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",35.4666666666667],PARAMETER["standard_parallel_2",34.0333333333333],PARAMETER["latitude_of_origin",33.5],PARAMETER["central_meridian",-118],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / CALIFORNIA ZONE VI':
        'PROJCS["NAD27 / California zone VI",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",33.8833333333333],PARAMETER["standard_parallel_2",32.7833333333333],PARAMETER["latitude_of_origin",32.1666666666667],PARAMETER["central_meridian",-116.25],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / CALIFORNIA ZONE VII':
        'PROJCS["NAD27 / California zone VII",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",34.4166666666667],PARAMETER["standard_parallel_2",33.8666666666667],PARAMETER["latitude_of_origin",34.1333333333333],PARAMETER["central_meridian",-118.333333333333],PARAMETER["false_easting",4186692.58],PARAMETER["false_northing",416926.74],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / COLORADO CENTRAL':
        'PROJCS["NAD27 / Colorado Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",39.75],PARAMETER["standard_parallel_2",38.45],PARAMETER["latitude_of_origin",37.8333333333333],PARAMETER["central_meridian",-105.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / COLORADO NORTH':
        'PROJCS["NAD27 / Colorado North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",39.7166666666667],PARAMETER["standard_parallel_2",40.7833333333333],PARAMETER["latitude_of_origin",39.3333333333333],PARAMETER["central_meridian",-105.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / COLORADO SOUTH':
        'PROJCS["NAD27 / Colorado South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",38.4333333333333],PARAMETER["standard_parallel_2",37.2333333333333],PARAMETER["latitude_of_origin",36.6666666666667],PARAMETER["central_meridian",-105.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / CONNECTICUT':
        'PROJCS["NAD27 / Connecticut",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.8666666666667],PARAMETER["standard_parallel_2",41.2],PARAMETER["latitude_of_origin",40.8333333333333],PARAMETER["central_meridian",-72.75],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / DELAWARE':
        'PROJCS["NAD27 / Delaware",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",38],PARAMETER["central_meridian",-75.4166666666667],PARAMETER["scale_factor",0.999995],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / FLORIDA EAST':
        'PROJCS["NAD27 / Florida East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",24.3333333333333],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / FLORIDA NORTH':
        'PROJCS["NAD27 / Florida North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",30.75],PARAMETER["standard_parallel_2",29.5833333333333],PARAMETER["latitude_of_origin",29],PARAMETER["central_meridian",-84.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / FLORIDA WEST':
        'PROJCS["NAD27 / Florida West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",24.3333333333333],PARAMETER["central_meridian",-82],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / GEORGIA EAST':
        'PROJCS["NAD27 / Georgia East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",30],PARAMETER["central_meridian",-82.1666666666667],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / GEORGIA WEST':
        'PROJCS["NAD27 / Georgia West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",30],PARAMETER["central_meridian",-84.1666666666667],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / HAWAII ZONE 1':
        'PROJCS["NAD27 / Hawaii zone 1",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",18.8333333333333],PARAMETER["central_meridian",-155.5],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / HAWAII ZONE 2':
        'PROJCS["NAD27 / Hawaii zone 2",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",20.3333333333333],PARAMETER["central_meridian",-156.666666666667],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / HAWAII ZONE 3':
        'PROJCS["NAD27 / Hawaii zone 3",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",21.1666666666667],PARAMETER["central_meridian",-158],PARAMETER["scale_factor",0.99999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / HAWAII ZONE 4':
        'PROJCS["NAD27 / Hawaii zone 4",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",21.8333333333333],PARAMETER["central_meridian",-159.5],PARAMETER["scale_factor",0.99999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / HAWAII ZONE 5':
        'PROJCS["NAD27 / Hawaii zone 5",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",21.6666666666667],PARAMETER["central_meridian",-160.166666666667],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / IDAHO CENTRAL':
        'PROJCS["NAD27 / Idaho Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",41.6666666666667],PARAMETER["central_meridian",-114],PARAMETER["scale_factor",0.999947368],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / IDAHO EAST':
        'PROJCS["NAD27 / Idaho East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",41.6666666666667],PARAMETER["central_meridian",-112.166666666667],PARAMETER["scale_factor",0.999947368],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / IDAHO WEST':
        'PROJCS["NAD27 / Idaho West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",41.6666666666667],PARAMETER["central_meridian",-115.75],PARAMETER["scale_factor",0.999933333],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ILLINOIS EAST':
        'PROJCS["NAD27 / Illinois East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",36.6666666666667],PARAMETER["central_meridian",-88.3333333333333],PARAMETER["scale_factor",0.999975],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ILLINOIS WEST':
        'PROJCS["NAD27 / Illinois West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",36.6666666666667],PARAMETER["central_meridian",-90.1666666666667],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / INDIANA EAST':
        'PROJCS["NAD27 / Indiana East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",37.5],PARAMETER["central_meridian",-85.6666666666667],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / INDIANA WEST':
        'PROJCS["NAD27 / Indiana West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",37.5],PARAMETER["central_meridian",-87.0833333333333],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / IOWA NORTH':
        'PROJCS["NAD27 / Iowa North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",43.2666666666667],PARAMETER["standard_parallel_2",42.0666666666667],PARAMETER["latitude_of_origin",41.5],PARAMETER["central_meridian",-93.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / IOWA SOUTH':
        'PROJCS["NAD27 / Iowa South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.7833333333333],PARAMETER["standard_parallel_2",40.6166666666667],PARAMETER["latitude_of_origin",40],PARAMETER["central_meridian",-93.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / KANSAS NORTH':
        'PROJCS["NAD27 / Kansas North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",39.7833333333333],PARAMETER["standard_parallel_2",38.7166666666667],PARAMETER["latitude_of_origin",38.3333333333333],PARAMETER["central_meridian",-98],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / KANSAS SOUTH':
        'PROJCS["NAD27 / Kansas South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",38.5666666666667],PARAMETER["standard_parallel_2",37.2666666666667],PARAMETER["latitude_of_origin",36.6666666666667],PARAMETER["central_meridian",-98.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / KENTUCKY NORTH':
        'PROJCS["NAD27 / Kentucky North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",37.9666666666667],PARAMETER["standard_parallel_2",38.9666666666667],PARAMETER["latitude_of_origin",37.5],PARAMETER["central_meridian",-84.25],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / KENTUCKY SOUTH':
        'PROJCS["NAD27 / Kentucky South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",36.7333333333333],PARAMETER["standard_parallel_2",37.9333333333333],PARAMETER["latitude_of_origin",36.3333333333333],PARAMETER["central_meridian",-85.75],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / LOUISIANA NORTH':
        'PROJCS["NAD27 / Louisiana North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",31.1666666666667],PARAMETER["standard_parallel_2",32.6666666666667],PARAMETER["latitude_of_origin",30.6666666666667],PARAMETER["central_meridian",-92.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / LOUISIANA SOUTH':
        'PROJCS["NAD27 / Louisiana South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",29.3],PARAMETER["standard_parallel_2",30.7],PARAMETER["latitude_of_origin",28.6666666666667],PARAMETER["central_meridian",-91.3333333333333],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MAINE EAST':
        'PROJCS["NAD27 / Maine East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",43.8333333333333],PARAMETER["central_meridian",-68.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MAINE WEST':
        'PROJCS["NAD27 / Maine West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",42.8333333333333],PARAMETER["central_meridian",-70.1666666666667],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MARYLAND':
        'PROJCS["NAD27 / Maryland",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",38.3],PARAMETER["standard_parallel_2",39.45],PARAMETER["latitude_of_origin",37.8333333333333],PARAMETER["central_meridian",-77],PARAMETER["false_easting",800000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MASSACHUSETTS ISLAND':
        'PROJCS["NAD27 / Massachusetts Island",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.2833333333333],PARAMETER["standard_parallel_2",41.4833333333333],PARAMETER["latitude_of_origin",41],PARAMETER["central_meridian",-70.5],PARAMETER["false_easting",200000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MASSACHUSETTS MAINLAND':
        'PROJCS["NAD27 / Massachusetts Mainland",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.7166666666667],PARAMETER["standard_parallel_2",42.6833333333333],PARAMETER["latitude_of_origin",41],PARAMETER["central_meridian",-71.5],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MINNESOTA CENTRAL':
        'PROJCS["NAD27 / Minnesota Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",45.6166666666667],PARAMETER["standard_parallel_2",47.05],PARAMETER["latitude_of_origin",45],PARAMETER["central_meridian",-94.25],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MINNESOTA NORTH':
        'PROJCS["NAD27 / Minnesota North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",47.0333333333333],PARAMETER["standard_parallel_2",48.6333333333333],PARAMETER["latitude_of_origin",46.5],PARAMETER["central_meridian",-93.1],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MINNESOTA SOUTH':
        'PROJCS["NAD27 / Minnesota South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",43.7833333333333],PARAMETER["standard_parallel_2",45.2166666666667],PARAMETER["latitude_of_origin",43],PARAMETER["central_meridian",-94],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MISSISSIPPI EAST':
        'PROJCS["NAD27 / Mississippi East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",29.6666666666667],PARAMETER["central_meridian",-88.8333333333333],PARAMETER["scale_factor",0.99996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MISSISSIPPI WEST':
        'PROJCS["NAD27 / Mississippi West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",30.5],PARAMETER["central_meridian",-90.3333333333333],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MISSOURI CENTRAL':
        'PROJCS["NAD27 / Missouri Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",35.8333333333333],PARAMETER["central_meridian",-92.5],PARAMETER["scale_factor",0.999933333],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MISSOURI EAST':
        'PROJCS["NAD27 / Missouri East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",35.8333333333333],PARAMETER["central_meridian",-90.5],PARAMETER["scale_factor",0.999933333],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MISSOURI WEST':
        'PROJCS["NAD27 / Missouri West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",36.1666666666667],PARAMETER["central_meridian",-94.5],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MONTANA CENTRAL':
        'PROJCS["NAD27 / Montana Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",47.8833333333333],PARAMETER["standard_parallel_2",46.45],PARAMETER["latitude_of_origin",45.8333333333333],PARAMETER["central_meridian",-109.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MONTANA NORTH':
        'PROJCS["NAD27 / Montana North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",48.7166666666667],PARAMETER["standard_parallel_2",47.85],PARAMETER["latitude_of_origin",47],PARAMETER["central_meridian",-109.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MONTANA SOUTH':
        'PROJCS["NAD27 / Montana South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",46.4],PARAMETER["standard_parallel_2",44.8666666666667],PARAMETER["latitude_of_origin",44],PARAMETER["central_meridian",-109.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / MTM ZONE 1':
        'PROJCS["NAD27 / MTM zone 1",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-53],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 10':
        'PROJCS["NAD27 / MTM zone 10",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-79.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 11':
        'PROJCS["NAD27 / MTM zone 11",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-82.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 12':
        'PROJCS["NAD27 / MTM zone 12",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 13':
        'PROJCS["NAD27 / MTM zone 13",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-84],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 14':
        'PROJCS["NAD27 / MTM zone 14",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-87],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 15':
        'PROJCS["NAD27 / MTM zone 15",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-90],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 16':
        'PROJCS["NAD27 / MTM zone 16",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-93],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 17':
        'PROJCS["NAD27 / MTM zone 17",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-96],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 2':
        'PROJCS["NAD27 / MTM zone 2",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-56],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 3':
        'PROJCS["NAD27 / MTM zone 3",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-58.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 4':
        'PROJCS["NAD27 / MTM zone 4",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-61.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 5':
        'PROJCS["NAD27 / MTM zone 5",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-64.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 6':
        'PROJCS["NAD27 / MTM zone 6",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-67.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 7':
        'PROJCS["NAD27 / MTM zone 7",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-70.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 8':
        'PROJCS["NAD27 / MTM zone 8",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-73.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / MTM ZONE 9':
        'PROJCS["NAD27 / MTM zone 9",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-76.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / NEBRASKA NORTH':
        'PROJCS["NAD27 / Nebraska North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.85],PARAMETER["standard_parallel_2",42.8166666666667],PARAMETER["latitude_of_origin",41.3333333333333],PARAMETER["central_meridian",-100],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEBRASKA SOUTH':
        'PROJCS["NAD27 / Nebraska South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",40.2833333333333],PARAMETER["standard_parallel_2",41.7166666666667],PARAMETER["latitude_of_origin",39.6666666666667],PARAMETER["central_meridian",-99.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEVADA CENTRAL':
        'PROJCS["NAD27 / Nevada Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",34.75],PARAMETER["central_meridian",-116.666666666667],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEVADA EAST':
        'PROJCS["NAD27 / Nevada East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",34.75],PARAMETER["central_meridian",-115.583333333333],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEVADA WEST':
        'PROJCS["NAD27 / Nevada West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",34.75],PARAMETER["central_meridian",-118.583333333333],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEW HAMPSHIRE':
        'PROJCS["NAD27 / New Hampshire",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",42.5],PARAMETER["central_meridian",-71.6666666666667],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEW JERSEY':
        'PROJCS["NAD27 / New Jersey",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",38.8333333333333],PARAMETER["central_meridian",-74.6666666666667],PARAMETER["scale_factor",0.999975],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEW MEXICO CENTRAL':
        'PROJCS["NAD27 / New Mexico Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",31],PARAMETER["central_meridian",-106.25],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEW MEXICO EAST':
        'PROJCS["NAD27 / New Mexico East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",31],PARAMETER["central_meridian",-104.333333333333],PARAMETER["scale_factor",0.999909091],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEW MEXICO WEST':
        'PROJCS["NAD27 / New Mexico West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",31],PARAMETER["central_meridian",-107.833333333333],PARAMETER["scale_factor",0.999916667],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEW YORK CENTRAL':
        'PROJCS["NAD27 / New York Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40],PARAMETER["central_meridian",-76.5833333333333],PARAMETER["scale_factor",0.9999375],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEW YORK EAST':
        'PROJCS["NAD27 / New York East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40],PARAMETER["central_meridian",-74.3333333333333],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEW YORK LONG ISLAND':
        'PROJCS["NAD27 / New York Long Island",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.0333333333333],PARAMETER["standard_parallel_2",40.6666666666667],PARAMETER["latitude_of_origin",40.5],PARAMETER["central_meridian",-74],PARAMETER["false_easting",1000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NEW YORK WEST':
        'PROJCS["NAD27 / New York West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40],PARAMETER["central_meridian",-78.5833333333333],PARAMETER["scale_factor",0.9999375],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NORTH CAROLINA':
        'PROJCS["NAD27 / North Carolina",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",34.3333333333333],PARAMETER["standard_parallel_2",36.1666666666667],PARAMETER["latitude_of_origin",33.75],PARAMETER["central_meridian",-79],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NORTH DAKOTA NORTH':
        'PROJCS["NAD27 / North Dakota North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",47.4333333333333],PARAMETER["standard_parallel_2",48.7333333333333],PARAMETER["latitude_of_origin",47],PARAMETER["central_meridian",-100.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / NORTH DAKOTA SOUTH':
        'PROJCS["NAD27 / North Dakota South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",46.1833333333333],PARAMETER["standard_parallel_2",47.4833333333333],PARAMETER["latitude_of_origin",45.6666666666667],PARAMETER["central_meridian",-100.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / OHIO NORTH':
        'PROJCS["NAD27 / Ohio North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",40.4333333333333],PARAMETER["standard_parallel_2",41.7],PARAMETER["latitude_of_origin",39.6666666666667],PARAMETER["central_meridian",-82.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / OHIO SOUTH':
        'PROJCS["NAD27 / Ohio South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",38.7333333333333],PARAMETER["standard_parallel_2",40.0333333333333],PARAMETER["latitude_of_origin",38],PARAMETER["central_meridian",-82.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / OKLAHOMA NORTH':
        'PROJCS["NAD27 / Oklahoma North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",35.5666666666667],PARAMETER["standard_parallel_2",36.7666666666667],PARAMETER["latitude_of_origin",35],PARAMETER["central_meridian",-98],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / OKLAHOMA SOUTH':
        'PROJCS["NAD27 / Oklahoma South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",33.9333333333333],PARAMETER["standard_parallel_2",35.2333333333333],PARAMETER["latitude_of_origin",33.3333333333333],PARAMETER["central_meridian",-98],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / OREGON NORTH':
        'PROJCS["NAD27 / Oregon North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",44.3333333333333],PARAMETER["standard_parallel_2",46],PARAMETER["latitude_of_origin",43.6666666666667],PARAMETER["central_meridian",-120.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / OREGON SOUTH':
        'PROJCS["NAD27 / Oregon South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",42.3333333333333],PARAMETER["standard_parallel_2",44],PARAMETER["latitude_of_origin",41.6666666666667],PARAMETER["central_meridian",-120.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / PENNSYLVANIA NORTH':
        'PROJCS["NAD27 / Pennsylvania North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",40.8833333333333],PARAMETER["standard_parallel_2",41.95],PARAMETER["latitude_of_origin",40.1666666666667],PARAMETER["central_meridian",-77.75],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / PENNSYLVANIA SOUTH':
        'PROJCS["NAD27 / Pennsylvania South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",39.9333333333333],PARAMETER["standard_parallel_2",40.8],PARAMETER["latitude_of_origin",39.3333333333333],PARAMETER["central_meridian",-77.75],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / PUERTO RICO':
        'PROJCS["NAD27 / Puerto Rico",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",18.4333333333333],PARAMETER["standard_parallel_2",18.0333333333333],PARAMETER["latitude_of_origin",17.8333333333333],PARAMETER["central_meridian",-66.4333333333334],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / RHODE ISLAND':
        'PROJCS["NAD27 / Rhode Island",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",41.0833333333333],PARAMETER["central_meridian",-71.5],PARAMETER["scale_factor",0.9999938],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / SCOPQ ZONE 2':
        'PROJCS["NAD27 / SCoPQ zone 2",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-55.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / SOUTH CAROLINA NORTH':
        'PROJCS["NAD27 / South Carolina North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",33.7666666666667],PARAMETER["standard_parallel_2",34.9666666666667],PARAMETER["latitude_of_origin",33],PARAMETER["central_meridian",-81],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / SOUTH CAROLINA SOUTH':
        'PROJCS["NAD27 / South Carolina South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",32.3333333333333],PARAMETER["standard_parallel_2",33.6666666666667],PARAMETER["latitude_of_origin",31.8333333333333],PARAMETER["central_meridian",-81],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / SOUTH DAKOTA NORTH':
        'PROJCS["NAD27 / South Dakota North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",44.4166666666667],PARAMETER["standard_parallel_2",45.6833333333333],PARAMETER["latitude_of_origin",43.8333333333333],PARAMETER["central_meridian",-100],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / SOUTH DAKOTA SOUTH':
        'PROJCS["NAD27 / South Dakota South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",42.8333333333333],PARAMETER["standard_parallel_2",44.4],PARAMETER["latitude_of_origin",42.3333333333333],PARAMETER["central_meridian",-100.333333333333],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / ST. CROIX':
        'PROJCS["NAD27 / St. Croix",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",18.4333333333333],PARAMETER["standard_parallel_2",18.0333333333333],PARAMETER["latitude_of_origin",17.8333333333333],PARAMETER["central_meridian",-66.4333333333334],PARAMETER["false_easting",500000],PARAMETER["false_northing",100000],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / TENNESSEE':
        'PROJCS["NAD27 / Tennessee",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",35.25],PARAMETER["standard_parallel_2",36.4166666666667],PARAMETER["latitude_of_origin",34.6666666666667],PARAMETER["central_meridian",-86],PARAMETER["false_easting",100000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / TEXAS CENTRAL':
        'PROJCS["NAD27 / Texas Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",30.1166666666667],PARAMETER["standard_parallel_2",31.8833333333333],PARAMETER["latitude_of_origin",29.6666666666667],PARAMETER["central_meridian",-100.333333333333],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / TEXAS NORTH':
        'PROJCS["NAD27 / Texas North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",34.65],PARAMETER["standard_parallel_2",36.1833333333333],PARAMETER["latitude_of_origin",34],PARAMETER["central_meridian",-101.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / TEXAS NORTH CENTRAL':
        'PROJCS["NAD27 / Texas North Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",32.1333333333333],PARAMETER["standard_parallel_2",33.9666666666667],PARAMETER["latitude_of_origin",31.6666666666667],PARAMETER["central_meridian",-97.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / TEXAS SOUTH':
        'PROJCS["NAD27 / Texas South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",26.1666666666667],PARAMETER["standard_parallel_2",27.8333333333333],PARAMETER["latitude_of_origin",25.6666666666667],PARAMETER["central_meridian",-98.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / TEXAS SOUTH CENTRAL':
        'PROJCS["NAD27 / Texas South Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",28.3833333333333],PARAMETER["standard_parallel_2",30.2833333333333],PARAMETER["latitude_of_origin",27.8333333333333],PARAMETER["central_meridian",-99],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / UTAH CENTRAL':
        'PROJCS["NAD27 / Utah Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",39.0166666666667],PARAMETER["standard_parallel_2",40.65],PARAMETER["latitude_of_origin",38.3333333333333],PARAMETER["central_meridian",-111.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / UTAH NORTH':
        'PROJCS["NAD27 / Utah North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",40.7166666666667],PARAMETER["standard_parallel_2",41.7833333333333],PARAMETER["latitude_of_origin",40.3333333333333],PARAMETER["central_meridian",-111.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / UTAH SOUTH':
        'PROJCS["NAD27 / Utah South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",37.2166666666667],PARAMETER["standard_parallel_2",38.35],PARAMETER["latitude_of_origin",36.6666666666667],PARAMETER["central_meridian",-111.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / UTM ZONE 10N':
        'PROJCS["NAD27 / UTM zone 10N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 11N':
        'PROJCS["NAD27 / UTM zone 11N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 12N':
        'PROJCS["NAD27 / UTM zone 12N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 13N':
        'PROJCS["NAD27 / UTM zone 13N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 14N':
        'PROJCS["NAD27 / UTM zone 14N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 15N':
        'PROJCS["NAD27 / UTM zone 15N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 16N':
        'PROJCS["NAD27 / UTM zone 16N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 17N':
        'PROJCS["NAD27 / UTM zone 17N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 18N':
        'PROJCS["NAD27 / UTM zone 18N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 19N':
        'PROJCS["NAD27 / UTM zone 19N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 20N':
        'PROJCS["NAD27 / UTM zone 20N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 21N':
        'PROJCS["NAD27 / UTM zone 21N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 22N':
        'PROJCS["NAD27 / UTM zone 22N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 3N':
        'PROJCS["NAD27 / UTM zone 3N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 4N':
        'PROJCS["NAD27 / UTM zone 4N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 5N':
        'PROJCS["NAD27 / UTM zone 5N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 6N':
        'PROJCS["NAD27 / UTM zone 6N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 7N':
        'PROJCS["NAD27 / UTM zone 7N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 8N':
        'PROJCS["NAD27 / UTM zone 8N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / UTM ZONE 9N':
        'PROJCS["NAD27 / UTM zone 9N",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD27 / VERMONT':
        'PROJCS["NAD27 / Vermont",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",42.5],PARAMETER["central_meridian",-72.5],PARAMETER["scale_factor",0.999964286],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / VIRGINIA NORTH':
        'PROJCS["NAD27 / Virginia North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",38.0333333333333],PARAMETER["standard_parallel_2",39.2],PARAMETER["latitude_of_origin",37.6666666666667],PARAMETER["central_meridian",-78.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / VIRGINIA SOUTH':
        'PROJCS["NAD27 / Virginia South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",36.7666666666667],PARAMETER["standard_parallel_2",37.9666666666667],PARAMETER["latitude_of_origin",36.3333333333333],PARAMETER["central_meridian",-78.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / WASHINGTON NORTH':
        'PROJCS["NAD27 / Washington North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",47.5],PARAMETER["standard_parallel_2",48.7333333333333],PARAMETER["latitude_of_origin",47],PARAMETER["central_meridian",-120.833333333333],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / WASHINGTON SOUTH':
        'PROJCS["NAD27 / Washington South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",45.8333333333333],PARAMETER["standard_parallel_2",47.3333333333333],PARAMETER["latitude_of_origin",45.3333333333333],PARAMETER["central_meridian",-120.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / WEST VIRGINIA NORTH':
        'PROJCS["NAD27 / West Virginia North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",39],PARAMETER["standard_parallel_2",40.25],PARAMETER["latitude_of_origin",38.5],PARAMETER["central_meridian",-79.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / WEST VIRGINIA SOUTH':
        'PROJCS["NAD27 / West Virginia South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",37.4833333333333],PARAMETER["standard_parallel_2",38.8833333333333],PARAMETER["latitude_of_origin",37],PARAMETER["central_meridian",-81],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / WISCONSIN CENTRAL':
        'PROJCS["NAD27 / Wisconsin Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",44.25],PARAMETER["standard_parallel_2",45.5],PARAMETER["latitude_of_origin",43.8333333333333],PARAMETER["central_meridian",-90],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / WISCONSIN NORTH':
        'PROJCS["NAD27 / Wisconsin North",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",45.5666666666667],PARAMETER["standard_parallel_2",46.7666666666667],PARAMETER["latitude_of_origin",45.1666666666667],PARAMETER["central_meridian",-90],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / WISCONSIN SOUTH':
        'PROJCS["NAD27 / Wisconsin South",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",42.7333333333333],PARAMETER["standard_parallel_2",44.0666666666667],PARAMETER["latitude_of_origin",42],PARAMETER["central_meridian",-90],PARAMETER["false_easting",2000000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / WYOMING EAST':
        'PROJCS["NAD27 / Wyoming East",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40.6666666666667],PARAMETER["central_meridian",-105.166666666667],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / WYOMING EAST CENTRAL':
        'PROJCS["NAD27 / Wyoming East Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40.6666666666667],PARAMETER["central_meridian",-107.333333333333],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / WYOMING WEST':
        'PROJCS["NAD27 / Wyoming West",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40.6666666666667],PARAMETER["central_meridian",-110.083333333333],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 / WYOMING WEST CENTRAL':
        'PROJCS["NAD27 / Wyoming West Central",GEOGCS["NAD27",DATUM["North_American_Datum_1927",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40.6666666666667],PARAMETER["central_meridian",-108.75],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["US survey foot",0.304800609601219]]',
    'NAD27 MICHIGAN':
        'GEOGCS["NAD27 Michigan",DATUM["NAD_Michigan",SPHEROID["Clarke 1866 Michigan",6378450.0472441,294.978684676652]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'NAD83':
        'GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'NAD83 / ALABAMA EAST':
        'PROJCS["NAD83 / Alabama East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",30.5],PARAMETER["central_meridian",-85.8333333333333],PARAMETER["scale_factor",0.99996],PARAMETER["false_easting",200000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ALABAMA WEST':
        'PROJCS["NAD83 / Alabama West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",30],PARAMETER["central_meridian",-87.5],PARAMETER["scale_factor",0.999933333],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ALASKA ZONE 1':
        'PROJCS["NAD83 / Alaska zone 1",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Hotine_Oblique_Mercator"],PARAMETER["latitude_of_center",57],PARAMETER["longitude_of_center",-133.666666666667],PARAMETER["azimuth",323.130102361111],PARAMETER["rectified_grid_angle",323.130102361111],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",5000000],PARAMETER["false_northing",-5000000],UNIT["metre",1]]',
    'NAD83 / ALASKA ZONE 10':
        'PROJCS["NAD83 / Alaska zone 10",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",53.8333333333333],PARAMETER["standard_parallel_2",51.8333333333333],PARAMETER["latitude_of_origin",51],PARAMETER["central_meridian",-176],PARAMETER["false_easting",1000000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ALASKA ZONE 2':
        'PROJCS["NAD83 / Alaska zone 2",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-142],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ALASKA ZONE 3':
        'PROJCS["NAD83 / Alaska zone 3",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-146],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ALASKA ZONE 4':
        'PROJCS["NAD83 / Alaska zone 4",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-150],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ALASKA ZONE 5':
        'PROJCS["NAD83 / Alaska zone 5",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-154],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ALASKA ZONE 6':
        'PROJCS["NAD83 / Alaska zone 6",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-158],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ALASKA ZONE 7':
        'PROJCS["NAD83 / Alaska zone 7",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-162],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ALASKA ZONE 8':
        'PROJCS["NAD83 / Alaska zone 8",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-166],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ALASKA ZONE 9':
        'PROJCS["NAD83 / Alaska zone 9",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",54],PARAMETER["central_meridian",-170],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ARIZONA  EAST':
        'PROJCS["NAD83 / Arizona  East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",31],PARAMETER["central_meridian",-110.166666666667],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",213360],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ARIZONA CENTRAL':
        'PROJCS["NAD83 / Arizona Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",31],PARAMETER["central_meridian",-111.916666666667],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",213360],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ARIZONA WEST':
        'PROJCS["NAD83 / Arizona West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",31],PARAMETER["central_meridian",-113.75],PARAMETER["scale_factor",0.999933333],PARAMETER["false_easting",213360],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ARKANSAS NORTH':
        'PROJCS["NAD83 / Arkansas North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",36.2333333333333],PARAMETER["standard_parallel_2",34.9333333333333],PARAMETER["latitude_of_origin",34.3333333333333],PARAMETER["central_meridian",-92],PARAMETER["false_easting",400000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ARKANSAS SOUTH':
        'PROJCS["NAD83 / Arkansas South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",34.7666666666667],PARAMETER["standard_parallel_2",33.3],PARAMETER["latitude_of_origin",32.6666666666667],PARAMETER["central_meridian",-92],PARAMETER["false_easting",400000],PARAMETER["false_northing",400000],UNIT["metre",1]]',
    'NAD83 / CALIFORNIA ZONE 1':
        'PROJCS["NAD83 / California zone 1",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.6666666666667],PARAMETER["standard_parallel_2",40],PARAMETER["latitude_of_origin",39.3333333333333],PARAMETER["central_meridian",-122],PARAMETER["false_easting",2000000],PARAMETER["false_northing",500000],UNIT["metre",1]]',
    'NAD83 / CALIFORNIA ZONE 2':
        'PROJCS["NAD83 / California zone 2",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",39.8333333333333],PARAMETER["standard_parallel_2",38.3333333333333],PARAMETER["latitude_of_origin",37.6666666666667],PARAMETER["central_meridian",-122],PARAMETER["false_easting",2000000],PARAMETER["false_northing",500000],UNIT["metre",1]]',
    'NAD83 / CALIFORNIA ZONE 3':
        'PROJCS["NAD83 / California zone 3",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",38.4333333333333],PARAMETER["standard_parallel_2",37.0666666666667],PARAMETER["latitude_of_origin",36.5],PARAMETER["central_meridian",-120.5],PARAMETER["false_easting",2000000],PARAMETER["false_northing",500000],UNIT["metre",1]]',
    'NAD83 / CALIFORNIA ZONE 4':
        'PROJCS["NAD83 / California zone 4",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",37.25],PARAMETER["standard_parallel_2",36],PARAMETER["latitude_of_origin",35.3333333333333],PARAMETER["central_meridian",-119],PARAMETER["false_easting",2000000],PARAMETER["false_northing",500000],UNIT["metre",1]]',
    'NAD83 / CALIFORNIA ZONE 5':
        'PROJCS["NAD83 / California zone 5",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",35.4666666666667],PARAMETER["standard_parallel_2",34.0333333333333],PARAMETER["latitude_of_origin",33.5],PARAMETER["central_meridian",-118],PARAMETER["false_easting",2000000],PARAMETER["false_northing",500000],UNIT["metre",1]]',
    'NAD83 / CALIFORNIA ZONE 6':
        'PROJCS["NAD83 / California zone 6",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",33.8833333333333],PARAMETER["standard_parallel_2",32.7833333333333],PARAMETER["latitude_of_origin",32.1666666666667],PARAMETER["central_meridian",-116.25],PARAMETER["false_easting",2000000],PARAMETER["false_northing",500000],UNIT["metre",1]]',
    'NAD83 / COLORADO CENTRAL':
        'PROJCS["NAD83 / Colorado Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",39.75],PARAMETER["standard_parallel_2",38.45],PARAMETER["latitude_of_origin",37.8333333333333],PARAMETER["central_meridian",-105.5],PARAMETER["false_easting",914401.8289],PARAMETER["false_northing",304800.6096],UNIT["metre",1]]',
    'NAD83 / COLORADO NORTH':
        'PROJCS["NAD83 / Colorado North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",40.7833333333333],PARAMETER["standard_parallel_2",39.7166666666667],PARAMETER["latitude_of_origin",39.3333333333333],PARAMETER["central_meridian",-105.5],PARAMETER["false_easting",914401.8289],PARAMETER["false_northing",304800.6096],UNIT["metre",1]]',
    'NAD83 / COLORADO SOUTH':
        'PROJCS["NAD83 / Colorado South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",38.4333333333333],PARAMETER["standard_parallel_2",37.2333333333333],PARAMETER["latitude_of_origin",36.6666666666667],PARAMETER["central_meridian",-105.5],PARAMETER["false_easting",914401.8289],PARAMETER["false_northing",304800.6096],UNIT["metre",1]]',
    'NAD83 / CONNECTICUT':
        'PROJCS["NAD83 / Connecticut",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.8666666666667],PARAMETER["standard_parallel_2",41.2],PARAMETER["latitude_of_origin",40.8333333333333],PARAMETER["central_meridian",-72.75],PARAMETER["false_easting",304800.6096],PARAMETER["false_northing",152400.3048],UNIT["metre",1]]',
    'NAD83 / DELAWARE':
        'PROJCS["NAD83 / Delaware",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",38],PARAMETER["central_meridian",-75.4166666666667],PARAMETER["scale_factor",0.999995],PARAMETER["false_easting",200000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / FLORIDA  EAST':
        'PROJCS["NAD83 / Florida  East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",24.3333333333333],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",200000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / FLORIDA  WEST':
        'PROJCS["NAD83 / Florida  West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",24.3333333333333],PARAMETER["central_meridian",-82],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",200000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / FLORIDA NORTH':
        'PROJCS["NAD83 / Florida North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",30.75],PARAMETER["standard_parallel_2",29.5833333333333],PARAMETER["latitude_of_origin",29],PARAMETER["central_meridian",-84.5],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / GEORGIA  EAST':
        'PROJCS["NAD83 / Georgia  East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",30],PARAMETER["central_meridian",-82.1666666666667],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",200000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / GEORGIA WEST':
        'PROJCS["NAD83 / Georgia West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",30],PARAMETER["central_meridian",-84.1666666666667],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",700000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / HAWAII  ZONE 1':
        'PROJCS["NAD83 / Hawaii  zone 1",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",18.8333333333333],PARAMETER["central_meridian",-155.5],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / HAWAII  ZONE 2':
        'PROJCS["NAD83 / Hawaii  zone 2",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",20.3333333333333],PARAMETER["central_meridian",-156.666666666667],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / HAWAII  ZONE 3':
        'PROJCS["NAD83 / Hawaii  zone 3",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",21.1666666666667],PARAMETER["central_meridian",-158],PARAMETER["scale_factor",0.99999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / HAWAII  ZONE 4':
        'PROJCS["NAD83 / Hawaii  zone 4",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",21.8333333333333],PARAMETER["central_meridian",-159.5],PARAMETER["scale_factor",0.99999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / HAWAII  ZONE 5':
        'PROJCS["NAD83 / Hawaii  zone 5",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",21.6666666666667],PARAMETER["central_meridian",-160.166666666667],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / IDAHO  CENTRAL':
        'PROJCS["NAD83 / Idaho  Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",41.6666666666667],PARAMETER["central_meridian",-114],PARAMETER["scale_factor",0.999947368],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / IDAHO  EAST':
        'PROJCS["NAD83 / Idaho  East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",41.6666666666667],PARAMETER["central_meridian",-112.166666666667],PARAMETER["scale_factor",0.999947368],PARAMETER["false_easting",200000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / IDAHO  WEST':
        'PROJCS["NAD83 / Idaho  West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",41.6666666666667],PARAMETER["central_meridian",-115.75],PARAMETER["scale_factor",0.999933333],PARAMETER["false_easting",800000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ILLINOIS  EAST':
        'PROJCS["NAD83 / Illinois  East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",36.6666666666667],PARAMETER["central_meridian",-88.3333333333333],PARAMETER["scale_factor",0.999975],PARAMETER["false_easting",300000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / ILLINOIS  WEST':
        'PROJCS["NAD83 / Illinois  West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",36.6666666666667],PARAMETER["central_meridian",-90.1666666666667],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",700000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / INDIANA  EAST':
        'PROJCS["NAD83 / Indiana  East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",37.5],PARAMETER["central_meridian",-85.6666666666667],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",100000],PARAMETER["false_northing",250000],UNIT["metre",1]]',
    'NAD83 / INDIANA  WEST':
        'PROJCS["NAD83 / Indiana  West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",37.5],PARAMETER["central_meridian",-87.0833333333333],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",900000],PARAMETER["false_northing",250000],UNIT["metre",1]]',
    'NAD83 / IOWA NORTH':
        'PROJCS["NAD83 / Iowa North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",43.2666666666667],PARAMETER["standard_parallel_2",42.0666666666667],PARAMETER["latitude_of_origin",41.5],PARAMETER["central_meridian",-93.5],PARAMETER["false_easting",1500000],PARAMETER["false_northing",1000000],UNIT["metre",1]]',
    'NAD83 / IOWA SOUTH':
        'PROJCS["NAD83 / Iowa South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.7833333333333],PARAMETER["standard_parallel_2",40.6166666666667],PARAMETER["latitude_of_origin",40],PARAMETER["central_meridian",-93.5],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / KANSAS NORTH':
        'PROJCS["NAD83 / Kansas North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",39.7833333333333],PARAMETER["standard_parallel_2",38.7166666666667],PARAMETER["latitude_of_origin",38.3333333333333],PARAMETER["central_meridian",-98],PARAMETER["false_easting",400000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / KANSAS SOUTH':
        'PROJCS["NAD83 / Kansas South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",38.5666666666667],PARAMETER["standard_parallel_2",37.2666666666667],PARAMETER["latitude_of_origin",36.6666666666667],PARAMETER["central_meridian",-98.5],PARAMETER["false_easting",400000],PARAMETER["false_northing",400000],UNIT["metre",1]]',
    'NAD83 / KENTUCKY NORTH':
        'PROJCS["NAD83 / Kentucky North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",38.9666666666667],PARAMETER["standard_parallel_2",37.9666666666667],PARAMETER["latitude_of_origin",37.5],PARAMETER["central_meridian",-84.25],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / KENTUCKY SOUTH':
        'PROJCS["NAD83 / Kentucky South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",37.9333333333333],PARAMETER["standard_parallel_2",36.7333333333333],PARAMETER["latitude_of_origin",36.3333333333333],PARAMETER["central_meridian",-85.75],PARAMETER["false_easting",500000],PARAMETER["false_northing",500000],UNIT["metre",1]]',
    'NAD83 / LOUISIANA NORTH':
        'PROJCS["NAD83 / Louisiana North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",32.6666666666667],PARAMETER["standard_parallel_2",31.1666666666667],PARAMETER["latitude_of_origin",30.5],PARAMETER["central_meridian",-92.5],PARAMETER["false_easting",1000000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / LOUISIANA SOUTH':
        'PROJCS["NAD83 / Louisiana South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",30.7],PARAMETER["standard_parallel_2",29.3],PARAMETER["latitude_of_origin",28.5],PARAMETER["central_meridian",-91.3333333333333],PARAMETER["false_easting",1000000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MAINE  EAST':
        'PROJCS["NAD83 / Maine  East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",43.6666666666667],PARAMETER["central_meridian",-68.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",300000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MAINE  WEST':
        'PROJCS["NAD83 / Maine  West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",42.8333333333333],PARAMETER["central_meridian",-70.1666666666667],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",900000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MARYLAND':
        'PROJCS["NAD83 / Maryland",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",39.45],PARAMETER["standard_parallel_2",38.3],PARAMETER["latitude_of_origin",37.6666666666667],PARAMETER["central_meridian",-77],PARAMETER["false_easting",400000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MASSACHUSETTS ISLAND':
        'PROJCS["NAD83 / Massachusetts Island",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.4833333333333],PARAMETER["standard_parallel_2",41.2833333333333],PARAMETER["latitude_of_origin",41],PARAMETER["central_meridian",-70.5],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MASSACHUSETTS MAINLAND':
        'PROJCS["NAD83 / Massachusetts Mainland",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",42.6833333333333],PARAMETER["standard_parallel_2",41.7166666666667],PARAMETER["latitude_of_origin",41],PARAMETER["central_meridian",-71.5],PARAMETER["false_easting",200000],PARAMETER["false_northing",750000],UNIT["metre",1]]',
    'NAD83 / MICHIGAN CENTRAL':
        'PROJCS["NAD83 / Michigan Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",45.7],PARAMETER["standard_parallel_2",44.1833333333333],PARAMETER["latitude_of_origin",43.3166666666667],PARAMETER["central_meridian",-84.3666666666667],PARAMETER["false_easting",6000000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MICHIGAN NORTH':
        'PROJCS["NAD83 / Michigan North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",47.0833333333333],PARAMETER["standard_parallel_2",45.4833333333333],PARAMETER["latitude_of_origin",44.7833333333333],PARAMETER["central_meridian",-87],PARAMETER["false_easting",8000000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MICHIGAN SOUTH':
        'PROJCS["NAD83 / Michigan South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",43.6666666666667],PARAMETER["standard_parallel_2",42.1],PARAMETER["latitude_of_origin",41.5],PARAMETER["central_meridian",-84.3666666666667],PARAMETER["false_easting",4000000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MINNESOTA CENTRAL':
        'PROJCS["NAD83 / Minnesota Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",47.05],PARAMETER["standard_parallel_2",45.6166666666667],PARAMETER["latitude_of_origin",45],PARAMETER["central_meridian",-94.25],PARAMETER["false_easting",800000],PARAMETER["false_northing",100000],UNIT["metre",1]]',
    'NAD83 / MINNESOTA NORTH':
        'PROJCS["NAD83 / Minnesota North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",48.6333333333333],PARAMETER["standard_parallel_2",47.0333333333333],PARAMETER["latitude_of_origin",46.5],PARAMETER["central_meridian",-93.1],PARAMETER["false_easting",800000],PARAMETER["false_northing",100000],UNIT["metre",1]]',
    'NAD83 / MINNESOTA SOUTH':
        'PROJCS["NAD83 / Minnesota South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",45.2166666666667],PARAMETER["standard_parallel_2",43.7833333333333],PARAMETER["latitude_of_origin",43],PARAMETER["central_meridian",-94],PARAMETER["false_easting",800000],PARAMETER["false_northing",100000],UNIT["metre",1]]',
    'NAD83 / MISSISSIPPI EAST':
        'PROJCS["NAD83 / Mississippi East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",29.5],PARAMETER["central_meridian",-88.8333333333333],PARAMETER["scale_factor",0.99995],PARAMETER["false_easting",300000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MISSISSIPPI WEST':
        'PROJCS["NAD83 / Mississippi West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",29.5],PARAMETER["central_meridian",-90.3333333333333],PARAMETER["scale_factor",0.99995],PARAMETER["false_easting",700000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MISSOURI  EAST':
        'PROJCS["NAD83 / Missouri  East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",35.8333333333333],PARAMETER["central_meridian",-90.5],PARAMETER["scale_factor",0.999933333],PARAMETER["false_easting",250000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MISSOURI  WEST':
        'PROJCS["NAD83 / Missouri  West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",36.1666666666667],PARAMETER["central_meridian",-94.5],PARAMETER["scale_factor",0.999941177],PARAMETER["false_easting",850000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MISSOURI CENTRAL':
        'PROJCS["NAD83 / Missouri Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",35.8333333333333],PARAMETER["central_meridian",-92.5],PARAMETER["scale_factor",0.999933333],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MONTANA':
        'PROJCS["NAD83 / Montana",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",49],PARAMETER["standard_parallel_2",45],PARAMETER["latitude_of_origin",44.25],PARAMETER["central_meridian",-109.5],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 1':
        'PROJCS["NAD83 / MTM zone 1",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-53],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 10':
        'PROJCS["NAD83 / MTM zone 10",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-79.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 11':
        'PROJCS["NAD83 / MTM zone 11",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-82.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 12':
        'PROJCS["NAD83 / MTM zone 12",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 13':
        'PROJCS["NAD83 / MTM zone 13",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-84],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 14':
        'PROJCS["NAD83 / MTM zone 14",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-87],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 15':
        'PROJCS["NAD83 / MTM zone 15",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-90],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 16':
        'PROJCS["NAD83 / MTM zone 16",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-93],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 17':
        'PROJCS["NAD83 / MTM zone 17",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-96],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 2':
        'PROJCS["NAD83 / MTM zone 2",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-56],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 3':
        'PROJCS["NAD83 / MTM zone 3",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-58.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 4':
        'PROJCS["NAD83 / MTM zone 4",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-61.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 5':
        'PROJCS["NAD83 / MTM zone 5",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-64.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 6':
        'PROJCS["NAD83 / MTM zone 6",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-67.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 7':
        'PROJCS["NAD83 / MTM zone 7",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-70.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 8':
        'PROJCS["NAD83 / MTM zone 8",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-73.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / MTM ZONE 9':
        'PROJCS["NAD83 / MTM zone 9",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-76.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NEBRASKA':
        'PROJCS["NAD83 / Nebraska",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",43],PARAMETER["standard_parallel_2",40],PARAMETER["latitude_of_origin",39.8333333333333],PARAMETER["central_meridian",-100],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NEVADA  EAST':
        'PROJCS["NAD83 / Nevada  East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",34.75],PARAMETER["central_meridian",-115.583333333333],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",200000],PARAMETER["false_northing",8000000],UNIT["metre",1]]',
    'NAD83 / NEVADA CENTRAL':
        'PROJCS["NAD83 / Nevada Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",34.75],PARAMETER["central_meridian",-116.666666666667],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",6000000],UNIT["metre",1]]',
    'NAD83 / NEVADA WEST':
        'PROJCS["NAD83 / Nevada West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",34.75],PARAMETER["central_meridian",-118.583333333333],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",800000],PARAMETER["false_northing",4000000],UNIT["metre",1]]',
    'NAD83 / NEW HAMPSHIRE':
        'PROJCS["NAD83 / New Hampshire",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",42.5],PARAMETER["central_meridian",-71.6666666666667],PARAMETER["scale_factor",0.999966667],PARAMETER["false_easting",300000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NEW JERSEY':
        'PROJCS["NAD83 / New Jersey",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",38.8333333333333],PARAMETER["central_meridian",-74.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",150000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NEW MEXICO CENTRAL':
        'PROJCS["NAD83 / New Mexico Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",31],PARAMETER["central_meridian",-106.25],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NEW MEXICO EAST':
        'PROJCS["NAD83 / New Mexico East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",31],PARAMETER["central_meridian",-104.333333333333],PARAMETER["scale_factor",0.999909091],PARAMETER["false_easting",165000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NEW MEXICO WEST':
        'PROJCS["NAD83 / New Mexico West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",31],PARAMETER["central_meridian",-107.833333333333],PARAMETER["scale_factor",0.999916667],PARAMETER["false_easting",830000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NEW YORK  EAST':
        'PROJCS["NAD83 / New York  East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",38.8333333333333],PARAMETER["central_meridian",-74.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",150000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NEW YORK  WEST':
        'PROJCS["NAD83 / New York  West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40],PARAMETER["central_meridian",-78.5833333333333],PARAMETER["scale_factor",0.9999375],PARAMETER["false_easting",350000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NEW YORK CENTRAL':
        'PROJCS["NAD83 / New York Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40],PARAMETER["central_meridian",-76.5833333333333],PARAMETER["scale_factor",0.9999375],PARAMETER["false_easting",250000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NEW YORK LONG ISLAND':
        'PROJCS["NAD83 / New York Long Island",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.0333333333333],PARAMETER["standard_parallel_2",40.6666666666667],PARAMETER["latitude_of_origin",40.1666666666667],PARAMETER["central_meridian",-74],PARAMETER["false_easting",300000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NORTH CAROLINA':
        'PROJCS["NAD83 / North Carolina",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",36.1666666666667],PARAMETER["standard_parallel_2",34.3333333333333],PARAMETER["latitude_of_origin",33.75],PARAMETER["central_meridian",-79],PARAMETER["false_easting",609601.22],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NORTH DAKOTA NORTH':
        'PROJCS["NAD83 / North Dakota North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",48.7333333333333],PARAMETER["standard_parallel_2",47.4333333333333],PARAMETER["latitude_of_origin",47],PARAMETER["central_meridian",-100.5],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / NORTH DAKOTA SOUTH':
        'PROJCS["NAD83 / North Dakota South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",47.4833333333333],PARAMETER["standard_parallel_2",46.1833333333333],PARAMETER["latitude_of_origin",45.6666666666667],PARAMETER["central_meridian",-100.5],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / OHIO NORTH':
        'PROJCS["NAD83 / Ohio North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.7],PARAMETER["standard_parallel_2",40.4333333333333],PARAMETER["latitude_of_origin",39.6666666666667],PARAMETER["central_meridian",-82.5],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / OHIO SOUTH':
        'PROJCS["NAD83 / Ohio South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",40.0333333333333],PARAMETER["standard_parallel_2",38.7333333333333],PARAMETER["latitude_of_origin",38],PARAMETER["central_meridian",-82.5],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / OKLAHOMA NORTH':
        'PROJCS["NAD83 / Oklahoma North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",36.7666666666667],PARAMETER["standard_parallel_2",35.5666666666667],PARAMETER["latitude_of_origin",35],PARAMETER["central_meridian",-98],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / OKLAHOMA SOUTH':
        'PROJCS["NAD83 / Oklahoma South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",35.2333333333333],PARAMETER["standard_parallel_2",33.9333333333333],PARAMETER["latitude_of_origin",33.3333333333333],PARAMETER["central_meridian",-98],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / OREGON NORTH':
        'PROJCS["NAD83 / Oregon North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",46],PARAMETER["standard_parallel_2",44.3333333333333],PARAMETER["latitude_of_origin",43.6666666666667],PARAMETER["central_meridian",-120.5],PARAMETER["false_easting",2500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / OREGON SOUTH':
        'PROJCS["NAD83 / Oregon South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",44],PARAMETER["standard_parallel_2",42.3333333333333],PARAMETER["latitude_of_origin",41.6666666666667],PARAMETER["central_meridian",-120.5],PARAMETER["false_easting",1500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / PENNSYLVANIA NORTH':
        'PROJCS["NAD83 / Pennsylvania North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.95],PARAMETER["standard_parallel_2",40.8833333333333],PARAMETER["latitude_of_origin",40.1666666666667],PARAMETER["central_meridian",-77.75],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / PENNSYLVANIA SOUTH':
        'PROJCS["NAD83 / Pennsylvania South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",40.9666666666667],PARAMETER["standard_parallel_2",39.9333333333333],PARAMETER["latitude_of_origin",39.3333333333333],PARAMETER["central_meridian",-77.75],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / PUERTO RICO & VIRGIN IS.':
        'PROJCS["NAD83 / Puerto Rico & Virgin Is.",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",18.4333333333333],PARAMETER["standard_parallel_2",18.0333333333333],PARAMETER["latitude_of_origin",17.8333333333333],PARAMETER["central_meridian",-66.4333333333334],PARAMETER["false_easting",200000],PARAMETER["false_northing",200000],UNIT["metre",1]]',
    'NAD83 / RHODE ISLAND':
        'PROJCS["NAD83 / Rhode Island",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",41.0833333333333],PARAMETER["central_meridian",-71.5],PARAMETER["scale_factor",0.99999375],PARAMETER["false_easting",100000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / SCOPQ ZONE 2':
        'PROJCS["NAD83 / SCoPQ zone 2",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-55.5],PARAMETER["scale_factor",0.9999],PARAMETER["false_easting",304800],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / SOUTH CAROLINA':
        'PROJCS["NAD83 / South Carolina",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",34.8333333333333],PARAMETER["standard_parallel_2",32.5],PARAMETER["latitude_of_origin",31.8333333333333],PARAMETER["central_meridian",-81],PARAMETER["false_easting",609600],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / SOUTH DAKOTA NORTH':
        'PROJCS["NAD83 / South Dakota North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",45.6833333333333],PARAMETER["standard_parallel_2",44.4166666666667],PARAMETER["latitude_of_origin",43.8333333333333],PARAMETER["central_meridian",-100],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / SOUTH DAKOTA SOUTH':
        'PROJCS["NAD83 / South Dakota South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",44.4],PARAMETER["standard_parallel_2",42.8333333333333],PARAMETER["latitude_of_origin",42.3333333333333],PARAMETER["central_meridian",-100.333333333333],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / TENNESSEE':
        'PROJCS["NAD83 / Tennessee",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",36.4166666666667],PARAMETER["standard_parallel_2",35.25],PARAMETER["latitude_of_origin",34.3333333333333],PARAMETER["central_meridian",-86],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / TEXAS CENTRAL':
        'PROJCS["NAD83 / Texas Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",31.8833333333333],PARAMETER["standard_parallel_2",30.1166666666667],PARAMETER["latitude_of_origin",29.6666666666667],PARAMETER["central_meridian",-100.333333333333],PARAMETER["false_easting",700000],PARAMETER["false_northing",3000000],UNIT["metre",1]]',
    'NAD83 / TEXAS NORTH':
        'PROJCS["NAD83 / Texas North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",36.1833333333333],PARAMETER["standard_parallel_2",34.65],PARAMETER["latitude_of_origin",34],PARAMETER["central_meridian",-101.5],PARAMETER["false_easting",200000],PARAMETER["false_northing",1000000],UNIT["metre",1]]',
    'NAD83 / TEXAS NORTH CENTRAL':
        'PROJCS["NAD83 / Texas North Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",33.9666666666667],PARAMETER["standard_parallel_2",32.1333333333333],PARAMETER["latitude_of_origin",31.6666666666667],PARAMETER["central_meridian",-98.5],PARAMETER["false_easting",600000],PARAMETER["false_northing",2000000],UNIT["metre",1]]',
    'NAD83 / TEXAS SOUTH':
        'PROJCS["NAD83 / Texas South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",27.8333333333333],PARAMETER["standard_parallel_2",26.1666666666667],PARAMETER["latitude_of_origin",25.6666666666667],PARAMETER["central_meridian",-98.5],PARAMETER["false_easting",300000],PARAMETER["false_northing",5000000],UNIT["metre",1]]',
    'NAD83 / TEXAS SOUTH CENTRAL':
        'PROJCS["NAD83 / Texas South Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",30.2833333333333],PARAMETER["standard_parallel_2",28.3833333333333],PARAMETER["latitude_of_origin",27.8333333333333],PARAMETER["central_meridian",-99],PARAMETER["false_easting",600000],PARAMETER["false_northing",4000000],UNIT["metre",1]]',
    'NAD83 / UTAH CENTRAL':
        'PROJCS["NAD83 / Utah Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",40.65],PARAMETER["standard_parallel_2",39.0166666666667],PARAMETER["latitude_of_origin",38.3333333333333],PARAMETER["central_meridian",-111.5],PARAMETER["false_easting",500000],PARAMETER["false_northing",2000000],UNIT["metre",1]]',
    'NAD83 / UTAH NORTH':
        'PROJCS["NAD83 / Utah North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",41.7833333333333],PARAMETER["standard_parallel_2",40.7166666666667],PARAMETER["latitude_of_origin",40.3333333333333],PARAMETER["central_meridian",-111.5],PARAMETER["false_easting",500000],PARAMETER["false_northing",1000000],UNIT["metre",1]]',
    'NAD83 / UTAH SOUTH':
        'PROJCS["NAD83 / Utah South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",38.35],PARAMETER["standard_parallel_2",37.2166666666667],PARAMETER["latitude_of_origin",36.6666666666667],PARAMETER["central_meridian",-111.5],PARAMETER["false_easting",500000],PARAMETER["false_northing",3000000],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 10N':
        'PROJCS["NAD83 / UTM zone 10N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 11N':
        'PROJCS["NAD83 / UTM zone 11N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 12N':
        'PROJCS["NAD83 / UTM zone 12N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 13N':
        'PROJCS["NAD83 / UTM zone 13N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 14N':
        'PROJCS["NAD83 / UTM zone 14N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 15N':
        'PROJCS["NAD83 / UTM zone 15N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 16N':
        'PROJCS["NAD83 / UTM zone 16N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 17N':
        'PROJCS["NAD83 / UTM zone 17N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 18N':
        'PROJCS["NAD83 / UTM zone 18N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 19N':
        'PROJCS["NAD83 / UTM zone 19N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 20N':
        'PROJCS["NAD83 / UTM zone 20N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 21N':
        'PROJCS["NAD83 / UTM zone 21N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 22N':
        'PROJCS["NAD83 / UTM zone 22N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 23N':
        'PROJCS["NAD83 / UTM zone 23N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 3N':
        'PROJCS["NAD83 / UTM zone 3N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 4N':
        'PROJCS["NAD83 / UTM zone 4N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 5N':
        'PROJCS["NAD83 / UTM zone 5N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 6N':
        'PROJCS["NAD83 / UTM zone 6N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 7N':
        'PROJCS["NAD83 / UTM zone 7N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 8N':
        'PROJCS["NAD83 / UTM zone 8N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / UTM ZONE 9N':
        'PROJCS["NAD83 / UTM zone 9N",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / VERMONT':
        'PROJCS["NAD83 / Vermont",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",42.5],PARAMETER["central_meridian",-72.5],PARAMETER["scale_factor",0.999964286],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / VIRGINIA NORTH':
        'PROJCS["NAD83 / Virginia North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",39.2],PARAMETER["standard_parallel_2",38.0333333333333],PARAMETER["latitude_of_origin",37.6666666666667],PARAMETER["central_meridian",-78.5],PARAMETER["false_easting",3500000],PARAMETER["false_northing",2000000],UNIT["metre",1]]',
    'NAD83 / VIRGINIA SOUTH':
        'PROJCS["NAD83 / Virginia South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",37.9666666666667],PARAMETER["standard_parallel_2",36.7666666666667],PARAMETER["latitude_of_origin",36.3333333333333],PARAMETER["central_meridian",-78.5],PARAMETER["false_easting",3500000],PARAMETER["false_northing",1000000],UNIT["metre",1]]',
    'NAD83 / WASHINGTON NORTH':
        'PROJCS["NAD83 / Washington North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",48.7333333333333],PARAMETER["standard_parallel_2",47.5],PARAMETER["latitude_of_origin",47],PARAMETER["central_meridian",-120.833333333333],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / WASHINGTON SOUTH':
        'PROJCS["NAD83 / Washington South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",47.3333333333333],PARAMETER["standard_parallel_2",45.8333333333333],PARAMETER["latitude_of_origin",45.3333333333333],PARAMETER["central_meridian",-120.5],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / WEST VIRGINIA NORTH':
        'PROJCS["NAD83 / West Virginia North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",40.25],PARAMETER["standard_parallel_2",39],PARAMETER["latitude_of_origin",38.5],PARAMETER["central_meridian",-79.5],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / WEST VIRGINIA SOUTH':
        'PROJCS["NAD83 / West Virginia South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",38.8833333333333],PARAMETER["standard_parallel_2",37.4833333333333],PARAMETER["latitude_of_origin",37],PARAMETER["central_meridian",-81],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / WISCONSIN CENTRAL':
        'PROJCS["NAD83 / Wisconsin Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",45.5],PARAMETER["standard_parallel_2",44.25],PARAMETER["latitude_of_origin",43.8333333333333],PARAMETER["central_meridian",-90],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / WISCONSIN NORTH':
        'PROJCS["NAD83 / Wisconsin North",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",46.7666666666667],PARAMETER["standard_parallel_2",45.5666666666667],PARAMETER["latitude_of_origin",45.1666666666667],PARAMETER["central_meridian",-90],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / WISCONSIN SOUTH':
        'PROJCS["NAD83 / Wisconsin South",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_2SP"],PARAMETER["standard_parallel_1",44.0666666666667],PARAMETER["standard_parallel_2",42.7333333333333],PARAMETER["latitude_of_origin",42],PARAMETER["central_meridian",-90],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / WYOMING EAST':
        'PROJCS["NAD83 / Wyoming East",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40.5],PARAMETER["central_meridian",-105.166666666667],PARAMETER["scale_factor",0.9999375],PARAMETER["false_easting",200000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAD83 / WYOMING EAST CENTRAL':
        'PROJCS["NAD83 / Wyoming East Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40.5],PARAMETER["central_meridian",-107.333333333333],PARAMETER["scale_factor",0.9999375],PARAMETER["false_easting",400000],PARAMETER["false_northing",100000],UNIT["metre",1]]',
    'NAD83 / WYOMING WEST':
        'PROJCS["NAD83 / Wyoming West",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40.5],PARAMETER["central_meridian",-110.083333333333],PARAMETER["scale_factor",0.9999375],PARAMETER["false_easting",800000],PARAMETER["false_northing",100000],UNIT["metre",1]]',
    'NAD83 / WYOMING WEST CENTRAL':
        'PROJCS["NAD83 / Wyoming West Central",GEOGCS["NAD83",DATUM["North_American_Datum_1983",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",40.5],PARAMETER["central_meridian",-108.75],PARAMETER["scale_factor",0.9999375],PARAMETER["false_easting",600000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAHRWAN 1967':
        'GEOGCS["Nahrwan 1967",DATUM["Nahrwan_1967",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'NAHRWAN 1967 / UTM ZONE 38N':
        'PROJCS["Nahrwan 1967 / UTM zone 38N",GEOGCS["Nahrwan 1967",DATUM["Nahrwan_1967",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAHRWAN 1967 / UTM ZONE 39N':
        'PROJCS["Nahrwan 1967 / UTM zone 39N",GEOGCS["Nahrwan 1967",DATUM["Nahrwan_1967",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAHRWAN 1967 / UTM ZONE 40N':
        'PROJCS["Nahrwan 1967 / UTM zone 40N",GEOGCS["Nahrwan 1967",DATUM["Nahrwan_1967",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NAPARIMA 1972':
        'GEOGCS["Naparima 1972",DATUM["Naparima_1972",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'NAPARIMA 1972 / UTM ZONE 20N':
        'PROJCS["Naparima 1972 / UTM zone 20N",GEOGCS["Naparima 1972",DATUM["Naparima_1972",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NDG (PARIS)':
        'GEOGCS["NDG (Paris)",DATUM["Nord_de_Guerre",SPHEROID["Plessis 1817",6376523,308.64]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]]',
    'NGN':
        'GEOGCS["NGN",DATUM["National_Geodetic_Network",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'NGN / UTM ZONE 38N':
        'PROJCS["NGN / UTM zone 38N",GEOGCS["NGN",DATUM["National_Geodetic_Network",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NGN / UTM ZONE 39N':
        'PROJCS["NGN / UTM zone 39N",GEOGCS["NGN",DATUM["National_Geodetic_Network",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NGO 1948':
        'GEOGCS["NGO 1948",DATUM["NGO_1948",SPHEROID["Bessel Modified",6377492.018,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'NORD SAHARA 1959':
        'GEOGCS["Nord Sahara 1959",DATUM["Nord_Sahara_1959",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'NORD SAHARA 1959 / UTM ZONE 29N':
        'PROJCS["Nord Sahara 1959 / UTM zone 29N",GEOGCS["Nord Sahara 1959",DATUM["Nord_Sahara_1959",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NORD SAHARA 1959 / UTM ZONE 30N':
        'PROJCS["Nord Sahara 1959 / UTM zone 30N",GEOGCS["Nord Sahara 1959",DATUM["Nord_Sahara_1959",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NORD SAHARA 1959 / UTM ZONE 31N':
        'PROJCS["Nord Sahara 1959 / UTM zone 31N",GEOGCS["Nord Sahara 1959",DATUM["Nord_Sahara_1959",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NORD SAHARA 1959 / UTM ZONE 32N':
        'PROJCS["Nord Sahara 1959 / UTM zone 32N",GEOGCS["Nord Sahara 1959",DATUM["Nord_Sahara_1959",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'NSWC 9Z-2':
        'GEOGCS["NSWC 9Z-2",DATUM["NSWC_9Z_2",SPHEROID["NWL 9D",6378145,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'NTF':
        'GEOGCS["NTF",DATUM["Nouvelle_Triangulation_Francaise",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'NTF (PARIS)':
        'GEOGCS["NTF (Paris)",DATUM["Nouvelle_Triangulation_Francaise",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]]',
    'NTF (PARIS) / CENTRE FRANCE':
        'PROJCS["NTF (Paris) / Centre France",GEOGCS["NTF (Paris)",DATUM["Nouvelle_Triangulation_Francaise",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",52],PARAMETER["central_meridian",0],PARAMETER["scale_factor",0.99987742],PARAMETER["false_easting",600000],PARAMETER["false_northing",200000],UNIT["metre",1]]',
    'NTF (PARIS) / CORSE':
        'PROJCS["NTF (Paris) / Corse",GEOGCS["NTF (Paris)",DATUM["Nouvelle_Triangulation_Francaise",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",46.85],PARAMETER["central_meridian",0],PARAMETER["scale_factor",0.99994471],PARAMETER["false_easting",234.358],PARAMETER["false_northing",185861.369],UNIT["metre",1]]',
    'NTF (PARIS) / FRANCE I':
        'PROJCS["NTF (Paris) / France I",GEOGCS["NTF (Paris)",DATUM["Nouvelle_Triangulation_Francaise",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",55],PARAMETER["central_meridian",0],PARAMETER["scale_factor",0.999877341],PARAMETER["false_easting",600000],PARAMETER["false_northing",1200000],UNIT["metre",1]]',
    'NTF (PARIS) / FRANCE II':
        'PROJCS["NTF (Paris) / France II",GEOGCS["NTF (Paris)",DATUM["Nouvelle_Triangulation_Francaise",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",52],PARAMETER["central_meridian",0],PARAMETER["scale_factor",0.99987742],PARAMETER["false_easting",600000],PARAMETER["false_northing",2200000],UNIT["metre",1]]',
    'NTF (PARIS) / FRANCE III':
        'PROJCS["NTF (Paris) / France III",GEOGCS["NTF (Paris)",DATUM["Nouvelle_Triangulation_Francaise",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",49],PARAMETER["central_meridian",0],PARAMETER["scale_factor",0.999877499],PARAMETER["false_easting",600000],PARAMETER["false_northing",3200000],UNIT["metre",1]]',
    'NTF (PARIS) / FRANCE IV':
        'PROJCS["NTF (Paris) / France IV",GEOGCS["NTF (Paris)",DATUM["Nouvelle_Triangulation_Francaise",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",46.85],PARAMETER["central_meridian",0],PARAMETER["scale_factor",0.99994471],PARAMETER["false_easting",234.358],PARAMETER["false_northing",4185861.369],UNIT["metre",1]]',
    'NTF (PARIS) / NORD FRANCE':
        'PROJCS["NTF (Paris) / Nord France",GEOGCS["NTF (Paris)",DATUM["Nouvelle_Triangulation_Francaise",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",55],PARAMETER["central_meridian",0],PARAMETER["scale_factor",0.999877341],PARAMETER["false_easting",600000],PARAMETER["false_northing",200000],UNIT["metre",1]]',
    'NTF (PARIS) / SUD FRANCE':
        'PROJCS["NTF (Paris) / Sud France",GEOGCS["NTF (Paris)",DATUM["Nouvelle_Triangulation_Francaise",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",49],PARAMETER["central_meridian",0],PARAMETER["scale_factor",0.999877499],PARAMETER["false_easting",600000],PARAMETER["false_northing",200000],UNIT["metre",1]]',
    'OBSERVATARIO':
        'GEOGCS["Observatario",DATUM["Observatario",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'OS(SN)80':
        'GEOGCS["OS(SN)80",DATUM["OS_SN_1980",SPHEROID["Airy 1830",6377563.396,299.32496]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'OSGB 1936':
        'GEOGCS["OSGB 1936",DATUM["OSGB_1936",SPHEROID["Airy 1830",6377563.396,299.32496]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'OSGB 1936 / BRITISH NATIONAL GRID':
        'PROJCS["OSGB 1936 / British National Grid",GEOGCS["OSGB 1936",DATUM["OSGB_1936",SPHEROID["Airy 1830",6377563.396,299.32496]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",49],PARAMETER["central_meridian",-2],PARAMETER["scale_factor",0.999601272],PARAMETER["false_easting",400000],PARAMETER["false_northing",-100000],UNIT["metre",1]]',
    'OSGB70':
        'GEOGCS["OSGB70",DATUM["OSGB_1970_SN",SPHEROID["Airy 1830",6377563.396,299.32496]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'PADANG':
        'GEOGCS["Padang",DATUM["Padang_1884",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'PADANG (JAKARTA)':
        'GEOGCS["Padang (Jakarta)",DATUM["Padang_1884",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Jakarta",106.807719444444],UNIT["degree",0.0174532925199433]]',
    'PALESTINE 1923':
        'GEOGCS["Palestine 1923",DATUM["Palestine_1923",SPHEROID["Clarke 1880 (Benoit)",6378300.79,293.466234570509]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'POINTE NOIRE':
        'GEOGCS["Pointe Noire",DATUM["Pointe_Noire",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'POINTE NOIRE / UTM ZONE 32S':
        'PROJCS["Pointe Noire / UTM zone 32S",GEOGCS["Pointe Noire",DATUM["Pointe_Noire",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'PSAD56':
        'GEOGCS["PSAD56",DATUM["Provisional_South_American_Datum_1956",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'PSAD56 / PERU CENTRAL ZONE':
        'PROJCS["PSAD56 / Peru central zone",GEOGCS["PSAD56",DATUM["Provisional_South_American_Datum_1956",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",-9.5],PARAMETER["central_meridian",-76],PARAMETER["scale_factor",0.99932994],PARAMETER["false_easting",720000],PARAMETER["false_northing",1039979.159],UNIT["metre",1]]',
    'PSAD56 / PERU EAST ZONE':
        'PROJCS["PSAD56 / Peru east zone",GEOGCS["PSAD56",DATUM["Provisional_South_American_Datum_1956",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",-9.5],PARAMETER["central_meridian",-70.5],PARAMETER["scale_factor",0.99952992],PARAMETER["false_easting",1324000],PARAMETER["false_northing",1040084.558],UNIT["metre",1]]',
    'PSAD56 / PERU WEST ZONE':
        'PROJCS["PSAD56 / Peru west zone",GEOGCS["PSAD56",DATUM["Provisional_South_American_Datum_1956",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",-6],PARAMETER["central_meridian",-80.5],PARAMETER["scale_factor",0.99983008],PARAMETER["false_easting",222000],PARAMETER["false_northing",1426834.743],UNIT["metre",1]]',
    'PSAD56 / UTM ZONE 17S':
        'PROJCS["PSAD56 / UTM zone 17S",GEOGCS["PSAD56",DATUM["Provisional_South_American_Datum_1956",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'PSAD56 / UTM ZONE 18N':
        'PROJCS["PSAD56 / UTM zone 18N",GEOGCS["PSAD56",DATUM["Provisional_South_American_Datum_1956",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PSAD56 / UTM ZONE 18S':
        'PROJCS["PSAD56 / UTM zone 18S",GEOGCS["PSAD56",DATUM["Provisional_South_American_Datum_1956",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'PSAD56 / UTM ZONE 19N':
        'PROJCS["PSAD56 / UTM zone 19N",GEOGCS["PSAD56",DATUM["Provisional_South_American_Datum_1956",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PSAD56 / UTM ZONE 19S':
        'PROJCS["PSAD56 / UTM zone 19S",GEOGCS["PSAD56",DATUM["Provisional_South_American_Datum_1956",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'PSAD56 / UTM ZONE 20N':
        'PROJCS["PSAD56 / UTM zone 20N",GEOGCS["PSAD56",DATUM["Provisional_South_American_Datum_1956",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PSAD56 / UTM ZONE 20S':
        'PROJCS["PSAD56 / UTM zone 20S",GEOGCS["PSAD56",DATUM["Provisional_South_American_Datum_1956",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'PSAD56 / UTM ZONE 21N':
        'PROJCS["PSAD56 / UTM zone 21N",GEOGCS["PSAD56",DATUM["Provisional_South_American_Datum_1956",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942':
        'GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'PULKOVO 1942 / GAUSS-KRUGER 10N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 10N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 11N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 11N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",63],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 12N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 12N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",69],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 13N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 13N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",75],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 14N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 14N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",81],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 15N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 15N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",87],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 16N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 16N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 17N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 17N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 18N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 18N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 19N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 19N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 20N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 20N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 21N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 21N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 22N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 22N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 23N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 23N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 24N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 24N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 25N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 25N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 26N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 26N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 27N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 27N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 28N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 28N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 29N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 29N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",171],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 2N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 2N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 30N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 30N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",177],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 31N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 31N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-177],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 32N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 32N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-171],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 3N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 3N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 4N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 4N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 5N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 5N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 6N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 6N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 7N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 7N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 8N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 8N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER 9N':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger 9N",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 10':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 10",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",1],PARAMETER["false_easting",10500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 11':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 11",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",63],PARAMETER["scale_factor",1],PARAMETER["false_easting",11500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 12':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 12",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",69],PARAMETER["scale_factor",1],PARAMETER["false_easting",12500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 13':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 13",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",75],PARAMETER["scale_factor",1],PARAMETER["false_easting",13500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 14':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 14",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",81],PARAMETER["scale_factor",1],PARAMETER["false_easting",14500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 15':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 15",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",87],PARAMETER["scale_factor",1],PARAMETER["false_easting",15500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 16':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 16",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",1],PARAMETER["false_easting",16500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 17':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 17",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",1],PARAMETER["false_easting",17500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 18':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 18",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",1],PARAMETER["false_easting",18500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 19':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 19",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",1],PARAMETER["false_easting",19500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 2':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 2",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",1],PARAMETER["false_easting",2500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 20':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 20",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",1],PARAMETER["false_easting",20500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 21':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 21",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",1],PARAMETER["false_easting",21500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 22':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 22",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",1],PARAMETER["false_easting",22500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 23':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 23",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",1],PARAMETER["false_easting",23500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 24':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 24",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",1],PARAMETER["false_easting",24500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 25':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 25",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",1],PARAMETER["false_easting",25500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 26':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 26",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",1],PARAMETER["false_easting",26500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 27':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 27",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",1],PARAMETER["false_easting",27500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 28':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 28",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",1],PARAMETER["false_easting",28500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 29':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 29",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",171],PARAMETER["scale_factor",1],PARAMETER["false_easting",29500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 3':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 3",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",1],PARAMETER["false_easting",3500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 30':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 30",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",177],PARAMETER["scale_factor",1],PARAMETER["false_easting",30500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 31':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 31",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-177],PARAMETER["scale_factor",1],PARAMETER["false_easting",31500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 32':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 32",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-171],PARAMETER["scale_factor",1],PARAMETER["false_easting",32500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 4':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 4",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",1],PARAMETER["false_easting",4500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 5':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 5",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",1],PARAMETER["false_easting",5500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 6':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 6",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",1],PARAMETER["false_easting",6500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 7':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 7",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",1],PARAMETER["false_easting",7500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 8':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 8",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",1],PARAMETER["false_easting",8500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1942 / GAUSS-KRUGER ZONE 9':
        'PROJCS["Pulkovo 1942 / Gauss-Kruger zone 9",GEOGCS["Pulkovo 1942",DATUM["Pulkovo_1942",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",1],PARAMETER["false_easting",9500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995':
        'GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'PULKOVO 1995 / GAUSS-KRUGER 10N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 10N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 11N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 11N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",63],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 12N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 12N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",69],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 13N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 13N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",75],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 14N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 14N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",81],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 15N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 15N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",87],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 16N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 16N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 17N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 17N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 18N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 18N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 19N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 19N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 20N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 20N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 21N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 21N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 22N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 22N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 23N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 23N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 24N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 24N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 25N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 25N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 26N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 26N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 27N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 27N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 28N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 28N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 29N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 29N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",171],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 30N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 30N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",177],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 31N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 31N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-177],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 32N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 32N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-171],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 4N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 4N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 5N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 5N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 6N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 6N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 7N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 7N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 8N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 8N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER 9N':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger 9N",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",1],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 10':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 10",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",1],PARAMETER["false_easting",10500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 11':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 11",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",63],PARAMETER["scale_factor",1],PARAMETER["false_easting",11500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 12':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 12",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",69],PARAMETER["scale_factor",1],PARAMETER["false_easting",12500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 13':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 13",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",75],PARAMETER["scale_factor",1],PARAMETER["false_easting",13500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 14':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 14",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",81],PARAMETER["scale_factor",1],PARAMETER["false_easting",14500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 15':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 15",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",87],PARAMETER["scale_factor",1],PARAMETER["false_easting",15500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 16':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 16",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",1],PARAMETER["false_easting",16500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 17':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 17",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",1],PARAMETER["false_easting",17500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 18':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 18",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",1],PARAMETER["false_easting",18500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 19':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 19",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",1],PARAMETER["false_easting",19500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 20':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 20",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",1],PARAMETER["false_easting",20500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 21':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 21",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",1],PARAMETER["false_easting",21500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 22':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 22",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",1],PARAMETER["false_easting",22500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 23':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 23",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",1],PARAMETER["false_easting",23500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 24':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 24",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",1],PARAMETER["false_easting",24500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 25':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 25",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",1],PARAMETER["false_easting",25500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 26':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 26",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",1],PARAMETER["false_easting",26500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 27':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 27",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",1],PARAMETER["false_easting",27500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 28':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 28",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",1],PARAMETER["false_easting",28500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 29':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 29",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",171],PARAMETER["scale_factor",1],PARAMETER["false_easting",29500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 30':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 30",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",177],PARAMETER["scale_factor",1],PARAMETER["false_easting",30500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 31':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 31",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-177],PARAMETER["scale_factor",1],PARAMETER["false_easting",31500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 32':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 32",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-171],PARAMETER["scale_factor",1],PARAMETER["false_easting",32500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 4':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 4",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",1],PARAMETER["false_easting",4500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 5':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 5",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",1],PARAMETER["false_easting",5500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 6':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 6",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",1],PARAMETER["false_easting",6500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 7':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 7",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",1],PARAMETER["false_easting",7500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 8':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 8",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",1],PARAMETER["false_easting",8500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'PULKOVO 1995 / GAUSS-KRUGER ZONE 9':
        'PROJCS["Pulkovo 1995 / Gauss-Kruger zone 9",GEOGCS["Pulkovo 1995",DATUM["Pulkovo_1995",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",1],PARAMETER["false_easting",9500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'QATAR':
        'GEOGCS["Qatar",DATUM["Qatar",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'QATAR / QATAR NATIONAL GRID':
        'PROJCS["Qatar / Qatar National Grid",GEOGCS["Qatar 1948",DATUM["Qatar_1948",SPHEROID["Helmert 1906",6378200,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",24.45],PARAMETER["central_meridian",51.2166666666667],PARAMETER["scale_factor",0.99999],PARAMETER["false_easting",200000],PARAMETER["false_northing",300000],UNIT["metre",1]]',
    'QATAR 1948':
        'GEOGCS["Qatar 1948",DATUM["Qatar_1948",SPHEROID["Helmert 1906",6378200,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'QORNOQ':
        'GEOGCS["Qornoq",DATUM["Qornoq",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'RD / NETHERLANDS NEW':
        'PROJCS["RD / Netherlands New",GEOGCS["Amersfoort",DATUM["Amersfoort",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Oblique_Stereographic"],PARAMETER["latitude_of_origin",52.1561605555556],PARAMETER["central_meridian",5.38763888888889],PARAMETER["scale_factor",0.9999079],PARAMETER["false_easting",155000],PARAMETER["false_northing",463000],UNIT["metre",1]]',
    'RD / NETHERLANDS OLD':
        'PROJCS["RD / Netherlands Old",GEOGCS["Amersfoort",DATUM["Amersfoort",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Oblique_Stereographic"],PARAMETER["latitude_of_origin",52.1561605555556],PARAMETER["central_meridian",5.38763888888889],PARAMETER["scale_factor",0.9999079],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'RT38':
        'GEOGCS["RT38",DATUM["Stockholm_1938",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'RT38 (STOCKHOLM)':
        'GEOGCS["RT38 (Stockholm)",DATUM["Stockholm_1938",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Stockholm",18.0582777777778],UNIT["degree",0.0174532925199433]]',
    'RT38 2.5 GON W':
        'PROJCS["RT38 2.5 gon W",GEOGCS["RT38",DATUM["Stockholm_1938",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15.8082777777778],PARAMETER["scale_factor",1],PARAMETER["false_easting",1500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'RT90':
        'GEOGCS["RT90",DATUM["Rikets_koordinatsystem_1990",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'RT90 2.5 GON W':
        'PROJCS["RT90 2.5 gon W",GEOGCS["RT90",DATUM["Rikets_koordinatsystem_1990",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15.8082777777778],PARAMETER["scale_factor",1],PARAMETER["false_easting",1500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SAD69':
        'GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'SAD69 / UTM ZONE 17S':
        'PROJCS["SAD69 / UTM zone 17S",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 18N':
        'PROJCS["SAD69 / UTM zone 18N",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 18S':
        'PROJCS["SAD69 / UTM zone 18S",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 19N':
        'PROJCS["SAD69 / UTM zone 19N",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 19S':
        'PROJCS["SAD69 / UTM zone 19S",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 20N':
        'PROJCS["SAD69 / UTM zone 20N",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 20S':
        'PROJCS["SAD69 / UTM zone 20S",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 21N':
        'PROJCS["SAD69 / UTM zone 21N",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 21S':
        'PROJCS["SAD69 / UTM zone 21S",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 22N':
        'PROJCS["SAD69 / UTM zone 22N",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 22S':
        'PROJCS["SAD69 / UTM zone 22S",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 23S':
        'PROJCS["SAD69 / UTM zone 23S",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 24S':
        'PROJCS["SAD69 / UTM zone 24S",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SAD69 / UTM ZONE 25S':
        'PROJCS["SAD69 / UTM zone 25S",GEOGCS["SAD69",DATUM["South_American_Datum_1969",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SAMBOJA':
        'GEOGCS["Samboja",DATUM["Samboja",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'SAMBOJA / UTM ZONE 50S':
        'PROJCS["Samboja / UTM zone 50S",GEOGCS["Samboja",DATUM["Samboja",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SAPPER HILL 1943':
        'GEOGCS["Sapper Hill 1943",DATUM["Sapper_Hill_1943",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'SAPPER HILL 1943 / UTM ZONE 20S':
        'PROJCS["Sapper Hill 1943 / UTM zone 20S",GEOGCS["Sapper Hill 1943",DATUM["Sapper_Hill_1943",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SAPPER HILL 1943 / UTM ZONE 21S':
        'PROJCS["Sapper Hill 1943 / UTM zone 21S",GEOGCS["Sapper Hill 1943",DATUM["Sapper_Hill_1943",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SCHWARZECK':
        'GEOGCS["Schwarzeck",DATUM["Schwarzeck",SPHEROID["Bessel Namibia",6377483.865,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'SCHWARZECK / UTM ZONE 33S':
        'PROJCS["Schwarzeck / UTM zone 33S",GEOGCS["Schwarzeck",DATUM["Schwarzeck",SPHEROID["Bessel Namibia",6377483.865,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'SEGORA':
        'GEOGCS["Segora",DATUM["Segora",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'SERINDUNG':
        'GEOGCS["Serindung",DATUM["Serindung",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'SOUTH AFRICAN COORDINATE SYSTEM ZONE 15':
        'PROJCS["South African Coordinate System zone 15",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SOUTH AFRICAN COORDINATE SYSTEM ZONE 17':
        'PROJCS["South African Coordinate System zone 17",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",17],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SOUTH AFRICAN COORDINATE SYSTEM ZONE 19':
        'PROJCS["South African Coordinate System zone 19",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",19],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SOUTH AFRICAN COORDINATE SYSTEM ZONE 21':
        'PROJCS["South African Coordinate System zone 21",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SOUTH AFRICAN COORDINATE SYSTEM ZONE 23':
        'PROJCS["South African Coordinate System zone 23",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",23],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SOUTH AFRICAN COORDINATE SYSTEM ZONE 25':
        'PROJCS["South African Coordinate System zone 25",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",25],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SOUTH AFRICAN COORDINATE SYSTEM ZONE 27':
        'PROJCS["South African Coordinate System zone 27",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SOUTH AFRICAN COORDINATE SYSTEM ZONE 29':
        'PROJCS["South African Coordinate System zone 29",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",29],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SOUTH AFRICAN COORDINATE SYSTEM ZONE 31':
        'PROJCS["South African Coordinate System zone 31",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",31],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SOUTH AFRICAN COORDINATE SYSTEM ZONE 33':
        'PROJCS["South African Coordinate System zone 33",GEOGCS["Cape",DATUM["Cape",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SOUTH WEST AFRICAN COORD. SYSTEM ZONE 11':
        'PROJCS["South West African Coord. System zone 11",GEOGCS["Schwarzeck",DATUM["Schwarzeck",SPHEROID["Bessel Namibia",6377483.865,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",-22],PARAMETER["central_meridian",11],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["German legal metre",1.0000135965]]',
    'SOUTH WEST AFRICAN COORD. SYSTEM ZONE 13':
        'PROJCS["South West African Coord. System zone 13",GEOGCS["Schwarzeck",DATUM["Schwarzeck",SPHEROID["Bessel Namibia",6377483.865,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",-22],PARAMETER["central_meridian",13],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["German legal metre",1.0000135965]]',
    'SOUTH WEST AFRICAN COORD. SYSTEM ZONE 15':
        'PROJCS["South West African Coord. System zone 15",GEOGCS["Schwarzeck",DATUM["Schwarzeck",SPHEROID["Bessel Namibia",6377483.865,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",-22],PARAMETER["central_meridian",15],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["German legal metre",1.0000135965]]',
    'SOUTH WEST AFRICAN COORD. SYSTEM ZONE 17':
        'PROJCS["South West African Coord. System zone 17",GEOGCS["Schwarzeck",DATUM["Schwarzeck",SPHEROID["Bessel Namibia",6377483.865,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",-22],PARAMETER["central_meridian",17],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["German legal metre",1.0000135965]]',
    'SOUTH WEST AFRICAN COORD. SYSTEM ZONE 19':
        'PROJCS["South West African Coord. System zone 19",GEOGCS["Schwarzeck",DATUM["Schwarzeck",SPHEROID["Bessel Namibia",6377483.865,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",-22],PARAMETER["central_meridian",19],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["German legal metre",1.0000135965]]',
    'SOUTH WEST AFRICAN COORD. SYSTEM ZONE 21':
        'PROJCS["South West African Coord. System zone 21",GEOGCS["Schwarzeck",DATUM["Schwarzeck",SPHEROID["Bessel Namibia",6377483.865,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",-22],PARAMETER["central_meridian",21],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["German legal metre",1.0000135965]]',
    'SOUTH WEST AFRICAN COORD. SYSTEM ZONE 23':
        'PROJCS["South West African Coord. System zone 23",GEOGCS["Schwarzeck",DATUM["Schwarzeck",SPHEROID["Bessel Namibia",6377483.865,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",-22],PARAMETER["central_meridian",23],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["German legal metre",1.0000135965]]',
    'SOUTH WEST AFRICAN COORD. SYSTEM ZONE 25':
        'PROJCS["South West African Coord. System zone 25",GEOGCS["Schwarzeck",DATUM["Schwarzeck",SPHEROID["Bessel Namibia",6377483.865,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator_South_Orientated"],PARAMETER["latitude_of_origin",-22],PARAMETER["central_meridian",25],PARAMETER["scale_factor",1],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["German legal metre",1.0000135965]]',
    'SUDAN':
        'GEOGCS["Sudan",DATUM["Sudan",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'SUDAN / UTM ZONE 35N':
        'PROJCS["Sudan / UTM zone 35N",GEOGCS["Sudan",DATUM["Sudan",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'SUDAN / UTM ZONE 36N':
        'PROJCS["Sudan / UTM zone 36N",GEOGCS["Sudan",DATUM["Sudan",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'TANANARIVE':
        'GEOGCS["Tananarive",DATUM["Tananarive_1925",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'TANANARIVE (PARIS)':
        'GEOGCS["Tananarive (Paris)",DATUM["Tananarive_1925",SPHEROID["International 1924",6378388,297]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]]',
    'TANANARIVE (PARIS) / LABORDE GRID':
        'PROJCS["Tananarive (Paris) / Laborde Grid",GEOGCS["Tananarive (Paris)",DATUM["Tananarive_1925",SPHEROID["International 1924",6378388,297]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]],PROJECTION["Oblique_Mercator"],PARAMETER["latitude_of_center",-21],PARAMETER["longitude_of_center",49],PARAMETER["azimuth",21],PARAMETER["rectified_grid_angle",21],PARAMETER["scale_factor",0.9995],PARAMETER["false_easting",400000],PARAMETER["false_northing",800000],UNIT["metre",1]]',
    'TANANARIVE / UTM ZONE 38S':
        'PROJCS["Tananarive / UTM zone 38S",GEOGCS["Tananarive",DATUM["Tananarive_1925",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'TANANARIVE / UTM ZONE 39S':
        'PROJCS["Tananarive / UTM zone 39S",GEOGCS["Tananarive",DATUM["Tananarive_1925",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'TC(1948)':
        'GEOGCS["TC(1948)",DATUM["Trucial_Coast_1948",SPHEROID["Helmert 1906",6378200,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'TC(1948) / UTM ZONE 39N':
        'PROJCS["TC(1948) / UTM zone 39N",GEOGCS["TC(1948)",DATUM["Trucial_Coast_1948",SPHEROID["Helmert 1906",6378200,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'TC(1948) / UTM ZONE 40N':
        'PROJCS["TC(1948) / UTM zone 40N",GEOGCS["TC(1948)",DATUM["Trucial_Coast_1948",SPHEROID["Helmert 1906",6378200,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'TETE':
        'GEOGCS["Tete",DATUM["Tete",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'TETE / UTM ZONE 36S':
        'PROJCS["Tete / UTM zone 36S",GEOGCS["Tete",DATUM["Tete",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'TETE / UTM ZONE 37S':
        'PROJCS["Tete / UTM zone 37S",GEOGCS["Tete",DATUM["Tete",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'TIMBALAI 1948':
        'GEOGCS["Timbalai 1948",DATUM["Timbalai_1948",SPHEROID["Everest 1830 (1967 Definition)",6377298.556,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'TIMBALAI 1948 / R.S.O. BORNEO (CH)':
        'PROJCS["Timbalai 1948 / R.S.O. Borneo (ch)",GEOGCS["Timbalai 1948",DATUM["Timbalai_1948",SPHEROID["Everest 1830 (1967 Definition)",6377298.556,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Hotine_Oblique_Mercator"],PARAMETER["latitude_of_center",4],PARAMETER["longitude_of_center",115],PARAMETER["azimuth",53.3158204722222],PARAMETER["rectified_grid_angle",53.1301023611111],PARAMETER["scale_factor",0.99984],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["chain (Sears)",20.1167651215526]]',
    'TIMBALAI 1948 / R.S.O. BORNEO (FT)':
        'PROJCS["Timbalai 1948 / R.S.O. Borneo (ft)",GEOGCS["Timbalai 1948",DATUM["Timbalai_1948",SPHEROID["Everest 1830 (1967 Definition)",6377298.556,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Hotine_Oblique_Mercator"],PARAMETER["latitude_of_center",4],PARAMETER["longitude_of_center",115],PARAMETER["azimuth",53.3158204722222],PARAMETER["rectified_grid_angle",53.1301023611111],PARAMETER["scale_factor",0.99984],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["foot (Sears)",0.304799471538676]]',
    'TIMBALAI 1948 / R.S.O. BORNEO (M)':
        'PROJCS["Timbalai 1948 / R.S.O. Borneo (m)",GEOGCS["Timbalai 1948",DATUM["Timbalai_1948",SPHEROID["Everest 1830 (1967 Definition)",6377298.556,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Hotine_Oblique_Mercator"],PARAMETER["latitude_of_center",4],PARAMETER["longitude_of_center",115],PARAMETER["azimuth",53.3158204722222],PARAMETER["rectified_grid_angle",53.1301023611111],PARAMETER["scale_factor",0.99984],PARAMETER["false_easting",0],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'TIMBALAI 1948 / UTM ZONE 49N':
        'PROJCS["Timbalai 1948 / UTM zone 49N",GEOGCS["Timbalai 1948",DATUM["Timbalai_1948",SPHEROID["Everest 1830 (1967 Definition)",6377298.556,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'TIMBALAI 1948 / UTM ZONE 50N':
        'PROJCS["Timbalai 1948 / UTM zone 50N",GEOGCS["Timbalai 1948",DATUM["Timbalai_1948",SPHEROID["Everest 1830 (1967 Definition)",6377298.556,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'TM65':
        'GEOGCS["TM65",DATUM["TM65",SPHEROID["Airy Modified 1849",6377340.189,299.32496]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'TM65 / IRISH NATIONAL GRID':
        'PROJCS["TM65 / Irish National Grid",GEOGCS["TM65",DATUM["TM65",SPHEROID["Airy Modified 1849",6377340.189,299.32496]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",53.5],PARAMETER["central_meridian",-8],PARAMETER["scale_factor",1.000035],PARAMETER["false_easting",200000],PARAMETER["false_northing",250000],UNIT["metre",1]]',
    'TM75':
        'GEOGCS["TM75",DATUM["TM75",SPHEROID["Airy Modified 1849",6377340.189,299.32496]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'TOKYO':
        'GEOGCS["Tokyo",DATUM["Tokyo",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'TRINIDAD 1903':
        'GEOGCS["Trinidad 1903",DATUM["Trinidad_1903",SPHEROID["Clarke 1858",6378293.63924683,294.260676369257]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'TRINIDAD 1903 / TRINIDAD GRID':
        'PROJCS["Trinidad 1903 / Trinidad Grid",GEOGCS["Trinidad 1903",DATUM["Trinidad_1903",SPHEROID["Clarke 1858",6378293.63924683,294.260676369257]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Cassini_Soldner"],PARAMETER["latitude_of_origin",10.4416666666667],PARAMETER["central_meridian",-61.3333333333333],PARAMETER["false_easting",430000],PARAMETER["false_northing",325000],UNIT["link",0.201166194975966]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7001':
        'GEOGCS["Unknown datum based upon ellipsoid 7001",DATUM["Not_specified_based_on_ellipsoid_7001",SPHEROID["Airy 1830",6377563.396,299.32496]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7002':
        'GEOGCS["Unknown datum based upon ellipsoid 7002",DATUM["Not_specified_based_on_ellipsoid_7002",SPHEROID["Airy Modified 1849",6377340.189,299.32496]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7003':
        'GEOGCS["Unknown datum based upon ellipsoid 7003",DATUM["Not_specified_based_on_ellipsoid_7003",SPHEROID["Australian National Spheroid",6378160,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7004':
        'GEOGCS["Unknown datum based upon ellipsoid 7004",DATUM["Not_specified_based_on_ellipsoid_7004",SPHEROID["Bessel 1841",6377397.155,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7005':
        'GEOGCS["Unknown datum based upon ellipsoid 7005",DATUM["Not_specified_based_on_ellipsoid_7005",SPHEROID["Bessel Modified",6377492.018,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7006':
        'GEOGCS["Unknown datum based upon ellipsoid 7006",DATUM["Not_specified_based_on_ellipsoid_7006",SPHEROID["Bessel Namibia",6377483.865,299.15281]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7007':
        'GEOGCS["Unknown datum based upon ellipsoid 7007",DATUM["Not_specified_based_on_ellipsoid_7007",SPHEROID["Clarke 1858",6378293.63924683,294.260676369257]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7008':
        'GEOGCS["Unknown datum based upon ellipsoid 7008",DATUM["Not_specified_based_on_ellipsoid_7008",SPHEROID["Clarke 1866",6378206.4,294.978698213901]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7009':
        'GEOGCS["Unknown datum based upon ellipsoid 7009",DATUM["Not_specified_based_on_ellipsoid_7009",SPHEROID["Clarke 1866 Michigan",6378450.0472441,294.978684676652]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7010':
        'GEOGCS["Unknown datum based upon ellipsoid 7010",DATUM["Not_specified_based_on_ellipsoid_7010",SPHEROID["Clarke 1880 (Benoit)",6378300.79,293.466234570509]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7011':
        'GEOGCS["Unknown datum based upon ellipsoid 7011",DATUM["Not_specified_based_on_ellipsoid_7011",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7012':
        'GEOGCS["Unknown datum based upon ellipsoid 7012",DATUM["Not_specified_based_on_ellipsoid_7012",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7013':
        'GEOGCS["Unknown datum based upon ellipsoid 7013",DATUM["Not_specified_based_on_ellipsoid_7013",SPHEROID["Clarke 1880 (Arc)",6378249.145,293.46631]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7014':
        'GEOGCS["Unknown datum based upon ellipsoid 7014",DATUM["Not_specified_based_on_ellipsoid_7014",SPHEROID["Clarke 1880 (SGA 1922)",6378249.2,293.46598]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7015':
        'GEOGCS["Unknown datum based upon ellipsoid 7015",DATUM["Not_specified_based_on_ellipsoid_7015",SPHEROID["Everest 1830 (1937 Adjustment)",6377276.345,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7016':
        'GEOGCS["Unknown datum based upon ellipsoid 7016",DATUM["Not_specified_based_on_ellipsoid_7016",SPHEROID["Everest 1830 (1967 Definition)",6377298.556,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7017':
        'GEOGCS["Unknown datum based upon ellipsoid 7017",DATUM["Not_specified_based_on_ellipsoid_7017",SPHEROID["Everest 1830 (1975 Definition)",6377301.243,300.801737341596]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7018':
        'GEOGCS["Unknown datum based upon ellipsoid 7018",DATUM["Not_specified_based_on_ellipsoid_7018",SPHEROID["Everest 1830 Modified",6377304.063,300.8017]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7019':
        'GEOGCS["Unknown datum based upon ellipsoid 7019",DATUM["Not_specified_based_on_ellipsoid_7019",SPHEROID["GRS 1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7020':
        'GEOGCS["Unknown datum based upon ellipsoid 7020",DATUM["Not_specified_based_on_ellipsoid_7020",SPHEROID["Helmert 1906",6378200,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7021':
        'GEOGCS["Unknown datum based upon ellipsoid 7021",DATUM["Not_specified_based_on_ellipsoid_7021",SPHEROID["Indonesian National Spheroid",6378160,298.247]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7022':
        'GEOGCS["Unknown datum based upon ellipsoid 7022",DATUM["Not_specified_based_on_ellipsoid_7022",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7024':
        'GEOGCS["Unknown datum based upon ellipsoid 7024",DATUM["Not_specified_based_on_ellipsoid_7024",SPHEROID["Krassowsky 1940",6378245,298.3]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7025':
        'GEOGCS["Unknown datum based upon ellipsoid 7025",DATUM["Not_specified_based_on_ellipsoid_7025",SPHEROID["NWL 9D",6378145,298.25]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7026':
        'GEOGCS["Unknown datum based upon ellipsoid 7026",DATUM["Not_specified_based_on_ellipsoid_7026",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7027':
        'GEOGCS["Unknown datum based upon ellipsoid 7027",DATUM["Not_specified_based_on_ellipsoid_7027",SPHEROID["Plessis 1817",6376523,308.64]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7028':
        'GEOGCS["Unknown datum based upon ellipsoid 7028",DATUM["Not_specified_based_on_ellipsoid_7028",SPHEROID["Struve 1860",6378298.3,294.73]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7029':
        'GEOGCS["Unknown datum based upon ellipsoid 7029",DATUM["Not_specified_based_on_ellipsoid_7029",SPHEROID["War Office",6378300.583,296]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7030':
        'GEOGCS["Unknown datum based upon ellipsoid 7030",DATUM["Not_specified_based_on_ellipsoid_7030",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7031':
        'GEOGCS["Unknown datum based upon ellipsoid 7031",DATUM["Not_specified_based_on_ellipsoid_7031",SPHEROID["GEM 10C",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7032':
        'GEOGCS["Unknown datum based upon ellipsoid 7032",DATUM["Not_specified_based_on_ellipsoid_7032",SPHEROID["OSU86F",6378136.2,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7033':
        'GEOGCS["Unknown datum based upon ellipsoid 7033",DATUM["Not_specified_based_on_ellipsoid_7033",SPHEROID["OSU91A",6378136.3,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7034':
        'GEOGCS["Unknown datum based upon ellipsoid 7034",DATUM["Not_specified_based_on_ellipsoid_7034",SPHEROID["Clarke 1880",6378249.13884613,293.466307655625]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7035':
        'GEOGCS["Unknown datum based upon ellipsoid 7035",DATUM["Not_specified_based_on_ellipsoid_7035",SPHEROID["Sphere",6371000,0]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'UNKNOWN DATUM BASED UPON ELLIPSOID 7036':
        'GEOGCS["Unknown datum based upon ellipsoid 7036",DATUM["Not_specified_based_on_ellipsoid_7036",SPHEROID["GRS 1967",6378160,298.247167427]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'VOIROL 1875':
        'GEOGCS["Voirol 1875",DATUM["Voirol_1875",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'VOIROL 1875 (PARIS)':
        'GEOGCS["Voirol 1875 (Paris)",DATUM["Voirol_1875",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]]',
    'VOIROL 1875 / NORD ALGERIE (ANCIENNE)':
        'PROJCS["Voirol 1875 / Nord Algerie (ancienne)",GEOGCS["Voirol 1875",DATUM["Voirol_1875",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",36],PARAMETER["central_meridian",2.7],PARAMETER["scale_factor",0.999625544],PARAMETER["false_easting",500000],PARAMETER["false_northing",300000],UNIT["metre",1]]',
    'VOIROL 1875 / SUD ALGERIE (ANCIENNE)':
        'PROJCS["Voirol 1875 / Sud Algerie (ancienne)",GEOGCS["Voirol 1875",DATUM["Voirol_1875",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",33.3],PARAMETER["central_meridian",2.7],PARAMETER["scale_factor",0.999625769],PARAMETER["false_easting",500000],PARAMETER["false_northing",300000],UNIT["metre",1]]',
    'VOIROL UNIFIE':
        'GEOGCS["Voirol Unifie",DATUM["Voirol_Unifie_1960",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'VOIROL UNIFIE (PARIS)':
        'GEOGCS["Voirol Unifie (Paris)",DATUM["Voirol_Unifie_1960",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Paris",2.5969213],UNIT["grad",0.015707963267949]]',
    'VOIROL UNIFIE / NORD ALGERIE':
        'PROJCS["Voirol Unifie / Nord Algerie",GEOGCS["Voirol Unifie",DATUM["Voirol_Unifie_1960",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",36],PARAMETER["central_meridian",2.7],PARAMETER["scale_factor",0.999625544],PARAMETER["false_easting",500135],PARAMETER["false_northing",300090],UNIT["metre",1]]',
    'VOIROL UNIFIE / SUD ALGERIE':
        'PROJCS["Voirol Unifie / Sud Algerie",GEOGCS["Voirol Unifie",DATUM["Voirol_Unifie_1960",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Lambert_Conformal_Conic_1SP"],PARAMETER["latitude_of_origin",33.3],PARAMETER["central_meridian",2.7],PARAMETER["scale_factor",0.999625769],PARAMETER["false_easting",500135],PARAMETER["false_northing",300090],UNIT["metre",1]]',
    'WGS 72':
        'GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'WGS 72 / UTM ZONE 10N':
        'PROJCS["WGS 72 / UTM zone 10N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 10S':
        'PROJCS["WGS 72 / UTM zone 10S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 11N':
        'PROJCS["WGS 72 / UTM zone 11N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 11S':
        'PROJCS["WGS 72 / UTM zone 11S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 12N':
        'PROJCS["WGS 72 / UTM zone 12N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 12S':
        'PROJCS["WGS 72 / UTM zone 12S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 13N':
        'PROJCS["WGS 72 / UTM zone 13N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 13S':
        'PROJCS["WGS 72 / UTM zone 13S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 14N':
        'PROJCS["WGS 72 / UTM zone 14N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 14S':
        'PROJCS["WGS 72 / UTM zone 14S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 15N':
        'PROJCS["WGS 72 / UTM zone 15N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 15S':
        'PROJCS["WGS 72 / UTM zone 15S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 16N':
        'PROJCS["WGS 72 / UTM zone 16N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 16S':
        'PROJCS["WGS 72 / UTM zone 16S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 17N':
        'PROJCS["WGS 72 / UTM zone 17N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 17S':
        'PROJCS["WGS 72 / UTM zone 17S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 18N':
        'PROJCS["WGS 72 / UTM zone 18N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 18S':
        'PROJCS["WGS 72 / UTM zone 18S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 19N':
        'PROJCS["WGS 72 / UTM zone 19N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 19S':
        'PROJCS["WGS 72 / UTM zone 19S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 1N':
        'PROJCS["WGS 72 / UTM zone 1N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-177],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 1S':
        'PROJCS["WGS 72 / UTM zone 1S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-177],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 20N':
        'PROJCS["WGS 72 / UTM zone 20N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 20S':
        'PROJCS["WGS 72 / UTM zone 20S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 21N':
        'PROJCS["WGS 72 / UTM zone 21N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 21S':
        'PROJCS["WGS 72 / UTM zone 21S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 22N':
        'PROJCS["WGS 72 / UTM zone 22N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 22S':
        'PROJCS["WGS 72 / UTM zone 22S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 23N':
        'PROJCS["WGS 72 / UTM zone 23N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 23S':
        'PROJCS["WGS 72 / UTM zone 23S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 24N':
        'PROJCS["WGS 72 / UTM zone 24N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 24S':
        'PROJCS["WGS 72 / UTM zone 24S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 25N':
        'PROJCS["WGS 72 / UTM zone 25N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 25S':
        'PROJCS["WGS 72 / UTM zone 25S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 26N':
        'PROJCS["WGS 72 / UTM zone 26N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 26S':
        'PROJCS["WGS 72 / UTM zone 26S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 27N':
        'PROJCS["WGS 72 / UTM zone 27N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 27S':
        'PROJCS["WGS 72 / UTM zone 27S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 28N':
        'PROJCS["WGS 72 / UTM zone 28N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 28S':
        'PROJCS["WGS 72 / UTM zone 28S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 29N':
        'PROJCS["WGS 72 / UTM zone 29N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 29S':
        'PROJCS["WGS 72 / UTM zone 29S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 2N':
        'PROJCS["WGS 72 / UTM zone 2N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-171],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 2S':
        'PROJCS["WGS 72 / UTM zone 2S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-171],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 30N':
        'PROJCS["WGS 72 / UTM zone 30N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 30S':
        'PROJCS["WGS 72 / UTM zone 30S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 31N':
        'PROJCS["WGS 72 / UTM zone 31N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 31S':
        'PROJCS["WGS 72 / UTM zone 31S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 32N':
        'PROJCS["WGS 72 / UTM zone 32N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 32S':
        'PROJCS["WGS 72 / UTM zone 32S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 33N':
        'PROJCS["WGS 72 / UTM zone 33N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 33S':
        'PROJCS["WGS 72 / UTM zone 33S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 34N':
        'PROJCS["WGS 72 / UTM zone 34N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 34S':
        'PROJCS["WGS 72 / UTM zone 34S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 35N':
        'PROJCS["WGS 72 / UTM zone 35N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 35S':
        'PROJCS["WGS 72 / UTM zone 35S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 36N':
        'PROJCS["WGS 72 / UTM zone 36N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 36S':
        'PROJCS["WGS 72 / UTM zone 36S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 37N':
        'PROJCS["WGS 72 / UTM zone 37N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 37S':
        'PROJCS["WGS 72 / UTM zone 37S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 38N':
        'PROJCS["WGS 72 / UTM zone 38N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 38S':
        'PROJCS["WGS 72 / UTM zone 38S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 39N':
        'PROJCS["WGS 72 / UTM zone 39N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 39S':
        'PROJCS["WGS 72 / UTM zone 39S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 3N':
        'PROJCS["WGS 72 / UTM zone 3N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 3S':
        'PROJCS["WGS 72 / UTM zone 3S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 40N':
        'PROJCS["WGS 72 / UTM zone 40N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 40S':
        'PROJCS["WGS 72 / UTM zone 40S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 41N':
        'PROJCS["WGS 72 / UTM zone 41N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 41S':
        'PROJCS["WGS 72 / UTM zone 41S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 42N':
        'PROJCS["WGS 72 / UTM zone 42N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 42S':
        'PROJCS["WGS 72 / UTM zone 42S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 43N':
        'PROJCS["WGS 72 / UTM zone 43N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 43S':
        'PROJCS["WGS 72 / UTM zone 43S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 44N':
        'PROJCS["WGS 72 / UTM zone 44N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 44S':
        'PROJCS["WGS 72 / UTM zone 44S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 45N':
        'PROJCS["WGS 72 / UTM zone 45N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 45S':
        'PROJCS["WGS 72 / UTM zone 45S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 46N':
        'PROJCS["WGS 72 / UTM zone 46N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 46S':
        'PROJCS["WGS 72 / UTM zone 46S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 47N':
        'PROJCS["WGS 72 / UTM zone 47N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 47S':
        'PROJCS["WGS 72 / UTM zone 47S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 48N':
        'PROJCS["WGS 72 / UTM zone 48N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 48S':
        'PROJCS["WGS 72 / UTM zone 48S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 49N':
        'PROJCS["WGS 72 / UTM zone 49N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 49S':
        'PROJCS["WGS 72 / UTM zone 49S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 4N':
        'PROJCS["WGS 72 / UTM zone 4N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 4S':
        'PROJCS["WGS 72 / UTM zone 4S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 50N':
        'PROJCS["WGS 72 / UTM zone 50N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 50S':
        'PROJCS["WGS 72 / UTM zone 50S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 51N':
        'PROJCS["WGS 72 / UTM zone 51N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 51S':
        'PROJCS["WGS 72 / UTM zone 51S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 52N':
        'PROJCS["WGS 72 / UTM zone 52N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 52S':
        'PROJCS["WGS 72 / UTM zone 52S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 53N':
        'PROJCS["WGS 72 / UTM zone 53N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 53S':
        'PROJCS["WGS 72 / UTM zone 53S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 54N':
        'PROJCS["WGS 72 / UTM zone 54N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 54S':
        'PROJCS["WGS 72 / UTM zone 54S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 55N':
        'PROJCS["WGS 72 / UTM zone 55N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 55S':
        'PROJCS["WGS 72 / UTM zone 55S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 56N':
        'PROJCS["WGS 72 / UTM zone 56N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 56S':
        'PROJCS["WGS 72 / UTM zone 56S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 57N':
        'PROJCS["WGS 72 / UTM zone 57N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 57S':
        'PROJCS["WGS 72 / UTM zone 57S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 58N':
        'PROJCS["WGS 72 / UTM zone 58N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 58S':
        'PROJCS["WGS 72 / UTM zone 58S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 59N':
        'PROJCS["WGS 72 / UTM zone 59N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",171],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 59S':
        'PROJCS["WGS 72 / UTM zone 59S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",171],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 5N':
        'PROJCS["WGS 72 / UTM zone 5N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 5S':
        'PROJCS["WGS 72 / UTM zone 5S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 60N':
        'PROJCS["WGS 72 / UTM zone 60N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",177],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 60S':
        'PROJCS["WGS 72 / UTM zone 60S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",177],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 6N':
        'PROJCS["WGS 72 / UTM zone 6N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 6S':
        'PROJCS["WGS 72 / UTM zone 6S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 7N':
        'PROJCS["WGS 72 / UTM zone 7N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 7S':
        'PROJCS["WGS 72 / UTM zone 7S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 8N':
        'PROJCS["WGS 72 / UTM zone 8N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 8S':
        'PROJCS["WGS 72 / UTM zone 8S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 9N':
        'PROJCS["WGS 72 / UTM zone 9N",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72 / UTM ZONE 9S':
        'PROJCS["WGS 72 / UTM zone 9S",GEOGCS["WGS 72",DATUM["WGS_1972",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE':
        'GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'WGS 72BE / UTM ZONE 10N':
        'PROJCS["WGS 72BE / UTM zone 10N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 10S':
        'PROJCS["WGS 72BE / UTM zone 10S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 11N':
        'PROJCS["WGS 72BE / UTM zone 11N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 11S':
        'PROJCS["WGS 72BE / UTM zone 11S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 12N':
        'PROJCS["WGS 72BE / UTM zone 12N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 12S':
        'PROJCS["WGS 72BE / UTM zone 12S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 13N':
        'PROJCS["WGS 72BE / UTM zone 13N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 13S':
        'PROJCS["WGS 72BE / UTM zone 13S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 14N':
        'PROJCS["WGS 72BE / UTM zone 14N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 14S':
        'PROJCS["WGS 72BE / UTM zone 14S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 15N':
        'PROJCS["WGS 72BE / UTM zone 15N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 15S':
        'PROJCS["WGS 72BE / UTM zone 15S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 16N':
        'PROJCS["WGS 72BE / UTM zone 16N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 16S':
        'PROJCS["WGS 72BE / UTM zone 16S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 17N':
        'PROJCS["WGS 72BE / UTM zone 17N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 17S':
        'PROJCS["WGS 72BE / UTM zone 17S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 18N':
        'PROJCS["WGS 72BE / UTM zone 18N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 18S':
        'PROJCS["WGS 72BE / UTM zone 18S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 19N':
        'PROJCS["WGS 72BE / UTM zone 19N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 19S':
        'PROJCS["WGS 72BE / UTM zone 19S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 1N':
        'PROJCS["WGS 72BE / UTM zone 1N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-177],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 1S':
        'PROJCS["WGS 72BE / UTM zone 1S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-177],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 20N':
        'PROJCS["WGS 72BE / UTM zone 20N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 20S':
        'PROJCS["WGS 72BE / UTM zone 20S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 21N':
        'PROJCS["WGS 72BE / UTM zone 21N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 21S':
        'PROJCS["WGS 72BE / UTM zone 21S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 22N':
        'PROJCS["WGS 72BE / UTM zone 22N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 22S':
        'PROJCS["WGS 72BE / UTM zone 22S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 23N':
        'PROJCS["WGS 72BE / UTM zone 23N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 23S':
        'PROJCS["WGS 72BE / UTM zone 23S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 24N':
        'PROJCS["WGS 72BE / UTM zone 24N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 24S':
        'PROJCS["WGS 72BE / UTM zone 24S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 25N':
        'PROJCS["WGS 72BE / UTM zone 25N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 25S':
        'PROJCS["WGS 72BE / UTM zone 25S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 26N':
        'PROJCS["WGS 72BE / UTM zone 26N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 26S':
        'PROJCS["WGS 72BE / UTM zone 26S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 27N':
        'PROJCS["WGS 72BE / UTM zone 27N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 27S':
        'PROJCS["WGS 72BE / UTM zone 27S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 28N':
        'PROJCS["WGS 72BE / UTM zone 28N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 28S':
        'PROJCS["WGS 72BE / UTM zone 28S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 29N':
        'PROJCS["WGS 72BE / UTM zone 29N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 29S':
        'PROJCS["WGS 72BE / UTM zone 29S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 2N':
        'PROJCS["WGS 72BE / UTM zone 2N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-171],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 2S':
        'PROJCS["WGS 72BE / UTM zone 2S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-171],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 30N':
        'PROJCS["WGS 72BE / UTM zone 30N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 30S':
        'PROJCS["WGS 72BE / UTM zone 30S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 31N':
        'PROJCS["WGS 72BE / UTM zone 31N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 31S':
        'PROJCS["WGS 72BE / UTM zone 31S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 32N':
        'PROJCS["WGS 72BE / UTM zone 32N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 32S':
        'PROJCS["WGS 72BE / UTM zone 32S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 33N':
        'PROJCS["WGS 72BE / UTM zone 33N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 33S':
        'PROJCS["WGS 72BE / UTM zone 33S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 34N':
        'PROJCS["WGS 72BE / UTM zone 34N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 34S':
        'PROJCS["WGS 72BE / UTM zone 34S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 35N':
        'PROJCS["WGS 72BE / UTM zone 35N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 35S':
        'PROJCS["WGS 72BE / UTM zone 35S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 36N':
        'PROJCS["WGS 72BE / UTM zone 36N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 36S':
        'PROJCS["WGS 72BE / UTM zone 36S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 37N':
        'PROJCS["WGS 72BE / UTM zone 37N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 37S':
        'PROJCS["WGS 72BE / UTM zone 37S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 38N':
        'PROJCS["WGS 72BE / UTM zone 38N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 38S':
        'PROJCS["WGS 72BE / UTM zone 38S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 39N':
        'PROJCS["WGS 72BE / UTM zone 39N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 39S':
        'PROJCS["WGS 72BE / UTM zone 39S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 3N':
        'PROJCS["WGS 72BE / UTM zone 3N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 3S':
        'PROJCS["WGS 72BE / UTM zone 3S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 40N':
        'PROJCS["WGS 72BE / UTM zone 40N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 40S':
        'PROJCS["WGS 72BE / UTM zone 40S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 41N':
        'PROJCS["WGS 72BE / UTM zone 41N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 41S':
        'PROJCS["WGS 72BE / UTM zone 41S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 42N':
        'PROJCS["WGS 72BE / UTM zone 42N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 42S':
        'PROJCS["WGS 72BE / UTM zone 42S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 43N':
        'PROJCS["WGS 72BE / UTM zone 43N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 43S':
        'PROJCS["WGS 72BE / UTM zone 43S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 44N':
        'PROJCS["WGS 72BE / UTM zone 44N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 44S':
        'PROJCS["WGS 72BE / UTM zone 44S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 45N':
        'PROJCS["WGS 72BE / UTM zone 45N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 45S':
        'PROJCS["WGS 72BE / UTM zone 45S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 46N':
        'PROJCS["WGS 72BE / UTM zone 46N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 46S':
        'PROJCS["WGS 72BE / UTM zone 46S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 47N':
        'PROJCS["WGS 72BE / UTM zone 47N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 47S':
        'PROJCS["WGS 72BE / UTM zone 47S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 48N':
        'PROJCS["WGS 72BE / UTM zone 48N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 48S':
        'PROJCS["WGS 72BE / UTM zone 48S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 49N':
        'PROJCS["WGS 72BE / UTM zone 49N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 49S':
        'PROJCS["WGS 72BE / UTM zone 49S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 4N':
        'PROJCS["WGS 72BE / UTM zone 4N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 4S':
        'PROJCS["WGS 72BE / UTM zone 4S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 50N':
        'PROJCS["WGS 72BE / UTM zone 50N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 50S':
        'PROJCS["WGS 72BE / UTM zone 50S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 51N':
        'PROJCS["WGS 72BE / UTM zone 51N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 51S':
        'PROJCS["WGS 72BE / UTM zone 51S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 52N':
        'PROJCS["WGS 72BE / UTM zone 52N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 52S':
        'PROJCS["WGS 72BE / UTM zone 52S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 53N':
        'PROJCS["WGS 72BE / UTM zone 53N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 53S':
        'PROJCS["WGS 72BE / UTM zone 53S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 54N':
        'PROJCS["WGS 72BE / UTM zone 54N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 54S':
        'PROJCS["WGS 72BE / UTM zone 54S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 55N':
        'PROJCS["WGS 72BE / UTM zone 55N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 55S':
        'PROJCS["WGS 72BE / UTM zone 55S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 56N':
        'PROJCS["WGS 72BE / UTM zone 56N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 56S':
        'PROJCS["WGS 72BE / UTM zone 56S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 57N':
        'PROJCS["WGS 72BE / UTM zone 57N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 57S':
        'PROJCS["WGS 72BE / UTM zone 57S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 58N':
        'PROJCS["WGS 72BE / UTM zone 58N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 58S':
        'PROJCS["WGS 72BE / UTM zone 58S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 59N':
        'PROJCS["WGS 72BE / UTM zone 59N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",171],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 59S':
        'PROJCS["WGS 72BE / UTM zone 59S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",171],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 5N':
        'PROJCS["WGS 72BE / UTM zone 5N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 5S':
        'PROJCS["WGS 72BE / UTM zone 5S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 60N':
        'PROJCS["WGS 72BE / UTM zone 60N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",177],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 60S':
        'PROJCS["WGS 72BE / UTM zone 60S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",177],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 6N':
        'PROJCS["WGS 72BE / UTM zone 6N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 6S':
        'PROJCS["WGS 72BE / UTM zone 6S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 7N':
        'PROJCS["WGS 72BE / UTM zone 7N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 7S':
        'PROJCS["WGS 72BE / UTM zone 7S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 8N':
        'PROJCS["WGS 72BE / UTM zone 8N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 8S':
        'PROJCS["WGS 72BE / UTM zone 8S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 9N':
        'PROJCS["WGS 72BE / UTM zone 9N",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 72BE / UTM ZONE 9S':
        'PROJCS["WGS 72BE / UTM zone 9S",GEOGCS["WGS 72BE",DATUM["WGS_1972_Transit_Broadcast_Ephemeris",SPHEROID["NWL 10D",6378135,298.26]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84':
        'GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'WGS 84 / TM 36 SE':
        'PROJCS["WGS 84 / TM 36 SE",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",36],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UPS NORTH':
        'PROJCS["WGS 84 / UPS North",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Polar_Stereographic"],PARAMETER["latitude_of_origin",90],PARAMETER["central_meridian",0],PARAMETER["scale_factor",0.994],PARAMETER["false_easting",2000000],PARAMETER["false_northing",2000000],UNIT["metre",1]]',
    'WGS 84 / UPS SOUTH':
        'PROJCS["WGS 84 / UPS South",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Polar_Stereographic"],PARAMETER["latitude_of_origin",-90],PARAMETER["central_meridian",0],PARAMETER["scale_factor",0.994],PARAMETER["false_easting",2000000],PARAMETER["false_northing",2000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 10N':
        'PROJCS["WGS 84 / UTM zone 10N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 10S':
        'PROJCS["WGS 84 / UTM zone 10S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 11N':
        'PROJCS["WGS 84 / UTM zone 11N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 11S':
        'PROJCS["WGS 84 / UTM zone 11S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 12N':
        'PROJCS["WGS 84 / UTM zone 12N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 12S':
        'PROJCS["WGS 84 / UTM zone 12S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 13N':
        'PROJCS["WGS 84 / UTM zone 13N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 13S':
        'PROJCS["WGS 84 / UTM zone 13S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 14N':
        'PROJCS["WGS 84 / UTM zone 14N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 14S':
        'PROJCS["WGS 84 / UTM zone 14S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 15N':
        'PROJCS["WGS 84 / UTM zone 15N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 15S':
        'PROJCS["WGS 84 / UTM zone 15S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 16N':
        'PROJCS["WGS 84 / UTM zone 16N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 16S':
        'PROJCS["WGS 84 / UTM zone 16S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 17N':
        'PROJCS["WGS 84 / UTM zone 17N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 17S':
        'PROJCS["WGS 84 / UTM zone 17S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 18N':
        'PROJCS["WGS 84 / UTM zone 18N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 18S':
        'PROJCS["WGS 84 / UTM zone 18S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 19N':
        'PROJCS["WGS 84 / UTM zone 19N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 19S':
        'PROJCS["WGS 84 / UTM zone 19S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 1N':
        'PROJCS["WGS 84 / UTM zone 1N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-177],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 1S':
        'PROJCS["WGS 84 / UTM zone 1S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-177],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 20N':
        'PROJCS["WGS 84 / UTM zone 20N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 20S':
        'PROJCS["WGS 84 / UTM zone 20S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 21N':
        'PROJCS["WGS 84 / UTM zone 21N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 21S':
        'PROJCS["WGS 84 / UTM zone 21S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 22N':
        'PROJCS["WGS 84 / UTM zone 22N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 22S':
        'PROJCS["WGS 84 / UTM zone 22S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 23N':
        'PROJCS["WGS 84 / UTM zone 23N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 23S':
        'PROJCS["WGS 84 / UTM zone 23S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 24N':
        'PROJCS["WGS 84 / UTM zone 24N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 24S':
        'PROJCS["WGS 84 / UTM zone 24S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 25N':
        'PROJCS["WGS 84 / UTM zone 25N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 25S':
        'PROJCS["WGS 84 / UTM zone 25S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 26N':
        'PROJCS["WGS 84 / UTM zone 26N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 26S':
        'PROJCS["WGS 84 / UTM zone 26S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 27N':
        'PROJCS["WGS 84 / UTM zone 27N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 27S':
        'PROJCS["WGS 84 / UTM zone 27S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 28N':
        'PROJCS["WGS 84 / UTM zone 28N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 28S':
        'PROJCS["WGS 84 / UTM zone 28S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 29N':
        'PROJCS["WGS 84 / UTM zone 29N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 29S':
        'PROJCS["WGS 84 / UTM zone 29S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 2N':
        'PROJCS["WGS 84 / UTM zone 2N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-171],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 2S':
        'PROJCS["WGS 84 / UTM zone 2S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-171],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 30N':
        'PROJCS["WGS 84 / UTM zone 30N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 30S':
        'PROJCS["WGS 84 / UTM zone 30S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 31N':
        'PROJCS["WGS 84 / UTM zone 31N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 31S':
        'PROJCS["WGS 84 / UTM zone 31S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",3],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 32N':
        'PROJCS["WGS 84 / UTM zone 32N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 32S':
        'PROJCS["WGS 84 / UTM zone 32S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",9],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 33N':
        'PROJCS["WGS 84 / UTM zone 33N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 33S':
        'PROJCS["WGS 84 / UTM zone 33S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 34N':
        'PROJCS["WGS 84 / UTM zone 34N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 34S':
        'PROJCS["WGS 84 / UTM zone 34S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",21],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 35N':
        'PROJCS["WGS 84 / UTM zone 35N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 35S':
        'PROJCS["WGS 84 / UTM zone 35S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",27],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 36N':
        'PROJCS["WGS 84 / UTM zone 36N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 36S':
        'PROJCS["WGS 84 / UTM zone 36S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",33],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 37N':
        'PROJCS["WGS 84 / UTM zone 37N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 37S':
        'PROJCS["WGS 84 / UTM zone 37S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",39],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 38N':
        'PROJCS["WGS 84 / UTM zone 38N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 38S':
        'PROJCS["WGS 84 / UTM zone 38S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",45],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 39N':
        'PROJCS["WGS 84 / UTM zone 39N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 39S':
        'PROJCS["WGS 84 / UTM zone 39S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",51],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 3N':
        'PROJCS["WGS 84 / UTM zone 3N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 3S':
        'PROJCS["WGS 84 / UTM zone 3S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 40N':
        'PROJCS["WGS 84 / UTM zone 40N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 40S':
        'PROJCS["WGS 84 / UTM zone 40S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 41N':
        'PROJCS["WGS 84 / UTM zone 41N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 41S':
        'PROJCS["WGS 84 / UTM zone 41S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",63],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 42N':
        'PROJCS["WGS 84 / UTM zone 42N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 42S':
        'PROJCS["WGS 84 / UTM zone 42S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",69],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 43N':
        'PROJCS["WGS 84 / UTM zone 43N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 43S':
        'PROJCS["WGS 84 / UTM zone 43S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",75],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 44N':
        'PROJCS["WGS 84 / UTM zone 44N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 44S':
        'PROJCS["WGS 84 / UTM zone 44S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",81],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 45N':
        'PROJCS["WGS 84 / UTM zone 45N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 45S':
        'PROJCS["WGS 84 / UTM zone 45S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",87],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 46N':
        'PROJCS["WGS 84 / UTM zone 46N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 46S':
        'PROJCS["WGS 84 / UTM zone 46S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",93],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 47N':
        'PROJCS["WGS 84 / UTM zone 47N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 47S':
        'PROJCS["WGS 84 / UTM zone 47S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",99],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 48N':
        'PROJCS["WGS 84 / UTM zone 48N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 48S':
        'PROJCS["WGS 84 / UTM zone 48S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",105],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 49N':
        'PROJCS["WGS 84 / UTM zone 49N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 49S':
        'PROJCS["WGS 84 / UTM zone 49S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",111],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 4N':
        'PROJCS["WGS 84 / UTM zone 4N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 4S':
        'PROJCS["WGS 84 / UTM zone 4S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 50N':
        'PROJCS["WGS 84 / UTM zone 50N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 50S':
        'PROJCS["WGS 84 / UTM zone 50S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",117],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 51N':
        'PROJCS["WGS 84 / UTM zone 51N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 51S':
        'PROJCS["WGS 84 / UTM zone 51S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",123],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 52N':
        'PROJCS["WGS 84 / UTM zone 52N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 52S':
        'PROJCS["WGS 84 / UTM zone 52S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 53N':
        'PROJCS["WGS 84 / UTM zone 53N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 53S':
        'PROJCS["WGS 84 / UTM zone 53S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 54N':
        'PROJCS["WGS 84 / UTM zone 54N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 54S':
        'PROJCS["WGS 84 / UTM zone 54S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 55N':
        'PROJCS["WGS 84 / UTM zone 55N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 55S':
        'PROJCS["WGS 84 / UTM zone 55S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 56N':
        'PROJCS["WGS 84 / UTM zone 56N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 56S':
        'PROJCS["WGS 84 / UTM zone 56S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 57N':
        'PROJCS["WGS 84 / UTM zone 57N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 57S':
        'PROJCS["WGS 84 / UTM zone 57S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",159],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 58N':
        'PROJCS["WGS 84 / UTM zone 58N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 58S':
        'PROJCS["WGS 84 / UTM zone 58S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",165],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 59N':
        'PROJCS["WGS 84 / UTM zone 59N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",171],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 59S':
        'PROJCS["WGS 84 / UTM zone 59S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",171],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 5N':
        'PROJCS["WGS 84 / UTM zone 5N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 5S':
        'PROJCS["WGS 84 / UTM zone 5S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-153],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 60N':
        'PROJCS["WGS 84 / UTM zone 60N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",177],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 60S':
        'PROJCS["WGS 84 / UTM zone 60S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",177],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 6N':
        'PROJCS["WGS 84 / UTM zone 6N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 6S':
        'PROJCS["WGS 84 / UTM zone 6S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-147],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 7N':
        'PROJCS["WGS 84 / UTM zone 7N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 7S':
        'PROJCS["WGS 84 / UTM zone 7S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-141],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 8N':
        'PROJCS["WGS 84 / UTM zone 8N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 8S':
        'PROJCS["WGS 84 / UTM zone 8S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-135],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 9N':
        'PROJCS["WGS 84 / UTM zone 9N",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'WGS 84 / UTM ZONE 9S':
        'PROJCS["WGS 84 / UTM zone 9S",GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-129],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",10000000],UNIT["metre",1]]',
    'YACARE':
        'GEOGCS["Yacare",DATUM["Yacare",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'YOFF':
        'GEOGCS["Yoff",DATUM["Yoff",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'YOFF / UTM ZONE 28N':
        'PROJCS["Yoff / UTM zone 28N",GEOGCS["Yoff",DATUM["Yoff",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.466021293627]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-15],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
    'ZANDERIJ':
        'GEOGCS["Zanderij",DATUM["Zanderij",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]',
    'ZANDERIJ / UTM ZONE 21N':
        'PROJCS["Zanderij / UTM zone 21N",GEOGCS["Zanderij",DATUM["Zanderij",SPHEROID["International 1924",6378388,297]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",-57],PARAMETER["scale_factor",0.9996],PARAMETER["false_easting",500000],PARAMETER["false_northing",0],UNIT["metre",1]]',
}

#
# The rest of this is for testing from the command line
#
if __name__ == "__main__":
    wgs84_descript = 'GEOGCS["WGS 84",DATUM["WGS_1984",SPHEROID["WGS 84",6378137,298.257223563]],PRIMEM["Greenwich",0],UNIT["degree",0.0174532925199433]]'

    descript = name_to_descript("wgs 84")
    if descript != wgs84_descript:
        raise ValueError("'wgs 84' as an argument:\n" \
                         "    expect: %s\n"
                         "    found:  %s" % (wgs84_descript, descript))

    num_descripts = len(_DESCRIPT_DICT)
    if num_descripts != 1337:
        raise ValueError("Number of entries in _DESCRIPT_DICT: expect: 1337, found: %d" % num_descripts)

    print "mapprj: SUCCESS"
