
/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Copyright (C) 2005,2006 Yaacov Zamir, Nir Soffer */

/* 	FriBidi python binding:
    
    Compile:
    gcc -fpic -shared `pkg-config --cflags --libs fribidi` 
      -I/usr/include/python2.3 pyfribidi.c -o pyfribidi.so
      
    Install:
    mv pyfribidi.so /usr/lib/python2.3/site-packages/
    
 */

#include <fribidi.h>
#undef _POSIX_C_SOURCE
#include <Python.h>

#define MAX_STR_LEN 65000

int simple_log2vis (char *instring, FriBidiCharType base, char *outstring);

static PyObject *_pyfribidi_log2vis (PyObject * self, PyObject * args);

void initpyfribidi (void);

int
simple_log2vis (char *instring, FriBidiCharType base, char *outstring)
{
	/* the fribidi unicode strings */
	FriBidiChar logical[MAX_STR_LEN];
	FriBidiChar visual[MAX_STR_LEN];

	/* fribidi unicode string length */
	FriBidiStrIndex len = strlen (instring);

	if (len < MAX_STR_LEN)
	{
		len = fribidi_utf8_to_unicode (instring, len, logical);
		fribidi_log2vis (logical, len, &base, visual, NULL, NULL, NULL);
		fribidi_unicode_to_utf8 (visual, len, outstring);
	}
	else
	{
		outstring[0] = '\0';
	}
	return 0;
}

/* From fribidi-0.10.7 README  */
PyDoc_STRVAR (_pyfribidi_log2vis__doc__,
	      "log2vis(logical_string, base_direction='RTL') -> visual_string\n\n"
	      "Reorder logical string to visual order using base direction.\n\n"
	      "- logical_string: string using utf-8 encoding\n"
	      "- base_direction: optional input and output base direction. Either\n"
	      "  'LTR', 'RTL' or 'ON'. 'ON' will cause fribidi to calculate the\n"
	      "  base direction according to the BiDi algorithm.\n"
	      "- returns: string in visual order using utf-8 encoding\n");

static PyObject *
_pyfribidi_log2vis (PyObject * self, PyObject * args)
{
	char *instring = NULL;
	const char *base_dir_string = "RTL";	/* Default */
	FriBidiCharType base_dir;
	char outstring[MAX_STR_LEN] = "";

	if (!PyArg_ParseTuple (args, "s|s", &instring, &base_dir_string))
		return NULL;

	/* Map base_dir_string to FriBidiCharType value */
	if ((strcmp (base_dir_string, "RTL")) == 0)
		base_dir = FRIBIDI_TYPE_RLE;
	else if ((strcmp (base_dir_string, "LTR")) == 0)
		base_dir = FRIBIDI_TYPE_LTR;
	else if ((strcmp (base_dir_string, "ON")) == 0)
		base_dir = FRIBIDI_TYPE_ON;
	else
	{
		PyErr_Format (PyExc_ValueError,
			      "invalid value '%s': exepcted 'RTL', 'LTR' or 'ON'",
			      base_dir_string);
		return NULL;
	}

	simple_log2vis (instring, base_dir, outstring);

	return Py_BuildValue ("s", outstring);
}

static PyMethodDef PyfribidiMethods[] = {
	{"log2vis", (PyCFunction) _pyfribidi_log2vis, METH_VARARGS,
	 _pyfribidi_log2vis__doc__},
	{NULL, NULL, 0, NULL}
};

void
initpyfribidi (void)
{
	(void) Py_InitModule ("pyfribidi", PyfribidiMethods);
}
