# -*- coding: iso-8859-1 -*-
#
# inotify.py - raw python interface to inotify
# Copyright (C) 2005-2006  Sébastien Martini <sebastien.martini@gmail.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
# 02111-1307, USA.

"""
Raw wrap of inotify.

This module wrap:
  1. 3 system calls: inotify_init, inotify_add_watch, inotify_rm_watch.
  2. 3 variables (accessible via proc fs): max_queued_events, max_user_instances, max_user_watches.


How-to manipulate the variables:
  1. Get the int value: tmp_int = max_queued_events.value
  2. Update the value: max_queued_events.value = 42

@var max_queued_events: access it through its 'value' member.
@type max_queued_events: MaxQueuedEvents

@var max_user_instances: access it through its 'value' member.
@type max_user_instances: MaxUserInstances

@var max_user_watches: access it through its 'value' member.
@type max_user_watches: MaxUserWatches

@var INOTIFY_PROC_PATH: inotify's proc path.
@type INOTIFY_PROC_PATH: str

@author: Sebastien Martini
@license: GPL 2
@contact: sebastien.martini@gmail.com
"""
from os.path import join, isfile, isdir
from _inotify import inotify_init, inotify_add_watch, inotify_rm_watch

__all__ = ['max_queued_events', 'max_user_instances',
           'max_user_watches', 'inotify_init', 'inotify_add_watch',
           'inotify_rm_watch']

__metaclass__ = type # Use new-style classes by default


class Singleton:
    """
    Singleton Class pattern
    from recipe:U{http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/66531}

    """
    def __new__(cls, *p, **k):
        if not '_the_instance' in cls.__dict__:
            cls._the_instance = object.__new__(cls)
        return cls._the_instance



class FileTools:
    """
    Used to manipulate (read and write) files.

    """
    def __init__(self, fpath):
        """
        @param fpath: file's path
        @type fpath: str
        """
        assert(isfile(fpath))
        self._fpath = fpath

    def read(self):
        """
        Read content from self._fpath and return it.

        @return: self._fpath's content.
        @rtype: str
        """
        ret = ''
        fo = file(self._fpath, 'r')
        try:
            ret = fo.read()
        finally:
            fo.close()
        return ret

    def write(self, data):
        """
        Write data to self._fpath.

        @param data: data to write.
        @type data: str
        """
        fo = file(self._fpath, 'w')
        try:
            fo.write(data)
        finally:
            fo.close()

    def __repr__(self):
        return self.read()


class ProcINotify(FileTools):
    """
    Manipulate inotify's variables through their representation
    in the inotify's directory of proc fs.

    @cvar INOTIFY_PROC_PATH: inotify directory's path.
    @type INOTIFY_PROC_PATH: str
    """
    def get_inotify_proc_path():
        """
        static method, old style (for python 2.4.3 compatibility)

        @return: inotify's proc path or raise an exception if it
                 couldn't locate this path.
        @rtype: str or RuntimeError
        """
        paths = ['/proc/sys/fs/inotify/',
                 '/proc/sys/filesystem/inotify/',
                 '/proc/sys/kernel/inotify/']
        for path in paths:
            if isdir(path):
                return path
        raise RuntimeError, "Unable to locate inotify's proc path"

    get_inotify_proc_path = staticmethod(get_inotify_proc_path)

    def get_val(self):
        """
        @return: stored value.
        @rtype: int
        """
        return int(self.read())

    def set_val(self, nval):
        """
        @param nval: replace current val by nval.
        @type nval: int
        """
        self.write('%d\n' % nval)

    value = property(get_val, set_val)

    def __repr__(self):
        return '%d' % self.get_val()


class MaxQueuedEvents(Singleton, ProcINotify):
    """
    Singleton class, represents max_queued_events.

    """
    def __init__(self, inotify_path):
        ProcINotify.__init__(self, join(inotify_path,
                                        'max_queued_events'))

class MaxUserInstances(Singleton, ProcINotify):
    """
    Singleton class, represents max_user_instances.

    """
    def __init__(self, inotify_path):
        ProcINotify.__init__(self, join(inotify_path,
                                        'max_user_instances'))

class MaxUserWatches(Singleton, ProcINotify):
    """
    Singleton class, represents max_user_watches.

    """
    def __init__(self, inotify_path):
        ProcINotify.__init__(self, join(inotify_path,
                                        'max_user_watches'))


# inotify's proc path
INOTIFY_PROC_PATH = ProcINotify.get_inotify_proc_path()


# singleton instances
#
# read: my_tmp_int = max_queued_events.value
# update: max_queued_events.value = 42

# defaulting value to 16384
max_queued_events = MaxQueuedEvents(INOTIFY_PROC_PATH)

# defaulting value to 128
max_user_instances = MaxUserInstances(INOTIFY_PROC_PATH)

# defaulting value to 8192
max_user_watches = MaxUserWatches(INOTIFY_PROC_PATH)

