########################################################################
#
# File Name:            Statement.py
#
# Documentation:        http://docs.4suite.org/4Rdf/Statement.py.html
#
"""
The model container for RDF meta-data: represents a graph
WWW: http://4suite.org/4RDF         e-mail: support@4suite.org

Copyright (c) 1999-2001 Fourthought Inc, USA.   All Rights Reserved.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""

from Ft.Rdf.Resource import Resource
from Ft.Rdf import RdfException
from Ft.Rdf import RDF_MS_BASE, OBJECT_TYPE_LITERAL, OBJECT_TYPE_UNKNOWN
#from Ft.Rdf import Container
from Ft import STRING_TYPES


# See this article for ideas on more efficient reification: http://lists.w3.org/Archives/Public/www-rdf-interest/2000Nov/0450.html


class Statement(Resource):
    """Represents an RDF triple or statement"""

    def __init__(self, subject, predicate, object, statementUri='',
                 scope='', objectType=OBJECT_TYPE_UNKNOWN):
        """
        Initializer for an RDF statement instance.
        statementUri is the URI of this statement.
        scope is usually the URI of the serialized source of this statement.
        """
        Resource.__init__(self, statementUri)
        if not type(subject) in STRING_TYPES:
            raise ValueError("Subject not of type string: %s"%str(subject))
        if not type(predicate) in STRING_TYPES:
            raise ValueError("Predicate not of type string: %s"%str(predicate))
        if not type(object) in STRING_TYPES:
            raise ValueError("Object not of type string: %s"%str(object))
        if statementUri is not None and not type(statementUri) in STRING_TYPES:
            raise ValueError("Statement URI not of type string: %s"%str(statementUri))
        if scope is not None and not type(scope) in STRING_TYPES:
            raise ValueError("Scope not of type string: %s"%str(scope))

        self.subject = subject
        self.predicate = predicate
        self.object = object
        self.scope = scope
        self.objectType = objectType
        return

    def __cmp__(self,other):
        if isinstance(other,Statement):
            return abs(cmp(self.subject,other.subject)) + abs(cmp(self.predicate,other.predicate)) + abs(cmp(self.object,other.object)) + abs(cmp(self.scope,other.scope)) + abs(cmp(self.objectType,other.objectType)) + abs(cmp(self.uri,other.uri))
        return cmp(str(self),str(other))
            

    def reify(self, model, uri=None):
        """Prepare the statement in the model so that it can be the subject of
        other statements.  See the RDF spec for details of how statements are expanded for
        reification.  Note that the original statement is not removed by reification.
        The statement's uri is used as the subject of the added statements, or a URI
        is generated if the statement's uri is an empty string."""
        if uri == None:
            uri = self.uri or model.generateUri()
        model.add(Statement(uri, RDF_MS_BASE+'type', RDF_MS_BASE+'Statement',
                            scope=uri))
        model.add(Statement(uri, RDF_MS_BASE+'subject', self.subject,
                            scope=uri))
        model.add(Statement(uri, RDF_MS_BASE+'predicate', self.predicate,
                            scope=uri))
        model.add(Statement(uri, RDF_MS_BASE+'object', self.object,
                            scope=uri))
        return uri

##    def matchSubject(self, testSubject):
##        return self.subject == testSubject

##    def matchPredicate(self, testPredicate):
##        return self.predicate == testPredicate

##    def matchObject(self, testObject):
##        if self.aboutEach and isinstance(self.object, Container):
##            return testObject in self.object
##        else:
##            return testObject == self.object

##    def match(self, subject, predicate, object):
##        return (not subject or self.matchSubject(subject)) and (not predicate or self.matchPredicate(predicate)) and (not object or self.matchObject(object))

##    def isEquivalent(self, other):
##        return self.match(other.subject, other.predicate, other.object)

##    def __cmp__(self, other):
##        return cmp((self.subject, self.predicate, self.object), (other.subject, other.predicate, other.object))

    def __repr__(self):
        """Print the statement, duh"""
        if type(self.object) in STRING_TYPES:
            st = '<RDF Statement at %s: [Subject: %s, Predicate: %s, Object: "%s"]>'%(id(self), self.subject, self.predicate, self.object)
        else:
            st = '<RDF Statement at %s: [Subject: %s, Predicate: %s, Object: %s]>'%(id(self), self.subject, self.predicate, self.object)
        return st

