########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Server/Server/SCore/RawFileImp.py,v 1.16 2003/02/18 04:38:43 molson Exp $
"""
Rawfile repository resource class.

Copyright 2003 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

import types

import ResourceMetaDataImp

from Ft.Server import FTSERVER_NAMESPACE
from Ft.Server.Common import ResourceTypes, Schema
from Ft.Server.Common import SchematronStylesheet, XmlLib
from Ft.Server.Server import FtServerServerException, Error
from Ft.Server.Server.Drivers import FtssDriver, FtssInputSource
from Ft.Xml.XLink import XLINK_NAMESPACE
from Ft.Xml.Xslt import StylesheetReader

class RawFileImp(ResourceMetaDataImp.ResourceMetaDataImp):
    """
    Every resource in the repository is a RawFile
    """
    resourceType = ResourceTypes.ResourceType.RAW_FILE

    def getMetaDataResource(self):
        """
        Get the string meta data of this resource
        """
        self._verifyTx()
        p = self._path.normalize(".;metadata;no-traverse")
        return self._fetchResource(p)

    def getContentResource(self):
        """
        Get the resource the represents the content of this object
        """
        self._verifyTx()
        return self

    def getImt(self):
        """
        Returns the Internet Media Type of the raw file resource
        """
        self._verifyTx()
        res = self._driver.getSystemModel().complete(self._path.absolutePath, Schema.IMT, None)
        if not res:
            raise FtServerServerException(Error.UNKNOWN_PATH,
                                          path = self._path)
        return res[0].object


    def getContent(self):
        """
        Get the string content of this resource
        """
        self._verifyTx()
        return self._driver.fetchResource(self._path)




    ##########################################
    #XML Interfaces
    ##########################################
    def asStylesheet(self):
        """
        Return this object as a non-live compiled stylesheet
        """
        self._verifyTx()

        f,e = self.getExtFunctionsAndElements()

        reader = StylesheetReader.StylesheetReader(extElements = e)

        isrc = FtssInputSource.FtssInputSourceFactory.fromString(
                self.getContent(),
                self._path.absolutePathAsUri,
                self._driver)
        return reader.fromSrc(isrc)

    def asSchematron(self):
        """
        Return this object as a non-live compiled schematron document
        """
        self._verifyTx()
        return SchematronStylesheet.ParseSchematron(
                self.getContent(),
                self._path.absolutePathAsUri,
                self._driver)


    ##########################################
    #Mutation Interfaces
    ##########################################
    def setImt(self,imt):
        """
        Sets the Internet Media Type of the raw file resource
        """
        self._verifyTx()
        xu = XmlLib.MakeString(SET_IMT_XUPDATE%(FTSERVER_NAMESPACE,XLINK_NAMESPACE,imt))
        self._driver.xupdateMetaData(self._path,xu)

    def setContent(self,src):
        """
        Set the string content of this resource
        """
        self._verifyTx()
        if type(src) != types.StringType:
            raise TypeError("src must be a string object")

        self._driver.updateResourceContent(self._path,src)




SET_IMT_XUPDATE="""<xupdate:modifications
  version="1.0"
  xmlns:xupdate="http://www.xmldb.org/xupdate"
  xmlns:ftss="%s"
  xmlns:xlink="%s"
>
  <xupdate:update select="/ftss:MetaData/ftss:Imt">%s</xupdate:update>
</xupdate:modifications>
"""



def NewRawFileXml(driver,path,acl,owner,imt,src):

    a = driver.aclToXml(acl)
    t = FtssDriver.CurrentTime()
    md = """<ftss:MetaData xmlns:ftss="%s" path='%s' type='%s' creation-date='%s'>
  %s
  <ftss:LastModifiedDate>%s</ftss:LastModifiedDate>
  <ftss:Owner>%s</ftss:Owner>
  <ftss:Imt>%s</ftss:Imt>
  <ftss:Size>%d</ftss:Size>
</ftss:MetaData>
        """ % (FTSERVER_NAMESPACE,
               path,
               Schema.g_rdfResourceTypes[ResourceTypes.ResourceType.RAW_FILE],
               t,
               a,
               t,
               owner,
               imt,
               len(src))
    return XmlLib.MakeString(md)
