########################################################################
#
# File Name:            User.py
#
"""
XSLT and XPath extensions supporting the 4SS User API
WWW: http://4suite.org        e-mail: support@4suite.org

Copyright (c) 2000-2001 Fourthought Inc, USA.   All Rights Reserved.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""

import sha
from Ft.Lib import boolean
from Ft.Server import FTSERVER_NAMESPACE
from Ft.Server.Server import FtServerServerException, Error
from Ft.Server.Common import ResourceTypes
from Ft.Xml.XPath import Conversions
from Ft.Xml.Xslt import XsltElement, ContentInfo, AttributeInfo

from Ns import SCORE_NS
import FtssXsltBase


def GetUserName(context, path):
    """
    Get the user name given the path to a user resource.  If the path is invalid or does not give a user object, '' is returned.
    """
    path = Conversions.StringValue(path)
    try:
        username = FtssXsltBase.FetchBaseObject(context.processor, path).getUsername()
    except FtServerServerException, AttributeError:
        username = ''
    return username


class SetUserNameElement(XsltElement):
    """
    Set the name of the user.
    """

    content = ContentInfo.Empty

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(required=1,
                                               description='The path of the user'),
        'new-name' : AttributeInfo.StringAvt(required=1,
					     description='The new name'),
        }
    childArgument = None

    def instantiate(self, context, processor):
        context.setProcessState(self)

        path = self._path.evaluate(context)
        name = self._new_name.evaluate(context)

        user = FtssXsltBase.FetchBaseObject(processor, path)
        user.setUsername(name)

        return (context,)


def SetUserData(context, key, value,path=None):
    key = Conversions.StringValue(key)
    value = Conversions.StringValue(value)
    repo = context.processor._repository
    if path:
        user = FtssXsltBase.FetchBaseObject(context.processor,
                                            Conversions.StringValue(path))
    else:
        user = repo.getCurrentUser()
    if user is not None:
        if user.has_key(key):
            print "Xslt/User: Value '" + value + "' replaces value '" + user[key] + "' for key '" + key + "' in user data."
        user[key] = value
        context.processor.extensionParams[(FTSERVER_NAMESPACE,'commit')]=1
        return boolean.true
    else:
        print "Xslt/User: set-user-data() called but there is no current repo user."
        return boolean.false
        
    
def GetUserData(context, key, path=None):
    key = Conversions.StringValue(key)
    repo = context.processor._repository

    if path:
        user = FtssXsltBase.FetchBaseObject(context.processor,Conversions.StringValue(path))
    else:
        user = repo.getCurrentUser()

    if user is not None:
        if user.has_key(key):
           return user[key]
        else:
            print "Xslt/User: Key '" + key + "' not found in user data."
    else:
        print "Xslt/User: get-user-data() called but there is no current repo user."
    return ''


def GetUserDataKeys(context,path=None):
    if path:
        user = FtssXsltBase.FetchBaseObject(context.processor,Conversions.StringValue(path))
    else:
        user = repo.getCurrentUser()
    doc = context.node.rootNode
    return map(doc.createTextNode, user.keys())


def RemoveUserData(context,key,path=None):
    key = Conversions.StringValue(key)
    repo = context.processor._repository
    if path:
        user = FtssXsltBase.FetchBaseObject(context.processor,Conversions.StringValue(path))
    else:
        user = repo.getCurrentUser()
    if user is not None:
        if user.has_key(key):
            print "Xslt/User: Key '" + key + "' with value '" + user[key] + "' removed from user data."
            del user[key]
            return boolean.true
        else:
            print "Xslt/User: Key '" + key + "not found in user data."
    else:
        print "Xslt/User: remove-user-data() called but there is no current repo user."
    return boolean.false


class ChangeUserPasswordElement(XsltElement):
    """
    Changes the password of the specified user
    """

    content = ContentInfo.Empty

    legalAttrs = {
        'path' : AttributeInfo.UriReferenceAvt(required=1,
                                               description='The path of the user'),
        'new-password' : AttributeInfo.StringAvt(default=None,
                                                 description='The new password. If '
                                                 'not specified, will remove the '
                                                 'password for this user.'),
        }
    childArgument = None

    def instantiate(self, context, processor):
        context.setProcessState(self)

        path = self._path.evaluate(context)
        password = self._new_password.evaluate(context) or ''

        hashedPW = sha.new(password).hexdigest()

        user = FtssXsltBase.FetchBaseObject(processor, path)
        user.setPassword(hashedPW)

        return (context,)


ExtFunctions = {
    (SCORE_NS, 'get-username'):     GetUserName,
    (SCORE_NS, 'get-user-data'):     GetUserData,
    (SCORE_NS, 'get-user-data-keys'):     GetUserDataKeys,
    (SCORE_NS, 'set-user-data'):     SetUserData,
    (SCORE_NS, 'remove-user-data'):  RemoveUserData,
}

ExtElements = {
    (SCORE_NS, 'change-password'): ChangeUserPasswordElement,
    (SCORE_NS, 'set-username'): SetUserNameElement,
}

