########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Xml/FtMiniDom/ValReader.py,v 1.16 2004/06/09 05:08:41 mbrown Exp $
"""
Minidom validating reader; requires xmlproc from PyXML.

If PyXML is not installed at all, then importing this module should
be harmless. If PyXML is installed, but is the wrong version, then the
import will fail with a SystemExit.

Copyright 2004 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

import warnings
import Handler
from Ft.Xml import ReaderException

class ValReader(Handler.Handler):

    ### Reader Interface ###

    def fromSrc(self, source, parseParamEntities=1):
        # DTD validation requires parameter entity parsing
        self.initHandler(source, 1)
        self.prepareParser()

        # Parse it
        self.parse(source)

        self.reset()
        return self.getRootNode()

    def prepareParser(self):
        # We do this here due to the long import time
        from xml.parsers.xmlproc import xmlval, xmlproc

        # Replace xmlproc's default uri resolution.  We handle it
        # in the resolve_*_pubid() methods.
        from xml.parsers.xmlproc import xmlutils, dtdparser
        xmlutils.join_sysids = lambda base, url: url
        xmlproc.join_sysids = lambda base, url: url
        dtdparser.join_sysids = lambda base, url: url

        if self.inputSource.encoding is not None:
            #FIXME: i18n
            msg = "xmlproc does not support external encoding declaration"
            warnings.warn(msg, RuntimeWarning, 2)
        self.parser = xmlval.XMLValidator()

        self.parser.set_application(self)
        self.parser.set_error_handler(self)
        self.parser.set_pubid_resolver(self)
        self.parser.set_inputsource_factory(self)
        self.parser.set_dtd_listener(self)

        # For validation
        self.parser.set_read_external_subset(1)
        return

    def parse(self, source):
        self.parser.parse_resource(source)

    def reset(self):
        self.parser = None
        return

    ### XMLPROC INTERFACES ###

    # --- InputSourceFactory methods

    def create_input_source(self, systemId):
        # systemId as returned from resolve_*_pubid()
        return systemId.stream

    # --- PubIdResolver methods

    def resolve_pe_pubid(self, publicId, systemId):
        #FIXME: this is a quick hack to get around a bug in this class.
        #the parser's InputSource is not kept up to date as entities
        #are resolved.  For some reason the locator's is
        if isinstance(self._locator.get_current_sysid(), type("")):
            resolver = self.parser.get_current_sysid()
        else:
            resolver = self._locator.get_current_sysid()
        #resolver = self._locator.get_current_sysid()
        return resolver.resolve(systemId, publicId, 'PARAMETER_ENTITY', 1)

    def resolve_doctype_pubid(self, publicId, systemId):
        #FIXME: this is a quick hack to get around a bug in this class.
        #the parser's InputSource is not kept up to date as entities
        #are resolved.  For some reason the locator's is
        if isinstance(self._locator.get_current_sysid(), type("")):
            resolver = self.parser.get_current_sysid()
        else:
            resolver = self._locator.get_current_sysid()
        #resolver = self._locator.get_current_sysid()
        return resolver.resolve(systemId, publicId, 'DOCTYPE_DECL', 1)

    def resolve_entity_pubid(self, publicId, systemId):
        #FIXME: this is a quick hack to get around a bug in this class.
        #the parser's InputSource is not kept up to date as entities
        #are resolved.  For some reason the locator's is
        if isinstance(self._locator.get_current_sysid(), type("")):
            resolver = self.parser.get_current_sysid()
        else:
            resolver = self._locator.get_current_sysid()
        #resolver = self._locator.get_current_sysid()
        return resolver.resolve(systemId, publicId, 'EXTERNAL_ENTITY')

    # --- DTDConsumer methods

    def dtd_start(self):
        "Called when DTD parsing starts."
        pass

    def dtd_end(self):
        "Called when the DTD is completely parsed."
        self.endDoctypeDecl()
        return

    def new_general_entity(self,name,val):
        "Receives internal general entity declarations."
        pass

    def new_external_entity(self,ent_name,pub_id,sys_id,ndata):
        """Receives external general entity declarations. 'ndata' is the
        empty string if the entity is parsed."""
        if ndata:
            base = self.inputSource.uri
            self.unparsedEntityDecl(ent_name, base, sys_id, pub_id, ndata)
        pass

    def new_parameter_entity(self,name,val):
        "Receives internal parameter entity declarations."
        pass

    def new_external_pe(self,name,pubid,sysid):
        "Receives external parameter entity declarations."
        pass

    def new_notation(self,name,pubid,sysid):
        "Receives notation declarations."
        pass

    def new_element_type(self,elem_name,elem_cont):
        "Receives the declaration of an element type."
        pass

    def new_attribute(self,elem,attr,a_type,a_decl,a_def):
        "Receives the declaration of a new attribute."
        pass

    # --- Application methods

    def set_locator(self, locator):
        self._locator = locator
        return

    def doc_start(self):
        pass

    def doc_end(self):
        pass

    def handle_comment(self, data):
        self.comment(data)
        return

    def handle_start_tag(self, name, attrs):
        self.startElement(name, attrs)
        return

    def handle_end_tag(self, name):
        self.endElement(name)
        return

    def handle_data(self, data, start, end):
        self.characters(data[start:end])
        return

    def handle_ignorable_data(self, data, start, end):
        self.characters(data[start:end])
        return

    def handle_pi(self, target, data):
        self.processingInstruction(target, data)
        return

    def handle_doctype(self, root, pubId, sysId):
        self.startDoctypeDecl(root, pubId, sysId, 0)
        return

    def set_entity_info(self, xmlver, enc, sddecl):
        pass

    # --- ErrorHandler methods

    def get_locator(self):
        return self._locator

    def fatal(self, message):
        raise ReaderException(ReaderException.XML_PARSE_ERROR,
                              self._locator.get_current_sysid().uri,
                              self._locator.get_line(),
                              self._locator.get_column(),
                              message)
    error = fatal

    def warning(self, message):
        uri = self._locator.get_current_sysid().uri,
        line = self._locator.get_line(),
        col = self._locator.get_column(),
        # FIXME: l10n
        msg = "in %s at line %s, column %s: %s" % (uri[0], line[0], col[0], message)
        warnings.warn(msg, RuntimeWarning, 2)
        return

try:
    from xml.parsers import xmlproc

except ImportError:
    # Create a dummy reader class
    class ValReader:
        def __init__(self, *varargs, **varkw):
            from Ft.Xml import CheckVersion
            CheckVersion('a validating parser')
        # stub for completeness (see valParse() below)
        def fromSrc(self, src):
            raise NotImplementedError("Did the validating parser import correctly?")


def valParse(src, readExtDtd=1):
    r = ValReader()
    return r.fromSrc(src, readExtDtd)
