########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Xml/Xslt/StylesheetTree.py,v 1.14 2003/03/18 08:03:52 uogbuji Exp $
"""
Node classes for the stylesheet tree

Copyright 2002 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""
from Ft.Xml import EMPTY_NAMESPACE
from Ft.Xml.Domlette import GetAllNs
from Ft.Xml.Domlette import XmlStrStrip
from Ft.Xml.Xslt import XSL_NAMESPACE, XsltException, XsltRuntimeException, Error
from Ft.Xml.Xslt import AttributeValueTemplate
from Ft.Xml.Xslt import CategoryTypes
from Ft.Xml.Xslt import ContentInfo

from Ft.Xml.XPath import parser
_xpath_parser = parser
from Ft.Xml.Xslt import parser
_xpattern_parser = parser
del parser

class XsltNode:

    # positional information (for debugger and errors)
    # baseUri also used to set base URI of rtf nodes
    baseUri = ''
    lineNumber = '??'
    columnNumber = '??'
    importIndex = None

    parent = None
    expandedName = (None, None)
    children = None
    attributes = None

    # hints for performance
    doesSetup = 0
    doesPrime = 0
    doesIdle = 0

    def __init__(self, root, baseUri):
        self.root = root
        self.baseUri = baseUri
        self.parent = None
        self.importIndex = -1
        return

    def isLastChild(self):
        siblings = self.parent.children
        if siblings.index(self) == len(siblings) - 1:
            return 1
        else:
            isLast = 1
            for node in siblings[siblings.index(self)+1:]:
                if not node.isPseudoNode():
                    isLast = 0
                    break
            return isLast

    def setup(self):
        return

    def prime(self, processor, context):
        return

    #def idle(self, processor):
    #    return

    def instantiate(self, context, processor):
        return (context,)

    def isPseudoNode(self):
        return


class XsltRoot(XsltNode):

    content = ContentInfo.Alt(ContentInfo.QName(XSL_NAMESPACE, 'xsl:stylesheet'),
                              ContentInfo.QName(XSL_NAMESPACE, 'xsl:transform'),
                              ContentInfo.ResultElements)

    validator = ContentInfo.Validator(content)

    def __init__(self, baseUri):
        XsltNode.__init__(self, self, baseUri)
        self.sources = {}
        self.sourceNodes = {}
        self.primeInstructions = []
        self.idleInstructions = []
        self.stylesheet = None
        self.children = []

    def appendChild(self, child):
        # The validator ensures that only one child will be added
        child.parent = self
        self.stylesheet = child
        self.children = [child]
        return

    def __str__(self):
        return "<XsltRoot at 0x%x>" % id(self)


# implements default behaviour for extension elements
class XsltElement(XsltNode):

    category = CategoryTypes.RESULT_ELEMENT
    content = ContentInfo.Template
    validator = ContentInfo.Validator(content)
    legalAttrs = None # this means no error checking or defaulting

    def __init__(self, root, namespaceUri, localName, baseUri):
        XsltNode.__init__(self, root, baseUri)
        self.expandedName = (namespaceUri, localName)
        self.children = []
        self.attributes = {}
        self.namespaces = {}
        return

    def insertChild(self, index, child):
        self.children.insert(index, child)
        child.parent = self
        if child.doesSetup:
            child.setup()
        return

    def appendChild(self, child):
        self.children.append(child)
        child.parent = self
        if child.doesSetup:
            child.setup()
        return

    def parseAVT(self, avt):
        if avt is None: return None
        try:
            return AttributeValueTemplate.AttributeValueTemplate(avt)
        except SyntaxError, error:
            raise XsltException(Error.INVALID_AVT, avt, self.baseUri,
                                self.lineNumber, self.columnNumber,
                                str(error))
        except XsltException, error:
            raise XsltException(Error.INVALID_AVT, avt, self.baseUri,
                                self.lineNumber, self.columnNumber,
                                error.args[0])

    def parseExpression(self, expression):
        if expression is None: return None
        p = _xpath_parser.new()
        try:
            return p.parse(expression)
        except SyntaxError, error:
            raise XsltException(Error.INVALID_EXPRESSION, expression,
                                self.baseUri, self.lineNumber,
                                self.columnNumber, str(error))

    def parsePattern(self, pattern):
        if pattern is None: return None
        p = _xpattern_parser.new()
        try:
            return p.parse(pattern)
        except SyntaxError, error:
            raise XsltException(Error.INVALID_PATTERN, pattern,
                                self.baseUri, self.lineNumber,
                                self.columnNumber, str(error))

    def splitQName(self, qname):
        if not qname: return None
        index = qname.find(':')
        if index != -1:
            split = (qname[:index], qname[index+1:])
        else:
            split = (None, qname)
        return split

    def expandQName(self, qname, refNode=None):
        if not qname: return None
        if refNode:
            namespaces = GetAllNs(refNode)
        else:
            namespaces = self.namespaces
        prefix, local = self.splitQName(qname)
        if prefix:
            try:
                expanded = (namespaces[prefix], local)
            except KeyError:
                raise XsltRuntimeException(Error.UNDEFINED_PREFIX,
                                           self, prefix)
        else:
            expanded = (EMPTY_NAMESPACE, local)
        return expanded

    def instantiate(self, context, processor):
        for child in self.children:
            if child.expandedName == (XSL_NAMESPACE, 'fallback'):
                child.instantiate(context, processor)
        return (context,)

    def processChildren(self, context, processor):
        context.processorNss = self.namespaces
        context.currentInstruction = self

        for child in self.children:
            context = child.instantiate(context, processor)[0]
        return (context, )

    def __str__(self):
        namespace, local = self.expandedName
        prefix = self.namespaces.get(namespace)
        if prefix:
            name = prefix + ':' + local
        else:
            name = local
        #FIXME: Should this use self.__class__ or sth rather than hardcoding "XsltElement"?
        return "<XsltElement at 0x%x: name %s, %d attributes, %d children>" % (
            id(self), repr(name), len(self.attributes), len(self.children)
            )


class XsltText(XsltNode):

    def __init__(self, root, baseUri, data):
        XsltNode.__init__(self, root, baseUri)
        self.data = data
        return

    def instantiate(self, context, processor):
        processor.writers[-1].text(self.data)
        return (context,)

    def __str__(self):
        if len(self.data) > 20:
            data = self.data[:20] + '...'
        else:
            data = self.data
        return "<XsltText at 0x%x: %s>" % (id(self), repr(data))

