#include "domlette.h"

/*Internal Functions*/

#define Element_VerifyState(ob)                                     \
  if (!PyElement_Check(ob) ||                                       \
      ((PyElementObject *)(ob))->namespaceURI == NULL ||            \
      ((PyElementObject *)(ob))->prefix == NULL ||                  \
      ((PyElementObject *)(ob))->localName == NULL ||               \
      ((PyElementObject *)(ob))->nodeName == NULL ||                \
      ((PyElementObject *)(ob))->childNodes == NULL ||              \
      ((PyElementObject *)(ob))->attributes == NULL) {              \
     DOMException_InvalidStateErr("Element in inconsistent state"); \
     return NULL;                                                   \
  } 

static PyObject *my_node_type;

static PyObject *buildAttrKey(PyAttrObject *attr)
{
  PyObject *key;
  PyObject *local;

  switch (PyObject_RichCompareBool(attr->namespaceURI, g_xmlnsNamespace, Py_EQ)) {
  case 0:
    /* normal attribute */
    local = attr->localName;
    break;
  case 1:
    /* namespace attribute */
    if (attr->prefix != Py_None) {
      /* xmlns:prefix = 'namespaceURI' */
      local = attr->localName;
    }
    else {
      /* xmlns = 'namespaceURI' */
      local = Py_None;
    }
    break;
  default:
    /* error */
    return NULL;
  }

  key = PyTuple_New(2);

  Py_INCREF(attr->namespaceURI);
  PyTuple_SET_ITEM(key, 0, attr->namespaceURI);

  Py_INCREF(local);
  PyTuple_SET_ITEM(key, 1, local);

  return key;
}

/* returns a new reference */
PyAttrObject *Element_SetAttributeNS(PyElementObject *self,
				     PyObject *namespaceURI,
                                     PyObject *qualifiedName,
				     PyObject *prefix,
				     PyObject *localName,
                                     PyObject *value)
{
  PyObject *key;
  register PyAttrObject *attr;

  Element_VerifyState(self);

  DOMString_NullCheck(prefix);
  DOMString_NullCheck(namespaceURI);

  /* new reference */
  attr = Document_CreateAttributeNS((PyDocumentObject*)self->ownerDocument,
                                    namespaceURI, qualifiedName, prefix,
                                    localName, value);

  Py_DECREF(attr->parentNode);
  attr->parentNode = (PyObject *)self;
  Py_INCREF(self);

  /* all attributes share a docIndex since they are unordered.
   * self->docIndex + 1 is for namespace nodes (assigned in XPath)
   * self->docIndex + 2 is for attribute nodes
   */
  attr->docIndex = self->docIndex + 2;

  key = buildAttrKey(attr);
  if (key) {
    PyDict_SetItem(self->attributes, key, (PyObject *)attr);
    Py_DECREF(key);
  } else {
    Py_DECREF(attr);
    attr = NULL;
  }

  return attr;
}


PyObject *Element_GetAttributeNodeNS(PyElementObject *elem,
                                     PyObject *namespaceURI,
                                     PyObject *localName)
{
  /* Returns a borrowed ref */
  PyObject *key;
  PyObject *attr;

  Element_VerifyState(elem);

  Py_INCREF(namespaceURI);
  Py_INCREF(localName);
  /* steals reference */
  key = PyTuple_New(2);
  PyTuple_SetItem(key, 0, namespaceURI);
  PyTuple_SetItem(key, 1, localName);

  attr = PyDict_GetItem(elem->attributes, key);
  Py_DECREF(key);

  return attr ? attr : Py_None;
}

PyElementObject *Element_CloneNode(PyObject *node, int deep,
				   PyNodeObject *newOwnerDocument)
{
  long tempId = 152;
  PyElementObject *element;
  PyObject *namespaceURI, *prefix, *localName, *qualifiedName;
  PyObject *attributes;
  int i, count;

  if (!PyDocument_Check(newOwnerDocument)) {
    PyErr_SetString(PyExc_TypeError, "newOwnerDocument must be a cDocument");
    return NULL;
  }

  namespaceURI = PyObject_GetAttrString(node, "namespaceURI");
  namespaceURI = DOMString_FromObjectInplace(namespaceURI);
  qualifiedName = PyObject_GetAttrString(node, "nodeName");
  qualifiedName = DOMString_FromObjectInplace(qualifiedName);
  prefix = PyObject_GetAttrString(node, "prefix");
  prefix = DOMString_FromObjectInplace(prefix);
  localName = PyObject_GetAttrString(node, "localName");
  localName = DOMString_FromObjectInplace(localName);

  /* attributes are cloned regardless of the deep argument */
  attributes = PyObject_GetAttrString(node, "attributes");
  if (attributes) {
    /* get the actual attribute nodes from the attributes mapping */
    PyObject *values = PyMapping_Values(attributes);
    Py_DECREF(attributes);
    attributes = values;
  }
  if (namespaceURI == NULL || qualifiedName == NULL || prefix == NULL || 
      localName == NULL || attributes == NULL) {
    Py_XDECREF(attributes);
    Py_XDECREF(localName);
    Py_XDECREF(prefix);
    Py_XDECREF(qualifiedName);
    Py_XDECREF(namespaceURI);
    return NULL;
  }

  /* We now have everything we need to create a shallow copy, do it */
  element = Document_CreateElementNS((PyDocumentObject *)newOwnerDocument,
                                     namespaceURI, qualifiedName, prefix,
                                     localName, &tempId);

  /* Done with these */
  Py_DECREF(namespaceURI);
  Py_DECREF(qualifiedName);
  Py_DECREF(prefix);
  Py_DECREF(localName);

  if (element == NULL) {
    Py_DECREF(attributes);
    return NULL;
  }

  /* copy the attributes */
  count = PySequence_Length(attributes);
  for (i = 0; i < count; i++) {
    PyAttrObject *attr;
    PyObject *value;
    PyObject *old_attr;

    old_attr = PySequence_GetItem(attributes, i);
    if (old_attr == NULL) {
      Py_DECREF(element);
      Py_DECREF(attributes);
      return NULL;
    }

    namespaceURI = PyObject_GetAttrString(old_attr, "namespaceURI");
    namespaceURI = DOMString_FromObjectInplace(namespaceURI);
    qualifiedName = PyObject_GetAttrString(old_attr, "nodeName");
    qualifiedName = DOMString_FromObjectInplace(qualifiedName);
    prefix = PyObject_GetAttrString(old_attr, "prefix");
    prefix = DOMString_FromObjectInplace(prefix);
    localName = PyObject_GetAttrString(old_attr, "localName");
    localName = DOMString_FromObjectInplace(localName);
    value = PyObject_GetAttrString(old_attr, "value");
    value = DOMString_FromObjectInplace(value);
    Py_DECREF(old_attr);
    if (namespaceURI == NULL || prefix == NULL || localName == NULL ||
        qualifiedName == NULL ||value == NULL) {
      Py_XDECREF(value);
      Py_XDECREF(qualifiedName);
      Py_XDECREF(localName);
      Py_XDECREF(prefix);
      Py_XDECREF(namespaceURI);
      Py_DECREF(element);
      Py_DECREF(attributes);
      return NULL;
    }

    attr = Element_SetAttributeNS(element, namespaceURI, qualifiedName, prefix,
                                  localName, value);
    Py_DECREF(value);
    Py_DECREF(localName);
    Py_DECREF(prefix);
    Py_DECREF(qualifiedName);
    Py_DECREF(namespaceURI);
    if (attr == NULL) {
      Py_DECREF(element);
      Py_DECREF(attributes);
      return NULL;
    }
    Py_DECREF(attr);
  }
  Py_DECREF(attributes);

  if (deep) {
    PyObject *childNodes;

    childNodes = PyObject_GetAttrString(node, "childNodes");
    if (childNodes == NULL) {
      Py_DECREF(element);
      return NULL;
    }
    count = PySequence_Length(childNodes);
    for (i = 0; i < count; i++) {
      PyObject *child;
      PyNodeObject *cloned_child;

      child = PySequence_GetItem(childNodes, i);
      if (child == NULL) {
	Py_DECREF(childNodes);
	Py_DECREF(element);
	return NULL;
      }

      cloned_child = Node_CloneNode(child, deep, newOwnerDocument);
      Py_DECREF(child);
      if (cloned_child == NULL) {
	Py_DECREF(childNodes);
	Py_DECREF(element);
	return NULL;
      }

      Node_AppendChild((PyNodeObject *)element, cloned_child);
      Py_DECREF(cloned_child);
    }
    Py_DECREF(childNodes);
  }

  return element;
}


/*
  Python Interface
*/


static char PyElement_getAttributeNodeNS__doc__[] =
"Retrieves an Attr node by local name and namespace URI.";

static PyObject *PyElement_getAttributeNodeNS(PyObject *self, PyObject *args)
{
  PyObject *namespaceURI, *localName;
  PyObject *attr;

  Element_VerifyState(self);

  if (!PyArg_ParseTuple(args, "OO:getAttributeNodeNS", 
                        &namespaceURI, &localName))
    return NULL;

  namespaceURI = DOMString_ConvertArgument(namespaceURI, "namespaceURI", 1);
  if (namespaceURI == NULL) {
    return NULL;
  }

  localName = DOMString_ConvertArgument(localName, "localName", 0);
  if (localName == NULL) {
    Py_DECREF(namespaceURI);
    return NULL;
  }

  attr = Element_GetAttributeNodeNS((PyElementObject *)self, namespaceURI, localName);

  Py_DECREF(namespaceURI);
  Py_DECREF(localName);

  Py_INCREF(attr);
  return attr;
}


static char PyElement_getAttributeNS__doc__[] =
"Retrieves an attribute value by local name and namespace URI.";

static PyObject *PyElement_getAttributeNS(PyObject *self, PyObject *args)
{
  PyObject *namespaceURI, *localName;
  PyObject *attr;

  Element_VerifyState(self);

  if (!PyArg_ParseTuple(args, "OO:getAttributeNS", &namespaceURI, &localName))
    return NULL;

  namespaceURI = DOMString_ConvertArgument(namespaceURI, "namespaceURI", 1);
  if (namespaceURI == NULL) {
    return NULL;
  }

  localName = DOMString_ConvertArgument(localName, "localName", 0);
  if (localName == NULL) {
    Py_DECREF(namespaceURI);
    return NULL;
  }

  attr = Element_GetAttributeNodeNS((PyElementObject *)self, namespaceURI, localName);

  Py_DECREF(namespaceURI);
  Py_DECREF(localName);

  if (attr == Py_None) {
    /* empty unicode string */
    return PyUnicode_FromUnicode(NULL, 0);
  } else {
    Py_INCREF(PyAttr_NODE_VALUE(attr));
    return PyAttr_NODE_VALUE(attr);
  }
}


static char PyElement_hasAttributeNS__doc__[] =
"Returns True when an attribute with a given local name and namespace URI\n\
is specified on this element or has a default value, False otherwise.";

static PyObject *PyElement_hasAttributeNS(PyObject *self, PyObject *args)
{
  PyObject *namespaceURI, *localName;
  PyObject *attr;

  Element_VerifyState(self);

  if (!PyArg_ParseTuple(args, "OO:hasAttributeNS", &namespaceURI, &localName))
    return NULL;

  namespaceURI = DOMString_ConvertArgument(namespaceURI, "namespaceURI", 1);
  if (namespaceURI == NULL) {
    return NULL;
  }

  localName = DOMString_ConvertArgument(localName, "localName", 0);
  if (localName == NULL) {
    Py_DECREF(namespaceURI);
    return NULL;
  }

  attr = Element_GetAttributeNodeNS((PyElementObject *)self, namespaceURI, localName);

  Py_DECREF(namespaceURI);
  Py_DECREF(localName);

  if (attr == Py_None) {
    Py_INCREF(Py_False);
    return Py_False;
  }

  Py_INCREF(Py_True);
  return Py_True;
}


static char PyElement_setAttributeNS__doc__[] =
"Adds a new attribute.  If an attribute with the same local name and\n\
namespace URI is already present on the element, its prefix is changed\n\
to be the prefix part of the qualifiedName, and its value is changed to\n\
be the value parameter.";

static PyObject *PyElement_setAttributeNS(PyObject *self, PyObject *args)
{
  PyObject *namespaceURI, *qualifiedName, *value, *prefix, *localName;
  PyAttrObject *attr;

  Element_VerifyState(self);

  if (!PyArg_ParseTuple(args, "OOO:setAttributeNS", 
                        &namespaceURI, &qualifiedName, &value))
    return NULL;

  namespaceURI = DOMString_ConvertArgument(namespaceURI, "namespaceURI", 1);
  if (namespaceURI == NULL) {
    return NULL;
  }

  qualifiedName = DOMString_ConvertArgument(qualifiedName, "qualifiedName", 0);
  if (qualifiedName == NULL) {
    Py_DECREF(namespaceURI);
    return NULL;
  }

  value = DOMString_ConvertArgument(value, "value", 0);
  if (value == NULL) {
    Py_DECREF(namespaceURI);
    Py_DECREF(qualifiedName);
    return NULL;
  }

  if (!SplitQName(qualifiedName, &prefix, &localName)) {
    Py_DECREF(namespaceURI);
    Py_DECREF(qualifiedName);
    return NULL;
  }

  /* returns new reference */
  attr = Element_SetAttributeNS((PyElementObject *)self, namespaceURI,
                                qualifiedName, prefix, localName, value);
  Py_DECREF(namespaceURI);
  Py_DECREF(qualifiedName);
  Py_DECREF(prefix);
  Py_DECREF(localName);
  Py_DECREF(value);

  return (PyObject *) attr;
}


static char PyElement_setAttributeNodeNS__doc__[] =
"Adds a new attribute. If an attribute with that local name and that\n\
namespace URI is already present in the element, it is replaced by the\n\
new one. Replacing an attribute node by itself has no effect.";

static PyObject *PyElement_setAttributeNodeNS(PyObject *self, PyObject *args)
{
  PyAttrObject *attr;
  PyObject *oldAttr, *key;

  Element_VerifyState(self);

  if (!PyArg_ParseTuple(args, "O!:setAttributeNodeNS", &PyDomletteAttr_Type,
                        &attr))
    return NULL;

  key = buildAttrKey(attr);

  /* Get the return value */
  oldAttr = PyDict_GetItem(PyElement_ATTRIBUTES(self), key);

  /* Set the new attribute */
  PyDict_SetItem(PyElement_ATTRIBUTES(self), key, (PyObject *)attr);
  Py_DECREF(key);

  /* Set the new attributes owner */
  PyNode_PARENT_NODE(attr) = self;
  Py_INCREF(self);

  /* all attributes share a docIndex since they are unordered.
   * self->docIndex + 1 is for namespace nodes (assigned in XPath)
   * self->docIndex + 2 is for attribute nodes
   */
  PyNode_INDEX(attr) = PyNode_INDEX(self) + 2;
    
  if (oldAttr == NULL) {
    /* new attribute */
    oldAttr = Py_None;
  } 
  else {
    /* Reset the removed attributes owner */
    Py_DECREF(PyNode_PARENT_NODE(oldAttr));
    PyNode_PARENT_NODE(oldAttr) = Py_None;
    Py_INCREF(Py_None);
  }

  Py_INCREF(oldAttr);
  return oldAttr;
}


static char PyElement_removeAttributeNode__doc__[] =
"Removes the specified attribute node.";

static PyObject *PyElement_removeAttributeNode(PyObject *self, PyObject *args)
{
  PyAttrObject *attr;
  PyObject *key;

  Element_VerifyState(self);

  if (!PyArg_ParseTuple(args, "O!:removeAttributeNode", &PyDomletteAttr_Type,
                        &attr))
    return NULL;

  key = buildAttrKey(attr);
  if (PyDict_DelItem(PyElement_ATTRIBUTES(self), key) == -1) {
    if (PyErr_ExceptionMatches(PyExc_KeyError)) {
      DOMException_NotFoundErr("attribute not found");
    }
    Py_DECREF(key);
    return NULL;
  }
  Py_DECREF(key);

  /* Reset the removed attributes owner */
  Py_DECREF(PyNode_PARENT_NODE(attr));
  PyNode_PARENT_NODE(attr) = Py_None;
  Py_INCREF(Py_None);

  Py_INCREF(Py_None);
  return Py_None;
}


static char PyElement_removeAttributeNS__doc__[] =
"Removes an attribute by local name and namespace URI.";

static PyObject *PyElement_removeAttributeNS(PyObject *self, PyObject *args)
{
  PyObject *namespaceURI, *qualifiedName, *prefix, *localName, *key, *attr;

  Element_VerifyState(self);

  if (!PyArg_ParseTuple(args, "OO:removeAttributeNS", 
                        &namespaceURI, &qualifiedName))
    return NULL;

  namespaceURI = DOMString_ConvertArgument(namespaceURI, "namespaceURI", 1);
  if (namespaceURI == NULL) {
    return NULL;
  }

  qualifiedName = DOMString_ConvertArgument(qualifiedName, "qualifiedName", 0);
  if (qualifiedName == NULL) {
    Py_DECREF(namespaceURI);
    return NULL;
  }

  if (!SplitQName(qualifiedName, &prefix, &localName)) {
    Py_DECREF(namespaceURI);
    Py_DECREF(qualifiedName);
    return NULL;
  }

  /* Done with these */
  Py_DECREF(qualifiedName);
  Py_DECREF(prefix);

  key = PyTuple_New(2);
  /* Let the tuple own these */
  PyTuple_SetItem(key, 0, namespaceURI);
  PyTuple_SetItem(key, 1, localName);

  attr = PyDict_GetItem(PyElement_ATTRIBUTES(self), key);
  if (attr) {
    /* Ensure that this doesn't go away while we're changing it */
    Py_INCREF(attr);

    /* Remove the entry from the attributes mapping */
    if (PyDict_DelItem(PyElement_ATTRIBUTES(self), key) == -1) {
      Py_DECREF(attr);
      Py_DECREF(key);
      return NULL;
    }

    /* Unset the parent node relationship */
    Py_DECREF(PyNode_PARENT_NODE(attr));
    Py_INCREF(Py_None);
    PyNode_PARENT_NODE(attr) = Py_None;

    /* Done with our changes */
    Py_DECREF(attr);
  }

  /* Destroy the key */
  Py_DECREF(key);

  Py_INCREF(Py_None);
  return Py_None;
}

#define PyElement_Method(meth) \
  { #meth, PyElement_##meth, METH_VARARGS, PyElement_##meth##__doc__ }

static struct PyMethodDef Element_methods[] = {
  NODE_METHODS,
  PyElement_Method(getAttributeNS),
  PyElement_Method(getAttributeNodeNS),
  PyElement_Method(setAttributeNS),
  PyElement_Method(setAttributeNodeNS),
  PyElement_Method(removeAttributeNS),
  PyElement_Method(removeAttributeNode),
  PyElement_Method(hasAttributeNS),
  { NULL }      /* sentinel */
};

#undef PyElement_Method

/*
  Our Type interface
*/

static PyObject *element_getattr(PyElementObject *self, char *name)
{
  int temp;
  PyObject *rt = NULL;

  Element_VerifyState(self);

  if (!strcmp(name, "tagName")) {
    rt = (PyObject *)self->nodeName;
  }
  else if (!strcmp(name, "nodeType")) {
    rt = my_node_type;
  }
  else if (!strcmp(name, "nodeName")) {
    rt = (PyObject *)self->nodeName;
  }
  else if (!strcmp(name, "localName")) {
    rt = (PyObject *)self->localName;
  }
  else if (!strcmp(name, "prefix")) {
    rt = (PyObject *)self->prefix;
  }
  else if (!strcmp(name, "namespaceURI")) {
    rt = self->namespaceURI;
  }
  else if (!strcmp(name, "attributes")) {
    rt = self->attributes;
  }
  else if (!strcmp(name, "childNodes")) {
    rt = self->childNodes;
  }
  else if (!strcmp(name, "lastChild")) {
    temp = PyList_GET_SIZE(self->childNodes);
    if (temp) {
      rt = PyList_GET_ITEM(self->childNodes, temp-1);
    } else {
      rt = Py_None;
    }
  }
  else if (!strcmp(name,"firstChild")) {
    if (PyList_GET_SIZE(self->childNodes)) {
      rt = PyList_GET_ITEM(self->childNodes,0);
    } else {
      rt = Py_None;
    }
  }
  if (rt) {
    Py_INCREF(rt);
    return rt;
  }
  return node_getattr((PyNodeObject*)self,name,Element_methods);
}

static int element_setattr(PyElementObject *self, char *name, PyObject *v)
{
  /* Set attribute 'name' to value 'v'. v==NULL means delete */
  if (v == NULL) {
    PyErr_Format(PyExc_AttributeError,
                 "Cannot delete attribute '%.400s' on '%.50s' object",
                 name, self->ob_type->tp_name);
  }
  else {
    PyErr_Format(PyExc_AttributeError,
                 "Cannot set attribute '%.400s' on '%.50s' object",
                 name, self->ob_type->tp_name);
  }
  return -1;
}

static void element_dealloc(PyElementObject *self)
{
  PyObject_GC_UnTrack((PyObject *) self);

  Py_XDECREF(self->namespaceURI);
  self->namespaceURI = NULL;

  Py_XDECREF(self->prefix);
  self->prefix = NULL;

  Py_XDECREF(self->localName);
  self->localName = NULL;

  Py_XDECREF(self->nodeName);
  self->nodeName = NULL;

  if (self->attributes) {
    PyDict_Clear(self->attributes);
    Py_DECREF(self->attributes);
    self->attributes = NULL;
  }

  Py_XDECREF(self->childNodes);
  self->childNodes = NULL;

  Node_Del(self);
}

static int element_clear(PyElementObject *self)
{
  (void) PyList_SetSlice(self->childNodes, 0, -1, 0);

  PyDict_Clear(self->attributes);

  return node_clear((PyNodeObject *)self);
}

static int element_traverse(PyElementObject *self, visitproc visit, void *arg)
{
  int rt;
  if (self->childNodes != NULL) {
    rt = visit(self->childNodes, arg);
    if (rt) return rt;
  }
  if (self->attributes != NULL) {
    rt = visit(self->attributes, arg);
    if (rt) return rt;
  }
  return node_traverse((PyNodeObject *)self,visit,arg);
}


static PyObject *element_repr(PyElementObject *element)
{
  char buf[512];
  char *tagname;
  PyObject *name = PyObject_Repr((PyObject *)(element->nodeName));

  if (name) {
    tagname = PyString_AsString(name);
    Py_DECREF(name);
  } else {
    tagname = "(null)";
  }

  sprintf(buf, "<cElement at %p: name %s, %d attributes, %d children>",
          element,
          tagname,
          PyDict_Size(element->attributes),
          PyList_GET_SIZE(element->childNodes)
          );
  return PyString_FromString(buf);
}


int element_test_ref_counts(PyObject *tester,PyElementObject *elem,long *childCtr,PyObject *internDict,int base) {
  /*Test the refcounts on an element, first, test our children, then our attrs, then ourselves.
    Also remeber to add our names to the intern counter */
  int ctr;
  PyObject *curChild;
  int numAttrs = 0;
  int i=0;
  PyObject *attr_name_tuple;
  PyObject *attr;

  /*Test our children*/
  for (ctr=0;ctr<PyList_GET_SIZE(elem->childNodes);ctr++) {
    curChild = PyList_GET_ITEM(elem->childNodes,ctr);
    if (!node_test_ref_counts(tester,(PyNodeObject *)curChild,childCtr,internDict,0)) {
      return 0;
    }
  }

  /*Test our attrs*/
  while (PyDict_Next(elem->attributes, &i, &attr_name_tuple, &attr)) {
    numAttrs ++;
    if (!node_test_ref_counts(tester, (PyNodeObject *)attr, childCtr, internDict, 0)) {
      return 0;
    }
#ifndef DISABLE_INTERN
    AddInternCtr(PyTuple_GET_ITEM(attr_name_tuple,0),internDict);
    AddInternCtr(PyTuple_GET_ITEM(attr_name_tuple,1),internDict);
#endif
  }


#ifdef DISABLE_INTERN
  if (elem->namespaceURI != Py_None)
    if (!TestRefCount(tester,(PyObject *)elem->namespaceURI,1, "namespaceURI")) return 0;

  if (elem->prefix != Py_None) {
    /*prefix is often shared accross ns state so we cannot test it here, must be tested with intern
      enabled
      if (!TestRefCount(tester,(PyObject *)elem->prefix,1, "prefix")) return 0;*/
    if (!TestRefCount(tester,(PyObject *)elem->nodeName,1, "nodeName")) return 0;
    if (!TestRefCount(tester,(PyObject *)elem->localName,1, "localName")) return 0;
  } else {
    if (!TestRefCount(tester,(PyObject *)elem->nodeName,2, "nodeName")) return 0;
    if (!TestRefCount(tester,(PyObject *)elem->localName,2, "localName")) return 0;
  }
#else
  /*Add our strings */
  PyObject_CallMethod(tester,"startTest","s","Add Intern Counters");
  AddInternCtr(elem->namespaceURI,internDict);
  AddInternCtr((PyObject *)elem->nodeName,internDict);
  AddInternCtr((PyObject *)elem->localName,internDict);
  AddInternCtr((PyObject *)elem->prefix,internDict);
  PyObject_CallMethod(tester,"testDone","");
#endif

  /*Our ref count is our number of children + number of attrs + 1 for our parent + base */
  PyObject_CallMethod(tester,"startTest","s","Element Child Node Ref Count");
  if (!PyObject_CallMethod(tester,"compare","ll",PyList_GET_SIZE(elem->childNodes) + numAttrs + 1 + base,elem->ob_refcnt)) {
    return 0;
  }
  PyObject_CallMethod(tester,"testDone","");

  return 1;
}


PyTypeObject PyDomletteElement_Type = {
    PyObject_HEAD_INIT(NULL)
    0,                              /*ob_size*/
    "cDomlette.Element",            /*tp_name*/
    GC_TP_BASICSIZE(PyElementObject),        /*tp_basicsize*/
    0,                              /*tp_itemsize*/
    /**** methods ****/
    (destructor)element_dealloc,    /*tp_dealloc*/
    (printfunc)0,       /*tp_print*/
    (getattrfunc)element_getattr,   /*tp_getattr*/
    (setattrfunc)element_setattr,   /*tp_setattr*/
    (cmpfunc)0,                     /*tp_compare*/
    (reprfunc)element_repr,         /*tp_repr*/
    0,                              /*tp_as_number*/
    0,                              /*tp_as_sequence*/
    0,                              /*tp_as_mapping*/
    (hashfunc)0,                    /*tp_hash*/
    (ternaryfunc)0,                 /*tp_call*/
    (reprfunc)0,                    /*tp_str*/
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_HAVE_GC,
    0,                          /* tp_doc */
    (traverseproc)element_traverse,  /* tp_traverse */
    (inquiry)element_clear,          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
};

int DomletteElement_Init(void)
{
  PyDomletteElement_Type.ob_type = &PyType_Type;

  my_node_type = PyInt_FromLong(ELEMENT_NODE);
  if (my_node_type == NULL) return 0;

  return 1;
}

void DomletteElement_Fini(void)
{
  Py_DECREF(my_node_type);
}
