#include "domlette.h"

static PyObject *my_node_type;

/*
  Internal Functions
*/
PyProcessingInstructionObject *ProcessingInstruction_CloneNode(
  PyObject *node, int deep, PyNodeObject *newOwnerDocument)
{
  long tempId;
  PyObject *nodeValue, *target;
  PyProcessingInstructionObject *newNode;

  if (!PyDocument_Check(newOwnerDocument)) {
    PyErr_SetString(PyExc_TypeError, "newOwnerDocument must be a cDocument");
    return NULL;
  }

  nodeValue = PyObject_GetAttrString(node, "nodeValue");
  nodeValue = DOMString_FromObjectInplace(nodeValue);
  target = PyObject_GetAttrString(node, "target");
  target = DOMString_FromObjectInplace(target);
  if (nodeValue == NULL || target == NULL) {
    Py_XDECREF(nodeValue);
    Py_XDECREF(target);
    return NULL;
  }

  newNode = Document_CreateProcessingInstruction(
    (PyDocumentObject *)newOwnerDocument, target, nodeValue, &tempId);

  Py_DECREF(target);
  Py_DECREF(nodeValue);
  return newNode;
}


/*
  External Functions

*/



static struct PyMethodDef Pi_methods[] = { NODE_METHODS,
  {NULL,     NULL}      /* sentinel */
};





/*
Type functions
*/

static PyObject *pi_getattr(PyProcessingInstructionObject *self, char *name)
{
  PyObject *rt = NULL;

  if (!strcmp(name, "data")) {
    rt = (PyObject *)self->nodeValue;
  }
  else if (!strcmp(name, "target")) {
    rt = (PyObject *)self->nodeName;
  }
  else if (!strcmp(name, "nodeValue")) {
    rt = (PyObject *)self->nodeValue;
  }
  else if (!strcmp(name, "nodeName")) {
    rt = (PyObject *)self->nodeName;
  }
  else if (!strcmp(name, "nodeType")) {
    rt = my_node_type;
  }

  if (rt) {
    Py_INCREF(rt);
    return rt;
  }

  return node_getattr((PyNodeObject *) self, name, Pi_methods);
}


static void pi_dealloc(PyProcessingInstructionObject *self)
{
  PyObject_GC_UnTrack((PyObject *)self);

  Py_XDECREF(self->nodeName);
  self->nodeName = NULL;

  Py_XDECREF(self->nodeValue);
  self->nodeValue = NULL;

  Node_Del(self);
}

static int pi_clear(PyProcessingInstructionObject *self)
{
  /*Called when we need to break cycles*/
  return node_clear((PyNodeObject *)self);
}

static int pi_traverse(PyProcessingInstructionObject *self, visitproc visit, void *arg)
{
  return node_traverse((PyNodeObject *)self,visit,arg);
}


static PyObject *pi_repr(PyProcessingInstructionObject *pi)
{
  char buf[256];
  PyObject *data = PyObject_Repr((PyObject *)(pi->nodeValue));
  PyObject *target = PyObject_Repr((PyObject *)(pi->nodeName));

  if (!data) {
    return NULL;
  }
  if (!target) {
    Py_DECREF(data);
    return NULL;
  }

  sprintf(buf, "<cProcessingInstruction at %p: target %s, data %s>",
          pi, PyString_AS_STRING(target), PyString_AS_STRING(data));
  Py_DECREF(target);
  Py_DECREF(data);
  return PyString_FromString(buf);
}

int pi_test_ref_counts(PyObject *tester,PyProcessingInstructionObject *node,long *childCtr,PyObject *internDict,int base) {
  /*Ref count is base + 1.  Out nodeValue and target should have a ref count of 1 as they not interned*/
  PyObject_CallMethod(tester,"startTest","s","Node");
  if (!PyObject_CallMethod(tester,"compare","ll",base+1,node->ob_refcnt)) return 0;
  PyObject_CallMethod(tester,"testDone","");
#ifdef DISABLE_INTERN
  if (!TestRefCount(tester,(PyObject *)node->nodeValue,1, "data")) return 0;
  if (!TestRefCount(tester,(PyObject *)node->nodeName,1, "target")) return 0;
#else
#endif
  return 1;
}


static int pi_setattr(PyProcessingInstructionObject *self, char *name, PyObject *v)
{
  /* Set attribute 'name' to value 'v'. v==NULL means delete */
  if (v == NULL) {
    PyErr_Format(PyExc_AttributeError,
                 "Cannot delete attribute '%.400s' on '%.50s' object",
                 name, self->ob_type->tp_name);
    return -1;
  }

  if (!strcmp(name, "data") || !strcmp(name, "nodeValue")) {
    PyObject *nodeValue = DOMString_ConvertArgument(v, name, 0);
    if (nodeValue == NULL) return -1;

    Py_DECREF(PyProcessingInstruction_DATA(self));
    PyProcessingInstruction_DATA(self) = nodeValue;
  } else {
    PyErr_Format(PyExc_AttributeError,
                 "Cannot set attribute '%.400s' on '%.50s' object",
                 name, self->ob_type->tp_name);
    return -1;
  }

  return 0;
}


PyTypeObject PyDomletteProcessingInstruction_Type = {
    PyObject_HEAD_INIT(0)
    0,
    "cDomlette.ProcessingInstruction",
    GC_TP_BASICSIZE(PyProcessingInstructionObject),
    0,
    (destructor)pi_dealloc, /*tp_dealloc*/
    (printfunc)0,    /*tp_print*/
    (getattrfunc)pi_getattr,    /*tp_getattr*/
    (setattrfunc)pi_setattr,    /*tp_setattr*/
    0,                          /*tp_compare*/
    (reprfunc)pi_repr,          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,              /*tp_as_sequence*/
    0,              /*tp_as_mapping*/
    0,                             /*tp_hash*/
    0,          /*tp_call*/
    0,          /*tp_str*/
    0,                      /*tp_getattro*/
    0,          /*tp_setattro*/
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_HAVE_GC,
    0,                          /* tp_doc */
    (traverseproc)pi_traverse,  /* tp_traverse */
    (inquiry)pi_clear,          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
};

int DomletteProcessingInstruction_Init(void)
{
  PyDomletteProcessingInstruction_Type.ob_type = &PyType_Type;

  my_node_type = PyInt_FromLong(PROCESSING_INSTRUCTION_NODE);
  if (my_node_type == NULL) return 0;

  return 1;
}

void DomletteProcessingInstruction_Fini(void)
{
  Py_DECREF(my_node_type);
}
