#!/usr/bin/env python
# file: test_pandaseq.py

# Tests for the pandaseq.py application controller.
# https://github.com/neufeld/pandaseq

from cogent.util.unit_test import TestCase, main
from cogent.util.misc import create_dir
from cogent.app.pandaseq import PandaSeq, join_paired_end_reads_pandaseq
import os 
from subprocess import Popen, PIPE, STDOUT
import shutil 
import gzip

__author__ = "Michael Robeson"
__copyright__ = "Copyright 2007-2013, The Cogent Project"
__credits__ = ["Michael Robeson"]
__license__ = "GPL"
__version__ = "1.9"
__maintainer__ = "Michael Robeson"
__email__ = "robesonms@ornl.gov"
__status__ = "Development"


class PandaSeqTests(TestCase):
    """Tests for PandaSeq application controller"""

    def setUp(self):
        """General setup for PandaSeq tests """
        # make directory test
        self.temp_dir_string = '/tmp/test_for_pandaseq'
        create_dir(self.temp_dir_string)

        # make directory with spaces test
        self.temp_dir_string_spaces = '/tmp/test for pandaseq'
        create_dir(self.temp_dir_string_spaces)

        # temp file paths
        self.test_fn1 = os.path.join(self.temp_dir_string,'reads1.fastq')
        self.test_fn1_space = os.path.join(self.temp_dir_string_spaces, 
                                        'reads1.fastq')
        self.test_fn2 = os.path.join(self.temp_dir_string,'reads2.fastq')
        self.test_fn2_space = os.path.join(self.temp_dir_string_spaces,
                                        'reads2.fastq')

    def writeTmpFastq(self, fw_reads_path, rev_reads_path):
        """write forward and reverse reads data to temp fastq files"""
        try:
            fq1 = open(fw_reads_path, "w+")
            fq1.write(reads1_string)
            fq1.close()
            fq2 = open(rev_reads_path, "w+")
            fq2.write(reads2_string)
            fq2.close()
        except OSError:
            pass

    def test_check_version(self):
        """ Set up some objects / data for use by tests"""

        # Check if pandaseq version is supported for this test
        accepted_version = (2,4)
        command = "pandaseq -v"
        version_cmd = Popen(command, shell=True, universal_newlines=True,
                            stdout=PIPE,stderr=STDOUT)
        stdout = version_cmd.stdout.read()
        #print stdout
        version_string = stdout.strip().split()[1]
        #print version_string
        try:
            version = tuple(map(int, version_string.split('.')))
            #print version
            pass_test = version == accepted_version
        except ValueError:
            pass_test = False
            version_string = stdout
        self.assertTrue(pass_test,
                        "Unsupported pandaseq version. %s is required, but running %s." 
                        %('.'.join(map(str, accepted_version)), version_string))

    def test_changing_working_dir(self):
        """WorkingDir should change properly.""" 
        c = PandaSeq(WorkingDir=self.temp_dir_string)
        self.assertEqual(c.BaseCommand,
                         ''.join(['cd "', self.temp_dir_string, '/"; ',
                          'pandaseq']))

        c = PandaSeq()
        c.WorkingDir = self.temp_dir_string + '2'
        self.assertEqual(c.BaseCommand,
                         ''.join(['cd "', self.temp_dir_string + '2', '/"; ',
                         'pandaseq']))

    def test_base_command(self):
        """pandaseq command should return correct BaseCommand"""
        c = PandaSeq()
        # test base command
        self.assertEqual(c.BaseCommand,
                         ''.join(['cd "', os.getcwd(), '/"; ', 'pandaseq']))
        # test turning on parameter
        c.Parameters['-o'].on('15')
        self.assertEqual(c.BaseCommand,
                         ''.join(['cd "', os.getcwd(), '/"; ', 'pandaseq -o 15']))

    def test_pandaseq_assembly(self):
        """ Runs PandaSeq with recomended default and alternate settings.

        Checks output of assembled and unassembled paired-ends.
        """

        # write temp files
        self.writeTmpFastq(self.test_fn1, self.test_fn2)

        ### Run with recomended default function params ##
        params = {}
        params['-f'] = self.test_fn1
        params['-r'] = self.test_fn2
        
        pandaseq_app = PandaSeq(params=params,
                                WorkingDir=self.temp_dir_string)

        pandaseq_app.Parameters['-F'].on()

        res = pandaseq_app([self.test_fn1, self.test_fn2])

        # assembly is sent to StdOut, check output
        self.assertEqual(res['StdOut'].read(), expected_default_assembly)
        
        res.cleanUp()

        ### Run with altered params ###
        # run with out -F option (output is FASTA format)
        params2 = {}
        params2['-f'] = self.test_fn1
        params2['-r'] = self.test_fn2
        
        pandaseq_app2 = PandaSeq(params=params2,
                                 WorkingDir=self.temp_dir_string)
        
        res2 = pandaseq_app2([self.test_fn1, self.test_fn2])

        # assembly is sent to StdOut, check output
        self.assertEqual(res2['StdOut'].read(), expected_default_assembly_fasta)
        
        res2.cleanUp()
        shutil.rmtree(self.temp_dir_string)


    def test_join_paired_end_reads_pandaseq(self):
        """join_paired_end_reads_pandaseq: should work as expected"""
        # write temp files
        self.writeTmpFastq(self.test_fn1, self.test_fn2)

        ### run with recomended defaults ###
        jp = join_paired_end_reads_pandaseq(self.test_fn1, self.test_fn2,
                          working_dir=self.temp_dir_string)
        
        jp_ends = open(jp['Assembled'],'U').read()
        self.assertEqual(jp_ends, expected_default_assembly)
        
        ### run with fastq (-F option) turned off ##
        jp2 = join_paired_end_reads_pandaseq(self.test_fn1, self.test_fn2,
                           fastq=False, 
                           working_dir=self.temp_dir_string) 
        
        jp_ends2 = open(jp2['Assembled'],'U').read()
        self.assertEqual(jp_ends2, expected_default_assembly_fasta)
        
        os.remove(self.test_fn1)
        os.remove(self.test_fn2)
        shutil.rmtree(self.temp_dir_string)

       


# Test Strings:

reads1_string ="""@MISEQ03:64:000000000-A2H3D:1:1101:14358:1530 1:N:0:TCCACAGGAGT
TNCAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGAAACTGACACTGAGGGGCGAAAGCGGGGGGGGCAAACG
+
?#5<????DDDDDDDDEEEEFFHHHHHHHHHHHHHHDCCHHFGDEHEH>CCE5AEEHHHHHHHHHHHHHHHHHFFFFHHHHHHEEADEEEEEEEEEEEEEEEEEEEEEEE?BEEEEEEEEEEEAEEEE0?A:?EE)8;)0ACEEECECCECAACEE?>)8CCC?CCA8?88ACC*A*::A??:0?C?.?0:?8884>'.''..'0?8C?C**0:0::?ECEE?############################
@MISEQ03:64:000000000-A2H3D:1:1101:14206:1564 1:N:0:TCCACAGGAGT
TACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTTTGTAAGTCAGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCGTTTGAAACTACAAGGCTAGAGTGTAGCAGAGGGGGGTAGAATTCCACGTGTAGCGGTGAAATGCGTAGAGATGGGGAGGAATACCAATGGCGAAGGCAGCCCCCGGGGTTAACACTGACGCCAAGGCACGAAAGCGGGGGGGGCAAACG
+
?????BB?DDDDDD@DDCEEFFH>EEFHHHHHHGHHHCEEFFDC5EECCCCCCDECEHF;?DFDDFHDDBBDF?CFDCCFEA@@::;EEEEEEEECBA,BBE?:>AA?CA*:**0:??A:8*:*0*0**0*:?CE?DD'..0????:*:?*0?EC*'.)4.?A***00)'.00*0*08)8??8*0:CEE*0:082.4;**?AEAA?#############################################
@MISEQ03:64:000000000-A2H3D:1:1101:14943:1619 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGGGGCGAAGGCGACCACCTGGACTGAAACTGACACTGAGGGGCGAAAGCGGGGGGGGCAAAAG
+
?AAAABBADDDEDEDDGGGGGGIIHHHIIIIIIIHHHCCHHFFEFHHHDCDH5CFHIIHIIIIHHHHHHHHHHHHHHHHHHHGGGEGGGGDDEGEGGGGGGGGGGGGGGEEEGCCGGGGGGEGCEEEECE?ECGE.84.8CEEGGGE:CCCC0:?C<8.48CC:C??.8.8?C:*:0:*9)??CCEE**)0'''42<2C8'8..8801**0*.1*1?:?EEEC?###########################
@MISEQ03:64:000000000-A2H3D:1:1101:15764:1660 1:N:0:TCCACAGGAGT
TACGAAGGGGGCTAGCGTTGCTCGGAATCACTGGGCGTAAAGCGCACGTAGGCGGATTGTTAAGTCAGGGGTGAAATCCTGGAGCTCAACTCCAGAACTGCCTTTGATACTGGCGATCTTGAGTCCGGGAGAGGTGAGTGGAACTGCGAGTGTAGAGGTGAAATTCGTAGATATTCGCAAGAACACCAGTGGCGAAGGCGGCTCACTGGCCCGGAACTGACGCTGAGGGGCGAAAGCTGGGGGAGCAAACG
+
???????@DDDDDDBDFEEFEFHEHHHHHHHHHHHHHEHHHHFEHHHHAEFHGEHAHHHHHHHHHHHHHHH=@FEFEEFEFEDAEEEFFE=CEBCFFFCECEFEFFFCEEEFFCD>>FFFEFF*?EED;?8AEE08*A*1)?E::???;>2?*01::A?EEEFEEE?:C.8:CE?:?8EE8AECEFE?C0::8'488DE>882)*1?A*8A########################################
@MISEQ03:64:000000000-A2H3D:1:1101:15211:1752 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGGGGTGGAATTTCCTGTGTAGCGGGGAAATGCGTAGATATGGGAAGGAACACCAGGGGCGAAGGCGACCACCTGGACTGATACTGACACTGGGGTGGGAAAGGGGGGGGAGGAAAAG
+
?????<B?DBBDDDBACCEEFFHFHHHHHHHHHHHHH5>EFFFEAACEC7>E5AFEFHHHHHHF=GHFGHFHHHHFHFHH;CED8@DDDE=4@EEEEECE=CECECEECCBB34,=CAB,40:?EEEE:?AAAE8'.4'..8*:AEEECCCA::A################################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:15201:1774 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGGGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGGGGGGGGCAAACG
+
?????BB?DDDDDDBDEEEEFFHFHHHHHHHHHHHFH>CEHDGDDCDE5CCEACFHHHHHHHHFFHHHHHHHHFHHFHHHHHHDEBFEEE@DEEEEEEEEEEEEEEBBCBECEEEEEEEEEEEEEEE?ACCEEEA)84)0.?EEE:AEACA?0?CEDD'.4?A:ACA)0'80:A:?*0*0)8CEEEEE?*0*)88888A'.5;2A)*0000*8:*0:?CEEEE?A*?A#######################
@MISEQ03:64:000000000-A2H3D:1:1101:15976:1791 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTCGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGGGAAATGCGTAGATATGGGAAGGAACACCGGGGGGGAGGGGGGCTCTCGGGTCCTTTTCGGCGGCTGGGGGCGGAAGGCAGGGGGGGCAACCG
+
?????BB?DDDDDDDDEEEEFFIFHHHHHHIIIHIFHCCHF@F>CECHCDDECCFEADEHHHHHHHHFGHHHHHHFHHHHHHF8:<DEEEADEEFFFFFFABEFFEFFECBCEEFEFFFFEACEEEEE*10*1??.08.888AEF?EEEC1:1:??>>'88AC?::?AA##################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:16031:1840 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGGGAAATGCGTAGATATAGGAAGGAACACCAGGGGCGAAGGCGACCACCTGGACGGATACTGACACTGAGGGGCGAAAGGGTGGGGAGAAAAAG
+
?????BB?DDDDDDDDGFEGGGIHHHHIIIIIIIHFE>CFFFFDCHCH>>CE-5EEIIHHHIHHHHHHHHHHGHFDFHFHEHGBEEEEGGEDGGGGEGGEGGGGGCEGCCEEGGG><CEECCGCEEEG?C:1?EG.8<.88CCCEEGE?C?C*:1:<>'.8?8:C:?00.0?:?*1::*9CC?EEEG*?##############################################################
@MISEQ03:64:000000000-A2H3D:1:1101:12964:1852 1:N:0:TCCACAGGAGT
TNCAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGCAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGAAGCGGGGAAATGCGTAGATATAGGAAGGAACACCAGGGGCGAAGGCAACCACCGGGACTGAAACTGAACCGGAGGGGGGAAAGCGGGGGGGGAAAACG
+
?#55????DDDDDDDDEEEEFFHHBHHHFFGHHFHDC+5>C/?E7DCHCCCD5AECFHHHFHHHHHHHHHFFFFFHFFDFEFF5)@=DEFDEFEEFF;AEAABC,4BECCC=B,5?C8?CC?CC*:?E010:?EA)0.)08C?A:?A########################################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:17245:1906 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAGGGAACACAAGGGGCGAAGGCGACCACCGGGACGGAAACTGCAACTGGGGGGGGAAAGCGGGGGGGGAAACAG
+
AAA??BB?DDDDDDDDGGEGGGIHGHIIIIIIHF?CFCCDFFFDCHEHC>DH5AFEHIHHHHHHHHHHHHHHFFFFFHHHHHGDBEEGGGGGGG@EGEGGGGGGGCGEGACC>EGEGGGGC:C0CEEG:0::CEE)88)08?:CECCE:C*10*104A.4CE:*:?C8)'8CC##############################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:18171:1983 1:N:0:TCCACAGGAGT
GACGTAGGAGGCGAGCGTTGTCCGGATTCATTGGGCGTAAAGAGCGCGCAGGCGGCTTGGTAAGTCGGATGTGAAATCCCGAGGCTCAACCTCGGGTCTGCATCCGATACTGCCCGGCTAGAGGTAGGTAGGGGAGATCGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAGGAACACCGGGGGCGAAGGCGGATCTCTGGGCCTTCCCTGACGCTCAGGCGCGAAAGCGGGGGGGGCGAACG
+
??????B?DDDDDDDDFFEFFFIHFEEEHHIHHHFHHEHHFGFFFHCEHEHCDECCEFFE4DDFDBEEEEEFFFFEEFFCE8B>BEFEEFFCEFE>8>EFFE*A?A?ADDAAEE8E>;>EA:??1*:?111?C<88AA08?ACECF:*:?*08:0:8<.4?EE*A:))'..0*01*?:08?A*?CA#################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:14225:2009 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGAAACGGACACTGAGGGGCGAAAGCGGGGGGGGCAAACG
+
?????BB?DDDDDDBDEEEEFFHHHHIIIIHHIIIIHHEHIFGEHHHHCCEHAEFHIIHIIIIHHHHHHHHHHFHHHHHHHHFFFEFFFFFEFFFFFFEFFFFFFEFFFEFCACEFFFFFF:C?CEEE*?:AAEE88;088?AEFCEAEECEEEFE>?).?ECCEEE8?4AFFE0?*0088ACFFFAAC*0?C888>>CD?D;8CE*:*:A?CF*::)0?DD?:::?########################
@MISEQ03:64:000000000-A2H3D:1:1101:16656:2052 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGGGGCGAAGGCGACCACCGGGACTGAAACTGACACTGAGGGGGGAAAGCGGGGGGGGAAAACA
+
?????BB?BDDDDDDDGFFEGGIIHHIIIIIHHHHIHCCFFDEEEHEHFFEH5AFHHIHIHIHGGHHHHHHHFHHFHHHHHHGEG@EGEGGEGGGGCEGGEGGGGEGGACECGGGGGGGGEGGCCEGG?CCCEGC088)0.?EGG?EC*::C*:??<8.48?C:?C808.8CEE#############################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:18209:2060 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGTAGGCGGCTCAGCAAGTCAGGGGTGAAATCCCGGGGCTCAACCCCGGAACTGCCCTTGAAACTGCTAAGCTAGAATCTTGGAGAGGCGAGTGGAATTCCGAGTGTAGAGGGGAAATTCGTAGATATTCGGAAGAACACCAGGGGCGAAGGCGACCCCCTGGACAAGCATTGACGCTGAGGGGGGAAAGCGGGGGGGGCAAAAG
+
?????BB?BDDDDDDDECEEFFHHHHAHFHHHHHHHHCCHHH=@DEEHFHFCGHHB)?ECGHHH?DHHHHHCCCFFHHHFEEEEEEEEEEEEEB)>EDACEECEECEEECEE:*0A:AEAECA:0::ACE??E?.8'4.88?EC*00:08).0:*00?)..8AAAAA*0)0::?::?0A8)?C:?A#################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:13824:2095 1:N:0:TCCACAGGAGT
TACGTAGGGGGCTAGCGTTGTCCGGAATCATTGGGCGTAAAGCGCGTGTAGGCGGCCCGGTAAGTCCGCTGTGAAAGTCGGGGGCTCAACCCTCGAAAGCCGGGGGATACTGTCGGGCTAGAGTACGGAAGAGGCGAGTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAGGAACACCCATTGCGAAGGCAGCTCGCTGGGACGTTACTGAGGCTGAGACCGGAAAGGGGGGGGGGCAAAAG
+
??A??BBADDDDDDBDFEEFFFHHHHHFHHHIHHFHHCCHHFHDCCDEEHHFIHAHHHHH@EFFDFFEBDEDEFFECBBEEEED?28CCFFECE;EF8?ECD;'488?EEFCE:A>>?>EECEE::A8E8.8?8).'.'08AEE*?:*::*001:?<D.'8??*:*))'''01***10*088CEEEEA8C#############################################################
@MISEQ03:64:000000000-A2H3D:1:1101:17839:2106 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACAACCGGGACGGAAACTGACACTGAGGGGCGAAAGCGGGGGGGGCAAAAG
+
AAA?ABB?DDDDDDDEGFEGGFHIHHIIIIIIIIDFH5CFHHGHEH=DC>CE5AEEHFHIHIFHHHHHHHHHFHHFHHHHHHGGGGGEEGGGGGDEGGGGGGGGGGGGGCE>AEGEGGGGEEECEGEE1:??CEC08>.88CEEECG*:C?CC:?0.4.4CE?CECC?)4?CC:*11?:?)CCEGG).9*1:?8<2<<C####################################################
@MISEQ03:64:000000000-A2H3D:1:1101:17831:2124 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGGGGGGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGAAACTGACACTGAGGGGCGAAAGCGGGGGGAGCAAACG
+
AAAAABB?DDDDDDDDGFGFGGIIHHIIIIIHIIDFH>CFHHGDCFDH>CDHAEFEHIEFFGGHHHHHHHFH=CFFHHHHEHG8DEEGEGGGGGDEEEEGEEGGGCGGEEECCACCEGGGCEE::?CE0?CCEGE'.<'..4CEGEGGEEEE*::C>20>?C?*1:C..'8:??*:*?*0)??9CEG8?*1*8'4.44?58<28?C#############################################
@MISEQ03:64:000000000-A2H3D:1:1101:12555:2129 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTCGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACGAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACGGAAACTGACACTGAGGTGCGAAAGCGTGGGGACCAACCG
+
????ABBADDDDDEDDGGGGGGIIHHIIIIIHIIHHHFFHHHFHHHHH>CDHAEFHFGHHFHHHHHHHHHFHHFHFFHHHHHGBEEAD;DGGGGEGGGCGCEEEGEGGGCE>>>CEDGDGE:C:CGGG:?C??EE08<)0?ECEGEGCCECEEGGGGG08CECE?CE8)4?CC:*:*:0989*9CEC8C*:?C'842.8'.4.2?E9?*:?'.8).::::?CC:*110*0C8C<8??C#############
@MISEQ03:64:000000000-A2H3D:1:1101:13627:2170 1:N:0:TCCACAGGAGT
GACAGAGGGTGCAAACGTTGTTCGGAATTACTGGGCATAAAGAGCACGTAGGTGGTCTGCTAAGTCACGTGTGAAATCCCCCGGCTCAACCGGGGAATTGCGCGTGATACTGGCCGGCTCGAGGTGGGTAGGGGGGAGCGGAACTCCAGGGGGAGCGGGGAAATGCGTAGATATCTGGAGGAACACCGGGGGCGAAAGCGGCTCACGGGACCCAATCTGACACTGAGGGGCGAAAGCTAGGGTGGCAAACG
+
?????BB?DDDDDDDDEFFFFFHHHHHIHIIHIIFHCEHIIHBFHIHHAAFH5CF@FHHHGHIIGHHHHFHIHIIIHIIIHHHHHHHHHHFHHHFFEFEFEDBE<>BBEEFECECE?D'..2AD)8A>40?AED''''.4<D>>AC**1?).2'888D'5<EACEEEAEDEFEE:*??*08A?AAC)58'4>2<>D8A:A82'.*:*.'?>E)AA####################################
@MISEQ03:64:000000000-A2H3D:1:1101:11781:2223 1:N:0:TCCACAGGAGT
TACGTAGGGCGCAAGCGTTATCCGGAATTATTGGGCGTAAAGAGCTCGTAGGCGGTTTGTCGCGTCTGCCGTGAAAGTCCGGGGCTCAACTCCGGATCTGCGGTGGGTACGGGCAGACTAGAGTGATGTAGGGGAGATTGGAATTCCTGGTGTAGCGGGGAAATGCGCAGATATCAGGAGGAACACCGATGGCGAAGGCAGGTCTCTGGGCATTAACTGACGCTGAGGAGCGAAAGCAGGGGGGGCGAACG
+
???A?BB?DDDDDDDDEEECFFHHHHHIHHHIIIHHHECEHFCFHGGH>CFEFEHHHHHFFDFHCDEFFHHEBFFFF?BBEEEEEEEFFFBEEEEAEDEFEDD.8A8.ACEDDD;AEFFFFEF:*1:?ACCFFD8<AE?EFFFF:EEEEFFFA:CEDD'.8??CEF?ADDDFF:C:?::?AEEFFFA>8'08:2448DE?E?8:*:*1A***0*:AA*?AEEEEE?#########################
@MISEQ03:64:000000000-A2H3D:1:1101:17996:2254 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGGGAAATGCGTAGATATAGGAAGGAACACCAGGGGCGAAGGCGACCACCGGGCCTGAAACTGACACTGAGGGGGGAAAGCGGGGGGGGAAAACG
+
?????BB?DDDDDDDDGGGGGGIHHHHIIIIHHHFFH>CHFHGHHHEHCCCE5AFEHIHHHHHHHHHHHHHHHHHHHHHHHHGGEEGGEGEGGGEGEGGGCGGGGGGGECGEECGAECGGEEEC**CE?C::CCC.8<)08?CCC:CCCEC?CC?:8>'4<?.1C:8082CCGG*:*:0C8?EC*0C89.?############################################################
@MISEQ03:64:000000000-A2H3D:1:1101:13712:2276 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGGGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGGGGGAAAGCGGGGGGGGAAAACG
+
?????BBADDDDDDDDGGEGGGIHHHHIIIIIIIIIHCCHHDCECHEHCDEH-AFEHIHIHHIHHHHHHHHHHHFFFHHHHHGEGEDDEEDDDGGGGGEGGGGGEEEGEEGEGGGGGGGCEGEGCEGG:C::CEE)88.8?EGGG:C?:?:C??:*52'.888:CEE).2CCGE*C??:C.?EGGGGC9*118>>.4>C''.8<CC*?*:**00*01?:CEGCC###########################
@MISEQ03:64:000000000-A2H3D:1:1101:15819:2301 1:N:0:TCCACAGGAGT
TACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTGAGTTAAGTCTGCTGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGGGGAATTTCCTGTGTAGCGGGGAAATGCGTAGATATAGGAAGGAACACCAGGGGCGAAGGCGACCACCTGGACTGAAACTGACACTGAGGGGCGAAAGCGGGGGGGGCAAACG
+
?????BB?DEDDDDDDFDEEFFHHHEFHHIIIIIIHHCHHFHHHCEEACCHHHEH)<<CCDGFFDFFFBCB@DFHFHFHHEEFB8@EEEFFEEFFFFFFFFFEFCEFFFCAAC?EF??AC???0*:?C*:::?EE)0>'.42AAECEFE:*0:AAC?D'..8C?:?A)).0001*11::??8A**?A################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:11958:2304 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGGGGAATTTCCTGTGTAGCGGGGAAATGCGTAGATATAGGAAGGAACACCAGGGGCGAAGGCGACCACCGGGACTGAAACTGACACTGAGGGGCGAAAGCGTGGGGGGCAAACG
+
????ABBADDDDDDDDEEEEFFHHHHHIFHHIIIHFEECEFGDECECE5>C:55EEHIHIFGHFGHHHHHFHFFHHFHHHHHFBFEEDEEFFFFEFFFEFEFABEEFFFEEBEFFEFF=::AE*:AEE0?:?CFE8A>'.<?EEE??E?A??CEEF<>'.8AC?ECE)848?0**::AAC???EEE)*0)084'48<'8'882<CA).2<408?*1:??EEE#############################
@MISEQ03:64:000000000-A2H3D:1:1101:19110:2311 1:N:0:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGGGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTAAAACTGACACTGAGGGGCGAAAGCGGGGGGGGCAAACG
+
????9BB?BBBBBBBBEEECFAFHHHHHHHFHHHHHHCCEFF/CDCCE>DCH5AECFHHHFHHHHHHHHHHHGFHHCFHHHHHEEEDEDEED@EEEEEEEEEEEEEEEEE;;BEEE?EEEEEE?*?CA?EE::?8'.''..?CEE*::/:?A:C?E??82?CCEEEE))4?EEEEA:?*80?AEEC#################################################################
""" 

reads2_string = """@MISEQ03:64:000000000-A2H3D:1:1101:14358:1530 2:N:0:TCCACAGGAGT
ACGGACTACAAGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGATTTCACATCCAACTTAACAACCCACATACCCGCCTTTTCGCCCAGGTAATCC
+
?????@@BDDBDDDBBCFFCFFHIIIIIIIIFGHHHHEHHHIIIHHHHHFHIIHIGHHIDGGHHHHIIIIICEFHIHHCDEHHHHHHFHHCFHDF?FHHFHHHFFDFFFDEDDD..=DDDE@<BFEEFCFFCECE==CACFE?*0:*CCAA?:*:*:0*A?A80:???A?*00:**0*1*:C??C?A?01*0?);>>'.8::A?###############################################
@MISEQ03:64:000000000-A2H3D:1:1101:14206:1564 2:N:0:TCCACAGGAGT
ACGGACTACAGGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGTGCATGAGCGTCAGTGTTAACCCAGGGGGCTGCCTTCGCCATTGGTATTCCTCCACATCTCTACGCATTTCACTGCTACACGTGGAATTCTACCCCCCTCTGCTACACTCTAGCCTTGTAGTTTCAAACGCAGTTCCCAGGTTGAGCCCGGGGCTTTCACATCTGCCTTACAAAACCGCCTGCGCACGCTTTACGCCCCGTAATTC
+
?????@@BDDBDDD?@CFFFFFHHHHHFFHHHHHHHHHHH@FFHEDFFH@FHBGCDHHHBFHHHHHHHEHHHHDCCEFFDFFFEE:=?FF?DFDFDFFF==BEE=DBDDEEEEEB,4??EE@EEE,3,3*3,?:?*0ACCEDD88:***?*0:*0***0*?C?00:AE:?EE:*A8'.?:CAA?A80*0*??AA88;28;C##################################################
@MISEQ03:64:000000000-A2H3D:1:1101:14943:1619 2:N:0:TCCACAGGAGT
ACGGACTACAGGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCAAGGTTGAGCCCGGGGCTTTCACATCCAACTTACAAAACCACCTACCCGACCTTTACGCCCAGAAATTC
+
?????@@BDDDDDD?AFFFFFFIIHHIHIIHIIIIHIHHHHHHHHHHHHHHHHHHIHHHIIHHIHIIIIII?EFEGHHHHHIIHHDHHFD@FFEHFHFHFHFHFFFFFFFFEEEEFFFDEB<E@EFEEABA=B=CAEFEEFEA?A:*AC::??:**10??:00::*??EC*?:?C*::A*?C*1:8A################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:15764:1660 2:N:0:TCCACAGGAGT
ACGGACTACCGGGGTATCTAATCCTGTTTGCTCCCCTAGCTTTCGCACCTCAGCGTCAGTACCGGGCCAGTGAGCCGCCTTCGCCACTGGTGTTCTTGCGAATATCTACGAATTTCACCTCTACACTCGCAGTTCCACTCACCTCTCCCGGACTCAAGACCGCCAGTACCAAAGGCAGTTCTGGAGTTGAGCTCCAGGTTTTCACCCCTGCTTTAAAAATCCCCCAACGCGGCCTTTCCCCCCAGTGACTC
+
?????@=BB?BBBB<?>ACFFCECFCHCFHH=CGHHDFH=E?ACDEHHCCFFGHHDHH@CBEFHHCHHHF,5@?DF)4<C3D4DD+=,BD5;DE=DBDE=<E<;?E?B;3?;A?;;;EBBC:??EEEEE?;AA*:A??#################################################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:15211:1752 2:N:0:TCCACAGGAGT
ACGGACTACCGGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGATTTCCCATACAACTTAACAAACCACTTACGCGGGTTTTCGCCCCACAAATTC
+
?????@@BB-<@BBBBEA?FFFA8ACFCBHHHGHHHHBHHHHFCDDD7CHHFE?CCDDCFGBHHHGGHFGFGFFHFDCDHHC?=CDHFFDCDDDF,EFF5?BFEDBBDB@@EEACCE;,?::@BEEEEACC*??//:AA*8AAAEE?ECC#####################################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:15201:1774 2:N:0:TCCACAGGAGT
ACGGACTACCGGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTAAGCCCGGGGATTTCCAATCCAACTAAACAAAACACACACCCGCTCTTTACGCCCACCATTTC
+
?????@=BBBBBBB<=CFFFFFFHFCFCEHHDGHHEFEHHHHHHHHHHHFHHGC-AEGFCGHHHFFHFHHBFGDEDDCEDDH+DDDHHF,,7D@DFDFFFBFFEDEDEEDE:@:B?C;,3@<><EEEE*BEEC?E*0AEEEEE*8*:CCE:CA*?*A?:AAA#########################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:15976:1791 2:N:0:TCCACAGGAGT
ACGGACTACAAGGGTTTCTAATCCCGTTTGCTCCCCTAGCTTTCGCACCTCAGCGTCAGAAATGGACCAGAGAGCCGCCTTCGCCACCGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGTTTTCCAATCCAATTTAACGACCAACCACGCGGCGCTTTAGGCCCGGTAATGC
+
?<???@=BBBBBBB556CFFBCEFFEHHHHHHHHHE=EHHHHHHHHHHHHHHHFHHEDCGFHHHHHHHHH;A?EFHHHHHHHHH+=EHHC)8@+?BFFFDFEEEEE;DDEEEEBCEECEEEBEEEEEEEEEEE:?*/:ACEE)*8*:C:A*0?:A*:C?A:?**:AECE?*?:*:C:????C#####################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:16031:1840 2:N:0:TCCACAGGAGT
ACGGACTACTGGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTTAACAATACTCTAGCTTGTCAGTTTTAAATGCAGTCCCCAGGTTGAGCCCGGGGCTTTCACATCAAATTTACAAAACCACCCACGCGCGTTCTACGCCGGACAATCC
+
?????==<BBBBBBBBFFFFF?FFF?FFFHHFEFFHHHH:@>CHEDHHHFFFGBCCDDFGGHHHHEHHHHH5AE+C*>==+EDHHDEFCFCDF3@.D=,CFH=,@,4DFFE:=DDDDEB:)1:1;;?B;BE;??,?EE;AEE??**0*/:0??:***:?E*:8?A*:CEE#################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:12964:1852 2:N:0:TCCACAGGAGT
ACGGACTACTCGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCAACCACCCTCTACCATACTCTAGCTTGTAAGTTTTGAATGCAGTTCCAAGGTTGAGCCCGGGGCTTTCACACCCAACTTAACAACCCACCTACGCGGCATTTACGCCCACTACTTC
+
?????@=BBBB9?=55AACCC?ECFFFHB>FFHGFHFHHHHHHHHHHHHFHHHGGGHHHGGHHHHHHDDFEGH;EBCEHD+AFE@C+@F=.7D+@CDCFFHHFFFD?DF@E+=:BDDB;D=@BE?BCE*,33,,?,3;;:?).0**::***0/*/0A??:*:****00/**8*0?AE:?AAC**0):??C#############################################################
@MISEQ03:64:000000000-A2H3D:1:1101:17245:1906 2:N:0:TCCACAGGAGT
ACGGACTACAGGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGCTTTCAAATCCAACTTAACAAACAACCAACCGGCGCTTTACGCCCAGTAATTC
+
?????@@BDDDDBDABEFFFFFHHHHHHHHGGHGHHHHHHEH@FEHEEHHHFHHH=EGHHHHDGHHHHFHHGGHHHCCEDEHHHHHHHHHDFHHF=DBDFHFD?BB.BF;@DDDD.=DD*>6)@==ABAACBB5=B,=,88A)???:E*::::??*:**1**8??CCCEE8A:A::AAACAC??A)1:0**1*)48;'42A?EA**1?*1*0::??:ACEF##############################
@MISEQ03:64:000000000-A2H3D:1:1101:18171:1983 2:N:0:TCCACAGGAGT
ACGGACTACCGGGGTTTCTAATCCTGTTCGCTCCCCACGCTTTCGCGCCTGAGCGTCAGGTAAGGCCCAGAGATCCGCCTTCGCCACCGGTGTTCCTCCTGATATCTGCGCATTTCACCGCTACACCAGGAATTCCGATCTCCCCTACATACCTCTAGCCGGGCAGTATCGGATGCAGACCCGAGGTTGAGCCCCGGGATTTCACATCGGCTTACCAAAGCGCCCGGCGCCCCCTTTACGCCCCAGAAACC
+
?????@@BDBDDDD=BCFFFFFIIIIHHFEHHHHIHIHHHEFCGDEHHHEFFEGC>EEHI?EHHGHHHHFH+C=,,?FHDDHFEE@EFE=1;A;EECCE==BEB,BBC=@@<?EE:?E:8>24;:CEAA8?CC*??:0?;*1?AE?CE*10:0*1:CAA*;22;2A#####################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:14225:2009 2:N:0:TCCACAGGAGT
ACGGACTACAGGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGCTTTCACATCCAACTTAACAAACAACCACGCCGCGCTTTAGCCCAGGTAATTC
+
?????@@BDDDDDD??CFFFFFHIHHHHIIIIHIIHIHHHHHIHHHHHHFFHHIHHHIHHHHHIIHIHIIIFFFEGHHEDEHHHHDHHHHCFFDFFHHHHHHFFFFFFF@EDEED=DDEED@EBFCEEEFECCCEEEFB<CA888:AEEFEFEA??CCEFF:?:ACCFFE?E:AC?:*:?EFE*:):???::A).;D>D>8:?################################################
@MISEQ03:64:000000000-A2H3D:1:1101:16656:2052 2:N:0:TCCACAGGAGT
ACGGACTACCCGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCCCCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCAAGGTTGAGCCCGGGGCTTTCACACCCAACTTAACAAACCACCTACGCGCGCTTTACGCCCAGCATTTC
+
?????@@BDDDDDD<@CFFFFFIHHFHHFHHHHHHHIHHHHHFHCEHHHHIIFHHAFHHHFFHIIHHIIIHGHFEH<DDEDH;DEHHHHHFFH;FFHFHFFFFD?FBFF=BDEDDDFEEAE@BEFFFF<BE=B,=,5?*).;>8A:*:::?E?*::A::?AE8AEFEEEC?A:CE?AEA:EF*008:?EF*:C)8;D228A0:??:*.8A8):*:*1::CE##############################
@MISEQ03:64:000000000-A2H3D:1:1101:18209:2060 2:N:0:TCCACAGGAGT
ACGGACTACTAGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGCGTCAATACTTGTCCAGCGAGTCCCCTTCGCCACTGGTGTTCTTCGGAATATCTACGAATTTCACCTCTACACCCGGAATTCCACTCCCCCTTCCAAGATTCCAGCTTAGCAGTTTCAAGGGCAGTTCCGGGGTTGGCACCCGGGATTTCACCCCTGCCTTGCTCAACCCCCCACGGGGCCTTTACCCCCAGCATTCC
+
=9<=9<7<@@@@@@<@A8A>C>8E>-8AE;C99CEEECC>>EECE@CCDE,C@E++5>E-A=E-C@@=@5@C>C<D-5A5CC<CD+<4DE3=<C+4==+<@D++2:9DEE1<9DE########################################################################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:13824:2095 2:N:0:TCCACAGGAGT
ACGGACTACCAGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCGTCTCAGCGTCAGTAACGTCCCAGCGAGCTGCCTTCGCAATGGGTGTTCCTCCTGATATCTGCGCATTTCACCGCTACACCAGGAATTCCACTCCCCCCTTCCGTACCCTAGCCCGACAGTACCCCCCGGCTTCCGAGGCTTGACCCCCCGCCTTTCACACCGGACTTACCGGGCCGCCTACCCGGCCTTTCGCCCCACCGTTTC
+
??<??@@BBBBBBBBBCFCFFFHHHHHHBHHGHHHHHCHHEH>GDEHCA:DFGHHEEEEFFHHHHHHDHED7<CHEGHFFDFFHEDHHHDDDE@D??DD;=B,,5B,,56)):?;BEE?*1::):?**8AEAC*?*:?#################################################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:17839:2106 2:N:0:TCCACAGGAGT
ACGGACTACAAGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGAAGTTCCAAGTTTGAGCCCGGGGCTTTCACATCCAATTTAAAAAACAACCAACGCGCGCTTCACCCCAGGTAATAC
+
?????@@BBBBBBB<5ACFFFFHHHHHHHHHHHHHHHHHHHHFFHHHHHHHHHGHHHFHGHHHHFGHHHHH?EFEECCEEEHHHEHHHHHDFHDFDHDHHHHFFDFFFHFEDDDD,5DD@BB<EEEEECBB?B3B;,?,3?E8CE?*?A*/:/:::??AE::**0:AEE##################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:17831:2124 2:N:0:TCCACAGGAGT
ACGGACTACAAGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTCGAATGCAGTTCCCAGGTTGAGCCCGGGCCTTCAACCTCCACTTTACAAAACAACCAAACGCCGCTTACCGCCACGAAATCC
+
?????@=BBBBBBB5<CFFFFCFHHHHHFHHHHHHHHHHHHHFHEEHHEHFGHGHFGDF?EEFHHHHDGHH=EHEGCCEEEHHHHHH@FFCFH+CFHCFHHHHHFFFHHFE:DDD,=EBDEBE<EEEE?;B?B?E?CEEEEEE8A?CC#######################################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:12555:2129 2:N:0:TCCACAGGAGT
ACGGACTACCGGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCCCGTCAGTTTTGAATGCATTTCCCAGGTTGAGCCCGGGGCTTTCAAACCCAACTAAACAAACAACCAACGCGCGTTTTCCGCCACGTAATTC
+
?????@@BDDDDDDDDEFFFFFHHHFHIIIHHHIIHHHHHEHFHEHHHHIFGHHIHIHFHI=FHFIHIIIHDFHHHHEHEDHHHHHHHHHHHH@FBFFFFFFFEFDEFE6@:@ACBEFFEEB@BCB=A<<A?C:A::C8AEE)0?A*?A:*:**10?1**1.4A88?EE?#################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:13627:2170 2:N:0:TCCACAGGAGT
ACGGACTACTAGGGTATCTAATCCCGTTTGCTACCCTAGCTTTCGCACCTCAGTGTCAGATTGGGTCCAGTGAGCCGCTTTCGCCACCGGTGTTCCTCCAGATATCTACGCATTTCACCGCTCCACCTGGAGTTCCGCTCACCCCTACCCACCTCGAGCCGGCCAGTATCCCGCGCAATTCCCCGTTTGAGCCGGGGGTTTTCCCAAGGGTCTTAACAGCCCACCTACGTCCCCTTTATGCCAGGTAATTC
+
?????@=BD?BBDD?58ACFFCHHHHHHHHFGHHHEEFHHHHHCDEEEECFDGFHDGHHHHFFFHEHHHHHHHFFFHAEHFFEHHHHFHH<DE:C--@F-CCF=,,4BDFE:@E,BBED@:)2>=C?;BC=?C,==*3.84?EC?88A8A:A?*8?###############################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:11781:2223 2:N:0:TCCACAGGAGT
ACGGACTACTGGGGTATCTAATCCTGTTCGCTCCCCATGCTTTCGCTCCTCAGCGTCAGTTAATGCCCAGAGACCTGCCTTCGCCATCGGTGTTCCTCCTGATATCTGCGCATTTCACCGCTACACCAGGAATTCCAATCTCCCCTACATCACTCTAGTCTGCCCGTACCCACCGCAGATCCGGAGTTGAGCCCCGGACTTTCACGGCAGACGCGCAAACCGCCCACAGAGCTCTTTCCCCCCAAAAATCC
+
?????@@BDDDDDDBDCFFFFFHHHHHHHHHHHEHEHDFHHHHHHEDEEHHHFHHHHEHHHHHFGHHHHDD;EFFHCFECAGFEEE+@E@3?E:DFFFHBDHC?BBDFFE8;=DD,,=DEE<@),==?B*44=,=,4**0*0:CA*A?::*0::?0AC:CE*?:*8.4AE?*8?)'82;*?0?####################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:17996:2254 2:N:0:TCCACAGGAGT
ACGGACTACCAGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTGGAGCCCGGGGCTTTACAATCAAACTTAACAAACCCCCTACGCGCGCTTTAGCCCCACAATTTC
+
?9???@@BBDDDDD<BEFFFFFHHHHCEFHGGHHHHHAEHEHFHEEHHCBFHFFHEGHFCGGFCGHHFHHFGHHHHDHHECEDEHDDHH?@?FDFDHHHFHFFDDHDFFFEEEEEDEDB=>BBEE=BECEEEE,B=?CBACE)*0**?C?*:*0:*:?:A:??**::?E**80::::??:CAC:C8C################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:13712:2276 2:N:0:TCCACAGGAGT
ACGGACTACTGGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGAAGTTCCAAGGTTAAGCCGGGGGCTTTCACATCCAACTAACCAAACCACTAACCGCCGCTTTAGGCCCAGCAATTC
+
?????@@BDDDDDDBDEFFFFFHFHHHIFFHIHIIHHEHHHHFHHHEHCFHHDFGGHIIHIHFGHHHGFHF-AEEGHHHHEGFHHHDFHB?FHEHCF?FDDFH??BFFFF=DDEEEFFEDE8>:BECCAEEAECE,ABAACEA*0**?*01?001*::*A**0:*::CE1*8:?**11:*CE#####################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:15819:2301 2:N:0:TCCACAGGAGT
ACGGACTACCGGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGAAGTTCCCAGGTTGACCCCGGGGCTTTCACAGCAGCCTTAACTCACCCCCTGCCAACCCTTTACTCCCCGAAATTC
+
?9???@=BDDDDDD<@CFFFFFHHHBFFHHHFGHHHHHHHEHFGEHHHHEDGD?FCGHHFHHHHHHHHBDF?FHHHFHH@DHHHH+DHHDCFHDFDFBFFDFFEDFEEDEEDEEC=CCEEEBEEFCEFEEFEE?:CEE*8CA800*:E*:AA::***1??*:**1::CF##################################################################################
@MISEQ03:64:000000000-A2H3D:1:1101:11958:2304 2:N:0:TCCACAGGAGT
ACGGACTACTGGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGCCTTTAAAATCCAACTTAAAAAACAACTAACCGCCGCTTTCCGCCCAGAAATCC
+
?????@@BDBDDD@@BFFFFFFHFHHHHFHHHHHHHHHHHHHHHHHHHHHFHHHHHFGH=CGEH=FHFGHFEFHEHHCEEEHHHDCEFHH.?FDFFHHHHBFHFFHFFFFEEEEEEEEEEB@EEEECE;CC?BCEEEC;;CEA*8:AE**00::C0A::?:*0*AEE?E?*A**:C?*:?:?**0)00::**?8>'8';ACA*0*0C?:?******::??E?CE###########################
@MISEQ03:64:000000000-A2H3D:1:1101:19110:2311 2:N:0:TCCACAGGAGT
ACGGACTACCAGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGGGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTAAGCCCGGGGCTTTCCAACCCAACTTAACAAACCACCTACCCCCGCTTTACGCCCAGAAATTC
+
?????@=BDDDDDD<5CFFFFFHHHHHF>CGFHHHHHHHEEHFHEHHHHHHHGAFGHHHHH-5AF5AFHHD+5*5CCCDDHFFHEEHFHHBFF:BDD4?=.=DEFFEBEDEBEEECEFFEE<::CEAACE?:A*1:*C88?AE.?:*::**1:**11*01***1?C??:?0?:C:C**1*1::*:8A?10*1?##########################################################
"""

# expected assembly in FASTA format
expected_default_assembly_fasta = """>MISEQ03:64:000000000-A2H3D:1:1101:14358:1530:TCCACAGGAGT
TNCAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCTTGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:14206:1564:TCCACAGGAGT
TACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTTTGTAAGTCAGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCGTTTGAAACTACAAGGCTAGAGTGTAGCAGAGGGGGGTAGAATTCCACGTGTAGCAGTGAAATGCGTAGAGATGTGGAGGAATACCAATGGCGAAGGCAGCCCCCTGGGTTAACACTGACGCTCATGCACGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCTGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:14943:1619:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:15764:1660:TCCACAGGAGT
TACGAAGGGGGCTAGCGTTGCTCGGAATCACTGGGCGTAAAGCGCACGTAGGCGGATTGTTAAGTCAGGGGTGAAATCCTGGAGCTCAACTCCAGAACTGCCTTTGATACTGGCGATCTTGAGTCCGGGAGAGGTGAGTGGAACTGCGAGTGTAGAGGTGAAATTCGTAGATATTCGCAAGAACACCAGTGGCGAAGGCGGCTCACTGGCCCGGTACTGACGCTGAGGTGCGAAAGCTAGGGGAGCAAACAGGATTAGATACCCCGGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:15211:1752:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:15201:1774:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:15976:1791:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTCGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCGGTGGCGAAGGCGGCTCTCTGGTCCATTTCTGACGCTGAGGTGCGAAAGCTAGGGGAGCAAACGGGATTAGAAACCCTTGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:16031:1840:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCAGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:12964:1852:TCCACAGGAGT
TNCAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGCAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCGAGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:17245:1906:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCTGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:18171:1983:TCCACAGGAGT
GACGTAGGAGGCGAGCGTTGTCCGGATTCATTGGGCGTAAAGAGCGCGCAGGCGGCTTGGTAAGTCGGATGTGAAATCCCGAGGCTCAACCTCGGGTCTGCATCCGATACTGCCCGGCTAGAGGTAGGTAGGGGAGATCGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAGGAACACCGGTGGCGAAGGCGGATCTCTGGGCCTTACCTGACGCTCAGGCGCGAAAGCGTGGGGAGCGAACAGGATTAGAAACCCCGGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:14225:2009:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:16656:2052:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCGGGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:18209:2060:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGTAGGCGGCTCAGCAAGTCAGGGGTGAAATCCCGGGGCTCAACCCCGGAACTGCCCTTGAAACTGCTAAGCTAGAATCTTGGAGAGGCGAGTGGAATTCCGAGTGTAGAGGGGAAATTCGTAGATATTCGGAAGAACACCAGTGGCGAAGGGGACTCGCTGGACAAGTATTGACGCTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTAGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:13824:2095:TCCACAGGAGT
TACGTAGGGGGCTAGCGTTGTCCGGAATCATTGGGCGTAAAGCGCGTGTAGGCGGCCCGGTAAGTCCGCTGTGAAAGTCGGGGGCTCAACCCTCGAAAGCCGGGGGATACTGTCGGGCTAGAGTACGGAAGAGGCGAGTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAGGAACACCCATTGCGAAGGCAGCTCGCTGGGACGTTACTGACGCTGAGACGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCTGGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:17839:2106:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTTGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:17831:2124:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTTGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:12555:2129:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTCGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACGAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:13627:2170:TCCACAGGAGT
GACAGAGGGTGCAAACGTTGTTCGGAATTACTGGGCATAAAGAGCACGTAGGTGGTCTGCTAAGTCACGTGTGAAATCCCCCGGCTCAACCGGGGAATTGCGCGTGATACTGGCCGGCTCGAGGTGGGTAGGGGTGAGCGGAACTCCAGGTGGAGCGGTGAAATGCGTAGATATCTGGAGGAACACCGGTGGCGAAAGCGGCTCACTGGACCCAATCTGACACTGAGGTGCGAAAGCTAGGGTAGCAAACGGGATTAGATACCCTAGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:11781:2223:TCCACAGGAGT
TACGTAGGGCGCAAGCGTTATCCGGAATTATTGGGCGTAAAGAGCTCGTAGGCGGTTTGTCGCGTCTGCCGTGAAAGTCCGGGGCTCAACTCCGGATCTGCGGTGGGTACGGGCAGACTAGAGTGATGTAGGGGAGATTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAGGAACACCGATGGCGAAGGCAGGTCTCTGGGCATTAACTGACGCTGAGGAGCGAAAGCATGGGGAGCGAACAGGATTAGATACCCCAGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:17996:2254:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCTGGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:13712:2276:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCAGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:15819:2301:TCCACAGGAGT
TACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTGAGTTAAGTCTGCTGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:11958:2304:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTTGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCAGTAGTCCGT
>MISEQ03:64:000000000-A2H3D:1:1101:19110:2311:TCCACAGGAGT
TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCCCCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTGGTAGTCCGT
"""


# NOTE need to deal with """ and ''' within the assembly string.
# so i opened output from manual pandaseq run within ipython and pasted here

expected_default_assembly = '@MISEQ03:64:000000000-A2H3D:1:1101:14358:1530:TCCACAGGAGT\nTNCAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCTTGTAGTCCGT\n+\n?#5<????DDDDDDDDEEEEFFHHH"HHHHH""HH"""CHH""DEH"H>C"E5AE"HHHHHHHHHHHHHHH>@::8.\'>>;)>0*10>@>A>>A:*1*0**:00*>@>>>908@>@*0:*:*:=@@AA)90*<BB)7:)0@@BABAAAABA>;>BA>=)..@@>A@@8>88@BB*@*::=>>:0>B>.>0:>8884=\'.\'\'..\'0?8B?B**0:!::>CBDC?IHFHH"HHIIIHHHE"HHHG"IIIIII"IHFFCFFCBBDDDBDDB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:14206:1564:TCCACAGGAGT\nTACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTTTGTAAGTCAGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCGTTTGAAACTACAAGGCTAGAGTGTAGCAGAGGGGGGTAGAATTCCACGTGTAGCAGTGAAATGCGTAGAGATGTGGAGGAATACCAATGGCGAAGGCAGCCCCCTGGGTTAACACTGACGCTCATGCACGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCTGTAGTCCGT\n+\n?????BB?DDDDDD@DDCEEFFH>EEFHHHH"HGHHHCEEFFDC5EECCCCCCDECEHF;?DFD"FHDDBBDF?@;82:88>=<9*0*08@>@@A9=.&8>*9:==9>?9)0**0*0***/\'7))*)\'\'0*6>@B>@?\').0<,3*2(2,*0>?A*!.)+.>@***00)\'.00*0!08)8>>8*0:A><*0:082.4;**>?C@@>"HHHHHHHFBHHHDCGB""@"FFDEHFF@HHH"HHHH"HHFFHH"HHFFFFFC@?DDDBDDB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:14943:1619:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT\n+\n?AAAABBADDDEDEDDGGGGGGIIHHHII"IIIIHHHCCHHF""FH"HDCDH5CFHII""IIIHHHHHHHHHHHHHHHHHHHGGGEGG@8:1!B>*@::*B>:>*BC>>*::009>>01**:>>::A@*:=>?CB.84.7@<@=@A@:AAA>08=A;8.48AA:A>>.8.8>B:*:0:*9)>>A>AB**!0\'\'\'42<2B8\'8..8801**0*.1!1>:>DCCB>HHHH"HHHHHHHHH"HIII"HIIHI""IIFFFFFFA?DDDDDDB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:15764:1660:TCCACAGGAGT\nTACGAAGGGGGCTAGCGTTGCTCGGAATCACTGGGCGTAAAGCGCACGTAGGCGGATTGTTAAGTCAGGGGTGAAATCCTGGAGCTCAACTCCAGAACTGCCTTTGATACTGGCGATCTTGAGTCCGGGAGAGGTGAGTGGAACTGCGAGTGTAGAGGTGAAATTCGTAGATATTCGCAAGAACACCAGTGGCGAAGGCGGCTCACTGGCCCGGTACTGACGCTGAGGTGCGAAAGCTAGGGGAGCAAACAGGATTAGATACCCCGGTAGTCCGT\n+\n???????@DDDDDDBDFEEFEFHEHH"HHHHHHHH"H"HHHH""H"HHA"FH"EHAHH"HHHH""HHHHHH=@FEF"EFEFEDAEEEFFE=CEBCFFFCECEFEFF"CEEEFFCD">FFFEFF*?EED;?8AEE08*=*1)*@97<>>;=2<*0199@:;;>@;>3:9>.77;B:8>8@A7@A:5B@,<*::3\'1874)=872)(1>@*8@HFE"C@HHDHHGFFCCH"EDCA?E=HF"HHGC=HHFCHC"CECFFCA>?<BBBB?BB=@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:15211:1752:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT\n+\n?????<B?DBBDDDBACCEEFFHFHHHHH""HHHH""">EF"F"AACEC7"E5AFEFHHHHHHF=GH"GHF"HHHFHFHH;CED8@DDDE=4@EEEEECE=CECECEECCBB34,=CAB,40:?EAAB9>@>>@5".4\',,7**?A@BAAA>979?,;"CCAEE@@BDBBDEFB"5FFE,FDDDCDFFH"C=?CHHDCDFHFFGFGFHGGHHHBGFCDDCC?EFH"C7DD"CFHHH"B"HHHGHH"BCF""8AFFF?AEBBBB@<-BB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:15201:1774:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT\n+\n?????BB?DDDDDDBDEEEEFFHFHHH"H""HHHHFH>CEHD"DDC"E5""EAC"HHHHHH"HFFHHHHH""HFHHFHHHHHHDEB"EEE@DEEEEEEEEEEEEEEBBCBECE@@@:>@*>*@A:BA>9*8*BB@)84().<ABB9*B@A@:0:>3,:!.49=7@@@)0\'80:@9>*0*0)7@7,,BC>*0*\'88888?\'.5;2>)*0000*8:*0:>A@-AC>@*>@HHHHHHHHHE"EHHG"HHECFC"HFFFFFFC=<BBBBBBB=@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:15976:1791:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTCGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCGGTGGCGAAGGCGGCTCTCTGGTCCATTTCTGACGCTGAGGTGCGAAAGCTAGGGGAGCAAACGGGATTAGAAACCCTTGTAGTCCGT\n+\n?????BB?DDDDDDDDEEEEFFIFH"HHHHI"IHIF"CCHF@F>C"""""DEC"F"ADEHHHH"HHHFGH""HHHF"HHHHHF8:<DEEEADEA>>>>:A9*:>*>BAB?9**>:@>A:*@9>0*@:A*)/\'(>>.06,*76@BB>BBBA1:19>>==!88>A>::>?:EEEEE"DFFFB?+@8)CHHE"+H"HH"HH"HFE?A;H"HHHHH"HFGC"E""FHHH"HH"""HHH"HH""EHHH"HHHH"EFFECBFFC655BBBBBBB=@???<?\n@MISEQ03:64:000000000-A2H3D:1:1101:16031:1840:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCAGTAGTCCGT\n+\n?????BB?DDDDDDDDGFEGGGIHH"HII""""IHFE>C"F"FDCHCH>"CE-5EEII""HIH"HH"HHHHHGHFDFHFHEHGBEEEEGGEDGGG"EGGEGGGG"ABB9*@>8:*=:9***9>>0:!*0!!1<BC.6;.7,>>:B@;@<::1*1(7;=!.8>77A:>0/*0,8>*1,8*.>3>AABB*>"HHDE+==>*C+EA5HHHHHEH"HHGGFDDCCBGFFFHH"DEHC>@:"HHHFFEFH"FFF""FF?FFFFFBBBBBBBB<==?????\n@MISEQ03:64:000000000-A2H3D:1:1101:12964:1852:TCCACAGGAGT\nTNCAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGCAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCGAGTAGTCCGT\n+\n?#55????DDDDDDDDEEEEFFHHBHH"FF"HHFHDC"5>C"""7DCHCCCD5AE"FHHHFHHHHHHH"HFFFFFHFFDFEFF5)>;>:)0*!A@@>7@@=0*8(*/00**!*,*9>7=0/*/0\'**:0))0.)=)0.)!7,,32,*ECB?E"@=D;B"DB:=+E@FD?DFFFHHFFCDC@+D7.=F@+"@EFA+DHEC"E;HGEF"DHHHHHH"GHHHG""H"HFHH"H"HHHHHHH"HFGH"F"BHFF"CE?CCCAA55=?9BBBB=@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:17245:1906:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCTGTAGTCCGT\n+\nAAA??BB?DDDDDDDDGGEGGGIHGHIIIIIIHF?CFCCDFFFDC""HC"DH5"CBB@:>>::0*1*>1*!@B>@24\';84)*1**0:1)@>>B=A@@@::@:@8ACAB?>>7**1**:*>9909:*C7099)@8),7(0759?A??@9<*1(*1)4?.-@A:*9:A8(\'8>@FHFDB"=FHHFDH"HH"HHHHEDECCHHHGGHH"HHHH"DH"HHGE""HHFH"HE"H"F@HEHHH"HHGH"G"HH"""HHFFFFFEBADBDDDDB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:18171:1983:TCCACAGGAGT\nGACGTAGGAGGCGAGCGTTGTCCGGATTCATTGGGCGTAAAGAGCGCGCAGGCGGCTTGGTAAGTCGGATGTGAAATCCCGAGGCTCAACCTCGGGTCTGCATCCGATACTGCCCGGCTAGAGGTAGGTAGGGGAGATCGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAGGAACACCGGTGGCGAAGGCGGATCTCTGGGCCTTACCTGACGCTCAGGCGCGAAAGCGTGGGGAGCGAACAGGATTAGAAACCCCGGTAGTCCGT\n+\n??????B?DDDDDDDDFFEFFFIHF"EE""I"HHFHHEHHFG"F"HCEH"HC""CCEF"E""D""B"EEEEFFFFEEFFCE"B>BEFEEFFCEFE>8>EFFE*A?A?AD?2:226*<:=:1*09.)!>111=>1*6=007<*@A>89*:>*031066;.3>B>*=9))\'.*0*01*>908>:*:1<EFE"EEFHDDHF?,,=C+HFHHHHGHHE"IHEE>CGEFFEHHHEDGCFEHHH"HIHH"HEFHHI"IIFFFFFCB=DDDDBDB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:14225:2009:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT\n+\n?????BB?DDDDDDBDEEEEFFHHHHIIIIH""II""HEHIFGEHH""""EHA"FHIIHIIIIHHHHHHHHH>:8>C>C;.)@::>>>:):*BCB>:*:>A@:B>BCCAA@9=9BCBAA>>9A>ABBB*9677@A78;088>?ABAB@BAA@B?BB=>).>BA@B?B8>4@CCC0>*0088?ACAD@@B*0>B888=>BB>B;8BD*:*:@?BD!::!0>CC>:::>H"HHHIHHHHH"HIIH"IIIHHH"IHFFFFFC??DDDDDDB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:16656:2052:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCGGGTAGTCCGT\n+\n?????BB?BDDDDDDDGFFEGGIIHHI"I"IHHHHIHCCFFDEEEHEHFFEH5ABB::1*:*:)8@8.!:>>:0@822C;8)B:*?B>:800!CC:?B@>BB:@>ACC@AB?8C@>::@::*>A>:::*9?8=;.!*7)+.,@=C=;A*::A*9>=;8.48>A:=<808.7@BBFFHFHFF;HFFHHHH"ED;HDEDD<HEFHGHI"IHHIIHF"HHHFAHHFIIHHH"E"HFHHHHH"HHHH"H"FHHFHHIFFFFFC@<DDDDDDB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:18209:2060:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGTAGGCGGCTCAGCAAGTCAGGGGTGAAATCCCGGGGCTCAACCCCGGAACTGCCCTTGAAACTGCTAAGCTAGAATCTTGGAGAGGCGAGTGGAATTCCGAGTGTAGAGGGGAAATTCGTAGATATTCGGAAGAACACCAGTGGCGAAGGGGACTCGCTGGACAAGTATTGACGCTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTAGTAGTCCGT\n+\n?????BB?BDDDDDDDECEEFFHHH"A"F"HHHHH"HCCHHH""D"EHF"FC"HH")"ECGHHH"DHHHHHCCCFFHHHFEEE"""EEEEEEEB)>EDACEECEECEEECEE:*0A:AE"ECA:0::ACE""E?"8\'4.88?EC*00:"8).0:*00?"..88;1@@*0).++>98!0<3\'>;83@D4<"DC<CC5A5"D<C"C"5@=@@C-E="-E>5++E@C,EDC"@"CEE>>CC"EEC9"C;EA8""E8>C>A8A@<@@@@@@<7<9=<9=\n@MISEQ03:64:000000000-A2H3D:1:1101:13824:2095:TCCACAGGAGT\nTACGTAGGGGGCTAGCGTTGTCCGGAATCATTGGGCGTAAAGCGCGTGTAGGCGGCCCGGTAAGTCCGCTGTGAAAGTCGGGGGCTCAACCCTCGAAAGCCGGGGGATACTGTCGGGCTAGAGTACGGAAGAGGCGAGTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAGGAACACCCATTGCGAAGGCAGCTCGCTGGGACGTTACTGACGCTGAGACGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCTGGTAGTCCGT\n+\n??A??BBADDDDDDBDFEEFFFHHHHH"H""IHHF"""CHHF""CC"EEHHFIHAHHHHH@EFFDFFE"DEDEFFEC"BEEEED"28CC"FECE;"F8?ECD;\'48"?EEFCE:A>>?>EE"EE::A8E8."?8".\'.\')7*A@*=6\'*9*(011*:A.\'6<9\')*)(&\'\'++***10*087AAAAC@8@EHFFDFFHGEHC<7DEHDHHHHHHFFEEEE"HGFD:ACH""G>HEH"C"HHHH"HHBHH""HHFFFCFCBBBBBBBBB@@??<??\n@MISEQ03:64:000000000-A2H3D:1:1101:17839:2106:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTTGTAGTCCGT\n+\nAAA?ABB?DDDDDDDEGFEGGFHIH"IIIII""ID"H5"FHHGHEH=DC"CE5"EEHF"IHIF"HHHHHHHHFHHFHHHHHHGGGGGEE"GG"GDEGGG"GGGGGGCC@90**::B@>>:::/9/*@>)9>>8B>.,;*683@>@?B*:A>A;9=0.4-+@A>@BAB>)4>AA:*11>:>)@BBBD).9*1:>8;2;;@EEF"?HH"HHGF"HH"GHFHHHGHHHHHH"HHFFHHHHH"HHHH"HHHHH""HHFFFFCA5<BBBBBBB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:17831:2124:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTTGTAGTCCGT\n+\nAAAAABB?DDDDDDDDGFGFGGIIH"III"I""IDFH""FHHGD"""">"DHA"FEHI""FGG""HHHH"FH""FFH"HHEHG8DEEGEGGGGGDEEEEGEEGGG"GGEEECCACCEGGGCEE::?CA0<?8BCB!.;!..4?>A;>CBBB;*9:@=20,>@>*1:B..\'8:>>*:*>*0)+>9@BC7>*1*8\'4.44>58<27>BGDHHHHFE"?FDGFHGHGFHEH"EEHFHHHHH"HHHHHHHFHHH"HFCFFFFC<5BBBBBBB=@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:12555:2129:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTCGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACGAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT\n+\n????ABBADDDDDEDDGGGGGGIIHHIIIII""IHHH"FHH"FHHHHH>"DHA"FHF"HHF"HHHHHH"H"HHFHFFHHHHHGBEEAD;DGGGGEGG"CGCEEEG>CC>88<4.1!*1>01**7*:@>*=>0)BB05;)0=:@>@;<?<@@@?ACCCC08@@>:<6B8)4>AA:*:*:0888*9BCB8B*:>B\'842.8\'.4.2>D9?*:>!.7!.::::?BB:*110*0A8A<8>>BHHIIHH"III"F"HHFFFFFEDDDDDDDDB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:13627:2170:TCCACAGGAGT\nGACAGAGGGTGCAAACGTTGTTCGGAATTACTGGGCATAAAGAGCACGTAGGTGGTCTGCTAAGTCACGTGTGAAATCCCCCGGCTCAACCGGGGAATTGCGCGTGATACTGGCCGGCTCGAGGTGGGTAGGGGTGAGCGGAACTCCAGGTGGAGCGGTGAAATGCGTAGATATCTGGAGGAACACCGGTGGCGAAAGCGGCTCACTGGACCCAATCTGACACTGAGGTGCGAAAGCTAGGGTAGCAAACGGGATTAGATACCCTAGTAGTCCGT\n+\n?????BB?DDDDDDDDEFFFFFHHHHHIHII""IFHCEHIIH"F"IHHAAFH5CF"FHH"GHII"H"H"FH"HIII"IIIHHHHHHHHH"FHHHFFEFEFEDBE">BBEEFECECE?D\'..)=?)77<3/<?B>!\'&\')39,=;<@**1>!./%678A!5,B=9BBA>4,,<BA9(><(+89>?;B)58!4>2<=C8>:@82\'.*:!.\'>>B)@@HHHGDHFGDFCEE"EDCHHHHHFEEHHH"FHHHHHHHHCFFCA85?DDBB?DB=@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:11781:2223:TCCACAGGAGT\nTACGTAGGGCGCAAGCGTTATCCGGAATTATTGGGCGTAAAGAGCTCGTAGGCGGTTTGTCGCGTCTGCCGTGAAAGTCCGGGGCTCAACTCCGGATCTGCGGTGGGTACGGGCAGACTAGAGTGATGTAGGGGAGATTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAGGAACACCGATGGCGAAGGCAGGTCTCTGGGCATTAACTGACGCTGAGGAGCGAAAGCATGGGGAGCGAACAGGATTAGATACCCCAGTAGTCCGT\n+\n???A?BB?DDDDDDDDEEECFFHHH"HIH"HIIIH"H"CEHFCFHGG"""FE""H"HH""FDFHCDEFFHHEBFFFF?BBEEEEEEEFFFBE>0>*;28\')>8)7@7-.8*:>*;?:A@0>::0\'17<>*?A:0*0**4,<,<43*@><<,)=8ABA<!*8>;:8B>@A@@>A:@9>::>?:B>3?@<+\'08:2448BA>B>87*:*1@***0*:@@*>@CCBCC>EEDEHHHHHHFD"EHEH"HHHHHH"HHFFFFFCDBDDDDDDB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:17996:2254:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCTGGTAGTCCGT\n+\n?????BB?DDDDDDDDGGGGGGIHHHH"I""HHHF"">CHFHGHHHEHCCCE"AFEHIHHHHHHHH"HHH"""HHHHHHHHHGGEEG"A8A:B@A:>>::9:08**C>:9**>>:>:>:*:0*9\'*@>**0*)A@.8;)08,AAA:@A?<A>??;88=!4;>.1A:8082@@CC*:*:0@8>B>*0B89!>EDECEHHDHHHHGFH"FH"GCFG"CFHGEHFFHFBCH"E"HFHEHEA"HHHH"G"FECH"HHFFFFFEB<DDDDDBB@@???9?\n@MISEQ03:64:000000000-A2H3D:1:1101:13712:2276:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCAGTAGTCCGT\n+\n?????BBADDDDDDDDGGEGGGIHHHHII"IIIIII"CCHHDCE"""HC""H-AFEHIH"H"IHHHHHHHHHHHFFFHHHH"GEGE"DEEDD"CB*:11!*>:8*1BB::*:0**@*::*100>10*>**0*?BA)78.,>AC@C:?>9>99;7:*52!.888:<BB).2>>DB*@><:@.>CC>ADA9*118==.4>B\'\'.7-AB*>*:**00*01?:BCC@BHFCH"H"HFHHHHE"HIIH"H"FIHH"FHFFFFFEDBDDDDDDB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:15819:2301:TCCACAGGAGT\nTACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTGAGTTAAGTCTGCTGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT\n+\n?????BB?DEDDDDDDFDEEFFHHHEFHH"I"III"HCHHFH"HC""ACCHH"EH)<<CCDGFF"FFFBCB@DFHFHFHHEEFB"@EEEFFEEFFFFFF"FFEFCECA:91**9*><1***990*7*A**007@A))=!.32@@BABCB9*0:>@A>@!..8A>:>?)).0001*119:>>8@**>@HD"HHHHD@HHFHHHF?FDBHHHHHHH"FHHGCF?DGDEHH"HEGFHEHHH"HHHG"HHHFFB"HHFFFFFC@<DDDDDDB=@???9?\n@MISEQ03:64:000000000-A2H3D:1:1101:11958:2304:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTTGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCAGTAGTCCGT\n+\n????ABBADDDDDDDDEEEEFFHHH"HIF"HIIIHFE"CEFGDE"""E5"":5!B>>:!******>:>B0!0!@B@;!8\'>8>**::00)0**>:>:*>A:**>*>B>BB@*0*:>::<099:)0**B096*?BA6:=!.;>@>A>:B>@>>A?@B;=!.8@A>BAB)848>0**::@@A>>>>.C)*0!084\'48;\'8\'882;A@!.2<308>!1:>>CCCHHFHHH"HHHHHHHHHHHHHH"HHFHHH"FHFFFFFFB@@DDDBDB@@?????\n@MISEQ03:64:000000000-A2H3D:1:1101:19110:2311:TCCACAGGAGT\nTACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCCCCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTGGTAGTCCGT\n+\n????9BB?BBBBBBBBEEECFAFHHHHHH"FHHHHHHCCEFF/C"C"E>DCH5AECFHHHFHHHHHHH"H""GFHH"FHHHH>1*0!>?8:*9:1*1**A:A:>0>:>>A:1***10*11**:1\'*99*:>.:=7!.\'&.,*?:>*99/:>97;>B>>82>AA@BAB))4>BA<.<94*809@B@BHFHEEHFFHDDCCC5*5"DHHFA5FA"-"HHHHGFAGHHHHH"HEHFHEEHH"HHHH"GC>FHH"HHFFFFFC5<DDDDDDB=@?????\n'

# below will not work due to """ and ''' within the fastq string
#expected_default_assembly_raw = """@MISEQ03:64:000000000-A2H3D:1:1101:14358:1530:TCCACAGGAGT
#TNCAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCTTGTAGTCCGT
#+
#?#5<????DDDDDDDDEEEEFFHHH"HHHHH""HH"""CHH""DEH"H>C"E5AE"HHHHHHHHHHHHHHH>@::8.'>>;)>0*10>@>A>>A:*1*0**:00*>@>>>908@>@*0:*:*:=@@AA)90*<BB)7:)0@@BABAAAABA>;>BA>=)..@@>A@@8>88@BB*@*::=>>:0>B>.>0:>8884='.''..'0?8B?B**0:!::>CBDC?IHFHH"HHIIIHHHE"HHHG"IIIIII"IHFFCFFCBBDDDBDDB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:14206:1564:TCCACAGGAGT
#TACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTTTGTAAGTCAGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCGTTTGAAACTACAAGGCTAGAGTGTAGCAGAGGGGGGTAGAATTCCACGTGTAGCAGTGAAATGCGTAGAGATGTGGAGGAATACCAATGGCGAAGGCAGCCCCCTGGGTTAACACTGACGCTCATGCACGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCTGTAGTCCGT
#+
#?????BB?DDDDDD@DDCEEFFH>EEFHHHH"HGHHHCEEFFDC5EECCCCCCDECEHF;?DFD"FHDDBBDF?@;82:88>=<9*0*08@>@@A9=.&8>*9:==9>?9)0**0*0***/'7))*)''0*6>@B>@?').0<,3*2(2,*0>?A*!.)+.>@***00)'.00*0!08)8>>8*0:A><*0:082.4;**>?C@@>"HHHHHHHFBHHHDCGB""@"FFDEHFF@HHH"HHHH"HHFFHH"HHFFFFFC@?DDDBDDB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:14943:1619:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT
#+
#?AAAABBADDDEDEDDGGGGGGIIHHHII"IIIIHHHCCHHF""FH"HDCDH5CFHII""IIIHHHHHHHHHHHHHHHHHHHGGGEGG@8:1!B>*@::*B>:>*BC>>*::009>>01**:>>::A@*:=>?CB.84.7@<@=@A@:AAA>08=A;8.48AA:A>>.8.8>B:*:0:*9)>>A>AB**!0'''42<2B8'8..8801**0*.1!1>:>DCCB>HHHH"HHHHHHHHH"HIII"HIIHI""IIFFFFFFA?DDDDDDB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:15764:1660:TCCACAGGAGT
#TACGAAGGGGGCTAGCGTTGCTCGGAATCACTGGGCGTAAAGCGCACGTAGGCGGATTGTTAAGTCAGGGGTGAAATCCTGGAGCTCAACTCCAGAACTGCCTTTGATACTGGCGATCTTGAGTCCGGGAGAGGTGAGTGGAACTGCGAGTGTAGAGGTGAAATTCGTAGATATTCGCAAGAACACCAGTGGCGAAGGCGGCTCACTGGCCCGGTACTGACGCTGAGGTGCGAAAGCTAGGGGAGCAAACAGGATTAGATACCCCGGTAGTCCGT
#+
#???????@DDDDDDBDFEEFEFHEHH"HHHHHHHH"H"HHHH""H"HHA"FH"EHAHH"HHHH""HHHHHH=@FEF"EFEFEDAEEEFFE=CEBCFFFCECEFEFF"CEEEFFCD">FFFEFF*?EED;?8AEE08*=*1)*@97<>>;=2<*0199@:;;>@;>3:9>.77;B:8>8@A7@A:5B@,<*::3'1874)=872)(1>@*8@HFE"C@HHDHHGFFCCH"EDCA?E=HF"HHGC=HHFCHC"CECFFCA>?<BBBB?BB=@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:15211:1752:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT
#+
#?????<B?DBBDDDBACCEEFFHFHHHHH""HHHH""">EF"F"AACEC7"E5AFEFHHHHHHF=GH"GHF"HHHFHFHH;CED8@DDDE=4@EEEEECE=CECECEECCBB34,=CAB,40:?EAAB9>@>>@5".4',,7**?A@BAAA>979?,;"CCAEE@@BDBBDEFB"5FFE,FDDDCDFFH"C=?CHHDCDFHFFGFGFHGGHHHBGFCDDCC?EFH"C7DD"CFHHH"B"HHHGHH"BCF""8AFFF?AEBBBB@<-BB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:15201:1774:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT
#+
#?????BB?DDDDDDBDEEEEFFHFHHH"H""HHHHFH>CEHD"DDC"E5""EAC"HHHHHH"HFFHHHHH""HFHHFHHHHHHDEB"EEE@DEEEEEEEEEEEEEEBBCBECE@@@:>@*>*@A:BA>9*8*BB@)84().<ABB9*B@A@:0:>3,:!.49=7@@@)0'80:@9>*0*0)7@7,,BC>*0*'88888?'.5;2>)*0000*8:*0:>A@-AC>@*>@HHHHHHHHHE"EHHG"HHECFC"HFFFFFFC=<BBBBBBB=@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:15976:1791:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTCGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCGGTGGCGAAGGCGGCTCTCTGGTCCATTTCTGACGCTGAGGTGCGAAAGCTAGGGGAGCAAACGGGATTAGAAACCCTTGTAGTCCGT
#+
#?????BB?DDDDDDDDEEEEFFIFH"HHHHI"IHIF"CCHF@F>C"""""DEC"F"ADEHHHH"HHHFGH""HHHF"HHHHHF8:<DEEEADEA>>>>:A9*:>*>BAB?9**>:@>A:*@9>0*@:A*)/'(>>.06,*76@BB>BBBA1:19>>==!88>A>::>?:EEEEE"DFFFB?+@8)CHHE"+H"HH"HH"HFE?A;H"HHHHH"HFGC"E""FHHH"HH"""HHH"HH""EHHH"HHHH"EFFECBFFC655BBBBBBB=@???<?
#@MISEQ03:64:000000000-A2H3D:1:1101:16031:1840:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCAGTAGTCCGT
#+
#?????BB?DDDDDDDDGFEGGGIHH"HII""""IHFE>C"F"FDCHCH>"CE-5EEII""HIH"HH"HHHHHGHFDFHFHEHGBEEEEGGEDGGG"EGGEGGGG"ABB9*@>8:*=:9***9>>0:!*0!!1<BC.6;.7,>>:B@;@<::1*1(7;=!.8>77A:>0/*0,8>*1,8*.>3>AABB*>"HHDE+==>*C+EA5HHHHHEH"HHGGFDDCCBGFFFHH"DEHC>@:"HHHFFEFH"FFF""FF?FFFFFBBBBBBBB<==?????
#@MISEQ03:64:000000000-A2H3D:1:1101:12964:1852:TCCACAGGAGT
#TNCAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGCAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCGAGTAGTCCGT
#+
#?#55????DDDDDDDDEEEEFFHHBHH"FF"HHFHDC"5>C"""7DCHCCCD5AE"FHHHFHHHHHHH"HFFFFFHFFDFEFF5)>;>:)0*!A@@>7@@=0*8(*/00**!*,*9>7=0/*/0'**:0))0.)=)0.)!7,,32,*ECB?E"@=D;B"DB:=+E@FD?DFFFHHFFCDC@+D7.=F@+"@EFA+DHEC"E;HGEF"DHHHHHH"GHHHG""H"HFHH"H"HHHHHHH"HFGH"F"BHFF"CE?CCCAA55=?9BBBB=@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:17245:1906:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCTGTAGTCCGT
#+
#AAA??BB?DDDDDDDDGGEGGGIHGHIIIIIIHF?CFCCDFFFDC""HC"DH5"CBB@:>>::0*1*>1*!@B>@24';84)*1**0:1)@>>B=A@@@::@:@8ACAB?>>7**1**:*>9909:*C7099)@8),7(0759?A??@9<*1(*1)4?.-@A:*9:A8('8>@FHFDB"=FHHFDH"HH"HHHHEDECCHHHGGHH"HHHH"DH"HHGE""HHFH"HE"H"F@HEHHH"HHGH"G"HH"""HHFFFFFEBADBDDDDB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:18171:1983:TCCACAGGAGT
#GACGTAGGAGGCGAGCGTTGTCCGGATTCATTGGGCGTAAAGAGCGCGCAGGCGGCTTGGTAAGTCGGATGTGAAATCCCGAGGCTCAACCTCGGGTCTGCATCCGATACTGCCCGGCTAGAGGTAGGTAGGGGAGATCGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAGGAACACCGGTGGCGAAGGCGGATCTCTGGGCCTTACCTGACGCTCAGGCGCGAAAGCGTGGGGAGCGAACAGGATTAGAAACCCCGGTAGTCCGT
#+
#??????B?DDDDDDDDFFEFFFIHF"EE""I"HHFHHEHHFG"F"HCEH"HC""CCEF"E""D""B"EEEEFFFFEEFFCE"B>BEFEEFFCEFE>8>EFFE*A?A?AD?2:226*<:=:1*09.)!>111=>1*6=007<*@A>89*:>*031066;.3>B>*=9))'.*0*01*>908>:*:1<EFE"EEFHDDHF?,,=C+HFHHHHGHHE"IHEE>CGEFFEHHHEDGCFEHHH"HIHH"HEFHHI"IIFFFFFCB=DDDDBDB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:14225:2009:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT
#+
#?????BB?DDDDDDBDEEEEFFHHHHIIIIH""II""HEHIFGEHH""""EHA"FHIIHIIIIHHHHHHHHH>:8>C>C;.)@::>>>:):*BCB>:*:>A@:B>BCCAA@9=9BCBAA>>9A>ABBB*9677@A78;088>?ABAB@BAA@B?BB=>).>BA@B?B8>4@CCC0>*0088?ACAD@@B*0>B888=>BB>B;8BD*:*:@?BD!::!0>CC>:::>H"HHHIHHHHH"HIIH"IIIHHH"IHFFFFFC??DDDDDDB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:16656:2052:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCGGGTAGTCCGT
#+
#?????BB?BDDDDDDDGFFEGGIIHHI"I"IHHHHIHCCFFDEEEHEHFFEH5ABB::1*:*:)8@8.!:>>:0@822C;8)B:*?B>:800!CC:?B@>BB:@>ACC@AB?8C@>::@::*>A>:::*9?8=;.!*7)+.,@=C=;A*::A*9>=;8.48>A:=<808.7@BBFFHFHFF;HFFHHHH"ED;HDEDD<HEFHGHI"IHHIIHF"HHHFAHHFIIHHH"E"HFHHHHH"HHHH"H"FHHFHHIFFFFFC@<DDDDDDB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:18209:2060:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCACGTAGGCGGCTCAGCAAGTCAGGGGTGAAATCCCGGGGCTCAACCCCGGAACTGCCCTTGAAACTGCTAAGCTAGAATCTTGGAGAGGCGAGTGGAATTCCGAGTGTAGAGGGGAAATTCGTAGATATTCGGAAGAACACCAGTGGCGAAGGGGACTCGCTGGACAAGTATTGACGCTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTAGTAGTCCGT
#+
#?????BB?BDDDDDDDECEEFFHHH"A"F"HHHHH"HCCHHH""D"EHF"FC"HH")"ECGHHH"DHHHHHCCCFFHHHFEEE"""EEEEEEEB)>EDACEECEECEEECEE:*0A:AE"ECA:0::ACE""E?"8'4.88?EC*00:"8).0:*00?"..88;1@@*0).++>98!0<3'>;83@D4<"DC<CC5A5"D<C"C"5@=@@C-E="-E>5++E@C,EDC"@"CEE>>CC"EEC9"C;EA8""E8>C>A8A@<@@@@@@<7<9=<9=
#@MISEQ03:64:000000000-A2H3D:1:1101:13824:2095:TCCACAGGAGT
#TACGTAGGGGGCTAGCGTTGTCCGGAATCATTGGGCGTAAAGCGCGTGTAGGCGGCCCGGTAAGTCCGCTGTGAAAGTCGGGGGCTCAACCCTCGAAAGCCGGGGGATACTGTCGGGCTAGAGTACGGAAGAGGCGAGTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAGGAACACCCATTGCGAAGGCAGCTCGCTGGGACGTTACTGACGCTGAGACGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCTGGTAGTCCGT
#+
#??A??BBADDDDDDBDFEEFFFHHHHH"H""IHHF"""CHHF""CC"EEHHFIHAHHHHH@EFFDFFE"DEDEFFEC"BEEEED"28CC"FECE;"F8?ECD;'48"?EEFCE:A>>?>EE"EE::A8E8."?8".'.')7*A@*=6'*9*(011*:A.'6<9')*)(&''++***10*087AAAAC@8@EHFFDFFHGEHC<7DEHDHHHHHHFFEEEE"HGFD:ACH""G>HEH"C"HHHH"HHBHH""HHFFFCFCBBBBBBBBB@@??<??
#@MISEQ03:64:000000000-A2H3D:1:1101:17839:2106:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTTGTAGTCCGT
#+
#AAA?ABB?DDDDDDDEGFEGGFHIH"IIIII""ID"H5"FHHGHEH=DC"CE5"EEHF"IHIF"HHHHHHHHFHHFHHHHHHGGGGGEE"GG"GDEGGG"GGGGGGCC@90**::B@>>:::/9/*@>)9>>8B>.,;*683@>@?B*:A>A;9=0.4-+@A>@BAB>)4>AA:*11>:>)@BBBD).9*1:>8;2;;@EEF"?HH"HHGF"HH"GHFHHHGHHHHHH"HHFFHHHHH"HHHH"HHHHH""HHFFFFCA5<BBBBBBB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:17831:2124:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTTGTAGTCCGT
#+
#AAAAABB?DDDDDDDDGFGFGGIIH"III"I""IDFH""FHHGD"""">"DHA"FEHI""FGG""HHHH"FH""FFH"HHEHG8DEEGEGGGGGDEEEEGEEGGG"GGEEECCACCEGGGCEE::?CA0<?8BCB!.;!..4?>A;>CBBB;*9:@=20,>@>*1:B..'8:>>*:*>*0)+>9@BC7>*1*8'4.44>58<27>BGDHHHHFE"?FDGFHGHGFHEH"EEHFHHHHH"HHHHHHHFHHH"HFCFFFFC<5BBBBBBB=@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:12555:2129:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTCGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACGAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT
#+
#????ABBADDDDDEDDGGGGGGIIHHIIIII""IHHH"FHH"FHHHHH>"DHA"FHF"HHF"HHHHHH"H"HHFHFFHHHHHGBEEAD;DGGGGEGG"CGCEEEG>CC>88<4.1!*1>01**7*:@>*=>0)BB05;)0=:@>@;<?<@@@?ACCCC08@@>:<6B8)4>AA:*:*:0888*9BCB8B*:>B'842.8'.4.2>D9?*:>!.7!.::::?BB:*110*0A8A<8>>BHHIIHH"III"F"HHFFFFFEDDDDDDDDB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:13627:2170:TCCACAGGAGT
#GACAGAGGGTGCAAACGTTGTTCGGAATTACTGGGCATAAAGAGCACGTAGGTGGTCTGCTAAGTCACGTGTGAAATCCCCCGGCTCAACCGGGGAATTGCGCGTGATACTGGCCGGCTCGAGGTGGGTAGGGGTGAGCGGAACTCCAGGTGGAGCGGTGAAATGCGTAGATATCTGGAGGAACACCGGTGGCGAAAGCGGCTCACTGGACCCAATCTGACACTGAGGTGCGAAAGCTAGGGTAGCAAACGGGATTAGATACCCTAGTAGTCCGT
#+
#?????BB?DDDDDDDDEFFFFFHHHHHIHII""IFHCEHIIH"F"IHHAAFH5CF"FHH"GHII"H"H"FH"HIII"IIIHHHHHHHHH"FHHHFFEFEFEDBE">BBEEFECECE?D'..)=?)77<3/<?B>!'&')39,=;<@**1>!./%678A!5,B=9BBA>4,,<BA9(><(+89>?;B)58!4>2<=C8>:@82'.*:!.'>>B)@@HHHGDHFGDFCEE"EDCHHHHHFEEHHH"FHHHHHHHHCFFCA85?DDBB?DB=@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:11781:2223:TCCACAGGAGT
#TACGTAGGGCGCAAGCGTTATCCGGAATTATTGGGCGTAAAGAGCTCGTAGGCGGTTTGTCGCGTCTGCCGTGAAAGTCCGGGGCTCAACTCCGGATCTGCGGTGGGTACGGGCAGACTAGAGTGATGTAGGGGAGATTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAGGAACACCGATGGCGAAGGCAGGTCTCTGGGCATTAACTGACGCTGAGGAGCGAAAGCATGGGGAGCGAACAGGATTAGATACCCCAGTAGTCCGT
#+
#???A?BB?DDDDDDDDEEECFFHHH"HIH"HIIIH"H"CEHFCFHGG"""FE""H"HH""FDFHCDEFFHHEBFFFF?BBEEEEEEEFFFBE>0>*;28')>8)7@7-.8*:>*;?:A@0>::0'17<>*?A:0*0**4,<,<43*@><<,)=8ABA<!*8>;:8B>@A@@>A:@9>::>?:B>3?@<+'08:2448BA>B>87*:*1@***0*:@@*>@CCBCC>EEDEHHHHHHFD"EHEH"HHHHHH"HHFFFFFCDBDDDDDDB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:17996:2254:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCTGGTAGTCCGT
#+
#?????BB?DDDDDDDDGGGGGGIHHHH"I""HHHF"">CHFHGHHHEHCCCE"AFEHIHHHHHHHH"HHH"""HHHHHHHHHGGEEG"A8A:B@A:>>::9:08**C>:9**>>:>:>:*:0*9'*@>**0*)A@.8;)08,AAA:@A?<A>??;88=!4;>.1A:8082@@CC*:*:0@8>B>*0B89!>EDECEHHDHHHHGFH"FH"GCFG"CFHGEHFFHFBCH"E"HFHEHEA"HHHH"G"FECH"HHFFFFFEB<DDDDDBB@@???9?
#@MISEQ03:64:000000000-A2H3D:1:1101:13712:2276:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCAGTAGTCCGT
#+
#?????BBADDDDDDDDGGEGGGIHHHHII"IIIIII"CCHHDCE"""HC""H-AFEHIH"H"IHHHHHHHHHHHFFFHHHH"GEGE"DEEDD"CB*:11!*>:8*1BB::*:0**@*::*100>10*>**0*?BA)78.,>AC@C:?>9>99;7:*52!.888:<BB).2>>DB*@><:@.>CC>ADA9*118==.4>B''.7-AB*>*:**00*01?:BCC@BHFCH"H"HFHHHHE"HIIH"H"FIHH"FHFFFFFEDBDDDDDDB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:15819:2301:TCCACAGGAGT
#TACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTGAGTTAAGTCTGCTGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT
#+
#?????BB?DEDDDDDDFDEEFFHHHEFHH"I"III"HCHHFH"HC""ACCHH"EH)<<CCDGFF"FFFBCB@DFHFHFHHEEFB"@EEEFFEEFFFFFF"FFEFCECA:91**9*><1***990*7*A**007@A))=!.32@@BABCB9*0:>@A>@!..8A>:>?)).0001*119:>>8@**>@HD"HHHHD@HHFHHHF?FDBHHHHHHH"FHHGCF?DGDEHH"HEGFHEHHH"HHHG"HHHFFB"HHFFFFFC@<DDDDDDB=@???9?
#@MISEQ03:64:000000000-A2H3D:1:1101:11958:2304:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTTGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCAGTAGTCCGT
#+
#????ABBADDDDDDDDEEEEFFHHH"HIF"HIIIHFE"CEFGDE"""E5"":5!B>>:!******>:>B0!0!@B@;!8'>8>**::00)0**>:>:*>A:**>*>B>BB@*0*:>::<099:)0**B096*?BA6:=!.;>@>A>:B>@>>A?@B;=!.8@A>BAB)848>0**::@@A>>>>.C)*0!084'48;'8'882;A@!.2<308>!1:>>CCCHHFHHH"HHHHHHHHHHHHHH"HHFHHH"FHFFFFFFB@@DDDBDB@@?????
#@MISEQ03:64:000000000-A2H3D:1:1101:19110:2311:TCCACAGGAGT
#TACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCCCCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTGGTAGTCCGT
#+
#????9BB?BBBBBBBBEEECFAFHHHHHH"FHHHHHHCCEFF/C"C"E>DCH5AECFHHHFHHHHHHH"H""GFHH"FHHHH>1*0!>?8:*9:1*1**A:A:>0>:>>A:1***10*11**:1'*99*:>.:=7!.'&.,*?:>*99/:>97;>B>>82>AA@BAB))4>BA<.<94*809@B@BHFHEEHFFHDDCCC5*5"DHHFA5FA"-"HHHHGFAGHHHHH"HEHFHEEHH"HHHH"GC>FHH"HHFFFFFC5<DDDDDDB=@?????
#"""
#

if __name__ == '__main__':
    main()
