# Copyright 2005 Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


"""Parse an RFC822-like header structure

RFC822 specifies the Internet e-mail header structure:

    Foo: blah blah blah de blah
    Bar: some more text
        comes here
    FooBar: yadda yadda yadda

This is familiar to a lot of programmers, so it is used in a number
of circumstances. This module contains a simple function that parses
such data in a minimal fashion.

Lars Wirzenius <liw@iki.fi>
"""


import sys


def rfc822_like_header_parse(input):
    """Parse RFC822-like header lines.
    
    Read input line by line until an empty line (just line terminator,
    not even spaces or tabs). Return list of headers as single strings,
    where multi-line headers are combined into one string with the line
    terminators intact. The header name is returned as part of the
    string, not split off.
    
    """
    
    headers = []
    while 1:
        line = input.readline()
        if not line or line in ["\r\n", "\n"]:
            break
        if headers and line and line[0].isspace():
            headers[-1] = headers[-1] + line
        else:
            headers.append(line)
    return headers


if __name__ == "__main__":
    # FIXME: This should be turned into a proper unit test instead
    while 1:
        headers = rfc822_like_header_parse(sys.stdin)
        if not headers:
            break
        print "------------------------"
        for header in headers:
            print ">>", header,
