# -*- coding: utf-8 -*-

# Copyright (C) 2010-2011 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# Python X2go is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Python X2go is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""\
L{X2goSessionProfiles} class - managing x2goclient session profiles.

L{X2goSessionProfiles} is a public API class. Use this class in your Python X2go based 
applications.

"""
__NAME__ = 'x2gosessionprofiles-pylib'

import copy
import types

# Python X2go modules
from x2go.defaults import X2GO_SESSIONPROFILES_CONFIGFILES as _X2GO_SESSIONPROFILES_CONFIGFILES
from x2go.defaults import X2GO_SESSIONPROFILE_DEFAULTS as _X2GO_SESSIONPROFILE_DEFAULTS
from x2go.defaults import X2GO_DESKTOPSESSIONS as _X2GO_DESKTOPSESSIONS
import x2go.inifiles as inifiles
import x2go.log as log
import x2go.utils as utils
from x2go.x2go_exceptions import X2goProfileException


class X2goSessionProfilesFILE(inifiles.X2goIniFile):

    defaultSessionProfile = _X2GO_SESSIONPROFILE_DEFAULTS
    _non_profile_sections = ('embedded')


    def __init__(self, config_files=_X2GO_SESSIONPROFILES_CONFIGFILES, defaults=None, session_profile_defaults=None, logger=None, loglevel=log.loglevel_DEFAULT):
        """\
        STILL UNDOCUMENTED

        """
        self.defaultValues = {}
        self._profile_metatypes = {}
        self._cached_profile_ids = []
        self._cached_profile_names = []

        if logger is None:
            self.logger = log.X2goLogger(loglevel=loglevel)
        else:
            self.logger = copy.deepcopy(logger)
        self.logger.tag = __NAME__

        # providing defaults for an X2goSessionProfiles instance will---in the worst case---override your
        # existing sessions file in your home directory once you write the sessions back to file...
        inifiles.X2goIniFile.__init__(self, config_files, defaults=defaults, logger=logger, loglevel=loglevel)

        if utils._checkSessionProfileDefaults(session_profile_defaults):
            self.defaultSessionProfile = session_profile_defaults

        self.session_profiles = [ p for p in self.iniConfig.sections() if p not in self._non_profile_sections ]
        for session_profile in self.session_profiles:
            self.get_profile_metatype(session_profile)
            for key, default_value in self.defaultSessionProfile.iteritems():
                if not self.iniConfig.has_option(session_profile, key):
                    self._storeValue(session_profile, key, default_value)

    def __call__(self, profile_id_or_name):
        """\
        STILL UNDOCUMENTED

        """
        _profile_id = check_profile_id_or_name(self, profile_id_or_name)
        return self.get_profile_config(profile_id=_profile_id)

    def get_profile_metatype(self, profile_id_or_name, force=False):

        _profile_id = self.check_profile_id_or_name(profile_id_or_name)
        if not self._profile_metatypes.has_key(_profile_id) or force:
            _config = self.get_profile_config(_profile_id)
            if _config['host']:
                if _config['rdpserver'] and _config['command'] == 'RDP':
                    _metatype = 'RDP/proxy'
                elif _config['rootless']:
                    _metatype = 'Single Applications'
                else:
                    if _config['command'] in _X2GO_DESKTOPSESSIONS.keys():
                        _metatype = '%s Desktop' % _config['command']
                    elif _config['command'] in _X2GO_DESKTOPSESSIONS.values():
                        _metatype = '%s Desktop' % [ s for s in _X2GO_DESKTOPSESSIONS.keys() if _config['command'] == _X2GO_DESKTOPSESSIONS[s] ][0]
                    else:
                        _metatype = 'CUSTOM Desktop'
            else:
                if _config['rdpserver'] and _config['command'] == 'RDP':
                    _metatype = 'RDP/direct'
                else:
                    _metatype = 'not supported'
            self._profile_metatypes[_profile_id] = _metatype
        else:
            return self._profile_metatypes[_profile_id]

    def get_profile_option_type(self, option):
        """\
        STILL UNDOCUMENTED

        """
        try:
            return type(self.defaultSessionProfile[option])
        except KeyError:
            return types.StringType

    def get_type(self, section, key):
        """\
        STILL UNDOCUMENTED

        """
        # we have to handle the get_type method separately...
        return self.get_profile_option_type(key)

    def get_profile_config(self, profile_id_or_name=None, profile_id=None):
        """\
        STILL UNDOCUMENTED

        """
        _profile_id = profile_id or self.check_profile_id_or_name(profile_id_or_name)
        _profile_config = {}
        for option in self.iniConfig.options(_profile_id):
            _profile_config[option] = self.get(_profile_id, option, key_type=self.get_profile_option_type(option))
        return _profile_config

    def default_profile_config(self):
        """\
        STILL UNDOCUMENTED

        """
        return copy.deepcopy(self.defaultSessionProfile)

    def has_profile(self, profile_id_or_name):
        try:
            _profile_id = self.check_profile_id_or_name(profile_id_or_name)
            return True
        except X2goProfileException:
            return False

    @property
    def profile_ids(self):
        """\
        STILL UNDOCUMENTED

        """
        if not self._cached_profile_ids:
            self._cached_profile_ids = [ s for s in self.iniConfig.sections() if s not in self._non_profile_sections ]
        return self._cached_profile_ids

    def has_profile_id(self, profile_id):
        """\
        STILL UNDOCUMENTED

        """
        return profile_id in self.profile_ids

    @property
    def profile_names(self):
        """\
        STILL UNDOCUMENTED

        """
        if not self._cached_profile_names:
            self._cached_profile_names = [ self.to_profile_name(p) for p in self.profile_ids ]
        return  self._cached_profile_names

    def has_profile_name(self, profile_name):
        """\
        STILL UNDOCUMENTED

        """
        return profile_name in self.profile_names

    def to_profile_id(self, profile_name):
        """\
        STILL UNDOCUMENTED

        """
        _profile_ids = [ p for p in self.profile_ids if self.to_profile_name(p) == profile_name ]
        if len(_profile_ids) == 1:
            return _profile_ids[0]
        elif len(_profile_ids) == 0:
            return None
        else:
            raise X2goProfileException('The sessions config file contains multiple session profiles with name: %s' % profile_name)

    def to_profile_name(self, profile_id):
        """\
        STILL UNDOCUMENTED

        """
        _profile_config = self.get_profile_config(profile_id=profile_id)
        if _profile_config.has_key('name'):
            return _profile_config['name']
        else:
            return ''

    def add_profile(self, profile_id=None, **kwargs):
        """\
        STILL UNDOCUMENTED

        """
        if profile_id is None:
            profile_id = utils._genSessionProfileId()
        for key, value in kwargs.items():
            if key in self.defaultSessionProfile:
                self.update_value(profile_id, key, value)
            else:
                raise X2goProfileException('keyword ,,%s\'\' not supported in X2go session profile' % key)

        for key, value in self.defaultSessionProfile.items():
            if key in kwargs: continue
            self.update_value(profile_id, key, value)

        self._cached_profile_ids = []
        self._cached_profile_names = []

        return profile_id

    def delete_profile(self, profile_id_or_name):
        """\
        STILL UNDOCUMENTED

        """
        _profile_id = self.check_profile_id_or_name(profile_id_or_name)
        self.iniConfig.remove_section(_profile_id)
        self.write_user_config = True
        self.write()
        self._cached_profile_ids = []
        self._cached_profile_names = []

    def update_value(self, section, key, value):
        """\
        STILL UNDOCUMENTED

        """
        profile_id = section
        if key == 'name':
            profile_name = value
            current_profile_name = self.get_value(section, key)
            if not profile_name:
                raise X2goProfileException('profile name for profile id %s may not be empty' % profile_id)
            else:
                if profile_name != current_profile_name and profile_name in self.profile_names:
                    raise X2goProfileException('a profile of name ,,%s'' already exists' % profile_name)
            self._cached_profile_names = []
        inifiles.X2goIniFile.update_value(self, section, key, value)

    def check_profile_id_or_name(self, profile_id_or_name):
        """\
        STILL UNDOCUMENTED

        """
        _profile_id = None
        if self.has_profile_name(profile_id_or_name):
            # we were given a sesion profile name...
            _profile_id = self.to_profile_id(profile_id_or_name)
        elif self.has_profile_id(profile_id_or_name):
            # we were given a session profile id...
            _profile_id = profile_id_or_name
        else:
            raise X2goProfileException('No session profile with id or name ,,%s\'\' exists.' % profile_id_or_name)
        return _profile_id

    def to_session_params(self, profile_id_or_name=None, profile_id=None):
        """\
        STILL UNDOCUMENTED

        """
        _profile_id = profile_id or self.check_profile_id_or_name(profile_id_or_name)
        return utils._convert_SessionProfileOptions_2_SessionParams(self.get_profile_config(_profile_id))

    def get_session_param(self, profile_id_or_name, param):
        """\
        STILL UNDOCUMENTED

        """
        return self.to_session_params(profile_id_or_name)[param]
