# -*- coding: utf-8 -*-

# Copyright (C) 2010-2011 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# Python X2go is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Python X2go is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
#
# This code was initially written by:
#       2010 Dick Kniep <dick.kniep@lindix.nl>
#
# Other contributors:
#       none so far

__NAME__ = 'x2goxserver-pylib'

from defaults import X2GOCLIENT_OS as _X2GOCLIENT_OS
if _X2GOCLIENT_OS == 'Windows':
    import wmi
    import win32process

# modules
import os
import threading
import gevent
import copy

# Python X2go modules
import log
from defaults import X2GO_XCONFIG_CONFIGFILES as _X2GO_XCONFIG_CONFIGFILES
from defaults import X2GO_CLIENTXCONFIG_DEFAULTS as _X2GO_CLIENTXCONFIG_DEFAULTS
import inifiles


class X2goClientXConfig(inifiles.X2goIniFile):
    """\
    Configuration file based XServer startup settings for X2goClient instances.

    This class is needed for Windows systems and (maybe soon) for Unix desktops using Wayland.

    """
    defaultValues = _X2GO_CLIENTXCONFIG_DEFAULTS

    def __init__(self, config_files=_X2GO_XCONFIG_CONFIGFILES, defaults=None, logger=None, loglevel=log.loglevel_DEFAULT):
        """\
        Constructs an L{X2goClientXConfig} instance. This is normally done by an L{X2goClient} instance.
        You can retrieve this L{X2goClientXConfig} instance with the C{X2goClient.get_client_xconfig()} 
        method.

        On construction the L{X2goClientXConfig} instance is filled with values from the configuration files::

            /etc/x2goclient/xconfig
            ~/.x2goclient/xconfig

        The files are read in the specified order and config options of both files are merged. Options
        set in the user configuration file (C{~/.x2goclient/xconfig}) override global options set in
        C{/etc/x2goclient/xconfig}.

        @param config_files: a list of configuration file names
        @type config_files: C{list}
        @param defaults: a Python dictionary with configuration file defaults (use on your own risk)
        @type defaults: C{dict}
        @param logger: you can pass an L{X2goLogger} object to the L{X2goClientXConfig} constructor
        @type logger: C{instance}
        @param loglevel: if no L{X2goLogger} object has been supplied a new one will be
            constructed with the given loglevel
        @type loglevel: C{int}

        """
        if _X2GOCLIENT_OS not in ("Windows"):
            import exceptions
            class OSNotSupportedException(exceptions.StandardError): pass
            raise OSNotSupportedException('classes of x2go.xserver module are for Windows only')

        inifiles.X2goIniFile.__init__(self, config_files, defaults=defaults, logger=logger, loglevel=loglevel)

    def get_xserver_config(self, xserver_name):
        """\
        Retrieve the XServer configuration (from the xconfig file) for the given XServer application.

        @param xserver_name: name of the XServer application
        @type xserver_name: C{str}

        @return: A Python dictionary containing the XServer's configuration settings
        @rtype: C{list}

        """
        _xserver_config = {}
        for option in self.iniConfig.options(xserver_name):
            _xserver_config[option] = self.get(xserver_name, option, key_type=self.get_type(xserver_name, option))
        return _xserver_config

    @property
    def known_xservers(self):
        """\
        Renders a list of XServers that are known to Python X2go.

        """
        return self.get_value('XServers', 'known_xservers')

    @property
    def installed_xservers(self):
        """\
        Among the known XServers renders a list of XServers that are actually
        installed on the system.

        """
        _installed = []
        for xserver_name in self.known_xservers:
            if os.path.exists(os.path.normpath(self.get_xserver_config(xserver_name)['test_installed'])):
                _installed.append(xserver_name)
        return _installed

    @property
    def running_xservers(self):
        """\
        Tries to render a list of running XServer processes from the system's process list.

        """
        _running = []
        _wmi = wmi.WMI()
        _p_names = []
        for process in _wmi.Win32_Process():
            _p_names.append(process.Name)

        for xserver_name in self.installed_xservers:
            process_name = self.get_xserver_config(xserver_name)['process_name']
            if process_name in _p_names:
                # XServer is already running
                _running.append(xserver_name)
                continue
        return _running

    @property
    def xserver_launch_possible(self):
        """\
        Detect if there is an XServer (that is known to Python X2go) installed on the system.
        Equals C{True} if we have found an installed XServer that we can launch.

        """
        return bool(self.installed_xservers)

    @property
    def xserver_launch_needed(self):
        """\
        Detect if an XServer launch is really needed (or if we use an already running XServer instance).
        Equals C{True} if we have to launch an XServer before we can start/resume
        X2go sessions.

        """
        return not bool(self.running_xservers)

    @property
    def preferred_xserver(self):
        """\
        Renders a list of preferred XServer names (most preferred on top).

        """
        if self.xserver_launch_possible and self.xserver_launch_needed:
            return (self.installed_xservers[0], self.get_xserver_config(self.installed_xservers[0]))
        else:
            return None


class X2goXServer(threading.Thread):
    """
    This class is responsible for starting/stopping an external XServer application.

    X2go applications require a running XServer on the client system. This class will
    manage/handle the XServer while your X2go application is running.

    """

    def __init__(self, xserver_name, xserver_config, logger=None, loglevel=log.loglevel_DEFAULT):
        """\
        Initialize an XServer thread.

        @param xserver_name: name of the XServer to start (refer to the xconfig file for available names)
        @type xserver_name: C{str}
        @param xserver_config: XServer configuration node (as derived from L{X2goClientXConfig.get_xserver_config()}
        @type xserver_config: C{dict}
        @param logger: you can pass an L{X2goLogger} object to the L{X2goClientXConfig} constructor
        @type logger: C{instance}
        @param loglevel: if no L{X2goLogger} object has been supplied a new one will be
            constructed with the given loglevel
        @type loglevel: C{int}

        """
        if _X2GOCLIENT_OS not in ("Windows"):
            import exceptions
            class OSNotSupportedException(exceptions.StandardError): pass
            raise OSNotSupportedException('classes of x2go.xserver module are for Windows only')

        if logger is None:
            self.logger = log.X2goLogger(loglevel=loglevel)
        else:
            self.logger = copy.deepcopy(logger)
        self.logger.tag = __NAME__

        self._keepalive = None

        self.xserver_name = xserver_name
        self.xserver_config = xserver_config
        if self.xserver_config.has_key('display'):
            self.logger('settings DISPLAY environment variable to %s' % self.xserver_config['display'], loglevel=log.loglevel_NOTICE)
            os.environ.update({'DISPLAY': str(self.xserver_config['display'])})
        threading.Thread.__init__(self)
        self.daemon = True
        self.start()

    def run(self):
        """\
        Start this L{X2goXServer} thread. This will launch the configured XServer application.

        """
        self._keepalive = True
        cmd_line = [self.xserver_config['run_command']]
        cmd_line.extend(self.xserver_config['parameters'])
        self.logger('starting XServer ,,%s\'\' with command line: %s' % (self.xserver_name, ' '.join(cmd_line)), loglevel=log.loglevel_DEBUG)

        if _X2GOCLIENT_OS == 'Windows':
            si = win32process.STARTUPINFO()
            p_info = win32process.CreateProcess(None,
                                                ' '.join(cmd_line),
                                                None,
                                                None,
                                                0,
                                                win32process.NORMAL_PRIORITY_CLASS,
                                                None,
                                                None,
                                                si,
                                               )
            (hProcess, hThread, processId, threadId) = p_info

        while self._keepalive:
            gevent.sleep(1)

        self.logger('terminating running XServer ,,%s\'\'' % self.xserver_name, loglevel=log.loglevel_DEBUG)

        if _X2GOCLIENT_OS == 'Windows':
            try:
                win32process.TerminateProcess(hProcess, 0)
            except win32process.error:
                pass

    def stop_thread(self):
        """\
        A call to this method will stop the XServer application and do a cleanup afterwards.

        """
        self.logger('stop_thread() method has been called', loglevel=log.loglevel_DEBUG)
        self._keepalive = False

