# -*- coding: ISO-8859-1 -*-

# Copyright (C) 2002, 2003 Jrg Lehmann <joerg@luga.de>
#
# Ogg Vorbis interface by Byron Ellacott <bje@apnic.net>.
#
# This file is part of PyTone (http://www.luga.de/pytone/)
#
# PyTone is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2
# as published by the Free Software Foundation.
#
# PyTone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PyX; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os.path, re, string, time
import locale, sys
import MP3Info

from helper import debug

fallbacklocalecharset = 'iso-8859-1'

# Try to determine "correct" character set for the reencoding of the
# unicode strings contained in Ogg Vorbis files 
try:
    # works only in python > 2.3
    localecharset = locale.getpreferredencoding()
except:
    try:
        localecharset = locale.getdefaultlocale()[1]
    except:
        try:
            localecharset = sys.getdefaultencoding()
        except:
            localecharset = fallbacklocalecharset
if localecharset in [None, 'ANSI_X3.4-1968']:
    localecharset = fallbacklocalecharset

try:
    import ogg.vorbis
    oggsupport = 1
except:
    oggsupport = 0
    debug("disabled Ogg Vorbis support, since module is not present\n")
    pass

tracknrandtitlere = re.compile("^\[?(\d+)\]? ?[- ] ?(.*)\.(mp3|ogg)$")
UNKNOWN = "Unknown"


class dbitem:

    """ base class for various items stored in database:

    songs, albums, artists, genres, years, playlists"""

    def __cmp__(self, other):
        try:
            return cmp(self.__class__, other.__class__) or cmp(self.id, other.id)
        except:
            return 1

    def __repr__(self):
        return "%s(%s)" % (self.__class__, self.id)


class song(dbitem):
    def __init__(self, path):
        self.path = os.path.normpath(path)
        # use path of file as its id
        self.id = self.path
        self.name = os.path.basename(path)
        if not os.access(self.path, os.R_OK):
            raise IOError("cannot read song")

        # guesses for title and tracknr
        match = re.match(tracknrandtitlere, self.name)
        if match:
            fntracknr = str(int(match.group(1)))
            fntitle = match.group(2)
        else:
            fntracknr = ""
            fntitle = self.name
            if fntitle.endswith(".mp3") or fntitle.endswith(".ogg"):
                fntitle = fntitle[:-4]

        try:
            if self.path.endswith(".mp3"):
                self.type = "mp3"
                mp3info = MP3Info.MP3Info(open(self.path, "rb"))
                self.title = mp3info.title
                self.album = mp3info.album
                self.artist = mp3info.artist
                self.year = mp3info.year
                self.genre  = mp3info.genre
                self.tracknr = mp3info.track
                if mp3info.id3.tags.has_key("TLEN"):
                    self.length = int(mp3info.id3.tags["TLEN"])/1000
                else:
                    self.length = mp3info.mpeg.length
            else:
                self.type = "ogg"
                vf = ogg.vorbis.VorbisFile(self.path)
                id3get = vf.comment().as_dict().get
                self.title = id3get('TITLE', [""])[0]
                self.title = self.title.encode(localecharset, 'replace')
                self.album = id3get('ALBUM', [""])[0]
                self.album = self.album.encode(localecharset, 'replace')
                self.artist = id3get('ARTIST', [""])[0]
                self.artist = self.artist.encode(localecharset, 'replace')
                self.year = id3get('DATE', [""])[0]
                self.year = self.year.encode(localecharset, 'replace')
                self.genre  = id3get('GENRE', [""])[0]
                self.genre = self.genre.encode(localecharset, 'replace')
                self.tracknr = id3get('TRACKNUMBER', [""])[0]
                self.tracknr = self.tracknr.encode(localecharset, 'replace')
                self.length = vf.time_total(0)
        except:
            self.title = fntitle
            self.album = ""
            self.artist = ""
            self.year = ""
            self.genre = ""
            self.tracknr = ""
            self.length = 0

        # sanity check for tracknr
        try:
            self.tracknr= str(int(self.tracknr))
        except:
            # treat track number like "3/12"
            try:
                self.tracknr= str(int(self.tracknr[:self.tracknr.index('/')]))
            except:
                self.tracknr= ""

        # do some further treatment of the song info

        # use title from filename, if it is a longer version of
        # the id3 tag title
        if not self.title or fntitle.startswith(self.title):
            self.title = fntitle

        # also try to use tracknr from filename, if not present as id3 tag
        if not self.tracknr or self.tracknr=="0":
            self.tracknr = fntracknr

        # we don't want empty album names
        if not self.album:
            self.album = UNKNOWN

        # nor empty artist names
        if not self.artist:
            self.artist = UNKNOWN

        # nor empty genres
        if not self.genre:
            self.genre = UNKNOWN

        if not self.year or self.year=="0":
            self.year = None
        else:
            try:
                self.year = int(self.year)
            except:
                self.year = None

        # XXX: make further treatment of tags configurable

        # strip leading "The " in artist names, often used inconsistently
        # XXX: make this configurable
        if self.artist.startswith("The ") and len(self.artist)>4:
            self.artist = self.artist[4:]

        # normalize artist, album and title
        self.artist = string.capwords(self.artist)
        self.album = string.capwords(self.album)
        self.title = string.capwords(self.title)

        if 0:
            translationtable = string.maketrans('',
                                                'AAAAEEEEIIIIOOOOUUUUaaaaeeeeiiiioooouuuu')
            self.artist = string.translate(self.artist, translationtable)
            self.album = string.translate(self.album, translationtable)
            self.title = string.translate(self.title, translationtable)

        self.nrplayed = 0
        self.lastplayed = None
        self.added = time.time()
        self.rating = None

    def __getattr__(self, name):
        if name=="albumid":
            return self.album
        elif name=="artistid":
            return artist
        elif name=="genreid":
            return genre
        else:
            raise AttributeError


class artist(dbitem):
    def __init__(self, name, path=None):
        self.id = name
        self.name = name
        self.albums = []
        self.songs = []
        self.genres = []
        self.years = []
        self.path = path
        self.rating = None


class album(dbitem):
    def __init__(self, name, path=None):
        self.id = name
        self.name = name
        self.artists = []
        self.songs = []
        self.genres = []
        self.years = []
        self.path = path
        self.rating = None


class playlist(dbitem):
    def __init__(self, path):
        assert os.path.isfile(path), "'%s' doesn't exist" % path
        self.path = self.id = os.path.normpath(path)
        self.name = os.path.basename(path)
        if self.name.endswith(".m3u"):
            self.name = self.name[:-4]
        self.songs = []

        file = open(self.path, "r")

        for line in file.xreadlines():
            # XXX: interpret extended m3u format (especially for streams)
            # see: http://forums.winamp.com/showthread.php?s=dbec47f3a05d10a3a77959f17926d39c&threadid=65772
            if not line.startswith("#") and not chr(0) in line:
                path = line.strip()
                if not path.startswith("/"):
                    path = os.path.join(self.path, path)
                if os.path.isfile(path):
                    self.songs.append(path)
        file.close()


class genre(dbitem):
    def __init__(self, name):
        self.name = self.id = name
        self.artists = []
        self.albums = []
        self.songs = []


class year(dbitem):
    def __init__(self, year):
        self.year = year
        self.id = `self.year`
        self.artists = []
        self.albums = []
        self.songs = []
