# -*- coding: ISO-8859-1 -*-

# Copyright (C) 2002, 2003, 2004 Jrg Lehmann <joerg@luga.de>
#
# This file is part of PyTone (http://www.luga.de/pytone/)
#
# PyTone is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2
# as published by the Free Software Foundation.
#
# PyTone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PyX; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import config
import events, hub, requests
import slist

#
# playlist class, which acts as glue layer between the playlist service
# and the playlist window
# 

class playlist(slist.slist):
    def __init__(self, win, playerid):
        slist.slist.__init__(self, win, config.playlistwindow.scrollmode=="page")
        self.playerid = playerid
        self.songdbid = "main"
        self.items, self.ptime, self.ttime, self.autoplaymode, self.playingsong = \
                    hub.hub.request(requests.playlistgetcontents())
        
        self.win.channel.subscribe(events.playlistchanged, self.playlistchanged)
        
    def _recenter(self):
        """ recenter playlist around currently playing (or alternatively last) song """
        for i in range(len(self)):
            if self[i] is self.playingsong or i==len(self)-1:
                self.selected = i
                h2 = self.win.ih/2
                if len(self)-i <= h2:
                    self.top = max(0, len(self)-self.win.ih)
                elif i >= h2:
                    self.top = i-h2
                else:
                    self.top = 0
                self._updatetop()
                break

    def getselected(self):
        playlistitem = slist.slist.getselected(self)
        if playlistitem:
            return playlistitem.song
        else:
            return None

    def deleteselected(self):
        "delete currently selected item"
        if self.selected is not None:
            hub.hub.notify(events.playlistdeletesong(self.items[self.selected].id))

    def moveitemup(self):
        "move selected item up, if not first"
        if self.selected is not None and self.selected>0:
            hub.hub.notify(events.playlistmovesongup(self.items[self.selected].id))

    def moveitemdown(self):
        "move selected item down, if not last"
        if self.selected is not None and self.selected<len(self)-1:
            hub.hub.notify(events.playlistmovesongdown(self.items[self.selected].id))

    def rateselection(self, rating):
        assert 0<=rating<=5, "only ratings between 0 and 5 are allowed"
        if self.selected is not None:
            self.getselected().rate(rating)
            hub.hub.notify(events.selectionchanged(self.getselected()))

    def rescanselection(self):
        if self.selected is not None:
            self.getselected().rescan()
            hub.hub.notify(events.selectionchanged(self.getselected()))

    def playselected(self):
        item = slist.slist.getselected(self)
        if item:
            # move item up before first not yet played song
            if not item.hasbeenplayed():
                self.deleteselected()
            hub.hub.notify(events.playlistaddsongtop(item.song))

    # event handler

    def playlistchanged(self, event):
        oldlen = len(self)
        if self.selected is not None:
            oldselecteditem = slist.slist.getselected(self)
            oldselected = self.selected
            self.set(event.songs)
            try:
                self.selected = self.index(oldselecteditem)
            except ValueError:
                if oldselected<len(self):
                    self.selected = oldselected
                elif len(self)>0:
                    self.selected = len(self)-1
                self._notifyselectionchanged()
            self._updatetop()
        else:
            self.set(event.songs)
            
        self.ptime = event.ptime
        self.ttime = event.ttime
        self.autoplaymode = event.autoplaymode
        self.playingsong = event.playingsong

        # recenter window if the playlist window doesn't have the focus.
        if not self.win.hasfocus():
            self._recenter()
        
        self.win.update()
