/* ====================================================================
 *
 *
 *   BSD LICENSE
 *
 *   Copyright(c) 2021-2024 Intel Corporation.
 *   All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in
 *       the documentation and/or other materials provided with the
 *       distribution.
 *     * Neither the name of Intel Corporation nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * ====================================================================
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <openssl/err.h>
#include <openssl/async.h>
#include <openssl/objects.h>
#include <openssl/bn.h>
#include <openssl/dsa.h>
#include <openssl/engine.h>
#ifdef QAT_OPENSSL_PROVIDER
# include <openssl/param_build.h>
#endif

#include "tests.h"
#include "../qat_utils.h"

static unsigned char BN_p[] = {
    0xD7, 0x57, 0x26, 0x2C, 0x45, 0x84, 0xC4, 0x4C,
    0x21, 0x1F, 0x18, 0xBD, 0x96, 0xE5, 0xF0, 0x61,
    0xC4, 0xF0, 0xA4, 0x23, 0xF7, 0xFE, 0x6B, 0x6B,
    0x85, 0xB3, 0x4C, 0xEF, 0x72, 0xCE, 0x14, 0xA0,
    0xD3, 0xA5, 0x22, 0x2F, 0xE0, 0x8C, 0xEC, 0xE6,
    0x5B, 0xE6, 0xC2, 0x65, 0x85, 0x48, 0x89, 0xDC,
    0x1E, 0xDB, 0xD1, 0x3E, 0xC8, 0xB2, 0x74, 0xDA,
    0x9F, 0x75, 0xBA, 0x26, 0xCC, 0xB9, 0x87, 0x72,
    0x36, 0x02, 0x78, 0x7E, 0x92, 0x2B, 0xA8, 0x44,
    0x21, 0xF2, 0x2C, 0x3C, 0x89, 0xCB, 0x9B, 0x06,
    0xFD, 0x60, 0xFE, 0x01, 0x94, 0x1D, 0xDD, 0x77,
    0xFE, 0x6B, 0x12, 0x89, 0x3D, 0xA7, 0x6E, 0xEB,
    0xC1, 0xD1, 0x28, 0xD9, 0x7F, 0x06, 0x78, 0xD7,
    0x72, 0x2B, 0x53, 0x41, 0xC8, 0x50, 0x6F, 0x35,
    0x82, 0x14, 0xB1, 0x6A, 0x2F, 0xAC, 0x4B, 0x36,
    0x89, 0x50, 0x38, 0x78, 0x11, 0xC7, 0xDA, 0x33,
};

static unsigned char BN_q[] = {
    0xC7, 0x73, 0x21, 0x8C, 0x73, 0x7E, 0xC8, 0xEE,
    0x99, 0x3B, 0x4F, 0x2D, 0xED, 0x30, 0xF4, 0x8E,
    0xDA, 0xCE, 0x91, 0x5F,
};

static unsigned char BN_g[] = {
    0x82, 0x26, 0x90, 0x09, 0xE1, 0x4E, 0xC4, 0x74,
    0xBA, 0xF2, 0x93, 0x2E, 0x69, 0xD3, 0xB1, 0xF1,
    0x85, 0x17, 0xAD, 0x95, 0x94, 0x18, 0x4C, 0xCD,
    0xFC, 0xEA, 0xE9, 0x6E, 0xC4, 0xD5, 0xEF, 0x93,
    0x13, 0x3E, 0x84, 0xB4, 0x70, 0x93, 0xC5, 0x2B,
    0x20, 0xCD, 0x35, 0xD0, 0x24, 0x92, 0xB3, 0x95,
    0x9E, 0xC6, 0x49, 0x96, 0x25, 0xBC, 0x4F, 0xA5,
    0x08, 0x2E, 0x22, 0xC5, 0xB3, 0x74, 0xE1, 0x6D,
    0xD0, 0x01, 0x32, 0xCE, 0x71, 0xB0, 0x20, 0x21,
    0x70, 0x91, 0xAC, 0x71, 0x7B, 0x61, 0x23, 0x91,
    0xC7, 0x6C, 0x1F, 0xB2, 0xE8, 0x83, 0x17, 0xC1,
    0xBD, 0x81, 0x71, 0xD4, 0x1E, 0xCB, 0x83, 0xE2,
    0x10, 0xC0, 0x3C, 0xC9, 0xB3, 0x2E, 0x81, 0x05,
    0x61, 0xC2, 0x16, 0x21, 0xC7, 0x3D, 0x6D, 0xAA,
    0xC0, 0x28, 0xF4, 0xB1, 0x58, 0x5D, 0xA7, 0xF4,
    0x25, 0x19, 0x71, 0x8C, 0xC9, 0xB0, 0x9E, 0xEF,
};


static unsigned char Pub_key[] = {
    0x52, 0x3C, 0x3E, 0x53, 0x41, 0xC3, 0xC8, 0xDF,
    0x22, 0x4E, 0x07, 0x0C, 0x99, 0x76, 0xFC, 0x7D,
    0xF1, 0x95, 0xD3, 0xC5, 0x1D, 0x67, 0x1A, 0xF2,
    0xC9, 0x68, 0xA2, 0xA1, 0x41, 0x35, 0x1F, 0xFC,
    0x64, 0x47, 0x65, 0xAD, 0xEB, 0xC4, 0x71, 0xD5,
    0x1B, 0xC2, 0xEF, 0x76, 0x21, 0xE9, 0xED, 0x6A,
    0xA6, 0xD9, 0xDB, 0x5B, 0xBB, 0x81, 0x43, 0x8F,
    0xC0, 0x1E, 0xE6, 0x49, 0x2A, 0xB7, 0xEA, 0x8F,
    0xCB, 0x6B, 0x93, 0x1E, 0x94, 0x8C, 0x24, 0x48,
    0xE3, 0x80, 0xD3, 0x9B, 0x4F, 0xDC, 0xBF, 0x45,
    0xE3, 0xC0, 0xFC, 0xDA, 0x1C, 0x06, 0x74, 0xF9,
    0x02, 0x2A, 0x5A, 0xAA, 0x18, 0x58, 0x1D, 0x12,
    0x4D, 0x07, 0x0F, 0x00, 0xB4, 0x5E, 0xA3, 0x62,
    0xB1, 0xFA, 0xEE, 0x05, 0x04, 0x89, 0x27, 0x5A,
    0x68, 0xBF, 0x8B, 0x19, 0x90, 0x84, 0x21, 0xB8,
    0xFB, 0x3B, 0x37, 0x04, 0x4D, 0x6B, 0xB6, 0xD2,
};

static unsigned char Priv_key[] = {
    0x6E, 0xB5, 0x08, 0x61, 0xE5, 0x25, 0xD7, 0xEF,
    0x6C, 0x53, 0xBD, 0x77, 0x72, 0x0F, 0x98, 0xE1,
    0x60, 0x3D, 0x67, 0x67,
};

#ifdef QAT_OPENSSL_PROVIDER
static unsigned char dsa1024_priv[] = {
    0x7d, 0x21, 0xda, 0xbb, 0x62, 0x15, 0x47, 0x36, 0x07, 0x67, 0x12, 0xe8,
    0x8c, 0xaa, 0x1c, 0xcd, 0x38, 0x12, 0x61, 0x18,
};

static unsigned char dsa1024_pub[] = {
    0x3c, 0x4e, 0x9c, 0x2a, 0x7f, 0x16, 0xc1, 0x25, 0xeb, 0xac, 0x78, 0x63,
    0x90, 0x14, 0x8c, 0x8b, 0xf4, 0x68, 0x43, 0x3c, 0x2d, 0xee, 0x65, 0x50,
    0x7d, 0x9c, 0x8f, 0x8c, 0x8a, 0x51, 0xd6, 0x11, 0x2b, 0x99, 0xaf, 0x1e,
    0x90, 0x97, 0xb5, 0xd3, 0xa6, 0x20, 0x25, 0xd6, 0xfe, 0x43, 0x02, 0xd5,
    0x91, 0x7d, 0xa7, 0x8c, 0xdb, 0xc9, 0x85, 0xa3, 0x36, 0x48, 0xf7, 0x68,
    0xaa, 0x60, 0xb1, 0xf7, 0x05, 0x68, 0x3a, 0xa3, 0x3f, 0xd3, 0x19, 0x82,
    0xd8, 0x82, 0x7a, 0x77, 0xfb, 0xef, 0xf4, 0x15, 0x0a, 0xeb, 0x06, 0x04,
    0x7f, 0x53, 0x07, 0x0c, 0xbc, 0xcb, 0x2d, 0x83, 0xdb, 0x3e, 0xd1, 0x28,
    0xa5, 0xa1, 0x31, 0xe0, 0x67, 0xfa, 0x50, 0xde, 0x9b, 0x07, 0x83, 0x7e,
    0x2c, 0x0b, 0xc3, 0x13, 0x50, 0x61, 0xe5, 0xad, 0xbd, 0x36, 0xb8, 0x97,
    0x4e, 0x40, 0x7d, 0xe8, 0x83, 0x0d, 0xbc, 0x4b
};

static unsigned char dsa1024_p[] = {
    0xA7, 0x3F, 0x6E, 0x85, 0xBF, 0x41, 0x6A, 0x29, 0x7D, 0xF0, 0x9F, 0x47,
    0x19, 0x30, 0x90, 0x9A, 0x09, 0x1D, 0xDA, 0x6A, 0x33, 0x1E, 0xC5, 0x3D,
    0x86, 0x96, 0xB3, 0x15, 0xE0, 0x53, 0x2E, 0x8F, 0xE0, 0x59, 0x82, 0x73,
    0x90, 0x3E, 0x75, 0x31, 0x99, 0x47, 0x7A, 0x52, 0xFB, 0x85, 0xE4, 0xD9,
    0xA6, 0x7B, 0x38, 0x9B, 0x68, 0x8A, 0x84, 0x9B, 0x87, 0xC6, 0x1E, 0xB5,
    0x7E, 0x86, 0x4B, 0x53, 0x5B, 0x59, 0xCF, 0x71, 0x65, 0x19, 0x88, 0x6E,
    0xCE, 0x66, 0xAE, 0x6B, 0x88, 0x36, 0xFB, 0xEC, 0x28, 0xDC, 0xC2, 0xD7,
    0xA5, 0xBB, 0xE5, 0x2C, 0x39, 0x26, 0x4B, 0xDA, 0x9A, 0x70, 0x18, 0x95,
    0x37, 0x95, 0x10, 0x56, 0x23, 0xF6, 0x15, 0xED, 0xBA, 0x04, 0x5E, 0xDE,
    0x39, 0x4F, 0xFD, 0xB7, 0x43, 0x1F, 0xB5, 0xA4, 0x65, 0x6F, 0xCD, 0x80,
    0x11, 0xE4, 0x70, 0x95, 0x5B, 0x50, 0xCD, 0x49,
};

static unsigned char dsa1024_q[] = {
    0xF7, 0x07, 0x31, 0xED, 0xFA, 0x6C, 0x06, 0x03, 0xD5, 0x85, 0x8A, 0x1C,
    0xAC, 0x9C, 0x65, 0xE7, 0x50, 0x66, 0x65, 0x6F,
};

static unsigned char dsa1024_g[] = {
    0x4D, 0xDF, 0x4C, 0x03, 0xA6, 0x91, 0x8A, 0xF5, 0x19, 0x6F, 0x50, 0x46,
    0x25, 0x99, 0xE5, 0x68, 0x6F, 0x30, 0xE3, 0x69, 0xE1, 0xE5, 0xB3, 0x5D,
    0x98, 0xBB, 0x28, 0x86, 0x48, 0xFC, 0xDE, 0x99, 0x04, 0x3F, 0x5F, 0x88,
    0x0C, 0x9C, 0x73, 0x24, 0x0D, 0x20, 0x5D, 0xB9, 0x2A, 0x9A, 0x3F, 0x18,
    0x96, 0x27, 0xE4, 0x62, 0x87, 0xC1, 0x7B, 0x74, 0x62, 0x53, 0xFC, 0x61,
    0x27, 0xA8, 0x7A, 0x91, 0x09, 0x9D, 0xB6, 0xF1, 0x4D, 0x9C, 0x54, 0x0F,
    0x58, 0x06, 0xEE, 0x49, 0x74, 0x07, 0xCE, 0x55, 0x7E, 0x23, 0xCE, 0x16,
    0xF6, 0xCA, 0xDC, 0x5A, 0x61, 0x01, 0x7E, 0xC9, 0x71, 0xB5, 0x4D, 0xF6,
    0xDC, 0x34, 0x29, 0x87, 0x68, 0xF6, 0x5E, 0x20, 0x93, 0xB3, 0xDB, 0xF5,
    0xE4, 0x09, 0x6C, 0x41, 0x17, 0x95, 0x92, 0xEB, 0x01, 0xB5, 0x73, 0xA5,
    0x6A, 0x7E, 0xD8, 0x32, 0xED, 0x0E, 0x02, 0xB8,
};

static unsigned char dsa2048_224_priv[] = {
    0x6c, 0xca, 0xee, 0xf6, 0xd7, 0x3b, 0x4e, 0x80,
    0xf1, 0x1c, 0x17, 0xb8, 0xe9, 0x62, 0x7c, 0x03,
    0x66, 0x35, 0xba, 0xc3, 0x94, 0x23, 0x50, 0x5e,
    0x40, 0x7e, 0x5c, 0xb7
};

static unsigned char dsa2048_224_pub[] = {
    0xa0, 0x12, 0xb3, 0xb1, 0x70, 0xb3, 0x07, 0x22,
    0x79, 0x57, 0xb7, 0xca, 0x20, 0x61, 0xa8, 0x16,
    0xac, 0x7a, 0x2b, 0x3d, 0x9a, 0xe9, 0x95, 0xa5,
    0x11, 0x9c, 0x38, 0x5b, 0x60, 0x3b, 0xf6, 0xf6,
    0xc5, 0xde, 0x4d, 0xc5, 0xec, 0xb5, 0xdf, 0xa4,
    0xa4, 0x1c, 0x68, 0x66, 0x2e, 0xb2, 0x5b, 0x63,
    0x8b, 0x7e, 0x26, 0x20, 0xba, 0x89, 0x8d, 0x07,
    0xda, 0x6c, 0x49, 0x91, 0xe7, 0x6c, 0xc0, 0xec,
    0xd1, 0xad, 0x34, 0x21, 0x07, 0x70, 0x67, 0xe4,
    0x7c, 0x18, 0xf5, 0x8a, 0x92, 0xa7, 0x2a, 0xd4,
    0x31, 0x99, 0xec, 0xb7, 0xbd, 0x84, 0xe7, 0xd3,
    0xaf, 0xb9, 0x01, 0x9f, 0x0e, 0x9d, 0xd0, 0xfb,
    0xaa, 0x48, 0x73, 0x00, 0xb1, 0x30, 0x81, 0xe3,
    0x3c, 0x90, 0x28, 0x76, 0x43, 0x6f, 0x7b, 0x03,
    0xc3, 0x45, 0x52, 0x84, 0x81, 0xd3, 0x62, 0x81,
    0x5e, 0x24, 0xfe, 0x59, 0xda, 0xc5, 0xac, 0x34,
    0x66, 0x0d, 0x4c, 0x8a, 0x76, 0xcb, 0x99, 0xa7,
    0xc7, 0xde, 0x93, 0xeb, 0x95, 0x6c, 0xd6, 0xbc,
    0x88, 0xe5, 0x8d, 0x90, 0x10, 0x34, 0x94, 0x4a,
    0x09, 0x4b, 0x01, 0x80, 0x3a, 0x43, 0xc6, 0x72,
    0xb9, 0x68, 0x8c, 0x0e, 0x01, 0xd8, 0xf4, 0xfc,
    0x91, 0xc6, 0x2a, 0x3f, 0x88, 0x02, 0x1f, 0x7b,
    0xd6, 0xa6, 0x51, 0xb1, 0xa8, 0x8f, 0x43, 0xaa,
    0x4e, 0xf2, 0x76, 0x53, 0xd1, 0x2b, 0xf8, 0xb7,
    0x09, 0x9f, 0xdf, 0x6b, 0x46, 0x10, 0x82, 0xf8,
    0xe9, 0x39, 0x10, 0x7b, 0xfd, 0x2f, 0x72, 0x10,
    0x08, 0x7d, 0x32, 0x6c, 0x37, 0x52, 0x00, 0xf1,
    0xf5, 0x1e, 0x7e, 0x74, 0xa3, 0x41, 0x31, 0x90,
    0x1b, 0xcd, 0x08, 0x63, 0x52, 0x1f, 0xf8, 0xd6,
    0x76, 0xc4, 0x85, 0x81, 0x86, 0x87, 0x36, 0xc5,
    0xe5, 0x1b, 0x16, 0xa4, 0xe3, 0x92, 0x15, 0xea,
    0x0b, 0x17, 0xc4, 0x73, 0x59, 0x74, 0xc5, 0x16
};

static unsigned char dsa2048_224_p[] = {
    0xa2, 0x9b, 0x88, 0x72, 0xce, 0x8b, 0x84, 0x23,
    0xb7, 0xd5, 0xd2, 0x1d, 0x4b, 0x02, 0xf5, 0x7e,
    0x03, 0xe9, 0xe6, 0xb8, 0xa2, 0x58, 0xdc, 0x16,
    0x61, 0x1b, 0xa0, 0x98, 0xab, 0x54, 0x34, 0x15,
    0xe4, 0x15, 0xf1, 0x56, 0x99, 0x7a, 0x3e, 0xe2,
    0x36, 0x65, 0x8f, 0xa0, 0x93, 0x26, 0x0d, 0xe3,
    0xad, 0x42, 0x2e, 0x05, 0xe0, 0x46, 0xf9, 0xec,
    0x29, 0x16, 0x1a, 0x37, 0x5f, 0x0e, 0xb4, 0xef,
    0xfc, 0xef, 0x58, 0x28, 0x5c, 0x5d, 0x39, 0xed,
    0x42, 0x5d, 0x7a, 0x62, 0xca, 0x12, 0x89, 0x6c,
    0x4a, 0x92, 0xcb, 0x19, 0x46, 0xf2, 0x95, 0x2a,
    0x48, 0x13, 0x3f, 0x07, 0xda, 0x36, 0x4d, 0x1b,
    0xdf, 0x6b, 0x0f, 0x71, 0x39, 0x98, 0x3e, 0x69,
    0x3c, 0x80, 0x05, 0x9b, 0x0e, 0xac, 0xd1, 0x47,
    0x9b, 0xa9, 0xf2, 0x85, 0x77, 0x54, 0xed, 0xe7,
    0x5f, 0x11, 0x2b, 0x07, 0xeb, 0xbf, 0x35, 0x34,
    0x8b, 0xbf, 0x3e, 0x01, 0xe0, 0x2f, 0x2d, 0x47,
    0x3d, 0xe3, 0x94, 0x53, 0xf9, 0x9d, 0xd2, 0x36,
    0x75, 0x41, 0xca, 0xca, 0x3b, 0xa0, 0x11, 0x66,
    0x34, 0x3d, 0x7b, 0x5b, 0x58, 0xa3, 0x7b, 0xd1,
    0xb7, 0x52, 0x1d, 0xb2, 0xf1, 0x3b, 0x86, 0x70,
    0x71, 0x32, 0xfe, 0x09, 0xf4, 0xcd, 0x09, 0xdc,
    0x16, 0x18, 0xfa, 0x34, 0x01, 0xeb, 0xf9, 0xcc,
    0x7b, 0x19, 0xfa, 0x94, 0xaa, 0x47, 0x20, 0x88,
    0x13, 0x3d, 0x6c, 0xb2, 0xd3, 0x5c, 0x11, 0x79,
    0xc8, 0xc8, 0xff, 0x36, 0x87, 0x58, 0xd5, 0x07,
    0xd9, 0xf9, 0xa1, 0x7d, 0x46, 0xc1, 0x10, 0xfe,
    0x31, 0x44, 0xce, 0x9b, 0x02, 0x2b, 0x42, 0xe4,
    0x19, 0xeb, 0x4f, 0x53, 0x88, 0x61, 0x3b, 0xfc,
    0x3e, 0x26, 0x24, 0x1a, 0x43, 0x2e, 0x87, 0x06,
    0xbc, 0x58, 0xef, 0x76, 0x11, 0x72, 0x78, 0xde,
    0xab, 0x6c, 0xf6, 0x92, 0x61, 0x82, 0x91, 0xb7
};

static unsigned char dsa2048_224_q[] = {
    0xa3, 0xbf, 0xd9, 0xab, 0x78, 0x84, 0x79, 0x4e,
    0x38, 0x34, 0x50, 0xd5, 0x89, 0x1d, 0xc1, 0x8b,
    0x65, 0x15, 0x7b, 0xdc, 0xfc, 0xda, 0xc5, 0x15,
    0x18, 0x90, 0x28, 0x67
};

static unsigned char dsa2048_224_g[] = {
    0x68, 0x19, 0x27, 0x88, 0x69, 0xc7, 0xfd, 0x3d,
    0x2d, 0x7b, 0x77, 0xf7, 0x7e, 0x81, 0x50, 0xd9,
    0xad, 0x43, 0x3b, 0xea, 0x3b, 0xa8, 0x5e, 0xfc,
    0x80, 0x41, 0x5a, 0xa3, 0x54, 0x5f, 0x78, 0xf7,
    0x22, 0x96, 0xf0, 0x6c, 0xb1, 0x9c, 0xed, 0xa0,
    0x6c, 0x94, 0xb0, 0x55, 0x1c, 0xfe, 0x6e, 0x6f,
    0x86, 0x3e, 0x31, 0xd1, 0xde, 0x6e, 0xed, 0x7d,
    0xab, 0x8b, 0x0c, 0x9d, 0xf2, 0x31, 0xe0, 0x84,
    0x34, 0xd1, 0x18, 0x4f, 0x91, 0xd0, 0x33, 0x69,
    0x6b, 0xb3, 0x82, 0xf8, 0x45, 0x5e, 0x98, 0x88,
    0xf5, 0xd3, 0x1d, 0x47, 0x84, 0xec, 0x40, 0x12,
    0x02, 0x46, 0xf4, 0xbe, 0xa6, 0x17, 0x94, 0xbb,
    0xa5, 0x86, 0x6f, 0x09, 0x74, 0x64, 0x63, 0xbd,
    0xf8, 0xe9, 0xe1, 0x08, 0xcd, 0x95, 0x29, 0xc3,
    0xd0, 0xf6, 0xdf, 0x80, 0x31, 0x6e, 0x2e, 0x70,
    0xaa, 0xeb, 0x1b, 0x26, 0xcd, 0xb8, 0xad, 0x97,
    0xbc, 0x3d, 0x28, 0x7e, 0x0b, 0x8d, 0x61, 0x6c,
    0x42, 0xe6, 0x5b, 0x87, 0xdb, 0x20, 0xde, 0xb7,
    0x00, 0x5b, 0xc4, 0x16, 0x74, 0x7a, 0x64, 0x70,
    0x14, 0x7a, 0x68, 0xa7, 0x82, 0x03, 0x88, 0xeb,
    0xf4, 0x4d, 0x52, 0xe0, 0x62, 0x8a, 0xf9, 0xcf,
    0x1b, 0x71, 0x66, 0xd0, 0x34, 0x65, 0xf3, 0x5a,
    0xcc, 0x31, 0xb6, 0x11, 0x0c, 0x43, 0xda, 0xbc,
    0x7c, 0x5d, 0x59, 0x1e, 0x67, 0x1e, 0xaf, 0x7c,
    0x25, 0x2c, 0x1c, 0x14, 0x53, 0x36, 0xa1, 0xa4,
    0xdd, 0xf1, 0x32, 0x44, 0xd5, 0x5e, 0x83, 0x56,
    0x80, 0xca, 0xb2, 0x53, 0x3b, 0x82, 0xdf, 0x2e,
    0xfe, 0x55, 0xec, 0x18, 0xc1, 0xe6, 0xcd, 0x00,
    0x7b, 0xb0, 0x89, 0x75, 0x8b, 0xb1, 0x7c, 0x2c,
    0xbe, 0x14, 0x44, 0x1b, 0xd0, 0x93, 0xae, 0x66,
    0xe5, 0x97, 0x6d, 0x53, 0x73, 0x3f, 0x4f, 0xa3,
    0x26, 0x97, 0x01, 0xd3, 0x1d, 0x23, 0xd4, 0x67
};

/* dsa 3072 */
static unsigned char dsa3072_priv[] = {
	0x1B, 0x28, 0x82, 0xDA, 0x37, 0xAC, 0xE5, 0x05,
    0x00, 0x0D, 0x3A, 0x4C, 0x86, 0x54, 0xEE, 0x2B,
	0x09, 0xA7, 0x88, 0x1F, 0xB3, 0x99, 0x56, 0x87,
	0x77, 0x7F, 0x67, 0xFD, 0x6F, 0x4D, 0x2F, 0xEC
};

static unsigned char dsa3072_pub[] = {
	0x95, 0x32, 0x25, 0xEB, 0x53, 0x7F, 0x26, 0x52,
    0xED, 0xF0, 0xC1, 0x3C, 0x89, 0x87, 0x3E, 0xAD,
	0x75, 0xE7, 0x76, 0xB0, 0x52, 0x59, 0x0C, 0x1B,
	0x13, 0x8F, 0x36, 0xB1, 0xE7, 0x53, 0x10, 0xA2,
	0x4E, 0x9A, 0x6F, 0xA4, 0x84, 0x52, 0x62, 0xDB,
	0xDE, 0xAA, 0xA2, 0x06, 0x9D, 0x3C, 0xF5, 0x4E,
	0xCA, 0x47, 0x9C, 0x9B, 0x40, 0x73, 0x1E, 0xAA,
	0x68, 0x00, 0x89, 0x2B, 0x8F, 0xEA, 0xAE, 0x3C,
	0x44, 0xC3, 0x55, 0xDE, 0x11, 0xFC, 0x75, 0x5C,
	0x75, 0x5C, 0xBC, 0x66, 0xE9, 0xB4, 0x37, 0x74,
	0xFD, 0x55, 0x8D, 0x05, 0x25, 0x5B, 0x7F, 0x34,
	0xEC, 0x33, 0x9E, 0x72, 0x1B, 0x9F, 0x9A, 0x93,
	0x04, 0x72, 0x90, 0xDF, 0x0C, 0xAA, 0x52, 0x32,
	0xC9, 0x9F, 0xC6, 0x21, 0xA3, 0x75, 0xB4, 0xAF,
	0xA6, 0x57, 0x6B, 0xFD, 0xB8, 0xDE, 0x78, 0xB3,
	0x03, 0x51, 0x11, 0xE5, 0xAE, 0x81, 0x22, 0xEB,
	0xDF, 0xFC, 0x21, 0x63, 0x60, 0xD8, 0x4F, 0x92,
	0xF1, 0x77, 0x09, 0x9E, 0xCD, 0x82, 0xB1, 0x0B,
	0x07, 0x04, 0xAB, 0x19, 0x65, 0x98, 0x9B, 0xC9,
	0x2A, 0x74, 0xA8, 0xEF, 0xED, 0x71, 0x2F, 0x9B,
	0x50, 0xA2, 0xB8, 0xF3, 0x2B, 0x60, 0x68, 0xE5,
	0xC7, 0xCB, 0xA7, 0x15, 0x4E, 0x56, 0x18, 0x6F,
	0x2C, 0xDF, 0x39, 0x8B, 0x24, 0x09, 0x04, 0x71,
	0x67, 0x6D, 0x88, 0x8F, 0x85, 0xAB, 0xF4, 0x69,
	0xE4, 0xCD, 0x66, 0x8D, 0xE6, 0x11, 0xE0, 0x3F,
	0x16, 0xA3, 0xC9, 0xC8, 0x19, 0xE3, 0xF5, 0x36,
	0x86, 0x3F, 0xB3, 0xD9, 0xBF, 0x94, 0x7E, 0x35,
	0xC5, 0x1E, 0xC6, 0x84, 0x6A, 0x44, 0x01, 0x26,
	0xA0, 0xFB, 0xA3, 0x7C, 0xE3, 0x3A, 0x99, 0x0C,
	0x66, 0x6B, 0x27, 0x37, 0xB7, 0x25, 0x6A, 0x4E,
	0x6F, 0x90, 0x43, 0x19, 0x91, 0x2F, 0x82, 0xD0,
	0x1A, 0x62, 0x87, 0x30, 0xF7, 0x93, 0x15, 0xEB,
	0x4F, 0xFE, 0xF8, 0xB8, 0xB6, 0x2B, 0xCA, 0xF0,
	0xEE, 0xA5, 0x65, 0x4B, 0x3B, 0x5A, 0xE8, 0xDC,
	0x79, 0xBE, 0xF0, 0xEB, 0x4E, 0x7B, 0xA9, 0xFB,
	0xC2, 0x83, 0x91, 0xE9, 0x4D, 0x9F, 0xEE, 0x04,
	0x0E, 0xC9, 0x6A, 0xB4, 0x11, 0x7A, 0x18, 0xD1,
	0xC0, 0x5C, 0x70, 0xA7, 0x68, 0xFB, 0xAB, 0x41,
	0xEF, 0xA8, 0x73, 0x20, 0xAE, 0xCC, 0x5E, 0xB4,
	0x73, 0x19, 0xF1, 0x7E, 0x7B, 0x0A, 0x0F, 0x5F,
	0x4D, 0x1B, 0xDB, 0xDB, 0x92, 0x98, 0x36, 0x92,
	0xFF, 0x17, 0x17, 0xC1, 0xDD, 0x1F, 0x85, 0x27,
	0x74, 0xC8, 0xDE, 0x61, 0x72, 0x25, 0xFA, 0xAA,
	0x75, 0xAA, 0xD4, 0x53, 0x63, 0x54, 0x77, 0xCA,
	0x69, 0xD6, 0x22, 0x3A, 0x18, 0xD7, 0xFD, 0x26,
	0x66, 0x91, 0xC2, 0xA1, 0xAB, 0x85, 0x19, 0x3E,
	0xF7, 0x1E, 0xAD, 0xB3, 0x5A, 0x87, 0x76, 0x98,
	0xF0, 0xAD, 0x0C, 0xE2, 0x70, 0xD2, 0x44, 0x29
};

static unsigned char dsa3072_p[] = {
	0xA8, 0x50, 0x7A, 0x44, 0x32, 0xCA, 0x8F, 0xCD,
    0xCC, 0x3B, 0xC4, 0x1C, 0xA7, 0x16, 0x30, 0x68,
	0x41, 0xAC, 0x8E, 0xE7, 0x39, 0xFB, 0xC5, 0x86,
	0xCA, 0x50, 0x66, 0x49, 0x33, 0x36, 0x2D, 0x9D,
	0xA9, 0x38, 0x15, 0x85, 0x34, 0xD9, 0x62, 0xAF,
	0x31, 0xA6, 0x1C, 0x3A, 0x7B, 0x96, 0x31, 0xF5,
	0x47, 0x02, 0x3B, 0x0A, 0xE7, 0x38, 0xED, 0x91,
	0x27, 0xEA, 0x40, 0x4A, 0xA5, 0xDA, 0xF7, 0xA1,
	0x30, 0x48, 0x8A, 0xB6, 0xD2, 0x5E, 0x06, 0x9A,
	0x0F, 0x8D, 0x3E, 0x3A, 0x49, 0xF7, 0x8B, 0xB1,
	0x06, 0xA1, 0x61, 0x6B, 0xB2, 0x5C, 0x64, 0x1E,
	0x02, 0x80, 0xF2, 0x3E, 0xB8, 0x81, 0xC1, 0x71,
	0x0F, 0x31, 0x85, 0xB2, 0x13, 0xCB, 0xBF, 0xC4,
	0x75, 0x2D, 0x9D, 0x92, 0x0C, 0x9D, 0xDD, 0x2B,
	0xA8, 0x91, 0x7A, 0xEF, 0x64, 0x81, 0xBC, 0x72,
	0x59, 0x73, 0xA2, 0x2C, 0xCD, 0x24, 0x31, 0x87,
	0xFC, 0x5E, 0xEC, 0xF1, 0x6B, 0x41, 0xA8, 0x40,
	0xD7, 0xDE, 0x94, 0x55, 0x7E, 0x75, 0xB2, 0x52,
	0xD4, 0xF4, 0x17, 0xCF, 0xFF, 0xE9, 0x52, 0x66,
	0x4C, 0x62, 0xE1, 0xB9, 0x18, 0x15, 0x36, 0x02,
	0xF7, 0xEC, 0xF3, 0x43, 0x84, 0x93, 0xF8, 0x7E,
	0x93, 0xDE, 0x83, 0x39, 0x85, 0x93, 0xE9, 0xF4,
	0x8E, 0x66, 0xFD, 0x36, 0xD3, 0x8B, 0x48, 0x44,
	0xEB, 0x3F, 0x9C, 0x56, 0xE1, 0x4C, 0x1C, 0x0B,
	0x11, 0xAB, 0xA9, 0xAA, 0x80, 0xDC, 0x75, 0x47,
	0x3D, 0xF1, 0x1B, 0x1C, 0x00, 0x85, 0xFC, 0x23,
	0x49, 0x6D, 0xD0, 0x9D, 0x89, 0x79, 0xA2, 0xBD,
	0xF6, 0xB5, 0xB4, 0x9D, 0xDB, 0x05, 0x17, 0xF3,
	0xD3, 0x2D, 0x38, 0xCD, 0xB4, 0xA1, 0x42, 0xE5,
	0x17, 0xB0, 0xF7, 0x3F, 0xD4, 0x84, 0xD6, 0x36,
	0x62, 0x64, 0x9C, 0xD5, 0x0A, 0xBD, 0xB2, 0xD9,
	0x50, 0x16, 0xC1, 0x49, 0x17, 0x23, 0xCE, 0xD0,
	0xC0, 0x29, 0x76, 0x40, 0xE0, 0x19, 0x8C, 0x75,
	0xA2, 0x28, 0x59, 0xC1, 0xC6, 0x63, 0x67, 0xEA,
	0x80, 0xF3, 0x16, 0xD1, 0x71, 0x0E, 0x44, 0xF8,
	0x00, 0xAE, 0x64, 0xB9, 0x7B, 0x1F, 0x6D, 0xB9,
	0x4C, 0x89, 0x36, 0x0C, 0x52, 0x9D, 0xEC, 0xE2,
	0x2E, 0x6C, 0x95, 0xC4, 0x7F, 0xFE, 0xCE, 0x54,
	0x2C, 0xA7, 0x03, 0xCD, 0x7A, 0xBE, 0xD0, 0x76,
	0x41, 0x5D, 0x64, 0xA8, 0xA3, 0x0C, 0x5B, 0x02,
	0x68, 0x6F, 0xA0, 0x34, 0xF8, 0x31, 0xF7, 0x4D,
	0x8E, 0x3B, 0x66, 0x7F, 0x23, 0xAC, 0x22, 0xD4,
	0x84, 0x1A, 0x4C, 0xC2, 0x72, 0xDB, 0x68, 0xFD,
	0xAE, 0x60, 0xAC, 0x05, 0xCB, 0xE7, 0x67, 0x05,
	0x86, 0x72, 0xF1, 0xB1, 0x33, 0x68, 0xF6, 0x70,
	0xDD, 0x99, 0xFE, 0x35, 0x1D, 0x98, 0x00, 0x74,
	0x3C, 0xEC, 0x39, 0x6E, 0x3C, 0x91, 0xB3, 0x80,
	0x58, 0x19, 0x38, 0xEC, 0xD7, 0xAC, 0x69, 0x53
};

static unsigned char dsa3072_q[] = {
	0x95, 0x76, 0xFA, 0x68, 0x5A, 0x13, 0x06, 0x2F,
        0xF8, 0xEA, 0x34, 0x4B, 0xCC, 0xF4, 0x14, 0xAA,
	0x14, 0x92, 0x73, 0xB6, 0xC7, 0x8D, 0xAF, 0x4D,
	0xE4, 0xEA, 0x5E, 0xFB, 0xF3, 0xA6, 0x84, 0xD5
};

static unsigned char dsa3072_g[] = {
	0x8B, 0xCA, 0xEB, 0xB9, 0x97, 0x57, 0x85, 0xB4,
    0x06, 0x45, 0xC0, 0x5D, 0x94, 0xB0, 0x84, 0x41,
	0x72, 0x0B, 0x0A, 0x6F, 0xF6, 0x82, 0x1B, 0x8D,
	0xB1, 0xDC, 0xB2, 0xCE, 0x07, 0x2E, 0x3E, 0xBE,
	0x07, 0xFB, 0xBC, 0x56, 0x4E, 0x6D, 0x59, 0xCB,
	0xE1, 0x07, 0x66, 0xE9, 0x4C, 0xE4, 0x16, 0xCF,
	0xE6, 0xFF, 0x80, 0xF0, 0xFD, 0x09, 0xA0, 0x13,
	0x0A, 0x24, 0xE6, 0x85, 0x00, 0x14, 0x37, 0x46,
	0x18, 0xC2, 0xC3, 0xAD, 0x8B, 0x31, 0x26, 0xF1,
	0x86, 0x42, 0x7C, 0xF8, 0x84, 0x00, 0xE9, 0x52,
	0x2A, 0x30, 0x8B, 0x4E, 0x44, 0x35, 0xE0, 0xB1,
	0xCA, 0xBA, 0xBB, 0x1C, 0xA7, 0xC7, 0x6F, 0x8D,
	0x85, 0x24, 0x9D, 0x8B, 0x1F, 0x12, 0x1D, 0xE0,
	0x25, 0x0F, 0x8E, 0x82, 0xD8, 0x76, 0x06, 0x96,
	0xDC, 0xE0, 0x98, 0xAD, 0x4F, 0xD8, 0x52, 0xE9,
	0x54, 0x44, 0x61, 0x2B, 0xC8, 0xFC, 0x2A, 0xAC,
	0x4A, 0x01, 0x65, 0xFE, 0x81, 0xCB, 0xE7, 0xDB,
	0x93, 0x6A, 0x1A, 0x5F, 0xC5, 0xDB, 0x2C, 0x8F,
	0x77, 0x94, 0xCB, 0xFD, 0x8F, 0xEF, 0xF0, 0x77,
	0x46, 0xA9, 0xBB, 0xCD, 0x44, 0xE3, 0xB5, 0x98,
	0xFA, 0x61, 0x21, 0x7F, 0x6C, 0x8D, 0x4F, 0x4E,
	0x4F, 0x1B, 0xFD, 0x4F, 0x2F, 0xFE, 0xB0, 0x7B,
	0x99, 0xD6, 0x4A, 0xF6, 0xE1, 0x7E, 0xD5, 0xF2,
	0xE2, 0xA0, 0xB4, 0x3B, 0x4E, 0xF3, 0xAE, 0xAC,
	0x06, 0xD6, 0x50, 0x1A, 0xBB, 0xE9, 0x52, 0xAB,
	0x57, 0x51, 0x3A, 0x35, 0x13, 0xD0, 0x41, 0x9C,
	0x22, 0xAA, 0x39, 0x18, 0x46, 0xAD, 0xC7, 0x32,
	0x48, 0x0C, 0x2A, 0xF1, 0x87, 0x1F, 0x5B, 0xB4,
	0x1F, 0xE2, 0x9A, 0x60, 0x61, 0xA2, 0xFD, 0x5A,
	0xCB, 0x03, 0x26, 0xA5, 0x57, 0x7D, 0xE8, 0x08,
	0xFA, 0x71, 0x09, 0x4D, 0x0E, 0x29, 0xFF, 0xE4,
	0xC0, 0x9A, 0x78, 0x5C, 0xEF, 0x1D, 0x39, 0xAF,
	0x28, 0x18, 0xC6, 0x47, 0x4F, 0x64, 0xBD, 0x68,
	0x76, 0x95, 0xE1, 0x4B, 0xC9, 0xFB, 0xB6, 0xBF,
	0xF0, 0x96, 0xE2, 0xD7, 0x5E, 0x32, 0x66, 0x34,
	0x1B, 0xB6, 0x36, 0xA1, 0xDA, 0x33, 0x5A, 0x29,
	0xA2, 0x18, 0x96, 0xC5, 0x78, 0x76, 0x53, 0x9D,
	0x02, 0xC9, 0x50, 0x49, 0xE7, 0xE6, 0xB7, 0x5C,
	0x88, 0xD7, 0x2B, 0x14, 0x01, 0x1D, 0x4B, 0x81,
	0xA1, 0x11, 0x8F, 0xB3, 0x45, 0x09, 0xBF, 0x87,
	0x09, 0xF4, 0x3C, 0xF7, 0xC6, 0x75, 0x49, 0xBC,
	0x0F, 0xE7, 0x3E, 0x8E, 0x84, 0x61, 0x6B, 0xA3,
	0x08, 0x69, 0x98, 0x70, 0x3F, 0x75, 0x73, 0x0E,
	0x33, 0xCB, 0xBC, 0x21, 0xFD, 0x2A, 0xC6, 0xF2,
	0x06, 0x57, 0x00, 0x57, 0xF7, 0x9D, 0x26, 0x7D,
	0xAF, 0x59, 0x0B, 0xBD, 0xA0, 0xFB, 0x8B, 0xB0,
	0x7F, 0x78, 0xD3, 0xF7, 0x85, 0x26, 0x20, 0x60,
	0xB0, 0xC7, 0xF5, 0xB6, 0x70, 0xFE, 0x28, 0xB1
};

/* dsa 2048-256 */
static unsigned char dsa2048_256_priv[] = {
	0x69, 0xC7, 0x54, 0x8C, 0x21, 0xD0, 0xDF, 0xEA,
	0x6B, 0x9A, 0x51, 0xC9, 0xEA, 0xD4, 0xE2, 0x7C,
	0x33, 0xD3, 0xB3, 0xF1, 0x80, 0x31, 0x6E, 0x5B,
	0xCA, 0xB9, 0x2C, 0x93, 0x3F, 0x0E, 0x4D, 0xBC
};

static unsigned char dsa2048_256_pub[] = {
	0x66, 0x70, 0x98, 0xC6, 0x54, 0x42, 0x6C, 0x78,
	0xD7, 0xF8, 0x20, 0x1E, 0xAC, 0x6C, 0x20, 0x3E,
	0xF0, 0x30, 0xD4, 0x36, 0x05, 0x03, 0x2C, 0x2F,
	0x1F, 0xA9, 0x37, 0xE5, 0x23, 0x7D, 0xBD, 0x94,
	0x9F, 0x34, 0xA0, 0xA2, 0x56, 0x4F, 0xE1, 0x26,
	0xDC, 0x8B, 0x71, 0x5C, 0x51, 0x41, 0x80, 0x2C,
	0xE0, 0x97, 0x9C, 0x82, 0x46, 0x46, 0x3C, 0x40,
	0xE6, 0xB6, 0xBD, 0xAA, 0x25, 0x13, 0xFA, 0x61,
	0x17, 0x28, 0x71, 0x6C, 0x2E, 0x4F, 0xD5, 0x3B,
	0xC9, 0x5B, 0x89, 0xE6, 0x99, 0x49, 0xD9, 0x65,
	0x12, 0xE8, 0x73, 0xB9, 0xC8, 0xF8, 0xDF, 0xD4,
	0x99, 0xCC, 0x31, 0x28, 0x82, 0x56, 0x1A, 0xDE,
	0xCB, 0x31, 0xF6, 0x58, 0xE9, 0x34, 0xC0, 0xC1,
	0x97, 0xF2, 0xC4, 0xD9, 0x6B, 0x05, 0xCB, 0xAD,
	0x67, 0x38, 0x1E, 0x7B, 0x76, 0x88, 0x91, 0xE4,
	0xDA, 0x38, 0x43, 0xD2, 0x4D, 0x94, 0xCD, 0xFB,
	0x51, 0x26, 0xE9, 0xB8, 0xBF, 0x21, 0xE8, 0x35,
	0x8E, 0xE0, 0xE0, 0xA3, 0x0E, 0xF1, 0x3F, 0xD6,
	0xA6, 0x64, 0xC0, 0xDC, 0xE3, 0x73, 0x1F, 0x7F,
	0xB4, 0x9A, 0x48, 0x45, 0xA4, 0xFD, 0x82, 0x54,
	0x68, 0x79, 0x72, 0xA2, 0xD3, 0x82, 0x59, 0x9C,
	0x9B, 0xAC, 0x4E, 0x0E, 0xD7, 0x99, 0x81, 0x93,
	0x07, 0x89, 0x13, 0x03, 0x25, 0x58, 0x13, 0x49,
	0x76, 0x41, 0x0B, 0x89, 0xD2, 0xC1, 0x71, 0xD1,
	0x23, 0xAC, 0x35, 0xFD, 0x97, 0x72, 0x19, 0x59,
	0x7A, 0xA7, 0xD1, 0x5C, 0x1A, 0x9A, 0x42, 0x8E,
	0x59, 0x19, 0x4F, 0x75, 0xC7, 0x21, 0xEB, 0xCB,
	0xCF, 0xAE, 0x44, 0x69, 0x6A, 0x49, 0x9A, 0xFA,
	0x74, 0xE0, 0x42, 0x99, 0xF1, 0x32, 0x02, 0x66,
	0x01, 0x63, 0x8C, 0xB8, 0x7A, 0xB7, 0x91, 0x90,
	0xD4, 0xA0, 0x98, 0x63, 0x15, 0xDA, 0x8E, 0xEC,
	0x65, 0x61, 0xC9, 0x38, 0x99, 0x6B, 0xEA, 0xDF
};

static unsigned char dsa2048_256_p[] = {
	0x9D, 0xB6, 0xFB, 0x59, 0x51, 0xB6, 0x6B, 0xB6,
	0xFE, 0x1E, 0x14, 0x0F, 0x1D, 0x2C, 0xE5, 0x50,
	0x23, 0x74, 0x16, 0x1F, 0xD6, 0x53, 0x8D, 0xF1,
	0x64, 0x82, 0x18, 0x64, 0x2F, 0x0B, 0x5C, 0x48,
	0xC8, 0xF7, 0xA4, 0x1A, 0xAD, 0xFA, 0x18, 0x73,
	0x24, 0xB8, 0x76, 0x74, 0xFA, 0x18, 0x22, 0xB0,
	0x0F, 0x1E, 0xCF, 0x81, 0x36, 0x94, 0x3D, 0x7C,
	0x55, 0x75, 0x72, 0x64, 0xE5, 0xA1, 0xA4, 0x4F,
	0xFE, 0x01, 0x2E, 0x99, 0x36, 0xE0, 0x0C, 0x1D,
	0x3E, 0x93, 0x10, 0xB0, 0x1C, 0x7D, 0x17, 0x98,
	0x05, 0xD3, 0x05, 0x8B, 0x2A, 0x9F, 0x4B, 0xB6,
	0xF9, 0x71, 0x6B, 0xFE, 0x61, 0x17, 0xC6, 0xB5,
	0xB3, 0xCC, 0x4D, 0x9B, 0xE3, 0x41, 0x10, 0x4A,
	0xD4, 0xA8, 0x0A, 0xD6, 0xC9, 0x4E, 0x00, 0x5F,
	0x4B, 0x99, 0x3E, 0x14, 0xF0, 0x91, 0xEB, 0x51,
	0x74, 0x3B, 0xF3, 0x30, 0x50, 0xC3, 0x8D, 0xE2,
	0x35, 0x56, 0x7E, 0x1B, 0x34, 0xC3, 0xD6, 0xA5,
	0xC0, 0xCE, 0xAA, 0x1A, 0x0F, 0x36, 0x82, 0x13,
	0xC3, 0xD1, 0x98, 0x43, 0xD0, 0xB4, 0xB0, 0x9D,
	0xCB, 0x9F, 0xC7, 0x2D, 0x39, 0xC8, 0xDE, 0x41,
	0xF1, 0xBF, 0x14, 0xD4, 0xBB, 0x45, 0x63, 0xCA,
	0x28, 0x37, 0x16, 0x21, 0xCA, 0xD3, 0x32, 0x4B,
	0x6A, 0x2D, 0x39, 0x21, 0x45, 0xBE, 0xBF, 0xAC,
	0x74, 0x88, 0x05, 0x23, 0x6F, 0x5C, 0xA2, 0xFE,
	0x92, 0xB8, 0x71, 0xCD, 0x8F, 0x9C, 0x36, 0xD3,
	0x29, 0x2B, 0x55, 0x09, 0xCA, 0x8C, 0xAA, 0x77,
	0xA2, 0xAD, 0xFC, 0x7B, 0xFD, 0x77, 0xDD, 0xA6,
	0xF7, 0x11, 0x25, 0xA7, 0x45, 0x6F, 0xEA, 0x15,
	0x3E, 0x43, 0x32, 0x56, 0xA2, 0x26, 0x1C, 0x6A,
	0x06, 0xED, 0x36, 0x93, 0x79, 0x7E, 0x79, 0x95,
	0xFA, 0xD5, 0xAA, 0xBB, 0xCF, 0xBE, 0x3E, 0xDA,
	0x27, 0x41, 0xE3, 0x75, 0x40, 0x4A, 0xE2, 0x5B
};

static unsigned char dsa2048_256_q[] = {
	0xF2, 0xC3, 0x11, 0x93, 0x74, 0xCE, 0x76, 0xC9,
	0x35, 0x69, 0x90, 0xB4, 0x65, 0x37, 0x4A, 0x17,
	0xF2, 0x3F, 0x9E, 0xD3, 0x50, 0x89, 0xBD, 0x96,
	0x9F, 0x61, 0xC6, 0xDD, 0xE9, 0x99, 0x8C, 0x1F
};

static unsigned char dsa2048_256_g[] = {
	0x5C, 0x7F, 0xF6, 0xB0, 0x6F, 0x8F, 0x14, 0x3F,
	0xE8, 0x28, 0x84, 0x33, 0x49, 0x3E, 0x47, 0x69,
	0xC4, 0xD9, 0x88, 0xAC, 0xE5, 0xBE, 0x25, 0xA0,
	0xE2, 0x48, 0x09, 0x67, 0x07, 0x16, 0xC6, 0x13,
	0xD7, 0xB0, 0xCE, 0xE6, 0x93, 0x2F, 0x8F, 0xAA,
	0x7C, 0x44, 0xD2, 0xCB, 0x24, 0x52, 0x3D, 0xA5,
	0x3F, 0xBE, 0x4F, 0x6E, 0xC3, 0x59, 0x58, 0x92,
	0xD1, 0xAA, 0x58, 0xC4, 0x32, 0x8A, 0x06, 0xC4,
	0x6A, 0x15, 0x66, 0x2E, 0x7E, 0xAA, 0x70, 0x3A,
	0x1D, 0xEC, 0xF8, 0xBB, 0xB2, 0xD0, 0x5D, 0xBE,
	0x2E, 0xB9, 0x56, 0xC1, 0x42, 0xA3, 0x38, 0x66,
	0x1D, 0x10, 0x46, 0x1C, 0x0D, 0x13, 0x54, 0x72,
	0x08, 0x50, 0x57, 0xF3, 0x49, 0x43, 0x09, 0xFF,
	0xA7, 0x3C, 0x61, 0x1F, 0x78, 0xB3, 0x2A, 0xDB,
	0xB5, 0x74, 0x0C, 0x36, 0x1C, 0x9F, 0x35, 0xBE,
	0x90, 0x99, 0x7D, 0xB2, 0x01, 0x4E, 0x2E, 0xF5,
	0xAA, 0x61, 0x78, 0x2F, 0x52, 0xAB, 0xEB, 0x8B,
	0xD6, 0x43, 0x2C, 0x4D, 0xD0, 0x97, 0xBC, 0x54,
	0x23, 0xB2, 0x85, 0xDA, 0xFB, 0x60, 0xDC, 0x36,
	0x4E, 0x81, 0x61, 0xF4, 0xA2, 0xA3, 0x5A, 0xCA,
	0x3A, 0x10, 0xB1, 0xC4, 0xD2, 0x03, 0xCC, 0x76,
	0xA4, 0x70, 0xA3, 0x3A, 0xFD, 0xCB, 0xDD, 0x92,
	0x95, 0x98, 0x59, 0xAB, 0xD8, 0xB5, 0x6E, 0x17,
	0x25, 0x25, 0x2D, 0x78, 0xEA, 0xC6, 0x6E, 0x71,
	0xBA, 0x9A, 0xE3, 0xF1, 0xDD, 0x24, 0x87, 0x19,
	0x98, 0x74, 0x39, 0x3C, 0xD4, 0xD8, 0x32, 0x18,
	0x68, 0x00, 0x65, 0x47, 0x60, 0xE1, 0xE3, 0x4C,
	0x09, 0xE4, 0xD1, 0x55, 0x17, 0x9F, 0x9E, 0xC0,
	0xDC, 0x44, 0x73, 0xF9, 0x96, 0xBD, 0xCE, 0x6E,
	0xED, 0x1C, 0xAB, 0xED, 0x8B, 0x6F, 0x11, 0x6F,
	0x7A, 0xD9, 0xCF, 0x50, 0x5D, 0xF0, 0xF9, 0x98,
	0xE3, 0x4A, 0xB2, 0x75, 0x14, 0xB0, 0xFF, 0xE7
};

typedef struct testdsa_st {
    unsigned char *priv;
    unsigned char *pub;
    unsigned char *p;
    unsigned char *g;
    unsigned char *q;
    int priv_l;
    int pub_l;
    int p_l;
    int g_l;
    int q_l;
} testdsa;

#define set_dsa_ptr(st, bits)                 \
    do                                        \
    {                                         \
        st.priv = dsa##bits##_priv;           \
        st.pub = dsa##bits##_pub;             \
        st.p = dsa##bits##_p;                 \
        st.g = dsa##bits##_g;                 \
        st.q = dsa##bits##_q;                 \
        st.priv_l = sizeof(dsa##bits##_priv); \
        st.pub_l = sizeof(dsa##bits##_pub);   \
        st.p_l = sizeof(dsa##bits##_p);       \
        st.g_l = sizeof(dsa##bits##_g);       \
        st.q_l = sizeof(dsa##bits##_q);       \
    } while (0)

EVP_PKEY *get_dsa(int dsa_bits, int q_size) {
    EVP_PKEY *pkey = NULL;
    BIGNUM *priv_key, *pub_key, *p, *q, *g;
    EVP_PKEY_CTX *pctx;
    testdsa dsa_t;
    OSSL_PARAM_BLD *tmpl = NULL;
    OSSL_PARAM *params = NULL;

    switch (dsa_bits) {
    case 1024:
        set_dsa_ptr(dsa_t, 1024);
        break;
    case 2048:
	if (q_size == 256)
	    set_dsa_ptr(dsa_t, 2048_256);
	else
	    set_dsa_ptr(dsa_t, 2048_224);
        break;
    case 3072:
        set_dsa_ptr(dsa_t, 3072);
        break;
    case 4096:
        dsa_t.priv = Priv_key;
        dsa_t.pub = Pub_key;
        dsa_t.p = BN_p;
        dsa_t.g = BN_g;
        dsa_t.q = BN_q;
        dsa_t.priv_l = sizeof(Priv_key);
        dsa_t.pub_l = sizeof(Pub_key);
        dsa_t.p_l = sizeof(BN_p);
        dsa_t.g_l = sizeof(BN_g);
        dsa_t.q_l = sizeof(BN_q);
        break;
    default:
        return NULL;
    }

    if ((pctx = EVP_PKEY_CTX_new_from_name(NULL, "DSA", NULL)) == NULL)
        return NULL;

    priv_key = BN_bin2bn(dsa_t.priv, dsa_t.priv_l, NULL);
    pub_key = BN_bin2bn(dsa_t.pub, dsa_t.pub_l, NULL);
    p = BN_bin2bn(dsa_t.p, dsa_t.p_l, NULL);
    q = BN_bin2bn(dsa_t.q, dsa_t.q_l, NULL);
    g = BN_bin2bn(dsa_t.g, dsa_t.g_l, NULL);
    if (priv_key == NULL || pub_key == NULL || p == NULL || q == NULL || g == NULL) {
        goto err;
    }
    if ((tmpl = OSSL_PARAM_BLD_new()) == NULL 
        || !OSSL_PARAM_BLD_push_BN(tmpl, "p", p)
        || !OSSL_PARAM_BLD_push_BN(tmpl, "q", q)
        || !OSSL_PARAM_BLD_push_BN(tmpl, "g", g)
        || !OSSL_PARAM_BLD_push_BN(tmpl, "priv", priv_key)
        || !OSSL_PARAM_BLD_push_BN(tmpl, "pub", pub_key)
        || (params = OSSL_PARAM_BLD_to_param(tmpl)) == NULL)
        goto err;

    if (EVP_PKEY_fromdata_init(pctx) <= 0
        || EVP_PKEY_fromdata(pctx, &pkey, EVP_PKEY_KEYPAIR, params) <= 0)
        pkey = NULL;
err:
    OSSL_PARAM_free(params);
    OSSL_PARAM_BLD_free(tmpl);
    BN_free(priv_key);
    BN_free(pub_key);
    BN_free(p);
    BN_free(q);
    BN_free(g);
    EVP_PKEY_CTX_free(pctx);
    return pkey;
}
#endif

/******************************************************************************
* function:
*           run_dsa(void *args)
*
* @param args [IN] - the test parameters
*
* description:
*   The function is design to test DSA sign and verify using qat engine.
*
*******************************************************************************/
static int run_dsa(void *args)
{
    TEST_PARAMS *temp_args = (TEST_PARAMS *)args;
    int size = temp_args->size;
    int ret = 0;
#ifndef QAT_OPENSSL_PROVIDER
    int count = *(temp_args->count);
    int print_output = temp_args->print_output;
    int verify = temp_args->verify;

    DSA *dsa = NULL;
    BIGNUM *q = NULL;
    BIGNUM *p = NULL;
    BIGNUM *g = NULL;
    BIGNUM *pub_key = NULL;
    BIGNUM *priv_key = NULL;

    int i = 0;
    unsigned char sig[256] = {0};
    unsigned int siglen = 0;

    unsigned char *DgstData = OPENSSL_malloc(size);

    if (DgstData == NULL) {
        WARN("# FAIL DSA DgstData Initial malloc failed ! \n");
        exit(EXIT_FAILURE);
    }

    /* setup input hash message */
    for (i = 0; i < size; i++)
        DgstData[i] = i % 16;

    if ((dsa = DSA_new()) == NULL) {
        WARN("# FAIL DSA DSA_new failed ! \n");
        goto end;
    }

    if (((q = BN_bin2bn(BN_q, sizeof(BN_q), NULL)) == NULL) ||
        ((p = BN_bin2bn(BN_p, sizeof(BN_p), NULL)) == NULL) ||
        ((g = BN_bin2bn(BN_g, sizeof(BN_g), NULL)) == NULL) ||
        ((pub_key = BN_bin2bn(Pub_key, sizeof(Pub_key), NULL)) == NULL) ||
        ((priv_key = BN_bin2bn(Priv_key, sizeof(Priv_key), NULL)) == NULL)) {
        ret = 0;
        WARN("# FAIL DSA set DSA Parameters failed! \n");
        if (q) BN_free(q);
        if (p) BN_free(p);
        if (g) BN_free(g);
        if (pub_key) BN_free(pub_key);
        if (priv_key) BN_free(priv_key);
        goto end;
    }

    DSA_set0_pqg(dsa, p, q, g);
    DSA_set0_key(dsa, pub_key, priv_key);

    for (i = 0; i < count; i++) {
        if (i == 0) {
            /*
             * DSA_sign() computes a digital signature on the size byte message
             * digest dgst using the private key dsa and places its ASN.1 DER
             * encoding at sig. The length of the signature is placed in
             * *siglen. sig must point to DSA_size(dsa) bytes of memory.
             */
            DEBUG("%s starting DSA sign \n", __func__);

            ret = DSA_sign(0, DgstData, size, sig, &siglen, dsa);
            if (ret <= 0) {
                WARN("# FAIL DSA signature failed! \n");
                goto end;
            } else
                DEBUG("# DSA signature completed! \n");

            if (print_output)
                tests_hexdump(" DSA Signature:", sig, siglen);
        }

        if (i == 0 || verify) {
            /*
             * DSA_verify() verifies that the signature sig of size siglen
             * matches a given message digest dgst of length 'size'. dsa is the
             * signer's public key. Returns * 1: correct signature * 0:
             * incorrect signature * -1: error
             */
            DEBUG("%s starting DSA verify! \n", __func__);

            ret = DSA_verify(0, DgstData, size, sig, siglen, dsa);

            if (ret <= 0) {
                INFO("# FAIL %s for DSA.\n", verify ? "verify" : "sign");
                goto end;
            } else if (i == 0)
                    INFO("# PASS %s for DSA.\n",
                         verify ? "verify" : "sign");
        }
    }
#endif
#ifdef QAT_OPENSSL_PROVIDER
    EVP_PKEY *dsa_key = NULL;
    EVP_PKEY_CTX *dsa_sign_ctx;
    EVP_PKEY_CTX *dsa_verify_ctx;
    unsigned char *buf = NULL;
    unsigned char *buf2 = NULL;
    size_t buflen = 16 * 1024;
    size_t sigsize = buflen;

    buf = OPENSSL_malloc(buflen);
    if (buf == NULL){
        WARN("# FAIL DSA buf malloc failed! \n");
        goto end;
    }
    buf2 = OPENSSL_malloc(buflen);
    if (buf2 == NULL){
        WARN("# FAIL DSA buf2 malloc failed! \n");
        goto end;
    }

    memset(buf, 0, buflen);
    memset(buf2, 0, buflen);

    if (RAND_bytes(buf, 36) <= 0){
        WARN("# FAIL DSA get rand bytes failed! \n");
        goto end;
    }

    if (size == 2048)
        dsa_key = get_dsa(size, temp_args->q_size);
    else
        dsa_key = get_dsa(size, 0);

    if (dsa_key == NULL){
        WARN("# FAIL DSA get dsa key failed! \n");
        goto end;
    }

    dsa_sign_ctx = EVP_PKEY_CTX_new(dsa_key, NULL);

    if (dsa_sign_ctx == NULL
        || EVP_PKEY_sign_init(dsa_sign_ctx) <= 0
        || EVP_PKEY_sign(dsa_sign_ctx, buf2, &sigsize, buf, 20) <= 0){
        WARN("# FAIL DSA PKEY sign failed! \n");
        goto end;
    }

    INFO("# PASS DSA sign.\n");

    dsa_verify_ctx = EVP_PKEY_CTX_new(dsa_key, NULL);
    if (dsa_verify_ctx == NULL
        || EVP_PKEY_verify_init(dsa_verify_ctx) <= 0
        || EVP_PKEY_verify(dsa_verify_ctx, buf2, sigsize, buf, 20) <= 0){
        WARN("# FAIL DSA verify init failed! \n");
        goto end;
    }
    INFO("# PASS DSA verify.\n");

#endif
end:
    DEBUG("%s start to clean up! \n", __func__);
#ifdef QAT_OPENSSL_PROVIDER
    OPENSSL_free(buf);
    OPENSSL_free(buf2);
#endif
#ifndef QAT_OPENSSL_PROVIDER
    if (DgstData)
        OPENSSL_free(DgstData);
    if (dsa != NULL)
        DSA_free(dsa);
#endif
    return ret;
}

void tests_run_dsa(TEST_PARAMS *args)
{
    int i = 1;

    if (!args->enable_async) {
        if (args->size == 2048) {
            for (i = 1; i < KEY_TYPE_MAX; i++) {
                if (i == 1) {
                    run_dsa(args);
                    INFO("DSA 2048-224 test finished\n");
                } else {
                    args->q_size = 256;
                    run_dsa(args);
                    INFO("DSA 2048-256 test finished\n");
                }
            }
        } else {
            run_dsa(args);
        }
    } else {
        start_async_job(args, run_dsa);
    }
}
