/***************************************************************************
 $RCSfile: transfinder.cpp,v $
                             -------------------
    cvs         : $Id: transfinder.cpp,v 1.15 2005/08/31 05:24:20 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "transfinder.h"
#include "textmatcher.h"
#include "kbanking.h"
#include "rulenameedit.h"

#include <qwidget.h>
#include <qgroupbox.h>
#include <qlistview.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <qstring.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qradiobutton.h>
#include <qbutton.h>
#include <qcheckbox.h>
#include <qdatetime.h>
#include <qdatetimeedit.h>
#include <qcombobox.h>
#include <qapplication.h>
#include <qpushbutton.h>
#include <qsplitter.h>
#include <qpopupmenu.h>
#include <qtabwidget.h>

#include <gwenhywfar/debug.h>

#ifdef WIN32
# define strcasecmp stricmp
# define snprintf _snprintf
#endif



TransactionFinder::TransactionFinder(KBanking *kb,
                                     GWEN_DB_NODE *dbRules,
				     const char *ruleName,
                                     GWEN_TYPE_UINT32 flags,
                                     QWidget* parent,
                                     const char* name,
                                     bool modal,
                                     WFlags fl)
:TransactionFinderUI(parent, name, modal, fl)
,_app(kb)
,_dbRules(0)
,_dbRule(0)
,_ruleName(ruleName)
,_activeRule(false)
,_modifying(false)
,_currentCategoryItem(0)
,_flags(flags){
  GWEN_DB_NODE *dbT;
  int i;
  int idx;

  _dbRules=GWEN_DB_Group_dup(dbRules);

  matchWorkingLabel->setHidden(true);

  accountListView->setSelectionMode(QListView::Multi);
  accountListView->setAllColumnsShowFocus(true);

  payeeListView->setSelectionMode(QListView::Multi);
  payeeListView->setAllColumnsShowFocus(true);

  categoryListView->setSelectionMode(QListView::Multi);
  categoryListView->setRootIsDecorated(true);
  categoryListView->setAllColumnsShowFocus(true);

  transactionListView->setSelectionMode(QListView::Multi);

  if (!(flags & TRANSFINDER_FLAGS_USE_CATEGORY))
    criteriaTab->removePage(categoryTab);

  if (flags & TRANSFINDER_FLAGS_EXT_PAYEES)
    criteriaTab->removePage(payeeTab);
  else
    criteriaTab->removePage(payeeTab2);

  QObject::connect((QObject*)rulesCombo, SIGNAL(activated(int)),
                   this, SLOT(slotRuleActivated(int)));
  QObject::connect((QObject*)allAccountsCheck, SIGNAL(toggled(bool)),
                   this, SLOT(slotAllAccountCheckToggled(bool)));
  QObject::connect((QObject*)rangeCombo, SIGNAL(activated(int)),
                   this, SLOT(slotDateRangeActivated(int)));

  QObject::connect((QObject*)anyAmountRadio, SIGNAL(toggled(bool)),
                   this, SLOT(slotAnyAmountRadioToggled(bool)));
  QObject::connect((QObject*)thisAmountRadio, SIGNAL(toggled(bool)),
                   this, SLOT(slotThisAmountRadioToggled(bool)));
  QObject::connect((QObject*)rangeAmountRadio, SIGNAL(toggled(bool)),
                   this, SLOT(slotRangeAmountRadioToggled(bool)));
  QObject::connect((QObject*)lessAmountRadio, SIGNAL(toggled(bool)),
                   this, SLOT(slotLessAmountRadioToggled(bool)));
  QObject::connect((QObject*)moreAmountRadio, SIGNAL(toggled(bool)),
                   this, SLOT(slotMoreAmountRadioToggled(bool)));

  QObject::connect((QObject*)payeeListView, SIGNAL(selectionChanged()),
                   this, SLOT(slotPayeeSelectionChanged()));
  QObject::connect((QObject*)allPayeesCheck, SIGNAL(toggled(bool)),
                   this, SLOT(slotAllPayeeCheckToggled(bool)));

  QObject::connect((QObject*)allCategoriesCheck, SIGNAL(toggled(bool)),
                   this, SLOT(slotAllCategoryCheckToggled(bool)));

  QObject::connect((QObject*)refreshButton, SIGNAL(clicked()),
                   this, SLOT(slotRefreshClicked()));
  QObject::connect((QObject*)resetButton, SIGNAL(clicked()),
                   this, SLOT(slotResetClicked()));
  QObject::connect((QObject*)closeButton, SIGNAL(clicked()),
                   this, SLOT(accept()));
  QObject::connect((QObject*)abortButton, SIGNAL(clicked()),
                   this, SLOT(reject()));
  QObject::connect((QObject*)newRuleButton, SIGNAL(clicked()),
                   this, SLOT(slotNewRuleClicked()));
  QObject::connect((QObject*)deleteRuleButton, SIGNAL(clicked()),
                   this, SLOT(slotDeleteRuleClicked()));
  QObject::connect((QObject*)saveRuleButton, SIGNAL(clicked()),
                   this, SLOT(slotSaveRuleClicked()));
  QObject::connect((QObject*)accountListView, SIGNAL(selectionChanged()),
                   this, SLOT(slotAccountSelectionChanged()));

  QObject::connect((QObject*)categoryListView,
                   SIGNAL(contextMenuRequested(QListViewItem *,
                                               const QPoint &,
                                               int)),
                   this,
                   SLOT(slotCategoryContextMenuRequested(QListViewItem *,
                                                         const QPoint &,
                                                         int)));
  QObject::connect((QObject*)categoryListView, SIGNAL(selectionChanged()),
                   this, SLOT(slotCategorySelectionChanged()));

  QObject::connect((QObject*)helpButton, SIGNAL(clicked()),
		   this, SLOT(slotHelp()));

  textMatcher->setLabel(tr("Text"));
  payeeMatcher->setLabel(tr("Payee"));

  /* fill rule select combo */
  dbT=GWEN_DB_FindFirstGroup(_dbRules, "rule");
  i=0;
  idx=-1;
  while(dbT) {
    const char *s;

    s=GWEN_DB_GetCharValue(dbT, "name", 0, 0);
    if (s) {
      if (strcasecmp(s, ruleName)==0 && idx==-1) {
        _dbRule=dbT;
        idx=i;
      }
    }
    _addRule(dbT);
    i++;
    dbT=GWEN_DB_FindNextGroup(dbT, "rule");
  }
  if (idx==-1) {
    DBG_NOTICE(0, "Rule %s does not exist", ruleName);
    _dbRule=GWEN_DB_GetGroup(_dbRules, GWEN_PATH_FLAGS_CREATE_GROUP,
                             "rule");
    assert(_dbRule);
    GWEN_DB_SetCharValue(_dbRule, GWEN_DB_FLAGS_DEFAULT,
                         "name", ruleName);
    rulesCombo->insertItem(ruleName);
    rulesCombo->setCurrentItem(i);
    slotRuleActivated(i);
  }
  else {
    rulesCombo->setCurrentItem(idx);
    slotRuleActivated(idx);
  }

}



TransactionFinder::~TransactionFinder(){
  GWEN_DB_Group_free(_dbRules);
}



bool TransactionFinder::init(){
  GWEN_DB_NODE *db;
  std::string s;
  int i, j;

  categoryListView->beginModify();
  db=_app->getAppData();
  if (db) {
    db=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                        "gui/dlgs/transactionFinder");
    if (db) {
      GWEN_DB_NODE *dbT;
      int w, h;

      w=GWEN_DB_GetIntValue(db, "width", 0, 640);
      h=GWEN_DB_GetIntValue(db, "height", 0, 480);
      resize(w, h);

      dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "accountList");
      if (dbT) {
        const char *p;

        /* found settings */
        i=GWEN_DB_GetIntValue(dbT, "sortColumn", 0, -1);
        if (i!=-1)
          accountListView->setSortColumn(i);
        p=GWEN_DB_GetCharValue(dbT, "sortOrder", 0, "ascending");
        if (p) {
          if (strcasecmp(p, "ascending")==0)
            accountListView->setSortOrder(Qt::Ascending);
          else
            if (strcasecmp(p, "descending")==0)
              accountListView->setSortOrder(Qt::Descending);
        }
        for (i=0; i<accountListView->columns(); i++) {
          accountListView->setColumnWidthMode(i, QListView::Manual);
          j=GWEN_DB_GetIntValue(dbT, "columns", i, -1);
          if (j!=-1)
          accountListView->setColumnWidth(i, j);
        } /* for */
      } // if account list data

      dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "categoryList");
      if (dbT) {
        const char *p;

        /* found settings */
        i=GWEN_DB_GetIntValue(dbT, "sortColumn", 0, -1);
        if (i!=-1)
          categoryListView->setSortColumn(i);
        p=GWEN_DB_GetCharValue(dbT, "sortOrder", 0, "ascending");
        if (p) {
          if (strcasecmp(p, "ascending")==0)
            categoryListView->setSortOrder(Qt::Ascending);
          else
            if (strcasecmp(p, "descending")==0)
              categoryListView->setSortOrder(Qt::Descending);
        }
        for (i=0; i<categoryListView->columns(); i++) {
          categoryListView->setColumnWidthMode(i, QListView::Manual);
          j=GWEN_DB_GetIntValue(dbT, "columns", i, -1);
          if (j!=-1)
          categoryListView->setColumnWidth(i, j);
        } /* for */
      } // if category list data

      dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                           "transactionList");
      if (dbT) {
        const char *p;

        /* found settings */
        i=GWEN_DB_GetIntValue(dbT, "sortColumn", 0, -1);
        if (i!=-1)
          transactionListView->setSortColumn(i);
        p=GWEN_DB_GetCharValue(dbT, "sortOrder", 0, "ascending");
        if (p) {
          if (strcasecmp(p, "ascending")==0)
            transactionListView->setSortOrder(Qt::Ascending);
          else
            if (strcasecmp(p, "descending")==0)
              transactionListView->setSortOrder(Qt::Descending);
        }

        for (i=0; i<transactionListView->columns(); i++) {
          transactionListView->setColumnWidthMode(i, QListView::Manual);
          j=GWEN_DB_GetIntValue(dbT, "columns", i, -1);
          if (j!=-1)
          transactionListView->setColumnWidth(i, j);
        } /* for */
      } // if transaction list data

      dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                           "payeeList");
      if (dbT) {
        const char *p;

        /* found settings */
        i=GWEN_DB_GetIntValue(dbT, "sortColumn", 0, -1);
        if (i!=-1)
          payeeListView->setSortColumn(i);
        p=GWEN_DB_GetCharValue(dbT, "sortOrder", 0, "ascending");
        if (p) {
          if (strcasecmp(p, "ascending")==0)
            payeeListView->setSortOrder(Qt::Ascending);
          else
            if (strcasecmp(p, "descending")==0)
              payeeListView->setSortOrder(Qt::Descending);
        }

        for (i=0; i<payeeListView->columns(); i++) {
          payeeListView->setColumnWidthMode(i, QListView::Manual);
          j=GWEN_DB_GetIntValue(dbT, "columns", i, -1);
          if (j!=-1)
          payeeListView->setColumnWidth(i, j);
        } /* for */
      } // if payee list data

      dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                           "splitter");
      if (dbT) {
        int i, j;
        QValueList<int> sizes;

        /* found settings */
        for (i=0; ; i++) {
          j=GWEN_DB_GetIntValue(dbT, "sizes", i, -1);
          if (j!=-1)
            sizes.push_back(j);
          else
            break;
        } /* for */

        if (!sizes.empty())
          splitter->setSizes(sizes);
      } // if splitter settings
    } /* if settings */
  }

  accountListView->addAccounts(_app->getAppAccounts());
  payeeListView->addPayees(_app->getPayees());
  categoryListView->addCategories(_app->getCategories());
  categoryListView->expandAll();
  slotRuleActivated(rulesCombo->currentItem());
  categoryListView->endModify();

  return true;
}



bool TransactionFinder::fini(){
  GWEN_DB_NODE *db;
  GWEN_DB_NODE *dbT;
  int i, j;
  QValueList<int> sizes;

  db=_app->getAppData();
  assert(db);
  db=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                      "gui/dlgs/transactionFinder");
  assert(db);
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_DEFAULT,
                      "width", width());
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_DEFAULT,
                      "height", height());

  // store account list info
  dbT=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "accountList");
  GWEN_DB_SetIntValue(dbT, GWEN_DB_FLAGS_DEFAULT,
                      "sortColumn", accountListView->sortColumn());
  switch(accountListView->sortOrder()) {
  case Qt::Ascending:
    GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "sortOrder", "ascending");
    break;
  case Qt::Descending:
    GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "sortOrder", "descending");
    break;
  default:
    break;
  }
  GWEN_DB_DeleteVar(dbT, "columns");
  for (i=0; i<accountListView->columns(); i++) {
    j=accountListView->columnWidth(i);
    GWEN_DB_SetIntValue(dbT, GWEN_DB_FLAGS_DEFAULT,
                        "columns", j);
  } /* for */

  // store payee list info
  dbT=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "payeeList");
  GWEN_DB_SetIntValue(dbT, GWEN_DB_FLAGS_DEFAULT,
                      "sortColumn", payeeListView->sortColumn());
  switch(payeeListView->sortOrder()) {
  case Qt::Ascending:
    GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "sortOrder", "ascending");
    break;
  case Qt::Descending:
    GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "sortOrder", "descending");
    break;
  default:
    break;
  }
  GWEN_DB_DeleteVar(dbT, "columns");
  for (i=0; i<payeeListView->columns(); i++) {
    j=payeeListView->columnWidth(i);
    GWEN_DB_SetIntValue(dbT, GWEN_DB_FLAGS_DEFAULT,
                        "columns", j);
  } /* for */

  // store category list info
  dbT=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "categoryList");
  GWEN_DB_SetIntValue(dbT, GWEN_DB_FLAGS_DEFAULT,
                      "sortColumn", categoryListView->sortColumn());
  switch(categoryListView->sortOrder()) {
  case Qt::Ascending:
    GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "sortOrder", "ascending");
    break;
  case Qt::Descending:
    GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "sortOrder", "descending");
    break;
  default:
    break;
  }
  GWEN_DB_DeleteVar(dbT, "columns");
  for (i=0; i<categoryListView->columns(); i++) {
    j=categoryListView->columnWidth(i);
    GWEN_DB_SetIntValue(dbT, GWEN_DB_FLAGS_DEFAULT,
                        "columns", j);
  } /* for */

  // store transaction list info
  dbT=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "transactionList");
  GWEN_DB_SetIntValue(dbT, GWEN_DB_FLAGS_DEFAULT,
                      "sortColumn", transactionListView->sortColumn());
  switch(transactionListView->sortOrder()) {
  case Qt::Ascending:
    GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "sortOrder", "ascending");
    break;
  case Qt::Descending:
    GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "sortOrder", "descending");
    break;
  default:
    break;
  }
  GWEN_DB_DeleteVar(dbT, "columns");
  for (i=0; i<transactionListView->columns(); i++) {
    j=transactionListView->columnWidth(i);
    GWEN_DB_SetIntValue(dbT, GWEN_DB_FLAGS_DEFAULT,
                        "columns", j);
  } /* for */

  /* save splitter settings */
  dbT=GWEN_DB_GetGroup(db,
                       GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                       "splitter");
  sizes=splitter->sizes();
  QValueList<int>::Iterator it=sizes.begin();
  while(it!=sizes.end()) {
    GWEN_DB_SetIntValue(dbT, GWEN_DB_FLAGS_DEFAULT,
                        "sizes", *it);
    ++it;
  }


  return true;
}



void TransactionFinder::_addRule(GWEN_DB_NODE *dbT){
  const char *s;
  QString qs;

  s=GWEN_DB_GetCharValue(dbT, "name", 0, 0);
  if (s) {
    qs=QString::fromUtf8(s);
  }
  else
    qs=tr("[unnamed]");
  s=GWEN_DB_GetCharValue(dbT, "short", 0, 0);
  if (s) {
    qs+=" (";
    qs+=QString::fromUtf8(s);
    qs+=")";
  }
  rulesCombo->insertItem(qs);
}




bool TransactionFinder::_fromDb(GWEN_DB_NODE *db){
  GWEN_DB_NODE *dbT;
  const char *s;
  int i;

  _resetRule();

  /* text tab */
  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "text");
  if (dbT) {
    if (!textMatcher->fromDb(dbT))
      return false;
  }

  /* account tab */
  accountListView->selectAccounts("*", "*", false);
  allAccountsCheck->setChecked(GWEN_DB_GetIntValue(db,
                                                   "account/allAccounts",
                                                   0, 0));
  if (allAccountsCheck->isChecked()) {
    /* select all */
    accountListView->selectAccounts("*", "*", true);
  }
  else {
    GWEN_DB_NODE *dbSelected;

    dbSelected=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                                "account/selected");
    if (dbSelected) {
      GWEN_DB_NODE *gr;

      gr=GWEN_DB_FindFirstGroup(dbSelected, "account");
      while(gr) {
        accountListView->selectAccounts(GWEN_DB_GetCharValue(gr, "bankId", 0, ""),
                                 GWEN_DB_GetCharValue(gr, "accountId", 0, ""),
                                 true);
        gr=GWEN_DB_FindNextGroup(gr, "account");
      }
    } /* if dbSelected */
  } /* if !all */

  /* date tab */
  s=GWEN_DB_GetCharValue(db, "date/fromDate", 0, 0);
  if (s) {
    int y,m,d;

    if (3!=sscanf(s, "%04d/%02d/%02d", &y, &m, &d)) {
      DBG_ERROR(0, "Bad fromdate (%s)", s);
    }
    else {
      QDate da=QDate(y, m, d);
      if (da.isValid())
        fromDateEdit->setDate(da);
    }
  }

  s=GWEN_DB_GetCharValue(db, "date/toDate", 0, 0);
  if (s) {
    int y,m,d;

    if (3!=sscanf(s, "%04d/%02d/%02d", &y, &m, &d)) {
      DBG_ERROR(0, "Bad todate (%s)", s);
    }
    else {
      QDate da=QDate(y, m, d);
      if (da.isValid())
        toDateEdit->setDate(da);
    }
  }

  i=GWEN_DB_GetIntValue(db, "date/range", 0, 0);
  rangeCombo->setCurrentItem(i);
  if (!i)
    slotDateRangeActivated(0);

  /* amount tab */
  switch (GWEN_DB_GetIntValue(db, "amount/mode", 0, 0)) {
  case TMatcherAmountRange_All:
    anyAmountRadio->setChecked(true);
    break;
  case TMatcherAmountRange_Exactly:
    thisAmountRadio->setChecked(true);
    break;
  case TMatcherAmountRange_Range:
    rangeAmountRadio->setChecked(true);
    break;
  case TMatcherAmountRange_Less:
    lessAmountRadio->setChecked(true);
    break;
  case TMatcherAmountRange_More:
    moreAmountRadio->setChecked(true);
    break;
  default:
    anyAmountRadio->setChecked(true);
    break;
  }

  s=GWEN_DB_GetCharValue(db, "amount/thisAmount", 0, "");
  thisAmountEdit->setText(QString::fromUtf8(s));

  s=GWEN_DB_GetCharValue(db, "amount/fromAmount", 0, "");
  fromAmountEdit->setText(QString::fromUtf8(s));

  s=GWEN_DB_GetCharValue(db, "amount/toAmount", 0, "");
  toAmountEdit->setText(QString::fromUtf8(s));

  s=GWEN_DB_GetCharValue(db, "amount/lessAmount", 0, "");
  lessAmountEdit->setText(QString::fromUtf8(s));

  s=GWEN_DB_GetCharValue(db, "amount/moreAmount", 0, "");
  moreAmountEdit->setText(QString::fromUtf8(s));

  /* payee tab1/tab2 */
  if (_flags & TRANSFINDER_FLAGS_EXT_PAYEES) {
    allPayeesCheck->setChecked(GWEN_DB_GetIntValue(db,
						   "payees2/allPayees",
						   0, 1));
    if (allPayeesCheck->isOn())
      payeeListView->selectAll(true);
    else {
      dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "payee2");
      if (dbT) {
	const char *s;
	unsigned int i;

	for (i=0; ; i++) {
	  s=GWEN_DB_GetCharValue(dbT, "payees", i, 0);
	  if (!s)
	    break;
	  payeeListView->selectPayee(s, true);
	}
      }
    }
  }
  else {
    dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "payee");
    if (dbT) {
      if (!payeeMatcher->fromDb(dbT))
	return false;
    }
  }

  /* category tab */

  allCategoriesCheck->setChecked(GWEN_DB_GetIntValue(db,
                                                   "category/allCategories",
                                                   0, 1));
  if (allCategoriesCheck->isChecked()) {
    /* select all */
    categoryListView->selectAll(true);
  }
  else {
    dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "category");
    if (dbT) {
      const char *s;
      unsigned int i;

      for (i=0; ; i++) {
	s=GWEN_DB_GetCharValue(dbT, "categories", i, 0);
	if (!s)
	  break;
	categoryListView->selectSingleCategory(s, true);
      }
    }
  } /* if !all */

  return true;
}



bool TransactionFinder::_toDb(GWEN_DB_NODE *db, bool showErrors){
  QString qs;
  QDate d;
  GWEN_DB_NODE *gr;
  int i;
  std::string s;
  AB_VALUE *v;

  /* text tab */
  gr=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "text");
  assert(gr);
  if (!textMatcher->toDb(gr)) {
    return false;
  }

  /* account tab */
  gr=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "account");
  assert(gr);
  GWEN_DB_SetIntValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "allAccounts",
                      allAccountsCheck->isChecked()?1:0);
  if (!allAccountsCheck->isChecked()) {
    std::list<Account*> al;
    std::list<Account*>::iterator it;

    /* list all accounts */
    al=accountListView->getSelectedAccounts();
    if (!al.empty()) {
      GWEN_DB_NODE *dbt1;

      dbt1=GWEN_DB_GetGroup(gr, GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                            "selected");
      assert(dbt1);
      for (it=al.begin(); it!=al.end(); it++) {
        GWEN_DB_NODE *dbt2;

        dbt2=GWEN_DB_GetGroup(dbt1, GWEN_PATH_FLAGS_CREATE_GROUP,
                              "account");
        GWEN_DB_SetCharValue(dbt2, GWEN_DB_FLAGS_OVERWRITE_VARS,
                             "bankId", (*it)->getBankCode().c_str());
        GWEN_DB_SetCharValue(dbt2, GWEN_DB_FLAGS_OVERWRITE_VARS,
                             "accountId", (*it)->getAccountNumber().c_str());
      } // for
    } // if there are accounts selected
  }

  /* date tab */
  gr=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "date");
  assert(gr);

  i=rangeCombo->currentItem();
  GWEN_DB_SetIntValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "range", i);

  if (i==TMatcherDateRange_UserDefined) {
    d=fromDateEdit->date();
    if (d.isValid()) {
      char dbuf[16];
  
      snprintf(dbuf, sizeof(dbuf), "%04d/%02d/%02d",
               d.year(), d.month(), d.day());
      GWEN_DB_SetCharValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                           "fromDate", dbuf);
    }
  
    d=toDateEdit->date();
    if (d.isValid()) {
      char dbuf[16];
  
      snprintf(dbuf, sizeof(dbuf), "%04d/%02d/%02d",
               d.year(), d.month(), d.day());
      GWEN_DB_SetCharValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                           "toDate", dbuf);
    }
  } // if userDefined

  // amount tab
  gr=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "amount");
  assert(gr);
  if (anyAmountRadio->isChecked())
    GWEN_DB_SetIntValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "mode", TMatcherAmountRange_All);
  else if (thisAmountRadio->isChecked()) {
    GWEN_DB_SetIntValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "mode", TMatcherAmountRange_Exactly);
    qs=thisAmountEdit->text();
    s=QBanking::QStringToUtf8String(qs);
    if (s.empty()) {
      if (showErrors)
        QMessageBox::critical(0,
                              tr("Empty amount"),
                              tr("The <i>exact</i> amount field is empty."),
                              tr("Dismiss"), 0, 0, 0);
      return false;
    }
    v=AB_Value_fromString(s.c_str());
    if (!v) {
      if (showErrors)
        QMessageBox::critical(0,
                              tr("Bad amount"),
                              tr("The <i>exact</i> amount field is invalid."),
                              tr("Dismiss"), 0, 0, 0);
      return false;
    }
    AB_Value_free(v);
    GWEN_DB_SetCharValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "thisAmount", s.c_str());
  }
  else if (rangeAmountRadio->isChecked()) {
    GWEN_DB_SetIntValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "mode", TMatcherAmountRange_Range);

    qs=fromAmountEdit->text();
    s=QBanking::QStringToUtf8String(qs);
    if (s.empty()) {
      if (showErrors)
        QMessageBox::critical(0,
                              tr("Empty amount"),
                              tr("The <i>from</i> amount field is empty."),
                              tr("Dismiss"), 0, 0, 0);
      return false;
    }
    v=AB_Value_fromString(s.c_str());
    if (!v) {
      if (showErrors)
        QMessageBox::critical(0,
                              tr("Bad amount"),
                              tr("The <i>from</i> amount field is invalid."),
                              tr("Dismiss"), 0, 0, 0);
      return false;
    }
    AB_Value_free(v);
    GWEN_DB_SetCharValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "fromAmount", s.c_str());

    qs=toAmountEdit->text();
    s=QBanking::QStringToUtf8String(qs);
    if (s.empty()) {
      if (showErrors)
        QMessageBox::critical(0,
                              tr("Empty amount"),
                              tr("The <i>to</i> amount field is empty."),
                              tr("Dismiss"), 0, 0, 0);
      return false;
    }
    v=AB_Value_fromString(s.c_str());
    if (!v) {
      if (showErrors)
        QMessageBox::critical(0,
                              tr("Bad amount"),
                              tr("The <i>to</i> amount field is invalid."),
                              tr("Dismiss"), 0, 0, 0);
      return false;
    }
    AB_Value_free(v);
    GWEN_DB_SetCharValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "toAmount", s.c_str());
  }
  else if (lessAmountRadio->isChecked()) {
    GWEN_DB_SetIntValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "mode", TMatcherAmountRange_Less);
    qs=lessAmountEdit->text();
    s=QBanking::QStringToUtf8String(qs);
    if (s.empty()) {
      if (showErrors)
        QMessageBox::critical(0,
                              tr("Empty amount"),
                              tr("The <i>less</i> amount field is empty."),
                              tr("Dismiss"), 0, 0, 0);
      return false;
    }
    v=AB_Value_fromString(s.c_str());
    if (!v) {
      if (showErrors)
        QMessageBox::critical(0,
                              tr("Bad amount"),
                              tr("The <i>less</i> amount field is invalid."),
                              tr("Dismiss"), 0, 0, 0);
      return false;
    }
    AB_Value_free(v);
    GWEN_DB_SetCharValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "lessAmount", s.c_str());
  }
  else if (moreAmountRadio->isChecked()) {
    GWEN_DB_SetIntValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "mode", TMatcherAmountRange_More);
    qs=moreAmountEdit->text();
    s=QBanking::QStringToUtf8String(qs);
    if (s.empty()) {
      if (showErrors)
        QMessageBox::critical(0,
                              tr("Empty amount"),
                              tr("The <i>more</i> amount field is empty."),
                              tr("Dismiss"), 0, 0, 0);
      return false;
    }
    v=AB_Value_fromString(s.c_str());
    if (!v) {
      if (showErrors)
        QMessageBox::critical(0,
                              tr("Bad amount"),
                              tr("The <i>more</i> amount field is invalid."),
                              tr("Dismiss"), 0, 0, 0);
      return false;
    }
    AB_Value_free(v);
    GWEN_DB_SetCharValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "moreAmount", s.c_str());
  }
  else {
    DBG_ERROR(0, "Uuups, none of the radio buttons is checked?");
    abort();
  }

  /* payee tab1/tab2 */
  if (_flags & TRANSFINDER_FLAGS_EXT_PAYEES) {
    GWEN_DB_DeleteGroup(db, "payee");
    gr=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "payee2");
    assert(gr);
    GWEN_DB_SetIntValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
			"allPayees",
			allPayeesCheck->isChecked()?1:0);
    if (!allPayeesCheck->isChecked()) {
      std::list<Payee*> pList=payeeListView->getSelectedPayees();
      if (!pList.empty()) {
	std::list<Payee*>::iterator it;
  
	for (it=pList.begin(); it!=pList.end(); it++) {
	  if (!(*it)->id().empty()) {
	    GWEN_DB_SetCharValue(gr, GWEN_DB_FLAGS_DEFAULT,
				 "payees", (*it)->id().c_str());
	  }
	}
      }
    }
  }
  else {
    GWEN_DB_DeleteGroup(db, "payee2");
    gr=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "payee");
    assert(gr);
    if (!payeeMatcher->toDb(gr)) {
      return false;
    }
  }

  /* category tab */
  if (_flags & TRANSFINDER_FLAGS_USE_CATEGORY) {
    gr=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "category");
    assert(gr);

    GWEN_DB_SetIntValue(gr, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "allCategories",
                        allCategoriesCheck->isChecked()?1:0);
    if (!allCategoriesCheck->isChecked()) {
      std::list<Category*> catList=categoryListView->getSelectedCategories();
      if (!catList.empty()) {
        std::list<Category*>::iterator it;

        for (it=catList.begin(); it!=catList.end(); it++) {
          if (!(*it)->getId().empty()) {
            GWEN_DB_SetCharValue(gr, GWEN_DB_FLAGS_DEFAULT,
                                 "categories", (*it)->getId().c_str());
          }
        }
      }
    }
  }
  else {
    GWEN_DB_DeleteGroup(db, "category");
  }
  return true;
}



void TransactionFinder::slotRuleActivated(int idx){
  GWEN_DB_NODE *dbT;
  int i;

  DBG_DEBUG(0, "Rule %d activated", idx);

  if (_activeRule) {
    if (!_saveRule()) {
      DBG_ERROR(0, "Bad rule");
      return;
    }
  }

  dbT=GWEN_DB_FindFirstGroup(_dbRules, "rule");
  i=idx;
  while(dbT && i--) dbT=GWEN_DB_FindNextGroup(dbT, "rule");
  if (!dbT) {
    DBG_ERROR(0, "Uuuups, index %d out of range?", idx);
    GWEN_DB_Dump(_dbRules, stderr, 2);
    abort();
  }
  _dbRule=dbT;
  dbT=GWEN_DB_GetGroup(_dbRule, GWEN_DB_FLAGS_DEFAULT, "data");
  assert(dbT);
  _fromDb(dbT);

  _activeRule=true;
}



void TransactionFinder::slotDateRangeActivated(int idx){
  DBG_DEBUG(0, "Selection date range %d", idx);
  if (idx==TMatcherDateRange_UserDefined) {
    fromDateEdit->setEnabled(true);
    toDateEdit->setEnabled(true);
  }
  else {
    fromDateEdit->setEnabled(false);
    toDateEdit->setEnabled(false);
  }
}



void TransactionFinder::slotAnyAmountRadioToggled(bool on){
  DBG_DEBUG(0, "anyAmountRadioToggled (%d)", on);
  if (on) {
    thisAmountEdit->setEnabled(false);
    fromAmountEdit->setEnabled(false);
    toAmountEdit->setEnabled(false);
    lessAmountEdit->setEnabled(false);
    moreAmountEdit->setEnabled(false);
  }
}



void TransactionFinder::slotThisAmountRadioToggled(bool on){
  DBG_DEBUG(0, "thisAmountRadioToggled (%d)", on);
  thisAmountEdit->setEnabled(on);
  if (on) {
    fromAmountEdit->setEnabled(false);
    toAmountEdit->setEnabled(false);
    lessAmountEdit->setEnabled(false);
    moreAmountEdit->setEnabled(false);
  }
}



void TransactionFinder::slotRangeAmountRadioToggled(bool on){
  DBG_DEBUG(0, "rangeAmountRadioToggled (%d)", on);
  fromAmountEdit->setEnabled(on);
  toAmountEdit->setEnabled(on);
  if (on) {
    thisAmountEdit->setEnabled(false);
    lessAmountEdit->setEnabled(false);
    moreAmountEdit->setEnabled(false);
  }
}



void TransactionFinder::slotLessAmountRadioToggled(bool on){
  DBG_DEBUG(0, "lessAmountRadioToggled (%d)", on);
  lessAmountEdit->setEnabled(on);
  if (on) {
    thisAmountEdit->setEnabled(false);
    fromAmountEdit->setEnabled(false);
    toAmountEdit->setEnabled(false);
    moreAmountEdit->setEnabled(false);
  }
}



void TransactionFinder::slotMoreAmountRadioToggled(bool on){
  DBG_DEBUG(0, "moreAmountRadioToggled (%d)", on);
  moreAmountEdit->setEnabled(on);
  if (on) {
    thisAmountEdit->setEnabled(false);
    fromAmountEdit->setEnabled(false);
    toAmountEdit->setEnabled(false);
    lessAmountEdit->setEnabled(false);
  }
}



void TransactionFinder::slotAllAccountCheckToggled(bool on){
  if (!_modifying) {
    _modifying=true;
    accountListView->selectAll(on);
    _modifying=false;
  }
}



void TransactionFinder::slotAllCategoryCheckToggled(bool on) {
  if (!_modifying) {
    _modifying=true;
    categoryListView->selectAll(on);
    _modifying=false;
  }
}



void TransactionFinder::_resetRule(){
  /* text tab */
  textMatcher->resetRule();

  /* account tab */
  accountListView->selectAccounts("*", "*", false);
  allAccountsCheck->setChecked(false);

  /* date tab */
  fromDateEdit->setDate(QDate(2000, 1, 1));
  toDateEdit->setDate(QDate(7999, 12, 31));
  rangeCombo->setCurrentItem(0);
  slotDateRangeActivated(0);

  /* amount tab */
  thisAmountRadio->setChecked(true);
  rangeAmountRadio->setChecked(true);
  anyAmountRadio->setChecked(true);
  thisAmountEdit->setText("");
  fromAmountEdit->setText("");
  toAmountEdit->setText("");

  /* payee tab1/2 */
  if (_flags & TRANSFINDER_FLAGS_EXT_PAYEES) {
    payeeListView->selectAll(true);
    allPayeesCheck->setChecked(true);
  }
  else {
    payeeMatcher->resetRule();
  }

  /* category tab */
  if (_flags & TRANSFINDER_FLAGS_USE_CATEGORY) {
    categoryListView->selectAll(true);
    allCategoriesCheck->setChecked(true);
  }

}



void TransactionFinder::slotResetClicked(){
  _resetRule();
}



void TransactionFinder::accept(){
  if (_saveRule()) {
    //GWEN_DB_Dump(_dbRules, stderr, 2);
    DBG_DEBUG(0, "Accept");
    QDialog::accept();
  }
}



void TransactionFinder::reject(){
  _saveRule(false);
  //GWEN_DB_Dump(_dbRules, stderr, 2);
  DBG_NOTICE(0, "Reject");
  QDialog::reject();
}



void TransactionFinder::slotNewRuleClicked(){
  QString qs;
  GWEN_DB_NODE *dbRule;

  RuleNameEdit w(tr("Create New Rule..."),
                 0,
                 0,
                 this,
                 "RuleNameEdit",
                 true);
  for (;;) {
    GWEN_DB_NODE *dbT;

    if (w.exec()!=QDialog::Accepted)
      return;
    qs=w.getName();

    dbT=GWEN_DB_FindFirstGroup(_dbRules, "rule");
    while(dbT) {
      const char *n;

      n=GWEN_DB_GetCharValue(dbT, "name", 0, 0);
      if (n) {
        if (strcasecmp(n, qs.utf8())==0)
          break;
      }
      dbT=GWEN_DB_FindNextGroup(dbT, "rule");
    }
    if (!dbT)
      break;

    if (QMessageBox::critical(0,
                              tr("Name already exists"),
                              tr("Please choose another name."),
                              tr("Ok"), tr("Abort"), 0, 0)!=0)
      return;
  } // for


  dbRule=GWEN_DB_Group_new("rule");
  qs=w.getName();
  GWEN_DB_SetCharValue(dbRule, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "name", qs.utf8());
  qs=w.getShortDescription();
  if (!qs.isEmpty())
    GWEN_DB_SetCharValue(dbRule, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "short", qs.utf8());

    GWEN_DB_SetIntValue(dbRule, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "data/account/allAccounts", 1);
    GWEN_DB_SetIntValue(dbRule, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "data/payee/allPayees", 1);
    GWEN_DB_SetIntValue(dbRule, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "data/category/allCategories", 1);

  _addRule(dbRule);
  GWEN_DB_AddGroup(_dbRules, dbRule);
  rulesCombo->setCurrentItem(rulesCombo->count()-1);
  slotRuleActivated(rulesCombo->count()-1);
}



bool TransactionFinder::_saveRule(bool showErrors){
  GWEN_DB_NODE *dbData;

  dbData=GWEN_DB_Group_new("data");
  if (_toDb(dbData, showErrors)) {
    GWEN_DB_DeleteGroup(_dbRule, "data");
    GWEN_DB_AddGroup(_dbRule, dbData);
    //GWEN_DB_Dump(_dbRule, stderr, 2);
    return true;
  }
  else {
    DBG_ERROR(0, "Could not save rule");
    GWEN_DB_Group_free(dbData);
    return false;
  }

}



void TransactionFinder::slotSaveRuleClicked(){
  QString qs;
  GWEN_DB_NODE *dbRule;
  GWEN_DB_NODE *dbData;

  assert(_dbRule);

  RuleNameEdit w(tr("Save Rule As..."),
                 GWEN_DB_GetCharValue(_dbRule, "name", 0, 0),
                 GWEN_DB_GetCharValue(_dbRule, "short", 0, 0),
                 this,
                 "RuleNameEdit",
                 true);
  for (;;) {
    GWEN_DB_NODE *dbT;

    if (w.exec()!=QDialog::Accepted)
      return;
    qs=w.getName();

    dbT=GWEN_DB_FindFirstGroup(_dbRules, "rule");
    while(dbT) {
      const char *n;

      n=GWEN_DB_GetCharValue(dbT, "name", 0, 0);
      if (n) {
        if (strcasecmp(n, qs.utf8())==0)
          break;
      }
      dbT=GWEN_DB_FindNextGroup(dbT, "rule");
    }
    if (!dbT)
      break;

    if (QMessageBox::critical(0,
                              tr("Name already exists"),
                              tr("Please choose another name."),
                              tr("Ok"), tr("Abort"), 0, 0)!=0)
      return;
  } // for


  dbRule=GWEN_DB_Group_new("rule");
  qs=w.getName();
  GWEN_DB_SetCharValue(dbRule, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "name", qs.utf8());
  qs=w.getShortDescription();
  if (!qs.isEmpty())
    GWEN_DB_SetCharValue(dbRule, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "short", qs.utf8());

  dbData=GWEN_DB_GetGroup(dbRule, GWEN_DB_FLAGS_DEFAULT, "data");
  if (_toDb(dbData, true)) {
    //GWEN_DB_Dump(dbRule, stderr, 2);
    _addRule(dbRule);
    GWEN_DB_AddGroup(_dbRules, dbRule);
    _activeRule=false;
    rulesCombo->setCurrentItem(rulesCombo->count()-1);
    slotRuleActivated(rulesCombo->count()-1);
  }
  else {
    DBG_ERROR(0, "Could not save rule");
    GWEN_DB_Group_free(dbRule);
  }
}



void TransactionFinder::slotDeleteRuleClicked(){
  if (rulesCombo->count()>1 && _dbRule) {
    if (QMessageBox::warning(0,
                             tr("Delete Rule"),
                             tr("Do you really want to delete this rule?"),
                             tr("Yes"), tr("No"), 0, 0)!=0)
      return;

    assert(_dbRule);
    _activeRule=false;
    rulesCombo->removeItem(rulesCombo->currentItem());
    GWEN_DB_UnlinkGroup( _dbRule);
    GWEN_DB_Group_free(_dbRule);
    _dbRule=GWEN_DB_FindFirstGroup(_dbRules, "rule");
    if (_dbRule) {
      rulesCombo->setCurrentItem(0);
      slotRuleActivated(0);
    }
    DBG_DEBUG(0, "Removed, ruleset now:");
    //GWEN_DB_Dump(_dbRules, stderr, 2);
  }
  else {
    QMessageBox::critical(0,
                          tr("Error"),
                          tr("You can not delete all rules"),
                          tr("Ok"), tr("Abort"), 0, 0);
  }
}




void TransactionFinder::slotRefreshClicked(){
  std::list<RefPointer<Transaction> > tl;
  GWEN_TYPE_UINT32 id;
  bool rv;
  bool h1, h2, h3;

  if (!_saveRule())
    return;

  transactionListView->clear();
  //DBG_DEBUG(0, "Checking this rule:");
  //GWEN_DB_Dump(_dbRule, stderr, 2);
  id=_app->showBox(0,
                   "Checking transactions",
                   "Scanning for matching transactions, please wait...");
  h1=transactionListView->isHidden();
  h2=xaMatchLabel->isHidden();
  h3=matchWorkingLabel->isHidden();
  transactionListView->setHidden(true);
  xaMatchLabel->setHidden(true);
  matchWorkingLabel->setHidden(false);
  qApp->processEvents();
  rv=_app->getMatchingTransactions(_dbRule, tl);
  _app->hideBox(id);
  if (rv) {
    std::list<RefPointer<Transaction> >::iterator it;
    QString qs;
    double sum;

    transactionListView->addTransactions(tl);
    transactionListView->selectAll(true);

    sum=0;
    for (it=tl.begin(); it!=tl.end(); it++) {
      const AB_VALUE *v;

      v=(*it).ref().getValue();
      if (v)
        sum+=AB_Value_GetValue(v);
    } // for

    qs=QString(tr("%1 transactions, total amount %2"))
      .arg(tl.size())
      .arg(sum, 0, 'f', 2);
    xaMatchLabel->setText(qs);
  }
  else {
    DBG_ERROR(0, "Uuups...");
  }
  transactionListView->setHidden(h1);
  xaMatchLabel->setHidden(h2);
  matchWorkingLabel->setHidden(h3);
  qApp->processEvents();
}



void TransactionFinder::slotAccountSelectionChanged() {
  if (!_modifying && allAccountsCheck->isChecked()) {
    _modifying=true;
    allAccountsCheck->setChecked(false);
    _modifying=false;
  }
}



void TransactionFinder::slotHelp(){
  _app->invokeHelp("TransFinder");
}



std::list<RefPointer<Transaction> >
TransactionFinder::getMatchingTransactions(){
  return transactionListView->getSelectedTransactions();
}



GWEN_DB_NODE *TransactionFinder::getRules(){
  return _dbRules;
}



GWEN_DB_NODE *TransactionFinder::getRule(){
  return _dbRule;
}



bool TransactionFinder::editRule(KBanking *kb,
                                 GWEN_DB_NODE *dbRule,
                                 const QString &title,
                                 GWEN_TYPE_UINT32 flags,
                                 QWidget* parent,
                                 WFlags fl) {
  std::string ruleName;
  std::string ruleDescr;
  GWEN_DB_NODE *dbRules;
  GWEN_DB_NODE *dbT;
  bool rv;

  ruleName=KBanking::QStringToUtf8String(tr("current"));
  ruleDescr=KBanking::QStringToUtf8String(tr("currently active rule"));
  dbRules=kb->getTransactionMatcherRules();

  dbT=GWEN_DB_FindFirstGroup(dbRules, "rule");
  while(dbT) {
    const char *s;

    s=GWEN_DB_GetCharValue(dbT, "name", 0, 0);
    if (s) {
      if (strcasecmp(s, ruleName.c_str())==0)
        break;
    }
    dbT=GWEN_DB_FindNextGroup(dbT, "rule");
  } // while
  if (dbT) {
    GWEN_DB_ClearGroup(dbT, 0);
  }
  else {
    dbT=GWEN_DB_GetGroup(dbRules, GWEN_PATH_FLAGS_CREATE_GROUP, "rule");
    DBG_DEBUG(0, "Added this group:");
    //GWEN_DB_Dump(dbT, stderr, 2);
  }
  assert(dbT);
  GWEN_DB_AddGroupChildren(dbT, dbRule);
  if (!(flags & TRANSFINDER_FLAGS_USE_CATEGORY))
    GWEN_DB_ClearGroup(dbT, "category");

  GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "name", ruleName.c_str());
  GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "short",
                       ruleDescr.c_str());
  TransactionFinder tf(kb,
                       dbRules,
		       ruleName.c_str(),
                       flags,
                       parent, "TransactionFinder",
                       true);
  if (!title.isEmpty())
    tf.setCaption(title);
  tf.closeButton->setText(tr("Finished"));
  tf.refreshButton->setText(tr("Test"));

  tf.init();
  if (tf.exec()==QDialog::Accepted) {
    DBG_DEBUG(0, "Accepted");
    dbT=tf.getRule();
    assert(dbT);
    GWEN_DB_ClearGroup(dbRule, 0);
    GWEN_DB_AddGroupChildren(dbRule, dbT);
    rv=true;
  }
  else {
    DBG_DEBUG(0, "Rejected");
    rv=false;
  }
  kb->setTransactionMatcherRules(tf.getRules());
  tf.fini();
  return rv;
}




bool
TransactionFinder::selectTransactions(KBanking *kb,
                                      std::list<RefPointer<Transaction> >*tl,
				      const QString &title,
                                      GWEN_TYPE_UINT32 flags,
                                      QWidget* parent,
                                      WFlags fl) {
  std::string ruleName;
  std::string ruleDescr;
  GWEN_DB_NODE *dbRules;
  GWEN_DB_NODE *dbT;
  bool rv;

  ruleName=KBanking::QStringToUtf8String(tr("current"));
  ruleDescr=KBanking::QStringToUtf8String(tr("currently active rule"));
  dbRules=kb->getTransactionMatcherRules();

  dbT=GWEN_DB_FindFirstGroup(dbRules, "rule");
  while(dbT) {
    const char *s;

    s=GWEN_DB_GetCharValue(dbT, "name", 0, 0);
    if (s) {
      if (strcasecmp(s, ruleName.c_str())==0)
        break;
    }
    dbT=GWEN_DB_FindNextGroup(dbT, "rule");
  } // while
  if (!dbT) {
    dbT=GWEN_DB_GetGroup(dbRules, GWEN_PATH_FLAGS_CREATE_GROUP, "rule");
  }
  assert(dbT);
  GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "name", ruleName.c_str());
  GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "short",
                       ruleDescr.c_str());
  TransactionFinder tf(kb,
                       dbRules,
		       ruleName.c_str(),
                       flags,
                       parent, "TransactionFinder",
                       true);
  if (!title.isEmpty())
    tf.setCaption(title);
  tf.closeButton->setText(tr("Return Selected"));
  tf.refreshButton->setText(tr("Apply"));

  tf.init();
  if (tf.exec()==QDialog::Accepted) {
    DBG_DEBUG(0, "Accepted");
    *tl=tf.transactionListView->getSelectedTransactions();
    rv=true;
  }
  else {
    DBG_DEBUG(0, "Rejected");
    rv=false;
  }
  kb->setTransactionMatcherRules(tf.getRules());
  tf.fini();
  return rv;
}






void TransactionFinder::slotCategoryContextMenuRequested(QListViewItem *lvi,
                                                         const QPoint &coord,
                                                         int column) {
  QPopupMenu men(categoryListView, "CategoryContextMenu");
  DBG_ERROR(0, "Right Button Clicked on column %d", column);

  _currentCategoryItem=dynamic_cast<CategoryListViewItem*>(lvi);
  assert(_currentCategoryItem);

  //_currentCategoryItem->setSelected(!_currentCategoryItem->isSelected());

  men.insertItem(tr("(Un)Select this &entry"),
                 this, SLOT(slotCatContextSelectEntry()));
  men.insertItem(tr("(Un)Select this &branch"),
                 this, SLOT(slotCatContextSelectBranch()));
  men.insertItem(tr("(Un)Select &all entries"),
                 this, SLOT(slotCatContextSelectAll()));
  men.exec(coord);
}



void TransactionFinder::slotCatContextSelectEntry() {
  allCategoriesCheck->setChecked(false);
  //_currentCategoryItem->setSelected(!_currentCategoryItem->isSelected());
}



void TransactionFinder::_catSelectBranch(QListViewItem *p, bool on) {
  QListViewItem *lvi;

  p->setSelected(on);
  lvi=p->firstChild();
  while(lvi) {
    _catSelectBranch(lvi, on);
    lvi=lvi->nextSibling();
  }
}


void TransactionFinder::slotCatContextSelectBranch() {
  bool on=_currentCategoryItem->isSelected();
  QListViewItem *lvi;

  //_currentCategoryItem->setSelected(on);
  lvi=_currentCategoryItem->firstChild();
  categoryListView->beginModify();
  while(lvi) {
    _catSelectBranch(lvi, on);
    lvi=lvi->nextSibling();
  }
  allCategoriesCheck->setChecked(false);
  categoryListView->updateAll();
  categoryListView->endModify();
}



void TransactionFinder::slotCatContextSelectAll() {
  bool on=_currentCategoryItem->isSelected();

  categoryListView->beginModify();
  allCategoriesCheck->setChecked(false);
  categoryListView->selectAll(on);
  categoryListView->endModify();
}



void TransactionFinder::slotCategorySelectionChanged() {
  if (!_modifying && allCategoriesCheck->isChecked()) {
    _modifying=true;
    allCategoriesCheck->setChecked(false);
    _modifying=false;
  }
}



void TransactionFinder::slotPayeeSelectionChanged() {
  if (!_modifying && allPayeesCheck->isChecked()) {
    _modifying=true;
    allPayeesCheck->setChecked(false);
    _modifying=false;
  }
}



void TransactionFinder::slotAllPayeeCheckToggled(bool on) {
  if (!_modifying) {
    _modifying=true;
    payeeListView->selectAll(on);
    _modifying=false;
  }
}









