/***************************************************************************
 $RCSfile: account.cpp,v $
                             -------------------
    cvs         : $Id: account.cpp,v 1.24 2005/08/25 15:27:35 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "account.h"
#include "app.h"

#include <gwenhywfar/debug.h>
#include <stdlib.h>
#include <assert.h>



Account::Account(App *app)
:_accountStatus(0)
,_bankingId(0)
,_banking(app)
,_optionUpdateTransactions(true)
,_optionUpdateBalance(true)
,_optionUpdateStandingOrders(true)
,_optionUpdateDatedTransfers(true)
,_optionUpdatePending(true)
,_optionExcludeFromGlobalCall(false)
,_storage(0)
,_lastTransactionDate(0) {
  assert(app);

}



Account::Account(App *app, const AB_ACCOUNT *a)
:_accountStatus(0)
,_bankingId(0)
,_banking(app)
,_optionUpdateTransactions(true)
,_optionUpdateBalance(true)
,_optionUpdateStandingOrders(true)
,_optionUpdateDatedTransfers(true)
,_optionUpdatePending(true)
,_optionExcludeFromGlobalCall(false)
,_storage(0)
,_lastTransactionDate(0){
  const char *p;

  assert(a);
  p=AB_Account_GetCountry(a);
  if (p)
    _country=p;
  p=AB_Account_GetAccountNumber(a);
  if (p)
    _accountNumber=p;
  p=AB_Account_GetAccountName(a);
  if (p)
    _accountName=p;
  p=AB_Account_GetBankCode(a);
  if (p)
    _bankCode=p;
  p=AB_Account_GetBankName(a);
  if (p)
    _bankName=p;
  p=AB_Account_GetOwnerName(a);
  if (p)
    _ownerName=p;
  _bankingId=AB_Account_GetUniqueId(a);
}


Account::Account(const Account &a)
:_accountStatus(0)
,_bankingId(a._bankingId)
,_banking(0)
,_optionUpdateTransactions(true)
,_optionUpdateBalance(true)
,_optionUpdateStandingOrders(true)
,_optionUpdateDatedTransfers(true)
,_optionUpdatePending(true)
,_optionExcludeFromGlobalCall(false)
,_storage(0){
  _bankCode=a._bankCode;
  _accountNumber=a._accountNumber;
  _bankName=a._bankName;
  _accountName=a._accountName;
  _ownerName=a._ownerName;
  _transactions=a._transactions;
  _bankingId=a._bankingId;
  _banking=a._banking;
  _lastTransactionDate=a._lastTransactionDate;
  if (a._accountStatus)
    _accountStatus=AB_AccountStatus_dup(a._accountStatus);

}


Account::~Account(){
  AB_AccountStatus_free(_accountStatus);
}



bool Account::fromDb(GWEN_DB_NODE *db) {
  GWEN_DB_NODE *dbT;
  const char *s;

  s=GWEN_DB_GetCharValue(db, "country", 0, "de");
  if (s && *s)
    _country=s;
  else
    _country="de";

  s=GWEN_DB_GetCharValue(db, "bankCode", 0, "");
  assert(s && *s);
  _bankCode=s;
  s=GWEN_DB_GetCharValue(db, "accountNumber", 0, "");
  assert(s && *s);
  _accountNumber=s;
  _bankName=GWEN_DB_GetCharValue(db, "bankName", 0, "");
  _accountName=GWEN_DB_GetCharValue(db, "accountName", 0, "");
  _ownerName=GWEN_DB_GetCharValue(db, "ownerName", 0, "");

  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "accountStatus");
  if (dbT)
    _accountStatus=AB_AccountStatus_fromDb(dbT);
  _bankingId=GWEN_DB_GetIntValue(db, "bankingId", 0, 0);
  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                       "lastTransactionDate");
  if (dbT)
    _lastTransactionDate=GWEN_Time_fromDb(dbT);

  /* read transfers */
  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                       "transfers");
  if (dbT) {
    GWEN_DB_NODE *dbTransfer;

    dbTransfer=GWEN_DB_FindFirstGroup(dbT, "transfer");
    while(dbTransfer) {
      _transfers.push_back(new Transfer(dbTransfer));
      dbTransfer=GWEN_DB_FindNextGroup(dbTransfer, "transfer");
    } // while
  }

  /* read standing orders */
  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                       "standingOrders");
  if (dbT) {
    GWEN_DB_NODE *dbTransfer;

    dbTransfer=GWEN_DB_FindFirstGroup(dbT, "standingOrder");
    while(dbTransfer) {
      _standingOrders.push_back(new StandingOrder(dbTransfer));
      dbTransfer=GWEN_DB_FindNextGroup(dbTransfer, "standingOrder");
    } // while
  }

  /* read dated transfers */
  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                       "datedTransfers");
  if (dbT) {
    GWEN_DB_NODE *dbTransfer;

    dbTransfer=GWEN_DB_FindFirstGroup(dbT, "datedTransfer");
    while(dbTransfer) {
      _datedTransfers.push_back(new Transfer(dbTransfer));
      dbTransfer=GWEN_DB_FindNextGroup(dbTransfer, "datedTransfer");
    } // while
  }

  // read options
  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                       "options");
  if (dbT) {
    _optionUpdateTransactions=GWEN_DB_GetIntValue(dbT,
                                                  "updateTransactions",
                                                  0, 1);
    _optionUpdateBalance=GWEN_DB_GetIntValue(dbT,
                                             "updateBalance",
                                             0, 1);
    _optionUpdateStandingOrders=GWEN_DB_GetIntValue(dbT,
                                                    "updateStandingOrders",
                                                    0, 1);
    _optionUpdateDatedTransfers=GWEN_DB_GetIntValue(dbT,
                                                    "updateDatedTransfers",
                                                    0, 1);
    _optionUpdatePending=GWEN_DB_GetIntValue(dbT,
                                             "updatePending",
                                             0, 1);
    _optionExcludeFromGlobalCall=GWEN_DB_GetIntValue(dbT,
                                                     "excludeFromGlobalCall",
                                                     0, 0);

  }

  DBG_INFO(0, "Loaded account from DB (%s/%s)",
           _bankCode.c_str(),
           _accountNumber.c_str());
  return true;
}



bool Account::toDb(GWEN_DB_NODE *db) {
  GWEN_DB_NODE *dbT;
  std::list<RefPointer<Transfer> >::iterator itt;
  std::list<RefPointer<StandingOrder> >::iterator its;

  if (!_country.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "country", _country.c_str());
  if (!_bankCode.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "bankCode", _bankCode.c_str());
  if (!_accountNumber.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "accountNumber", _accountNumber.c_str());
  if (!_bankName.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "bankName", _bankName.c_str());
  if (!_accountName.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "accountName", _accountName.c_str());
  if (!_ownerName.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "ownerName", _ownerName.c_str());
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "bankingId", _bankingId);

  // write options
  if (1) {
    GWEN_DB_NODE *dbT;

    dbT=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                         "options");
    assert(dbT);
    GWEN_DB_SetIntValue(dbT,
                        GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "updateTransactions",
                        _optionUpdateTransactions);
    GWEN_DB_SetIntValue(dbT,
                        GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "updateBalance",
                        _optionUpdateBalance);

    GWEN_DB_SetIntValue(dbT,
                        GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "updateStandingOrders",
                        _optionUpdateStandingOrders);

    GWEN_DB_SetIntValue(dbT,
                        GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "updateDatedTransfers",
                        _optionUpdateDatedTransfers);

    GWEN_DB_SetIntValue(dbT,
                        GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "updatePending",
                        _optionUpdatePending);

    GWEN_DB_SetIntValue(dbT,
                        GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "excludeFromGlobalCall",
                        _optionExcludeFromGlobalCall);
  }

  if (_accountStatus) {
    GWEN_DB_NODE *dbT;

    dbT=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_DEFAULT, "accountStatus");
    assert(dbT);
    if (AB_AccountStatus_toDb(_accountStatus, dbT)) {
      DBG_ERROR(0, "Could not store account status");
      return false;
    }
  }

  if (_lastTransactionDate) {
    GWEN_DB_NODE *dbT;

    dbT=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_DEFAULT, "lastTransactionDate");
    assert(dbT);
    if (GWEN_Time_toDb(_lastTransactionDate, dbT)) {
      DBG_ERROR(0, "Could not store date of last transaction");
      return false;
    }
  }

  // save transfers
  dbT=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "transfers");
  for (itt=_transfers.begin(); itt!=_transfers.end(); itt++) {
    GWEN_DB_NODE *dbTransfer;

    dbTransfer=GWEN_DB_GetGroup(dbT, GWEN_PATH_FLAGS_CREATE_GROUP,
                                "transfer");
    assert(dbTransfer);
    if (!(*itt).ref().toDb(dbTransfer)) {
      DBG_ERROR(0, "Could not save transfer");
      return AB_ERROR_GENERIC;
    }
  }

  // save standing orders
  dbT=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "standingOrders");
  for (its=_standingOrders.begin(); its!=_standingOrders.end(); its++) {
    GWEN_DB_NODE *dbTransfer;

    dbTransfer=GWEN_DB_GetGroup(dbT, GWEN_PATH_FLAGS_CREATE_GROUP,
                                "standingOrder");
    assert(dbTransfer);
    if (!(*its).ref().toDb(dbTransfer)) {
      DBG_ERROR(0, "Could not save transfer");
      return AB_ERROR_GENERIC;
    }
  }

  // save dated transfers
  dbT=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "datedTransfers");
  for (itt=_datedTransfers.begin(); itt!=_datedTransfers.end(); itt++) {
    GWEN_DB_NODE *dbTransfer;

    dbTransfer=GWEN_DB_GetGroup(dbT, GWEN_PATH_FLAGS_CREATE_GROUP,
                                "datedTransfer");
    assert(dbTransfer);
    if (!(*itt).ref().toDb(dbTransfer)) {
      DBG_ERROR(0, "Could not save dated transfer");
      return AB_ERROR_GENERIC;
    }
  }

  return true;
}



const std::string &Account::getCountry() const{
  return _country;
}



void Account::setCountry(const std::string &s){
  _country=s;
}



const std::string &Account::getBankCode() const{
  return _bankCode;
}



void Account::setBankCode(const std::string &s){
  _bankCode=s;
}



const std::string &Account::getAccountNumber() const{
  return _accountNumber;
}



void Account::setAccountNumber(const std::string &s){
  _accountNumber=s;
}



const std::string &Account::getBankName() const{
  return _bankName;
}



void Account::setBankName(const std::string &s){
  _bankName=s;
}



const std::string &Account::getAccountName() const{
  return _accountName;
}



void Account::setAccountName(const std::string &s){
  _accountName=s;
}



const std::string &Account::getOwnerName() const{
  return _ownerName;
}



void Account::setOwnerName(const std::string &s){
  _ownerName=s;
}



GWEN_TYPE_UINT32 Account::getBankingId() const{
  return _bankingId;
}



void Account::setBankingId(GWEN_TYPE_UINT32 i){
  _bankingId=i;
}



const AB_ACCOUNT_STATUS *Account::getAccountStatus() const{
  return _accountStatus;
}



void Account::setAccountStatus(const AB_ACCOUNT_STATUS *s){
  assert(s);
  _accountStatus=AB_AccountStatus_dup(s);;
}



const std::list<RefPointer<Transaction> >  &Account::getTransactions() {
  if (_transactions.size()==0) {
    if (!_loadTransactions()) {
      DBG_ERROR(0, "Could not load transactions");
    }
  }
  return _transactions;
}



bool Account::_loadTransactions(unsigned int fromYear,
                                unsigned int fromMonth,
                                unsigned int fromDay) {
  int year, month, day;
  int rv;
  GWEN_TYPE_UINT32 ymd;
  GWEN_IDLIST *idl;

  DBG_INFO(0, "Loading transactions");
  _transactions.clear();
  ymd=(fromYear<<16)+(fromMonth<<8)+fromDay;
  if (!getStorage())
    return false;

  idl=AH_Storage_GetAvailableDays(_storage);
  if (!idl) {
    DBG_ERROR(0, "Error building storage index");
    return false;
  }
  rv=AH_Storage_GetFirstDay(_storage, idl, &year, &month, &day);
  while(!rv) {
    GWEN_TYPE_UINT32 id;
    GWEN_DB_NODE *db;
    GWEN_TYPE_UINT32 cymd;

    DBG_DEBUG(0, "Loading day %04d/%02d/%02d",
              year, month, day);
    cymd=(year<<16)+(month<<8)+day;
    if (cymd>=ymd) {
      id=AH_Storage_OpenDay(_storage, year, month, day, 1);
      if (!id) {
        DBG_ERROR(0, "Error loading day %04d/%02d/%02d",
                  year, month, day);
        GWEN_IdList_free(idl);
        return false;
      }

      db=AH_Storage_GetFirstTransaction(_storage, id);
      if (!db) {
        DBG_WARN(0, "No transactions in day %04d/%02d/%02d",
                 year, month, day);
      }
      while(db) {
        RefPointer<Transaction> t;

        if (!t.ref().fromDb(db)) {
          DBG_ERROR(0, "Bad format of transaction in day %04d/%02d/%02d",
                    year, month, day);
          AH_Storage_AbandonDay(_storage, id);
          GWEN_IdList_free(idl);
          return false;
        }
        DBG_VERBOUS(0, "Got a transaction");
        _transactions.push_back(t);
        db=AH_Storage_GetNextTransaction(_storage, id);
      }
      if (AH_Storage_CloseDay(_storage, id)) {
        DBG_ERROR(0, "Error closing day %04d/%02d/%02d",
                  year, month, day);
        GWEN_IdList_free(idl);
        return false;
      }
    }
    rv=AH_Storage_GetNextDay(_storage, idl, &year, &month, &day);
    if (rv) {
      DBG_DEBUG(0, "End of day list.");
    }
  } /* if first day */
  GWEN_IdList_free(idl);

  return true;
}



bool Account::loadTransactions(GWEN_TIME *fromDate,
                               GWEN_TIME *toDate,
                               std::list<Transaction> &tl) {
  int year, month, day;
  int rv;
  GWEN_TYPE_UINT32 fromYmd;
  GWEN_TYPE_UINT32 toYmd;
  GWEN_IDLIST *idl;

  if (fromDate) {
    if (GWEN_Time_GetBrokenDownUtcDate(fromDate,
                                       &day, &month, &year)) {
      DBG_ERROR(0, "Error breaking down date");
      return false;
    }
    fromYmd=(year<<16)+((month+1)<<8)+day;
  }
  else
    fromYmd=0;

  if (toDate) {
    if (GWEN_Time_GetBrokenDownUtcDate(toDate,
                                       &day, &month, &year)) {
      DBG_ERROR(0, "Error breaking down date");
      return false;
    }
    toYmd=(year<<16)+((month+1)<<8)+day;
  }
  else
    toYmd=0xffffffff;

  DBG_INFO(0, "Loading transactions");
  if (!getStorage())
    return false;

  idl=AH_Storage_GetAvailableDays(_storage);
  if (!idl) {
    DBG_ERROR(0, "Error building storage index");
    return false;
  }
  rv=AH_Storage_GetFirstDay(_storage, idl, &year, &month, &day);
  while(!rv) {
    GWEN_TYPE_UINT32 id;
    GWEN_DB_NODE *db;
    GWEN_TYPE_UINT32 cymd;

    cymd=(year<<16)+(month<<8)+day;
    if (cymd>=fromYmd && cymd<=toYmd) {
      DBG_DEBUG(0, "Loading day %04d/%02d/%02d",
                year, month, day);

      id=AH_Storage_OpenDay(_storage, year, month, day, 1);
      if (!id) {
        DBG_ERROR(0, "Error loading day %04d/%02d/%02d",
                  year, month, day);
        GWEN_IdList_free(idl);
        return false;
      }

      db=AH_Storage_GetFirstTransaction(_storage, id);
      if (!db) {
        DBG_WARN(0, "No transactions in day %04d/%02d/%02d",
                 year, month, day);
      }
      while(db) {
        Transaction t;

        if (!t.fromDb(db)) {
          DBG_ERROR(0, "Bad format of transaction in day %04d/%02d/%02d",
                    year, month, day);
          AH_Storage_AbandonDay(_storage, id);
          GWEN_IdList_free(idl);
          return false;
        }
        DBG_DEBUG(0, "Got a transaction");
        tl.push_back(t);
        db=AH_Storage_GetNextTransaction(_storage, id);
      }
      if (AH_Storage_CloseDay(_storage, id)) {
        DBG_ERROR(0, "Error closing day %04d/%02d/%02d",
                  year, month, day);
        GWEN_IdList_free(idl);
        return false;
      }
    }
    rv=AH_Storage_GetNextDay(_storage, idl, &year, &month, &day);
    if (rv) {
      DBG_INFO(0, "End of day list.");
    }
  } /* if first day */
  GWEN_IdList_free(idl);

  return true;
}



bool Account::loadTransactions(GWEN_TIME *fromDate,
                               GWEN_TIME *toDate,
                               std::list<RefPointer<Transaction> > &tl) {
  int year, month, day;
  int rv;
  GWEN_TYPE_UINT32 fromYmd;
  GWEN_TYPE_UINT32 toYmd;
  GWEN_IDLIST *idl;

  if (fromDate) {
    if (GWEN_Time_GetBrokenDownUtcDate(fromDate,
                                       &day, &month, &year)) {
      DBG_ERROR(0, "Error breaking down date");
      return false;
    }
    fromYmd=(year<<16)+((month+1)<<8)+day;
  }
  else
    fromYmd=0;

  if (toDate) {
    if (GWEN_Time_GetBrokenDownUtcDate(toDate,
                                       &day, &month, &year)) {
      DBG_ERROR(0, "Error breaking down date");
      return false;
    }
    toYmd=(year<<16)+((month+1)<<8)+day;
  }
  else
    toYmd=0xffffffff;

  DBG_INFO(0, "Loading transactions");
  if (!getStorage())
    return false;

  idl=AH_Storage_GetAvailableDays(_storage);
  if (!idl) {
    DBG_ERROR(0, "Error building storage index");
    return false;
  }

  rv=AH_Storage_GetFirstDay(_storage, idl, &year, &month, &day);
  while(!rv) {
    GWEN_TYPE_UINT32 id;
    GWEN_DB_NODE *db;
    GWEN_TYPE_UINT32 cymd;

    cymd=(year<<16)+(month<<8)+day;
    if (cymd>=fromYmd && cymd<=toYmd) {
      DBG_DEBUG(0, "Loading day %04d/%02d/%02d",
                year, month, day);

      id=AH_Storage_OpenDay(_storage, year, month, day, 1);
      if (!id) {
        DBG_ERROR(0, "Error loading day %04d/%02d/%02d",
                  year, month, day);
        GWEN_IdList_free(idl);
        return false;
      }

      db=AH_Storage_GetFirstTransaction(_storage, id);
      if (!db) {
        DBG_WARN(0, "No transactions in day %04d/%02d/%02d",
                 year, month, day);
      }
      while(db) {
        RefPointer<Transaction> t;

        t=new Transaction();
        if (!t.ref().fromDb(db)) {
          DBG_ERROR(0, "Bad format of transaction in day %04d/%02d/%02d",
                    year, month, day);
          AH_Storage_AbandonDay(_storage, id);
          GWEN_IdList_free(idl);
          return false;
        }
        DBG_DEBUG(0, "Got a transaction");
        tl.push_back(t);
        db=AH_Storage_GetNextTransaction(_storage, id);
      }
      if (AH_Storage_CloseDay(_storage, id)) {
        DBG_ERROR(0, "Error closing day %04d/%02d/%02d",
                  year, month, day);
        GWEN_IdList_free(idl);
        return false;
      }
    }
    rv=AH_Storage_GetNextDay(_storage, idl, &year, &month, &day);
    if (rv) {
      DBG_INFO(0, "End of day list.");
    }
  } /* if first day */
  GWEN_IdList_free(idl);

  return true;
}



App *Account::getApi() const {
  return _banking;
}



AH_STORAGE *Account::getStorage(){
  if (_storage==0) {
    GWEN_BUFFER *pbuf;

    if (_bankCode.empty() || _accountNumber.empty()){
      DBG_ERROR(0, "Bank code AND account number needed");
      return false;
    }

    pbuf=GWEN_Buffer_new(0, 256, 0, 1);
    if (_banking->getAppUserDataDir(pbuf)) {
      DBG_ERROR(0, "Error getting data path");
      GWEN_Buffer_free(pbuf);
      return false;
    }
    GWEN_Buffer_AppendString(pbuf, "/transactions/280/");
    GWEN_Buffer_AppendString(pbuf, _bankCode.c_str());
    GWEN_Buffer_AppendString(pbuf, "/");
    GWEN_Buffer_AppendString(pbuf, _accountNumber.c_str());
    _storage=AH_Storage_new(GWEN_Buffer_GetStart(pbuf));
    GWEN_Buffer_free(pbuf);
  }
  assert(_storage);
  return _storage;
}



void Account::infoTransactionsChanged(){
  _transactions.clear();
}



void Account::setLastTransactionDate(const GWEN_TIME *t){
  if (_lastTransactionDate && t) {
    if (GWEN_Time_Diff(t, _lastTransactionDate)>0) {
      GWEN_Time_free(_lastTransactionDate);
      _lastTransactionDate=GWEN_Time_dup(t);
    }
  }
  else {
    GWEN_Time_free(_lastTransactionDate);
    if (t) _lastTransactionDate=GWEN_Time_dup(t);
    else _lastTransactionDate=0;
  }
}



const GWEN_TIME *Account::getLastTransactionDate() const{
  return _lastTransactionDate;
}



bool Account::saveTransaction(const Transaction *t) {
  const GWEN_TIME *ti;
  int year, month, day;
  GWEN_TYPE_UINT32 id;
  GWEN_TYPE_UINT32 tid;
  GWEN_DB_NODE *dbTransaction;
  GWEN_DB_NODE *dbT;

  dbTransaction=GWEN_DB_Group_new("transaction");
  if (!t->toDb(dbTransaction)) {
    DBG_ERROR(0, "Could not store transaction to db");
    GWEN_DB_Group_free(dbTransaction);
    return false;
  }
  ti=t->getDate();
  if (!ti)
    ti=t->getValutaDate();
  if (!ti) {
    DBG_ERROR(0, "No date in transaction, can not save");
    GWEN_DB_Group_free(dbTransaction);
    return false;
  }
  if (GWEN_Time_GetBrokenDownUtcDate(ti, &day, &month, &year)) {
    DBG_ERROR(0, "Bad date in transaction, can not save");
    GWEN_DB_Group_free(dbTransaction);
    return false;
  }

  id=AH_Storage_OpenDay(_storage, year, month+1, day, 0);
  if (!id) {
    DBG_ERROR(0, "Error loading day %04d/%02d/%02d",
              year, month, day);
    GWEN_DB_Group_free(dbTransaction);
    return false;
  }

  tid=t->getTransactionId();
  assert(tid);
  dbT=AH_Storage_GetFirstTransaction(_storage, id);
  while(dbT) {
    GWEN_TYPE_UINT32 i;

    i=GWEN_DB_GetIntValue(dbT, "transactionId", 0, -1);
    if (tid==i)
      break;
    dbT=AH_Storage_GetNextTransaction(_storage, id);
  }
  if (dbT)
    GWEN_DB_ClearGroup(dbT, 0);

  if (!dbT) {
    DBG_INFO(0, "Transaction not found, adding it");
    if (AH_Storage_AddTransaction(_storage, id, dbTransaction)) {
      GWEN_DB_Group_free(dbTransaction);
      AH_Storage_AbandonDay(_storage, id);
      return false;
    }
  }
  else {
    DBG_INFO(0, "Transaction found, modifying");
    GWEN_DB_AddGroupChildren(dbT, dbTransaction);
    if (AH_Storage_CloseDay(_storage, id)) {
      DBG_ERROR(0, "Could not close day %04d/%02d/%02d",
                year, month, day);
      GWEN_DB_Group_free(dbTransaction);
      return false;
    }
  }
  GWEN_DB_Group_free(dbTransaction);

  return true;
}



bool Account::removeTransaction(const Transaction *t) {
  const GWEN_TIME *ti;
  int year, month, day;
  GWEN_TYPE_UINT32 id;
  GWEN_TYPE_UINT32 tid;
  GWEN_DB_NODE *dbT;

  ti=t->getDate();
  if (!ti)
    ti=t->getValutaDate();
  if (!ti) {
    DBG_ERROR(0, "No date in transaction, can not remove");
    return false;
  }
  if (GWEN_Time_GetBrokenDownUtcDate(ti, &day, &month, &year)) {
    DBG_ERROR(0, "Bad date in transaction, can not remove");
    return false;
  }

  id=AH_Storage_OpenDay(_storage, year, month+1, day, 0);
  if (!id) {
    DBG_ERROR(0, "Error loading day %04d/%02d/%02d",
              year, month, day);
    return false;
  }

  tid=t->getTransactionId();
  assert(tid);
  dbT=AH_Storage_GetFirstTransaction(_storage, id);
  while(dbT) {
    GWEN_TYPE_UINT32 i;

    i=GWEN_DB_GetIntValue(dbT, "transactionId", 0, -1);
    if (tid==i)
      break;
    dbT=AH_Storage_GetNextTransaction(_storage, id);
  }

  if (dbT) {
    DBG_ERROR(0, "Transaction found, removing");
    GWEN_DB_UnlinkGroup(dbT);
    GWEN_DB_Group_free(dbT);
    if (AH_Storage_CloseDay(_storage, id)) {
      DBG_ERROR(0, "Could not close day %04d/%02d/%02d",
                year, month, day);
      return false;
    }
  }

  return true;
}



bool Account::saveTransactions() {
  return true;
}



std::list<RefPointer<Transfer> > &Account::getTransfers(){
  return _transfers;
}



void Account::addTransfer(RefPointer<Transfer> t){
  _transfers.push_back(t);
}



std::list<RefPointer<StandingOrder> > &Account::getStandingOrders(){
  return _standingOrders;
}



void Account::addStandingOrder(RefPointer<StandingOrder> t){
  _standingOrders.push_back(t);
}



void Account::clearStandingOrders() {
  _standingOrders.clear();
}



std::list<RefPointer<Transfer> > &Account::getDatedTransfers() {
  return _datedTransfers;
}



void Account::addDatedTransfer(RefPointer<Transfer> t) {
  _datedTransfers.push_back(t);
}



void Account::clearDatedTransfers() {
  _datedTransfers.clear();
}



bool Account::getOptionUpdateTransactions() const {
  return _optionUpdateTransactions;
}



void Account::setOptionUpdateTransactions(bool b) {
  _optionUpdateTransactions=b;
}



bool Account::getOptionUpdateBalance() const {
  return _optionUpdateBalance;
}



void Account::setOptionUpdateBalance(bool b) {
  _optionUpdateBalance=b;
}



bool Account::getOptionUpdateStandingOrders() const {
  return _optionUpdateStandingOrders;
}



void Account::setOptionUpdateStandingOrders(bool b) {
  _optionUpdateStandingOrders=b;
}



bool Account::getOptionUpdateDatedTransfers() const {
  return _optionUpdateDatedTransfers;
}



void Account::setOptionUpdateDatedTransfers(bool b) {
  _optionUpdateDatedTransfers=b;
}



bool Account::getOptionUpdatePending() const {
  return _optionUpdatePending;
}



void Account::setOptionUpdatePending(bool b) {
  _optionUpdatePending=b;
}



bool Account::getOptionExcludeFromGlobalCall() const {
  return _optionExcludeFromGlobalCall;
}



void Account::setOptionExcludeFromGlobalCall(bool b) {
  _optionExcludeFromGlobalCall=b;
}














