/***************************************************************************
 $RCSfile: datedxferview.cpp,v $
                             -------------------
    cvs         : $Id: datedxferview.cpp,v 1.2 2005/07/13 13:47:35 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "datedxferview.h"
#include "edittransaction.h"
#include "kbanking.h"


#ifdef WIN32
# define strcasecmp stricmp
#endif

#include <qevent.h>
#include <qpushbutton.h>
#include <qgroupbox.h>
#include <qmessagebox.h>
#include <qlayout.h>

#include <gwenhywfar/debug.h>
#include <aqbanking/jobcreatedatedtransfer.h>
#include <aqbanking/jobmodifydatedtransfer.h>
#include <aqbanking/jobdeletedatedtransfer.h>


#define BUTTON_WIDTH 110


DatedTransferView::DatedTransferView(KBanking *app,
                                     QWidget* parent,
                                     const char* name,
                                     WFlags fl)
:DatedTransferViewUi(parent, name, fl), _app(app) {
  assert(app);

  // Manually create and add layout here because the .ui-generated
  // QGroupBox doesn't have one.
  transferBox->setColumnLayout(0, Qt::Vertical );
  QBoxLayout *transferBoxLayout = new QHBoxLayout( transferBox->layout() );
  transferBoxLayout->setAlignment( Qt::AlignTop );

  _xaList=new TransferListView((QWidget*)transferBox, name);
  transferBoxLayout->addWidget(_xaList);

  QObject::connect((QObject*)app->flagStaff(),
                   SIGNAL(signalDatedTransfersUpdated()),
                   this, SLOT(slotUpdated()));

  QObject::connect((QObject*)detailsButton, SIGNAL(clicked()),
                   this, SLOT(slotDetails()));
  QObject::connect((QObject*)newDatedTransferButton, SIGNAL(clicked()),
                   this, SLOT(slotNewDatedTransfer()));
  QObject::connect((QObject*)modifyDatedTransferButton, SIGNAL(clicked()),
                   this, SLOT(slotModifyDatedTransfer()));
  QObject::connect((QObject*)deleteDatedTransferButton, SIGNAL(clicked()),
                   this, SLOT(slotDeleteDatedTransfer()));
}



DatedTransferView::~DatedTransferView(){
}


bool DatedTransferView::init(){
  GWEN_DB_NODE *db;
  std::string s;
  int i, j;

  db=_app->getAppData();
  if (db) {
    db=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                        "gui/views/datedTransferView");
    if (db) {
      int w;
      int h;
      const char *p;

      /* found settings */
      p=GWEN_DB_GetCharValue(db, "sortOrder", 0, "ascending");
      if (p) {
        if (strcasecmp(p, "ascending")==0)
          _xaList->setSortOrder(Qt::Ascending);
        else
          if (strcasecmp(p, "descending")==0)
            _xaList->setSortOrder(Qt::Descending);
      }
      i=GWEN_DB_GetIntValue(db, "sortColumn", 0, -1);
      if (i!=-1)
        _xaList->setSortColumn(i);
      w=GWEN_DB_GetIntValue(db, "width", 0, 640);
      h=GWEN_DB_GetIntValue(db, "height", 0, 480);
      resize(w, h);
      for (i=0; i<_xaList->columns(); i++) {
        _xaList->setColumnWidthMode(i, QListView::Manual);
        j=GWEN_DB_GetIntValue(db, "columns", i, -1);
        if (j!=-1)
          _xaList->setColumnWidth(i, j);
      } /* for */
    } // if transaction view setting
  } // if application settings
  slotUpdated();

  return true;
}



bool DatedTransferView::fini(){
  GWEN_DB_NODE *db;
  int i, j;
  std::string s;

  db=_app->getAppData();
  assert(db);
  db=GWEN_DB_GetGroup(db,
                      GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                      "gui/views/datedTransferView");
  assert(db);
  db=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                      s.c_str());
  assert(db);
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_DEFAULT,
                      "width", width());
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_DEFAULT,
                      "height", height());

  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_DEFAULT,
                      "sortColumn", _xaList->sortColumn());
  switch(_xaList->sortOrder()) {
  case Qt::Ascending:
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT,
                         "sortOrder", "ascending");
    break;
  case Qt::Descending:
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT,
                         "sortOrder", "descending");
    break;
  default:
    break;
  }

  for (i=0; i<_xaList->columns(); i++) {
    j=_xaList->columnWidth(i);
    GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_DEFAULT,
                        "columns", j);
  } /* for */

  return true;
}



void DatedTransferView::closeEvent(QCloseEvent *e){
  if (!fini()) {
    DBG_ERROR(0, "Could not deinit transaction view");
  }
  e->accept();
}



void DatedTransferView::slotUpdated() {
  _xaList->clear();
  _xaList->addTransfers(_app->getDatedTransfers());
}



void DatedTransferView::slotDetails() {
}



AB_JOB *DatedTransferView::_newDatedXferJob(AB_JOB_TYPE jt,
                                      RefPointer<Transfer> tmpl){
  RefPointer<Transfer> t;
  bool b;
  int rv;
  AB_JOB *job=0;

  if (tmpl.isValid())
    t=new Transfer(tmpl.ref());
  else {
    t=new Transfer();
    t.ref().setTextKey(52);
  }

  while(1) {
    GWEN_DB_NODE *dbT;
    int bad=false;
    AB_BANKINFO_CHECKRESULT res;
    int i;
    AB_ACCOUNT *ba;
    Account *acc;

    switch(jt) {
    case AB_Job_TypeCreateDatedTransfer:
      b=EditTransaction::createDatedTransfer(_app, t,
                                             tr("New Dated Transfer"),
                                             this, true);
      break;

    case AB_Job_TypeModifyDatedTransfer:
      DBG_ERROR(0, "Modifying existing dated transfer");
      b=EditTransaction::modifyDatedTransfer(_app, t,
                                             tr("Modify Dated Transfer"),
                                             this, true);
      break;

    default:
      b=false;
    }

    if (!b) {
      DBG_INFO(0, "User aborted");
      AB_Job_free(job);
      return 0;
    }

    // dump transaction
    DBG_NOTICE(0, "Accepted");
    dbT=GWEN_DB_Group_new("DatedTransfer");
    if (!t.ref().toDb(dbT)) {
      DBG_ERROR(0, "Could not write DB");
    }
    else {
#ifndef WIN32
	  // In MSVC this function crashes
      GWEN_DB_Dump(dbT, stderr, 2);
#endif
    }
    GWEN_DB_Group_free(dbT);


    // get account for this new transaction
    acc=_app->findAccount(t.ref().getLocalBankCode().c_str(),
                          t.ref().getLocalAccountNumber().c_str());
    assert(acc);
    ba=_app->getAccount(acc->getBankingId());
    if (!ba) {
      DBG_ERROR(0, "Account not available");
      QMessageBox::critical(0,
                            tr("Account Not Available"),
                            tr("The account you requested is not "
                               "available\n"
                               "with any backend."),
                            tr("Dismiss"), 0, 0, 0);
      return 0;
    }

    // create job
    switch(jt) {
    case AB_Job_TypeCreateDatedTransfer:
      job=AB_JobCreateDatedTransfer_new(ba);
      break;

    case AB_Job_TypeModifyDatedTransfer:
      job=AB_JobModifyDatedTransfer_new(ba);
      break;

    default:
      return 0;
    }

    rv=AB_Job_CheckAvailability(job);
    if (rv) {
      DBG_NOTICE(0, "Job is not available (%d)", rv);
      QMessageBox::critical(0,
                            tr("Job Not Available"),
                            tr("The job you requested is not "
                               "available with\n"
                               "the backend which handles this account.\n"),
                            tr("Dismiss"), 0, 0, 0);
      AB_Job_free(job);
      return 0;
    }

    res=AB_Banking_CheckAccount(_app->getCInterface(),
                                t.ref().getRemoteCountry().c_str(),
                                0,
                                t.ref().getRemoteBankCode().c_str(),
                                t.ref().getRemoteAccountNumber().c_str());
    switch(res) {
    case AB_BankInfoCheckResult_NotOk:
      DBG_INFO(0, "Account check result: %d", res);
      i=QMessageBox::warning(0,
                             tr("Warning"),
                             tr("<qt>"
                                "<p>"
                                "The given bank code and account number "
                                "combination is invalid."
                                "</p>"
                                "Do you want to review the order?"
                                "<qt>"
                               ),
                             tr("Yes"), tr("No"), tr("Abort"));
      if (i==0)
        bad=true;
      else if (i!=1) {
        AB_Job_free(job);
        return 0;
      }
      break;

    case AB_BankInfoCheckResult_UnknownBank:
      DBG_INFO(0, "Account check result: %d", res);
      i=QMessageBox::warning(0,
                             tr("Warning"),
                             tr("<qt>"
                                "<p>"
                                "The given bank code is invalid."
                                "</p>"
                                "Do you want to review the order?"
                                "<qt>"
                               ),
                             tr("Yes"), tr("No"), tr("Abort"));
      if (i==0)
        bad=true;
      else if (i!=1) {
        AB_Job_free(job);
        return 0;
      }
      break;

    case AB_BankInfoCheckResult_UnknownResult:
      DBG_INFO(0, "Nothing known about account");
      break;
    case AB_BankInfoCheckResult_Ok:
      DBG_INFO(0, "Bank code/account number combination is ok");
      break;
    default:
      break;
    }

    if (!bad) {
      AB_TRANSACTION *abt;

      abt=t.ref().toBankingTransaction();
      assert(abt);

      switch(jt) {
      case AB_Job_TypeCreateDatedTransfer:
        rv=AB_JobCreateDatedTransfer_SetTransaction(job, abt);
        break;

      case AB_Job_TypeModifyDatedTransfer:
        rv=AB_JobModifyDatedTransfer_SetTransaction(job, abt);
        break;

      default:
        AB_Transaction_free(abt);
        AB_Job_free(job);
        return 0;
      }

      if (rv) {
        DBG_NOTICE(0, "Could not set transfer (%d)", rv);
        if (QMessageBox::critical(0,
                                  tr("Bad Transfer Data"),
                                  tr("<qt>"
                                     "<p>"
                                     "There seems to be some errors "
                                     "in the dated transfer data."
                                     "</p>"
                                     "<p>"
                                     "Do you want to review the order?"
                                     "</p>"
                                     "<qt>"
                                    ),
                                  tr("Yes"), tr("Abort"), 0, 0)!=0){
          AB_Transaction_free(abt);
          AB_Job_free(job);
          return 0;
        }
        bad=true;
      }
      AB_Transaction_free(abt);
    } // if ok

    if (!bad) {
      break;
    }
    AB_Job_free(job);
    job=0;
  } // while

  return job;
}



void DatedTransferView::_createDatedTransfer(RefPointer<Transfer> tmpl){
  AB_JOB *job;

  job=_newDatedXferJob(AB_Job_TypeCreateDatedTransfer, tmpl);
  if (job) {
    int rv;

    DBG_NOTICE(0, "Enqueuing job");
    rv=_app->enqueueJob(job);
    if (rv) {
      DBG_NOTICE(0, "Error %d", rv);
      AB_Job_free(job);
      return;
    }
    AB_Job_free(job);

    QMessageBox::information(0,
                             tr("Job enqueued"),
                             tr("<qt>"
                                "A dated transfer creation job has been "
                                "enqueued.\n"
                                "Please go to the <i>outbox</i> and press\n"
                                "the <i>execute</i> button there to "
                                "actually send\n"
                                "the enqueued jobs."
                                "</qt>"),
                             tr("Dismiss"), 0, 0, 0);
  }
}



void DatedTransferView::_modifyDatedTransfer(RefPointer<Transfer> tmpl){
  AB_JOB *job;

  job=_newDatedXferJob(AB_Job_TypeModifyDatedTransfer, tmpl);
  if (job) {
    int rv;

    DBG_NOTICE(0, "Enqueuing job");
    rv=_app->enqueueJob(job);
    if (rv) {
      DBG_NOTICE(0, "Error %d", rv);
      AB_Job_free(job);
      return;
    }
    AB_Job_free(job);

    QMessageBox::information(0,
                             tr("Job enqueued"),
                             tr("<qt>"
                                "A dated transfer modification job has been "
                                "enqueued.\n"
                                "Please go to the <i>outbox</i> and press\n"
                                "the <i>execute</i> button there to "
                                "actually send\n"
                                "the enqueued jobs."
                                "</qt>"),
                             tr("Dismiss"), 0, 0, 0);
  }
}



void DatedTransferView::slotNewDatedTransfer() {
  _createDatedTransfer();
}



void DatedTransferView::slotModifyDatedTransfer() {
  RefPointer<Transfer> t;

  t=_xaList->getCurrentTransfer();
  if (!t.isValid()) {
    DBG_ERROR(0, "No current dated transfer");
    QMessageBox::critical(0,
                          tr("No Selection"),
                          tr("<qt>"
                             "Please select a dated transfer for "
                             "modification"
                             "</qt>"),
                          tr("Dismiss"), 0, 0, 0, 0);
    return;
  }

  _modifyDatedTransfer(t);
}



void DatedTransferView::slotDeleteDatedTransfer() {
  RefPointer<Transfer> t;
  AB_ACCOUNT *ba;
  const char *bankCode;
  const char *accountNumber;
  AB_JOB *job;
  int rv;
  AB_TRANSACTION *abt;

  t=_xaList->getCurrentTransfer();
  if (!t.isValid()) {
    DBG_ERROR(0, "No current dated transfer");
    QMessageBox::critical(0,
                          tr("No Selection"),
                          tr("<qt>"
                             "Please select a dated transfer for "
                             "modification"
                             "</qt>"),
                          tr("Dismiss"), 0, 0, 0, 0);
    return;
  }

  bankCode=t.ref().getLocalBankCode().c_str();
  accountNumber=t.ref().getLocalAccountNumber().c_str();
  ba=AB_Banking_GetAccountByCodeAndNumber(_app->getCInterface(),
                                          bankCode,
                                          accountNumber);
  if (!ba) {
    DBG_ERROR(0, "Account for dated transfer not found");
    QMessageBox::critical(0,
                          tr("Unknown Account"),
                          tr("<qt>"
                             "This dated transfer does not belong "
                             "to any known account."
                             "</qt>"),
                          tr("Dismiss"), 0, 0, 0, 0);
    return;
  }

  job=AB_JobDeleteDatedTransfer_new(ba);
  rv=AB_Job_CheckAvailability(job);
  if (rv) {
    DBG_ERROR(0, "Job is not available (%d)", rv);
    QMessageBox::critical(0,
                          tr("Job Not Available"),
                          tr("<qt>"
                             "The selected job is not available with "
                             "your bank and this account."
                             "</qt>"),
                          tr("Dismiss"), 0, 0, 0, 0);
    AB_Job_free(job);
    return;
  }

  abt=t.ref().toBankingTransaction();
  assert(abt);

  rv=AB_JobDeleteDatedTransfer_SetTransaction(job, abt);
  if (rv) {
    DBG_ERROR(0, "Error on SetTransaction: %d", rv);
    QMessageBox::critical(0,
                          tr("Internal Error"),
                          tr("<qt>"
                             "An internal error occurred. "
                             "Please check console logs and report to "
                             "<b>martin@libchipcard.de</b>"
                             "</qt>"),
                          tr("Dismiss"), 0, 0, 0, 0);
    AB_Job_free(job);
    return;
  }

  rv=_app->enqueueJob(job);
  AB_Job_free(job);
  if (rv) {
    DBG_NOTICE(0, "Error enqueing job: %d", rv);
    QMessageBox::critical(0,
                          tr("Job Not Enqueued"),
                          tr("<qt>"
                             "The selected job is available with "
                             "your bank and this account but an error "
                             "occurred while enqueuing it."
                             "</qt>"),
                          tr("Dismiss"), 0, 0, 0, 0);
    return;
  }

  slotUpdated();

  QMessageBox::information(0,
                           tr("Job enqueued"),
                           tr("<qt>"
                              "A job to delete a dated transfer has been "
                              "enqueued.\n"
                              "Please go to the <i>outbox</i> and press\n"
                              "the <i>execute</i> button there to "
                              "actually send\n"
                              "the enqueued jobs."
                              "</qt>"),
                           tr("Dismiss"), 0, 0, 0);

}






