/***************************************************************************
 $RCSfile: kbanking.cpp,v $
                             -------------------
    cvs         : $Id: kbanking.cpp,v 1.49 2006/03/08 15:38:34 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "kbanking_p.h"
#include "ktimport.h"
#include "flagstaff.h"
#include "checkduplicates.h"
#include "transactionwindow.h"
#include "selecttransfers.h"
#include "edittransaction.h"
#include "editeutransaction.h"
#include "mainwindow.h"
#include "cfgmoduleqbm.h"

#include "report.h"
#include "simplereport.h"
#include "summaryreport.h"
#include "categoryreport.h"
#include "graphreport.h"
/* Note: add new reports to fn _sampleReportModules() */

#include <assert.h>
#include <qstring.h>
#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qapplication.h>
#include <qdatetime.h>

#include <aqbanking/jobsingletransfer.h>
#include <aqbanking/jobsingledebitnote.h>
#include <aqbanking/jobeutransfer.h>

#include <gwenhywfar/debug.h>
#include <gwenhywfar/text.h>

#include <qwidget.h>


KBanking::KBanking(const char *appname,
                   const char *fname,
                   const char *name)
:App(appname, fname)
,_mainWindow(0)
,_lastWidgetId(0)
,_logLevel(AB_Banking_LogLevelInfo)
,_useColoursForCategoryText(true)
,_lastReportId(0){
  _flagStaff=new FlagStaff();
  _cfgModule=new CfgModuleQbm(this);
  setAppConfigModule(_cfgModule);
}



KBanking::~KBanking(){
  delete _cfgModule;
  delete _flagStaff;
}



FlagStaff *KBanking::flagStaff(){
  return _flagStaff;
}



std::string KBanking::tr(const char *s) {
  return QBanking::QStringToUtf8String(QWidget::tr(s));
}



int KBanking::executeQueue(AB_IMEXPORTER_CONTEXT *ctx){
  int rv;

  rv=QBanking::executeQueue(ctx);
  _flagStaff->queueUpdated();
  return rv;
}



void KBanking::transfersUpdated(){
  _flagStaff->transfersUpdated();
}



void KBanking::standingOrdersUpdated(){
  _flagStaff->standingOrdersUpdated();
}



bool KBanking::importContext(AB_IMEXPORTER_CONTEXT *ctx,
                             GWEN_TYPE_UINT32 flags){
  KTransactionImporter imp(this, ctx, flags);

  if (flags & QBANKING_IMPORTER_FLAGS_AS_ORDERS) {
    AB_IMEXPORTER_ACCOUNTINFO *ai;
    std::list<RefPointer<Transfer> > tl;
    std::list<RefPointer<Transfer> > stl;

    ai=AB_ImExporterContext_GetFirstAccountInfo(ctx);
    if (!ai) {
      DBG_ERROR(0, "No accounts");
    }
    while(ai) {
      const AB_TRANSACTION *t;

      DBG_DEBUG(0, "Checking account (%s/%s)",
                AB_ImExporterAccountInfo_GetBankCode(ai),
                AB_ImExporterAccountInfo_GetAccountNumber(ai));
      t=AB_ImExporterAccountInfo_GetFirstTransaction(ai);
      while(t) {
        Transfer *tr;

        if (AB_Transaction_GetType(t)==AB_Transaction_TypeDebitNote)
          tr=new Transfer(t, Transfer::TransferType_DebitNote);
        else
          tr=new Transfer(t, Transfer::TransferType_Simple);
        tl.push_back(tr);
        t=AB_ImExporterAccountInfo_GetNextTransaction(ai);
      } /* while */

      ai=AB_ImExporterContext_GetNextAccountInfo(ctx);
    } /* while accounts */

    /* now we have a list of transfer orders */
    if (!SelectTransfers::selectTransfers(this, tl, stl)) {
      return false;
    }
    else {
      std::list<RefPointer<Transfer> >::iterator it;
      int i=0;
      int rv = AB_ERROR_USER_ABORT;

      for (it=stl.begin(); it!=stl.end(); it++) {
        switch((*it).ref().getTransferType()) {
        case Transfer::TransferType_Simple:
          rv=transferFromTemplate((*it).cast<Transaction>(),
                                  getParentWidget());
          break;
        case Transfer::TransferType_DebitNote:
          rv=debitNoteFromTemplate((*it).cast<Transaction>(),
                                   getParentWidget());
          break;
        case Transfer::TransferType_EuTransfer:
          rv=euTransferFromTemplate((*it).cast<Transaction>(),
                                    getParentWidget());
          break;
        case Transfer::TransferType_Unknown:
          abort();
        }

        if (rv==AB_ERROR_USER_ABORT) {
          DBG_ERROR(0, "User aborted");
          if (i)
            _flagStaff->transfersUpdated();
          return rv;
        }
        i++;
      } // for
      if (i)
        _flagStaff->transfersUpdated();
      else {
        DBG_ERROR(0, "No transfers selected...");
      }
    }
  }
  else {
    DBG_INFO(0, "Importing transactions...");
    if (!imp.importContext()) {
      DBG_INFO(0, "Error importing transactions");
      return false;
    }

    _flagStaff->accountsUpdated();
    _flagStaff->transfersUpdated();
    _flagStaff->datedTransfersUpdated();
    _flagStaff->standingOrdersUpdated();
    _flagStaff->payeesUpdated();
  }

  return true;
}



void KBanking::infoTransactionsChanged(Account *a){
  _flagStaff->accountsUpdated();
  appInfoTransactionsChanged(a);
}



void KBanking::rebuildExportMenu(){
  _flagStaff->rebuildExportMenu();
}



int KBanking::init(){
  GWEN_DB_NODE *db;
  int rv;
  GWEN_TYPE_UINT32 lastVersion;

  rv=App::init();
  if (rv)
    return rv;

  lastVersion=getLastVersion();
  if ((lastVersion!=APP_LAST_VERSION_NONE) &&
      (lastVersion<
       ((0<<24)+
        (9<<16)+
        (18<<8)+
        5))) {
    DBG_NOTICE(0, "Updating from old version, updating transactions");
    CheckDuplicates w(this);
    w.show();
    w.adjustDates();
  }

  if ((lastVersion!=APP_LAST_VERSION_NONE) &&
      (lastVersion<
       ((0<<24)+
        (9<<16)+
        (28<<8)+
        1))) {
    DBG_NOTICE(0, "Updating from old version, updating transactions");
    CheckDuplicates w(this);
    w.show();
    w.gatherInfoFromPurpose();
  }

  db=getAppData();
  assert(db);
  db=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_DEFAULT,
                      "kbanking");
  assert(db);
  _lastReportId=GWEN_DB_GetIntValue(db, "lastReportId", 0, 0);
  _useColoursForCategoryText=GWEN_DB_GetIntValue(db,
                                                 "useColoursForCategoryText",
                                                 0, 1)!=0;

  _sampleReportModules();

  return 0;
}



int KBanking::fini(){
  GWEN_DB_NODE *db;

  db=getAppData();
  assert(db);
  db=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_DEFAULT,
                      "kbanking");
  assert(db);
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "lastReportId",
                      _lastReportId);

  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "useColoursForCategoryText",
                      _useColoursForCategoryText?1:0);

  return App::fini();
}



bool KBanking::useColoursForCategoryText() const {
  return _useColoursForCategoryText;
}



void KBanking::setUseColoursForCategoryText(bool b) {
  _useColoursForCategoryText=b;
}



void KBanking::setMainWindow(MainWindow *w){
  _mainWindow=w;
}



MainWindow *KBanking::getMainWindow(){
  return _mainWindow;
}



Report *KBanking::findReport(const char *name){
  std::list<Report*>::iterator it;

  for (it=_reports.begin(); it!=_reports.end(); it++) {
    std::string s;

    s=QStringToUtf8String((*it)->name());
    if (strcasecmp(s.c_str(), name)==0)
      return *it;
  } // for

  return 0;
}



std::list<Report*> KBanking::getReports(){
  return _reports;
}



GWEN_DB_NODE *KBanking::getReportProfiles(Report *r) {
  GWEN_DB_NODE *db;
  GWEN_BUFFER *nbuf;
  std::string name;

  name=QStringToUtf8String(r->name());
  db=getAppData();
  assert(db);
  db=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_DEFAULT,
                      "kbanking/reports");
  assert(db);
  nbuf=GWEN_Buffer_new(0, 256, 0, 1);
  if (GWEN_Text_EscapeToBuffer(name.c_str(), nbuf)) {
    DBG_ERROR(0, "Could not escape report name");
    GWEN_Buffer_free(nbuf);
    return 0;
  }
  GWEN_Buffer_AppendString(nbuf, "/profiles");
  db=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_DEFAULT,
                      GWEN_Buffer_GetStart(nbuf));
  assert(db);
  GWEN_Buffer_free(nbuf);

  return db;
}



GWEN_DB_NODE *KBanking::findReportProfile(Report *r, const char *id){
  GWEN_DB_NODE *db;

  db=getReportProfiles(r);
  if (!db)
    return 0;
  db=GWEN_DB_FindFirstGroup(db, "profile");
  while(db) {
    const char *s;

    s=GWEN_DB_GetCharValue(db, "id", 0, 0);
    if (!s) {
      DBG_WARN(0, "Report with no id detected");
    }
    else {
      if (strcasecmp(s, id)==0)
        return db;
    }
    db=GWEN_DB_FindNextGroup(db, "profile");
  } // while

  return 0;
}



int KBanking::addReportProfile(Report *r, GWEN_DB_NODE *dbProfile){
  GWEN_DB_NODE *db;
  char numbuf[32];

  db=getReportProfiles(r);
  if (!db)
    return -1;

  snprintf(numbuf, sizeof(numbuf), "%d", ++_lastReportId);
  GWEN_DB_SetCharValue(dbProfile, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "id", numbuf);
  if (strcasecmp(GWEN_DB_GroupName(dbProfile), "profile")!=0)
    GWEN_DB_GroupRename(dbProfile, "profile");
  GWEN_DB_AddGroup(db, dbProfile);

  return 0;
}



void KBanking::removeReportProfile(Report *r,
                                   GWEN_DB_NODE *dbProfile){
  GWEN_DB_UnlinkGroup(dbProfile);
  GWEN_DB_Group_free(dbProfile);
}



void KBanking::_sampleReportModules(){
  Report *r;

  r=new SimpleReport(this);
  _reports.push_back(r);
  r=new SummaryReport(this);
  _reports.push_back(r);
  r=new CategoryReport(this);
  _reports.push_back(r);
  r=new GraphReport(this);
  _reports.push_back(r);
}



void KBanking::addTransactionWindow(TransactionWindow *w) {
  _tWindows.push_back(w);
}



void KBanking::delTransactionWindow(TransactionWindow *w) {
  std::list<RefPointer<TransactionWindow> >::iterator it;

  for (it=_tWindows.begin();
       it!=_tWindows.end();
       it++) {
    if ((*it).ptr()==w) {
      // found
      _tWindows.erase(it);
      return;
    }
  }
}



void KBanking::delTransactionWindows() {
  std::list<RefPointer<TransactionWindow> >::iterator it;

  while(!_tWindows.empty()) {
    RefPointer<TransactionWindow> wp=_tWindows.front();
    wp.ref().close();
  }
}



int KBanking::_newTransferJob(Transfer::TransferType tt,
                                  RefPointer<Transfer> tmpl,
                                  AB_JOB **retjob,
                                  QWidget *parent){
  RefPointer<Transfer> t;
  bool b;
  int rv;
  AB_JOB *job=0;

  if (tmpl.isValid()) {
    const AB_VALUE *val;

    t=new Transfer(tmpl.ref());
    if (t.ref().getRemoteName().empty() ||
	t.ref().getRemoteBankCode().empty() ||
	t.ref().getRemoteAccountNumber().empty()) {
      std::string pname;

      /* we need to fill some fields from the payee */
      pname=t.ref().getPayee();
      if (!pname.empty()) {
	Payee *payee;

        payee=findPayeeById(pname.c_str());
	if (payee) {
	  std::string s;

	  s=payee->name();
          if (!s.empty())
	    t.ref().addRemoteName(payee->name());
	  if (t.ref().getRemoteBankCode().empty())
	    t.ref().setRemoteBankCode(payee->bankCode());
	  if (t.ref().getRemoteAccountNumber().empty())
	    t.ref().setRemoteAccountNumber(payee->accountNumber());
	}
      }
    }
    val=t.ref().getValue();
    if (val) {
      if (AB_Value_IsNegative(val)) {
	AB_VALUE *nv;

	nv=AB_Value_dup(val);
	AB_Value_Negate(nv);
	t.ref().setValue(nv);
        AB_Value_free(nv);
      }
    }
  }
  else {
    t=new Transfer(tt);
    if (tt==Transfer::TransferType_DebitNote)
      t.ref().setTextKey(5);
    else
      t.ref().setTextKey(51);
  }
  t.ref().setTransactionId(getNextUniqueId());

  while(1) {
    GWEN_DB_NODE *dbT;
    int bad=false;
    AB_BANKINFO_CHECKRESULT res;
    int i;
    AB_ACCOUNT *ba;
    Account *acc;

    switch(tt) {
    case Transfer::TransferType_Simple:
      b=EditTransaction::editTransfer(this, t,
                                      QWidget::tr("New Transfer"),
                                      parent, true);
      break;
    case Transfer::TransferType_DebitNote:
      b=EditTransaction::editDebitNote(this, t,
                                       QWidget::tr("New Debit Note"),
                                       parent, true);
      break;
    case Transfer::TransferType_EuTransfer:
      b=EditEuTransaction::editEuTransfer(this, t,
                                          QWidget::tr("New Euro Transfer"),
                                          parent, true);
      break;
    default:
      b=false;
    }

    if (!b) {
      DBG_INFO(0, "User aborted");
      AB_Job_free(job);
      return AB_ERROR_USER_ABORT;
    }

    // dump transaction
    DBG_NOTICE(0, "Accepted");
    dbT=GWEN_DB_Group_new("Transaction");
    if (!t.ref().toDb(dbT)) {
      DBG_ERROR(0, "Could not write DB");
    }
    else {
#ifndef WIN32
	  // In MSVC this function crashes
      GWEN_DB_Dump(dbT, stderr, 2);
#endif
    }
    GWEN_DB_Group_free(dbT);


    // get account for this new transaction
    acc=findAccount(t.ref().getLocalBankCode().c_str(),
                          t.ref().getLocalAccountNumber().c_str());
    assert(acc);
    ba=getAccount(acc->getBankingId());
    if (!ba) {
      DBG_ERROR(0, "Account not available");
      QMessageBox::critical(parent,
                            QWidget::tr("Account Not Available"),
                            QWidget::tr("The account you requested is not "
                               "available\n"
                               "with any backend."),
                            QWidget::tr("Dismiss"), QString::null);
      return AB_ERROR_GENERIC;
    }

    // create job
    switch(tt) {
    case Transfer::TransferType_Simple:
      job=AB_JobSingleTransfer_new(ba);
      break;
    case Transfer::TransferType_DebitNote:
      job=AB_JobSingleDebitNote_new(ba);
      break;
    case Transfer::TransferType_EuTransfer:
      job=AB_JobEuTransfer_new(ba);
      break;
    default:
      return AB_ERROR_INVALID;
    }

    rv=AB_Job_CheckAvailability(job);
    if (rv) {
      DBG_NOTICE(0, "Job is not available (%d)", rv);
      AB_Job_free(job);
      QMessageBox::critical(parent,
                            QWidget::tr("Job Not Available"),
                            QWidget::tr("The job you requested is not "
                               "available with\n"
                               "the backend which handles this account.\n"),
                            QWidget::tr("Dismiss"), QString::null);
      AB_Job_free(job);
      return AB_ERROR_NOT_AVAILABLE;
    }

    res=AB_Banking_CheckAccount(getCInterface(),
                                t.ref().getRemoteCountry().c_str(),
                                0,
                                t.ref().getRemoteBankCode().c_str(),
                                t.ref().getRemoteAccountNumber().c_str());
    switch(res) {
    case AB_BankInfoCheckResult_NotOk:
      DBG_INFO(0, "Account check result: %d", res);
      i=QMessageBox::warning(parent,
                             QWidget::tr("Warning"),
                             QWidget::tr("<qt>"
                                "<p>"
                                "The given bank code and account number "
                                "combination is invalid."
                                "</p>"
                                "Do you want to review the transfer?"
                                "<qt>"
                               ),
                             QMessageBox::Yes,
                             QMessageBox::No,
                             QMessageBox::Abort);
      if (i==0 || i == QMessageBox::Yes)
        bad=true;
      else if (i != 1 && i != QMessageBox::No) {
        AB_Job_free(job);
        return AB_ERROR_USER_ABORT;
      }
      break;

    case AB_BankInfoCheckResult_UnknownBank:
      DBG_INFO(0, "Account check result: %d", res);
      i=QMessageBox::warning(parent,
                             QWidget::tr("Warning"),
                             QWidget::tr("<qt>"
                                "<p>"
                                "The given bank code is invalid."
                                "</p>"
                                "Do you want to review the transfer?"
                                "<qt>"
                               ),
                             QMessageBox::Yes,
                             QMessageBox::No,
                             QMessageBox::Abort);
      if (i == 0 || i == QMessageBox::Yes)
        bad=true;
      else if (i != 1 && i != QMessageBox::No) {
        AB_Job_free(job);
        return AB_ERROR_USER_ABORT;
      }
      break;

    case AB_BankInfoCheckResult_UnknownResult:
      DBG_INFO(0, "Nothing known about account");
      break;
    case AB_BankInfoCheckResult_Ok:
      DBG_INFO(0, "Bank code/account number combination is ok");
      break;
    default:
      break;
    }

    if (!bad) {
      AB_TRANSACTION *abt;

      abt=t.ref().toBankingTransaction();
      assert(abt);

      switch(tt) {
      case Transfer::TransferType_Simple:
        rv=AB_JobSingleTransfer_SetTransaction(job, abt);
        break;
      case Transfer::TransferType_DebitNote:
        rv=AB_JobSingleDebitNote_SetTransaction(job, abt);
        break;
      case Transfer::TransferType_EuTransfer:
        rv=AB_JobEuTransfer_SetTransaction(job, abt);
        break;
      default:
        AB_Transaction_free(abt);
        AB_Job_free(job);
        return AB_ERROR_INVALID;
      }

      if (rv) {
        DBG_NOTICE(0, "Could not set transfer (%d)", rv);
        int res = QMessageBox::critical(parent,
                                  QWidget::tr("Bad Transfer Data"),
                                  QWidget::tr("<qt>"
                                     "<p>"
                                     "There seems to be some errors "
                                     "in the transfer data."
                                     "</p>"
                                     "<p>"
                                     "Do you want to review the transfer?"
                                     "</p>"
                                     "<qt>"
                                    ),
                                  QMessageBox::Yes,QMessageBox::Abort);
	if (res !=0 && res != QMessageBox::Yes){
          AB_Transaction_free(abt);
          AB_Job_free(job);
          return AB_ERROR_INVALID;
        }
        bad=true;
      }
      AB_Transaction_free(abt);
    } // if ok

    if (!bad) {
      break;
    }
    AB_Job_free(job);
    job=0;
  } // while

  *retjob=job;
  return 0;
}



int KBanking::_newTransfer(RefPointer<Transfer> tmpl,
                               QWidget *parent) {
  AB_JOB *job;
  int rv;

  rv=_newTransferJob(Transfer::TransferType_Simple, tmpl, &job, parent);
  if (rv) {
    DBG_INFO(0, "here (%d)", rv);
    return rv;
  }
  else {
    int rv;
    const AB_TRANSACTION *bat;

    assert(job);
    DBG_NOTICE(0, "Enqueuing job");
    rv=enqueueJob(job);
    if (rv) {
      DBG_NOTICE(0, "Error %d", rv);
      AB_Job_free(job);
      return rv;
    }
    bat=AB_JobSingleTransfer_GetTransaction(job);
    if (bat) {
      RefPointer<Transfer> t;
      GWEN_TIME *ti;

      t=new Transfer(bat, Transfer::TransferType_Simple);
      ti=GWEN_CurrentTime();
      if (ti) {
        t.ref().setDate(ti);
        GWEN_Time_free(ti);
      }
      addTransfer(t);
    }
    AB_Job_free(job);

    QMessageBox::information(parent,
                             QWidget::tr("Job enqueued"),
                             QWidget::tr("<qt>"
                                "A transfer job has been enqueued.\n"
                                "Please go to the <i>outbox</i> and press\n"
                                "the <i>execute</i> button there to "
                                "actually send\n"
                                "the enqueued jobs."
                                "</qt>"),
                             QWidget::tr("Dismiss"), QString::null);
  }

  return 0;
}



int KBanking::_newDebitNote(RefPointer<Transfer> tmpl,
                                QWidget *parent) {
  AB_JOB *job;
  int rv;

  rv=_newTransferJob(Transfer::TransferType_DebitNote, tmpl,
                     &job,
                     parent);
  if (rv) {
    DBG_INFO(0, "here (%d)", rv);
    return rv;
  }
  else {
    int rv;
    const AB_TRANSACTION *bat;

    DBG_NOTICE(0, "Enqueuing job");
    rv=enqueueJob(job);
    if (rv) {
      DBG_NOTICE(0, "Error %d", rv);
      AB_Job_free(job);
      return rv;
    }
    bat=AB_JobSingleDebitNote_GetTransaction(job);
    if (bat) {
      RefPointer<Transfer> t;
      GWEN_TIME *ti;

      t=new Transfer(bat, Transfer::TransferType_DebitNote);
      ti=GWEN_CurrentTime();
      if (ti) {
        t.ref().setDate(ti);
        GWEN_Time_free(ti);
      }
      addTransfer(t);
    }
    AB_Job_free(job);

    QMessageBox::information(parent,
                             QWidget::tr("Job enqueued"),
                             QWidget::tr("<qt>"
                                "A debit note job has been enqueued.\n"
                                "Please go to the <i>outbox</i> and press\n"
                                "the <i>execute</i> button there to "
                                "actually send\n"
                                "the enqueued jobs."
                                "</qt>"),
                             QWidget::tr("Dismiss"), QString::null);
  }

  return 0;
}



int KBanking::_newEuTransfer(RefPointer<Transfer> tmpl,
                                 QWidget *parent) {
  AB_JOB *job;
  int rv;

  rv=_newTransferJob(Transfer::TransferType_EuTransfer, tmpl,
                     &job,
                     parent);
  if (rv) {
    DBG_INFO(0, "here (%d)", rv);
    return rv;
  }
  else {
    int rv;
    const AB_TRANSACTION *bat;

    DBG_NOTICE(0, "Enqueuing job");
    rv=enqueueJob(job);
    if (rv) {
      DBG_NOTICE(0, "Error %d", rv);
      AB_Job_free(job);
      return rv;
    }
    bat=AB_JobEuTransfer_GetTransaction(job);
    if (bat) {
      RefPointer<Transfer> t;
      GWEN_TIME *ti;

      t=new Transfer(bat, Transfer::TransferType_EuTransfer);
      ti=GWEN_CurrentTime();
      if (ti) {
        t.ref().setDate(ti);
        GWEN_Time_free(ti);
      }
      addTransfer(t);
    }
    AB_Job_free(job);

    QMessageBox::information(parent,
                             QWidget::tr("Job enqueued"),
                             QWidget::tr("<qt>"
                                "An EU-transfer job has been enqueued.\n"
                                "Please go to the <i>outbox</i> and press\n"
                                "the <i>execute</i> button there to "
                                "actually send\n"
                                "the enqueued jobs."
                                "</qt>"),
                             QWidget::tr("Dismiss"), QString::null);
  }

  return 0;
}



int KBanking::transferFromTemplate(RefPointer<Transaction> tmpl,
                                   QWidget *parent) {
  int rv;

  if (tmpl.isValid()) {
    RefPointer<Transfer> t;

    t=new Transfer(tmpl.ref(), Transfer::TransferType_Simple);
    rv=_newTransfer(t, parent);
  }
  else
    rv=_newTransfer(0, parent);
  if (rv) {
    DBG_INFO(0, "here (%d)", rv);
    return rv;
  }

  return 0;
}



int KBanking::euTransferFromTemplate(RefPointer<Transaction> tmpl,
                                      QWidget *parent) {
  int rv;

  if (tmpl.isValid()) {
    RefPointer<Transfer> t;

    t=new Transfer(tmpl.ref(), Transfer::TransferType_EuTransfer);
    rv=_newEuTransfer(t, parent);
  }
  else
    rv=_newEuTransfer(0, parent);
  if (rv) {
    DBG_INFO(0, "here (%d)", rv);
    return rv;
  }

  return 0;
}



int KBanking::debitNoteFromTemplate(RefPointer<Transaction> tmpl,
                                    QWidget *parent) {
  int rv;

  if (tmpl.isValid()) {
    RefPointer<Transfer> t;

    t=new Transfer(tmpl.ref(), Transfer::TransferType_DebitNote);
    rv=_newDebitNote(t, parent);
  }
  else
    rv=_newDebitNote(0, parent);
  if (rv) {
    DBG_INFO(0, "here (%d)", rv);
    return rv;
  }

  return 0;
}








