/***************************************************************************
  hydrometertool.h
  -------------------
  A hydrometer correction utility for QBrew
  -------------------
  Copyright 2004-2007, David Johnson <david@usermode.org>
  Based on code Copyright 2001, Abe Kabakoff <abe_kabakoff@gmx.de>
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <QLabel>

#include "resource.h"
#include "hydrometertool.h"

//////////////////////////////////////////////////////////////////////////////
// HydrometerTool()
// ----------------
// Constructor

HydrometerTool:: HydrometerTool(QWidget* parent)
    : QDialog(parent)
{
    ui.setupUi(this);

    // TODO: use appropriate units (F or C)
    QString SUFFIX(Resource::DEGREE);
    SUFFIX += " F";

    setWindowTitle(Resource::TITLE + " - Hydrometer Tool");

    // additional setup
    QFont fnt(font());
    fnt.setBold(true);
    ui.corrected->setFont(fnt);

    ui.calibrated->setSuffix(SUFFIX);
    ui.sample->setSuffix(SUFFIX);

    // connections
    connect(ui.sample, SIGNAL(valueChanged(double)), this, SLOT(recalc()));
    connect(ui.calibrated, SIGNAL(valueChanged(double)), this, SLOT(recalc()));
    connect(ui.reading, SIGNAL(valueChanged(double)), this, SLOT(recalc()));

    recalc();
}

//////////////////////////////////////////////////////////////////////////////
// recalc()
// --------
// the signal to calculate the actual SG

void HydrometerTool::recalc()
{
    // see http://www.hbd.org/brewery/library/HydromCorr0992.html

    const double COEFF1 = 1.313454;
    const double COEFF2 = 0.132674;
    const double COEFF3 = 2.057793E-3;
    const double COEFF4 = 2.627634E-6;

    double calterm;   // E is the correction factor for the calibration temp
    double corr;      // the corrected reading
    double temp = ui.sample->value();
    double cal = ui.calibrated->value();
    double read = ui.reading->value();

    // Correction(@59F) = 1.313454 - 0.132674*T + 2.057793e-3*T**2 - 2.627634e-6*T**3
    // the equation is set up for 59F (15C)
    calterm = (COEFF1
               - (COEFF2 * cal)
               + (COEFF3 * (cal*cal))
               - (COEFF4 * (cal*cal*cal)));
    corr = (COEFF1
            - (COEFF2 * temp)
            + (COEFF3 * (temp*temp))
            - (COEFF4 * (temp*temp*temp))) - calterm;

    corr /= 1000.0;
    corr = 1.0 - corr;
    corr = read / corr;

    // now that we have the value change the reading
    ui.corrected->setText(QString::number(corr, 'f', 3));
}
