/***************************************************************************
  stylemodel.cpp
  -------------------
  Style model
  -------------------
  Copyright 2006-2007, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <QApplication>
#include <QMessageBox>
#include <QPair>

#include "data.h"
#include "stylemodel.h"

//////////////////////////////////////////////////////////////////////////////
// StyleModel()
// ------------
// Constructor

StyleModel::StyleModel(QObject *parent, StyleList *list)
    : QAbstractTableModel(parent), list_(list)
{}

StyleModel::~StyleModel(){}

//////////////////////////////////////////////////////////////////////////////
// flush()
// -------
// Reset the model

void StyleModel::flush()
{
    reset();
}

//////////////////////////////////////////////////////////////////////////////
// data()
// ------
// Return data at index

QVariant StyleModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid()) return QVariant();
    if (index.row() >= list_->count()) return QVariant();

    // row is the entry in the QList
    const Style &style = list_->at(index.row());

    // column is the style "field"
    if (role == Qt::DisplayRole) {
        switch (index.column()) {
          case Style::NAME:
              return style.name();
          case Style::OGLOW:
              return QString::number(style.oglow_, 'f', 3);
          case Style::OGHI:
              return QString::number(style.oghi_, 'f', 3);
          case Style::FGLOW:
              return QString::number(style.fglow_, 'f', 3);
          case Style::FGHI:
              return QString::number(style.fghi_, 'f', 3);
          case Style::IBULOW:
              return QString::number(style.ibulow_);
          case Style::IBUHI:
              return QString::number(style.ibuhi_);
          case Style::SRMLOW:
              return QString::number(style.srmlow_);
          case Style::SRMHI:
              return QString::number(style.srmhi_);
          default:
              return QVariant();
        }
    } else if (role == Qt::EditRole) {
        switch (index.column()) {
          case Style::NAME:
              return style.name();
          case Style::OGLOW:
              return style.oglow_;
          case Style::OGHI:
              return style.oghi_;
          case Style::FGLOW:
              return style.fglow_;
          case Style::FGHI:
              return style.fghi_;
          case Style::IBULOW:
              return style.ibulow_;
          case Style::IBUHI:
              return style.ibuhi_;
          case Style::SRMLOW:
              return style.srmlow_;
          case Style::SRMHI:
              return style.srmhi_;
          default:
              return QVariant();
        }
    } else if (role == Qt::TextAlignmentRole) {
        switch (index.column()) {
          case Style::NAME:
              return Qt::AlignLeft;
          case Style::OGLOW:
          case Style::OGHI:
          case Style::FGLOW:
          case Style::FGHI:
          case Style::IBULOW:
          case Style::IBUHI:
          case Style::SRMLOW:
          case Style::SRMHI:
          default:
              return Qt::AlignRight;
        }
    } else {
        return QVariant();
    }
}

//////////////////////////////////////////////////////////////////////////////
// setData()
// ---------
// Set data at index

bool StyleModel::setData(const QModelIndex &index,
                       const QVariant &value, int role)
{
    static bool deleting = false;

    Style style;
    QString name;
    int row = index.row();
    int column = index.column();

    if (!index.isValid()) return false;
    if (role != Qt::EditRole) return false;
    if ((row >= list_->count())
        && (column != Style::NAME)) return false;

    // grab existing style
    if (row < list_->count()) style = list_->value(row);

    switch (column) {
      case Style::NAME:
          // editing name as several special cases
          name = value.toString();

          // deleting name deletes style
          if (name.isEmpty()) {
              if (row >= list_->count()) return false; // empty

              // TODO: for some reason this gets entered recursively...
              if (deleting) return false;
              deleting = true;

              // remove style
              beginRemoveRows(index.parent(), row, row);

              list_->removeAt(row);
              emit modified();
              endRemoveRows();

              deleting = false;
              return true;
          }

          // changed name
          style.setName(name);
          break;

      case Style::OGLOW:
          style.oglow_ = value.toDouble();
          break;

      case Style::OGHI:
          style.oghi_ = value.toDouble();
          break;

      case Style::FGLOW:
          style.fglow_ = value.toDouble();
          break;

      case Style::FGHI:
          style.fghi_ = value.toDouble();
          break;

      case Style::IBULOW:
          style.ibulow_ = value.toUInt();
          break;

      case Style::IBUHI:
          style.ibuhi_ = value.toUInt();
          break;

      case Style::SRMLOW:
          style.srmlow_ = value.toUInt();
          break;

      case Style::SRMHI:
          style.srmhi_ = value.toUInt();
          break;

      default:
          return false;
    }

    list_->replace(row, style);
    emit modified();

    // whole row may have changed
    emit dataChanged(index.sibling(row, Style::NAME),
                     index.sibling(row, Style::SRMHI));

    return true;
}

//////////////////////////////////////////////////////////////////////////////
// insertRows()
// ------------
// Insert rows into table

bool StyleModel::insertRows(int row, int count, const QModelIndex&)
{
    if (count != 1) return false; // only insert one row at a time
    if ((row < 0) || (row >= list_->count())) row = list_->count();

    Style style = Data::instance()->style("Generic");

    beginInsertRows(QModelIndex(), row, row);
    list_->insert(row, style);
    emit modified();
    endInsertRows();
    return true;
}

//////////////////////////////////////////////////////////////////////////////
// removeRows()
// ------------
// Remove rows from table

bool StyleModel::removeRows(int row, int count, const QModelIndex&)
{
    if (count != 1) return false; // only remove one row at a time
    if ((row < 0) || (row >= list_->count())) return false;

    if (QMessageBox::question(QApplication::activeWindow(),
                              Resource::TITLE + " - Delete?",
                              "Do you wish to remove this entry?",
                              "Remove", "Cancel")) {
        return false;
    }

    beginRemoveRows(QModelIndex(), row, row);
    list_->removeAt(row);
    emit modified();
    endRemoveRows();
    return true;
}

//////////////////////////////////////////////////////////////////////////////
// headerData()
// ------------
// Return header information

QVariant StyleModel::headerData(int section, Qt::Orientation orientation,
                                int role) const
{
    if (orientation == Qt::Horizontal && role == Qt::DisplayRole) {
        switch (section) {
          case Style::NAME:
              return "Style";
          case Style::OGLOW:
              return "Min. OG";
          case Style::OGHI:
              return "Max. OG";
          case Style::FGLOW:
              return "Min. FG";
          case Style::FGHI:
              return "Max. FG";
          case Style::IBULOW:
              return "Min. IBU";
          case Style::IBUHI:
              return "Max. IBU";
          case Style::SRMLOW:
              return "Min. SRM";
          case Style::SRMHI:
              return "Max. SRM";
           default:
              return QVariant();
        }
    }

    return QVariant();
}

//////////////////////////////////////////////////////////////////////////////
// flags()
// ------
// Return flags at index

Qt::ItemFlags StyleModel::flags(const QModelIndex &index) const
{
    if (!index.isValid()) return Qt::ItemIsEnabled;

    return QAbstractTableModel::flags(index) | Qt::ItemIsEditable;
}

//////////////////////////////////////////////////////////////////////////////
// rowCount()
// ----------
// Return number of rows of data

int StyleModel::rowCount(const QModelIndex &) const
{
    return list_->count();
}

//////////////////////////////////////////////////////////////////////////////
// columnCount()
// -------------
// Return number of columns of data

int StyleModel::columnCount(const QModelIndex &) const
{
    return Style::COUNT;
}

//////////////////////////////////////////////////////////////////////////////
// sort()
// ------
// Sort by column

void StyleModel::sort(int column, Qt::SortOrder order)
{
    QList<QPair<QString,Style> > sortlist;

    foreach(Style style, *list_) {
        QString field;
        switch (column) {
          case Style::NAME:
              field = style.name();
              break;
          case Style::OGLOW:
              field = QString::number(style.oglow_).rightJustified(6,'0');
              break;
          case Style::OGHI:
              field = QString::number(style.oghi_).rightJustified(6,'0');
              break;
          case Style::FGLOW:
              field = QString::number(style.fglow_).rightJustified(6,'0');
              break;
          case Style::FGHI:
              field = QString::number(style.fghi_).rightJustified(6,'0');
              break;
          case Style::IBULOW:
              field = QString::number(style.ibulow_).rightJustified(6,'0');
              break;
          case Style::IBUHI:
              field = QString::number(style.ibuhi_).rightJustified(6,'0');
              break;
          case Style::SRMLOW:
              field = QString::number(style.srmlow_).rightJustified(6,'0');
              break;
          case Style::SRMHI:
              field = QString::number(style.srmhi_).rightJustified(6,'0');
              break;
          default:
              field = QString();
              break;
        }
        sortlist.append(QPair<QString,Style>(field, style));
    }

    // sort list
    qSort(sortlist.begin(), sortlist.end());

    // create new list
    list_->clear();
    QPair<QString,Style> pair;
    foreach(pair, sortlist) {
        if (order == Qt::AscendingOrder)
            list_->append(pair.second);
        else
            list_->prepend(pair.second);
    }

    emit layoutChanged();
}
