/***************************************************************************
  helpwindow.cpp
  -------------------
  General purpose help file viewer
  -------------------
  Copyright (c) 1999-2002 David Johnson
  Please see the header file for copyright and license information.
***************************************************************************/

#include <qaction.h>
#include <qapplication.h>
#include <qdir.h>
#include <qiconset.h>
#include <qmenubar.h>
#include <qpaintdevicemetrics.h>
#include <qpopupmenu.h>
#include <qpainter.h>
#include <qprinter.h>
#include <qtextbrowser.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qsimplerichtext.h>
#include <qstatusbar.h>
#include <qstylesheet.h>

#include "resource.h"
#include "helpwindow.h"

using namespace AppResource;

//////////////////////////////////////////////////////////////////////////////
// Construction                                                             //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// HelpWindow()
// ------------
// Constructor

HelpWindow::HelpWindow(const QString& home, const QString& data,
                       QWidget* parent, const char *name)
    : QMainWindow(parent, name, WDestructiveClose),
      browser_(0), backward_(0), forward_(0), home_(0), print_(0), close_(0)
{
    setUsesBigPixmaps(ID_PREF_LARGE_ICONS_DEFAULT);
#if QT_VERSION > 300
    QIconSet::setIconSize(QIconSet::Large, QSize(22, 22));
#endif
    browser_ = new QTextBrowser(this);
    browser_->mimeSourceFactory()->setFilePath(QFileInfo(home).dirPath(true));
    browser_->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    if (!home.isEmpty()) browser_->setSource(home);
    textChanged();
    setCentralWidget(browser_);

    // set a reasonable width
    int width = 480;
    if ((width * 3) > (QApplication::desktop()->width() * 2))
        width = QApplication::desktop()->width() * 2 / 3;
    else if ((width * 2) < (QApplication::desktop()->width()))
        width = QApplication::desktop()->width() / 2;
    // set a reasonable height
    int height = QApplication::desktop()->height() * 3 / 4;
    resize(width, height);

    // setup actions
    QString smallicon = data + "icons/16x16/actions/";
    QString largeicon = data + "icons/22x22/actions/";

    print_ = new QAction("Print",
                         QIconSet(QPixmap(smallicon + "fileprint.png"),
                                  QPixmap(largeicon + "fileprint.png")),
                         "&Print...", CTRL+Key_P, this);
    close_ = new QAction("Close",
                         QIconSet(QPixmap(smallicon + "exit.png"),
                                  QPixmap(largeicon + "exit.png")),
                         "&Close", CTRL+Key_Q, this);
    backward_ = new QAction("Back",
                            QIconSet(QPixmap(smallicon + "back.png"),
                                     QPixmap(largeicon + "back.png")),
                            "&Backward", ALT+Key_Left, this);
    forward_ = new QAction("Forward",
                           QIconSet(QPixmap(smallicon + "forward.png"),
                                    QPixmap(largeicon + "forward.png")),
                           "&Forward", ALT+Key_Right, this);
    home_ = new QAction("Home",
                        QIconSet(QPixmap(smallicon + "gohome.png"),
                                 QPixmap(largeicon + "gohome.png")),
                        "&Home", CTRL+Key_Home, this);

    print_->setStatusTip("Print the document");
    close_->setStatusTip("Close the help viewer");
    backward_->setStatusTip("Go to the previous page");
    forward_->setStatusTip("Go to the next page");
    home_->setStatusTip("Go to the index page");

    connect(print_, SIGNAL(activated()), this, SLOT(print()));
    connect(close_, SIGNAL(activated()), this, SLOT(close()));
    connect(backward_, SIGNAL(activated()), browser_, SLOT(backward()));
    connect(forward_, SIGNAL(activated()), browser_, SLOT(forward()));
    connect(home_, SIGNAL(activated()), browser_, SLOT(home()));

    QPopupMenu* filemenu = new QPopupMenu(this);
    print_->addTo(filemenu);
    filemenu->insertSeparator();
    close_->addTo(filemenu);

    QPopupMenu* navmenu = new QPopupMenu(this);
    backward_->addTo(navmenu);
    forward_->addTo(navmenu);
    home_->addTo(navmenu);

    menuBar()->insertItem("&File", filemenu);
    menuBar()->insertItem("&Navigate", navmenu);

    QToolBar* toolbar = new QToolBar(this);
    backward_->addTo(toolbar);
    forward_->addTo(toolbar);
    home_->addTo(toolbar);
    toolbar->addSeparator();
    print_->addTo(toolbar);
    close_->addTo(toolbar);

    backward_->setEnabled(false);
    forward_->setEnabled(false);

    // setup connections
    connect(browser_, SIGNAL(backwardAvailable(bool)),
            backward_, SLOT(setEnabled(bool)));
    connect(browser_, SIGNAL(forwardAvailable(bool)),
            forward_, SLOT(setEnabled(bool)));
    connect(browser_, SIGNAL(textChanged()),
            this, SLOT(textChanged()));
    connect(browser_, SIGNAL(highlighted(const QString&)), statusBar(),
            SLOT(message(const QString&)));

    browser_->setFocus();
}

//////////////////////////////////////////////////////////////////////////////
// ~HelpWindow()
// -------------
// Destructor

HelpWindow::~HelpWindow() { ; }

//////////////////////////////////////////////////////////////////////////////
// textChanged()
// -------------
// The displayed document has changed, so change caption

void HelpWindow::textChanged()
{
    if (browser_->documentTitle().isNull())
        setCaption(ID_TITLE +  " Help - " + browser_->context());
    else
        setCaption(ID_TITLE + " Help - " + browser_->documentTitle());
}

//////////////////////////////////////////////////////////////////////////////
// print()
// -------
// Print the contents of the help window

void HelpWindow::print()
{
    QPrinter printer;
    printer.setFullPage(true);
    if (printer.setup()) {
        QPainter p(&printer);
        QPaintDeviceMetrics metrics(p.device());
        int dpix = metrics.logicalDpiX();
        int dpiy = metrics.logicalDpiY();
        const int margin = 72; // pt
        QRect body(margin * dpix / 72, margin * dpiy / 72,
            metrics.width() - margin * dpix / 72 * 2,
            metrics.height() - margin * dpiy / 72 * 2);
        QFont font("times", 10);
        QSimpleRichText richText(browser_->text(), font, browser_->context(),
                                 browser_->styleSheet(),
                                 browser_->mimeSourceFactory(), body.height());
        richText.setWidth(&p, body.width());
        QRect view(body);
        int page = 1;
        while (true) {
            richText.draw(&p, body.left(), body.top(), view, colorGroup());
            view.moveBy(0, body.height());
            p.translate(0, -body.height());
            p.setFont(font);
            p.drawText(view.right() -
                       p.fontMetrics().width(QString::number(page)),
                       view.bottom() + p.fontMetrics().ascent() + 5,
                       QString::number(page));
            if (view.top() >= richText.height()) break;
            printer.newPage();
            ++page;
        }
    }
}
