/***************************************************************************
  grain.h
  -------------------
  A grain class
  -------------------
  Copyright 1999-2004, David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#ifndef GRAIN_H
#define GRAIN_H

#include <qlistview.h>
#include <qstring.h>
#include <qvaluelist.h>

#include "quantity.h"

enum { GRAIN_OTHER, GRAIN_MASHED, GRAIN_STEEPED, GRAIN_EXTRACT };

class Grain {
public:
    // default constructor
    Grain();
    // constructor
    Grain(const QString &name, const Weight &quantity, const double &extract,
          const double &color, const QString &use);
    // copy constructor
    Grain(const Grain &g);
    // assignment operator
    Grain operator=(const Grain &g);
    // comparison operator
    bool operator==(const Grain &g);
    // destructor
    ~Grain();

    // get/set name
    QString name() const;
    void setName(const QString &name);
    // get/set weight
    Weight &weight();
    void setWeight(const Weight &weight);
    // get/set extract
    double extract();
    void setExtract(double extract);
    // get/set color
    double color();
    void setColor(double color);
    // get/set usage
    int use();
    void setUse(int use);

    // return the yield (quantity times extract)
    double yield();
    // return the HCU (quantity times color)
    double HCU();
        
    // return use as string
    QString useString();

    // return a list of use strings
    static QStringList useStringList();
    // convert use string to enum
    static int useStringToEnum(const QString &name);
    
private:
    QString name_;
    Weight weight_;
    double extract_;
    double color_;
    int use_;
    double yield_;
    double hcu_;    // color yield for grain
};

// TODO map will be more efficient
typedef QValueList<Grain> GrainList;
typedef QValueList<Grain>::Iterator GrainIterator;

// TODO: remove this from model
class GrainListItem : public QListViewItem {
public:
    // constructor
    GrainListItem(QListView * parent, QString, QString = QString::null,
                  QString = QString::null, QString = QString::null,
                  QString = QString::null, QString = QString::null,
                  QString = QString::null, QString = QString::null );
    // destructor
    ~GrainListItem();
    // get iterator to Grain item
    GrainIterator grainIterator();
    // set iterator to Grain item
    void setGrainIterator(GrainIterator &it);
    // set grain to match iterator
    void setToIterator();

    // custom sorting (to get numerical sort order)
    int compare(QListViewItem* item, int col, bool asc) const;

private:
    GrainIterator git_;
};

//////////////////////////////////////////////////////////////////////////////
// Inlined Methods

inline void Grain::setName(const QString &name) { name_ = name; }

inline QString Grain::name() const { return name_; }

inline void Grain::setWeight(const Weight &weight) { weight_ = weight; }

inline Weight& Grain::weight() { return weight_; }

inline void Grain::setExtract(double extract) { extract_ = extract; }

inline double Grain::extract() { return extract_; }

inline void Grain::setColor(double color) { color_ = color; }

inline double Grain::color() { return color_; }

inline void Grain::setUse(int use) { use_ = use; }

inline int Grain::use() { return use_; }

inline double Grain::yield() { return yield_; }

inline double Grain::HCU() { return hcu_; }

inline GrainIterator GrainListItem::grainIterator() { return git_; }

inline void GrainListItem::setGrainIterator(GrainIterator &it) { git_ = it; }

#endif // GRAIN_H
