/***************************************************************************
  controller.cpp
  -------------------
  Control class for QBrew
  -------------------
  Copyright (c) 2005 David Johnson
  Please see the header file for copyright and license information.
***************************************************************************/

#include <qapplication.h>
#include <qdir.h>
#include <qfiledialog.h>
#include <qlabel.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qpaintdevicemetrics.h>
#include <qpainter.h>
#include <qprinter.h>
#include <qregexp.h>
#include <qsettings.h>
#include <qsimplerichtext.h>
#include <qsplashscreen.h>
#include <qstatusbar.h>
#include <qstyle.h>
#include <qstylefactory.h>
#include <qtimer.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qwhatsthis.h>

#include "alcoholtool.h"
#include "configure.h"
#include "helpwindow.h"
#include "hydrometertool.h"
#include "databasetool.h"
#include "images.h"
#include "model.h"
#include "view.h"

#include "controller.h"

using namespace AppResource;

Controller *Controller::instance_ = 0;

const QImage& Controller::findImage(const QString& name)
{
    return qembed_findImage(name);
}

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction, Initialization                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// Controller()
// ------------
// Private constructor

Controller::Controller()
    : QMainWindow(0, "QBrew", WType_TopLevel | WDestructiveClose),
      model_(0), view_(0), configure_(0), helpwin_(0), primerwin_(0),
      alcoholtool_(0), hydrometertool_(0), filename_(), newflag_(false),
      filerecentmenu_(0), filemenu_(0), optionsmenu_(0), toolsmenu_(0),
      helpmenu_(0), toolbar_(0), autosave_(0), printer_(0), backed_(false)
{ ; }

void Controller::initialize(const QString &filename)
{
    // NOTE: the following is a temporary conversion
    convertPreferences();
    readConfig();

    // show splashscreen
    if (state_.general.showsplash
        && QFile::exists(dataBase() + "splash.png")) {
        QSplashScreen *splash  = new QSplashScreen(dataBase() + "splash.png");
        QTimer::singleShot(3000, splash, SLOT(close()));
        splash->show();
        QApplication::flushX();
    }

    // restore window state
    restoreState();

    // setup look and feel
    setIcon(qembed_findImage("qbrew"));
    QIconSet::setIconSize(QIconSet::Large, QSize(22, 22));
    setUsesBigPixmaps(state_.general.largeicons);

    if ((state_.general.lookfeel != QApplication::style().name()) &&
        (QStyleFactory::keys().contains(state_.general.lookfeel))) {
        QApplication::setStyle(QStyleFactory::create(state_.general.lookfeel));
    }

    // initialize frame elements
    initActions();
    initMenuBar();
    initToolBar();
    initStatusBar();

    // create and initialize model
    model_ = Model::instance();
    if (state_.calc.units == UNIT_METRIC) {
        model_->setDefaultSize(Volume(state_.recipe.batch, Volume::liter));
        model_->setDefaultGrainUnit(Weight::kilogram);
        model_->setDefaultHopUnit(Weight::gram);
    } else if (state_.calc.units == UNIT_US) {
        model_->setDefaultSize(Volume(state_.recipe.batch, Volume::gallon));
        model_->setDefaultGrainUnit(Weight::pound);
        model_->setDefaultHopUnit(Weight::ounce);
    }
    model_->setDefaultStyle(state_.recipe.style);
    model_->setDefaultMash(state_.recipe.mash);
    model_->setDefaultHopform(state_.recipe.hopform);
    model_->setDefaultMiscUnit(Quantity::generic);
    connect(model_, SIGNAL(recipeModified()), this, SLOT(documentModified()));

    // create and initialize view
    view_ = new View(this, model_);
    connect(model_, SIGNAL(recipeChanged()), view_, SLOT(flush()));
    connect(model_, SIGNAL(recipeModified()), view_, SLOT(refresh()));

    // initialize calculations
    Calc::setEfficiency(state_.calc.efficiency);
    Calc::setTinseth(state_.calc.tinseth);
    Calc::setMorey(state_.calc.morey);

    // load or create a recipe
    if (state_.general.loadlast &&
        (filename.isEmpty() || (filename == ID_DEFAULT_FILE))) {
        filename_ = *state_.general.recentfiles.begin();
    } else {
        filename_ = filename;
    }
    if ((!filename_.isEmpty()) && (filename_ != ID_DEFAULT_FILE)) {
        openFile(filename_);
    } else {
        model_->newRecipe();
        newflag_ = true;
        setCaption(ID_TITLE + " " + VERSION);
    }

    // adjust size, otherwise it grabs too much space
    adjustSize();

    // other initialization
    initAutoSave();
    setCentralWidget(view_);
}

//////////////////////////////////////////////////////////////////////////////
// ~Controller()
// -------------
// Private destructor

Controller::~Controller()
{
    saveState();

    if (autosave_) { autosave_->stop(); delete autosave_; }

    if (view_) delete view_;
    if (filerecentmenu_) delete filerecentmenu_;
    if (filemenu_) delete filemenu_;
    if (toolsmenu_) delete toolsmenu_;
    if (optionsmenu_) delete optionsmenu_;
    if (helpmenu_) delete helpmenu_;
    if (toolbar_) delete toolbar_;
    if (printer_) delete printer_;
}

//////////////////////////////////////////////////////////////////////////////
// instance()
// ----------
// Return pointer to the controller.

Controller *Controller::instance()
{
    if (!instance_)
        instance_ = new Controller();
    return instance_;
}

//////////////////////////////////////////////////////////////////////////////
// initActions()
// -------------
// Initialize the actions

void Controller::initActions()
{
    QAction *action;
    actions_.setAutoDelete(true);

    // file actions
    action = new QAction("New",
                         QIconSet(qembed_findImage("sm_filenew"),
                                  qembed_findImage("filenew")),
                         "&New", CTRL+Key_N, this);
    action->setStatusTip("Create a new recipe");
    action->setWhatsThis("Use this command to create a new recipe.");
    connect(action, SIGNAL(activated()), this, SLOT(fileNew()));
    actions_.insert("filenew", action);

    action = new QAction("Open",
                         QIconSet(qembed_findImage("sm_fileopen"),
                                  qembed_findImage("fileopen")),
                         "&Open...", CTRL+Key_O, this);
    action->setStatusTip("Open an existing recipe");
    action->setWhatsThis("Use this command to open a recipe.");
    connect(action, SIGNAL(activated()), this,  SLOT(fileOpen()));
    actions_.insert("fileopen", action);

    action = new QAction("Save",
                         QIconSet(qembed_findImage("sm_filesave"),
                                  qembed_findImage("filesave")),
                         "&Save",  CTRL+Key_S, this);
    action->setStatusTip("Save the recipe");
    action->setWhatsThis("Use this command to save the current recipe.");
    action->setEnabled(false);
    connect(action, SIGNAL(activated()), this, SLOT(fileSave()));
    actions_.insert("filesave", action);

    action = new QAction("Save As",
                         QIconSet(qembed_findImage("sm_filesaveas"),
                                  qembed_findImage("filesaveas")),
                         "Save &as...", 0, this);
    action->setStatusTip("Save the recipe under a new name");
    connect(action, SIGNAL(activated()), this, SLOT(fileSaveAs()));
    actions_.insert("filesaveas", action);

    action = new QAction("Print",
                         QIconSet(qembed_findImage("sm_fileprint"),
                                  qembed_findImage("fileprint")),
                         "&Print...", CTRL+Key_P, this);
    action->setStatusTip("Print the recipe");
    action->setWhatsThis("Use this command to print the recipe.");
    connect(action, SIGNAL(activated()), this, SLOT(filePrint()));
    actions_.insert("fileprint", action);

    action = new QAction("Export", "E&xport...", 0, this);
    action->setStatusTip("Export the recipe to a non-native format");
    connect(action, SIGNAL(activated()), this, SLOT(fileExport()));
    actions_.insert("fileexport", action);

    action = new QAction("Quit",
                         QIconSet(qembed_findImage("sm_exit"),
                                  qembed_findImage("exit")),
                         "&Quit", CTRL+Key_Q, this);
    action->setStatusTip("Quit the application");
    connect(action, SIGNAL(activated()), qApp, SLOT(closeAllWindows()));
    actions_.insert("filequit", action);

    // tools actions
    action = new QAction("Alcohol Percentage",
                         "&Alcohol Percentage...", 0, this);
    action->setStatusTip("Alcohol percentage calculator");
    connect(action, SIGNAL(activated()), this, SLOT(toolsAlcohol()));
    actions_.insert("toolsalcohol", action);

    action = new QAction("Hydrometer Correction",
                         "&Hydrometer Correction...", 0, this);
    action->setStatusTip("Hydrometer correction utility");
    connect(action, SIGNAL(activated()), this, SLOT(toolsHydrometer()));
    actions_.insert("toolshydrometer", action);

    action = new QAction("Database Editor",
                         "&Database Editor...", 0, this);
    action->setStatusTip("Database editor");
    connect(action, SIGNAL(activated()), this, SLOT(toolsDatabase()));
    actions_.insert("toolsdatabase", action);

    // options actions
    action = new QAction("Toolbar", "&Toolbar", 0, this, 0, true);
    action->setStatusTip("Enable or disable the Toolbar");
    connect(action, SIGNAL(activated()), this, SLOT(optionsToolbar()));
    actions_.insert("optionstoolbar", action);

    action = new QAction("Statusbar", "&Statusbar", 0, this, 0, true);
    action->setStatusTip("Enable or disable the Statusbar");
    connect(action, SIGNAL(activated()), this, SLOT(optionsStatusbar()));
    actions_.insert("optionsstatusbar", action);

    action = new QAction("Configure", "&Configure...", 0, this);
    action->setStatusTip("Display the configuration dialog");
    connect(action, SIGNAL(activated()), this, SLOT(optionsConfigure()));
    actions_.insert("optionsconfigure", action);

    // help actions
    action = new QAction("Help",
                         QIconSet(qembed_findImage("sm_contents"),
                                  qembed_findImage("contents")),
                         "&Contents", Key_F1, this);
    action->setStatusTip("Display the application handbook");
    connect(action, SIGNAL(activated()), this, SLOT(helpContents()));
    actions_.insert("helpcontents", action);

    action = new QAction("Primer",
                         QIconSet(qembed_findImage("sm_contents"),
                                  qembed_findImage("contents")),
                         "&Primer", 0, this);
    action->setStatusTip("Display the brewing primer");
    connect(action, SIGNAL(activated()), this, SLOT(helpPrimer()));
    actions_.insert("helpprimer", action);

    action = new QAction("About", "&About...", 0, this);
    action->setStatusTip("Application information");
    connect(action, SIGNAL(activated()), this, SLOT(helpAbout()));
    actions_.insert("helpabout", action);

    action = new QAction("Context",
                         QIconSet(qembed_findImage("sm_contexthelp"),
                                  qembed_findImage("contexthelp")),
                         "&What's This?", SHIFT+Key_F1, this);
    action->setStatusTip("Context sensitive help");
    connect(action, SIGNAL(activated()), this, SLOT(whatsThis()));
    actions_.insert("helpcontext", action);
}

//////////////////////////////////////////////////////////////////////////////
// initMenuBar()
// -------------
// Initialize the menu bar

void Controller::initMenuBar()
{
    QAction *action;

    // file menu
    filemenu_ = new QPopupMenu();

    action = actions_.find("filenew");
    if (action) action->addTo(filemenu_);
    action = actions_.find("fileopen");
    if (action) action->addTo(filemenu_);

    // recent file submenu
    filerecentmenu_ = new QPopupMenu();
    connect(filerecentmenu_, SIGNAL(aboutToShow()),
            this, SLOT(recentMenuShow()));
    connect(filerecentmenu_, SIGNAL(activated(int)),
            this, SLOT(fileRecent(int)));
    filemenu_->insertItem(QIconSet(qembed_findImage("sm_fileopen"),
                                   qembed_findImage("fileopen")),
                          "Open &Recent", filerecentmenu_);

    filemenu_->insertSeparator();

    action = actions_.find("filesave");
    if (action) action->addTo(filemenu_);
    action = actions_.find("filesaveas");
    if (action) action->addTo(filemenu_);

    filemenu_->insertSeparator();

    action = actions_.find("fileexport");
    if (action) action->addTo(filemenu_);
    action = actions_.find("fileprint");
    if (action) action->addTo(filemenu_);

    filemenu_->insertSeparator();

    action = actions_.find("filequit");
    if (action) action->addTo(filemenu_);

    // tools menu
    toolsmenu_ = new QPopupMenu();
    action = actions_.find("toolsalcohol");
    if (action) action->addTo(toolsmenu_);
    action = actions_.find("toolshydrometer");
    if (action) action->addTo(toolsmenu_);
    action = actions_.find("toolsdatabase");
    if (action) action->addTo(toolsmenu_);

    // options menu
    optionsmenu_ = new QPopupMenu();
    optionsmenu_->setCheckable(true);
    action = actions_.find("optionstoolbar");
    if (action) action->addTo(optionsmenu_);
    action = actions_.find("optionsstatusbar");
    if (action) action->addTo(optionsmenu_);

    optionsmenu_->insertSeparator();

    action = actions_.find("optionsconfigure");
    if (action) action->addTo(optionsmenu_);

    // help menu
    helpmenu_ = new QPopupMenu();
    action = actions_.find("helpcontents");
    if (action) action->addTo(helpmenu_);
    action = actions_.find("helpprimer");
    if (action) action->addTo(helpmenu_);
    action = actions_.find("helpcontext");
    if (action) action->addTo(helpmenu_);

    helpmenu_->insertSeparator();

    action = actions_.find("helpabout");
    if (action) action->addTo(helpmenu_);

    // insert submenus into main menu
    menuBar()->insertItem("&File", filemenu_);
    menuBar()->insertItem("&Tools", toolsmenu_);
    menuBar()->insertItem("&Options", optionsmenu_);

    menuBar()->insertSeparator();

    menuBar()->insertItem("&Help", helpmenu_);
}

//////////////////////////////////////////////////////////////////////////////
// initToolBar()
// -------------
// Initialize the toolbar

void Controller::initToolBar()
{
    QAction *action;

    setRightJustification(false);
    toolbar_ = new QToolBar("Main Toolbar", this);

    action = actions_.find("filenew");
    if (action) action->addTo(toolbar_);
    action = actions_.find("fileopen");
    if (action) action->addTo(toolbar_);
    action = actions_.find("filesave");
    if (action) action->addTo(toolbar_);
    action = actions_.find("fileprint");
    if (action) action->addTo(toolbar_);

    toolbar_->addSeparator();

    action = actions_.find("helpcontext");
    if (action) action->addTo(toolbar_);

    // now show or hide toolbar depending on initial setting
    toolbar_->setShown(state_.window.toolbar);
    if (actions_.find("optionstoolbar"))
        actions_.find("optionstoolbar")->setOn(state_.window.toolbar);
}

//////////////////////////////////////////////////////////////////////////////
// initStatusBar()
// ---------------
// Initialize the status bar

void Controller::initStatusBar()
{
    statusBar()->message(ID_READY, 2000);
    // now show or hide statusbar depending on initial setting
    statusBar()->setShown(state_.window.statusbar);
    if (actions_.find("optionsstatusbar"))
        actions_.find("optionsstatusbar")->setOn(state_.window.statusbar);
}

//////////////////////////////////////////////////////////////////////////////
// initAutoSave()
// ---------------
// Initialize the autosave timer

void Controller::initAutoSave()
{
    // destroy existing timer, if any
    if (autosave_) {
        autosave_->stop();
        disconnect(autosave_, SIGNAL(timeout()), this, SLOT(autoSave()));
        delete autosave_; autosave_ = 0;
    }

    if (state_.general.autosave) {
        autosave_ = new QTimer(this, CONF_GEN_AUTOSAVE);
        connect(autosave_, SIGNAL(timeout()), this, SLOT(autoSave()));
        autosave_->start(state_.general.saveinterval * 60000, false);
    }
}

//////////////////////////////////////////////////////////////////////////////
// File Menu Implementation                                                 //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// fileNew()
// ---------
// Create a new recipe

void Controller::fileNew()
{
    if (model_->modified()) {
        // file needs to be saved, what do we do
        switch (querySave()) {
          case QMessageBox::Yes: // yes, save the file
              fileSave();
              break;
          case QMessageBox::No: // no, don't save the file
              break;
          case QMessageBox::Cancel: // cancel creating new file
              statusBar()->message("Canceled...", 2000);
              // exit function
              return;
        }
    }

    // create a new file
    statusBar()->message("Creating new recipe...");
    model_->newRecipe();
    newflag_ = true;
    backed_ = false;

    if (actions_.find("filesave")) actions_.find("filesave")->setEnabled(false);

    filename_ = ID_DEFAULT_FILE;
    setCaption(ID_TITLE + " - " + filename_);
    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// fileOpen()
// ----------
// Open a recipe

void Controller::fileOpen()
{
    if (model_->modified()) {
        // file needs to be saved, what do we do
        switch (querySave()) {
            case QMessageBox::Yes:
                fileSave();
                break;
            case QMessageBox::No: // no, don't save the file
                break;
            case QMessageBox::Cancel: //
                statusBar()->message("Canceled...", 2000);
                return;
        }
    }

    // open the file
    statusBar()->message("Opening recipe...");
    QString selected = ID_FILE_FILTER;
    QString fname = QFileDialog::getOpenFileName(0, ID_OPEN_FILTER, this,
                                                 "fileopendialog",
                                                 "Open...",
                                                 &selected);
    if (openFile(fname)) {
        statusBar()->message("Loaded recipe", 2000);
    } else {
        statusBar()->message("Load failed", 2000);
    }
}

//////////////////////////////////////////////////////////////////////////////
// fileRecent()
// -----------
// Selection has been made from recent file menu

void Controller::fileRecent(int item)
{
    if (item < 0) return;

    if (model_->modified()) {
        // file needs to be saved, what do we do
        switch (querySave()) {
          case QMessageBox::Yes:
              fileSave();
              break;
          case QMessageBox::No:
              break;
          case QMessageBox::Cancel:
              statusBar()->message("Canceled...", 2000);
              return;
        }
    }

    // open the file
    QString fname = filerecentmenu_->text(item);
    if (openFile(fname)) {
        statusBar()->message("Loaded recipe", 2000);
        addRecent(fname);
    }
}

//////////////////////////////////////////////////////////////////////////////
// fileSave()
// ----------
// Save a recipe

void Controller::fileSave()
{
    if (newflag_) {
        fileSaveAs();
    } else {
        // file exists so save it
        statusBar()->message("Saving recipe...");
        // TODO: backupFile() returns a bool, so use it...
        if (state_.general.autobackup) backupFile();
        if (model_->saveRecipe(filename_)) {
            // successful in saving file
            newflag_ = false;
            if (actions_.find("filesave"))
                actions_.find("filesave")->setEnabled(false);
            statusBar()->message(ID_READY, 2000);
        } else {
            // error in saving file
            QMessageBox::warning(this, ID_TITLE, ID_TITLE +
                                 " was unable to save the recipe " +
                                 filename_);
            statusBar()->message("Error in saving recipe", 2000);
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// fileSaveAs()
// ------------
// Save a recipe under a new name

void Controller::fileSaveAs()
{
    statusBar()->message("Saving recipe under new filename...");

    QString selected = ID_FILE_FILTER;
    QString fname = QFileDialog::getSaveFileName(filename_,
                                                 ID_SAVE_FILTER, this,
                                                 "savedialog", "Save As...",
                                                 &selected);

    if (!fname.isEmpty()) { // we got a filename
        QFileInfo finfo(fname);
        if (state_.general.autobackup) backupFile();
        if (finfo.exists()) {
            // overwrite?
            switch (queryOverwrite(fname)) {
              case QMessageBox::Yes:
                  break;
              case QMessageBox::Cancel:
                  statusBar()->message("Canceled...", 2000);
                  return;
            }
        }

        if (model_->saveRecipe(fname)) {
            // successfully saved
            newflag_ = false;
            if (actions_.find("filesave"))
                actions_.find("filesave")->setEnabled(false);
            setCaption(ID_TITLE + " - " + finfo.fileName());
            statusBar()->message(ID_READY, 2000);
            // save name of file
            filename_ = fname;
        } else {
            // error in saving
            QMessageBox::warning(this, ID_TITLE,
                                 QString("Unable to save to %1")
                                 .arg(finfo.fileName()));
            statusBar()->message("Error in saving recipe", 2000);
        }
    } else {
        // no file chosen
        statusBar()->message("Saving aborted", 2000);
    }
}

//////////////////////////////////////////////////////////////////////////////
// fileExport()
// ------------
// Export recipe to non-native format

void Controller::fileExport()
{
    statusBar()->message("Exporting recipe...");

    bool status;
    QString selected = QString::null;

    // issue warning to user about data loss
    // TODO: use a "don't show this again" checkbox?
    if (QMessageBox::warning(this, ID_TITLE + " - Export",
                             "<p>Exporting a recipe will result in the loss "
                             " of some data, particularly in regards to "
                             "weights and volumes. Continue?",
                             QMessageBox::Yes,
                             QMessageBox::Cancel) == QMessageBox::Cancel) {
        statusBar()->message("Export aborted", 2000 );
        return;
    }

    // TODO: need custom save dialog that will switch extension when switching filters
    QString fname
        = QFileDialog::getSaveFileName(filename_.left(filename_.findRev('.')),
                                       ID_EXPORT_FILTER, this,
                                       "exportdialog", "Export...",
                                       &selected);

    if (!fname.isEmpty()) { // we got a filename
        QFileInfo finfo(fname);
        if (finfo.exists()) {
            // overwrite?
            switch (queryOverwrite(fname)) {
              case QMessageBox::Yes:
                  break;
              case QMessageBox::Cancel:
                  statusBar()->message("Canceled...", 2000);
                  return;
            }
        }

        // export recipe
        if (selected == ID_HTML_FILTER) {
            status = model_->exportHTML(fname);
        } else if (selected == ID_BEERXML_FILTER) {
            status = model_->exportBeerXML(fname);
        } else if (selected == ID_TEXT_FILTER) {
            status = model_->exportText(fname);
        }
        if (status) {
            // successfully exported
            statusBar()->message(ID_READY, 2000);
        } else {
            // error in exporting
            QMessageBox::warning(this, ID_TITLE,
                                 QString("Unable to export to %1")
                                 .arg(finfo.fileName()));
            statusBar()->message("Error exporting recipe", 2000);
        }
    } else {
        // no file chosen
        statusBar()->message("Export aborted", 2000 );
    }
}

//////////////////////////////////////////////////////////////////////////////
// filePrint()
// -----------
// Print the recipe. Much of this method derived from Qt example program
// "helpviewer" copyright 1992-2002 Troll Tech AS.

void Controller::filePrint()
{
    statusBar()->message("Printing...");

    if (!printer_) printer_ = new QPrinter(QPrinter::HighResolution);

    printer_->setFullPage(true);
    if (printer_->setup(this)) {
        QPainter painter(printer_);
	if (!painter.isActive()) return;
        QPaintDeviceMetrics metrics(painter.device());

        int dpix = metrics.logicalDpiX();
        int dpiy = metrics.logicalDpiY();
        const int marginx = (int) (dpix * 2 / 2.54); // two cm margins
        const int marginy = (int) (dpiy * 2 / 2.54);

        QFont font("sans", 10);
	QRect body(marginx, marginy,
                   metrics.width() - (2 * marginx),
                   metrics.height() - (2 * marginx));
	QSimpleRichText richtext(model_->recipeHTML(),
                                 font,
                                 QString::null,
                                 0,
				 QMimeSourceFactory::defaultFactory(),
                                 body.height());
	richtext.setWidth(&painter, body.width() );

	int page = 1;
	while (true) {
	    richtext.draw(&painter, marginx, marginy, body, colorGroup());
	    body.moveBy(0, body.height());
	    painter.translate(0, -body.height());
	    painter.drawText(body.right()
                             - painter.fontMetrics().width(QString::number(page)),
                             body.bottom() + painter.fontMetrics().ascent() + 5,
                             QString::number(page));
	    if (body.top()-marginy >= richtext.height())
		break;
	    printer_->newPage();
	    page++;
	}
        statusBar()->message(ID_READY, 2000);
    } else {
        // cancelled
        statusBar()->message("Printing cancelled", 2000);
    }
}

//////////////////////////////////////////////////////////////////////////////
// Tools Menu Implementation                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// toolsAlcohol()
// --------------
// A utility dialog for hydrometer correction

void Controller::toolsAlcohol()
{
    if (!alcoholtool_) alcoholtool_
        = new AlcoholTool(this, "alcoholtool");
    alcoholtool_->show();
    alcoholtool_->raise();
    if (alcoholtool_->isMinimized()) alcoholtool_->showNormal();

    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// toolsHydrometer()
// -----------------
// A utility dialog for hydrometer correction

void Controller::toolsHydrometer()
{
    if (!hydrometertool_) hydrometertool_
        = new HydrometerTool(this, "hydrometertool");
    hydrometertool_->show();
    hydrometertool_->raise();
    if (hydrometertool_->isMinimized()) hydrometertool_->showNormal();

    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// toolsDatabase()
// ---------------
// A database editor for ingredients

void Controller::toolsDatabase()
{
    if (!databasetool_)
        databasetool_ = new DatabaseTool(0, "databasetool", model_);
    databasetool_->show();
    databasetool_->raise();
    if (databasetool_->isMinimized()) databasetool_->showNormal();

    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// Options Menu Implementation                                              //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// optionsToolbar()
// ----------------
// Toggle toolbar status

void Controller::optionsToolbar()
{
    bool show = !toolbar_->isVisible();
    if (actions_.find("optionstoolbar"))
        actions_.find("optionstoolbar")->setOn(show);
    toolbar_->setShown(show);
    state_.window.toolbar = show;
    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// optionsSatusbar()
// -----------------
// Toggle statusbar status

void Controller::optionsStatusbar()
{
    bool show = !statusBar()->isVisible();
    if (actions_.find("optionsstatusbar"))
        actions_.find("optionsstatusbar")->setOn(show);
    statusBar()->setShown(show);
    state_.window.statusbar = show;
    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// optionsConfigure()
// ------------------
// Display the setup dialog

void Controller::optionsConfigure()
{
    statusBar()->message("Configuring " + ID_TITLE + "...", 2000);

    if (!configure_) {
        configure_ = new Configure(this, "configure");

        connect(configure_, SIGNAL(generalApply(const GenConfigState&)),
                this, SLOT(applyGeneralState(const GenConfigState&)));
        connect(configure_, SIGNAL(recipeApply(const RecipeConfigState&)),
                this, SLOT(applyRecipeState(const RecipeConfigState&)));
        connect(configure_, SIGNAL(calcApply(const CalcConfigState&)),
                this, SLOT(applyCalcState(const CalcConfigState&)));
        connect(configure_, SIGNAL(configureAccept()),
                this, SLOT(saveConfig()));
    }

    if (!configure_->isVisible()) {
        configure_->setState(state_);
    }
    configure_->show(); // non-modal
    configure_->raise();
    if (configure_->isMinimized()) configure_->showNormal();

}

//////////////////////////////////////////////////////////////////////////////
// Help Menu Implementation                                                 //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// helpContents()
// --------------
// Display the application manual

void Controller::helpContents()
{
    QString home = QDir(docBase() + ID_HELP_FILE).absPath();

    if (!helpwin_) helpwin_ = new HelpWindow(home, 0, "help");
    helpwin_->show();
    helpwin_->raise();
    if (helpwin_->isMinimized()) helpwin_->showNormal();

    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// helpPrimer()
// ------------
// Display the brewing primer

void Controller::helpPrimer()
{
    QString home = QDir(docBase() + ID_PRIMER_FILE).absPath();

    if (!primerwin_) primerwin_ = new HelpWindow(home, 0, "primer");
    primerwin_->show();
    primerwin_->raise();
    if (primerwin_->isMinimized()) primerwin_->showNormal();

    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// helpAbout()
// -----------
// Display the About dialog

void Controller::helpAbout()
{
    QString message;

    message = "<center><nobr><big><strong>" + ID_TITLE
        + " Version " + VERSION + "</strong></big></nobr></center>";
    message += "<p align=center><strong>" + ID_DESCRIPTION + "</strong>";
    message += "<p align=center>" + ID_COPYRIGHT + ' ' + ID_AUTHOR;
    message += "<p align=center>Contributions by " + ID_CONTRIBUTORS;

    QMessageBox::about(this, "About " + ID_TITLE, message);
}

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// openFile()
// ----------
// Open the named file

bool Controller::openFile(const QString &filename)
{
    if (!filename.isEmpty()) {
        QFileInfo finfo(filename);
        if (model_->nativeFormat(filename)) {
            if (model_->loadRecipe(filename)) {
                filename_ = filename;
                setCaption(ID_TITLE + " - " + finfo.fileName());
            } else {
                // load was unsuccessful
                QMessageBox::warning(this, ID_TITLE, ID_TITLE +
                                     " was unable to load the recipe " +
                                     finfo.fileName());
                return false;
            }
        } else {
            // attempt to import file
            if (model_->importRecipe(filename)) {
                // set filename to the default (forces a saveas)
                filename_ = ID_DEFAULT_FILE;
                setCaption(ID_TITLE + " - " + ID_DEFAULT_FILE);
            } else {
                // import was unsuccessful
                QMessageBox::warning(this, ID_TITLE, ID_TITLE +
                                     " was unable to import the file " +
                                     finfo.fileName());
                return false;
            }
        }

        newflag_ = false;
        backed_ = false;
        addRecent(filename);
        if (actions_.find("filesave"))
            actions_.find("filesave")->setEnabled(false);

        return true;
    }
    // empty filename
    return false;
}

//////////////////////////////////////////////////////////////////////////////
// backupFile()
// ------------
// Backup the current file

bool Controller::backupFile()
{
    if (backed_) return true;

    bool status = false;
    if ((!filename_.isEmpty()) && (filename_ != ID_DEFAULT_FILE)) {
        QFile source(filename_);
        QFile dest(filename_ + '~');

        // copy the file
        if (source.open(IO_ReadOnly)) {
            if (dest.open(IO_WriteOnly)) {
                char *buffer = new char[2048];
                int result = 0;
                while ((result >= 0) && (!source.atEnd())) {
                    result = source.readBlock(buffer, 2048);
                    if (result >= 0) dest.writeBlock(buffer, result);
                }
                status = result >= 0;
            }
            dest.close();
            source.close();
        }

        if (status) {
            backed_ = true;
        } else {
            QMessageBox::information(this, ID_TITLE + " - Backup",
                                     "Unable to make a backup of the "
                                     "original file");
        }
    }
    return status;
}

//////////////////////////////////////////////////////////////////////////////
// recentMenuShow()
// ----------------
// Initialize the recent file menu

void Controller::recentMenuShow()
{
    filerecentmenu_->clear();
    int item = 0;
    QStringList::ConstIterator it;
    for (it = state_.general.recentfiles.begin();
         it!=state_.general.recentfiles.end(); ++it) {
	filerecentmenu_->insertItem(*it, item++);
    }
}

//////////////////////////////////////////////////////////////////////////////
// addRecent()
// -----------
// Add a file to the recent files list

void Controller::addRecent(const QString &filename)
{
    if (state_.general.recentnum == 0) return;

    QFileInfo finfo(filename);
    finfo.convertToAbs();
    QString filepath = finfo.filePath();
    if (state_.general.recentfiles.find(filepath)
        != state_.general.recentfiles.end())
	state_.general.recentfiles.remove(filepath);
    if (state_.general.recentfiles.count() >= state_.general.recentnum)
	state_.general.recentfiles.pop_back();
    state_.general.recentfiles.prepend(filepath);
}

//////////////////////////////////////////////////////////////////////////////
// applyGeneralState()
// -------------------
// Set the general configuration state

void Controller::applyGeneralState(const GenConfigState &state)
{
    GenConfigState oldstate = state_.general;
    state_.general = state;

    // apply state changes
    if (state.lookfeel != oldstate.lookfeel) {
        QApplication::setStyle(QStyleFactory::create(state.lookfeel));
    }
    if (state.largeicons != oldstate.largeicons) {
        setUsesBigPixmaps(state.largeicons);
    }
    if ((state.autosave != oldstate.autosave) ||
        (state.saveinterval != oldstate.saveinterval)) {
        initAutoSave();
    }
    if (state.recentnum < oldstate.recentnum) {
        while (state_.general.recentfiles.count() > state.recentnum) {
            state_.general.recentfiles.pop_back();
        }
    }
    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// applyRecipeState()
// ------------------
// Set the recipe configuration state

void Controller::applyRecipeState(const RecipeConfigState &state)
{
    RecipeConfigState oldstate = state_.recipe;
    state_.recipe = state;

    // apply state changes
    if (state.batch != oldstate.batch) {
        if (state_.calc.units == UNIT_METRIC) {
            model_->setDefaultSize(Volume(state.batch, Volume::liter));
        } else if (state_.calc.units == UNIT_US) {
            model_->setDefaultSize(Volume(state.batch, Volume::gallon));
        }
    }
    if (state.style != oldstate.style) {
        model_->setDefaultStyle(state.style);
    }
    if (state.hopform != oldstate.hopform) {
        model_->setDefaultHopform(state.hopform);
    }
    if (state.mash != oldstate.mash) {
        model_->setDefaultMash(state.mash);
    }

    // recalculate
    model_->recalc();
    view_->flush();

    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// applyCalcState()
// ----------------
// Set the calc configuration state

void Controller::applyCalcState(const CalcConfigState &state)
{
    CalcConfigState oldstate = state_.calc;
    state_.calc = state;

    // apply state changes
    if (state.efficiency != oldstate.efficiency) {
        Calc::setEfficiency(state.efficiency);
    }
    if (state.morey != oldstate.morey) {
        Calc::setMorey(state.morey);
    }
    if (state.tinseth != oldstate.tinseth) {
        Calc::setTinseth(state.tinseth);
    }
    if (state.units != oldstate.units) {
        if (state.units == UNIT_METRIC) {
            model_->setDefaultSize(Volume(state_.recipe.batch,
                                          Volume::liter));
            model_->setDefaultGrainUnit(Weight::kilogram);
            model_->setDefaultHopUnit(Weight::gram);
        } else if (state.units == UNIT_US) {
            model_->setDefaultSize(Volume(state_.recipe.batch,
                                          Volume::gallon));
            model_->setDefaultGrainUnit(Weight::pound);
            model_->setDefaultHopUnit(Weight::ounce);
        }
    }

    // recalculate
    model_->recalc();
    view_->flush();

    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// saveConfig()
// -----------
// Save the configuration (when OK pressed in dialog)

void Controller::saveConfig()
{
    statusBar()->message("Saving configuration...");
    writeConfig();
    statusBar()->message(ID_READY, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// autoSave()
// ----------
// Time to autosave

void Controller::autoSave()
{
    if (!model_->modified()) return;
    if ((filename_.isEmpty()) || (filename_ == ID_DEFAULT_FILE)) {
        model_->saveRecipe(QDir::currentDirPath()
                           + QString("/autosave.%1").arg(ID_FILE_EXT));
    } else {
        model_->saveRecipe(filename_);
    }
}

//////////////////////////////////////////////////////////////////////////////
// documentModified()
// ----------------------
// Received when document is modified

void Controller::documentModified()
{
    if (actions_.find("filesave"))
        actions_.find("filesave")->setEnabled(true);
    QFileInfo finfo(filename_);
    setCaption(ID_TITLE + " - " + finfo.fileName() + " [modified]");
}

//////////////////////////////////////////////////////////////////////////////
// querySave()
// -----------
// Ask the user if they want to save their work before going on

int Controller::querySave()
{
    return QMessageBox::information(this, ID_TITLE + " - Save?",
                                    "<p>Do you wish to save your work first?",
                                    QMessageBox::Yes,
                                    QMessageBox::No,
                                    QMessageBox::Cancel);
}

//////////////////////////////////////////////////////////////////////////////
// queryOverwrite()
// -----------
// Ask the user if they want to overwrite an existing file

int Controller::queryOverwrite(const QString filename)
{
    return QMessageBox::information(this, ID_TITLE + " - Overwrite?",
                                    QString("<p>Are you sure you want to "
                                            "overwrite the file \"%1\"")
                                    .arg(filename),
                                    QMessageBox::Yes,
                                    QMessageBox::Cancel);
}

//////////////////////////////////////////////////////////////////////////////
// readConfig()
// ------------
// Read configuration from settings file

void Controller::readConfig()
{
    QSettings config;
    config.setPath("usermode.org", "QBrew");
    config.beginGroup('/' + ID_TITLE);

    // read general config
    config.beginGroup(CONFGROUP_GENERAL);
    state_.general.appdir  =
        config.readEntry(CONF_GEN_APPDIR, state_.general.appdir);
    state_.general.lookfeel =
        config.readEntry(CONF_GEN_LOOK_FEEL, state_.general.lookfeel);
    state_.general.largeicons =
        config.readBoolEntry(CONF_GEN_LARGE_ICONS, state_.general.largeicons);
    state_.general.showsplash =
        config.readBoolEntry(CONF_GEN_SHOW_SPLASH, state_.general.showsplash);
    state_.general.autosave =
        config.readBoolEntry(CONF_GEN_AUTOSAVE, state_.general.autosave);
    state_.general.saveinterval =
        config.readNumEntry(CONF_GEN_SAVEINTERVAL, state_.general.saveinterval);
    state_.general.autobackup =
        config.readBoolEntry(CONF_GEN_AUTOBACKUP, state_.general.autobackup);
    state_.general.loadlast =
        config.readBoolEntry(CONF_GEN_LOADLAST, state_.general.autobackup);
    state_.general.recentnum =
        config.readNumEntry(CONF_GEN_RECENTNUM, state_.general.recentnum);
    config.endGroup();

    // read recipe config
    config.beginGroup(CONFGROUP_RECIPE);
    state_.recipe.batch =
        config.readDoubleEntry(CONF_RECIPE_BATCH, state_.recipe.batch);
    state_.recipe.style =
        config.readEntry(CONF_RECIPE_STYLE, state_.recipe.style);
    state_.recipe.hopform =
        config.readEntry(CONF_RECIPE_HOPFORM, state_.recipe.hopform);
    state_.recipe.mash =
        config.readBoolEntry(CONF_RECIPE_MASH, state_.recipe.mash);
    config.endGroup();

    // read calc config
    config.beginGroup(CONFGROUP_CALC);
    state_.calc.efficiency =
        config.readDoubleEntry(CONF_CALC_EFFICIENCY, state_.calc.efficiency);
    state_.calc.morey =
        config.readBoolEntry(CONF_CALC_MOREY, state_.calc.morey);
    state_.calc.tinseth =
        config.readBoolEntry(CONF_CALC_TINSETH, state_.calc.tinseth);
    state_.calc.units =
        config.readEntry(CONF_CALC_UNITS, state_.calc.units);
    config.endGroup();

    config.endGroup();
}

//////////////////////////////////////////////////////////////////////////////
// writeConfig()
// -------------
// Write configuration to settings file

void Controller::writeConfig()
{
    QSettings config;
    config.setPath("usermode.org", "QBrew");
    config.beginGroup('/' + ID_TITLE);

    // write general config
    config.beginGroup(CONFGROUP_GENERAL);
    config.writeEntry(CONF_GEN_APPDIR, state_.general.appdir);
    config.writeEntry(CONF_GEN_LOOK_FEEL, state_.general.lookfeel);
    config.writeEntry(CONF_GEN_LARGE_ICONS, state_.general.largeicons);
    config.writeEntry(CONF_GEN_SHOW_SPLASH, state_.general.showsplash);
    config.writeEntry(CONF_GEN_AUTOSAVE, state_.general.autosave);
    config.writeEntry(CONF_GEN_SAVEINTERVAL, (int)state_.general.saveinterval);
    config.writeEntry(CONF_GEN_AUTOBACKUP, state_.general.autobackup);
    config.writeEntry(CONF_GEN_LOADLAST, state_.general.loadlast);
    config.writeEntry(CONF_GEN_RECENTNUM, (int)state_.general.recentnum);
    config.endGroup();

    // write recipe config
    config.beginGroup(CONFGROUP_RECIPE);
    config.writeEntry(CONF_RECIPE_BATCH, (double)state_.recipe.batch);
    config.writeEntry(CONF_RECIPE_STYLE, state_.recipe.style);
    config.writeEntry(CONF_RECIPE_HOPFORM, state_.recipe.hopform);
    config.writeEntry(CONF_RECIPE_MASH, state_.recipe.mash);
    config.endGroup();

    // write calc config
    config.beginGroup(CONFGROUP_CALC);
    config.writeEntry(CONF_CALC_EFFICIENCY, (double)state_.calc.efficiency);
    config.writeEntry(CONF_CALC_MOREY, state_.calc.morey);
    config.writeEntry(CONF_CALC_TINSETH, state_.calc.tinseth);
    config.writeEntry(CONF_CALC_UNITS, state_.calc.units);
    config.endGroup();

    config.endGroup();
}

//////////////////////////////////////////////////////////////////////////////
// restoreState()
// ------------
// Restore application state

void Controller::restoreState()
{
    QSettings config;
    config.setPath("usermode.org", "QBrew");
    config.beginGroup('/' + ID_TITLE);

    // read window state
    config.beginGroup(CONFGROUP_WINDOW);
    state_.window.toolbar = config.readBoolEntry(CONF_WIN_TOOLBAR,
                                                 state_.window.toolbar);
    state_.window.statusbar = config.readBoolEntry(CONF_WIN_STATUSBAR,
                                                   state_.window.statusbar);

    QString mainwindow = config.readEntry(CONF_WIN_MAINWINDOW);
    QTextStream stream(&mainwindow, IO_ReadOnly);
    stream >> *this; // dockwindow layout // TODO: doesn't work right...
    config.endGroup();

    // read general state
    config.beginGroup(CONFGROUP_GENERAL);
    state_.general.recentfiles = config.readListEntry(CONF_GEN_RECENTFILES, ',');
    config.endGroup();

    config.endGroup();
}

//////////////////////////////////////////////////////////////////////////////
// saveSate()
// ----------
// Save application state

void Controller::saveState()
{
    // load config from settings
    QSettings config;
    config.setPath("usermode.org", "QBrew");
    config.beginGroup('/' + ID_TITLE);

    // write window state
    config.beginGroup(CONFGROUP_WINDOW);
    config.writeEntry(CONF_WIN_TOOLBAR, state_.window.toolbar);
    config.writeEntry(CONF_WIN_STATUSBAR, state_.window.statusbar);

    QString mainwindow;
    QTextStream stream(&mainwindow, IO_WriteOnly);
    stream << *this; // dockwindow layout
    config.writeEntry(CONF_WIN_MAINWINDOW, mainwindow);

    config.endGroup();

    // write general state
    config.beginGroup(CONFGROUP_GENERAL);
    config.writeEntry(CONF_GEN_RECENTFILES, state_.general.recentfiles, ',');
    config.endGroup();

    config.endGroup();
}

//////////////////////////////////////////////////////////////////////////////
// state()
// -------
// Return the application state

const ConfigState& Controller::state()
{
    return state_;
}

//////////////////////////////////////////////////////////////////////////////
// closeEvent()
// -----------
// Catch the close event

void Controller::closeEvent(QCloseEvent* e)
{
    if (!model_->modified()) {
        e->accept();
    } else {
        // file needs to be saved, what do we do
        switch (querySave()) {
          case QMessageBox::Yes:
              fileSave();
              e->accept();
              break;
          case QMessageBox::Cancel:
              statusBar()->message(ID_READY, 2000);
              e->ignore();
              break;
          case QMessageBox::No:
          default:
              e->accept();
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// dataBase()
// ---------
// Figure out the base directory for the data

QString Controller::dataBase()
{
    QString base = state_.general.appdir;
    if (base.isEmpty()) {
        base = qApp->applicationDirPath();
    } else {
        base = QDir(base).absPath();
    }

#if defined(Q_WS_X11)
    if (QRegExp("qbrew/?$", false).match(base) != -1) {
        // we have our own application directory (it ends in 'qbrew')
        base += "/";
    } else if (QRegExp("bin/?$", false).match(base) != -1) {
        // we are in the bin dir of a filesystem hiearchy like '/usr/local/bin'
        base += "/../share/qbrew/";
    } else {
        // we are in a filesystem hierarchy like '/usr/local'
        base += "/share/qbrew/";
    }

#elif defined(Q_WS_MACX)
    if (QRegExp("Contents/MacOS/?$", false).match(base) != -1) {
        // we're pointing inside an application bundle
        base += "/../Resources/";
    } else {
        // otherwise punt
        base += "/";
    }

#else // Win32 and others
    base += "/";
#endif
    return base;
}

//////////////////////////////////////////////////////////////////////////////
// docBase()
// ---------
// Figure out the base directory for the documentation

QString Controller::docBase()
{
    QString base = state_.general.appdir;
    if (base.isEmpty()) {
        base = qApp->applicationDirPath();
    } else {
        base = QDir(base).absPath();
    }

#if defined(Q_WS_X11)
    if (QRegExp("qbrew/?$", false).match(base) != -1) {
        // we have our own application directory (it ends in 'qbrew')
        base += "/doc/en/";
    } else if (QRegExp("bin/?$", false).match(base) != -1) {
        // we are in the bin dir of a filesystem hiearchy like '/usr/local/bin'
        base += "/../share/doc/qbrew/en/";
    } else {
        // we are in a hierarchy like '/usr/local'
        base += "/share/doc/qbrew/en/";
    }

#elif defined(Q_WS_MACX)
    if (QRegExp("Contents/MacOS/?$", false).match(base) != -1) {
        // we're pointing inside an application bundle
        base += "/../Resources/en.lproj/";
    } else if (QFile::exists(base + "/en.lproj/")) {
        // some other hierarchy using Mac style l10n
        base += "/en.lproj/"
    } else {
        // otherwise punt
        base += "/doc/en";
    }

#else // Win32 and others
    base += "/doc/en/";
#endif
    return base;
}

//////////////////////////////////////////////////////////////////////////////
// TODO: the following is a temporary conversion
// It needs to remain for at least one year after the 0.3.4 (10/2004) release
// Get rid of QDIR_HOME and QPREFIX at that time as well

#include "preferences.h"

void Controller::convertPreferences()
{
    if (QFile::exists(QDIR_HOME + "/" + ID_PREFERENCES_FILE)) {
        QString prefname(QDIR_HOME + "/" + ID_PREFERENCES_FILE);
        int status = QMessageBox::question(this, ID_TITLE + " - Old format",
            "<strong>" + prefname + "</strong>"
            "<p>An old configuration file was found. Do you wish to convert "
            "this to the new configuration format? This will remove the file.",
            QMessageBox::Yes | QMessageBox::Default,
            QMessageBox::No | QMessageBox::Escape);
        if (status == QMessageBox::Yes) {
            // convert it
            Preferences *prefs = new Preferences(prefname, PACKAGE, VERSION);
            QSettings config;
            config.setPath("usermode.org", "QBrew");
            config.beginGroup('/' + ID_TITLE);

            // convert general config
            config.beginGroup(CONFGROUP_GENERAL);
            config.writeEntry(CONF_GEN_APPDIR,
                              prefs->getString(ID_PREF_APPDIR,
                                               ID_PREF_APPDIR_DEFAULT));
            config.writeEntry(CONF_GEN_LOOK_FEEL,
                              prefs->getString(ID_PREF_WIDGET_STYLE,
                                               ID_PREF_WIDGET_STYLE_DEFAULT));
            config.writeEntry(CONF_GEN_LARGE_ICONS,
                              prefs->getBool(ID_PREF_LARGE_ICONS,
                                             ID_PREF_LARGE_ICONS_DEFAULT));
            config.endGroup();

            // convert recipe config
            config.beginGroup(CONFGROUP_RECIPE);
            config.writeEntry(CONF_RECIPE_BATCH,
                              prefs->getDouble(ID_PREF_BATCH,
                                               ID_PREF_BATCH_DEFAULT));
            config.writeEntry(CONF_RECIPE_STYLE,
                              prefs->getString(ID_PREF_RECIPE_STYLE,
                                               ID_PREF_RECIPE_STYLE_DEFAULT));
            config.endGroup();

            // convert calc config
            config.beginGroup(CONFGROUP_CALC);
            config.writeEntry(CONF_CALC_EFFICIENCY,
                              prefs->getDouble(ID_PREF_EFFICIENCY,
                                               ID_PREF_EFFICIENCY_DEFAULT));
            config.writeEntry(CONF_CALC_TINSETH,
                              prefs->getBool(ID_PREF_TINSETH,
                                             ID_PREF_TINSETH_DEFAULT));
            config.writeEntry(CONF_CALC_UNITS,
                              prefs->getString(ID_PREF_UNITS,
                                               ID_PREF_UNITS_DEFAULT));
            config.endGroup();
            config.endGroup();
            delete prefs;

            // now remove the old file
            QFile::remove(prefname);
            if (QFile::exists(prefname)) {
                // couldn't remove it!
                QMessageBox::warning(this, ID_TITLE + " - Warning",
                                     "Could not remove " + prefname);
            }
        }
    }
}
