/***************************************************************************
  export.cpp
  -------------------
  Export methods for the Model class
  -------------------
  Copyright (c) 2005 David Johnson
  Please see the header file for copyright and license information.
 ***************************************************************************/

#include <qdom.h>
#include <qfile.h>
#include <qtextstream.h>

#include "controller.h"
#include "resource.h"

#include "model.h"

using namespace AppResource;
using namespace CalcResource;

QString escape(const QString &s);
QString underline(const QString &line);

//////////////////////////////////////////////////////////////////////////////
// escape()
// --------
// Escape special xml/html characters

QString escape(const QString &s)
{
    QString x = s;
    x.replace("&", "&amp;");
    x.replace("<", "&lt;");
    x.replace(">", "&gt;");
    x.replace("'", "&apos;");
    x.replace("\"", "&quot;");
    return x;
}

QString underline(const QString &line)
{
    QString text;
    text = line + '\n' + text.fill('-', line.length()) + '\n';
    return text;
}

//////////////////////////////////////////////////////////////////////////////
// recipeText()
// ------------
// Get the ascii text of the recipe for exporting

QString Model::recipeText()
{
    // title stuff
    QString text = underline(recipe_->title());
    text += "Brewer: " + recipe_->brewer() + '\n';
    text += "Style: " +
        recipe_->style().name() + '\n';
    text += "Batch: " + recipe_->size().toQString(2);
    if (recipe_->mashed()) text += ", Mashed";
    text += "\n\n";

    // style stuff
    text += underline("Characteristics");
    text += "Recipe Gravity: " +
        QString::number(Calc::OG(recipe_), 'f', 3) + " OG\n";
    text += "Recipe Bitterness: " +
        QString::number(Calc::IBU(recipe_), 'f', 0) + " IBU\n";
    text += "Recipe Color: " +
        QString::number(Calc::SRM(recipe_), 'f', 0) + CHAR_LATIN_DEGREE + " SRM\n";
    text += "Estimated FG: " +
        QString::number(Calc::FGEstimate(recipe_), 'f', 3) + '\n';
    text += "Alcohol by Volume: " +
        QString::number(Calc::ABV(recipe_) * 100.0, 'f', 1) + "%\n";
    text += "Alcohol by Weight: " +
        QString::number(Calc::ABW(recipe_) * 100.0, 'f', 1) + "%\n\n";

    // ingredients
    text += underline("Ingredients");

    // grains
    GrainList *grainlist = recipe_->grains();
    GrainIterator itg;
    for (itg=grainlist->begin(); itg != grainlist->end(); ++itg) {
        text += (*itg).name().leftJustify(30, ' ');
        text += (*itg).weight().toQString(2) + ", ";
        text += (*itg).useString() + '\n';
    }
    text += '\n';

    // hops
    HopList *hoplist = recipe_->hops();
    HopList::Iterator ith;
    for (ith=hoplist->begin(); ith != hoplist->end(); ++ith) {
        text += (*ith).name().leftJustify(30, ' ');
        text += (*ith).weight().toQString(2) + ", ";
        text += (*ith).form() + ", ";
        text += QString::number((*ith).time()) + " minutes\n";
    }
    text += '\n';

    // misc ingredients
    MiscIngredientList *misclist = recipe_->miscs();
    MiscIngredientList::Iterator itm;
    for (itm=misclist->begin(); itm != misclist->end(); ++itm) {
        text += (*itm).name().leftJustify(30, ' ');
        text += (*itm).quantity().toQString(2) + ", ";
        text += (*itm).notes() + '\n';
    }
    text += '\n';

    // notes
    text += underline("Notes");

    // TODO: wrap long notes
    text += "Recipe Notes:\n" + recipe_->recipeNotes() + "\n\n";
    text += "Batch Notes:\n" + recipe_->batchNotes() + "\n\n";

    return text;
}

//////////////////////////////////////////////////////////////////////////////
// exportText()
// ------------
// Export recipe to plain text

bool Model::exportText(const QString &filename)
{
    QString content = recipeText();

    if (!filename.isEmpty()) {
        QFile f(filename);
        if (f.open(IO_WriteOnly)) {
            // file opened successfully
            if (f.writeBlock(content, content.length()) == -1) return false;
            f.flush();
            f.close();
            return true;
        } else {
            return false;
        }
    }
    return false;
}

//////////////////////////////////////////////////////////////////////////////
// recipeHTML()
// ------------
// Get the html of the recipe for printing and exporting

QString Model::recipeHTML()
{
    const QString header = "<big><strong>%1</strong></big>\n";
    const QString table = "<table summary=\"%1\" border=0 cellpadding=0 cellspacing=%2>\n";
    const QString th = "<td><strong>%1</strong></td>\n";
    
    // heading
    QString html = "<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n";
    html += "<html>\n<head>\n";
    html += "<meta name=\"generator\" content=\"" + ID_TITLE + " " + VERSION + " \">\n";
    html += "<title>" + escape(recipe_->title()) + "</title>\n";
    html += "<meta name=\"author\" content=\"" + escape(recipe_->brewer()) + "\">\n";
    html += "</head>\n\n";

    html += "<body>\n";

    html += table.arg("header").arg("5 bgcolor=\"#CCCCCC\" width=\"100%\"");
    html += "<tr><td>\n" + header.arg(escape(recipe_->title())) + "</td></tr>\n";
    html += "</table>\n<br>\n\n";

    // recipe table
    html += table.arg("recipe").arg(0);
    html += "<tbody>\n<tr>\n";
    html += th.arg("Recipe");
    html += "<td width=\"15\"></td>\n";
    html += "<td>" + escape(recipe_->title()) + "</td>\n";
    html += "<td width=\"25\"></td>\n";

    html += th.arg("Style");
    html += "<td width=\"15\"></td>\n";
    html += "<td>" + escape(recipe_->style().name()) + "</td>\n";
    html += "</tr>\n<tr>\n";

    html += th.arg("Brewer");
    html += "<td></td>\n";
    html += "<td>" + escape(recipe_->brewer()) + "</td>\n";
    html += "<td></td>\n";

    html += th.arg("Batch");
    html += "<td></td>\n";
    html += "<td>" + recipe_->size().toQString(2) + "</td>\n";

    if (recipe_->mashed()) html += "</tr>\n<tr>\n" + th.arg("Mashed");
    html += "</tr>\n</tbody>\n</table>\n<br>\n\n";

    // characteristics table
    html += header.arg("Recipe Characteristics");
    html += table.arg("characteristics").arg(0);
    html += "<tbody>\n<tr>\n";

    html += th.arg("Recipe Gravity");
    html += "<td width=\"15\"></td>\n";
    html += "<td>" + QString::number(Calc::OG(recipe_), 'f', 3) + " OG</td>\n";
    html += "<td width=\"25\"></td>\n";

    html += th.arg("Estimated FG");
    html += "<td width=\"15\"></td>\n";
    html += "<td>" + QString::number(Calc::FGEstimate(recipe_), 'f', 3) + " FG</td>\n";
    html += "</tr>\n<tr>\n";

    html += th.arg("Recipe Bitterness");
    html += "<td></td>\n";
    html += "<td>" + QString::number(Calc::IBU(recipe_), 'f', 0) + " IBU</td>\n";
    html += "<td></td>\n";

    html += th.arg("Alcohol by Volume");
    html += "<td></td>\n";
    html += "<td>" + QString::number(Calc::ABV(recipe_) * 100.0, 'f', 1) + "%</td>\n";
    html += "</tr>\n<tr>\n";

    html += th.arg("Recipe Color");
    html += "<td></td>\n";
    html += "<td>" + QString::number(Calc::SRM(recipe_), 'f', 0) + CHAR_LATIN_DEGREE + " SRM</td>\n";
    html += "<td></td>\n";

    html += th.arg("Alcohol by Weight");
    html += "<td></td>\n";
    html += "<td>" + QString::number(Calc::ABW(recipe_) * 100.0, 'f', 1) + "%</td>\n";
    html += "</tr>\n</tbody>\n</table>\n<br>\n\n";

    // ingredients table
    html += header.arg("Ingredients");
    html += table.arg("ingredients").arg(0);
    html += "<tbody>\n";

    // grains
    html += "<tr>\n" + th.arg("Quantity");
    html += "<td width=\"15\"></td>\n";
    html += th.arg("Grain");
    html += "<td width=\"15\"></td>\n";
    html += th.arg("Use");
    html += "</tr>\n\n";

    GrainList *grainlist = recipe_->grains();
    GrainIterator itg;
    for (itg=grainlist->begin(); itg != grainlist->end(); ++itg) {
        html += "<tr>\n<td>" + (*itg).weight().toQString(2) + "</td>\n";
        html += "<td></td>\n";
        html += "<td>" + escape((*itg).name()) + "</td>\n";
        html += "<td></td>\n";
        html += "<td colspan=3>" + (*itg).useString() + "</td>\n</tr>\n\n";
    }

    // hops
    html += "<tr>\n" + th.arg("Quantity");
    html += "<td width=\"15\"></td>\n";
    html += th.arg("Hop");
    html += "<td width=\"15\"></td>\n";
    html += th.arg("Form");
    html += "<td width=\"15\"></td>\n";
    html += th.arg("Time");
    html += "</tr>\n\n";

    HopList *hoplist = recipe_->hops();
    HopList::Iterator ith;
    for (ith=hoplist->begin(); ith != hoplist->end(); ++ith) {
        html += "<tr>\n<td>" + (*ith).weight().toQString(2) + "</td>\n";
        html += "<td></td>\n";
        html += "<td>" + escape((*ith).name()) + "</td>\n";
        html += "<td></td>\n";
        html += "<td>" + (*ith).form() + "</td>\n";
        html += "<td></td>\n";
        html += "<td>" + QString::number((*ith).time()) + " minutes</td>\n</tr>\n\n";
    }

    // misc ingredients
    html += "<tr>\n" + th.arg("Quantity");
    html += "<td width=\"15\"></td>\n";
    html += th.arg("Misc");
    html += "<td width=\"15\"></td>\n";
    html += th.arg("Notes");
    html += "</tr>\n\n";

    MiscIngredientList *misclist = recipe_->miscs();
    MiscIngredientList::Iterator itm;
    for (itm=misclist->begin(); itm != misclist->end(); ++itm) {
        html += "<tr>\n<td>" + (*itm).quantity().toQString(2) + "</td>\n";
        html += "<td></td>\n";
        html += "<td>" + escape((*itm).name()) + "</td>\n";
        html += "<td></td>\n";
        html += "<td colspan=3>" + escape((*itm).notes()) + "</td>\n</tr>\n\n";
    }

    html += "</tbody>\n</table>\n<br>\n\n";

    // notes
    // TODO: using replace() might be dangerous if we ever use richtext in notes
    html += header.arg("Recipe Notes") + "\n";
    html += "<p>" + escape(recipe_->recipeNotes()).replace('\n', "<br>\n") + "\n</p>\n<br>\n";

    html += header.arg("Batch Notes") + "\n";
    html += "<p>" + escape(recipe_->batchNotes()).replace('\n', "<br>\n") + "\n</p>\n<br>\n";

    html += "</body>\n</html>\n";

    return html;
}

//////////////////////////////////////////////////////////////////////////////
// exportText()
// ------------
// Export recipe to plain text

bool Model::exportHTML(const QString &filename)
{
    QString content = recipeHTML();

    if (!filename.isEmpty()) {
        QFile f(filename);
        if (f.open(IO_WriteOnly)) {
            // file opened successfully
            if (f.writeBlock(content, content.length()) == -1) return false;
            f.flush();
            f.close();
            return true;
        } else {
            return false;
        }
    }
    return false;
}

//////////////////////////////////////////////////////////////////////////////
// exportBeerXML()
// ---------------
// Export recipe to BeerXML format

bool Model::exportBeerXML(const QString &filename)
{
    QDomDocument doc; // BeerXML 1 doesn't have a doctype
    doc.appendChild(doc.createProcessingInstruction("xml", "version=\"1.0\""));

    // create the root element
    QDomElement root = doc.createElement(tagRECIPES);
    doc.appendChild(root);

    QDomComment comment =
        doc.createComment(QString("BeerXML generated by %1 %2")
                          .arg(PACKAGE).arg(VERSION));
    doc.appendChild(comment);

    // start record
    QDomElement record = doc.createElement(tagRECIPE);
    root.appendChild(record);
    QDomElement element = doc.createElement(tagVERSION);
    element.appendChild(doc.createTextNode(beerXMLVersion));
    record.appendChild(element);

    // title
    element = doc.createElement(tagNAME);
    element.appendChild(doc.createTextNode(recipe_->title()));
    record.appendChild(element);

    // type
    bool mash = false;
    element = doc.createElement(tagTYPE);
    switch (recipe_->recipeType()) {
      case Recipe::TypeExtract:
          mash = false;
          element.appendChild(doc.createTextNode(TYPE_EXTRACT));
          break;
      case Recipe::TypePartial:
          mash = true;
          element.appendChild(doc.createTextNode(TYPE_PARTIAL));
          break;
      case Recipe::TypeAllGrain:
          mash = true;
          element.appendChild(doc.createTextNode(TYPE_ALLGRAIN));
          break;
    }
    record.appendChild(element);

    // brewer
    element = doc.createElement(tagBREWER);
    element.appendChild(doc.createTextNode(recipe_->brewer()));
    record.appendChild(element);

    // style record
    Style style = recipe_->style();
    QDomElement subrecord = doc.createElement(tagSTYLE);
    // version
    element = doc.createElement(tagVERSION);
    element.appendChild(doc.createTextNode(beerXMLVersion));
    subrecord.appendChild(element);
    // name
    element = doc.createElement(tagNAME);
    element.appendChild(doc.createTextNode(style.name()));
    subrecord.appendChild(element);
    // TODO: unfinished...
    // category
    // category number
    // style letter
    // style guide
    // type
    // og
    element = doc.createElement(tagOGMIN);
    element.appendChild(doc.createTextNode
                        (QString::number(style.OGLow(),'f',4)));
    subrecord.appendChild(element);
    element = doc.createElement(tagOGMAX);
    element.appendChild(doc.createTextNode
                        (QString::number(style.OGHi(),'f',4)));
    subrecord.appendChild(element);
    // fg
    element = doc.createElement(tagFGMIN);
    element.appendChild(doc.createTextNode
                        (QString::number(style.FGLow(),'f',4)));
    subrecord.appendChild(element);
    element = doc.createElement(tagFGMAX);
    element.appendChild(doc.createTextNode
                        (QString::number(style.FGHi(),'f',4)));
    subrecord.appendChild(element);
    // ibu
    element = doc.createElement(tagIBUMIN);
    element.appendChild(doc.createTextNode
                        (QString::number(style.IBULow(),'f',2)));
    subrecord.appendChild(element);
    element = doc.createElement(tagIBUMAX);
    element.appendChild(doc.createTextNode
                        (QString::number(style.IBUHi(),'f',2)));
    subrecord.appendChild(element);
    // color
    element = doc.createElement(tagCOLORMIN);
    element.appendChild(doc.createTextNode
                        (QString::number(style.SRMLow(),'f',2)));
    subrecord.appendChild(element);
    element = doc.createElement(tagCOLORMAX);
    element.appendChild(doc.createTextNode
                        (QString::number(style.SRMHi(),'f',2)));
    subrecord.appendChild(element);
    // add style record
    record.appendChild(subrecord);

    // batch size
    element = doc.createElement(tagBATCHSIZE);
    double size = recipe_->size().amount(Volume::liter);
    element.appendChild(doc.createTextNode(QString::number(size,'f',8)));
    record.appendChild(element);

    if (mash) {
        // efficiency
        element = doc.createElement(tagEFFICIENCY);
        element.appendChild(doc.createTextNode
                            (QString::number(Calc::efficiency()*100.0,'f',1)));
        record.appendChild(element);
    }

    // fermentables list
    element = doc.createElement(tagFERMENTABLES);
    GrainIterator git;
    QDomElement subelement;
    // iterate through _grains list
    for (git=recipe_->grains()->begin(); git!=recipe_->grains()->end(); ++git) {
        // fermentable
        subrecord = doc.createElement(tagFERMENTABLE);
        // version
        subelement = doc.createElement(tagVERSION);
        subelement.appendChild(doc.createTextNode(beerXMLVersion));
        subrecord.appendChild(subelement);
        // name
        subelement = doc.createElement(tagNAME);
        subelement.appendChild(doc.createTextNode((*git).name()));
        subrecord.appendChild(subelement);
        // type // TODO: ???
        // amount
        subelement = doc.createElement(tagAMOUNT);
        size = (*git).weight().amount(Weight::kilogram);
        subelement.appendChild(doc.createTextNode(QString::number(size,'f',8)));
        subrecord.appendChild(subelement);
        // color
        subelement = doc.createElement(tagCOLOR);
        subelement.appendChild(doc.createTextNode
                               (QString::number((*git).color(),'f',2)));
        subrecord.appendChild(subelement);
        // yield
        subelement = doc.createElement(tagYIELD);
        double yield = Calc::extractToYield((*git).extract()) * 100.0;
        subelement.appendChild(doc.createTextNode(QString::number(yield,'f',2)));
        subrecord.appendChild(subelement);
        // use??? // TODO: beerxml doesn't have this
        element.appendChild(subrecord);
    }
    record.appendChild(element);

    // hop list
    element = doc.createElement(tagHOPS);
    HopIterator hit;
    QString buf;
    // iterate through hops list
    for (hit=recipe_->hops()->begin(); hit!=recipe_->hops()->end(); ++hit) {
        // hop
        subrecord = doc.createElement(tagHOP);
        // version
        subelement = doc.createElement(tagVERSION);
        subelement.appendChild(doc.createTextNode(beerXMLVersion));
        subrecord.appendChild(subelement);
        // name
        subelement = doc.createElement(tagNAME);
        subelement.appendChild(doc.createTextNode((*hit).name()));
        subrecord.appendChild(subelement);
        // amount
        subelement = doc.createElement(tagAMOUNT);
        size = (*hit).weight().amount(Weight::kilogram);
        subelement.appendChild(doc.createTextNode(QString::number(size,'f',8)));
        subrecord.appendChild(subelement);
        // alpha
        subelement = doc.createElement(tagALPHA);
        subelement.appendChild(doc.createTextNode
                               (QString::number((*hit).alpha(),'f',2)));
        subrecord.appendChild(subelement);
        // use // TODO: note that I'm using "Boil" for all hops...
        subelement = doc.createElement(tagUSE);
        subelement.appendChild(doc.createTextNode("Boil"));
        subrecord.appendChild(subelement);
        // time
        subelement = doc.createElement(tagTIME);
        subelement.appendChild(doc.createTextNode
                               (QString::number((*hit).time())));
        subrecord.appendChild(subelement);
        // form
        subelement = doc.createElement(tagFORM);
        buf = "Leaf";;
        if ((*hit).form() == HOP_PELLET) buf = "Pellet";
        if ((*hit).form() == HOP_PLUG) buf = "Plug";
        subelement.appendChild(doc.createTextNode(buf));
        subrecord.appendChild(subelement);
        element.appendChild(subrecord);
    }
    record.appendChild(element);
 
    // yeasts
    element = doc.createElement(tagYEASTS);
    MiscIngredientIterator mit;
    // iterate through list looking for yeasts
    // TODO: need to separate yeasts from other miscs in database
    for (mit=recipe_->miscs()->begin(); mit!=recipe_->miscs()->end(); ++mit) {
        if ((*mit).name().find("yeast", false) != -1) {
            // yeast
            subrecord = doc.createElement(tagYEAST);
            // version
            subelement = doc.createElement(tagVERSION);
            subelement.appendChild(doc.createTextNode(beerXMLVersion));
            subrecord.appendChild(subelement);
            // name
            subelement = doc.createElement(tagNAME);
            subelement.appendChild(doc.createTextNode((*mit).name()));
            subrecord.appendChild(subelement);
            // amount
            subelement = doc.createElement(tagAMOUNT);
            subelement.appendChild(doc.createTextNode("0.00")); // TODO: fixup for miscs
            subrecord.appendChild(subelement);
            // notes
            subelement = doc.createElement(tagNOTES);
            subelement.appendChild(doc.createTextNode((*mit).notes()));
            subrecord.appendChild(subelement);            
            // type ???
            // form ???
            element.appendChild(subrecord);
        }
    }
    record.appendChild(element);

    // miscs
    element = doc.createElement(tagMISCS);
    // iterate through list looking for non-yeast miscs
    for (mit=recipe_->miscs()->begin(); mit!=recipe_->miscs()->end(); ++mit) {
        if ((*mit).name().find("yeast", false) == -1) {
            // misc
            subrecord = doc.createElement(tagMISC);
            // version
            subelement = doc.createElement(tagVERSION);
            subelement.appendChild(doc.createTextNode(beerXMLVersion));
            subrecord.appendChild(subelement);
            // name
            subelement = doc.createElement(tagNAME);
            subelement.appendChild(doc.createTextNode((*mit).name()));
            subrecord.appendChild(subelement);
            // amount
            subelement = doc.createElement(tagAMOUNT);
            subelement.appendChild(doc.createTextNode("0.00")); // TODO: fixup for miscs
            subrecord.appendChild(subelement);
            // notes
            subelement = doc.createElement(tagNOTES);
            subelement.appendChild(doc.createTextNode((*mit).notes()));
            subrecord.appendChild(subelement);            
            element.appendChild(subrecord);
            // type ???
            // use ???
            // time ???
        }
    }
    record.appendChild(element);

    // waters
    element = doc.createElement(tagWATERS);
    // NOTE: not currently supporting water
    record.appendChild(element);

    // mash
    // NOTE: not currently supporting mash

    // notes
    if ((recipe_->recipeNotes().length() + recipe_->batchNotes().length()) > 0) {
        element = doc.createElement(tagNOTES);
        buf = recipe_->recipeNotes();
        if (recipe_->batchNotes().length() > 0) {
            if (buf.length() > 0) buf += "\n\n";
            buf += recipe_->batchNotes();
        }
        element.appendChild(doc.createTextNode(buf));
        record.appendChild(element);
    }

    // open file
    QFile* datafile = new QFile(filename);
    if (!datafile->open(IO_WriteOnly)) {
        // error opening file
        qWarning("Error: Cannot open file " + filename);
        datafile->close();
        return false;
    }

    // write it out
    QTextStream textstream(datafile);
    doc.save(textstream, 2);
    datafile->close();
    delete (datafile);

    return true;
}
