/***************************************************************************
  fixedspinbox.cpp
  -------------------
  Spinbox for fixed numbers
  -------------------
  Copyright 1999-2005, David Johnson
  Please see the header file for copyright and license information
***************************************************************************/

#include <cmath>
#include <qregexp.h>

#include "fixedspinbox.h"

#define fixedToInt(value) int(divisor_ * value + (value < 0 ? -tv_ : tv_))
#define intToFixed(value) double(value) / divisor_

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// FixedSpinBox()
// --------------
// Constructor

FixedSpinBox::FixedSpinBox(QWidget *parent, const char *name,
                           double minvalue, double maxvalue,
                           double step, unsigned precision)
    : QSpinBox(parent, name), min_(minvalue), max_(maxvalue),
      step_(step), precision_(precision)
{
    // NOTE: QDoubleValidator doesn't work with prefixes/suffixes
    QSpinBox::setValidator(0);

    divisor_ = int(pow(10.0, double(precision_)));
    // tiny value to prevent rounding errors
    tv_ = 1.0 / pow(10.0, double(precision_+1));

    QSpinBox::setMaxValue(fixedToInt(max_));
    QSpinBox::setMinValue(fixedToInt(min_));
    QSpinBox::setLineStep(fixedToInt(step_));
}

FixedSpinBox::~FixedSpinBox() { ; }

//////////////////////////////////////////////////////////////////////////////
// Accessors                                                                //
//////////////////////////////////////////////////////////////////////////////

void FixedSpinBox::setMinimum(double value)
{
    min_ = value;
    QSpinBox::setMinValue(fixedToInt(value));
}

double FixedSpinBox::minimum() const
    { return intToFixed(QSpinBox::minValue()); }

void FixedSpinBox::setMaximum(double value)
{
    max_ = value;
    QSpinBox::setMaxValue(fixedToInt(value));
}

double FixedSpinBox::maximum() const
    {  return intToFixed(QSpinBox::maxValue()); }

void FixedSpinBox::setSingleStep(double value)
{
    step_ = value;
    QSpinBox::setLineStep(fixedToInt(value));
}

double FixedSpinBox::singleStep() const
    { return intToFixed(QSpinBox::lineStep()); }

void FixedSpinBox::setPrecision(unsigned precision)
{
    precision_ = precision;
    divisor_ = int(pow(10.0, double(precision_)));
    tv_ = 1.0 / pow(10.0, double(precision_+1));
    QSpinBox::setMaxValue(fixedToInt(max_));
    QSpinBox::setMinValue(fixedToInt(min_));
    QSpinBox::setLineStep(fixedToInt(step_));
}

unsigned FixedSpinBox::precision() const
    { return precision_; }

void FixedSpinBox::setValue(double value)
    { QSpinBox::setValue(fixedToInt(value)); }

void FixedSpinBox::setValue(const QString &value)
{
    // grab just the number
    QRegExp regex("[0-9.]+");
    regex.search(value);
    setValue(regex.cap(0).toDouble());
}

double FixedSpinBox::value() const
    { return intToFixed(QSpinBox::value()); }

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// valueChange()
// -------------
// QSpinBox override, the value of the spinbox has changed

void FixedSpinBox::valueChange()
{
    QSpinBox::valueChange();
    emit valueChanged( intToFixed(QSpinBox::value()) );
}

//////////////////////////////////////////////////////////////////////////////
// mapValueToText()
// ----------------
// Convert the integer value to text

QString FixedSpinBox::mapValueToText(int value)
{
    return QString::number(intToFixed(value), 'f', precision_);
}

//////////////////////////////////////////////////////////////////////////////
// mapTextToValue()
// ----------------
// Convert the text value to an integer

int FixedSpinBox::mapTextToValue(bool *ok)
{
    double value = cleanText().toDouble(ok);
    return fixedToInt(value);
}

