/***************************************************************************
  graindb.cpp
  -------------------
  A dialog page for grains
  -------------------
  Copyright 2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcombobox.h>
#include <qimage.h>
#include <qlabel.h>
#include <qlistview.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qstringlist.h>

#include "fixedspinbox.h"

#include "resource.h"
#include "controller.h"
#include "model.h"
#include "view.h"

#include "graindb.h"

using namespace AppResource;

// Note that this class is very similar to the GrainPage class. This is
// because they both implement very similar interfaces. They are separate
// for maintenance reasons.

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// GrainDB()
// ---------
// Constructor

GrainDB::GrainDB(QWidget *parent, const char *name,  Model *model)
    : GrainView(parent, name), model_(model), currentitem_(0)
{
    // additional widget setup
    grainslistview->setColumnAlignment(GITEM_NAME, AlignLeft);
    grainslistview->setColumnAlignment(GITEM_EXTRACT, AlignRight);
    grainslistview->setColumnAlignment(GITEM_COLOR, AlignRight);
    grainslistview->setColumnAlignment(GITEM_USE, AlignRight);

    extractlabel->setBuddy(extractspin);
    extractspin->setMinimum(1.000);
    extractspin->setMaximum(1.100);
    extractspin->setSingleStep(0.001);
    extractspin->setPrecision(3);

    colorlabel->setBuddy(colorspin);
    colorspin->setMinimum(0.0);
    colorspin->setMaximum(500.0);
    colorspin->setSingleStep(1.0);
    colorspin->setPrecision(1);
    colorspin->setSuffix(AppResource::CHAR_LATIN_DEGREE);

    usecombo->insertStringList(Grain::useStringList());

    addbutton->setPixmap(Controller::findImage("sm_add"));
    applybutton->setPixmap(Controller::findImage("sm_apply"));
    removebutton->setPixmap(Controller::findImage("sm_remove"));

    // hide quantity in database editor
    grainslistview->setColumnWidthMode(GITEM_QUANTITY, QListView::Manual);
    grainslistview->hideColumn(GITEM_QUANTITY);
    quantitylabel->hide();
    quantityspin->hide();

    // adjust size of leftmost labels so they line up
    namelabel->adjustSize();
    int leftmost = namelabel->width();
    extractlabel->adjustSize();
    leftmost = QMAX(leftmost, extractlabel->width());
    extractlabel->setMinimumWidth(leftmost);
    namelabel->setMinimumWidth(leftmost);

    // fixup tab orders
    QWidget::setTabOrder(namecombo, quantityspin);
    QWidget::setTabOrder(quantityspin, extractspin);
    QWidget::setTabOrder(extractspin, colorspin);
    QWidget::setTabOrder(colorspin, usecombo);

    // setup connections
    connect(addbutton, SIGNAL(clicked()), SLOT(addGrain()));
    connect(applybutton, SIGNAL(clicked()), SLOT(applyGrain()));
    connect(removebutton, SIGNAL(clicked()), SLOT(removeGrain()));
    connect(namecombo, SIGNAL(activated(const QString &)),
            SLOT(updateFields(const QString &)));
    connect(grainslistview, SIGNAL(selectionChanged(QListViewItem *)),
            SLOT(itemSelected(QListViewItem *)));
}

GrainDB::~GrainDB() {}

//////////////////////////////////////////////////////////////////////////////
// Implementation                                                           //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// refresh()
// ---------
// initialize list and fields

void GrainDB::refresh()
{
    // update list
    GrainIterator it;
    GrainItem *item;
    GrainList *list = model_->grainDB();
    grainslistview->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new GrainItem(grainslistview);
        item->setText(GITEM_NAME, (*it).name());
        item->setText(GITEM_EXTRACT,
                      QString::number((*it).extract(),'f',3));
        item->setText(GITEM_COLOR,
                      QString::number((*it).color(),'f',1)+CHAR_LATIN_DEGREE);
        item->setText(GITEM_USE, (*it).useString());
        item->setIterator(it);
    }

    // update fields
    namecombo->clear();
    namecombo->insertStringList(model_->grainsList());
    updateFields(namecombo->currentText());
}

//////////////////////////////////////////////////////////////////////////////
// updateFields()
// --------------
// Update the fields in the entry area

void GrainDB::updateFields(const QString &name)
{
    Grain *thisgrain = model_->grain(name);
    if (thisgrain) { // sanity check
        extractspin->setValue(thisgrain->extract());
        colorspin->setValue(thisgrain->color());
    }
}

//////////////////////////////////////////////////////////////////////////////
// addGrain()
// ----------
// Add a grain to the recipe

void GrainDB::addGrain()
{
    GrainIterator it;
    GrainItem *item;

    Grain grain(namecombo->currentText(),
                Weight(quantityspin->value(), model_->defaultGrainUnit()),
                extractspin->value(), colorspin->value(),
                usecombo->currentText());

    if (model_->checkGrain(grain.name())) {
        // already exists, query user
        if (QMessageBox::information(this, ID_TITLE + " - Replace?",
                                     QString("<p>Are you sure you want to "
                                             "replace the entry \"%1\"")
                                     .arg(grain.name()),
                                     QMessageBox::No | QMessageBox::Escape,
                                     QMessageBox::Yes)
            == QMessageBox::No) return;

        // remove existing item from listview
        item = (GrainItem*)grainslistview->findItem(grain.name(), GITEM_NAME);
        if (item) {
            grainslistview->takeItem(item);
            delete item;
        }
    }

    // create new listview item
    item = new GrainItem(grainslistview);
    item->setText(GITEM_NAME, grain.name());
    item->setText(GITEM_EXTRACT,
                  QString::number(grain.extract(),'f',3));
    item->setText(GITEM_COLOR,
                  QString::number(grain.color(),'f',1)+CHAR_LATIN_DEGREE);
    item->setText(GITEM_USE, grain.useString());

    // add grain to database
    it = model_->addGrain(grain, true);
    item->setIterator(it);

    grainslistview->ensureItemVisible(item);
    grainslistview->setCurrentItem(item);
}

//////////////////////////////////////////////////////////////////////////////
// applyGrain()
// ------------
// Apply a grain to the database

void GrainDB::applyGrain()
{
    if (currentitem_ != 0) {
        Grain grain(namecombo->currentText(),
                    Weight(quantityspin->value(), model_->defaultGrainUnit()),
                    extractspin->value(), colorspin->value(), 
                    usecombo->currentText());
        // apply grain (overwrite is simplest way)
        GrainIterator it = model_->addGrain(grain, true);
        currentitem_->setIterator(it);
        currentitem_->refresh();
    }
}

//////////////////////////////////////////////////////////////////////////////
// removeGrain()
// -------------
// Remove a grain from the database

void GrainDB::removeGrain()
{
    if (currentitem_ != 0) {
        // remove grain from recipe
        model_->removeGrain(currentitem_->iterator());
        delete currentitem_; currentitem_ = 0;
    }
}

//////////////////////////////////////////////////////////////////////////////
// itemSelected()
// --------------
// A grain is selected in the listview

void GrainDB::itemSelected(QListViewItem *item)
{
    if (item) {
        currentitem_ = dynamic_cast<GrainItem*>(item);
        if (currentitem_ == 0) {
            qWarning("qbrew: bad dynamic cast, line" + __LINE__);
            return;
        }
        namecombo->setEditText(item->text(GITEM_NAME));
        extractspin->setValue(item->text(GITEM_EXTRACT));
        colorspin->setValue(item->text(GITEM_COLOR));
        View::setComboItem(usecombo, item->text(GITEM_USE));
    }
}
