/***************************************************************************
  hopspage.cpp
  -------------------
  A dialog page for hops
  -------------------
  Copyright 1999-2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcombobox.h>
#include <qimage.h>
#include <qlabel.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qstringlist.h>

#include "fixedspinbox.h"

#include "resource.h"
#include "controller.h"
#include "model.h"
#include "view.h"

#include "hopspage.h"

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// HopPage()
// ----------
// Constructor

HopPage::HopPage(QWidget *parent, const char *name, Model *model)
    : HopView(parent, name), model_(model), currentitem_(0)
{
    // additional widget setup
    hoplistview->setColumnAlignment(HITEM_NAME, AlignLeft);
    hoplistview->setColumnAlignment(HITEM_QUANTITY, AlignRight);
    hoplistview->setColumnAlignment(HITEM_ALPHA, AlignRight);
    hoplistview->setColumnAlignment(HITEM_FORM, AlignRight);
    hoplistview->setColumnAlignment(HITEM_TIME, AlignRight);

    quantitylabel->setBuddy(quantityspin);
    quantityspin->setMinimum(0.00);
    quantityspin->setMaximum(100.0);
    quantityspin->setSingleStep(0.25);
    quantityspin->setPrecision(2);

    alphalabel->setBuddy(alphaspin);
    alphaspin->setMinimum(0.0);
    alphaspin->setMaximum(50.0);
    alphaspin->setSingleStep(0.1);
    alphaspin->setPrecision(1);
    alphaspin->setSuffix("%");

    formcombo->insertStringList(model_->formsList());

    addbutton->setPixmap(Controller::findImage("sm_add"));
    applybutton->setPixmap(Controller::findImage("sm_apply"));
    removebutton->setPixmap(Controller::findImage("sm_remove"));

    // adjust size of leftmost labels so they line up
    namelabel->adjustSize();
    int leftmost = namelabel->width();
    quantitylabel->adjustSize();
    leftmost = QMAX(leftmost, quantitylabel->width());
    quantitylabel->setMinimumWidth(leftmost);
    namelabel->setMinimumWidth(leftmost);

    // fixup tab orders
    QWidget::setTabOrder(namecombo, quantityspin);
    QWidget::setTabOrder(quantityspin, alphaspin);
    QWidget::setTabOrder(alphaspin, formcombo);
    QWidget::setTabOrder(alphaspin, timespin);

    // setup connections
    connect(addbutton, SIGNAL(clicked()), SLOT(addHop()));
    connect(applybutton, SIGNAL(clicked()), SLOT(applyHop()));
    connect(removebutton, SIGNAL(clicked()), SLOT(removeHop()));
    connect(namecombo, SIGNAL(activated(const QString &)),
            SLOT(updateFields(const QString &)));
    connect(hoplistview, SIGNAL(selectionChanged(QListViewItem *)),
            SLOT(itemSelected(QListViewItem *)));
}

HopPage::~HopPage() { ; }

//////////////////////////////////////////////////////////////////////////////
// Implementation                                                           //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// refresh()
// ---------
// initialize list and fields (new/opened document)

void HopPage::refresh()
{
    // initialize list
    HopIterator it;
    HopItem *item;
    HopList *list = model_->recipe()->hops();
    hoplistview->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new HopItem(hoplistview);
        item->setText(HITEM_NAME,
                      (*it).name());
        item->setText(HITEM_QUANTITY,
                      QString::number((*it).weight().amount(),'f',2)
                      + " " + (*it).weight().unit().symbol());
        item->setText(HITEM_ALPHA,
                      QString::number((*it).alpha(),'f',1) + '%');
        item->setText(HITEM_FORM,
                      (*it).form());
        item->setText(HITEM_TIME,
                      QString::number((*it).time()));
        item->setIterator(it);
    }

    // initialize fields
    namecombo->clear();
    namecombo->insertStringList(model_->hopsList());
    updateFields(namecombo->currentText());
}

//////////////////////////////////////////////////////////////////////////////
// updateFields()
// --------------
// Update the fields in the entry area

void HopPage::updateFields(const QString &name)
{
    Hop *thishop = model_->hop(name);
    if (thishop) { // sanity check
        quantityspin->setValue(thishop->weight().amount());
        quantityspin->setSuffix(" " + thishop->weight().unit().symbol());
        alphaspin->setValue(thishop->alpha());
        timespin->setValue(thishop->time());
        // set form according to preferences
        if (thishop->form().isEmpty()) {
            View::setComboItem(formcombo, model_->defaultHopform());
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// addHop()
// ---------
// Add a hop to recipe

void HopPage::addHop()
{
    HopIterator it;
    HopItem *item;
    Hop hop(namecombo->currentText(),
             Weight(quantityspin->value(), model_->defaultHopUnit()),
             formcombo->currentText(), alphaspin->value(),
             timespin->value());
    // add hop to database if it doesn't exist
    model_->addHop(hop, false);

    // create new listview item
    item = new HopItem(hoplistview);
    item->setText(HITEM_NAME, hop.name());
    item->setText(HITEM_QUANTITY,
                  QString::number(hop.weight().amount(),'f',2)
                  + " " + hop.weight().unit().symbol());
    item->setText(HITEM_ALPHA,
                  QString::number(hop.alpha(),'f',1) + '%');
    item->setText(HITEM_FORM, hop.form());
    item->setText(HITEM_TIME, QString::number(hop.time()));

    hoplistview->ensureItemVisible(item);
    hoplistview->setCurrentItem(item);

    // add hop to recipe
    it = model_->recipe()->addHop(hop);
    model_->setModified(true);
    item->setIterator(it);

    emit modified();
}

//////////////////////////////////////////////////////////////////////////////
// applyHop()
// -----------
// Apply a hop to recipe

void HopPage::applyHop()
{
    if (currentitem_ != 0) {
        Hop hop(namecombo->currentText(),
                 Weight(quantityspin->value(), model_->defaultHopUnit()),
                 formcombo->currentText(), alphaspin->value(),
                 timespin->value());
        // add hop to database if it doesn't exist
        model_->addHop(hop, false);

        // change the hop in the recipe
        model_->recipe()->applyHop(currentitem_->iterator(), hop);
        model_->setModified(true);
        currentitem_->refresh();

        emit modified();
    }
}

//////////////////////////////////////////////////////////////////////////////
// removeHop()
// ------------
// Remove a hop from recipe

void HopPage::removeHop()
{
    if (currentitem_ != 0) {
        // remove hop from recipe
        model_->recipe()->removeHop(currentitem_->iterator());
        model_->setModified(true);
        delete currentitem_; currentitem_ = 0;
        emit modified();
    }
}

//////////////////////////////////////////////////////////////////////////////
// itemSelected()
// -----------------
// Received when user selects an item in listview

void HopPage::itemSelected(QListViewItem *item)
{
    if (item) {
        currentitem_ = dynamic_cast<HopItem*>(item);
        if (currentitem_ == 0) {
            qWarning("qbrew: bad dynamic cast, line" + __LINE__);
            return;
        }
        namecombo->setEditText(item->text(HITEM_NAME));
        quantityspin->setValue(item->text(HITEM_QUANTITY));
        alphaspin->setValue(item->text(HITEM_ALPHA));
        formcombo->setEditText(item->text(HITEM_FORM));
        // NOTE: cannot have any suffixes or toUInt() will fail
        timespin->setValue(item->text(HITEM_TIME).toUInt());
    }
}
