/***************************************************************************
  miscpage.cpp
  -------------------
  A dialog page for misc ingredients
  -------------------
  Copyright 1999-2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcombobox.h>
#include <qimage.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qstringlist.h>

#include "fixedspinbox.h"
#include "controller.h"
#include "model.h"

#include "miscpage.h"

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// MiscPage()
// ----------
// Constructor

MiscPage::MiscPage(QWidget *parent, const char *name, Model *model)
    : MiscView(parent, name), model_(model), currentitem_(0)
{
    // additional widget setup
    misclistview->setColumnAlignment(MITEM_NAME, AlignLeft);
    misclistview->setColumnAlignment(MITEM_QUANTITY, AlignRight);
    misclistview->setColumnAlignment(MITEM_NOTES, AlignLeft);

    quantitylabel->setBuddy(quantityspin);
    quantityspin->setMinimum(0.00);
    quantityspin->setMaximum(100.0);
    quantityspin->setSingleStep(0.25);
    quantityspin->setPrecision(2);

    addbutton->setPixmap(Controller::findImage("sm_add"));
    applybutton->setPixmap(Controller::findImage("sm_apply"));
    removebutton->setPixmap(Controller::findImage("sm_remove"));

    // adjust size of leftmost labels so they line up
    namelabel->adjustSize();
    int leftmost = namelabel->width();
    quantitylabel->adjustSize();
    leftmost = QMAX(leftmost, quantitylabel->width());
    quantitylabel->setMinimumWidth(leftmost);
    namelabel->setMinimumWidth(leftmost);

    // fixup tab orders
    QWidget::setTabOrder(namecombo, quantityspin);
    QWidget::setTabOrder(quantityspin, notesedit);

    // setup connections
    connect(addbutton, SIGNAL(clicked()), SLOT(addMiscIngredient()));
    connect(applybutton, SIGNAL(clicked()), SLOT(applyMiscIngredient()));
    connect(removebutton, SIGNAL(clicked()), SLOT(removeMiscIngredient()));
    connect(namecombo, SIGNAL(activated(const QString &)),
            SLOT(updateFields(const QString &)));
    connect(misclistview, SIGNAL(selectionChanged(QListViewItem *)),
            SLOT(itemSelected(QListViewItem *)));
}

MiscPage::~MiscPage() { ; }

//////////////////////////////////////////////////////////////////////////////
// Implementation                                                           //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// refresh()
// ---------
// initialize list and fields (new/opened document)

void MiscPage::refresh()
{
    // initialize list
    MiscIngredientIterator it;
    MiscItem *item;
    MiscIngredientList *list = model_->recipe()->miscs();
    misclistview->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new MiscItem(misclistview);
        item->setText(MITEM_NAME,
                      (*it).name());
        item->setText(MITEM_QUANTITY,
                      QString::number((*it).quantity().amount(),'f',2)
                      + " " + (*it).quantity().unit().symbol());
        item->setText(MITEM_NOTES,
                      (*it).notes());
        item->setIterator(it);
    }

    // initialize fields
    namecombo->clear();
    namecombo->insertStringList(model_->miscList());
    updateFields(namecombo->currentText());
}

//////////////////////////////////////////////////////////////////////////////
// updateFields()
// ------------------
// Update the fields in the entry area

void MiscPage::updateFields(const QString &name)
{
    MiscIngredient *thismisc = model_->misc(name);
    if (thismisc) { // sanity check
        quantityspin->setValue(thismisc->quantity().amount());
        quantityspin->setSuffix(" " + thismisc->quantity().unit().symbol());
        notesedit->setText(thismisc->notes());
    }
}

//////////////////////////////////////////////////////////////////////////////
// addMiscIngredient()
// -----------------------
// adds a misc ingredient to recipe

void MiscPage::addMiscIngredient()
{
    MiscIngredientIterator it;
    MiscItem *item;

    MiscIngredient misc(namecombo->currentText(),
                        Quantity(quantityspin->value(),
                                 model_->defaultMiscUnit()),
                        notesedit->text());
    // add ingredient to database if it doesn't exist
    model_->addMisc(misc, false);

    // create new listview item
    item = new MiscItem(misclistview);
    item->setText(MITEM_NAME, misc.name());
    item->setText(MITEM_QUANTITY,
                  QString::number(misc.quantity().amount(),'f',2)
                  + " " + misc.quantity().unit().symbol());
    item->setText(MITEM_NOTES, misc.notes());

    misclistview->ensureItemVisible(item);
    misclistview->setCurrentItem(item);

    // add ingredient to recipe
    it = model_->recipe()->addMisc(misc);
    model_->setModified(true);
    item->setIterator(it);

    emit modified();
}

//////////////////////////////////////////////////////////////////////////////
// applyMiscIngredient()
// -------------------------
// applies a misc ingredient to recipe

void MiscPage::applyMiscIngredient()
{
    if (currentitem_ != 0) {
        MiscIngredient misc(namecombo->currentText(),
                            Quantity(quantityspin->value(),
                                     model_->defaultMiscUnit()),
                            notesedit->text());
        // add ingredient to database if it doesn't exist
        model_->addMisc(misc, false);

        // change the ingredient in the recipe
        model_->recipe()->applyMisc(currentitem_->iterator(), misc);
        model_->setModified(true);
        currentitem_->refresh();

        emit modified();
     }
}

//////////////////////////////////////////////////////////////////////////////
// removeMiscIngredient()
// --------------------------
// rremoves a misc ingredient from recipe

void MiscPage::removeMiscIngredient()
{
    if (currentitem_ != 0) {
        // remove ingredient from recipe
        model_->recipe()->removeMisc(currentitem_->iterator());
        model_->setModified(true);
        delete currentitem_; currentitem_ = 0;
        emit modified();
    }
}

//////////////////////////////////////////////////////////////////////////////
// itemSelected()
// ------------------
// Received when user selects an item in the list view

void MiscPage::itemSelected(QListViewItem *item)
{
    if (item) {
        currentitem_ = dynamic_cast<MiscItem*>(item);
        if (currentitem_ == 0) {
            qWarning("qbrew: bad dynamic cast, line" + __LINE__);
            return;
        }
        namecombo->setEditText(item->text(MITEM_NAME));
        quantityspin->setValue(item->text(MITEM_QUANTITY));
        notesedit->setText(item->text(MITEM_NOTES));
    }
}
