/***************************************************************************
                          rinputpanel.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/


/****************************************************************************
** rinputpanel.cpp 1998/09/20 A. Mustun RibbonSoft
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#include "rinputpanel.h"

#include <qapplication.h>
#include <qradiobutton.h>
#include <qslider.h>
#include <qstring.h>
#include <qtooltip.h>
#include <qwindowdefs.h>

#include "rappwin.h"
#include "rcadwidget.h"
#include "reventdef.h"
#include "rgraphic.h"
#include "rlayoutdef.h"
#include "rlineedit.h"
#include "rlog.h"
#include "rmath.h"
#include "rstring.h"
#include "rtoolbutton.h"
#include "rxpminp.h"



// Constructor:
//
RInputPanel::RInputPanel(QWidget* _parent, 
               const char* _name,
               WFlags _f)
:RWidget(_parent, _name, _f)
{
  graphic=0;

  setFrameStyle(Panel|Raised);
  setLineWidth(1);

  gap=2;
  posY=gap;
  bSize=17;
  panelWidth = 150;
  panelHeight= 0;
  group=0;
  
  setFixedWidth(panelWidth);

  bLock.setAutoDelete(true);
  eInput.setAutoDelete(true);
  lLabel.setAutoDelete(true);
  gGroup.setAutoDelete(true);
  rButton.setAutoDelete(true);
}



// Destructor:
//
RInputPanel::~RInputPanel()
{

}



// Reset all layout objects in input panel:
//
void
RInputPanel::reset()
{
  bLock.clear();
  eInput.clear();
  lLabel.clear();
  gGroup.clear();
  rButton.clear();
  posY=gap;
  group=0;
  hide();
}



// Set the pointer to a graphic and update the layers:
//
void
RInputPanel::setGraphic(RGraphic* _graphic) 
{
  graphic=_graphic;
}




// Lock / unlock an edit field:
//
void
RInputPanel::lock(int /*_ind*/)
{

}



// New group for new collection of radiobuttons:
//
void
RInputPanel::newGroup(const QString& _title)
{
  RLOG("\nRInputPanel: newGroup...");

  // New group:
  //
  RPanelButtonGroup* tmpGroup = new RPanelButtonGroup(_title, this, "buttongroup");
  tmpGroup->move(2*gap, posY);
  tmpGroup->resize(width()-4*gap-1, 0);
  gGroup.append(tmpGroup);
  posY+=8*gap;

  adjustSize();
  
  RLOG("\n...RInputPanel: newGroup");
}



// End current group for collection of radiobuttons:
//
void
RInputPanel::endGroup()
{
  RLOG("\nRInputPanel: endGroup...");

  if(gGroup.current()) {
    posY+=4*gap;
    gGroup.current()->resize(gGroup.current()->width(), posY-gGroup.current()->y());
    posY+=2*gap;

    adjustSize();
  }
  
  RLOG("\n...RInputPanel: endGroup");
}




// Add a line edit to this panel:
//
void
RInputPanel::addLineEdit(const QString& _label, bool _lockButton)
{
  RLOG("\nRInputPanel: addLineEdit...");

  // Label:
  //
  RPanelLabel* tmpLLabel = new RPanelLabel(_label, this, "lLabel");
  tmpLLabel->setGeometry(2*gap, posY, width()/2-3*gap, bSize);
  tmpLLabel->setAlignment(AlignRight);
  lLabel.append(tmpLLabel);
  
  if(tmpLLabel->sizeHint().width()>width()/2) {
    panelWidth=tmpLLabel->sizeHint().width()>width()*2+4*gap;
  }

  // Input field:
  //
  RPanelLineEdit* tmpEInput = new RPanelLineEdit(this, "eInput");
  if(_lockButton) tmpEInput->setGeometry(width()/2+gap/2, posY, width()/2-2*gap-bSize-2, bSize);
  else            tmpEInput->setGeometry(width()/2+gap/2, posY, width()/2-2*gap, bSize);
  tmpEInput->setMaxLength(16);
  
  // Connect change of this edit line with slot "valueChanged()":
  //
  connect(tmpEInput, SIGNAL(textChanged(const QString&)),
          this, SLOT(valueChanged(const QString&)));
  eInput.append(tmpEInput);

  // Add the lock button:
  //
  RPanelToolButton* tmpBLock = new RPanelToolButton(this, "bLock");
  if(_lockButton) {
    tmpBLock->setPixmap(QPixmap(lock02_xpm));
    tmpBLock->setGeometry(width()-gap-bSize, posY,
                          bSize, bSize);
    tmpBLock->setToggleButton(true);
    QToolTip::add(tmpBLock, tr("Lock / unlock this value"));
    connect(tmpBLock, SIGNAL(clicked()),
            this, SLOT(valueChanged()));
    connect(tmpBLock, SIGNAL(clicked()), 
            this, SLOT(updateIcons()));
  }
  else {
    tmpBLock->hide();
  }
  bLock.append(tmpBLock);

  posY+=bSize+gap;

  adjustSize();
  
  RLOG("\n...RInputPanel: addLineEdit");
}



// Add a radiobutton to this panel:
//
//  _qInfo: Quickinfo text
//  _col:   Column 0...
//  _maxCol: Max number of columns 1...
//
void
RInputPanel::addRadioButton(QPixmap _pixmap,
                            const QString& _qInfo,
                            int _col, int _maxCol)
{
  int itemWidth = (width()-(8+_maxCol)*gap)/_maxCol-2;

  // Radio Button:
  //
  RPanelRadioButton* tmpRadioButton = new RPanelRadioButton(this, "rbut");
  tmpRadioButton->setGeometry(6*gap+_col*itemWidth+gap,
                              posY,
                              itemWidth, bSize);
  tmpRadioButton->setPixmap(_pixmap);
  if(_qInfo) QToolTip::add(tmpRadioButton, _qInfo);
  gGroup.current()->insert(tmpRadioButton);

  connect(tmpRadioButton, SIGNAL(clicked()),
          this, SLOT(buttonChanged()));

  rButton.append(tmpRadioButton);

  // Max number of columns reached / next line:
  //
  if(_col==_maxCol-1) posY+=bSize+gap;

  adjustSize();
}



// Adjust size automatically:
//
void
RInputPanel::adjustSize()
{
  RLOG("\nRInputPanel::adjustSize()...");

  panelHeight=posY+gap;
  resize(panelWidth, panelHeight);
  //move(0,0);
  
  RLOG("\n...RInputPanel::adjustSize()");
}



// Set Value of a line edit:
//
void
RInputPanel::setValue(int _ind, double _value)
{
  if(_ind>=0 && _ind<(int)eInput.count()) {
    //eInput.at(_ind)->setText(strFloatToString(_value));

    //QString str;
    //str.setNum(_value, 'g', 3);
    //eInput.at(_ind)->setText(str);
    eInput.at(_ind)->setFloat(_value);
  }
}




// Get Value from a line edit as double:
//
double
RInputPanel::getValue(int _ind)
{
  if(_ind>=0 && _ind<(int)eInput.count()) {
  
    return mtCalculate( eInput.at(_ind)->text() );
    
    /*
    QString str = eInput.at(_ind)->text();
    //return str.toFloat();
    ExprParser parser;
    parser.setExpression( str.latin1() );
    ret = parser.getValue();
    if( !parser.errorOccurred() ) {
      return ret;
    }
    */
  }
  else {
    return 0.0;
  }
}



// Set the checked Radio Button index:
//
void
RInputPanel::setCheckedRadioButton(int _ind)
{
  if(_ind>=0 && _ind<(int)rButton.count()) {
    rButton.at(_ind)->setChecked(true);
  }
}



// Find out if Radio Button with this index is checked:
//
int
RInputPanel::isRadioButtonChecked(int _ind)
{
  if(_ind>=0 && _ind<(int)rButton.count()) {
    return (int)rButton.at(_ind)->isChecked();
  }
  return -1;
}



// Update the icons of the lock buttons:
//
void
RInputPanel::updateIcons()
{
  QToolButton* but;
  for(but=bLock.first(); but!=0; but=bLock.next()) {
    if(but->isOn()) {
      but->setPixmap(QPixmap(lock01_xpm));
    }
    else {
      but->setPixmap(QPixmap(lock02_xpm));
    }
  }
}


// Value in one of the edit lines has changed:
//
void
RInputPanel::valueChanged(const QString&)
{
  if(graphic) graphic->distributeEvent(REVENT_MOUSEMOVE);
}



// Button status changed:
//
void
RInputPanel::buttonChanged()
{
  if(graphic) graphic->distributeEvent(REVENT_MOUSEMOVE);
}


/**
 * Mouse entered - but we don't want the preview to disappear!.
 */
void
RInputPanel::enterEvent( QEvent* /*_ev*/ )
{
  if(graphic) graphic->distributeEvent(REVENT_MOUSEMOVE);
}


/**
 * Mouse left - tell the graphic.
 */
void
RInputPanel::leaveEvent( QEvent* /*_ev*/ )
{
  if(graphic) graphic->distributeEvent(REVENT_MOUSELEAVE);
}


void
RInputPanel::mouseMoveEvent( QMouseEvent* /*_ev*/ )
{
  if(graphic) graphic->distributeEvent(REVENT_MOUSEMOVE);
}


/**
 * Key released. Test if it's enter and tell the graphic if so...
 */
void
RInputPanel::keyReleaseEvent ( QKeyEvent * e )
{
  RWidget::keyReleaseEvent( e );

  if(e->key()==Key_Enter || e->key()==Key_Return) {
    if(graphic) {
      graphic->distributeEvent(REVENT_LBUTTONUP);
      graphic->distributeEvent(REVENT_MOUSEMOVE);
    }
  }
  
  e->accept();
}


/**
 * Key pressed. Accept...
 */
void
RInputPanel::keyPressEvent ( QKeyEvent * e )
{
  RWidget::keyPressEvent( e );
  e->accept();
}




// EOF









