/******************************************************************

Copyright (C) 1996 by Brian Scearce

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
and/or distribute copies of the Software, and to permit persons to whom
the Software is furnished to do so, subject to the following conditions:

1. The above copyright notice and this permission notice shall be
   included in all copies or substantial portions of the Software.

2. Redistribution for profit requires the express, written permission of
   the author.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL BRIAN SCEARCE BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.
******************************************************************/

/** Edge
	Routine to do edge detection.  The idea is that we
	consider the box around a pixel:

		1 2 3
		4 5 6
		7 8 9

	The pixel under consideration is at 5.  If any three
	adjacent surrounding pixels have a wildly different
	value than the other six, we're probably looking at
	an edge.
*/

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "qcam.h"

static int new_scan[MAX_HEIGHT][MAX_WIDTH];
struct e_delta { int dx[3]; int dy[3]; };


int
qc_edge_detect(const struct qcam *q, scanbuf *scan, int tolerance)
{
  int width = q->width / q->transfer_scale;
  int height = q->height / q->transfer_scale;
  int x, y;
  int dx, dy;
  int total;
  int edge_total;
  int i, i2;
  int diff;
  int max_diff;
  int pixelrange = (q->bpp == 4) ? 15 : 63;
  static struct e_delta edge_delta[8] = {
	{ {-1, -1, -1}, {-1,  0,  1} }, /* left edge */
	{ {-1, -1,  0}, { 0, -1, -1} }, /* upper left corner */
	{ {-1,  0,  1}, {-1, -1, -1} }, /* top edge */
	{ { 0,  1,  1}, {-1, -1,  0} }, /* upper right corner */
	{ { 1,  1,  1}, {-1,  0,  1} }, /* right edge */
        { { 1,  1,  0}, { 0,  1,  1} }, /* lower right corner */
	{ { 1,  0, -1}, { 1,  1,  1} }, /* lower edge */
	{ { 0, -1, -1}, { 1,  1,  0} }  /* lower left edge */
  };

  if (tolerance == -1)
    tolerance = (pixelrange+1)/16;

  for (x = 1; x < width-1; x++) {
    for (y = 1; y < height-1; y++) {
      total = 0;
      for (dx = -1; dx <= 1; dx++) {
	for (dy = -1; dy <= 1; dy++) {
	  total += scan[(y+dy)*width + x+dx];
	}
      }

      max_diff = 0;
#if 0
      for (i = 0; i < 8; i++) {
	edge_total = 0;
	for (i2 = 0; i2 < 3; i2++) {
	  edge_total += scan[(y+edge_delta[i].dy[i2])*width + x+edge_delta[i].dx[i2]];
	}
	diff = 2*edge_total - (total - edge_total);
	diff = abs(diff);
#ifdef DEBUG
	if (diff > pixelrange) {
	  fprintf(stderr, "Pixel too big at %d, %d,", x, y);
	  fprintf(stderr, "total = %d, edge_total = %d, diff = %d\n",
		total, edge_total, diff);
	}
#endif
	if (diff < tolerance)
	  diff = 0;
	if (diff > max_diff)
	  max_diff = diff;
      }
      new_scan[y][x] = pixelrange-max_diff/6;
#endif
      for (i = 0; i < 4; i++) {
	edge_total = 0;
	for (i2 = 0; i2 < 3; i2++) {
	  edge_total += scan[(y+edge_delta[i].dy[i2])*width + x+edge_delta[i].dx[i2]] -
		scan[(y+edge_delta[i+4].dy[i2])*width + x+edge_delta[i+4].dx[i2]];
	}
	diff = abs(edge_total);
	if (diff < tolerance) diff = 0;
	else if (diff > max_diff) max_diff = diff;
      }
      new_scan[y][x] = pixelrange-max_diff/3;
    }
  }
  for (x = 1; x < width-1; x++) {
    for (y = 1; y < height-1; y++) {
      scan[y*width + x] = new_scan[y][x];
    }
  }
  for (x = 0; x < width; x++)
    scan[x] = scan[(height-1)*width + x] = pixelrange;
  for (y = 0; y < height; y++)
    scan[y*width] = scan[y*width + width-1] = pixelrange;
  return 0;
}

