/***************************************************************************
                         qgsgraduatedsymboldialog.cpp  -  description
                             -------------------
    begin                : Oct 2003
    copyright            : (C) 2003 by Marco Hugentobler
    email                : mhugent@geo.unizh.ch
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/* $Id: qgsgraduatedsymboldialog.cpp 6087 2006-11-15 17:35:28Z mhugent $ */

#include "qgsgraduatedsymboldialog.h"
#include "qgsfield.h"
#include "qgsgraduatedsymbolrenderer.h"
#include "qgsludialog.h"
#include "qgsvectordataprovider.h"
#include "qgsvectorlayer.h"


QgsGraduatedSymbolDialog::QgsGraduatedSymbolDialog(QgsVectorLayer * layer): QDialog(), mVectorLayer(layer), sydialog(layer)
{
    setupUi(this);
#ifdef QGISDEBUG
    qWarning("constructor QgsGraduatedSymbolDialog");
#endif

    setOrientation(Qt::Vertical);
    setSizeGripEnabled(true);

    //find out the numerical fields of mVectorLayer
    QgsVectorDataProvider *provider;
    if (provider = dynamic_cast<QgsVectorDataProvider *>(mVectorLayer->getDataProvider()))
    {
	std::vector < QgsField > const & fields = provider->fields();
	int fieldnumber = 0;
	QString str;
	
	for (std::vector < QgsField >::const_iterator it = fields.begin(); 
             it != fields.end(); 
             ++it)
        {
	    QString type = (*it).type();
	    if (type != "String" && type != "varchar" && type != "geometry")
            {
		str = (*it).name();
		classificationComboBox->insertItem(str);
		mFieldMap.insert(std::make_pair(str, fieldnumber));
            }
	    fieldnumber++;
        }
    } 
    else
    {
	qWarning("Warning, data provider is null in QgsGraduatedSymbolDialog::QgsGraduatedSymbolDialog(...)");
	return;
    }

    modeComboBox->insertItem("Empty");
    modeComboBox->insertItem("Equal Interval");
    
    //restore the correct settings
    const QgsGraduatedSymbolRenderer* renderer = dynamic_cast < const QgsGraduatedSymbolRenderer * >(layer->renderer());
    
    if (renderer)
    {
	std::list < QgsSymbol * >list = renderer->symbols();
	
	//display the classification field
	QString classfield="";
	for(std::map<QString,int>::iterator it=mFieldMap.begin();it!=mFieldMap.end();++it)
	{
	    if(it->second==renderer->classificationField())
	    {
		classfield=it->first;
		break;
	    }
	}
	classificationComboBox->setCurrentText(classfield);

	QGis::VectorType m_type = mVectorLayer->vectorType();
	numberofclassesspinbox->setValue(list.size());
	//fill the items of the renderer into mValues
	for(std::list<QgsSymbol*>::iterator it=list.begin();it!=list.end();++it)
	{
	    //todo: make an assignment operator and a copy constructor for QgsSymbol
		QString classbreak=(*it)->lowerValue()+" - "+(*it)->upperValue();
		QgsSymbol* sym=new QgsSymbol(mVectorLayer->vectorType(), (*it)->lowerValue(), (*it)->upperValue(), (*it)->label());
		sym->setPen((*it)->pen());
		sym->setBrush((*it)->brush());
		sym->setNamedPointSymbol((*it)->pointSymbolName());
		sym->setPointSize((*it)->pointSize());
		mEntries.insert(std::make_pair(classbreak,sym));
		mClassBreakBox->insertItem(classbreak);
	}
	
    }
    
    //do the necessary signal/slot connections
    QObject::connect(numberofclassesspinbox, SIGNAL(valueChanged(int)), this, SLOT(adjustClassification()));
    QObject::connect(classificationComboBox, SIGNAL(activated(int)), this, SLOT(adjustClassification()));
    QObject::connect(modeComboBox, SIGNAL(activated(int)), this, SLOT(adjustClassification()));
    QObject::connect(mClassBreakBox, SIGNAL(selectionChanged()), this, SLOT(changeCurrentValue()));
    QObject::connect(&sydialog, SIGNAL(settingsChanged()), this, SLOT(applySymbologyChanges()));
    QObject::connect(mClassBreakBox, SIGNAL(doubleClicked(Q3ListBoxItem*)), this, SLOT(changeClass(Q3ListBoxItem*)));

    mSymbolWidgetStack->addWidget(&sydialog);
    mSymbolWidgetStack->raiseWidget(&sydialog); 

    mClassBreakBox->setCurrentItem(0);
}

QgsGraduatedSymbolDialog::QgsGraduatedSymbolDialog(): QDialog(), mVectorLayer(0), sydialog(0)
{
    setupUi(this);
#ifdef QGISDEBUG
    qWarning("constructor QgsGraduatedSymbolDialog");
#endif
}

QgsGraduatedSymbolDialog::~QgsGraduatedSymbolDialog()
{
#ifdef QGISDEBUG
    qWarning("destructor QgsGraduatedSymbolDialog");
#endif
}

void QgsGraduatedSymbolDialog::adjustNumberOfClasses()
{
    //find out the number of the classification field
    QString fieldstring = classificationComboBox->currentText();
    
    if (fieldstring.isEmpty())    //don't do anything, it there is no classification field
    {
	show();
	return;
    }
    
    std::map < QString, int >::iterator iter = mFieldMap.find(fieldstring);
    int field = iter->second;
}

void QgsGraduatedSymbolDialog::apply()
{
	if (classificationComboBox->currentText().isEmpty())  //don't do anything, it there is no classification field
        {
	    return;
        }
	
	QgsGraduatedSymbolRenderer* renderer = new QgsGraduatedSymbolRenderer(mVectorLayer->vectorType());

	for (int item=0;item<mClassBreakBox->count();++item)
        {
	    QString classbreak=mClassBreakBox->text(item);
	    std::map<QString,QgsSymbol*>::iterator it=mEntries.find(classbreak);
	    if(it==mEntries.end())
	    {
		continue;
	    }
	
	    QString lower_bound=it->second->lowerValue();
	    QString upper_bound=it->second->upperValue();
	    QString label=it->second->label();

	    QgsSymbol* sy = new QgsSymbol(mVectorLayer->vectorType(), lower_bound, upper_bound, label);
	    
	    sy->setColor(it->second->pen().color());
	    sy->setLineStyle(it->second->pen().style());
	    sy->setLineWidth(it->second->pen().width());
	    
	    if (mVectorLayer->vectorType() == QGis::Point)
	    {
		sy->setNamedPointSymbol(it->second->pointSymbolName());
		sy->setPointSize(it->second->pointSize());
	     
	    }
	    
	    if (mVectorLayer->vectorType() != QGis::Line)
            {
		sy->setFillColor(it->second->brush().color());
		sy->setFillStyle(it->second->brush().style());
            }
	    
	    //test, if lower_bound is numeric or not (making a subclass of QString would be the proper solution)
	    bool lbcontainsletter = false;
	    for (uint j = 0; j < lower_bound.length(); j++)
            {
		if (lower_bound.ref(j).isLetter())
                {
		    lbcontainsletter = true;
                }
            }
	    
	    //test, if upper_bound is numeric or not (making a subclass of QString would be the proper solution)
	    bool ubcontainsletter = false;
	    for (uint j = 0; j < upper_bound.length(); j++)
            {
		if (upper_bound.ref(j).isLetter())
                {
		    ubcontainsletter = true;
                }
            }
	    if (lbcontainsletter == false && ubcontainsletter == false && lower_bound.length() > 0 && upper_bound.length() > 0) //only add the item if the value bounds do not contain letters and are not null strings
            {
		renderer->addSymbol(sy);
	    }
	    else
	    {
		delete sy;
	    }
        }
	
	std::map<QString,int>::iterator iter=mFieldMap.find(classificationComboBox->currentText());
	if(iter!=mFieldMap.end())
	{
	   renderer->setClassificationField(iter->second);
	}
	mVectorLayer->setRenderer(renderer);
	mVectorLayer->refreshLegend();
}

void QgsGraduatedSymbolDialog::adjustClassification()
{
    mClassBreakBox->clear();
    QGis::VectorType m_type = mVectorLayer->vectorType();
    QgsVectorDataProvider *provider = dynamic_cast<QgsVectorDataProvider *>(mVectorLayer->getDataProvider());
    double minimum, maximum;
    
    //delete all previous entries
    for(std::map<QString, QgsSymbol*>::iterator it=mEntries.begin();it!=mEntries.end();++it)
    {
	delete it->second;
    }
    mEntries.clear();

    //find out the number of the classification field
    QString fieldstring = classificationComboBox->currentText();

    if (fieldstring.isEmpty())    //don't do anything, it there is no classification field
    {
	show();
	return;
    }
    
    std::map < QString, int >::iterator iter = mFieldMap.find(fieldstring);
    int field = iter->second;

    if (provider)
    {
	if (modeComboBox->currentText() == "Equal Interval")
	{
	    minimum = provider->minValue(field).toDouble();
	    maximum = provider->maxValue(field).toDouble();
	} 
	else                    //don't waste performance if mMode is QgsGraduatedSymbolDialog::EMPTY
	{
	    minimum = 0;
	    maximum = 0;
	}
    }
 
    
    QString listboxtext;
    for(int i=0;i<numberofclassesspinbox->value();++i)
    {
	QgsSymbol* symbol = new QgsSymbol(m_type);
	symbol->setLabel("");
	QPen pen;
	QBrush brush;

	if (modeComboBox->currentText() == "Empty")
	{
	    listboxtext="Empty"+QString::number(i+1);
	    mClassBreakBox->insertItem(listboxtext);
	}
	else if(modeComboBox->currentText() == "Equal Interval")
	{
	    double lower=minimum + (maximum - minimum) / numberofclassesspinbox->value() * i;
	    double upper=minimum + (maximum - minimum) / numberofclassesspinbox->value() * (i+1);
	    if(i==0)//make sure all feature attributes are between minimum and maximum value (round off problem)
	      {
		lower-=0.001;
	      }
	    if(i==numberofclassesspinbox->value()-1)
	      {
		upper+=0.001;
	      }
	    symbol->setLowerValue(QString::number(lower,'f',3));
	    symbol->setUpperValue(QString::number(upper,'f',3));
	    listboxtext=QString::number(lower,'f',3)+" - " +QString::number(upper,'f',3);
	    mClassBreakBox->insertItem(listboxtext);
	}
	    //set default symbology

	    //apply a nice color range from red to green as default
	    if (i == 0)
	    {
		if (m_type == QGis::Line)
		{
		    pen.setColor(QColor(255, 0, 0));
		} 
		else                //point or polygon
		{
		    brush.setColor(QColor(255, 0, 0));
		    pen.setColor(Qt::black);
		}
	    } 
	    else
	    {
		if (m_type == QGis::Line)
		{
		  pen.setColor(QColor(255 - (255 / numberofclassesspinbox->value() * (i+1)), 255 / numberofclassesspinbox->value() * (i+1), 0));
		} 
		else                //point or polygon
		{
		  brush.setColor(QColor(255 - (255 / numberofclassesspinbox->value() * (i+1)), 255 / numberofclassesspinbox->value() * (i+1), 0));  
		  pen.setColor(Qt::black);
		}
	    }
	    pen.setWidth(1);
	    brush.setStyle(Qt::SolidPattern);
	    symbol->setPen(pen);
	    symbol->setBrush(brush);
       
	mEntries.insert(std::make_pair(listboxtext,symbol));
    }
    mClassBreakBox->setCurrentItem(0);
}

void QgsGraduatedSymbolDialog::changeCurrentValue()
{
    sydialog.blockSignals(true);//block signals to prevent sydialog from changing the current QgsRenderItem
    Q3ListBoxItem* item=mClassBreakBox->selectedItem();
    if(item)
    {
	QString value=item->text();
	std::map<QString,QgsSymbol*>::iterator it=mEntries.find(value);
	if(it!=mEntries.end())
	{
	    sydialog.set((*it).second);
	    sydialog.setLabel((*it).second->label());
	}
    }
    sydialog.blockSignals(false);
}

void QgsGraduatedSymbolDialog::applySymbologyChanges()
{
    Q3ListBoxItem* item=mClassBreakBox->selectedItem();
    if(item)
    {
	QString value=item->text();
	std::map<QString,QgsSymbol*>::iterator it=mEntries.find(value);
	if(it!=mEntries.end())
	{
	    sydialog.apply((*it).second);
	    it->second->setLabel((*it).second->label());
	}
    }
}

void QgsGraduatedSymbolDialog::changeClass(Q3ListBoxItem* item)
{
    QString currenttext=item->text();
    QgsSymbol* symbol=0;
    std::map<QString,QgsSymbol*>::iterator iter=mEntries.find(currenttext);
    if(iter!=mEntries.end())
    {
	symbol=iter->second;
    }
    QgsLUDialog dialog(this);
    
    if(symbol)
    {
	dialog.setLowerValue(symbol->lowerValue());
	dialog.setUpperValue(symbol->upperValue());
    }

    if(dialog.exec()==QDialog::Accepted)
    {
	if(symbol)
	{
	    mEntries.erase(currenttext);
	    symbol->setLowerValue(dialog.lowerValue());
	    symbol->setUpperValue(dialog.upperValue());
	    QString newclass=dialog.lowerValue()+"-"+dialog.upperValue();
	    mEntries.insert(std::make_pair(newclass,symbol));
	    int index=mClassBreakBox->index(item);
	    QObject::disconnect(mClassBreakBox, SIGNAL(selectionChanged()), this, SLOT(changeCurrentValue()));
	    mClassBreakBox->removeItem(index);
	    mClassBreakBox->insertItem(newclass,index);
	    mClassBreakBox->setSelected(index,true);
	    QObject::connect(mClassBreakBox, SIGNAL(selectionChanged()), this, SLOT(changeCurrentValue()));
	}	
    }
}
