
#include "qgssymbolv2propertiesdialog.h"

#include <QFile>
#include <QStandardItem>
#include <QKeyEvent>

#include "qgssymbollayerv2.h"
#include "qgssymbolv2.h"
#include "qgssymbollayerv2registry.h"

#include "qgsapplication.h"

#include "qgssymbollayerv2widget.h"

static const int SymbolLayerItemType = QStandardItem::UserType + 1;

class SymbolLayerItem : public QStandardItem
{
  public:
    SymbolLayerItem( QgsSymbolLayerV2* layer )
    {
      setLayer( layer );
    }

    void setLayer( QgsSymbolLayerV2* layer )
    {
      mLayer = layer;
      updatePreview();
    }

    void updatePreview()
    {
      QIcon icon = QgsSymbolLayerV2Utils::symbolLayerPreviewIcon( mLayer, QSize( 16, 16 ) );
      setIcon( icon );
    }

    int type() const { return SymbolLayerItemType; }

    QVariant data( int role ) const
    {
      if ( role == Qt::DisplayRole )
        return QVariant( mLayer->layerType() );
      if ( role == Qt::SizeHintRole )
        return QVariant( QSize( 32, 32 ) );
      if ( role == Qt::CheckStateRole )
        return QVariant(); // could be true/false
      return QStandardItem::data( role );
    }

  protected:
    QgsSymbolLayerV2* mLayer;
};

//////////

static QString iconPath( QString iconFile )
{
  // try active theme
  QString path = QgsApplication::activeThemePath();
  if ( QFile::exists( path + iconFile ) )
    return path + iconFile;

  // use default theme
  return QgsApplication::defaultThemePath() + iconFile;
}

//////////

QgsSymbolV2PropertiesDialog::QgsSymbolV2PropertiesDialog( QgsSymbolV2* symbol, QWidget* parent )
    : QDialog( parent ), mSymbol( symbol )
{
  setupUi( this );

  // setup icons
  btnAddLayer->setIcon( QIcon( iconPath( "symbologyAdd.png" ) ) );
  btnRemoveLayer->setIcon( QIcon( iconPath( "symbologyRemove.png" ) ) );
  btnLock->setIcon( QIcon( iconPath( "symbologyLock.png" ) ) );
  btnUp->setIcon( QIcon( iconPath( "symbologyUp.png" ) ) );
  btnDown->setIcon( QIcon( iconPath( "symbologyDown.png" ) ) );

  // set widget functions
  // (should be probably moved somewhere else)
  QgsSymbolLayerV2Registry::instance()->setLayerTypeWidgetFunction( "SimpleLine", QgsSimpleLineSymbolLayerV2Widget::create );
  QgsSymbolLayerV2Registry::instance()->setLayerTypeWidgetFunction( "MarkerLine", QgsMarkerLineSymbolLayerV2Widget::create );
  QgsSymbolLayerV2Registry::instance()->setLayerTypeWidgetFunction( "LineDecoration", QgsLineDecorationSymbolLayerV2Widget::create );

  QgsSymbolLayerV2Registry::instance()->setLayerTypeWidgetFunction( "SimpleMarker", QgsSimpleMarkerSymbolLayerV2Widget::create );
  QgsSymbolLayerV2Registry::instance()->setLayerTypeWidgetFunction( "SvgMarker", QgsSvgMarkerSymbolLayerV2Widget::create );

  QgsSymbolLayerV2Registry::instance()->setLayerTypeWidgetFunction( "SimpleFill", QgsSimpleFillSymbolLayerV2Widget::create );

  loadSymbol();

  connect( btnUp, SIGNAL( clicked() ), this, SLOT( moveLayerUp() ) );
  connect( btnDown, SIGNAL( clicked() ), this, SLOT( moveLayerDown() ) );
  connect( btnAddLayer, SIGNAL( clicked() ), this, SLOT( addLayer() ) );
  connect( btnRemoveLayer, SIGNAL( clicked() ), this, SLOT( removeLayer() ) );
  connect( btnLock, SIGNAL( clicked() ), this, SLOT( lockLayer() ) );

  populateLayerTypes();
  connect( cboLayerType, SIGNAL( currentIndexChanged( int ) ), this, SLOT( layerTypeChanged() ) );

  loadPropertyWidgets();

  updateUi();

  // set first layer as active
  QModelIndex newIndex = listLayers->model()->index( 0, 0 );
  listLayers->setCurrentIndex( newIndex );
}


void QgsSymbolV2PropertiesDialog::loadSymbol()
{
  QStandardItemModel* model = new QStandardItemModel( this );
  listLayers->setModel( model );

  QItemSelectionModel* selModel = listLayers->selectionModel();
  connect( selModel, SIGNAL( currentChanged( const QModelIndex&, const QModelIndex& ) ), this, SLOT( layerChanged() ) );

  int count = mSymbol->symbolLayerCount();
  for ( int i = count - 1; i >= 0; i-- )
  {
    model->appendRow( new SymbolLayerItem( mSymbol->symbolLayer( i ) ) );
  }

  updatePreview();
}


void QgsSymbolV2PropertiesDialog::populateLayerTypes()
{
  QStringList types = QgsSymbolLayerV2Registry::instance()->symbolLayersForType( mSymbol->type() );

  cboLayerType->clear();
  for ( int i = 0; i < types.count(); i++ )
    cboLayerType->addItem( types[i] );
}


void QgsSymbolV2PropertiesDialog::updateUi()
{
  int row = currentRowIndex();
  btnUp->setEnabled( row > 0 );
  btnDown->setEnabled( row < listLayers->model()->rowCount() - 1 && row != -1 );
  btnRemoveLayer->setEnabled( row != -1 );
}

void QgsSymbolV2PropertiesDialog::updatePreview()
{
  QImage preview = mSymbol->bigSymbolPreviewImage();
  lblPreview->setPixmap( QPixmap::fromImage( preview ) );
}

void QgsSymbolV2PropertiesDialog::updateLayerPreview()
{
  // get current layer item and update its icon
  SymbolLayerItem* item = currentLayerItem();
  if ( item )
    item->updatePreview();

  // update also preview of the whole symbol
  updatePreview();
}

void QgsSymbolV2PropertiesDialog::updateSymbolLayerWidget( QgsSymbolLayerV2* layer )
{
  QString layerType = layer->layerType();

  // stop updating from the original widget
  if ( stackedWidget->currentWidget() != pageDummy )
    disconnect( stackedWidget->currentWidget(), SIGNAL( changed() ), this, SLOT( updateLayerPreview() ) );

  // update active properties widget
  if ( mWidgets.contains( layerType ) )
  {
    stackedWidget->setCurrentWidget( mWidgets[layerType] );
    mWidgets[layerType]->setSymbolLayer( layer );

    // start recieving updates from widget
    connect( mWidgets[layerType], SIGNAL( changed() ), this, SLOT( updateLayerPreview() ) );
  }
  else
  {
    // use dummy widget instead
    stackedWidget->setCurrentWidget( pageDummy );
  }
}

void QgsSymbolV2PropertiesDialog::loadPropertyWidgets()
{
  QgsSymbolLayerV2Registry* pReg = QgsSymbolLayerV2Registry::instance();

  QStringList layerTypes = pReg->symbolLayersForType( mSymbol->type() );

  for ( int i = 0; i < layerTypes.count(); i++ )
  {
    QString layerType = layerTypes[i];
    QgsSymbolLayerV2WidgetFunc f = pReg->symbolLayerMetadata( layerType ).widgetFunction();
    if ( f == NULL ) // check whether the function is assigned
      continue;

    QgsSymbolLayerV2Widget* w = f();
    if ( w == NULL ) // check whether the function returns correct widget
      continue;

    mWidgets[layerType] = w;
    stackedWidget->addWidget( w );
  }
}

int QgsSymbolV2PropertiesDialog::currentRowIndex()
{
  QModelIndex idx = listLayers->selectionModel()->currentIndex();
  if ( !idx.isValid() )
    return -1;
  return idx.row();
}

int QgsSymbolV2PropertiesDialog::currentLayerIndex()
{
  return listLayers->model()->rowCount() - currentRowIndex() - 1;
}

SymbolLayerItem* QgsSymbolV2PropertiesDialog::currentLayerItem()
{
  int index = currentRowIndex();
  if ( index < 0 )
    return NULL;

  QStandardItemModel* model = qobject_cast<QStandardItemModel*>( listLayers->model() );
  if ( model == NULL )
    return NULL;
  QStandardItem* item = model->item( index );
  if ( item->type() != SymbolLayerItemType )
    return NULL;
  return static_cast<SymbolLayerItem*>( item );
}

QgsSymbolLayerV2* QgsSymbolV2PropertiesDialog::currentLayer()
{
  int idx = currentLayerIndex();
  if ( idx < 0 )
    return NULL;

  return mSymbol->symbolLayer( idx );
}


void QgsSymbolV2PropertiesDialog::layerChanged()
{
  updateUi();

  // get layer info
  QgsSymbolLayerV2* layer = currentLayer();
  if ( layer == NULL )
    return;

  // update layer type combo box
  int idx = cboLayerType->findText( layer->layerType() );
  cboLayerType->setCurrentIndex( idx );

  updateSymbolLayerWidget( layer );

  updateLockButton();
}


void QgsSymbolV2PropertiesDialog::updateLockButton()
{
  QgsSymbolLayerV2* layer = currentLayer();
  if ( layer == NULL ) return;

  btnLock->setChecked( layer->isLocked() );
}


void QgsSymbolV2PropertiesDialog::layerTypeChanged()
{
  QgsSymbolLayerV2* layer = currentLayer();
  if ( layer == NULL ) return;

  QString newLayerType = cboLayerType->currentText();
  if ( layer->layerType() == newLayerType )
    return;

  // get creation function for new layer from registry
  QgsSymbolLayerV2Registry* pReg = QgsSymbolLayerV2Registry::instance();
  QgsSymbolLayerV2CreateFunc f = pReg->symbolLayerMetadata( newLayerType ).createFunction();
  if ( f == NULL ) // check whether the function is assigned
    return;

  // change layer to a new (with different type)
  QgsSymbolLayerV2* newLayer = f( QgsStringMap() );
  mSymbol->changeSymbolLayer( currentLayerIndex(), newLayer );

  updateSymbolLayerWidget( newLayer );

  // update symbol layer item
  SymbolLayerItem* item = currentLayerItem();
  item->setLayer( newLayer );
  item->updatePreview();

  updatePreview();
}


void QgsSymbolV2PropertiesDialog::addLayer()
{
  QgsSymbolLayerV2* newLayer = QgsSymbolLayerV2Registry::instance()->defaultSymbolLayer( mSymbol->type() );

  mSymbol->appendSymbolLayer( newLayer );

  loadSymbol();

  QModelIndex newIndex = listLayers->model()->index( 0, 0 );
  listLayers->setCurrentIndex( newIndex );

  updateUi();
}


void QgsSymbolV2PropertiesDialog::removeLayer()
{
  int idx = currentLayerIndex();
  if ( idx < 0 ) return;
  mSymbol->deleteSymbolLayer( idx );

  loadSymbol();

  updateUi();
}


void QgsSymbolV2PropertiesDialog::moveLayerDown()
{
  moveLayerByOffset( +1 );
}

void QgsSymbolV2PropertiesDialog::moveLayerUp()
{
  moveLayerByOffset( -1 );
}

void QgsSymbolV2PropertiesDialog::moveLayerByOffset( int offset )
{
  int rowIdx = currentRowIndex();
  int layerIdx = currentLayerIndex();

  // switch layers
  QgsSymbolLayerV2* tmpLayer = mSymbol->takeSymbolLayer( layerIdx );
  mSymbol->insertSymbolLayer( layerIdx - offset, tmpLayer );

  loadSymbol();

  QModelIndex newIndex = listLayers->model()->index( rowIdx + offset, 0 );
  listLayers->setCurrentIndex( newIndex );

  updateUi();
}


void QgsSymbolV2PropertiesDialog::lockLayer()
{
  QgsSymbolLayerV2* layer = currentLayer();
  if ( layer == NULL ) return;

  layer->setLocked( btnLock->isChecked() );
}

#include "qgslogger.h"

void QgsSymbolV2PropertiesDialog::keyPressEvent( QKeyEvent * e )
{
  // Ignore the ESC key to avoid close the dialog without the properties window
  if ( !isWindow() && e->key() == Qt::Key_Escape )
  {
    e->ignore();
  }
  else
  {
    QDialog::keyPressEvent( e );
  }
}
