/*
 * QCameraInterpolationTool.cpp
 * $Id: 
 *
 * Copyright (C) 2001 Thomas Woerner, Michael Meissner, Markus Janich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */


// Qt
///////
#include <qdom.h>
#include <qfile.h>
#include <qfiledialog.h>
#include <qpushbutton.h>
#include <qtextstream.h>
#include <qmessagebox.h>
#include <qfiledialog.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qgl.h>

// QGLViewer
//////////////
#include "QCameraDrag.h"
#include "QGLViewerXML.h"
#include "QGLViewerIO.h"
#include "QCameraInterpolationTool.h"
#include "CCameraPathInterpolator.h"



// Function  : QCameraInterpolationTool
// Parameters: QWidget* parent, const char* name, bool modal, WFlags fl
// Purpose   : default constructor
// Comments  : 
QCameraInterpolationTool::QCameraInterpolationTool(QGLViewer *pQGLViewer,
						   QWidget* parent, const char* name,
						   bool modal, WFlags fl)
    : QDialog(parent, name, modal, fl),
      m_pQGLViewer(pQGLViewer)
/**************************************************************/
{
   resize(200,150);

   if (!name)
      setName("QCameraInterpolationTool");
   setCaption(tr("CameraPathInterpolator"));
   setSizeGripEnabled(TRUE);
   setSizePolicy(QSizePolicy((QSizePolicy::SizeType)7, 
                             (QSizePolicy::SizeType)7,
                             sizePolicy().hasHeightForWidth()));

   QGridLayout *pLayout = new QGridLayout(this, 5, 3, 10, 10);
   pLayout->setSpacing(10);
   pLayout->setMargin(10);

   QMenuBar *pMenuBar = new QMenuBar(this, "menu");
   Q_CHECK_PTR(pMenuBar);

   QPopupMenu *pMenu = new QPopupMenu();
   Q_CHECK_PTR(pMenu);
   pMenu->insertItem("Load key camera path", this, SLOT(sltKeyLoad()), CTRL+Key_L);
   pMenu->insertItem("Save key camera path", this, SLOT(sltKeySave()), CTRL+Key_S);
   pMenu->insertItem("Save camera path", this, SLOT(sltSave()), CTRL+Key_P);
   pMenu->insertItem("Save camera path as key camera path", this, SLOT(sltShapeSave()), CTRL+Key_K);
   pMenuBar->insertItem("&File", pMenu);

   m_pOptions = new QIconOptions(this, tr("Icon Options"));
   connect(m_pOptions, SIGNAL(sigArrowChanged(double)),
           this, SLOT(sltArrowSize(double)));
   connect(m_pOptions, SIGNAL(sigFrustumChanged(double)),
           this, SLOT(sltFrustumSize(double)));

   pMenu = new QPopupMenu();
   Q_CHECK_PTR(pMenu);
   pMenu->insertItem("Icon Size", this, SLOT(sltShowOptions()), CTRL+Key_O);
   pMenuBar->insertItem("&Options", pMenu);

   pLayout->setMenuBar(pMenuBar);

   QLabel *pLabel = new QLabel(this, "algoLabel");
   pLabel->setText(tr("Algorithm"));
   pLayout->addWidget(pLabel, 0, 0);

   m_pAlgoCombo = new QComboBox(FALSE, this, "algoCombo");
   m_pAlgoCombo->insertItem(QString("Line"), CCameraPathInterpolator::SHAPE_LINE);
   m_pAlgoCombo->insertItem(QString("TCB"), CCameraPathInterpolator::SHAPE_TCB);
   m_pAlgoCombo->insertItem(QString("Catmull-Rom"), CCameraPathInterpolator::SHAPE_CATMULL_ROM);
   pLayout->addWidget(m_pAlgoCombo, 0, 1);

   pLabel = new QLabel(this, "pathLabel");
   pLabel->setText(tr("Path"));
   pLayout->addWidget(pLabel, 1, 0);

   m_pPathCombo = new QComboBox(FALSE, this, "pathCombo");
   m_pPathCombo->insertItem(QString("open"), CCameraPathInterpolator::PATH_OPEN);
   m_pPathCombo->insertItem(QString("closed"), CCameraPathInterpolator::PATH_CLOSED);
   pLayout->addWidget(m_pPathCombo, 1, 1);

   pLabel = new QLabel(this, "keysLabel");
   pLabel->setText(tr("Key camera path points"));
   pLayout->addWidget(pLabel, 2, 0);

   m_pKeysLine = new QLineEdit(this, "keysLine");
   m_pKeysLine->setText("0");
   m_pKeysLine->setReadOnly(true);
   pLayout->addWidget(m_pKeysLine, 2, 1);

   m_pKeyDropSite = new QCameraKeyPathDropSite(this, "keyDropSite");
   m_pKeyDropSite->setCameraPath(&m_KeyPathList);
   pLayout->addWidget(m_pKeyDropSite, 2, 2);
   connect(m_pKeyDropSite, SIGNAL(sigCameraKeyPathDropped(const CList<CCameraKeyPathPoint> &)),
	   this, SLOT(sltSetCameraKeyPath(const CList<CCameraKeyPathPoint> &)));

   pLabel = new QLabel(this, "framesLabel");
   pLabel->setText(tr("Camera path points"));
   pLayout->addWidget(pLabel, 3, 0);

   m_pFramesLine = new QLineEdit(this, "framesLine");
   m_pFramesLine->setText("0");
   m_pFramesLine->setReadOnly(true);
   pLayout->addWidget(m_pFramesLine, 3, 1);

   m_pDropSite = new QCameraPathDropSite(this, "dropSite");
   m_pDropSite->setEnabled(false);
   pLayout->addWidget(m_pDropSite, 3, 2);
   connect(m_pDropSite, SIGNAL(sigCameraPathDropped(const CList<CCamera> &)),
	   this, SLOT(sltSetCameraPath(const CList<CCamera> &)));

   QHBoxLayout *pBoxLayout = new QHBoxLayout();
   pBoxLayout->setSpacing(5);
   pBoxLayout->setMargin(0);
   
   m_pInterpolateButton = new QPushButton(this, "interpolateButton");
   m_pInterpolateButton->setText(tr("Interpolate"));
   m_pInterpolateButton->setEnabled(false);
   connect(m_pInterpolateButton, SIGNAL(clicked()),
           this, SLOT(sltInterpolate()));
   pBoxLayout->addWidget(m_pInterpolateButton);

   m_pUndoButton = new QPushButton(this, "undoButton");
   m_pUndoButton->setText(tr("Undo"));
   m_pUndoButton->setEnabled(false);
   connect(m_pUndoButton, SIGNAL(clicked()), this, SLOT(sltUndo()));
   pBoxLayout->addWidget(m_pUndoButton);
    
   pLayout->addMultiCellLayout(pBoxLayout, 4, 4, 0, 2);

   setAcceptDrops(TRUE);

   m_rfIconSize = 1.0;
   m_rfFrustumSize = 1.0;

   // create display list for a arrow
   ////////////////////////////////////
   pQGLViewer->makeCurrent();
   makeArrowDispList();
}



void QCameraInterpolationTool::makeArrowDispList() 
/**************************************************************/
{
  int nSteps = 20;
  float rfX,rfY,rfZ;
  float rfAngle = 2*M_PI/ nSteps;
	
  if (glIsList(m_glArrowDispList))
    glDeleteLists(m_glArrowDispList, 1);
  m_glArrowDispList = glGenLists(1);

  glNewList(m_glArrowDispList, GL_COMPILE);
  //draw Vector
  ////////////////
  glBegin(GL_LINES);
  glLineWidth(m_rfIconSize);
  glVertex3f(0,0,0);
  glVertex3f(0,0,m_rfIconSize);
  glEnd();
	
  glBegin(GL_TRIANGLE_FAN);
  glVertex3f(0,0,m_rfIconSize);
  rfZ = m_rfIconSize;
  for (int l = 0; l<nSteps+1; l++) {
    rfX = cos(l*rfAngle);
    rfY = sin(l*rfAngle);
    glVertex3f(0.1*m_rfIconSize*rfX,0.1*m_rfIconSize*rfY,rfZ);
  }
  glEnd();
	
  glBegin(GL_TRIANGLE_FAN);
  glVertex3f(0,0,1.2*m_rfIconSize);
  rfZ = m_rfIconSize;
  for (int m = 0; m<nSteps+1; m++) {
    rfX = cos(m*rfAngle);
    rfY = sin(m*rfAngle);
    glVertex3f(0.1*m_rfIconSize*rfX,0.1*m_rfIconSize*rfY,rfZ);
  }
  glEnd();
  glEndList();
}


void QCameraInterpolationTool::makePathDispList() 
/**************************************************************/
{
  if (glIsList(m_glPathDispList))
    glDeleteLists(m_glPathDispList, 1);
  m_glPathDispList = glGenLists(1);

  glNewList(m_glPathDispList, GL_COMPILE);
  // save lighting state
  /////////////////////////
  glPushAttrib(GL_LIGHTING);
  glDisable(GL_LIGHTING);

  //draw path
  /////////////
  glColor3f(1.0, 1.0, 1.0);
  glBegin(GL_LINE_STRIP);
  glLineWidth(1.0);
  for (int k=0; k<m_ShapePathList.getNumObjects(); k++) {
    glVertex3f(m_ShapePathList[k].getCamera().getEyePos().getX(),
	       m_ShapePathList[k].getCamera().getEyePos().getY(),
	       m_ShapePathList[k].getCamera().getEyePos().getZ());
  }
  glEnd();
    
  for (int i=0; i<m_ShapePathList.getNumObjects(); i++) {
    CP3D Eye(m_ShapePathList[i].getCamera().getEyePos());
    CV3D ViewVector(m_ShapePathList[i].getCamera().getViewDir());
    CV3D UpVector(m_ShapePathList[i].getCamera().getViewUp());
    CV3D CrossVector = -(UpVector | ViewVector);
	
    glPushMatrix();

    float m[16];
    m[0] = CrossVector.getX(); m[4] = UpVector.getX();  m[8] = ViewVector.getX();  m[12] = Eye.getX(); 
    m[1] = CrossVector.getY(); m[5] = UpVector.getY();  m[9] = ViewVector.getY();  m[13] = Eye.getY(); 
    m[2] = CrossVector.getZ(); m[6] = UpVector.getZ();  m[10] = ViewVector.getZ(); m[14] = Eye.getZ(); 
    m[3] = 0.0;                m[7] = 0.0;              m[11] = 0.0;               m[15] = 1.0; 

    glMultMatrixf(m);

    glColor3f(1.0,0.0,0.0);                  //ViewVector
    glCallList(m_glArrowDispList);

    glColor3f(0.0,1.0,1.0);

    //draw frustum
    ////////////////
    glBegin(GL_TRIANGLE_FAN);
    glVertex3f(0,0,0);
	
    double rdVAngleDeg,rdHAngleDeg;
    double rdVAngleRad,rdHAngleRad;
    double rdRatio;
	
    rdVAngleDeg = ((m_ShapePathList[i].getCamera().getFovy())/2);
    rdRatio = m_ShapePathList[i].getCamera().getRatio();
    rdHAngleDeg = rdRatio * rdVAngleDeg;
    rdVAngleRad = rdVAngleDeg * M_PI / 180;
    rdHAngleRad = rdHAngleDeg * M_PI / 180;
	
    glVertex3f(sin(rdHAngleRad)*m_rfFrustumSize,
	       sin(rdVAngleRad)*m_rfFrustumSize,
	       cos(rdVAngleRad)*m_rfFrustumSize);
    glVertex3f(-sin(rdHAngleRad)*m_rfFrustumSize,
	       sin(rdVAngleRad)*m_rfFrustumSize,
	       cos(rdVAngleRad)*m_rfFrustumSize);
    glVertex3f(-sin(rdHAngleRad)*m_rfFrustumSize,
	       -sin(rdVAngleRad)*m_rfFrustumSize,
	       cos(rdVAngleRad)*m_rfFrustumSize);
    glVertex3f(sin(rdHAngleRad)*m_rfFrustumSize,
	       -sin(rdVAngleRad)*m_rfFrustumSize,
	       cos(rdVAngleRad)*m_rfFrustumSize);
    glVertex3f(sin(rdHAngleRad)*m_rfFrustumSize,
	       sin(rdVAngleRad)*m_rfFrustumSize,
	       cos(rdVAngleRad)*m_rfFrustumSize);
    glEnd();
	
    glBegin(GL_LINES);
    glColor3f(0.7,0.7,0.7);
    glVertex3f(0.0,0.0,0.0);
    glVertex3f(sin(rdHAngleRad)*m_rfFrustumSize,
	       sin(rdVAngleRad)*m_rfFrustumSize,
	       cos(rdVAngleRad)*m_rfFrustumSize);
    glEnd();
	
    glBegin(GL_LINES);
    glVertex3f(0.0,0.0,0.0);
    glVertex3f(-sin(rdHAngleRad)*m_rfFrustumSize,
	       sin(rdVAngleRad)*m_rfFrustumSize,
	       cos(rdVAngleRad)*m_rfFrustumSize);
    glEnd();
      
    glBegin(GL_LINES);
    glVertex3f(0.0,0.0,0.0);
    glVertex3f(-sin(rdHAngleRad)*m_rfFrustumSize,
	       -sin(rdVAngleRad)*m_rfFrustumSize,
	       cos(rdVAngleRad)*m_rfFrustumSize);
    glEnd();
	
    glBegin(GL_LINES);
    glVertex3f(0.0,0.0,0.0);
    glVertex3f(sin(rdHAngleRad)*m_rfFrustumSize,
	       -sin(rdVAngleRad)*m_rfFrustumSize,
	       cos(rdVAngleRad)*m_rfFrustumSize);
    glEnd();
	
    glPushMatrix();
    glRotatef(-90,1,0,0);
    glColor3f(0.0,1.0,0.0);                  //UpVector
    glCallList(m_glArrowDispList);
    glPopMatrix();
	
    glPushMatrix();
    glRotatef(90,0,1,0);
    glColor3f(0.0,0.0,1.0);                  //CrossVector
    glCallList(m_glArrowDispList);
    glPopMatrix();
      
    glPopMatrix();
  }
  // restore lighting state
  ///////////////////////////
  glPopAttrib();

  glEndList();
}


void QCameraInterpolationTool::updateBoundingBox() 
/**************************************************************/
{
   CBoundingBox3D box =  m_pQGLViewer->getCameraPtr()->getBoundingBox();

   for (int i=0; i<m_ShapePathList.getNumObjects(); i++)
      box.addPoint(CP3D(m_ShapePathList[i].getCamera().getEyePos().getX(),
			m_ShapePathList[i].getCamera().getEyePos().getY(),
			m_ShapePathList[i].getCamera().getEyePos().getZ()));

   m_pQGLViewer->setBoundingBox(box, false);
}


void QCameraInterpolationTool::sltKeyLoad()
/**************************************************************/
{
   // load XML file to keys
   QString fileName = QFileDialog::getOpenFileName(QString::null, "*.xml", this);
   if (QFile::exists(fileName)) {
      if (!fileName.isEmpty()) {
         QFile file(fileName);
         if (!QGLViewerIO::read(file, m_KeyPathList)) {
            cout << "Can not open file" << endl;
            return;
         } 
         else {
            if (m_KeyPathList.getNumObjects() > 0) {
	      m_pKeyDropSite->setCameraPath(&m_KeyPathList);
	      m_pKeyDropSite->setEnabled(true);

	      sltUndo();
            }
         }
      }
   }
}



void QCameraInterpolationTool::sltKeySave()
/**************************************************************/
{
    if (m_KeyPathList.getNumObjects() > 0) {
	QString fileName = QFileDialog::getSaveFileName(QString::null, "*.xml",
						     this);
	if (!fileName.isEmpty()) {
	    if (QFile::exists(fileName)) {
		int res = QMessageBox::warning(this, "Warning",
					       "File "+fileName+" exists.\n\n"
					       "Overwrite?\n",
					       "Yes", "No", 0, 0, 1);
		if (res == 1)
		    return;
	    }
	    
	    QFile file(fileName);
	    if (! QGLViewerIO::write(file, m_KeyPathList))
		cout << "Can not open file" << endl;
	}
    }
}



void QCameraInterpolationTool::sltShapeSave()
/**************************************************************/
{
   if (m_ShapePathList.getNumObjects() > 0) {
      QString fileName = QFileDialog::getSaveFileName(QString::null, "*.xml",
						     this);
      if (!fileName.isEmpty()) {
         if (QFile::exists(fileName)) {
            int res = QMessageBox::warning(this, "Warning",
                                           "File "+fileName+" exists.\n\n"
                                           "Overwrite?\n",
                                           "Yes", "No", 0, 0, 1);
            if (res == 1)
               return;
         }
	    
         QFile file(fileName);
         if (! QGLViewerIO::write(file, m_ShapePathList))
            cout << "Can not open file" << endl;
      }
   }
}



void QCameraInterpolationTool::sltSave()
/**************************************************************/
{
   if (m_ShapePathList.getNumObjects() > 0) {
      // save keys to XML file
      QString fileName = QFileDialog::getSaveFileName(QString::null, "*.xml",
						     this);
      if (QFile::exists(fileName)) {
         int res = QMessageBox::warning(this, "Warning",
                                        "File "+fileName+" exists.\n\n"
                                        "Overwrite?\n",
                                        "Yes", "No", 0, 0, 1);
         if (res == 1)
            return;
      }
      CList<CCamera> CameraList;
      for (int i=0; i<m_ShapePathList.getNumObjects(); i++)
         CameraList.insertAsLast(new CCamera(m_ShapePathList[i].getCamera()));

      QDomDocument doc("save CCameraKeyPath");
      QDomElement elem = doc.createElement("CameraKeyPath");
      doc.appendChild(elem);
	
      if (!fileName.isEmpty()) {
         // save keys to XML file
         if (QGLViewerXML::writeXML(elem, CameraList, "CameraPath")) {
		
            QFile qOutFile(fileName);
            if (!qOutFile.open(IO_WriteOnly))
               cout << "Can not open file" << endl;
		
            QTextStream qStream(&qOutFile);
            doc.firstChild().save(qStream, 0);
            qOutFile.close();
      
         }
      }
	
      CameraList.clear(1);
   }
}



void QCameraInterpolationTool::sltInterpolate()
/**************************************************************/
{
   // create interpolator and calculate 
   CCameraPathInterpolator interpolator(m_KeyPathList, 
                                        static_cast<CCameraPathInterpolator::ShapeType>(m_pAlgoCombo->currentItem()),
                                        static_cast<CCameraPathInterpolator::PathType>(m_pPathCombo->currentItem()));

   // replace keys with result from calculation
   int i;
   m_ShapePathList.clear(1);
   for (i=0; i<interpolator.getNumFrames(); i++)
      m_ShapePathList.insertAsLast(interpolator.getFrame(i));

   m_pFramesLine->setText(QString::number(m_ShapePathList.getNumObjects()));

   makePathDispList();
   updateBoundingBox();

   m_pAlgoCombo->setEnabled(false);
   m_pPathCombo->setEnabled(false);

   m_pInterpolateButton->setEnabled(false);
   m_pUndoButton->setEnabled(true);

   m_CameraList.clear(1);
   for (i=0; i<m_ShapePathList.getNumObjects(); i++)
      m_CameraList.insertAsLast(new CCamera(m_ShapePathList[i].getCamera()));

   m_pDropSite->setCameraPath(&m_CameraList);
   m_pDropSite->setEnabled(true);

   emit(sigCameraPathChanged(m_CameraList));
   emit(sigRedraw());
}



void QCameraInterpolationTool::sltUndo()
/**************************************************************/
{
   m_pKeysLine->setText(QString::number(m_KeyPathList.getNumObjects()));
   m_pFramesLine->setText("0");

   m_pAlgoCombo->setEnabled(true);
   m_pPathCombo->setEnabled(true);

   m_ShapePathList.clear(1);
   m_ShapePathList = *m_KeyPathList.getFullDuplicate();

   makePathDispList();

   updateBoundingBox();
   // redraw
   emit(sigRedraw());

   m_pDropSite->setEnabled(false);
   m_pInterpolateButton->setEnabled(true);
   m_pUndoButton->setEnabled(false);    
}



void QCameraInterpolationTool::draw()
/**************************************************************/
{
  if (m_ViewingMode != hidden)
    if (glIsList(m_glPathDispList))
      glCallList(m_glPathDispList);
}



void QCameraInterpolationTool::sltShowOptions()
/**************************************************************/
{
   m_pOptions->show();
}



void QCameraInterpolationTool::sltArrowSize(double d)
/**************************************************************/
{
   m_rfIconSize = d;
   makeArrowDispList();
   emit(sigRedraw());
}



void QCameraInterpolationTool::sltFrustumSize(double d)
/**************************************************************/
{
   m_rfFrustumSize = d;
   makePathDispList();
   emit(sigRedraw());
}



// Function  : dragEnterEvent
// Parameters: QDragEnterEvent *event
// Purpose   : handles QDragEnterEvents.
// Comments  : See docu for details.
void QCameraInterpolationTool::dragEnterEvent(QDragEnterEvent *event)
/**************************************************************/
{
   if (QCameraKeyPathDrag::canDecode(event))
      event->accept();

   if (QCameraPathDrag::canDecode(event))
      event->accept();
}



// Function  : sltSetCameraPath
// Parameters: const CList<CCamera> &list
// Purpose   : Sets the camera path
// Comments  : See docu for details.
void QCameraInterpolationTool::sltSetCameraPath(const CList<CCamera> &list)
/**************************************************************/
{
  m_CameraList.clear(1);
  m_CameraList = *list.getFullDuplicate();

  sltUndo();
}



// Function  : sltSetCameraKeyPath
// Parameters: const CList<CCameraKeyPathPoint> &path
// Purpose   : Sets the camera key path
// Comments  : See docu for details.
void QCameraInterpolationTool::sltSetCameraKeyPath(const CList<CCameraKeyPathPoint> &path)
/**************************************************************/
{
  m_KeyPathList.clear(1);  
  m_KeyPathList = *path.getFullDuplicate();
  
  sltUndo();
}



// Function  : dropEvent
// Parameters: QDropEvent *event
// Purpose   : handles QDropEvents.
// Comments  : See docu for details.
void QCameraInterpolationTool::dropEvent(QDropEvent *event)
/**************************************************************/
{
   CList<CCamera> camPath;
   CList<CCameraKeyPathPoint> keyPath;

   if (event->source() != m_pKeyDropSite) {
      if (QCameraKeyPathDrag::decode(event, keyPath))
         sltSetCameraKeyPath(keyPath);

      if (QCameraPathDrag::decode(event, camPath)) {
	 sltSetCameraPath(camPath);
      }
   }
}
