/*
 * CLightSource.h
 * $Id: CLightSource.h,v 1.9 2001/11/23 02:03:35 mjanich Exp $
 *
 * Copyright (C) 1999, 2000 Michael Meissner, Michael Guthe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Definition of the CLightSource class 
//  Purpose     : Managment of class providing the interface to a linked
//                list of objects.


#ifndef __CLIGHTSOURCE_H
#define __CLIGHTSOURCE_H


// Own
////////
#include "CV3D.h"
#include "CP3D.h"
#include "GeoGeneric.h"


// defines
////////////


/** This class defines a light source with different
  * attributes like type, color, ...
  *
  * @author Michael Meissner, Michael Guthe */

class CLightSource
/*************************/
{
  public:
      enum LightStatus {
	 ON,
	 OFF
      };

      enum LightSourceType {
         DIRECTIONAL,
	 POINT,
	 CONE,
	 CONE_DIFFUSE
      };

    //constructors

      /** Default constructor.
	* Default type is a point light source  which
	* is switched on and emits white color light
	* on full intensity. */
      CLightSource() 
        : m_LightSourceType(POINT),
	  m_LightStatus(ON),
	  m_Point(CP3D(0,0,0)),
	  m_Direction(CV3D(0,0,-1)),
	  m_rfIntensity(1.0),
          m_rfAngle(180.0),
          m_rfExponent(0.0),
          m_rfConstantAttenuation(1.0),
          m_rfLinearAttenuation(0.0),
          m_rfQuadraticAttenuation(0.0)

      {
         setColor(1.0, 1.0, 1.0);
      };

      /** Copy Constructor. */
      CLightSource(const CLightSource &);

      /** Constructor defining the type. */
      CLightSource(LightSourceType nType);

      /** Constructor defining a directional lightsource. */
      CLightSource(const CV3D &direction, float rfIntensity=1.0);

      /** Constructor defining a point lightsource. */
      CLightSource(const CP3D &point, float rfIntensity=1.0);

      /** Constructor defining a cone lightsource. */
      CLightSource(const CP3D &point, const CV3D &direction, 
		   float rfAngle, float rfIntensity=1.0);

      /** Constructor defining a diffuse cone lightsource. */
      CLightSource(const CP3D &point, const CV3D &direction, float rfAngle, 
	           float rfExponent=50.0, float rfIntensity=1.0);

     // the default copy constructor should be ok, also the = operator

    //destructors
      virtual ~CLightSource() { ;};

  
    //methods
      /** Returns the type of the lightsource. */
      LightSourceType getType() const { return m_LightSourceType; };

      /** Returns the status of the light. */
      LightStatus getStatus() const { return m_LightStatus; };

      /** Returns the color of the light. */
      const float* getColor() const { return m_arfColor; };

      /** Returns the position of the light. */
      const CP3D &getPosition() const { return m_Point; };

      /** Returns the light direction at the given point. */
      CV3D getDirection(const CP3D &point) const;

      /** Returns the direction of the light. */
      const CV3D &getDirection() const { return m_Direction; };

      /** Returns the inverse direction of the light. */
      CV3D getDirectionToLight() const { return -1 * m_Direction; };

      /** Returns the intensity of the light. */
      float getIntensity() const { return m_rfIntensity; };

      /** Returns the light intensity at the given point. */
      float getIntensity(const CP3D &point) const;

      /** Returns the exponent of the intensity distribution
       * of the light (used for cone light sources with diffuse
       * light distribution properties). . */
      float getExponent() const { return m_rfExponent; };

      /** Returns the opening angle of the light. */
      float getAngle() const { return m_rfAngle; };

      /** Returns the const attenuation. */
      float getConstantAttenuation() const { return m_rfConstantAttenuation; };

      /** Returns the linear attenuation. */
      float getLinearAttenuation() const { return m_rfLinearAttenuation; };

      /** Returns the quadratic attenuation. */
      float getQuadraticAttenuation() const { return m_rfQuadraticAttenuation; };

      /** Sets the type of the lightsource. */
      void setType(LightSourceType nType) { m_LightSourceType = nType; };

      /** Sets the position of the light. */
      int setStatus(LightStatus status) { m_LightStatus = status; return 1; };

      /** Sets the position of the light. */
      int setPosition(const CP3D &point);

      /** Sets the direction of the light. */
      int setDirection(const CV3D &direction);

      /** Sets the opening angle of the light. */
      int setAngle(float rfAngle);

      /** Sets the exponent of the intensity distribution
       * of the light (used for cone light sources with diffuse
       * light distribution properties). . */
      int setExponent(float rfExponent);

      /** Sets the intensity of the light. */
      int setIntensity(float rfIntensity);

      /** Sets the color of the light source. */
      void setColor(float rfRed, float rfGreen, float rfBlue);

      /** Sets the const attenuation. */
      void setConstantAttenuation(float rfValue) { m_rfConstantAttenuation = rfValue; };

      /** Sets the linear attenuation. */
      void setLinearAttenuation(float rfValue) { m_rfLinearAttenuation = rfValue; };

      /** Sets the quadratic attenuation. */
      void setQuadraticAttenuation(float rfValue) { m_rfQuadraticAttenuation = rfValue; };

  protected:
    //methods

  private:
    //data
      LightSourceType m_LightSourceType;
      LightStatus     m_LightStatus;

      CP3D m_Point;
      CV3D m_Direction;

      float m_arfColor[3];
      float m_rfIntensity;
      float m_rfAngle;
      float m_rfExponent;
      float m_rfConstantAttenuation;
      float m_rfLinearAttenuation;
      float m_rfQuadraticAttenuation;
};
#endif
