/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/

#include "IMap.h"
#include "IProjection.h"
#include "GeoMath.h"
#include "CCentralResources.h"

IMap::IMap(QObject * parent)
: QObject(parent)
, x_off(0.0)
, y_off(0.0)
, scale(1.0)
, easyDraw(false)
, needRedraw(false)
, showCityNames(true)
, showPOIs(true)
, showStreetNames(true)
{

}


IMap::~IMap()
{

}


void IMap::setEasyDraw(bool f)
{
    easyDraw = f;
    needRedraw = true;
}


void IMap::resize(const QSize& s)
{

    size = s;
    rect.setSize(size);
    calcWorldRect();

    scaleIcon.setTopLeft(QPoint(s.width() - 200, s.height() - 50));
    scaleIcon.setBottomRight(QPoint(s.width() - 100, s.height() - 47));
    scaleTextRect.setTopLeft(QPoint(s.width() - 200, s.height() - 65));
    scaleTextRect.setBottomRight(QPoint(s.width() - 100, s.height() - 52));

    pixBottom   = QPixmap(s);

    needRedraw = true;
}


void IMap::convertPt2Deg(double& u, double& v)
{

    XY pt;
    pt.u = x_off + u * scale;
    pt.v = y_off - v * scale;
    pt = pj_inv(pt,*gpProj);
    u = pt.u * RAD_TO_DEG;
    v = pt.v * RAD_TO_DEG;
}


void IMap::convertPt2Rad(double& u, double& v)
{
    XY pt;
    pt.u = x_off + u * scale;
    pt.v = y_off - v * scale;
    pt = pj_inv(pt,*gpProj);
    u = pt.u;
    v = pt.v;
}

void IMap::convertRad2Pt(double& u, double& v)
{
    XY pt;
    pt.u = u;
    pt.v = v;
    pt = pj_fwd(pt,*gpProj);
    u = u2px(pt.u);
    v = v2px(pt.v);
}


void IMap::convertPt2M(double& u, double& v)
{
    u = x_off + u * scale;
    v = y_off - v * scale;
}


void IMap::move(const QPoint& old, const QPoint& next)
{

    XY pt1,pt2;

    pt1.u = old.x() * scale;
    pt1.v = old.y() * scale;

    pt2.u = next.x() * scale;
    pt2.v = next.y() * scale;

    x_off += pt1.u - pt2.u;
    y_off -= pt1.v - pt2.v;

    calcWorldRect();

    needRedraw = true;
}


void IMap::draw(QPainter& p,bool buffered)
{
    if(buffered) {
        if(needRedraw) {
            drawBackground();
        }
        p.drawPixmap(0,0,pixBottom);
    }
    else {
        drawBackground(&p);
    }


    QPoint px1(rect.toRect().bottomRight() - QPoint(100,50));

    // step I: get the approximate distance for 200px in the bottom right corner
    double u1 = px1.x();
    double v1 = px1.y();

    double u2 = px1.x() - 200;
    double v2 = v1;

    convertPt2M(u1,v1);
    convertPt2M(u2,v2);

    double d = u1 - u2;

    // step II: derive the actual scale length in [m]
    double a = (int)log10(d);
    double b = log10(d) - a;

//     qDebug() << log10(d) << d << a << b;

    if(0 <= b && b < log10(3)){
        d = 1 * pow(10,a);
    }
    else if(log10(3) < b && b < log10(5)){
        d = 3 * pow(10,a);
    }
    else{
        d = 5 * pow(10,a);
    }

//     qDebug() << "----" << d;

    // step III: convert the scale length from [m] into [px]
    XY pt1, pt2;
    pt1.u = px1.x();
    pt1.v = px1.y();
    convertPt2Rad(pt1.u,pt1.v);
    pt2 = GPS_Math_Wpt_Projection(pt1, d, -90 * DEG_TO_RAD);
    convertRad2Pt(pt2.u, pt2.v);

    // step IV: draw the scale
    QPoint px2(px1 - QPoint(px1.x() - pt2.u,0));

    p.setRenderHint(QPainter::Antialiasing,false);
    p.setPen(QPen(Qt::white, 9));
    p.drawLine(px1, px2 + QPoint(9,0));
    p.setPen(QPen(Qt::black, 7));
    p.drawLine(px1, px2 + QPoint(9,0));
    p.setPen(QPen(Qt::white, 5));
    p.drawLine(px1, px2 + QPoint(9,0));

    QVector<qreal> pattern;
    pattern << 2 << 4;
    QPen pen(Qt::black, 5, Qt::CustomDashLine);
    pen.setDashPattern(pattern);
    p.setPen(pen);
    p.drawLine(px1, px2 + QPoint(9,0));
    p.setRenderHint(QPainter::Antialiasing,true);

    QPoint px3(px2.x() + (px1.x() - px2.x())/2, px2.y());

    QString str = gpResources->distance2str(d);

    QFont           f = gpResources->getMapFont();
    QFontMetrics    fm(f);
    QRect           r = fm.boundingRect(str);

    r.moveCenter(px3);

    p.setPen(Qt::white);
    p.setFont(f);

    p.drawText(r.topLeft() - QPoint(-1,-1), str);
    p.drawText(r.topLeft() - QPoint( 0,-1), str);
    p.drawText(r.topLeft() - QPoint(+1,-1), str);

    p.drawText(r.topLeft() - QPoint(-1, 0), str);
    p.drawText(r.topLeft() - QPoint(+1, 0), str);

    p.drawText(r.topLeft() - QPoint(-1,+1), str);
    p.drawText(r.topLeft() - QPoint( 0,+1), str);
    p.drawText(r.topLeft() - QPoint(+1,+1), str);

    p.setFont(f);
    p.setPen(Qt::black);
    p.drawText(r.topLeft(),str);

    needRedraw = false;
}
