/***************************************************************************
 *   Copyright (C) 2000-2008 by Johan Maes                                 *
 *   on4qz@telenet.be                                                      *
 *   http://users.telenet.be/on4qz                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "imageviewer.h"

#include <QtGui>
#include <QMenu>
#include "dispatcher.h"
#include "global.h"
#include "supportfunctions.h"
#include "configparams.h"
#include "editor/editor.h"
#include "editor/editorscene.h"
#include "logging.h"

/**
	\class imageViewer 
	
	The configDialog provides access to the application-wide configuration parameters.
*/	


imageViewer::imageViewer(QWidget *parent): QScrollArea(parent)
{
	imageLabel = new QLabel;
	setWidget(imageLabel);

	popup=new QMenu (this);

	newAct = new QAction(tr("&New"),this);
	connect(newAct, SIGNAL(triggered()), this, SLOT(slotNew()));

	loadAct = new QAction(tr("&Load"), this);
  connect(loadAct, SIGNAL(triggered()), this, SLOT(slotLoad()));

	toTXAct = new QAction(tr("&To TX"), this);
  connect(toTXAct, SIGNAL(triggered()), this, SLOT(slotToTX()));

	editAct = new QAction(tr("&Edit"), this);
	connect(editAct, SIGNAL(triggered()), this, SLOT(slotEdit()));

	printAct = new QAction(tr("&Print"), this);
	connect(printAct, SIGNAL(triggered()), this, SLOT(slotPrint()));

	deleteAct = new QAction(tr("&Delete"), this);
	connect(deleteAct, SIGNAL(triggered()), this, SLOT(slotDelete()));

	propertiesAct = new QAction(tr("Propert&ies"), this);
	connect(propertiesAct, SIGNAL(triggered()), this, SLOT(slotProperties()));

	zoomInAct = new QAction(tr("Zoom In"), this);
	connect(zoomInAct, SIGNAL(triggered()), this, SLOT(slotZoomIn()));

	zoomOutAct = new QAction(tr("Zoom Out"), this);
	connect(zoomOutAct, SIGNAL(triggered()), this, SLOT(slotZoomOut()));

	fitAct = new QAction(tr("Fit to Window"), this);
	connect(fitAct, SIGNAL(triggered()), this, SLOT(slotFit()));

	popup->addAction(newAct);
	popup->addAction(loadAct);
	popup->addAction(toTXAct);
	popup->addAction(editAct);
	popup->addAction(printAct);
	popup->addAction(deleteAct);
	popup->addAction(propertiesAct);
	popup->addSeparator();
	popup->addSeparator();
	popup->addAction(zoomInAct);
	popup->addAction(zoomOutAct);
	popup->addAction(fitAct);

	imagePtr=NULL;
  originalImagePtr=NULL;
  init();
	slotFit();
}

imageViewer::~imageViewer()
{
  addToLog("deleting ImageViewer",DBIMAG);
  if (originalImagePtr!=NULL)
  {
     delete originalImagePtr;
     originalImagePtr=NULL;
   }
  if (imagePtr!=NULL)
  {
    delete imagePtr;
    imagePtr=NULL;
  }
}

bool imageViewer::readThumbSettings(QSettings *qs)
{
	qs->beginGroup("Thumbnails");
	addToLog(QString("/THUMB/"+objectName()+"/"),DBIMAG);
	fileName =qs->value(objectName()+"/filename","").toString();
  fit=TRUE;
	qs->endGroup();
  return(open(fileName,FALSE));
}

void imageViewer::writeThumbSettings(QSettings *qs)
{
	qs->beginGroup("Thumbnails");
	qs->setValue(objectName()+"/filename",fileName);
	qs->endGroup();
}

void imageViewer::setType(thumbType t)
{
	ttype=t;
	switch(t)
		{
			case RXTHUMB:
				path=rxImagesPath; 
			break;
			case TXTHUMB:
				path=txImagesPath;
			break;
			case TEMPLATETHUMB:
				path=templatesPath;
			break;
		}
}

bool imageViewer::reload()
{
  return open(fileName,TRUE);
}

void imageViewer::open(QImage image)
{
  fileName="";
  if (originalImagePtr!=NULL) delete originalImagePtr;
  originalImagePtr=new QImage(image);
  setImage();
}

bool imageViewer::open(QString fn,bool showMessage)
{
  editorScene ed;
  QFile fi(fn);
  if (fn.isEmpty()) return FALSE;
  if (originalImagePtr!=NULL) delete originalImagePtr;
  originalImagePtr =new QImage();
  if (!originalImagePtr->load(fn))
   {
    //try to load it as a template file
    if(!ed.load(fi))
      {
        if(showMessage) QMessageBox::information(this, tr("Image Viewer"),
                      tr("Cannot load %1.").arg(fn));
        fileName="";
        delete originalImagePtr;
        originalImagePtr=NULL;
        return FALSE;
      }
      delete originalImagePtr;
      originalImagePtr=new QImage(ed.renderImage()->copy());
    }
  setImage();
  fileName=fn;
  return TRUE;
}

QImage *imageViewer::getImagePtr(int w,int h)
{

  if(imagePtr!=NULL)
  {
    delete imagePtr;
    imagePtr=NULL;
  }
  if(originalImagePtr==NULL) return NULL;
  if((w!=0)&&(h!=0))
    {
      imagePtr=new QImage(originalImagePtr->scaled(QSize(w,h)));
    }
  else
    {
      imagePtr=new QImage(*originalImagePtr);
    }
  addToLog(QString("imageViewer:%1 getImagePtr imagePtr=%2").arg(QString::number((ulong)this,16)).arg(QString::number((ulong)imagePtr,16)),DBIMAG);
  return imagePtr;
}






void imageViewer::setImage()
{
  imageLabel->setPixmap(QPixmap::fromImage(*originalImagePtr));
  if (!fit) slotNormalSize();
  else slotFit();
}


void imageViewer::slotZoomIn()
{
		calcDefaultScale();;
    scaleImage(1.25);
}

void imageViewer::slotZoomOut()
{
		calcDefaultScale();
    scaleImage(0.8);
}

void imageViewer::slotNormalSize()
{
		calcDefaultScale();
    scaleFactor = 1.0;
}

void imageViewer::calcDefaultScale()
{
	double s1,s2;
  if(fit == TRUE)
		{
			s1=((double)imageLabel->size().width())/((double)imageLabel->pixmap()->size().width());
			s2=((double)imageLabel->size().height())/((double)imageLabel->pixmap()->size().height());
			scaleFactor=qMax(s1,s2);
		}
	fit=FALSE;
	setWidgetResizable(FALSE);
	imageLabel->adjustSize();
}

void imageViewer::slotFit()
{
	fit = TRUE;

  addToLog(QString("slotFit() labelSize=%1 x %2").arg(imageLabel->size().width()).arg(imageLabel->size().height()),DBIMAG);
  if(imageLabel->pixmap())
  addToLog(QString("slotFit() pixmapSize=%1 x %2").arg(imageLabel->pixmap()->size().width()).arg(imageLabel->pixmap()->size().height()),DBIMAG);
  setWidgetResizable(fit);
}

void imageViewer::fitToWindow(bool bfit)
{
    fit = bfit;
    setWidgetResizable(fit);
    if (!fit)  slotNormalSize();
}

void imageViewer::scaleImage(double factor)
{
		if( ((scaleFactor*factor)<=3.) && ((scaleFactor*factor)>=0.03))
			{
    		scaleFactor *= factor;
			}
    imageLabel->resize(scaleFactor * imageLabel->pixmap()->size());

    adjustScrollBar(horizontalScrollBar(), factor);
    adjustScrollBar(verticalScrollBar(), factor);
}

void imageViewer::adjustScrollBar(QScrollBar *scrollBar, double factor)
{
    scrollBar->setValue(int(factor * scrollBar->value()
                            + ((factor - 1) * scrollBar->pageStep()/2)));
}


void imageViewer::setFilename(QString fn)
{
	fileName=fn;
}

void imageViewer::mousePressEvent( QMouseEvent *e )
{
  QString temp;
  QString fn;
  if (( e->button() == Qt::LeftButton )&&(!fileName.isEmpty()))
    {
	      
    }
  else if (e->button() == Qt::RightButton)
    {
      if(popupEnabled)
        {
           popup->popup(QCursor::pos());
         }
    }
}


void imageViewer::slotLoad()
{
	QString fileNameTmp;
	dirDialog dd;
	dd.show();
  fileNameTmp=dd.openFileName(path);
	open(fileNameTmp,TRUE);
  if(ttype==TEMPLATETHUMB)
    {
      templatesChangedEvent *ce = new templatesChangedEvent( );
      QApplication::postEvent(dispatchPtr, ce );  // Qt will delete it when done
    }

}

void imageViewer::slotNew()
{
	callEditorEvent *ce = new callEditorEvent(this, "" );
	QApplication::postEvent(dispatchPtr, ce );  // Qt will delete it when done
}

void imageViewer::slotDelete()
{
//	imageLabel->setPixmap(QPixmap::fromImage(QImage(":/icons/transparency.png")));
	imageLabel->setPixmap(QPixmap());
	if(fileName.isEmpty()) return;
  if(QMessageBox::question(this,"Delete file","Do you want to delete the file and\n move it to the trash folder?",QMessageBox::Yes|QMessageBox::No)==QMessageBox::No)
  {
    trash(fileName,TRUE);
  }
  fileName="";
  emit layoutChanged();
}

void imageViewer::slotToTX()
{
  dispatchPtr->setTXImage(fileName);
}


void imageViewer::slotEdit()
{

	if(fileName.isEmpty())
		{
			slotLoad();
			if (fileName.isEmpty()) return;
		}
	callEditorEvent *ce = new callEditorEvent( this,fileName );
	QApplication::postEvent(dispatchPtr, ce );  // Qt will delete it when done
}


void imageViewer::slotPrint()
{
    if(imageLabel->pixmap()->isNull()) return;
    QPrintDialog dialog(&printer, this);
    if (dialog.exec())
    {
        QPainter painter(&printer);
        QRect rect = painter.viewport();
        QSize size = imageLabel->pixmap()->size();
        size.scale(rect.size(), Qt::KeepAspectRatio);
        painter.setViewport(rect.x(), rect.y(), size.width(), size.height());
        painter.setWindow(imageLabel->pixmap()->rect());
        painter.drawPixmap(0, 0, *imageLabel->pixmap());
    }
}
/*
void imageViewer::setImage(QImage *)
{
	open(fileName); // simply relaod from disk
}
*/
/*!
    \fn imageViewer::slotProperties()
 */
void imageViewer::slotProperties()
{
	QFileInfo fi(fileName);
  if(originalImagePtr==NULL) return;
   QMessageBox::information(this,"Image Properties",
	 "File: " + fileName 
	 + "\n File size:     " + QString::number(fi.size())
   + "\n Image width:   " + QString::number(originalImagePtr->width())
   + "\n Image heigth:  " + QString::number(originalImagePtr->height())
	 + "\n Last Modified: " + fi.lastModified().toString()
	 ,QMessageBox::Ok);

}
void imageViewer::clear()
{
  if(imagePtr) imagePtr->fill(qRgb ( 80, 80,80));
}

void imageViewer::init()
{
	QBrush b;
	QPalette palette;
	b.setTexture(QPixmap::fromImage(QImage(":/icons/transparency.png")));
	palette.setBrush(QPalette::Active,QPalette::Base,b);
	palette.setBrush(QPalette::Inactive,QPalette::Base,b);
	palette.setBrush(QPalette::Disabled,QPalette::Base,b);
	imageLabel->setPalette(palette);
	imageLabel->setBackgroundRole(QPalette::Base);
	imageLabel->setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Ignored);
	imageLabel->setScaledContents(true);
	setBackgroundRole(QPalette::Dark);

	fileName="";
	path="";
	ttype=RXTHUMB;
	ed=NULL;
	popupEnabled=TRUE;
	fit=TRUE;

  addToLog("init ImageViewer",DBIMAG);
  if (originalImagePtr!=NULL)
    {
      delete originalImagePtr;
      originalImagePtr=NULL;
    }
 if (imagePtr!=NULL)
  {
    delete imagePtr;
    imagePtr=NULL;
  }
 imageLabel->setPixmap(QPixmap());
}

QRgb * imageViewer::getLineAddress(unsigned int line)
{
  addToLog(QString ("imageViewer:%1 imagePtr=%2").arg(QString::number((ulong)this,16)).arg(QString::number((ulong)imagePtr,16)),DBIMAG);
  addToLog("dummy",DBIMAG);
  return (QRgb *)imagePtr->scanLine(line);
}

void imageViewer::adaptImage( int w, int h)
{
  addToLog(QString("imageViewer:%1 adaptImage_1 imagePtr=%2").arg(QString::number((ulong)this,16)).arg(QString::number((ulong)imagePtr,16)),DBIMAG);
  if(imagePtr!=NULL)
  {
    if((imagePtr->width()!=w) || (imagePtr->height()!=h))  // we need rescaling
    {
      QImage imag(imagePtr->copy());
      delete imagePtr;
      imagePtr=new QImage(imag.scaled(QSize(w,h)));
    }
  }
  else
    {
      imagePtr=new QImage(w,h,QImage::Format_RGB32);
      imagePtr->fill(qRgb ( 80, 80,80));
      fitToWindow(FALSE);
    }
  imageLabel->setGeometry(0,0,w,h);
  addToLog(QString("imageViewer:%1 adaptImage_2 imagePtr=%2").arg(QString::number((ulong)this,16)).arg(QString::number((ulong)imagePtr,16)),DBIMAG);

}



void imageViewer::updateImage()
{
  if(imagePtr) imageLabel->setPixmap(QPixmap::fromImage(*imagePtr));
}

void imageViewer::save(QString n,QString fmt)
{
 if(imagePtr) imagePtr->save(n,fmt.toLatin1().data());
}


