/***************************************************************************
 *   Copyright (C) 2000-2008 by Johan Maes                                 *
 *   on4qz@telenet.be                                                      *
 *   http://users.telenet.be/on4qz                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef SSTVPARAM_H
#define SSTVPARAM_H
#include <QString>
#include "global.h"

/** \file */

/**
SSTV Parameter functions
@author Johan Maes - ON4QZ
*/

#define NUMSSTVMODES 32 //!< Number of SSTV Modes for RX and TX
#define NUMFAXMODES 3 //!< Number of FAX Modes for RX and TX


#define GREENLINE 0 //!< index for green line buffer
#define BLUELINE 1  //!< index for blue line buffer
#define REDLINE 2		//!< index for red line buffer
#define YLINEODD 3 //!< index for intensity line buffer 0 
#define YLINEEVEN 4 //!< index for intensity line buffer 1
#define VIDEOBW 800 
#define MAXLINES 800

#define DEMODAM 0
#define DEMODFM 1
#define MINSYNCWIDTH 0.005
#define MAXSYNCWIDTH 0.020
#define RETRACEWIDTH 0.300

/**
	\brief SSTV Modes
	
*/
enum esstvMode
{
M1,
M2,
S1,
S2,
SDX,
SC2_60,
SC2_120,
SC2_180,
R24,
R36,
R72,
P3,
P5,
P7,
BW8,
BW12,
PD50,
PD90,
PD120,
PD160,
PD180,
PD240,
PD290,
MP73,
MP115,
MP140,
MP175,
FAX480,
AVT24,
AVT90,
AVT94,
NOTVALID
};

enum efaxMode
{
	NOAA,
	HFFAX,
	FAXCUSTOM,
	FAXNONE
};

/*

// struc used in frequency detection 
struct sTimeFreq
{
  DSPFLOAT t;            // time in sec
  DSPFLOAT dt;           // minimum duration in sec
  unsigned int f;     // frequency to detect (freq=0 if any freq)
};
*/

/** \brief sstv parameter structure

	Structure containing all the mode depended parameters  Some of the parameters are updated at run time
	
*/

struct sSSTVParam
{
  QString name;
  QString shortName;
	enum esstvMode mode;
  DSPFLOAT imageTime;
  unsigned int numberOfPixels; // NumberOfPixels per Line
  unsigned int numberOfDisplayLines;
	unsigned int numberOfDataLines;  // data lines
  unsigned short int VISCode;
	float sync;			//used for rx
	float fp;
	float bp;
	float blank;
	float synct;		// used for tx
	float fpt;
	float bpt;
	float blankt;
	DSPFLOAT pixelDuration;
	int subcarrier;
	int deviation;
};

struct sFAXParam
{
	~sFAXParam(){}
  QString name;
  QString shortName;
	enum efaxMode mode;
	uint modulation;
	double lpm;
	unsigned int ioc;
	unsigned int numberOfDisplayLines;
	unsigned int numberOfPixels;
	int subcarrier;
	int deviation;
	int aptStartDuration;
	int aptStartFreq;
	int aptStopDuration;
	int aptStopFreq;
	unsigned int numberOfPhasingLines;
	bool inverted;
	unsigned int colorMode;
};


extern sSSTVParam SSTVTable[NUMSSTVMODES+1];
extern sFAXParam FAXTable[NUMFAXMODES+1];

extern sSSTVParam rxSSTVParam;
extern sFAXParam  rxFAXParam;
extern sSSTVParam txSSTVParam;
extern sFAXParam txFAXParam;
extern DSPFLOAT *lineTimeTableRX;
extern DSPFLOAT *lineTimeTableTX;


void setupSSTVLineTimeTable(esstvMode modeIndex,DSPFLOAT clock,bool transmit);
DSPFLOAT lineLength(esstvMode modeIndex,DSPFLOAT clock);
DSPFLOAT syncWidth(esstvMode modeIndex,DSPFLOAT clock);
void setupFAXLineTimeTable(DSPFLOAT clock);
esstvMode initializeParametersVIS(unsigned int viscode,bool tx);
bool  initializeSSTVParametersIndex(esstvMode modeIndex,bool tx);
bool  initializeFAXParametersIndex(efaxMode modeIndex,bool tx);
esstvMode lookupVIS(unsigned int vc);
QString getSSTVModeNameLong(esstvMode m);
QString getSSTVModeNameShort(esstvMode m);
QString getFAXModeNameLong(efaxMode m);
QString getFAXModeShort(efaxMode m);

esstvMode modeLookup(unsigned int lineLength,DSPFLOAT clock);
DSPFLOAT longestLine(DSPFLOAT clock);
bool lineIsValid(esstvMode mode,unsigned int lineLength,DSPFLOAT clock);
void printActiveSSTVParam(bool tx);
void copyCustomParam(bool tx);






#endif





